;***********************************************************************
;*                                                                     *
;*                 Run-time system for BCPL on CP/M 2.2                *
;*                                                                     *
;*             R.D. Eager   University of Kent   MCMLXXXIII            *
;*                                                                     *
;***********************************************************************
;
            name  cprts
;
;
;***********************************************************************
;*                                                                     *
;*                             Store layout                            *
;*                                                                     *
;***********************************************************************
;
; [ high ]        +------------------------------------+
;                 |                                    |
;                 |               FDOS                 |
;                 |                                    |
;                 +------------------------------------+ <--- fdosbase
;                 |                                    |
;                 |    CCP (if 'reboot' is 'false')    |
;                 |                                    |
;                 +------------------------------------+
;                 |                                    |
;                 |     BCPL stream control blocks     |
;                 |                                    |
;                 +------------------------------------+ <--- scbbase
;                 |             Z80 stack              |
;                 +------------------------------------+
;                 |                                    |
;                 |             BCPL stack             |
;                 |                                    |
;                 +------------------------------------+ <--- stckbase
;                 |            Global vector           |
;                 +------------------------------------+ <--- globbase
;                 |                                    |
;                 |                                    |
;                 |            BCPL modules            |
;                 |                                    |
;                 |                                    |
;                 +------------------------------------+
;                 |                                    | <--- endmain
;                 |      Run-time support system       |
;                 |                                    |
;           0100h +------------------------------------+ <--- bcplmain
;                 |      CP/M page zero area           |
; [ low ]   0000h +------------------------------------+
;
; The  lowest  addressed  SCB is pre-allocated for the stream opened via
; 'findfile'.
;
;
;***********************************************************************
;*                                                                     *
;*                    Layout of stream control block                   *
;*                                                                     *
;***********************************************************************
;
; [ high ]   0a6h +------------------------------------+ 0a7h
;                 |                                    |
;                 |                                    |
;                 |              Buffer                |
;                 |                                    |
;                 |                                    |
;            026h +------------------+                 + 027h
;                 |                  |                 |
;                 |                  +-----------------+ 025h
;                 |                                    |
;                 |         File Control Block         |
;                 |                                    |
;            004h +------------------+                 |
;                 |    last char     |                 |
;            002h +------------------+-----------------+ 003h
;                 |     status       |  char pointer   |
; [ low ]    000h +------------------+-----------------+ 001h
;
;
;***********************************************************************
;*                                                                     *
;*                           Symbolic Equates                          *
;*                                                                     *
;***********************************************************************
;
true:       equ   0ffffh                ; BCPL value TRUE
false:      equ   \true                 ; BCPL value FALSE
;
; Assembly options
;
chef:       equ   true                  ; version for use with CHEF
fixed:      equ   true                  ; fixed global vector
            if    fixed eq true
            if    chef eq false
globsize:   equ   400                   ; size of fixed global vector
            else
globsize:   equ   270                   ; size of fixed global vector
            endif
            endif
reboot:     equ   true                  ; reboot at 'stop' (CCP overwritten)
trace:      equ   false                 ; support tracing code
debug:      equ   false                 ; support debug code
blkio:      equ   true                  ; support block I/O code
;
nscbs:      equ   4                     ; number of SCBs to be allocated
;
fdos:       equ   5                     ; entry point to CP/M
fdosbase:   equ   6                     ; start address of FDOS
dfcb:       equ   05ch                  ; default file control block
dbuff:      equ   080h                  ; default buffer address
            if    trace eq true
tbsize:     equ   128                   ; size of trace buffer
            endif
terminal:   equ   -1                    ; terminal I/O flag
fcbsize:    equ   36                    ; size of file control block
bufsize:    equ   128                   ; size of I/O buffer
scbsize:    equ   3+fcbsize+bufsize     ; size (in bytes) of stream control block
zstack:     equ   40                    ; size of Z80 stack
            if    reboot eq true
scbspace:   equ   nscbs+1*scbsize
            else
ccpsize:    equ   0806h                 ; size of CCP
scbspace:   equ   nscbs+1*scbsize+ccpsize
            endif
parmsize:   equ   32                    ; size of BCPL parameter buffer
            if    debug eq true
$debin:     equ   0
proceed:    equ   3
            endif
;
; Bits in the stream control block status byte
;
unrbit:     equ   010h                  ; stream in 'unread' state
rdbit:      equ   020h                  ; input stream
wrbit:      equ   040h                  ; output stream
usebit:     equ   080h                  ; SCB in use
;
; Character codes
;
bs:         equ   008h                  ; backspace
lf:         equ   00ah                  ; linefeed
cr:         equ   00dh                  ; carriage return
ctrlz:      equ   01ah                  ; control-Z (end of file)
space:      equ   020h                  ; space character
delete:     equ   07fh                  ; delete character
starn:      equ   00ah                  ; BCPL newline character
stckcons:   equ   't<8!'s               ; stack initialisation pattern
;
;
;***********************************************************************
;*                                                                     *
;*                           Main entry point                          *
;*                                                                     *
;***********************************************************************
;
            org   0100h
;
bcplmain:   jp    $main$                ; skip over the jump table
;
;***********************************************************************
;*                                                                     *
;*         Table of routine addresses used by the compiled code        *
;*                                                                     *
;***********************************************************************
;
$103:       jp    $apply                ; apply a routine or function
$106:       jp    $setl                 ; set up linkage in new stack frame
$109:       jp    $retn                 ; return from routine or function
;
$10c:       jp    $mult                 ; 16 bit multiplication
$10f:       jp    $div                  ; 16 bit division
$112:       jp    $rem                  ; 16 bit remainder
$115:       jp    $lsh                  ; logical left shift
$118:       jp    $rsh                  ; logical right shift
;
$11b:       jp    $eq                   ; test equal
$11e:       jp    $ne                   ; test not equal
$121:       jp    $ls                   ; test less than
$124:       jp    $ge                   ; test greater than or equal
;
$127:       jp    $lvix                 ; address of local variable
$12a:       jp    $lviy                 ; address of global variable
$12d:       jp    $lix                  ; load local variable
$130:       jp    $liy                  ; load global variable
$133:       jp    $six                  ; store local variable
$136:       jp    $siy                  ; store global variable
;
$139:       jp    $cgt                  ; test greater than for SWITCHON
$13c:       jp    $clt                  ; test less than for SWITCHON
$13f:       jp    $linsch               ; linear search for SWITCHON
;
$142:       jp    $abs                  ; absolute value
$145:       jp    $trace                ; trace routine or function entry
$148:       jp    $finish               ; terminate execution
;
; Global vector (if not dynamically allocated).
;
            if    fixed eq true
            defs  5                     ; padding - globals start at offset 050h
globvec:    defs  2*globsize+2          ; global vector
            endif
;
;
;***********************************************************************
;*                                                                     *
;*                $MAIN$    Main initialisation routine                *
;*                ======    ===========================                *
;*                                                                     *
;* Initialisation proceeds as follows:                                 *
;*                                                                     *
;*   (1) - Scan  all  loaded modules to pick up the END of the modules *
;*         (marked by the first 4 bytes NOT being 'BCPL') and also the *
;*         maximum global referenced.                                  *
;*                                                                     *
;*   (2) - Initialise  all  globals  that  are found at the end of the *
;*         BCPL modules (all other  globals  are  initialised  to  the *
;*         address  of  a routine which prints out an error message if *
;*         entered).                                                   *
;*                                                                     *
;*   (3) - Initialise the SCBs to be 'empty' and 'unused'.             *
;*                                                                     *
;*   (4) - Initialise the stack to the character constant "st".        *
;*                                                                     *
;*   (5) - Pick  up the parameter string from 080h, and transfer it to *
;*         a free area to pass it to 'start'.                          *
;*                                                                     *
;*   (6) - Set up the default FCB for a call of 'findfile'.            *
;*                                                                     *
;***********************************************************************
;
$main$:     equ   $                     ; entry point to initialisation code
;
            if    reboot eq false
            pop   hl                    ; return address to CP/M
            ld    (cpmret),hl           ; save for use later
            endif
            ld    hl,(fdosbase)         ; top of SCB space
            ld    (scbtop),hl           ; save for later
            ld    de,scbspace
            and   a                     ; clear carry
            sbc   hl,de                 ; point to base of SCB area
            ld    (scbbase),hl          ; save this pointer for later
            ld    sp,hl                 ; Z80 stack runs down from here
            ld    ix,endmain            ; end of run-time system
            call  init1                 ; first initialisation step
;
; On exit from 'init1', IX points to just beyond the last loaded module;
; i.e, to the base of the global vector (unless it is fixed and  resides
; immediately  above  the  jump  table).   The BC register pair contains
; 'bytesperword*maxgn'.
;
            ld    de,errglob            ; address of error handler
;
            if    fixed eq true
            ld    hl,globvec+2          ; point HL to global 1
            ld    (globvec),de          ; get address of error handler
            ld    de,globvec            ; point DE to global 0
            else
            ld    (globbase),ix         ; save for use later
            push  ix                    ; save for later
            push  ix
            pop   hl                    ; point HL to global vector
            ld    (hl),e                ; initialise global zero
            inc   hl
            ld    (hl),d                ; high byte
            inc   hl                    ; point HL to global 1
            pop   de                    ; point DE to global 0
            endif
;
            push  de                    ; save for later
            ex    de,hl                 ; HL -> global 0, DE -> global 1
            ldir                        ; initialise whole global vector
;
            if    fixed eq false
            ld    (stckbase),de         ; save for later
            else
            ld    (stckbase),ix         ; save for later
            endif
;
; Initialisation stage 2.
; Scan the modules to initialise all globals that  are  at  the  end  of
; loaded modules.  The format for these modules is:
;
; [ low ]  +-----------------+
;          |   B    |   C    |
;          +--------+--------+
;          |   P    |   L    |
;          +-----------------+
;          |     length      |
;          +-----------------+
;              ..........
;              ..........
;          +-----------------+
;          |  maxgn  |   0   |
;          +-----------------+
;          |    gn   | rel @ |
;          +-----------------+
;              ...........
;              ...........
;          +-----------------+
;          |    gn   | rel @ |
; [ high ] +-----------------+
;
            pop   de                    ; point DE to global vector
            ld    ix,endmain            ; end of run-time system
            call  init2                 ; initialise all globals
;
; Now initialise the stream control blocks (if any).
;
            if    nscbs+1 ne 0
            ld    b,nscbs+1
;
            ld    ix,(scbtop)           ; point to top of SCB space
            ld    de,-scbsize           ; size of stream control block
;
ioloop:     add   ix,de                 ; decrement the pointer
            call  initscb               ; initialise SCB addressed by IX
            djnz  ioloop                ; loop until finished
            endif
;
; Now initialise the Z80 and BCPL stacks to be repeated  copies  of  the
; character  constant "st".  This is useful when doing a dump of memory,
; being able to see how much of each stack has been used.
;
            if    chef eq false
            ld    hl,stckcons           ; "st"
            push  hl                    ; onto first two bytes of Z80 stack
            ld    hl,(scbbase)
            dec   hl                    ; point to second word of Z80 stack
            push  hl                    ; save source pointer on the way
            dec   hl
            dec   hl
            push  hl                    ; save destination pointer
            ld    de,(stckbase)         ; base of BCPL stack
            and   a                     ; clear carry
            sbc   hl,de                 ; get length for copy
            push  hl                    ; transfer to...
            pop   bc                    ; ...BC for copy
            pop   de                    ; destination pointer
            pop   hl                    ; source pointer
            lddr                        ; initialise it all
            pop   hl                    ; clear stack
            endif
;
; Initialise the trace buffer and trace pointer, if present.
;
            if    trace eq true
            ld    de,trcbuf             ; address of trace buffer
            push  de
            pop   hl                    ; copy to HL
            xor   a                     ; A := 0
            ld    (de),a                ; initialise first byte
            inc   de                    ; point to second byte
            ld    bc,tbsize-1           ; count for copy
            ldir                        ; clear it all out
            ld    h,a
            ld    l,0                   ; HL := 0
            ld    (trcptr),hl           ; initialise trace pointer
            endif
;
; Now  move  the  CP/M  parameter string from 080h (where it was left by
; CP/M) to the BCPL parameter area.
;
            ld    de,bcplparm           ; BCPL parameter address
            ld    hl,dbuff              ; address of CP/M parameter
            ld    bc,parmsize           ; size of parameter buffer
            ldir                        ; do the move
;
; Move the default FCB from 05ch to the 'findfile' buffer.
;
            if    chef eq false
            ld    de,(scbbase)          ; address of 'findfile' SCB
            inc   de
            inc   de
            inc   de                    ; point to FCB
            ld    hl,dfcb               ; address of default FCB
            ld    bc,fcbsize
            ldir
            endif
;
; All that there is left to do now is to set up the registers and  stack
; frame for calling 'start'.
;
; The registers are:
;
;    IX  ==>  current stack frame
;    IY  ==>  global vector, if not fixed
;    HL' ==>  parameter string
;    HL  ==>  address of 'start'
;    BC  ==>  increase in stack size
;
            if    fixed ne true
            ld    iy,(globbase)
            ld    bc,128
            add   iy,bc                 ; point IY to global 64
            endif
;
            ld    hl,terminal           ; default I/O stream
            ld    (cis),hl              ; select for input...
            ld    (cos),hl              ; ...and for output
;
            ld    hl,bcplparm
            ld    a,(hl)                ; length of parameter string
            cp    parmsize              ; is it greater than space available?
            jp    m,notrunc             ; j if not...
            ld    a,parmsize-1
            ld    (hl),a                ; ...otherwise truncate
;
notrunc:    srl   h                     ; get word address of parameter string
            rr    l
;
            if    fixed eq true
            ld    (30*2+globvec),hl     ; 'param' (global 30)
            else
            ld    (iy-66),l
            ld    (iy-65),h             ; 'param' (global 30)
            endif
;
; Routine application of 'start'
;
            exx                         ; preserve HL for use in 'start'
;
            if    debug eq true
            if    fixed eq true
            ld    de,(3*2+globvec)      ; see if 'debug' present (global 3)
            else
            ld    e,(iy-122)
            ld    d,(iy-121)            ; see if 'debug' present (global 3)
            endif
;
            ld    hl,errglob            ; 'uninitialised global' value
            and   a                     ; clear carry
            sbc   hl,de                 ; sets Z if not there
            jr    z,start               ; j if not present
;
; 'debug' is present - zap all relevant locations
;
            ld    hl,proceed
;
            if    fixed eq true
            ld    (2*2+globvec),hl
            else
            ld    (iy-124),l
            ld    (iy-123),h            ; update 'debugmode' (global 2)
            endif
;
            ld    hl,jptable
            ld    de,$103
            ld    bc,9
            ldir                        ; zap the jump table
            endif
;
start:      equ   $
;
            if    fixed eq true
            ld    hl,(1*2+globvec)      ; 'start'
            else
            ld    l,(iy-126)            ; low byte...
            ld    h,(iy-125)            ; ...and high byte of 'start'
            endif
;
;
            ld    ix,(stckbase)         ; IX = stack pointer (IY already set if required)
            ld    bc,128
            add   ix,bc                 ; point IX 64 words up stack
            ld    bc,0
;
            call  $103                  ; apply 'start'
;
; A return from 'start' comes here
;
            if    chef eq false
            jp    $finish               ; treat as FINISH
            endif
;
; Special jump table entries used only when running in debug mode.
;
            if    debug eq true
jptable:    jp    $dapply
            jp    $setl
            jp    $retn
            endif
;
;***********************************************************************
;*                                                                     *
;*               Internal initialisation subroutines                   *
;*                                                                     *
;***********************************************************************
;
; Phase  1 - find the highest referenced global, and point IX beyond the
; last module.
;
init1:      equ   $                     ; first scan through modules
            if    fixed eq false
            ld    bc,0                  ; maximum global so far
            else
            ld    bc,2*globsize         ; known already
            endif
;
mod1:       equ   $
;
            if    fixed eq false
            push  ix                    ; pointer to end of module
            ld    de,-4                 ; step count
;
loop1:      add   ix,de                 ; IX := IX - 4
            ld    a,(ix+2)              ; low byte of reloc address
            or    (ix+3)                ; sets Z if zero
            jr    nz,loop1              ; j if not - not at end
;
            ld    l,(ix+0)              ; low byte of maximum global
            ld    h,(ix+1)              ; high byte
            push  hl
            and   a                     ; clear carry
            sbc   hl,bc                 ; subtract previous maximum
            pop   hl
            jp    m,notmax              ; if not current max global
;
            ld    b,h
            ld    c,l                   ; update maximum global so far
;
notmax:     pop   ix                    ; restore module pointer
            endif
;
            call  chkmod                ; check for 'BCPL'
            ret   nz                    ; return if not
;
; We  assume  that  if we drop through, this is a BCPL module, and hence
; needs to be processed.
;
            ld    e,(ix+4)              ; low byte of length
            ld    d,(ix+5)              ; high byte
            add   ix,de                 ; increment module pointer...
            jr    mod1                  ; ...and try the next module
;
;-----------------------------------------------------------------------
;
; Phase 2 - initialise all globals. DE points to base of global vector.
;
init2:      equ   $
;
mod2:       push  ix                    ; save module pointer
nextg:      ld    bc,-4                 ; loop count
            add   ix,bc                 ; IX := IX - 4
            ld    c,(ix+2)              ; low byte of reloc address
            ld    b,(ix+3)              ; high byte
;
            ld    a,b
            or    c                     ; sets Z if last one
            jr    z,nextone             ; j if so
;
            ld    l,(ix+0)              ; low byte of global offset
            ld    h,(ix+1)              ; high byte
            add   hl,de                 ; form absolute pointer
;
            ld    (hl),c                ; store low byte
            inc   hl
            ld    (hl),b                ; store high byte
;
            jr    nextg                 ; go to do next one
;
nextone:    pop   ix                    ; restore module pointer
            call  chkmod                ; check for 'BCPL'
            ret   nz                    ; return if not
;
; Here we assume that we are indeed in a BCPL module.
;
            ld    c,(ix+4)              ; low byte of length
            ld    b,(ix+5)              ; high byte
            add   ix,bc                 ; increment module pointer...
            jr    mod2                  ; ...try the next module
;
;-----------------------------------------------------------------------
;
; Initialise the SCB pointed to by IX.
;
initscb:    equ   $
            push  ix                    ; to put in HL
            push  ix                    ; save another copy too
            exx                         ; save registers
            pop   hl                    ; HL points to SCB
;
            ld    (hl),0                ; initialise first byte
            pop   de
            inc   de                    ; point to second byte of SCB
            ld    bc,scbsize-1          ; byte count
            ldir                        ; initialise the rest
;
            exx                         ; restore registers
            ret
;
;-----------------------------------------------------------------------
;
; Check  that  IX  points at a sequence of four bytes containing 'BCPL',
; this marking the start of a BCPL module.   On  return,  the  condition
; codes  indicate  zero  if  'BCPL'  was  found, otherwise they indicate
; nonzero.
;
chkmod:     ld    a,(ix+0)
            cp    'B
            ret   nz                    ; return if not
            ld    a,(ix+1)
            cp    'C
            ret   nz                    ; return if not
            ld    a,(ix+2)
            cp    'P
            ret   nz                    ; return if not
            ld    a,(ix+3)
            cp    'L
            ret
;
;
;***********************************************************************
;*                                                                     *
;*                Subroutines used by the compiled code                *
;*                                                                     *
;*             Part I - function and routine entry and exit            *
;*                                                                     *
;***********************************************************************
;
; Apply a function or routine call.
; On entry, the parameters are as follows:
;
;    HL  =  address of the routine to jump to
;    BC  =  increase in stack size
;   (SP) =  return address from the function/routine
;
$apply:     push  ix                    ; save stack pointer
            pop   de                    ; set DE to old stack pointer
            ex    (sp),hl               ; swap calling and return addresses
            ret                         ; jump to routine
;
;-----------------------------------------------------------------------
;
; Set  up the linkage information on the new stack frame, on entry to a
; routine or function.  Effectively called  immediately  after  leaving
; '$apply'.
;
$setl:      add   ix,bc                 ; set up new stack pointer
            ld    (ix-128),e            ; save low byte of old SP
            ld    (ix-127),d            ; save high byte
            ld    (ix-126),l            ; save low byte of return address
            ld    (ix-125),h            ; save high byte
            exx                         ; restore parameters for routine
            ret
;
;-----------------------------------------------------------------------
;
; Return  from  a routine or function.  Restore all linkage information
; (old stack pointer at offset 0 on stack frame, link address at offset
; 2).
;
$retn:      ld    e,(ix-128)            ; low byte of old stack pointer
            ld    d,(ix-127)            ; high byte
            ld    c,(ix-126)            ; low byte of link address
            ld    b,(ix-125)            ; high byte
            push  bc
            push  de                    ; save old SP
            pop   ix                    ; restore stack pointer...
            ret                         ; ...and return
;
;-----------------------------------------------------------------------
;
; Apply a routine or function if 'debug' is present.
;
            if    debug eq true
$dapply:    push  ix                    ; save stack pointer
            pop   de                    ; set DE to old stack pointer
            ex    (sp),hl               ; swap calling and return addresses
;
            ld    a,(iy-124)            ; low byte of 'debugmode'
            or    a
            jr    z,nodebug             ; don't enter debug
;
            ld    ($si),bc              ; stack increase
            ld    ($sp),de              ; stack pointer
            ld    ($ra),hl              ; return address
;
            call  $setl                 ; set up new stack frame
;
            ld    ($arg1),hl            ; save args held in registers
            ld    ($arg2),de
            ld    ($arg3),bc
;
            ld    (ix-124),l            ; stack them all as well...
            ld    (ix-123),h            ; ...to complete new frame
            ld    (ix-122),e
            ld    (ix-121),d
            ld    (ix-120),c
            ld    (ix-119),b
;
            pop   hl
            ld    ($ro),hl              ; address of routine being called
            ld    de,$debin             ; debug into a program
            push  ix
            add   ix,bc
            push  ix
            pop   bc
            pop   ix                    ; BC = vector to previous stack frame
            srl   b
            rr    c                     ; get BCPL pointer
;
            exx
;
            ld    bc,34
            if    fixed eq true
            ld    hl,(3*2+globvec)
            else
            ld    l,(iy-122)
            ld    h,(iy-121)            ; address of 'debug'
            endif
            call  $apply
;
            ld    hl,($arg1)
            ld    de,($arg2)
            ld    bc,($arg3)
            exx
            ld    hl,($ro)
            push  hl
            ld    hl,($ra)
            ld    de,($sp)
            ld    bc,($si)
            ld    ix,($sp)
;
nodebug:    ret
            endif
;
;-----------------------------------------------------------------------
;
; Trace routine.  Dumps the  link  into  a  circular  buffer  for  later
; analysis  by  debugging  routines.   If the compiler 'trace' option is
; enabled, the entry sequence is:
;
;    call   trace          (comes here)
;    call   setl
;
$trace:     equ   $
;
            if    trace eq true
            call  save                  ; save all registers
            ld    hl,trcbuf
            ld    de,(trcptr)           ; in range 0 to 254
            add   hl,de                 ; point to next slot
            pop   bc                    ; get link...
            push  bc                    ; ...but save copy for return
            ld    (hl),c                ; save low byte
            inc   hl
            ld    (hl),b                ; save high byte
            ld    a,e                   ; update pointer
            add   a,2
            ld    (trcptr),a
            jp    restore               ; restore registers and return
;
trcbuf:     defs  tbsize                ; the trace buffer
trcptr:     defs  2                     ; the trace pointer
            else
            ret                         ; if no tracing
            endif
;
;-----------------------------------------------------------------------
;
            if    debug eq true
$ro:        defs  2
$ra:        defs  2
$sp:        defs  2
$si:        defs  2
$arg1:      defs  2
$arg2:      defs  2
$arg3:      defs  2
            endif
;
;***********************************************************************
;*                                                                     *
;*                Subroutines used by the compiled code                *
;*                                                                     *
;*                     Part II - arithmetic support                    *
;*                                                                     *
;***********************************************************************
;
; 16 bit multiply routine.  The arguments are the top two items on  the
; Z80 stack.
;
$mult:      call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
            ld    c,h                   ; set up CA as a register
            ld    a,l
            ld    b,16                  ; loop count
            ld    hl,0
;
mul1:       srl   c
            rra                         ; next bit of multiplicand
            jr    nc,mul2               ; don't add in
            add   hl,de
;
mul2:       ex    de,hl
            add   hl,hl                 ; (DE) := (DE) << 1
            ex    de,hl
            djnz  mul1                  ; continue until finished
;
            ex    (sp),hl               ; return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; 16  bit  divide  routine.   Operands are the top two items on the Z80
; stack.
;
$div:       call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
            call  dodiv                 ; do the divide
;
            ld    h,c                   ; answer is in CA register pair
            ld    l,a
            ex    af,af'                ; see if answer to be negated
            jp    p,notneg              ; j if not
;
            dec   hl
            ld    a,l                   ; negate is decrement, then complement
            cpl
            ld    l,a                   ; low byte of register
            ld    a,h
            cpl
            ld    h,a                   ; high byte
;
notneg:     ex    (sp),hl               ; return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; 16 bit remainder after divide.  Operands are the same as for divide.
;
$rem:       call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
            call  dodiv                 ; do the divide
            ex    (sp),hl               ; swap remainder with return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Do a 16 bit division.  On entry, arguments are in HL and DE.  On exit,
; the quotient is in CA, while the remainder is in HL.  DE contains  the
; quotient unchanged.
;
dodiv:      ld    a,h                   ; test to see...
            xor   d                     ; ...if same sign
            ex    af,af'                ; save S flag for later
;
            bit   7,d                   ; is DE negative?
            jr    z,dd1                 ; j if not
;
            dec   de                    ; negate DE
            ld    a,d
            cpl
            ld    d,a                   ; complement high byte
            ld    a,e
            cpl
            ld    e,a                   ; complement low byte
;
dd1:                                    ; see if HL needs to be negated
            bit   7,h                   ; is HL negative?
            jr    z,dd2                 ; j if not
;
            dec   hl                    ; negate HL
            ld    a,h
            cpl
            ld    h,a                   ; complement high byte
            ld    a,l
            cpl
            ld    l,a                   ; complement low byte
;
dd2:                                    ; set up registers for divide loop
            ld    c,h
            ld    a,l                   ; set up CA register pair
            ld    hl,0                  ; HL will hold remainder
            ld    b,17                  ; loop count
            and   a                     ; clear carry
            jr    d1                    ; start the division
;
d2:         adc   hl,hl                 ; HL := HL << 1 + carry
            jp    m,derr                ; j if overflow
;
d1:         sbc   hl,de                 ; test sign of remainder
            jr    nc,d3                 ; not enough yet
            add   hl,de
;
d3:         ccf
            rla
            rl    c                     ; rotate in the answer
            djnz  d2                    ; loop until done
;
derr:       ret
;
;-----------------------------------------------------------------------
;
; Compare  the  top  two  items on the stack to see if 'arg2' is >= than
; 'arg1'.  If so, leave -1 (TRUE) on the stack, otherwise 0.
;
$ge:        call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
;
            ld    a,h                   ; 'sign' byte
            xor   d                     ; see if they are the same sign
            jp    p,gt1                 ; if so, subtract without fear!
;
            rl    d                     ; C := <sign bit> of DE
            sbc   hl,hl                 ; set answer
            jr    gt2
;
gt1:        sbc   hl,de                 ; subtract arguments
            ccf                         ; >= is NOT <
            sbc   hl,hl                 ; set answer
;
gt2:        ex    (sp),hl               ; return address
            push  hl                    ; save the answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Compare the top two items on the stack to see if 'arg2' < 'arg1'.
; TRUE or FALSE are returned as answers.
;
$ls:        call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
;
            ld    a,h                   ; 'sign' bit of HL
            xor   d                     ; are the operands of same sign?
            jp    p,lt1                 ; if so, subtract
;
            rl    h                     ; C := <sign bit> of HL
            sbc   hl,hl                 ; set answer
            jr    lt2
;
lt1:        sbc   hl,de                 ; subtract operands
            sbc   hl,hl                 ; set amswer
;
lt2:        ex    (sp),hl               ; return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Compare the top two items on the stack, to see if 'arg2' is equal
; to 'arg1'.  TRUE or FALSE are returned as answers.
;
$eq:        call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
;
            and   a                     ; clear carry
            sbc   hl,de                 ; see if equal
            scf                         ; carry set if equal
            jr    z,eq1                 ; j if equal
;
            ccf                         ; complement if not equal
;
eq1:        sbc   hl,hl                 ; set answer
            ex    (sp),hl               ; return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Compare  the  top two items on the stack, to see if 'arg1' is not
; equal to 'arg1'.  TRUE or FALSE are returned as answers.
;
$ne:        call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
;
            and   a                     ; clear carry
            sbc   hl,de                 ; sets Z if HL = DE
            jr    z,neq1                ; j if equal
;
            scf                         ; set carry
;
neq1:       sbc   hl,hl                 ; set answer
            ex    (sp),hl               ; return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Takes the top two items on the stack, and returns 'arg2' << 'arg1'.
;
$lsh:       call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
;
            ld    a,e
            or    a                     ; sets Z if E is zero
;
lsh1:       jr    z,lsh2                ; end of shifting
            add   hl,hl                 ; HL := HL << 1
            dec   a
            jr    lsh1                  ; loop until finished
;
lsh2:       ex    (sp),hl               ; return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Takes the top two items on the stack, and returns 'arg2' >> 'arg1'.
;
$rsh:       call  save                  ; save all registers
            pop   bc                    ; return address
            pop   de                    ; argument 1
            pop   hl                    ; argument 2
            push  bc                    ; save return address
;
            ld    a,e
            or    a                     ; sets Z if E is zero
;
rsh1:       jr    z,rsh2                ; end of loop
            srl   h
            rr    l                     ; HL := HL >> 1
            dec   a
            jr    rsh1
;
rsh2:       ex    (sp),hl               ; return address
            push  hl                    ; save answer
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Takes the top item on the stack, and returns the value
;    IX + <item on stack>.
; i.e. the address of local variable <item on stack>/2.
;
$lvix:      call  save                  ; save all registers
            pop   hl                    ; return address
            ld    c,(hl)                ; low byte of offset
            inc   hl
            ld    b,(hl)                ; high byte
            inc   hl
;
            ex    de,hl
            push  ix                    ; current stack pointer
            pop   hl
            add   hl,bc                 ; get offset needed (byte)
            srl   h
            rr    l                     ; now into BCPL pointer
            push  hl                    ; answer
            push  de                    ; return address
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; As for '$lvix', but returns the value
;    IY + <item in stack>
; i.e. the address of a global variable rather than a local.
;
$lviy:      equ   $
            if    fixed eq false
            call  save                  ; save all registers
            pop   hl                    ; return address
            ld    c,(hl)                ; low byte of offset
            inc   hl
            ld    b,(hl)                ; high byte
            inc   hl
;
            ex    de,hl
            push  iy                    ; current global pointer
            pop   hl
            add   hl,bc                 ; get offset needed (byte)
            srl   h
            rr    l                     ; now into BCPL pointer
            push  hl                    ; answer
            push  de                    ; return address
            jp    restore               ; restore registers and return
            endif
;-----------------------------------------------------------------------
;
; Load local variable onto the stack.  Used for variables out  of  range
; of the (IX+d) instructions.
;
$lix:       call  save                  ; save all registers
            pop   hl                    ; return address
            ld    c,(hl)                ; low byte of offset
            inc   hl
            ld    b,(hl)                ; high byte
            inc   hl
;
            ex    de,hl
            push  ix                    ; save stack pointer
            add   ix,bc
            ld    l,(ix+0)
            ld    h,(ix+1)
            pop   ix
;
            push  hl                    ; answer
            push  de
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Load  global variable onto the stack.  Like '$lix', used if the offset
; is out of range for the (IY+d) instructions.
;
$liy:       equ   $
            if    fixed eq false
            call  save                  ; save all registers
            pop   hl                    ; return address
            ld    c,(hl)                ; low byte of offset
            inc   hl
            ld    b,(hl)                ; high byte
            inc   hl
;
            ex    de,hl
            push  iy                    ; save global pointer
            add   iy,bc
            ld    l,(iy+0)
            ld    h,(iy+1)
            pop   iy
;
            push  hl                    ; answer
            push  de
            jp    restore               ; restore registers and return
            endif
;
;-----------------------------------------------------------------------
;
; Reverse of '$lix'; stores an item at an offset from  the  IX  register
; (one which is out of range for direct access).
;
$six:       call  save                  ; save all registers
            pop   hl                    ; return address
            ld    c,(hl)
            inc   hl
            ld    b,(hl)                ; BC := offset
            inc   hl
            ex    de,hl
;
            pop   hl                    ; thing to be stored
            push  ix                    ; save stack pointer
            add   ix,bc
            ld    (ix+0),l
            ld    (ix+1),h
            pop   ix
            push  de                    ; save return address
            jp    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Identical  to '$six', but works on offsets from the IY register.  Used
; for storing globals that are out of range of direct instructions.
;
$siy:       equ   $
            if    fixed eq false
            call  save                  ; save all registers
            pop   hl                    ; return address
            ld    c,(hl)
            inc   hl
            ld    b,(hl)                ; BC := offset
            inc   hl
            ex    de,hl
;
            pop   hl                    ; thing to be stored
            push  iy                    ; save global pointer
            add   iy,bc
            ld    (iy+0),l
            ld    (iy+1),h
            pop   iy
            push  de                    ; save return address
            jp    restore               ; restore registers and return
            endif
;
;-----------------------------------------------------------------------
;
; Compare Greater Than; takes two arguments (in DE and HL)  and  returns
; with the C flag set if HL is greater than DE.  Used only in SWITCHONs.
;
$cgt:       ld    a,h
            xor   d                     ; see if HL and DE are same sign
            jp    p,cgt1                ; if same sign, subtract
;
            push  de                    ; save value
            rl    d                     ; C := sign bit of DE
            pop   de                    ; restore value
            ret
;
cgt1:       push  hl                    ; if same sign
            sbc   hl,de                 ; subtract operands
            jr    z,cgteq               ; special case if =
;
            ccf
;
cgteq:      pop   hl                    ; restore HL
            ret
;
;-----------------------------------------------------------------------
;
; Compare Less Than.  Returns with the C flag set if HL is less than DE.
; Used only in SWITCHONs.
;
$clt:       ld    a,h
            xor   d                     ; see if same sign
            jp    p,clt1                ; j if so
;
            push  hl                    ; save value
            rl    h                     ; C := sign bit of HL
            pop   hl                    ; restore value
            ret
;
clt1:       push  hl                    ; save value
            sbc   hl,de                 ; subtract operands
            pop   hl                    ; restore value
            ret
;
;-----------------------------------------------------------------------
;
; Do  a  linear  search of a table (entry point at the return address of
; the routine).  The number of items in the table is  given  in  B.   HL
; contains the value to be compared.
;
$linsch:    pop   de                    ; pointer to search table
;
lsloop:     ld    a,(de)                ; first byte
            cp    l                     ; same as low byte of HL?
            jr    nz,ls1                ; j if not
            inc   de                    ; point to next byte
            ld    a,(de)
            cp    h                     ; top byte equal as well?
            jr    nz,ls2                ; j if not
;
            inc   de                    ; required entry found
            ex    de,hl
            ld    e,(hl)
            inc   hl
            ld    d,(hl)                ; load the label address
            ex    de,hl
            jp    (hl)                  ; jump to label
;
ls1:        inc   de
ls2:        inc   de
            inc   de                    ; move past label
            inc   de
            djnz  lsloop                ; loop until B = 0
;
            push  de                    ; now holds return address
            ret
;
;-----------------------------------------------------------------------
;
; Return the absolute value of the top item on the stack.
;
$abs:       call  save                  ; save all registers
            pop   de                    ; return address
            pop   hl                    ; the value we want the ABS of
            bit   7,h                   ; is HL negative?
            jr    z,abs1                ; j if not
;
            dec   hl                    ; negate HL by...
            ld    a,h                   ; ...decrementing and...
            cpl                         ; ...complementing it
            ld    h,a
            ld    a,l
            cpl
            ld    l,a
;
abs1:       push  hl                    ; save answer
            push  de                    ; restore return address
            jr    restore               ; restore registers and return
;
;-----------------------------------------------------------------------
;
; Save all registers, for restoration after a subroutine call.
;
save:       ld    ($hl),hl
            ld    ($de),de
            ld    ($bc),bc
            ret
;
;-----------------------------------------------------------------------
;
; Restore all registers after a subroutine call.
;
restore:    ld    hl,($hl)
            ld    de,($de)
            ld    bc,($bc)
            ret
;
;-----------------------------------------------------------------------
;
; Register save area
;
$hl:        defs  2
$de:        defs  2
$bc:        defs  2
;
;
;***********************************************************************
;*                                                                     *
;*             ERRGLOB - the initial state for all globals             *
;*                                                                     *
;***********************************************************************
;
errglob:    ld    de,errmsg
            ld    c,9
            call  fdos                  ; print out error message
            jp    $finish               ; effectively call 'stop'
;
errmsg:     defb  cr,lf
            if    chef eq false
            defm  "**Call to undefined global"
            else
            defm  "Amok"
            endif
crlf:       defb  cr,lf,'$
;
;
;***********************************************************************
;*                                                                     *
;*                              Data areas                             *
;*                                                                     *
;***********************************************************************
;
            align 2
;
            if    fixed eq  false
globbase:   defs  2                     ; base of global vector
            endif
scbtop:     defs  2                     ; top of space for SCBs
stckbase:   defs  2                     ; base of the BCPL stack
scbbase:    defs  2                     ; base of stream control blocks
bcplparm:   defs  parmsize              ; area for parameter string
;
wfcb:       defs  1                     ; workspace file control block
wfname:     defs  8
wfext:      defs  3
            defs  24
;
            if    chef eq false
workname:   defs  16                    ; extra filename area
            endif
;
            align 2
;
cis:        defs  2                     ; current input stream
cos:        defs  2                     ; current output stream
;
            if    reboot eq false
cpmret:     defs  2                     ; return address for CP/M
            endif
;
;
;***********************************************************************
;*                                                                     *
;*                STOP - called at the end of a BCPL run               *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; 'stop(rc)'  causes  the  execution of the current BCPL program to stop
; with return code 'rc'.  If 'writef' is present, then this return  code
; is given exactly.
; All I/O streams have to be terminated and closed.
;
stop:       equ   $
;
            if    chef eq false
            exx
            ld    a,h                   ; get return code
            or    l
            jr    z,$finish             ; j if program terminated correctly
;
            push  hl                    ; save return code
;
            if    fixed eq true
            ld    hl,(globvec)          ; global zero is 'undefined global'
            ld    de,(76*2+globvec)     ; address of 'writef'
            else
            ld    l,(iy-128)            ; global zero is 'undefined global'
            ld    h,(iy-127)
            ld    e,(iy+24)             ; address of 'writef'
            ld    d,(iy+25)
            endif
;
            and   a                     ; clear carry
            sbc   hl,de                 ; if zero, then no 'writef'
            jr    z,nowritef            ; j if not
;
            ex    de,hl                 ; set HL to address of 'writef'
            exx                         ; change to argument register set
            ld    hl,smsg1              ; stop message
            srl   h
            rr    l                     ; get BCPL pointer
            pop   de                    ; get return code back
            ld    bc,terminal
            ld    (cos),bc              ; selectoutput(terminal)
            exx                         ; change to call register set
;
            ld    bc,6                  ; stack increase
            call  $apply                ; call 'writef'
            jr    $finish
;
nowritef:   pop   hl                    ; just output a simple message
            ld    de,smsg2
            ld    c,9
            call  fdos                  ; print it out
            endif
;
; Code to tidy up files, etc.
;
$finish:    equ   $
;
            if    nscbs+1 ne 0
            ld    b,nscbs+1
            ld    hl,(scbtop)           ; top of SCB area
            ld    de,-scbsize           ; get ready to close all files
;
clloop:     add   hl,de                 ; get address of next SCB
            ld    a,(hl)                ; get status byte
            and   usebit
            call  nz,close              ; if the stream was open
            djnz  clloop                ; loop until finished
            endif
;
; *** Return to CP/M ***
;
            if    reboot eq true
            jp    0                     ; reboot the CCP
            else
            ld    hl,(cpmret)
            jp    (hl)                  ; return to CP/M
            endif
;
;-----------------------------------------------------------------------
;
; Internal  subroutine  used  by  'stop'  to close a file.  On entry, HL
; points to the base of the associated SCB.
;
close:      push  hl                    ; save registers
            push  de
            push  bc
;
            ld    a,(hl)                ; status byte of FCB
            and   wrbit                 ; open for writing?
            jr    nz,clw                ; j if so
;
            ld    (cis),hl              ; selectinput(HL)
            call  iendread              ; close the file
            jr    clret                 ; join common code
;
clw:        ld    (cos),hl              ; selectoutput(HL)
            call  iendwrite             ; close the file
;
clret:      pop   bc                    ; restore registers
            pop   de
            pop   hl
            ret
;
;-----------------------------------------------------------------------
;
; Messages
;
            if    chef eq false
smsg2:      defb  cr,lf
            defm  '**RC > 0'
            defb  cr,lf,'$
;
            align 2
;
smsg1:      defb  11,starn
            defm  '**RC = %N'
            defb  starn
            endif
;
;
;***********************************************************************
;*                                                                     *
;*        WRCH - write a character to the current output stream        *
;*                                                                     *
;***********************************************************************
;
            align 2
;
wrch:       push  hl                    ; save return address
            ld    hl,(cos)              ; current output stream
            ld    a,h
            or    l                     ; no selected output?
            ret   z                     ; return if no output selected
;
            ld    de,terminal
            and   a                     ; clear carry
            sbc   hl,de                 ; is the output to the terminal?
            jr    nz,wrch1              ; j if not
;
            exx                         ; get character to be written
            jr    wrcht
;
wrch1:      exx                         ; get character to be written
            ld    de,(cos)              ; point to SCB
            ld    a,l
            cp    starn                 ; is character a newline?
            jr    nz,bwrch              ; j if not - write character and return
;
            push  de
            pop   hl                    ; get pointer to SCB
            inc   hl
            inc   hl                    ; point to 'last char' position
            ld    (hl),a                ; save copy of character
;
            ld    l,cr
            call  bwrch                 ; binary write carriage return
            ld    l,lf                  ; drop through for linefeed, then return
;
; Write a character in binary to the currently selected output stream.
;
bwrch:      push  de                    ; save pointer to SCB
            inc   de                    ; point to current character pointer
            ld    a,(de)                ; pick it up
            push  de                    ; save SCB pointer
            push  hl                    ; save character
            or    a                     ; test character pointer
            call  m,bwrec               ; write out record if buffer full
            pop   hl                    ; restore character
            pop   de                    ; restore SCB pointer
            or    a                     ; set flags on result of write
            jp    m,bwrch2              ; j if failure
;
            push  hl                    ; save character
            push  de                    ; copy SCB pointer...
            pop   hl                    ; ...to HL
;
            ld    bc,fcbsize+2          ; offset to buffer
            add   hl,bc                 ; point HL to buffer
            ld    c,a                   ; C := character pointer
            add   hl,bc                 ; point to next free byte in buffer
;
            pop   bc                    ; C := character
            ld    (hl),c                ; store character in buffer
            inc   a
            ld    (de),a                ; update character pointer field
;
            inc   de                    ; point to 'last char' field
            ld    a,c
            ld    (de),a                ; update 'last char' field
bwrch1:     pop   de                    ; restore SCB pointer
            ret                         ; HL <> 0 indicates success
;
bwrch2:     ld    hl,0                  ; HL = 0 indicates failure
            jr    bwrch1
;
; Write a record in binary to the currently selected output stream.
; On entry, DE points to FCB - 2 (SCB + 1).
;
bwrec:      inc   de
            inc   de                    ; DE now points to FCB
            push  de                    ; save copies for later
            push  de
            ld    hl,fcbsize
            add   hl,de                 ; point HL to buffer
            ex    de,hl                 ; point DE to buffer
            ld    c,26                  ; set DMA address
            call  fdos
;
            pop   de                    ; point DE to FCB
            ld    c,34                  ; write random
            call  fdos                  ; write the record
            pop   hl                    ; point HL to FCB
            call  nextrec               ; increment record number
            or    a                     ; set flags on result of write
            ret   z                     ; return unless failure
            ld    a,-1                  ; signal failure
            ret
;
; Write a character to the terminal.
;
wrcht:      ld    a,l                   ; character to be printed
;
            if    chef eq true
            cp    delete                ; ignore delete in case...
            jr    z,retone              ; ...it prints as a blob
            endif
;
            cp    starn                 ; newline character?
            jr    nz,binwr              ; j if not
;
            ld    a,cr
            call  binwr                 ; write out the character
;
            ld    a,lf                  ; drop through to write linefeed
;
; Write the binary character in A to the terminal.
;
binwr:      ld    c,2
            ld    e,a                   ; write character to terminal...
            call  fdos
            jr    retone                ; ...and return
;
;***********************************************************************
;*                                                                     *
;*           BINWRCH - write a character in binary to the COS          *
;*                                                                     *
;***********************************************************************
;
            if    chef eq false
            align 2
;
; Write  the character in HL to the currently selected output stream, in
; binary.
;
binwrch:    push  hl                    ; save return address
            ld    hl,(cos)
            ld    a,h
            or    l                     ; any selected stream?
            ret   z                     ; return if no output selected
;
            ld    de,terminal
            and   a                     ; clear carry
            sbc   hl,de
            jr    nz,binw2              ; j if not terminal
;
            ld    a,l
            jr    binwr                 ; use common terminal output code
;
binw2:      exx                         ; get character to be written
            ld    de,(cos)              ; point to SCB
            jr    bwrch                 ; write character and return
            endif
;
;-----------------------------------------------------------------------
;
; Increment the random record number in the FCB pointed to by HL.
;
nextrec:    ld    de,33                 ; offset of record number in FCB
            add   hl,de                 ; point to record number
            inc   (hl)                  ; do the low byte
            ret   nz                    ; return if no overflow...
            inc   hl                    ; ...else move to high byte...
            inc   (hl)                  ; ...and increment that
            ret
;
;-----------------------------------------------------------------------
;
; Return a nonzero value in HL, indicating success.
;
retone:     ld    hl,1
            ret
;
;-----------------------------------------------------------------------
;
; Return zero in HL, indicating failure.
;
retzero:    ld    hl,0
            ret
;
;
;***********************************************************************
;*                                                                     *
;*        RDCH - read a character from the current input stream        *
;*                                                                     *
;***********************************************************************
;
            align 2
;
rdch:       push  hl                    ; save return address
            ld    de,(cis)              ; current input stream
            ld    a,d
            or    e                     ; no selected input?
            jr    z,retzero             ; return if no input selected
;
            ld    hl,terminal           ; is the CIS to the terminal?
            and   a                     ; clear carry
            sbc   hl,de
            jr    z,rdcht               ; j if so and read from terminal
;
            if    chef eq false
            ld    a,(de)                ; get status byte
            and   unrbit                ; see if pending 'unrdch'
            jr    z,nounrd              ; j if not
;
            ld    a,(de)                ; get status byte again
            and   \unrbit               ; clear the bit
            ld    (de),a
            inc   de
            inc   de                    ; point to last character returned
            ld    a,(de)
            ld    l,a
            ld    h,0
            ret                         ; return with last character
;
nounrd:     equ   $
            endif
;
            push  de                    ; save SCB pointer
            call  brdch                 ; read character in binary
            pop   de                    ; restore SCB pointer
            ld    a,h
            or    a
            ret   m                     ; return if 'endstreamch'
;
            ld    a,l                   ; test for special characters
            cp    cr                    ; is it carriage return?
            jr    z,rdchnl              ; j if so
;
            cp    ctrlz                 ; is it control-Z?
            ret   nz                    ; return if not
;
rdeof:      ld    hl,-1                 ; return 'endstreamch'
            ret
;
rdchnl:     call  brdch                 ; ignore the linefeed
            ld    hl,(cis)
            inc   hl
            inc   hl                    ; point HL to 'last char' field
            ld    (hl),starn
;
            ld    hl,starn
            ret
;
; Read a character in binary from the currently selected input stream.
;
brdch:      push  de                    ; copy SCB pointer to HL
            pop   hl                    ; check to see if 'endstreamch'...
            inc   hl                    ; ...has been sent
            inc   hl                    ; if so, return it again!
            ld    a,(hl)
            cp    ctrlz                 ; see if control-Z
            jr    nz,binrch             ; j if not
;
            ld    h,0
            ld    l,a
            ret
;
binrch:     inc   de                    ; point to character pointer field
            ld    a,(de)                ; pick up character pointer
            or    a
            jp    p,brdch1              ; j if new record not needed
;
            push  de                    ; save SCB pointer
            inc   de
            inc   de                    ; point to FCB
            push  de                    ; save copies for later
            push  de
;
            ld    hl,fcbsize
            add   hl,de                 ; point HL to buffer
            ex    de,hl                 ; point DE to buffer
            ld    c,26                  ; set DMA address
            call  fdos
;
            pop   hl                    ; restore FCB pointer
            call  nextrec               ; increment record number
;
            pop   de                    ; restore FCB pointer
            ld    c,33                  ; read next record
            call  fdos
;
            or    a                     ; did it work?
            pop   de                    ; restore SCB pointer
            jr    nz,rdeof              ; return 'endstreamch' if failure
;
brdch1:     push  de                    ; ready to grab another character
            pop   hl                    ; get character pointer
            ld    bc,fcbsize+2          ; offset to buffer
            add   hl,bc                 ; point to start of buffer
;
            ld    c,a
            add   hl,bc                 ; point to next character
            ld    l,(hl)                ; get character
            ld    h,0
            inc   a                     ; increment character pointer...
            ld    (de),a                ; ...and store it away
;
            if    chef eq false
            ld    a,l
            inc   de                    ; point to 'last char' field
            ld    (de),a                ; save copy of last character
            endif
;
            ret
;
rdcht:      call  binrd                 ; grab a character from the terminal
            cp    ctrlz                 ; is it control-Z?
            jr    z,rdeof               ; j if so to give 'endstreamch'
;
            cp    cr                    ; is it carriage return?
            jr    nz,rdcht1             ; j if not
;
            ld    c,9
            ld    de,crlf               ; carriage return, linefeed
            call  fdos                  ; rack it up
;
            ld    a,starn               ; convert to newline
rdcht1:     ld    l,a
            ld    h,0                   ; HL contains returned character
            ret
;
binrd:      ld    c,1
            call  fdos                  ; read character
            if    chef eq true          ; map backspace to delete
            cp    bs                    ; backspace?
            ret   nz                    ; return if not
            ld    a,delete              ; replacement value
            endif
            ret
;
;***********************************************************************
;*                                                                     *
;*      BINRDCH - return the next character from the CIS in binary     *
;*                                                                     *
;***********************************************************************
;
            if    chef eq false
            align 2
;
; Returns  the  next character from the currently selected input stream,
; in binary.
;
binrdch:    push  hl                    ; save return address
            ld    hl,(cis)              ; pick up stream selection
            ld    a,h
            or    l                     ; unselected?
            ret   z                     ; return if no input selected
;
            ld    de,terminal
            and   a                     ; clear carry
            ex    de,hl                 ; save HL for later
            sbc   hl,de                 ; CIS = terminal?
            jr    nz,binrch             ; j if not
;
            call  binrd                 ; grab character in binary
            jr    rdcht1
            endif
;
;
;***********************************************************************
;*                                                                     *
;*         RDBLOCK - read a block from the current input stream        *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; Input from the terminal is not allowed with this function.
;
            if    blkio eq true
rdblock:    push  hl                    ; save return address
            ld    de,(cis)              ; current input stream
            ld    a,d
            or    e                     ; no selected input?
            jp    z,retzero             ; j if not
;
            ld    hl,terminal           ; is the CIS to the terminal?
            and   a                     ; clear carry
            sbc   hl,de
            ret   z                     ; return if so - invalid here
;
            exx                         ; get BCPL parameters
;
; The buffer address is in HL, and the block number is in DE.
;
            sla   l
            rl    h                     ; form byte address of buffer
            push  hl                    ; save buffer address
;
            ld    hl,(cis)              ; point to SCB
            inc   hl
            inc   hl
            inc   hl                    ; point to FCB
            push  hl                    ; save FCB pointer
            ld    bc,33                 ; offset to random record number
            add   hl,bc                 ; point to record number
            ld    (hl),e                ; set low byte
            inc   hl                    ; point to high byte
            ld    (hl),d                ; set high byte
;
            pop   hl                    ; get FCB pointer again
            pop   de                    ; get buffer address
            push  hl                    ; save FCB pointer
            ld    c,26                  ; set DMA address
            call  fdos
;
            pop   de                    ; get FCB pointer again
            ld    c,33                  ; read random record
            call  fdos
            or    a                     ; test result
            jp    nz,retzero            ; j if read failed
            jp    retone                ; return nonzero for success
            endif
;
;
;***********************************************************************
;*                                                                     *
;*         WRBLOCK - write a block to the current output stream        *
;*                                                                     *
;***********************************************************************
;
; Output to the terminal is not allowed with this function.
;
            if    blkio eq true
wrblock:    push  hl                    ; save return address
            ld    de,(cos)              ; current output stream
            ld    a,d
            or    e                     ; no selected output?
            jp    z,retzero             ; j if not
;
            ld    hl,terminal           ; is the COS to the terminal?
            and   a                     ; clear carry
            sbc   hl,de
            ret   z                     ; return if so - invalid here
;
            exx                         ; get BCPL parameters
;
; The buffer address is in HL, and the block number is in DE.
;
            sla   l
            rl    h                     ; form byte address of buffer
            push  hl                    ; save user buffer address
;
            ld    hl,(cos)              ; point to SCB
            inc   hl
            inc   hl
            inc   hl                    ; point to FCB
            push  hl                    ; save FCB pointer
            ld    bc,33                 ; offset to random record number
            add   hl,bc                 ; point to record number
            ld    (hl),e                ; set low byte
            inc   hl                    ; point to high byte
            ld    (hl),d                ; set high byte
;
            pop   hl                    ; get FCB pointer again
            pop   de                    ; get buffer address
            push  hl                    ; save FCB pointer
            ld    c,26                  ; set DMA address
            call  fdos
;
            pop   de                    ; point to FCB
            ld    c,34                  ; write random record
            call  fdos
;
            or    a                     ; test result
            jp    nz,retzero            ; j if write failed
            jp    retone                ; return nonzero for success
            endif
;
;
;***********************************************************************
;*                                                                     *
;*        ENDREAD - close and deselect the current input stream        *
;*                                                                     *
;***********************************************************************
;
            align 2
;
endread:    push  hl                    ; save return address
iendread:   ld    hl,(cis)
            ld    a,h                   ; is the CIS null?
            or    l
            ret   z                     ; return if no selection
;
            ld    de,terminal
            and   a                     ; clear carry
            ex    de,hl                 ; preserve HL
            sbc   hl,de                 ; was CIS the terminal?
            ex    de,hl                 ; restore HL
            jr    nz,er1                ; j if not
            ld    a,1
            jr    er2                   ; return nonzero for success
;
er1:        ld    (hl),0                ; deallocate the SCB
            inc   hl
            inc   hl
            inc   hl                    ; point HL to FCB
            ex    de,hl                 ; point DE to FCB
            ld    c,16                  ; close file
            call  fdos
            xor   0ffh                  ; set zero if failure
er2:        ld    hl,0
            ld    (cis),hl              ; deselect the stream
            ld    l,a                   ; result in HL
            ret
;
;***********************************************************************
;*                                                                     *
;*      ENDWRITE - closes and deselects the current output stream      *
;*                                                                     *
;***********************************************************************
;
            align 2
;
endwrite:   push  hl                    ; save return address
iendwrite:  ld    hl,(cos)
            ld    a,h
            or    l                     ; is the COS null?
            ret   z                     ; return if no output selected
;
            ld    de,terminal
            and   a                     ; clear carry
            ex    de,hl                 ; preserve HL
            sbc   hl,de                 ; was COS the terminal?
            ex    de,hl                 ; restore HL
            jr    nz,ew4                ; j if not
            ld    a,1                   ; return nonzero for success
            push  af                    ; save result
            jr    ew1
;
ew4:        ld    (hl),0                ; deallocate the SCB
            push  hl
            pop   de                    ; point DE to SCB
            inc   hl                    ; point to character pointer field
            inc   hl                    ; point to 'last char' field
            ld    a,(hl)                ; pick up character
            cp    starn                 ; was it a newline?
            jr    z,nofill              ; j if so
;
            ld    hl,cr                 ; terminate incomplete line
            call  bwrch
            ld    hl,lf
            call  bwrch
;
nofill:     inc   de                    ; point to character pointer field
            ld    a,(de)
            push  de                    ; save SCB pointer
            or    a
            jp    p,ew6
            call  bwrec                 ; write record if buffer full
            jr    ew5
ew6:        xor   a                     ; A := 0
ew5:        pop   de                    ; restore SCB pointer
            jr    nz,ew3                ; j if failure
            ld    a,(de)                ; get character pointer
            or    a
            jp    m,ew3                 ; j if record written
;
            push  de
            pop   hl                    ; point HL to character pointer field
            ld    bc,fcbsize+2
            add   hl,bc                 ; point HL to buffer
            ld    c,a
            add   hl,bc                 ; point to next free byte in buffer
;
ewloop:     or    a                     ; reached end of buffer yet?
            jp    m,ew2                 ; j if so
            ld    (hl),ctrlz            ; pad with control-Z characters
            inc   hl
            inc   a
            jr    ewloop
;
; Now write out the last record, and close the file.
;
ew2:        push  de                    ; save pointer to character pointer
            call  bwrec
            pop   de                    ; restore it
;
ew3:        inc   de
            inc   de                    ; point to FCB
            push  af                    ; save result of write
            ld    c,16                  ; close file
            call  fdos
            pop   bc                    ; combine results
            or    c
            push  af
;
ew1:        ld    hl,0
            ld    (cos),hl              ; deselect the stream
            pop   af                    ; get error flag
            or    a                     ; set flags on A
            ret   m                     ; return if failure
            inc   hl                    ; return nonzero to indicate success
            ret
;
;
;***********************************************************************
;*                                                                     *
;*  FINDFILE - open a stream to the file specified on the command line *
;*                                                                     *
;***********************************************************************
;
            if    chef eq false
            align 2
;
; Routine  to  create an I/O stream to the file specified on the command
; line.  The FCB for this is left at 05ch by CP/M.
;
; Parameters to the routine:
;
;    HL  =  0     means open the file for output
;    HL  <> 0     means open the file for input
;
; If  no  file is found, zero is returned, otherwise a pointer to a BCPL
; stream control block.
;
findfile:   push  hl                    ; save return address
            ld    hl,(scbbase)          ; point to first SCB
            ld    a,(hl)                ; get status byte
            and   usebit                ; see if already open
            jr    nz,fferror            ; j if so
;
            inc   hl                    ; point to FCB + 1
            inc   hl
            inc   hl
            inc   hl
            ld    a,(hl)                ; first byte of filename
            cp    space
            jr    z,fferror             ; j if invalid
            dec   hl                    ; point back to FCB
            ld    de,wfcb               ; set up for copy
            ld    bc,fcbsize
            ldir                        ; copy to work area
;
            exx                         ; get entry parameters
            ld    a,h                   ; was it an output stream?
            or    l
            jr    nz,fftry              ; if not, skip delete and create
;
            exx                         ; back to call register set
            ld    de,wfcb
            ld    c,19                  ; delete file
            call  fdos                  ; ignore result
;
            ld    de,wfcb               ; address of work FCB
            ld    c,22                  ; create file
            call  fdos
;
            or    a                     ; directory full?
            jp    m,fferror             ; j if so
            xor   a                     ; A := 0, indicating output
            jr    fftry2
;
; Now try to open the file.
;
fftry:      ld    a,-1                  ; indicates input
fftry2:     push  af                    ; save direction indicator
            ld    de,wfcb
            xor   a                     ; A := 0
            ld    (wfcb+12),a           ; current extent
            ld    c,15                  ; open file
            call  fdos
;
            xor   0ffh                  ; was the open successful?
            jr    nz,ffopen             ; j if ok
            pop   af                    ; lose direction indicator
            jr    fferror
;
; If we drop through, then the file is open, ready for what  has  to  be
; done to it.
;
ffopen:     pop   af                    ; A := 0 for output, -1 for input
            ld    c,a                   ; initial random record number
            call  setfcb                ; set up FCB fields
            ld    hl,(scbbase)          ; point to SCB
            push  hl                    ; save copy
            inc   hl                    ; point to character pointer
            ld    a,c                   ; direction indicator to C
            ld    (hl),a
            inc   hl                    ; point to 'last char' field
            xor   a                     ; A := 0
            ld    (hl),a
            inc   hl                    ; point to FCB in SCB
            ex    de,hl                 ; swap for move
            ld    hl,wfcb
            ld    bc,fcbsize
            ldir                        ; copy work FCB to real one
;
            exx                         ; get BCPL parameters
            ex    de,hl                 ; move argument 1 to DE
;
            pop   hl                    ; restore SCB pointer
            ld    a,d                   ; high byte of param
            or    e                     ; sets Z if DE = 0
            jr    z,ffout               ; j if for output
;
            ld    a,usebit!rdbit
            ld    (hl),a                ; file open for input
            ret                         ; HL has SCB pointer
;
ffout:      ld    a,usebit!wrbit
            ld    (hl),a                ; file open for output
ffret:      ret                         ; HL has SCB pointer
;
; Error in opening file
;
fferror:    jp    retzero
            endif
;
;-----------------------------------------------------------------------
;
; Set up sundry fields in the FCB. C contains -1 if the FCB  is  for  an
; input stream, and 0 if it is for an output stream.
;
setfcb:     xor   a                     ; A := 0
            ld    hl,wfcb+32
            ld    (hl),a                ; current record
            inc   hl
            ld    (hl),c                ; low byte of random record number
            inc   hl
            ld    (hl),c                ; high byte of random record number
            inc   hl
            ld    (hl),a                ; third byte always zero
            ret
;
;
;***********************************************************************
;*                                                                     *
;*          FINDINPUT - return a descriptor to an input stream         *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; Returns  a  descriptor  to  a  BCPL  stream control block for the file
; specified as a string argument to the function.  The call fails if the
; file is not found, or there are no SCBs  left  to  allocate;  in  this
; case, zero is returned.
;
findin:     push  hl                    ; save return address
            exx                         ; get string filename
            call  format                ; formats the name to 'wfcb'
;
            ld    a,(wfcb+1)
            cp    space
            jr    z,fierr2              ; j if invalid name
;
fi1:        xor   a                     ; A := 0
            push  af                    ; save flag indicating no '*'
            ld    a,(wfcb)              ; get drive name
            cp    '*                    ; is it 'wild'?
            jr    nz,notwild            ; j if not
            ld    a,1                   ; otherwise set to A: ...
            ld    (wfcb),a
            pop   af
            inc   a
            push  af                    ; ... and save flag indicating so
;
notwild:    ld    de,wfcb               ; address of work FCB
            xor   a                     ; A := 0
            ld    (wfcb+12),a           ; current extent number
            ld    c,15                  ; open file
            call  fdos
;
            xor   0ffh                  ; is return code -1?
            jr    nz,openok             ; j if not - file is now open
            pop   af                    ; get 'wild' flag
            push  af                    ; save it again
            or    a
            jr    z,fierr               ; j if not wild
            ld    a,(wfcb)              ; get drive name (A:)
            inc   a                     ; change to B:
            ld    (wfcb),a              ; store it away in FCB
            pop   af                    ; clear stack
            jr    fi1                   ; go to try again
;
openok:     pop   af                    ; clear stack
            call  findscb               ; find a free SCB
            ld    a,h                   ; HL = 0 means no SCB available
            or    l
            jr    z,fierr2              ; j if no SCBs left
;
            push  hl                    ; final result of 'findinput'
            ld    (hl),usebit!rdbit     ; file open for reading
            ld    a,-1                  ; initial character pointer
            inc   hl
            ld    (hl),a                ; current character field
            xor   a                     ; A := 0
            inc   hl
            ld    (hl),a                ; 'last char' field
            inc   hl                    ; point HL to FCB
;
            push  hl                    ; save FCB pointer
            ld    c,-1                  ; indicates input FCB
            call  setfcb                ; set up FCB fields
            pop   hl                    ; restore FCB pointer
;
            ld    de,wfcb               ; set up for move
            ld    bc,fcbsize
            ex    de,hl
            ldir                        ; move work FCB to real one
;
            pop   hl                    ; restore final result
            ret
;
fierr:      pop   af                    ; clear stack
fierr2:     jp    retzero
;
;
;***********************************************************************
;*                                                                     *
;*         FINDOUTPUT - return a descriptor to an output stream        *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; Returns a descriptor to a BCPL  stream  control  block  for  the  file
; specified as a string argument to the function.  The call fails if the
; file cannot be opened; in this case, zero is returned.
;
findout:    push  hl                    ; save return address
            exx                         ; get string filename
;
            call  format                ; format the string to 'wfcb'
;
            ld    a,(wfcb)              ; get drive name
            cp    '*                    ; is it 'wild'?
            jr    z,foerr               ; j if so - not allowed for output
            ld    a,(wfcb+1)
            cp    space
            jr    z,foerr               ; j if invalid name
;
            ld    de,wfcb
            ld    c,19                  ; delete file
            call  fdos                  ; ignore result
;
            xor   a                     ; A := 0
            ld    (wfcb+12),a           ; set current extent
            ld    de,wfcb
            ld    c,22                  ; make file
            call  fdos
;
            xor   0ffh                  ; is the return code -1?
            jr    z,foerr               ; j if so - failed to create file
;
            call  findscb               ; find a free SCB
            ld    a,h                   ; if HL = 0 then unsuccessful
            or    l
            ret   z                     ; return if no SCBs left
;
            push  hl                    ; final result of function
            ld    (hl),usebit!wrbit     ; file open for writing
            xor   a                     ; A := 0
            inc   hl
            ld    (hl),a                ; current character field
            inc   hl                    ; 'last char' field
            ld    (hl),a
            inc   hl                    ; point to FCB
;
            push  hl                    ; save FCB pointer
            ld    c,0                   ; indicates output FCB
            call  setfcb                ; set FCB fields
            pop   hl                    ; restore FCB pointer
;
            ld    de,wfcb               ; set up for move
            ld    bc,fcbsize
            ex    de,hl
            ldir                        ; move work FCB to real one
;
            pop   hl                    ; final result
            ret
;
foerr:      jp    retzero
;
;-----------------------------------------------------------------------
;
; Format  a  BCPL  string  into a file control block.  On entry, HL is a
; BCPL pointer to the string.
; The file name is of the form:
;
;    [ <drivename>: ] <filename> [ .<extension> ]
;
format:     ld    b,fcbsize
            ld    de,wfcb               ; clear the work FCB
            ld    a,space
;
form1:      ld    (de),a
            inc   de                    ; set all of FCB to spaces
            djnz  form1
;
            add   hl,hl                 ; get byte address of string
            ld    b,(hl)                ; get length
            inc   hl                    ; point to first text byte
;
            ld    a,b
            or    a                     ; is length zero?
            ret   z                     ; if a null string
;
            ld    a,(hl)                ; first character of filename...
            ex    af,af'                ; ...or drive name
            inc   hl
            dec   b
            jr    z,currdisc            ; j if one letter filename!
;
            ld    a,(hl)                ; second character, or ':'...
            cp    ':                    ; ...if drive name
            jr    nz,currdisc           ; assume on current disc
;
            ex    af,af'                ; get back drive name
            call  ucase                 ; ensure upper case
            cp    '*                    ; is it wild?
            jr    z,form2               ; j if so - store as that
            sub   'A-1                  ; get drive number
form2:      ld    (wfcb),a              ; store drive number in FCB
;
            inc   hl
            dec   b
            ret   z                     ; A: (etc) not a legal filename
;
            ld    a,(hl)                ; get first character of filename
            inc   hl
            dec   b
            jr    fname
;
currdisc:   xor   a                     ; A := 0 - means current disc
            ld    (wfcb),a
            ex    af,af'                ; restore first character
;
fname:      ld    de,wfname
            call  ucase                 ; ensure upper case
            ld    (de),a                ; store first character
            inc   de
            ld    a,b
            or    a                     ; any more characters left?
;
fnloop:     ret   z                     ; return if end of string
            ld    a,(hl)                ; next character
            cp    '.                    ; start of extension?
            jr    z,fext                ; j if so
;
            inc   hl
            call  ucase                 ; ensure upper case
            ld    (de),a                ; store the character
            inc   de
            dec   b
            jr    fnloop                ; loop until end
;
fext:       ld    de,wfext              ; point to FCB extension area
            inc   hl
            dec   b
;
feloop:     ret   z                     ; return if end of string
            ld    a,(hl)                ; get next character
            inc   hl
            call  ucase                 ; ensure upper case
            ld    (de),a                ; store it in FCB
            inc   de
            dec   b
            jr    feloop                ; loop until end
;
;-----------------------------------------------------------------------
;
; Convert the character in A to upper case.
;
ucase:      cp    'a
            ret   m                     ; return if out of range
            cp    'z+1
            ret   p                     ; return if out of range
            sub   'a-'A                 ; convert it
            ret
;
;-----------------------------------------------------------------------
;
; Find a free SCB for a particular stream.
;
findscb:    ld    b,nscbs               ; loop count
            ld    a,b
            or    a                     ; is number of SCBs zero?
            jp    z,retzero
;
            ld    de,-scbsize
            ld    hl,(scbtop)           ; top of SCBs
;
fsloop:     add   hl,de
            ld    a,(hl)                ; get status byte
            and   usebit                ; is buffer in use?
            ret   z                     ; if not, return with answer
            djnz  fsloop                ; else continue looking
;
;
;***********************************************************************
;*                                                                     *
;*           SELECTINPUT - select a new current input stream           *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; Selects  as  the current  input  stream  the  item  described  by  the
; parameter.  This will either be a file descriptor (the address  of  an
; SCB), or a symbolic descriptor such as the terminal.
;
slcti:      push  hl                    ; save return address
            exx                         ; get BCPL parameter
            ex    de,hl                 ; DE := descriptor
;
; Now test if this is a valid input stream descriptor.
;
            ld    hl,terminal           ; simplest case
            and   a                     ; clear carry
            sbc   hl,de                 ; sets Z if equal
            jr    z,siok                ; j if correct
;
; Look in the SCB area for an SCB matching the parameter.
;
            if    nscbs+1 ne 0
            ld    b,nscbs+1
            ld    hl,(scbtop)           ; top of SCBs
            ld    a,b
            or    a                     ; is number of SCBs zero?
;
siloop:     jr    z,selierr             ; no more SCBs left to try
            push  bc                    ; save loop count
            ld    bc,-scbsize           ; decrement HL
            add   hl,bc
            pop   bc                    ; restore loop count
;
            push  hl
            and   a                     ; clear carry
            sbc   hl,de                 ; is this the one?
            pop   hl
            jr    nz,sinext             ; j if not
            ld    a,(de)
            and   usebit!rdbit          ; get status byte
            cp    usebit!rdbit
            jr    z,siok                ; j if it is input stream too
;
sinext:     dec   b
            jr    siloop                ; loop until end
            else
            jr    selierr
            endif
;
siok:       ld    (cis),de              ; new input stream
            ret
;
selierr:    ld    hl,0                  ; null stream control block
            ld    (cis),hl
            ret
;
;***********************************************************************
;*                                                                     *
;*          SELECTOUTPUT - select a new current input stream           *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; Selects  as  the current output  stream  the  item  described  by  the
; parameter.  This will either be a file descriptor (the address  of  an
; SCB), or a symbolic descriptor such as the terminal.
;
slcto:      push  hl                    ; save return address
            exx                         ; get BCPL parameter
            ex    de,hl                 ; DE := descriptor
;
; Now test if this is a valid output stream descriptor.
;
            ld    hl,terminal           ; simplest case
            and   a                     ; clear carry
            sbc   hl,de                 ; sets Z if equal
            jr    z,sook                ; j if correct
;
; Look in the SCB area for an SCB matching the parameter.
;
            if    nscbs+1 ne 0
            ld    b,nscbs+1
            ld    hl,(scbtop)           ; base of SCBs
            ld    a,b
            or    a                     ; is number of SCBs zero?
;
soloop:     jr    z,seloerr             ; no more SCBs left to try
            push  bc                    ; save loop count
            ld    bc,-scbsize           ; decrement HL
            add   hl,bc
            pop   bc                    ; restore loop count
;
            push  hl
            and   a                     ; clear carry
            sbc   hl,de                 ; is this the one?
            pop   hl
            jr    nz,sonext             ; j if not
            ld    a,(de)
            and   usebit!wrbit          ; get status byte
            cp    usebit!wrbit
            jr    z,sook                ; j if it is output stream too
;
sonext:     dec   b
            jr    soloop                ; loop until end
            else
            jr    seloerr
            endif
;
sook:       ld    (cos),de              ; new output stream
            ret
;
seloerr:    ld    hl,0                  ; null stream control block
            ld    (cos),hl
            ret
;
;
;***********************************************************************
;*                                                                     *
;*            UNRDCH - backspaces the CIS - only works once            *
;*                                                                     *
;***********************************************************************
;
            if    chef eq false
            align 2
;
; Sets a flag in the stream control block, causing the next call of RDCH
; to return the last character sent.
;
unrdch:     push  hl                    ; save return address
            ld    de,(cis)
            ld    a,d
            or    e
            jp    z,retzero
;
            ld    hl,terminal
            sbc   hl,de                 ; sets Z if equal
            ret   z                     ; return if terminal - can't backspace
;
            ld    a,(de)
            or    unrbit
            ld    (de),a                ; set the flag
            ret
            endif
;
;
;***********************************************************************
;*                                                                     *
;*                 DELETEFILE - delete a specified file                *
;*                                                                     *
;***********************************************************************
;
            align 2
;
deletefi:   push  hl                    ; save return address
            exx                         ; get string filename
            call  format                ; format the string to 'wfcb'
;
            ld    a,(wfcb+1)
            cp    space
            jp    z,retzero             ; j if invalid filename
;
            ld    de,wfcb
            ld    c,19                  ; delete file
            call  fdos
            xor   -1                    ; convert -1 result to 0
            ld    l,a
            ld    h,0
            ret
;
;
;***********************************************************************
;*
;*                 RENAMEFILE - rename a specified file
;*
;***********************************************************************
;
            if    chef eq false
            align 2
;
renamefi:   push  hl                    ; save return address
            exx                         ; get BCPL parameters
            push  hl                    ; save pointer to old filename
;
            ex    de,hl                 ; get pointer to new filename in HL
            call  format                ; format new filename to 'wfcb'
            ld    hl,wfcb               ; set up to copy name out of the way
            ld    de,workname
            ld    bc,16                 ; length of filename
            ldir                        ; do the copy
;
            pop   hl                    ; get pointer to old filename again
            call  format                ; format old filename to 'wfcb'
            ld    hl,workname           ; set up to copy new name back
            ld    de,wfcb+16
            ld    bc,16                 ; length of filename
            ldir                        ; copy in new filename to FCB
;
            ld    de,wfcb
            ld    c,23                  ; rename file
            call  fdos
            xor   0ffh                  ; convert error result to zero
            ld    l,a
            ld    h,a
            ret
            endif
;
;
;***********************************************************************
;*                                                                     *
;*         GETBYTE and PUTBYTE - old versions of the % operator        *
;*                                                                     *
;***********************************************************************
;
            if    chef eq false
            align 2
;
; 'getbyte(a,b)'  returns  the  byte at offset 'b' from the BCPL pointer
; 'a'.
;
getbyte:    push  hl                    ; save return address
            exx                         ; get BCPL parameters
            add   hl,hl                 ; form byte address
            add   hl,de                 ; add the offset
            ld    l,(hl)                ; get the required byte
            ld    h,0
            ret
;
            align 2
;
; 'putbyte(a,b,c)  puts the byte 'c' at offset 'b' from the BCPL pointer
; 'a'.
;
putbyte:    push  hl                    ; save return address
            exx                         ; get BCPL parameters
            add   hl,hl                 ; form byte address
            add   hl,de
            ld    (hl),c                ; store the byte
            ret
            endif
;
;
;***********************************************************************
;*                                                                     *
;*                     MOVE - move a block of bytes                    *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; 'move(len,from,to)' copies 'len' bytes from the byte address 'from' to
; the byte address 'to'.
;
move:       push  hl                    ; save return address
            exx                         ; get BCPL parameters
            ex    de,hl                 ; move source address to HL
            push  de
            push  bc
            pop   de                    ; move destination address to DE
            pop   bc                    ; byte count
            ld    a,b                   ; zero bytes to do?
            or    c
            ret   z                     ; return if so
            ldir                        ; else do the move
            ret
;
;
;***********************************************************************
;*                                                                     *
;*          SEEK - position an input file at a specified byte          *
;*                                                                     *
;***********************************************************************
;
            align 2
;
seek:       push  hl                    ; save return address
            exx                         ; get BCPL parameters
            ex    de,hl                 ; DE := descriptor
            push  hl                    ; save required position
;
; Test if this is a valid file stream.
;
            ld    hl,terminal
            and   a                     ; clear carry
            sbc   hl,de                 ; sets Z if equal
            jr    z,seekerr             ; j if terminal - not allowed
;
; Look in the input SCB area for an SCB matching the parameter.
;
            if    nscbs+1 ne 0
            ld    b,nscbs+1
            ld    hl,(scbtop)           ; top of buffer area
            ld    a,b
            or    a                     ; number of buffers zero?
;
skloop:     jr    z,seekerr             ; no more buffers left to try
            push  bc                    ; save loop count
            ld    bc,-scbsize           ; decrement HL
            add   hl,bc
            pop   bc                    ; restore loop count
;
            push  hl                    ; save SCB pointer
            and   a                     ; clear carry
            sbc   hl,de                 ; is this the one?
            pop   hl
            jr    nz,sknext             ; j if not
            ld    a,(de)                ; get status byte
            xor   usebit!rdbit          ; is SCB in use for input?
            jr    z,seekok              ; j if so
;
sknext:     dec   b
            jr    skloop                ; loop until end
;
; DE points to the stream control block.
;
seekok:     pop   hl                    ; required file position
            push  de                    ; save SCB pointer
            ld    a,l
            and   07fh                  ; get byte offset in block
            push  af                    ; save for later
            push  hl                    ; shift the file position...
            ld    hl,7                  ; ...7 places to the right...
            push  hl                    ; ...to get the number...
            call  $rsh                  ; ...of the file block
            pop   bc                    ; get block number
;
            pop   af                    ; get byte offset
            pop   hl                    ; get SCB pointer
            push  af                    ; save byte offset
            push  hl                    ; save SCB pointer
            ld    de,36                 ; offset of record number in SCB
            add   hl,de                 ; point to record number
;
            ld    e,(hl)                ; get current record number
            inc   hl
            ld    d,(hl)
            dec   hl                    ; point back to low byte
            push  hl                    ; save pointer
            push  bc                    ; copy new record number...
            pop   hl                    ; ...to HL
            and   a                     ; clear carry
            sbc   hl,de                 ; old and new numbers the same?
            pop   hl                    ; restore pointer to record number in SCB
            jr    z,seek5               ; j if they are the same block
            ld    (hl),c                ; set low byte
            inc   hl
            ld    (hl),b                ; set high byte
            inc   hl
            inc   hl                    ; point to buffer
            ex    de,hl                 ; to DE for FDOS
            ld    c,26
            call  fdos                  ; set DMA address
;
            pop   de                    ; get SCB pointer
seek4:      push  de                    ; save it again
            inc   de
            inc   de
            inc   de                    ; point to FCB
            ld    c,33                  ; read random
            call  fdos                  ; read random
            or    a                     ; test result
            jp    m,skerr2              ; j if failed
seek5:      pop   hl                    ; point to SCB character pointer
            inc   hl
            pop   af                    ; byte offset in buffer
            ld    (hl),a                ; set pointer
            ret
;
skerr2:     pop   hl
            endif
;
seekerr:    pop   hl                    ; clear stack
            jp    retzero
;
;
;***********************************************************************
;*                                                                     *
;*       INPUT and OUTPUT - return descriptors to the CIS and COS      *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; Return the descriptor for the currently selected input stream.
;
input:      push  hl                    ; save return address
            ld    hl,(cis)
            ret
;
;-----------------------------------------------------------------------
;
            align 2
;
; Return the descriptor for the currently selected output stream.
;
output:     push  hl                    ; save return address
            ld    hl,(cos)
            ret
;
;
;***********************************************************************
;*                                                                     *
;*                 TESTFLAGS - test various conditions                 *
;*                                                                     *
;***********************************************************************
;
            if    chef eq false
            align 2
;
; Bit 2**0 in the parameter asks if there is terminal input available.
;
testflags:  push  hl                    ; save return address
            exx                         ; get BCPL parameter
            dec   hl
            jr    nz,rfalse             ; j if parameter not 1
;
            ld    c,11                  ; get console status
            call  fdos
            xor   0ffh                  ; A := 0 if character ready
            jr    nz,rfalse             ; j if nothing there
            ld    hl,true
            ret
;
rfalse:     ld    hl,false
            ret
            endif
;
;
;***********************************************************************
;*                                                                     *
;*               LEVEL and LONGJUMP - for non-local jumps              *
;*                                                                     *
;***********************************************************************
;
            align 2
;
; Return the stack pointer for the current activation level.  This value
; is used mainly by 'longjump'.
;
level:      push  hl                    ; save return address
            ex    de,hl                 ; DE held old stack pointer
            ret
;
;-----------------------------------------------------------------------
;
            align 2
;
; 'longjump(p,l)' jumps to the label 'l' at activation level 'p'.
;
longjump:   exx                         ; get BCPL parameters
            ld    sp,(scbbase)          ; reset the Z80 stack
            push  hl                    ; new BCPL stack pointer
            pop   ix                    ; set it up
            push  de                    ; new starting address
            ret                         ; jump to it
;
;
;***********************************************************************
;*                                                                     *
;*         APTOVEC - apply a variable sized vector to a routine        *
;*                                                                     *
;***********************************************************************
;
; Causes a routine application of  its  first  argument,  setting  up  a
; variable length vector to be passed as a parameter to this routine.
;
            if    chef eq false
            align 2
;
aptovec:    push  hl                    ; save return address
            push  bc                    ; save stack size increase
            exx                         ; get BCPL parameters
            pop   bc                    ; BC = increase in stack size
            push  hl                    ; address of routine to be applied
            push  ix                    ; current stack pointer
            pop   hl                    ; first argument to routine
            add   hl,bc                 ; past current stack frame
            ld    bc,-128
            add   hl,bc                 ; (IX points 128 bytes up frame)
            srl   h
            rr    l                     ; form BCPL address
            push  de                    ; vector size
            exx                         ; now apply routine
            pop   hl                    ; vector size
            add   hl,hl
            add   hl,bc
            inc   hl
            inc   hl                    ; HL = vector size + stack increase (bytes)
            push  hl
            pop   bc                    ; vector size
            pop   hl                    ; vector address
            jp    $apply                ; and apply routine
            endif
;
;
;***********************************************************************
;*
;*                          Global definitions
;*
;***********************************************************************
;
            align 2
;
            defw  2*99,0                ; maximum global referenced
            defw  2*23,wrch
            defw  2*22,rdch
            if    blkio eq true
            defw  2*26,rdblock
            defw  2*27,wrblock
            endif
            defw  2*11,findout
            defw  2*10,findin
            defw  2*24,input
            defw  2*25,output
            defw  2*51,level
            defw  2*52,longjump
            defw  2*14,slcti
            defw  2*15,slcto
            defw  2*4,stop
            defw  2*18,endread
            defw  2*19,endwrite
            defw  2*41,deletefi
            defw  2*42,seek
            defw  2*82,move
            if    chef eq false
            defw  2*9,findfile
            defw  2*29,unrdch
            defw  2*63,binrdch
            defw  2*64,binwrch
            defw  2*40,renamefi
            defw  2*38,testflags
            defw  2*58,getbyte
            defw  2*59,putbyte
            defw  2*53,aptovec
            endif
;
endmain:    equ   $
;
            end
