; cluster.lsp               Gordon S. Novak Jr.              ; 05 Apr 07

; Copyright (c) 2007 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Functions to make clusters for data structures built on top of user types,
; e.g. a list, tree, or array of user records.

; 17 Nov 94; 05 Jan 95; 18 Jan 96; 14 Nov 96; 28 Apr 98; 03 Jan 00; 22 Oct 02


; 28 Nov 89; 29 Mar 90; 02 Apr 90
(defun gldefclusterspec (l)
  (setf (glclusterspec (car l)) (cdr l))
  (setf (glclusterinst (car l)) nil) )

; 28 Nov 89; 26 Dec 89; 28 Mar 90; 29 Mar 90; 30 Mar 90; 03 Apr 90; 04 Oct 93
; 20 Oct 93; 18 Jan 96; 05 Apr 07
; Specialize a cluster of types for specified type parameters.
; Example: (glspecializecluster 'xlistof '(integer))
; 04 Oct 93 added ability to put parameters at end of cluster definition,
;           e.g. (arrayof emprec filled-size nemployees)
(defun glspecializecluster (cluster argtypes)
  (prog (clusterdef prefix typesubs newtype newtypes newargtypes newarg
		arg newcluster tmp lst newpairs params str props)
    (unless (setq clusterdef (glclusterspec cluster))
	    (error "Undefined cluster spec ~A" cluster))

; commented out since parameters would change length    04 Oct 93
;    (unless (= (length argtypes) (length (first clusterdef)))
;	    (error "Wrong number of args"))

; Make a prefix for the types to be created.

    (setq prefix (symbol-name cluster))
    (dotimes (i (length (first clusterdef)))
      (setq arg (nth i argtypes))
      (setq newarg (if (or (symbolp arg)
			   (consp arg))
		       (glxtrtype arg)
		       arg))
      (setq newargtypes (nconc1 newargtypes newarg))
      (if (symbolp newarg)
	  (setq prefix (concat prefix "-" (symbol-name newarg)))))
    (setq newcluster (intern prefix))

; Make a list of type substitutions for argument types

    (mapc #'(lambda (absttype argtype)
              (if (or (gldescendantp argtype (second absttype))
                      (gltypematch argtype (second absttype)))
		  (push (cons (car absttype) argtype) typesubs)
		  (error "Specified type ~A does not match ~A"
			 argtype absttype)))
	  (first clusterdef)
	  newargtypes)

; Make type substitutions for cluster types

    (dolist (typedes (second clusterdef))
	    (setq newtype (intern (concat prefix "-"
					  (symbol-name (car typedes)))))
	    (push newtype newtypes)
            (setq newpairs (nconc1 newpairs
				   (list (car typedes) newtype)))
	    (push (cons (car typedes) (car newtypes))
		  typesubs))
    (setq lst (list 'roles newpairs))
    (if (third clusterdef) (nconc lst (list 'supers (third clusterdef))))
    (setf (glclusterdef newcluster) lst)	

    (setq params (nthcdr (length (first clusterdef)) argtypes))
 ; (format t "argtypes = ~A   params = ~A~%" argtypes params) ; debug
    (dolist (typedes (second clusterdef))
      (setq str (copy-tree (sublis typesubs typedes)))
      (setq props (cddr str))
      (while props
	(if (member (car props) '(prop adj isa msg))
	    (setf (cadr props)
		  (mapcan #'(lambda (x)
			      (if (symbolp x)
				  (if (setq tmp (getf params x))
				      (list (list x (list tmp))))
				  (list x)))
			  (cadr props))))
	(setq props (cddr props)))
      (gldefstr str nil))

    (dolist (pair newpairs)
	    (setf (glcluster (cadr pair)) newcluster))

; put in superclass links from types to corresponding super-cluster types
    (dolist (tp typesubs)
      (dolist (super (third clusterdef))
	(if (setq tmp (glclusterrole super (car tp)))
	    (gladdsuper (cdr tp) tmp)) ) )
    (dolist (s (third clusterdef))
	    (pushnew newcluster (glsubclusters s)))
    (return newcluster)
  ))


; 28 Nov 89; 28 Mar 90; 02 Apr 90
; Find the cluster with the specified argtypes.  If not defined, make one.
(defun glfindcluster (clusterspec)
  (let ((clustername (first clusterspec))
	(argtypes (rest clusterspec)) pair)
    (when (glclusterspec clustername)
	  (dolist (c (glclusterinst clustername))
		  (if (equal (first c) argtypes) (setq pair c)))
	  (unless pair
		  (setq pair
			(list argtypes
			      (glspecializecluster clustername argtypes)))
		  (push pair (glclusterinst clustername)) )
	  (second pair)) ))


; 02 Apr 90; 18 Jan 96
; Find the "top" type for a cluster with the specified argtypes.
; If not defined, make one.
(defun glfindclustertype (clusterspec)
  (let (clustername)
    (if (setq clustername (or (and (symbolp clusterspec)
				   (glclusterdef clusterspec)
				   clusterspec)
			      (glfindcluster clusterspec)))
	(cadar (glclusterroles clustername)) ) ))
    

    
; 28 Nov 89; 28 Dec 89; 28 Mar 90; 29 Mar 90
; Delete a cluster spec due to a change
(defun glcchg (clusterspec)
  (let ((clustername (first clusterspec))
	(argtypes (rest clusterspec)) pair)
    (when (glclusterspec clustername)
	  (dolist (c (glclusterinst clustername))
		  (if (equal (first c) argtypes) (setq pair c)))
	  (when pair
		(setf (glclusterinst clustername)
		      (delete pair (glclusterinst clustername)))
		(dolist (s (second pair))
		  (setf (glstructure s) nil)
		  (setf (glcluster s) nil) ) ) ) ))


; 22 Mar 90; 29 Mar 90; 08 Nov 90; 17 Nov 94; 18 Jan 96; 28 Apr 98
; Experimental cluster definition function

; Define a cluster data structure:
;    (gldefclusterc 'cluster-name '(cluster-types) '(cluster-supers) ...)
;       where each cluster-type is (role-name (glisp-type-spec))
;       and ... is <parameter-type> (<values>) ...
(defun gldefclusterc (cluster-name cluster-types cluster-supers &rest stuff)
  (let (lst tmp supers)
    (setq lst (list 'roles (mapcar #'(lambda (l) (list (car l) (caadr l)))
				   cluster-types)))
    (if cluster-supers (nconc lst (list 'supers cluster-supers)))
    (nconc lst stuff)
    (setf (glclusterdef cluster-name) lst)	
; define glisp types of components when specified, pointer from type to cluster
    (dolist (tp cluster-types)
; add superclass pointers between component types and super-cluster types
      (setq supers (getf (cddadr tp) 'supers))
      (dolist (super cluster-supers)
	(if (setq tmp (glclusterrole super (car tp)))
	    (setq supers (nconc supers (list tmp)))))
      (when (caadr tp)
	(when (cdadr tp)
	  (setf (getf (cddadr tp) 'supers) supers)
	  (gldefstr (cadr tp) nil))
	(setf (glcluster (caadr tp)) cluster-name)) )
; put in subcluster pointers from super-clusters
    (dolist (s cluster-supers)
	    (pushnew cluster-name (glsubclusters s)))
    cluster-name ))

; 29 Mar 90; 24 Apr 90; 09 Jul 90; 17 Jul 90; 20 Jul 90
; Add a superclass to a structure description
(defun gladdsuper (strname super)
  (let (pl)
    (if (and (symbolp strname)
	     (setq pl (glstr strname)))
	(unless (member super (getf (cdr pl) 'supers))
	  (setf (getf (cdr pl) 'supers)
		(nconc1 (getf (cdr pl) 'supers) super)))
	(error " ~S  has no structure definition." strname) ) ))

; 12 Nov 96
; Find the type that fills <role> in the cluster of <type>.
(defun clustertype (type role)
  (glclusterrole (glcluster type) role))

(setf (glfnresulttype 'clustertype) 'gltype)
