; conn.lsp            Gordon S. Novak Jr.           ; 09 Mar 10

; Copyright (c) 2010 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Interactive graphical connections
; initially derived from vip.lsp, which was derived from makev.lsp

; 28 Jul 96; 23 Oct 97; 05 Feb 98; 17 Mar 98; 18 Mar 98; 24 Mar 98; 29 May 98
; 03 Jun 98; 22 Dec 98, 30 Dec 98; 07 Jan 99; 23 Mar 99; 02 Apr 99; 22 Apr 99
; 20 Sep 01; 28 Feb 02; 30 May 02; 31 Dec 03; 05 Jan 04; 16 Feb 04; 21 May 04
; 01 Jun 04; 03 Jun 04; 20 Jun 04; 05 Aug 04; 30 Jun 06; 14 Sep 06; 28 Sep 06
; 29 Sep 06; 14 Nov 06; 22 Nov 06; 30 Oct 06; 13 Nov 06; 06 Dec 07; 13 Dec 07
; 24 Dec 07; 26 Dec 07; 27 Dec 07; 31 Dec 07; 08 Jan 08; 09 Jan 08; 11 Jan 08
; 15 Jan 08; 17 Jan 08; 22 Jan 08; 29 Feb 08; 18 Mar 08; 20 Mar 08; 03 Apr 08
; 04 Apr 08; 08 Apr 08; 10 Apr 08; 30 May 08; 19 Dec 08; 22 Dec 08; 31 Dec 08
; 01 Jan 09; 23 Oct 09; 18 Feb 10

; Files needed: dwindow.lsp vector.lsp menu-set.lsp

; Used in:
;   dag
;   makev
;   vip
;   ccm (example for Ben)

; Notes:
; conn-prop doesn't do much; should allow select from all props

(defvar *conn-constants*          nil)  ; menu tree of physical constants
(defvar *conn-laws*               nil)  ; menu tree of physical laws
(defvar *conn-programs*           nil)  ; menu tree of programs
(defvar *conn-program-actions*    nil)  ; alist of actions from program menu
(defvar *conn-trace*              nil)  ; set to t for in-process traces
(defvar *conn-user-laws*          nil)  ; user-defined law names
(defvar *conn-message-area-width* 0)    ; width of area for typein
(defvar *conn-default-args*       '(in in2 in3 in4 in5 in6 in7))

(glispobjects

  (conn-group (list (sources      (listof glnametype))
		    (goals        (listof glnametype))
		    (mconns       menu-conns)
		    (boxes        (listof conn-box))
		    (connpfn      anything)        ; connection predicate
		    (connfn       anything)        ; connection function
		    (source-names (listof symbol))
		    (letvars      (listof glnametype))
		    (windowsize   vector)
                    (verticalp    boolean))
     prop ((window        ((window (menu-set (mconns self)))))
	   (menu-set      ((menu-set (mconns self))))
	   (command-menu  ((menu (named-item (menu-set (mconns self))
					     'command))) )
	   (message-area-x ( (max 0 ((base-x (command-menu self))
			              - *conn-message-area-width*) / 2)))
	   (message-area-y ((max 0 (height (window self)) - 40))) )
     msg  ((redraw        conn-group-redraw)
	   (message       conn-group-message)
	   (erase-message conn-group-erase-message)
	   (get-input     conn-group-get-input)
	   (sourcep       conn-group-sourcep)
	   (goalp         conn-group-goalp)
           (box           (glambda (self nm) (that (boxes self)
                                                   with boxname == nm)))  ) )

  (conn-box (list (boxname   symbol)         ; a unique name
		  (boxtype   symbol)         ; e.g. variable, law, constant
		  (name      symbol)         ; e.g. gravitation
		  (type      gltype)
		  (direction symbol)         ; e.g. input, output
		  (portvals  (listof conn-port))
		  (equations equation-set))
   prop ((bnm     ((if (varp self) (name self) (boxname self)))) )
    msg ((port    (glambda (self nm) (that portvals with portname == nm))))
    adj ((fnp     ((member (boxtype self) '(op law fn prop adj msg str))) )
         (opp     ((eq (boxtype self) 'op)))
         (varp    ((eq (boxtype self) 'variable)))
         (constp  ((eq (boxtype self) 'constant)))
         (datap   ((member (boxtype self) '(variable constant))) ) ) )

  (conn-port (list (portname symbol)
		   (portcode anything)
 		   (type     gltype)
		   (filled   boolean)        ; T if port has a value
                   (direction symbol) )      ; e.g. input, output
    adj ((inputp  ((eq (direction self) 'input)))
         (outputp ((eq (direction self) 'output))) ) )

  (conn-save-group (list (name symbol)
                         (sources      (listof glnametype))
                         (goals        (listof glnametype))
                         (connections  (listof menu-set-conn))
                         (boxes        (listof conn-save-box))
                         (connpfn      anything)          ; connection function
                         (letvars      (listof glnametype))
                         (windowsize   vector)) )

  (conn-save-box (list (boxname   symbol)         ; a unique name
                       (boxtype   symbol)         ; e.g. variable, law, constant
                       (name      symbol)         ; e.g. gravitation
                       (type      gltype)
                       (direction symbol)         ; e.g. input, output
                       (value     anything)       ; for constants
                       (offset    vector) ) )


  ) ; glispobjects

(glispglobals (*conn-message-area-width* integer))

; 21 Nov 92; 23 Nov 92; 27 Nov 92; 30 Nov 92; 01 Dec 92; 29 Apr 93; 30 Apr 93
; 18 Aug 93; 14 Sep 93; 16 Sep 93; 06 Jan 94; 07 Jan 94; 20 Jan 94; 24 Jan 94
; 27 Jan 94; 17 Feb 94; 25 Nov 94; 01 Dec 94; 18 Mar 98; 23 Mar 99; 30 May 02
; 31 Dec 03; 18 Mar 08; 18 Feb 10
; Make a display for editing a set of connections between specified
; sets of sources and goals.
(gldefun conn-init ((sources (listof glnametype))
		    (goals (listof glnametype)) (w window)
		    (sourceprops symbol) (goalprops symbol)
		    (menuitems (listof anything)) (connpfn anything))
  (result conn-group)
  (let (group top (vspace 20) menu sizev command-menu)
    (group = (a conn-group with sources = sources  goals = goals
		  mconns = (a menu-conns with
				   menu-set = (a menu-set with window = w))
		  connpfn = connpfn))
    (open w)
    (clear w)
    (sizev = (size w))
    (remove-items (mconns group))
    (command-menu =
	       (menu-create
		 (or menuitems
		     '(("Quit" . quit)
		       ("Done" . done)
		       ("Redo" . redo)
		       ("Redraw" . redraw)
		       ("Move" . move)
		       ("Expand" . expand)
                       ("Minimize" . minimize)
		       ("Delete" . delete)
		       ("Geometry" . geometry)
		       ("Physics" . physics)
		       ("User Law" . userlaw)
                       ("Function From" . conn-function-from)
		       ("Program" . program)
                       ("Prop" . prop)
		       ("Make Var" . makevar)
		       ("Specify Type" . specifytype)
		       ("OP" . op)
		       ("Type-in" . type-in)
		       ("Constant" . constant)))
		 "Commands" w 0 0 t t))
    (init command-menu)
    (add-item (mconns group) 'command nil command-menu)
    (adjust (menu-set group) 'command 'top nil 1)
    (adjust (menu-set group) 'command 'right nil 2)
    ((source-names group) = (list 'command))
 ; put source var menus into the window
    (top = (height w) - vspace)
    (for source in sources do
      (menu = (conn-menu group source sourceprops))
      (sizev = (size menu))
      (moveto-xy menu 10 (top - (y sizev)) )
      (conn-add-box group menu 'variable (name source) (type source) 'input
		    (list (a conn-port portname = (name source)
			               portcode = (name source)
				       type     = (type source)
				       filled   = T )))
      (top _- ((height menu) + vspace)) )
 ; put goal menus into the window
    (top = (height w) - vspace)
    (for goal in goals do
      (menu = (conn-menu group goal goalprops))
      (sizev = (size menu))
      (moveto-xy menu ((base-x command-menu) - (x sizev) - 10)
		      (top - (y sizev)) )
      (conn-add-box group menu 'variable (name goal) (type goal) 'output
		    (list (a conn-port portname = (name goal)
			               portcode = (name goal)
				       type     = (type goal))))
      (top _- ((height menu) + vspace)) )
    group ))

; 07 Jan 94; 10 Jan 94; 17 Jan 94; 20 Jan 94; 24 Jan 94; 24 Jul 95; 18 Mar 98
; 20 Sep 01; 05 Jan 04; 30 May 08
; Make a menu for a type.
; Props is: NIL   for name only
;           FN    to use function arguments
;           PROPS if computed properties are to be included
;           else  data fields only
(gldefun conn-menu ((group conn-group) (goal glnametype) (props symbol))
  (result menu)
  (let ((tmp (listof glnametype)) items m pm pmspec type)
    (type = (conn-type (type goal)))
    (if (and (symbolp type) (pmspec = (picmenu-spec type)))
	(pm = (picmenu-create-from-spec
	        (copy-tree pmspec)
		(unless (get type 'picmenu-nobox) (conn-menu-title goal))
		(window group) 0 0 t t
		(not (get type 'picmenu-nobox))))
        (if (and (symbolp type) props)
	    (progn
		 (tmp = (if (eq props 'fn)
			    (glarguments type)
			    (conn-getnames type (eq props 'props))))
	         (items = (if (and tmp (consp tmp)
				     (not (and (null (cdr tmp))
					       (consp (car tmp))
					       (eq (caar tmp) 'identity))))
			      (for pair in tmp collect (name pair))
			      (list (name goal))))
		 (m = (menu-create items
				     (if (or (cdr items)
					     (and type
						  (not (glbasictypep type))))
					 (conn-menu-title goal)
				         "")
				     (window group) 0 0 t t '9x15)))
	  (m = (menu-create (list (name goal)) ""
			    (window group) 0 0 t t '9x15)) ) )
    (or m pm) ))

; 20 Jan 94; 05 Jan 03
; Get type, eliminating units if present.
(gldefun conn-type (type)
  (if (and (consp type) (eq (car type) 'units))
      (cadr type)
      type) )

; 10 Jan 94; 18 Jan 94; 20 Jan 94; 17 Mar 95; 23 Mar 99
(gldefun conn-menu-title ((goal glnametype))
  (conn-string-limit
    (if (name goal)
        (if (and (type goal)
		 (not (and (symbolp (name goal)) (symbolp (type goal))
			   (> (length (symbol-name (name goal)))
			      (length (symbol-name (type goal))))
			   (string= (symbol-name (name goal))
				    (symbol-name (type goal))
				    :end1 (length (symbol-name (type goal)))))))
		 (concatenate 'string (symbol-name (name goal)) ":"
			      (princ-to-string (type goal)) )
	         (stringify (name goal)) )
        (princ-to-string (type goal)))
    20))

; 10 Jan 94; 05 Aug 04
(defun conn-string-limit (s max) (string-limit s max))

(defun string-limit (str max)
  (let ((s (stringify str)))
    (if  (> (length s) max)  (subseq s 0 max)  s)))

; 07 Jan 94; 16 Jan 94; 10 Feb 94; 17 Mar 94; 11 Jan 96; 22 Apr 99; 09 Jan 08
; Get names of data, properties, and views of type.
; props is T to include computed properties.
(setf (glfnresulttype 'conn-getnames) '(listof glnametype))
(defun conn-getnames (type props)
  (let (datanames propnames tmp)
    (setq datanames
	  (or (and (consp type) (eq (car type) '^)
		   (list (list '^. (cadr type))))
	      (gldatanames type)
	      (and (glbasictypep type) (list (list 'identity type)))
	      (and (symbolp type)
		   (fboundp type)
		   (vip-args type))
	      (if (and (symbolp type)
		       (get type 'equations))
		  (progn (setq props nil)
			 (mapcar #'(lambda (x) (list x nil))
				 (get type 'variables))))))
    (if props
	(setq propnames
	      (if (setq tmp (assoc type '((string ((length integer))))))
		  (cadr tmp)
		  (if (or (glbasictypep type) (consp type))
		      nil
		      (nconc (gevpropnames type 'prop t)
			     (glviewnames type)))) ) )
    (setq propnames (delete-if #'(lambda (l)
				   (member (if (consp l) (car l) l)
					   '(displayprops shortvalue)))
			       propnames))
    (nconc datanames propnames) ))

; 27 Nov 92; 30 Nov 92; 06 Jan 94; 07 Jan 94; 17 Jan 94; 20 Jan 94; 24 Jan 94
; 01 Dec 94
(gldefun conn-add-box ((group conn-group) (m menu) boxtype name type
					 direction portvals)
  (let (boxname)
    (boxname = (conn-uniquename name))
    (add-item (mconns group) boxname name m)
    ((source-names group) _+ boxname)
    ((boxes group) _+ (a conn-box with boxname   = boxname
			               boxtype   = boxtype
				       name      = name
				       type      = type
				       direction = direction
				       portvals  = portvals) )
    boxname))

; 24 Jan 94; 17 Mar 95; 05 Jan 04
(gldefun conn-uniquename ((name symbol))  (result symbol)
  (let ((s (stringify name)) n)
    (n = (length s))
    (if (digit-char-p (char s (1- n)))
        name
        (glmkatom name) ) ))

; 25 Nov 92; 27 Nov 92; 30 Nov 92; 07 Apr 93; 07 Jan 94; 09 Feb 94; 10 Feb 94
; 22 Dec 98; 05 Jan 04
; Make an operation box and add to window
(gldefun conn-make-op ((group conn-group))
  (result symbol)
  (let (op)
    (if (op = (conngetexprop group 'number))
        (conn-make-op-box group op 'op)) ))

; 22 Dec 98; 30 Dec 98; 22 Nov 06; 13 Dec 07; 09 Jan 09
; Find the number of arguments of a function
(gldefun conn-nargs ((fn symbol))
  (or (cadr (assoc fn '((sin 1) (cos 1) (tan 1) (sqrt 1) (atan 2) (exp 1)
			(log 1) (expt 2) (+ 2) (- 2) (* 2) (/ 2) (^ 2)
			(min 2) (max 2) (if 3) (string-downcase 1) (both 2))))
      (glarity fn)) )

; 10 Feb 94; 01 Dec 94; 17 Mar 95; 26 Sep 95; 22 Dec 98; 05 Jan 04
; 20 Jun 04; 03 Apr 08; 18 Dec 08
; Make an operation box and add to window
; optype = op, prop
(gldefun conn-make-op-box ((group conn-group) (op symbol) (optype symbol))
  (result symbol)
  (let (fnname m boxwidth boxheight nargs w boxname)
    (w = (window group))
    (fnname = (intern (concatenate 'string "CONN-DRAW-" (symbol-name op))))
    (boxwidth = (+ 25 (string-width w (stringify op))))
    (nargs = (or (conn-nargs op)
                 (if (member optype '(prop param)) 1 2)))
    (boxheight = (if (verticalp group)
                     24
                     (+ 24 (* 12 (max 0 (- nargs 2))))))
    (unless (and (fboundp fnname) (symbol-function fnname))
      (setf (symbol-function fnname)
	    `(lambda (w x y)
	       (conn-draw-op-box w x y ,(- boxwidth 4) ,boxheight ',op))))
    (m = (picmenu-create-from-spec
	     (list 'picmenu-spec boxwidth boxheight
                   (conn-args-pos boxwidth boxheight nargs (verticalp group))
		   t fnname '9x15)
	     nil w 0 0 t t nil))
    (menu-reposition m)
    (boxname = (conn-add-box group m optype op (conn-op-type op) nil nil))
    (draw m)
    boxname ))

; 13 Dec 07; 26 Dec 07; 31 Dec 07; 15 Jan 08; 29 Feb 08; 03 Apr 08; 04 Apr 08
; 18 Dec 08; 19 Dec 08; 23 Feb 10
; Make a function box and add to window
; fn      = function or prop name
; nm      = name to use when drawing the box
; optype  = op, prop
; restype = result type
(gldefun conn-make-fn-box ((group conn-group) (fn symbol) (nm symbol)
                           (optype symbol) (args (listof glnametype))
                           (restype gltype))
  (result symbol)
  (let (pmspec fnname boxtitle m boxwidth boxheight w boxname ports)
    (w = (window group))
    (unless (pmspec = (picmenu-spec fn))
      (or nm (nm = fn))
      (boxtitle = (string-limit (stringify (or nm fn)) 12))
      (boxwidth = (+ 25 (string-width w boxtitle)))
      (boxheight = (+ 24 (* 12 (max 0 (- (length args) 2)))))
      (fnname = (intern (concatenate 'string "CONN-DRAW-"
                                     (symbol-name (or nm fn)))))
      (unless (and (fboundp fnname) (symbol-function fnname))
        (setf (symbol-function fnname)
              `(lambda (w x y)
                 (conn-draw-op-box w x y ,(- boxwidth 4) ,boxheight
                                   ',boxtitle))))
      (pmspec = (list 'picmenu-spec boxwidth boxheight
                   (conn-args-pos boxwidth boxheight (length args)
                                  (verticalp group)
                                  (for arg in args collect
                                       (if (consp arg) (car arg) arg)))
                      t fnname '9x15) ) )
    (m = (picmenu-create-from-spec pmspec nil w 0 0 t t nil))
    (menu-reposition m)
    (ports = (cons (a conn-port  portname = 'out
                                 direction = 'output
                                 type = restype)
                   (for arg in args
                        collect (a conn-port
                                   portname = (name arg)
                                   direction = 'input
                                   type =     (type arg)))))
    (boxname = (conn-add-box group m optype fn      ; 23 Feb 10: was nm
                             (conn-xtrtype (or restype
                                               (glresulttype fn nil)))
                             nil ports))
    (draw m)
    boxname ))

; 18 Feb 10
; handle a function-from selection
(gldefun conn-function-from ((group conn-group))
  (let (newbox fnname)
    (newbox = (dag-function-from group))
    (if (and newbox
             (member (boxtype newbox) '(prop adj isa msg)) )
        ((boxtype newbox) = 'fn))
    newbox))

; 04 Apr 08
; extract a real type if fn result is virtual
(defun conn-xtrtype (type)
  (if (and (consp type)
           (eq (car type) 'virtual))
      (cadr type)
      type))


; 22 Dec 98; 30 Dec 98; 30 Oct 07; 13 Nov 07
; Make input button positions for an n-input function box
(defun conn-input-pos (n &optional vars)
  (let (res (y (+ -6 (* 12 n))))
    (or vars (setq vars *conn-default-args*))
    (if (<= n 1)
	'((in (2 12)))
        (dotimes (i n (nreverse res))
	  (push (list (pop vars) (list 2 y)) res)
	  (setq y (- y 12)) ) ) ))

; 13 Dec 07; 29 Feb 08
; Make input button positions for a function box
; n    = number of arg inputs
; vars = list of (name type)
(defun conn-input-pos-b (n vars)
  (let (var res (y (+ -6 (* 12 n))))
    (or vars (setq vars '((in anything) (in2 anything) (in3 anything)
                          (in4 anything) (in5 anything) (in6 anything)
                          (in7 anything)) ) )
    (dotimes (i n (nreverse res))
      (setq var (pop vars))
      (push (list (if (consp var) (car var) var) (list 2 y)) res)
      (setq y (- y 12)) ) ))

; 03 Apr 08
; Make input button positions for an n-input function box
(defun conn-args-pos (boxwidth boxheight n &optional vertical vars outvars)
  (or vars (setq vars *conn-default-args*))
  (or outvars (setq outvars '(out)))
  (if vertical
      (append (conn-button-pos-h boxwidth (- boxheight 2) n vars)
              (conn-button-pos-h boxwidth 2 (length outvars) outvars))
      (append (conn-button-pos-v boxheight 2 n vars)
              (conn-button-pos-v boxheight (- boxwidth 2)
                                 (length outvars) outvars)) ) )

; 03 Apr 08
; Make horizontal button positions for n buttons
(defun conn-button-pos-h (width y n names)
  (let (delta x res)
    (setq delta (round width n))
    (setq x (truncate delta 2))
    (dotimes (i n)
      (push (list (pop names) (list x y)) res)
      (incf x delta) )
    res))

; 03 Apr 08
; Make vertical button positions for n buttons
(defun conn-button-pos-v (height x n names)
  (let (delta y res)
    (setq delta (round height n))
    (setq y (- height (truncate delta 2)))
    (dotimes (i n)
      (push (list (pop names) (list x y)) res)
      (decf y delta) )
    res))

; 25 Nov 92; 07 Apr 93; 17 Mar 95; 22 Dec 98
(defun conn-draw-op-box (w x y boxwidth boxheight op)
  (window-draw-box-xy w (+ x 2) (+ y 0) boxwidth boxheight)
  (window-printat-xy w op (+ x 13) (+ y 5))
  (window-force-output w))

; 26 Sep 95
(defun conn-op-type (op)
  (case op ((> < >= <= = <> == != not) 'boolean) ) )

; 24 Nov 92; 27 Nov 92; 30 Nov 92; 02 Dec 92; 07 Jan 94; 18 Jan 94; 20 Jan 94
; 01 Dec 94; 05 Jan 04
; Make a box containing a constant and add to window
(gldefun conn-make-constant ((group conn-group))
  (result menu)
  (let (m const name)
    (const = (conn-getconst nil))
    (if (consp const)
	(progn (name = (cadr const))
	       (m = (menu-create (list name) nil (window group) 0 0 t t '9x15))
	       (menu-reposition m)
	       (conn-add-box group m 'constant name (cadar const) nil
			     (list (a conn-port portname = name
				                portcode = (caar const)
						type     = (cadar const)
						filled   = T)))
	       (draw m)
	       m) )))

; 24 Nov 92; 27 Nov 92; 30 Nov 92; 22 Mar 93; 08 Apr 93; 07 Jan 94; 18 Jan 94
; 20 Jan 94; 27 Jan 94; 07 Mar 94; 22 Mar 94; 01 Apr 94; 20 Apr 94; 01 Dec 94
; 05 Mar 96; 05 Jan 04
; Make a box containing a typed-in constant and add to window
(gldefun conn-typein-constant ((group conn-group))
  (result menu)
  (let (m const)
    (const = (get-input group "Enter constant: "))
    (if (and (consp const)
	     (numberp (car const))
	     (consp (cdr const))
	     (null (cddr const))
	     (glunitp (cadr const)))
	(const = (cons 'q const)))
    (m = (menu-create
	    (list (cons (if (and (consp const)
				 (eq (car const) 'q))
			    (format nil "~A ~A" (cadr const) (caddr const))
			    (princ-to-string const))
			'constant))
			nil (window group) 0 0 t t '9x15))
    (menu-reposition m)
    (conn-add-box group m 'constant 'constant (glconstanttype const) nil
		 (list (a conn-port portname = 'constant
			            portcode = (kwote const)
				    type     = (glconstanttype const)
				    filled   = T)))
    (draw m)
    m ))

; 22 Mar 94; 03 May 94
; Read from string.  Returns a list in case of multiple items.
(defun conn-read-from-string (s)
  (let (lst val end lng)
    (setq lng (length s))
    (multiple-value-setq (val end) (read-from-string s))
    (if (>= end lng)
	val
	(progn (push val lst)
	       (while (< end lng)
		 (multiple-value-setq (val end)
				      (read-from-string s nil nil :start end))
		 (if val (push val lst)))
	       (nreverse lst))) ))

; 25 Nov 92; 27 Nov 92; 30 Nov 92; 01 Dec 92; 07 Jan 94; 18 Jan 94; 20 Jan 94
; 24 Jan 94; 01 Dec 94; 27 Apr 95; 05 Mar 96
; Make a box for a variable and add to window
(gldefun conn-make-var ((group conn-group))
  (result menu)
  (let ((menu menu-set-menu) type name)
    (name = (get-input group "Enter variable name: "))
    (type = (get-input group "Enter variable type: "))
    (menu = (conn-menu group (a glnametype name = name type = type) 'data))
    (menu-reposition menu)
    (conn-add-box group menu 'variable name type 'local
		  (list (a conn-port portname = name portcode = name
		          type = type)))
    (draw menu)
    menu ))

; 29 Apr 93; 30 Apr 93; 07 May 93; 18 Jan 94; 28 Jan 94; 03 May 94; 01 Dec 94
; 05 Feb 98; 28 Feb 02; 05 Jan 04; 16 Feb 04
; Expand a box for a variable to show its components
(gldefun conn-expand ((group conn-group))
  (result menu)
  (let (mitem box name type oldm m datanames pt)
    (pt = (get-crosshairs (window group)))
    (if (mitem = (find-item (mconns group) pt))
     (progn
      (box = (conn-box group (menu-name mitem)))
      (if ((boxtype box) == 'variable)
       (progn
	(name = (name box))
	(oldm = (menu mitem))
	(type = (type box))
	(datanames = (gldatanames type))
	((portvals box) =
          (cons (a conn-port portname = name portcode = name type = type)
	    (for nt in datanames collect
	      (a conn-port portname = (name nt)
		 portcode = (list (name nt) name) type = (type nt)))))
	(m = (menu-create (if datanames
			      (mapcar #'car datanames)
                                 ; was (cons (cons "Whole" name) ... )
			      (list name))
			    (if datanames (or (name box) name))
			    (window group) 0 0 t t '9x15))
	(menu-reposition m)
	((menu mitem) = m)
	(clear oldm)
	(draw m)
	m )) )) ))

; 03 Apr 08
; Minimize a box
(gldefun conn-minimize ((group conn-group))
  (result menu)
  (let (mitem box name type oldm m pt)
    (pt = (get-crosshairs (window group)))
    (if (and (mitem = (find-item (mconns group) pt))
             (box = (conn-box group (menu-name mitem))))
        (progn
          (name = (name box))
          (oldm = (menu mitem))
          (type = (type box))
          (m = (menu-create nil
                            (conn-menu-title
                              (a glnametype name = name type = type))
                            (window group) 0 0 t t '9x15))
          (menu-reposition m)
          ((menu mitem) = m)
          (clear oldm)
          (draw m)
          m )) ))

; 29 Apr 93; 30 Apr 93; 06 Jan 94; 18 Jan 94; 27 Jan 94; 23 Oct 97; 05 Jan 04
; 22 Jan 08
(gldefun conn-delete ((group conn-group))
  (let (pt mitem conn id box)
    (pt = (get-cross (window group)))
    (if (mitem = (find-item (mconns group) pt))
	(progn (id = (menu-name mitem))
	       (delete-item (mconns group) mitem)
	       ((source-names group) _- id)
	       (if (box = (that (boxes group)
				with ((boxname (that conn-box)) == id) ) )
		   ((boxes group) _- box)))
        (if (conn = (find-conn (mconns group) pt))
	    (delete-conn (mconns group) conn)))
    (if (or mitem conn)
	(redraw group)) ))

; 21 Nov 92; 23 Nov 92; 24 Nov 92; 25 Nov 92; 19 Mar 93; 22 Mar 93; 29 Apr 93
; 30 Apr 93; 06 May 93; 07 May 93; 14 Sep 93; 16 Sep 93; 06 Jan 94; 17 Jan 94
; 18 Jan 94; 25 Jan 94; 27 Jan 94; 08 Feb 94; 17 Feb 94; 03 May 94; 25 Nov 94
; 27 Dec 94; 18 Mar 98; 29 May 98; 03 Jun 98; 28 Feb 02; 05 Jan 04; 14 Sep 06
; 18 Mar 08; 03 Apr 08; 08 Apr 08
; Acquire views and connections
(gldefun conn-edit-conns ((group conn-group))
  (let (ms done (sel menu-selection) res (redraw t) box pair-type pair-rhs
	   (newsel menu-selection))
    (ms = (menu-set group))
    (while (not done)
      (if (not ((windowsize group) == (size (window group)))) ; check resize
	  (progn
	    ((windowsize group) = (size (window group)))
	    (adjust (menu-set group) 'command 'top nil 1)
	    (adjust (menu-set group) 'command 'right nil 2)
	    (redraw = t)))
      (if redraw (progn (redraw group) (redraw = nil)))
      (sel = (or newsel (select ms)))
      (newsel = nil)
      (if sel
        (if ((menu-name sel) == 'command)
	    (case (port sel)
		   ((quit done redo)   (done = t) (res = (port sel)))
		   (op                 (conn-make-op group))
		   (constant           (conn-make-constant group))
		   (type-in            (conn-typein-constant group))
		   ((geometry physics userlaw)
		                       (conn-law-view group (port sel)))
		   (program            (conn-call group))
                   (prop               (conn-prop group))
		   (makevar            (conn-make-var group))
		   (specifytype        (conn-specify-type group))
		   (redraw             (setq redraw t))
		   (expand             (conn-expand group))
		   (minimize           (conn-minimize group))
		   (delete             (conn-delete group))
		   (move               (move ms) (setq redraw t))
                   (t     (if (fboundp (port sel))
                              (funcall (port sel) group)
                              (error "Bad selection ~A in conn-edit-conns"
                                     (port sel)) ) ) )
	    (if ((menu-name sel) != 'background)
		(progn (box = (conn-box group (menu-name sel)))
		       ((port sel) = (conn-box-port box (port sel)))
		       (pair-type = (conn-find-type (type box) (port sel)))
		       (pair-rhs = (conn-getexpr group pair-type
						 (list (menu-name sel))))
		       (if (atom pair-rhs)
			   (case pair-rhs 
			     (quit (done = t) (res = 'quit) )
			     (done (done = t))
			     (redo)
			     (t (newsel = (a menu-selection
					     with port = pair-rhs
						  menu-name  = 'command))) )
			   (conn-add-conn group sel (car pair-rhs))) ) ) )))
    res ))




; 10 Feb 94; 05 Jan 04; 27 Dec 07
(gldefun conn-add-conn ((group conn-group) (from menu-port) (to menu-port))
  (let (conn fn)
    (conn = (a menu-set-conn with from = from  to = to))
    (fn = (connpfn group))
    (if (or (null fn)
	    (funcall fn group conn))
	(progn (push conn (connections (mconns group)))
	       (conn-draw-conn (menu-set group) conn)
               (if (connfn group)
                   (funcall (connfn group) group from to))) )))

; 27 Jan 94; 02 Mar 94; 17 Jan 08
(gldefun conn-group-redraw ((group conn-group))
  (let ()
    (clear (window (mconns group)))
    (draw (menu-set (mconns group)))
    (for c in (connections (mconns group))
         (conn-draw-conn (menu-set (mconns group)) c) )
    (draw-border (window group))
    (if (> *conn-message-area-width* 0)
	(draw-box-xy (window group) (message-area-x group)
		     (message-area-y group)
		     *conn-message-area-width* 30))  ))

; 08 Feb 94; 05 Jan 04; 17 Jan 08
; Draw a connection.  May have to wade through properties to get to the
; displayed menu.
(gldefun conn-draw-conn ((ms menu-set) (conn menu-set-conn))
  (let (fnname)
    (if (not (symbolp (menu-name (from conn))))
        (conn-draw-conn ms (a menu-set-conn with from = (cadr (from conn))
                              to = (to conn)))
        (if (not (symbolp (menu-name (to conn))))
            (conn-draw-conn ms (a menu-set-conn with from = (from conn)
				                     to = (cadr (to conn))))

            (if (and (fnname = (sym (named-item ms (menu-name (to conn)))))
                     (dag-inputp fnname (port (to conn)))
                     (itemp ms (menu-name (from conn)) fnname) )
                (draw-conn ms (a menu-set-conn with
                               from = (a menu-port with port = fnname
                                         menu-name = (menu-name from))
                               to = (to conn)))
                (draw-conn ms conn) ) ) ) ))

; 27 Jan 94; 02 Mar 94
(gldefun conn-group-message ((group conn-group) (msg string))
  (let (mx my)
    (mx = (message-area-x group))
    (my = (message-area-y group))
    (erase-area-xy (window group) (mx + 1) (my + 1)
		   (*conn-message-area-width* - 2) 38)
    (printat-xy (window group) (stringify msg) (mx + 10) (my + 12)) ))

; 27 Jan 94; 02 Mar 94; 03 May 94
(gldefun conn-group-erase-message ((group conn-group))
  (let ()
    (erase-area-xy (window group) ((message-area-x group) + 1)
		   ((message-area-y group) + 1)
		   (*conn-message-area-width* - 2) 38) ))

; 27 Jan 94; 02 Mar 94; 01 Apr 94; 28 Feb 02
(gldefun conn-group-tty-input ((msg string))
  (let (input)
    (princ msg) (princ " ")
    (while ( (input = (read-line nil)) == "") )
    input))

; 27 Apr 95; 05 Mar 96; 05 Jan 04
; Get input from user.  Returns a list if multiple items are input.
(gldefun conn-group-get-input ((group conn-group) (msg string))
  (conn-read-from-string
   (if (> *conn-message-area-width* 0)
       (conn-group-window-input group msg)
       (conn-group-tty-input msg))))

; 11 Apr 08; 09 Mar 10
(gldefun conn-group-window-input ((group conn-group) (msg string))
  (let (mx my mw)
    (mx = (message-area-x group))
    (my = (message-area-y group))
    (mw = (string-width (window group) msg))
    (conn-msg group msg)
    (input-string (window group) "" (mx + mw + 10) (my + 12)
		  (*conn-message-area-width* - (mw + 10))) ))

; 11 Apr 08
; Output a message
(gldefun conn-msg ((group conn-group) (msg string))
  (let (mx my mw)
    (if (> *conn-message-area-width* 0)
        (progn (mx = (message-area-x group))
               (my = (message-area-y group))
               (mw = (string-width (window group) msg))
               (printat-xy (window group) msg (mx + 10) (my + 12)))
        (progn (princ msg) (terpri)) ) ))
    
; 01 Feb 94
(gldefun conn-group-sourcep ((group conn-group) (name symbol))
  (assoc name (sources group)) )

; 01 Feb 94
(gldefun conn-group-goalp ((group conn-group) (name symbol))
  (assoc name (goals group)) )

; 17 Jan 94; 09 Jan 08
; Find the type of a field or property of a box
(gldefun conn-find-type ((type gltype) (name symbol))
    (or (gldatatype type name)
	(gevproptype type 'prop name)
	(glviewtype type name) ) )

; 25 Jan 94; 30 Nov 94; 01 Dec 94; 05 Mar 96; 18 Feb 10
; If port is nil, coerce to variable name
(gldefun conn-box-port ((box conn-box) (port symbol))
  (or port
      (and (member (boxtype box) '(variable fn law))
	   (boxname box)) ))

; 23 Nov 92; 24 Nov 92; 27 Nov 92; 30 Nov 92; 20 Dec 92; 05 May 93; 07 Jan 94
; 18 Jan 94; 03 May 94; 01 Dec 94
; Get a selection of a view associated with physical or geometric laws
(gldefun conn-law-view ((group conn-group) (area symbol))
  (let (law menu)
    (law = (conn-select-law area))
    (unless (or (null law) (eq law 'quit))
      (menu = (conn-menu group (a glnametype with type = law) t))
      (menu-reposition menu)
      (conn-add-box group menu 'law law nil nil nil)
      (menu-mdraw menu) ) ))

(setq *conn-laws*
      '(z (geometry (plane line-segment vector circle right-triangle
			   rectangle region)
		    (solid sphere cylinder cone mercator) )
	  (physics  (kinematics uniform-motion uniform-acceleration falling
				projectile)
		    (dynamics   centrifugal-force)
		    (force      weight friction gravitation centrifugal-force
                                inclined-plane)
		    (energy     gravity-kinetic kinetic-energy) )
	  ))

; 23 Nov 92; 17 Feb 94; 17 Mar 98
; Get a selection of a view associated with physical or geometric laws
(gldefun conn-select-law ((area symbol))
  (conn-tree-select
    (case area
      (userlaw (cons nil *conn-user-laws*))
      (nil     *conn-laws*)
      (t       (assoc area (rest *conn-laws*))))))

; 25 Nov 94; 19 Sep 95; 05 Jan 04
; Select item from a tree-structured set of items
(gldefun conn-tree-select (tree)
  (let (done sels sel prev tmp result (savetree tree))
    (if tree
	(while (not done)
	  (sels = (mapcar #'(lambda (x) (if (consp x) (car x) x))
			    (cdr tree)))
	  (sel = (menu (cons '("Quit" . quit)
			   (if prev
			       (cons '("Redo" . redo)
				     (cons '("Back up" . backup) sels))
			       sels))))
	  (case sel
	    (quit (done = t) (result = 'quit))
	    (redo (tree = savetree)
		  (prev = nil))
	    (backup (tree -_ prev))
	    (t (tmp = (conn-tree-assoc sel (cdr tree)))
	       (if (consp tmp)
		   (if (and (atom (cadr tmp)) (null (cddr tmp)))
			    (progn (result = (cadr tmp))
				   (done = t))
		       (progn (prev +_ tree)
			      (tree = tmp)))
		   (progn (result = tmp) (done = t)) ) ) ) ))
    result))

; 19 Sep 95
; assoc that allows symbols in the list
(defun conn-tree-assoc (sym alist)
  (if alist
      (if (eql sym (if (consp (car alist))
		       (caar alist)
		       (car alist)))
	  (car alist)
	  (conn-tree-assoc sym (rest alist))) ) )

; 25 Nov 94; 01 Dec 94; 11 Jan 96; 05 Mar 96; 18 Mar 98; 24 Mar 98; 28 Feb 02
; 05 Jan 04; 14 Nov 06; 06 Dec 07
; Add a program box.  area = program
(gldefun conn-call ((group conn-group))
  (let (fn)
    (fn = (if *conn-programs*
              (conn-tree-select *conn-programs*)
              'type-in))
    (if (fn == 'type-in)
	(fn = (get-input group "Enter function name: "))
        (if (assoc fn *conn-program-actions*)
            (setq fn (funcall (cadr (assoc fn *conn-program-actions*)) group))))
    (when (and fn (not (eq fn 'quit)) (symbolp fn)
	       (or (fboundp fn) (conn-iteratorp fn)))
      (conn-call-fn group fn) ) ))

; 08 Jan 08
; Add a program box to call a function
(gldefun conn-call-fn ((group conn-group) (fn symbol))
  (let (menu args restp)
    (restp = (glfnresulttype! fn))
    (args = (glarguments fn))
    (menu = (conn-menu group (a glnametype with type = fn) 'fn))
    (menu-reposition menu)
    (conn-add-box group menu 'fn fn restp nil
                  (cons (a conn-port portname = fn  type = restp)
                        (for arg in args collect
                             (a conn-port portname = (name arg)
                                type = (type arg)))))
    (menu-mdraw menu) ))

; 24 Mar 98
(gldefun conn-iteratorp ((fn symbol))
  (result boolean)
  (member fn '(iterator)))

; 18 Mar 08; 23 Oct 09
; Get a prop from an existing box
(gldefun conn-prop ((group conn-group))
  (let (done sel (result anything) box tp)
    (while ~ done do
      (sel = (select (menu-set group)))
      (if (member (menu-name sel) '(command background))
	  (done = t)
          (if (box = (conn-box group (menu-name sel)))
              (progn (done = t)
                     (if (port sel)
                         (tp = (cadr (assoc (port sel)
                                            (conn-getnames (type box) t))))
                         (progn 
                           ((port sel) = (conn-box-port box (port sel)))
                           (tp = (type box))))
                     (result = (conngetexprfld group sel tp 'anything))))))
    result))



; 25 Jul 91; 28 Sep 92; 29 Sep 92; 02 Oct 92; 06 Oct 92; 26 Oct 92; 19 Nov 92
; 24 Nov 92; 25 Nov 92; 19 Mar 93; 07 May 93; 06 Jan 94; 17 Jan 94; 18 Jan 94
; 25 Jan 94; 27 Dec 94; 28 Feb 02; 05 Jan 04; 21 May 04; 03 Jun 04; 22 Dec 08
; Get an expression for an item of the given type.
; Returns (code type) or atomic command
(gldefun conn-getexpr ((group conn-group) (goal-type gltype) &optional except)
  (let (done sel (result anything) res-type box)
    (while ~ done do
      (sel = (select (menu-set group) nil
		       (set-difference (source-names group) except)))
      (if ((menu-name sel) == 'command)
	  (progn (done = t)
		 (result = (port sel)))
	  (if ((menu-name sel) == 'background)
	      nil
	      (if (box = (conn-box group (menu-name sel)))
		  (progn (done = t)
			 ((port sel) = (conn-box-port box (port sel)))
			 (res-type = (conn-find-type (type box) (port sel)))
			 (result =
				 (if (or (null goal-type)
                                         (and (goal-type != 'anything)
                                              (conngetexprmatch res-type
                                                                goal-type)))
				     (list sel res-type)
				     (conngetexprfld group sel res-type
						     goal-type))))))))
    result))

; Following functions derived from getexpr.lsp


; 27 Sep 89; 25 Jul 91; 25 Sep 92; 28 Sep 92; 14 Oct 92; 08 Feb 94; 10 Feb 94
; 03 May 94; 28 Dec 98; 18 Mar 08; 20 Mar 08
; Get a field spec starting from TYPE
(defun conngetexprfld (group start type goaltype)
  (let (props matches lst sel boxname)
    (if (and goaltype
	     (not (eq goaltype 'anything))
	     (conngetexprmatch type goaltype))
	(list start type)
	(progn (setq props (conn-getnames type t))
	       (setq matches
                     (subset #'(lambda (x) (conngetexprmatch (cadr x) goaltype))
                             props))
	       (if (and goaltype matches (null (cdr matches)))
		   (setq sel (caar matches))
		   (if props
		       (progn (setq lst (mapcar #'car props))
			      (if (or (null goaltype)
				      (conngetexprmatch type goaltype))
				  (push '("Done" . conngetexprmatch-done) lst))
			      (setq sel
				    (menu (cons '("Redo" .
						  conngetexprmatch-redo)
						lst))))
		       (setq sel 'conngetexprmatch-done)))
	       (if (eq sel 'conngetexprmatch-redo)
		   'redo
		   (if (member sel '(identity conngetexprmatch-done))
		       (list start type)
		       (progn (setq boxname (conn-make-op-box group sel 'prop))
			      (conn-add-conn group start (list 'in boxname))
			      (conngetexprfld group
					      (list 'out boxname)
					      (cadr (assoc sel props))
					      goaltype) ) ) ) ) ) ))

; 2-Aug-87; 28 Sep 92; 12 Oct 92; 05 Mar 93; 17 Mar 93; 18 Mar 93; 08 Feb 94
; 08 Mar 94
; Test whether a given type matches a goal type 
(defun conngetexprmatch (type goaltype)
  (setq type (glxtrtype type))
  (cond ((or (null type)
	     (eq type goaltype)
	     (and (null goaltype)
		  (member type *glbasictypes*))))
	((symbolp goaltype)
	  (case goaltype (anything t)
		         ((number real integer)
			   (or (member type '(number real integer))
			       (and (consp type)
				    (eq (car type) 'units)
				    (consp (cdr type))
				    (member (cadr type)
					    '(number real integer)))))
			 (boolean (or (eq type 'boolean)
				      (and (consp type)
					   (eq (car type) 'listof))))
			 (string (eq type goaltype))
			 ((atom symbol)
			   (member type '(number real integer symbol
						 atom boolean)))
			 (t (or (gltypeint type goaltype)
				(and (glunitp goaltype)
				     (member type '(number real integer)))))))
	((and (consp goaltype)
	      (eq (car goaltype) 'listof))
	 (and (consp type)
	      (eq (car type) 'listof)
	      (conngetexprmatch (cadr type) (cadr goaltype))))
	((and (consp goaltype)
	      (eq (car goaltype) 'units)
	      (consp (cdr goaltype))
	      (consp (cddr goaltype)))
	  (or (member type '(number real integer))
	      (and (consp type)
		   (eq (car type) 'units)
		   (consp (cdr type))
		   (consp (cddr type))
		   (glconvertunit (caddr type) (caddr goaltype)))))
	(t nil)))


; 12-Aug-87; 04 Jan 91; 25 Sep 92; 28 Sep 92; 07 Apr 93; 19 Sep 95; 22 Dec 98
; 30 Dec 98; 28 Feb 02; 05 Jan 04; 09 Jan 09
(gldefun conngetexprop ((group conn-group) resulttype)
  (let (op)
    (setq op (if (eq resulttype 'boolean)
		 (menu '(< <= == >= > != and or not))
		 (menu '(+ - * / ^ min max if fn bool))))
    (if (eq op 'fn)
	(progn (setq op (menu '(sqrt sin cos tan atan exp log expt type-in)))
	       (if (eq op 'type-in)
		   (progn (op = (get-input group "Enter function name: "))
			  (or (conn-nargs op)
			      (setf (glnargs op)
				    (get-input group
					       "Enter number of arguments: "))))))
        (if (eq op 'bool) (op = (conngetexprop group 'boolean) )) )
    op))

; 12-Aug-87; 28 Sep 92
; Infer type of result of an arithmetic operation 
(defun conngetexprtypeinf (op lhstype rhstype)
(cond ((member op '(< > <= >= <> != = == and or not)) 'boolean)
      ((member op '(+ - * / ^))
        (cond ((and (eq lhstype 'integer)
		    (eq rhstype 'integer))
	        'integer)
	      ((or (eq lhstype 'real)
		   (eq rhstype 'real))
	        'real)
	      ((or (eq lhstype 'number)
		   (eq rhstype 'number))
	        'number)
	      (t lhstype)))))

(setq *conn-constants*
      '(z (speed (zero        0.0        (/ m s) 0)
		 (light       2.997925e8 (/ m s) c)
		 (sound       1090.0     (/ ft s)))
	  (mass  (m           1.0        kg m)
		 (earth       5.98e24 kg)
		 (moon        7.36e22 kg)
		 (sun         1.99e30 kg)
		 (electron    9.1091e-31 kg)
		 (proton      1.67252e-27 kg)
		 (neutron     1.67482e-27 kg))
	  (length (zero       0.0      m 0)
		  (earth-radius 6.366198e6 m)
		  (earth-sun    1.49e11 m)
		  (earth-moon   3.80e8 m))
	  (time   (zero         0.0  s 0)
		  (minute       60.0 s)
		  (hour         3600.0 s)
		  (day          86400.0 s)
		  (year         31557600.0 s))
	  (charge (electron     1.60210e-19 coulomb))
	  (acceleration (earth-gravity 9.81 (/ m (* s s))))
	  (basic (pi 3.1415926535897931 unity)
		 (e  2.7182818284590452 unity)
		 (gravitation 6.67e-11 (/ (* nt m m) (* kg kg))) )
	))

; 26 Oct 92; 23 Nov 92; 24 Nov 92; 04 Aug 93; 03 May 94; 05 Jan 04; 01 Jun 04
; Get a physical constant as a part of an expression
(gldefun conn-getconst ((goal-type gltype))
  (let (done tree tmp prev sel (result anything) sels)
    (tree = *conn-constants*)
    (if (and goal-type
	     (tmp = (assoc goal-type (rest tree))))
	(tree = tmp))
    (while ~ done do
      (sels = (mapcar #'car (cdr tree)))
      (sel = (menu (cons '("Quit" . quit)
			   (if prev
			       (cons '("Redo" . redo)
				     (cons '("Back up" . backup) sels))
			       sels))))					
      (case sel
	(quit (done = t) (result = 'quit))
	(redo (tree = *conn-constants*)
	      (prev = nil))
	(backup (tree -_ prev))
	(t (tmp = (assoc sel (cdr tree)))
	   (if (numberp (cadr tmp))
	       (progn (result =
		        (list (list (cadr tmp)
				    (list 'units
					  (if (integerp (cadr tmp))
					      'integer 'real)
					  (caddr tmp)))
			      (or (cadddr tmp) (car tmp))))
		      (done = t))
	       (progn (prev +_ tree)
		      (tree = tmp)))) ))
    result ))

; 22 Mar 93; 23 Mar 93; 30 Apr 93; 07 May 93; 06 Jan 94; 17 Jan 94; 22 Mar 94
; 19 May 94; 28 Dec 94; 05 Mar 96; 28 Jul 97; 05 Jan 04
; Specify the type of a box/port
(gldefun conn-specify-type ((group conn-group))
  (let (sel mi box (port conn-port) type)
    (sel = (select (menu-set group) nil (source-names group)))
    (if (mi = (named-item (menu-set group) (menu-name sel)))
	(progn (box = (conn-box group (menu-name sel)))
	       (port = (or (that (portvals box)
				 with portname = (port sel))
			   (if (null (portvals box))
			       (conn-add-port box (port sel) nil nil))))
	       (if (or port (eq (boxtype box) 'variable))
		(progn (type = (get-input group "Enter type or unit: "))
		       (if (and (symbolp type)
				(not (glbasictypep type))
				(glunitp type))
			   (setq type (list 'units 'real type)))
	              (if port ((type port) = type))
		      (if (eq (boxtype box) 'variable)
			  ((type box) = type))) ) ))))

; 08 Apr 93; 06 Jan 94
; Find the box with a given name
(gldefun conn-box ((group conn-group) (boxname symbol)) (result conn-box)
  (assoc boxname (boxes group)))

; 17 Feb 94
; Find the box with a given name
(gldefun conn-var-box ((group conn-group) (var symbol)) (result conn-box)
  (let (boxname)
    (boxname = (car (find var (menu-items (menu-set (mconns group)))
			    :key #'cadr)))
    (assoc boxname (boxes group)) ))

; 19 May 94; 01 Jun 04
; Add a port to a box
(gldefun conn-add-port ((target-box conn-box) (portname symbol) code type)
  (let (newport)
    (newport = (a conn-port with portname = portname
		                   portcode = code
				   type = type))
    ((portvals target-box) _+ newport)
    newport))

; 28 Sep 06
; Make a smaller version of a conn-group to save in a file
(gldefun conn-save ((group conn-group) (name symbol)) (result conn-save-group)
  (let ()
    (a conn-save-group
       name = name
       sources = (sources group)
       goals = (goals group)
       connections = (connections (mconns group))
       boxes = (for box in (boxes group) collect
                    (a conn-save-box boxname   = (boxname box)
                                     boxtype   = (boxtype box)
                                     name      = (name box)
                                     type      = (type box)
                                     direction = (direction box)
                                     value     =
          (if (eq (boxtype box) 'constant)
              (portcode (first (portvals box))))
                                     offset    =
          (offset (named-menu (menu-set (mconns group)) (boxname box)))))
       connpfn = (connpfn group)
       letvars = (letvars group)
       windowsize = (windowsize group) ) ))

