; dag.lsp                   Gordon S. Novak Jr.               ; 07 Apr 17

; Copyright (c) 2017 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; DAG Programming by graphical connections

; 06 Nov 07; 13 Nov 07; 14 Nov 07; 06 Dec 07; 12 Dec 07; 13 Dec 07; 21 Dec 07
; 24 Dec 07; 26 Dec 07; 27 Dec 07; 28 Dec 07; 02 Jan 08; 03 Jan 08; 09 Jan 08
; 10 Jan 08; 15 Jan 08; 17 Jan 08; 18 Jan 08; 22 Jan 08; 20 Mar 08; 02 Apr 08
; 08 Apr 08; 09 Apr 08; 10 Apr 08; 15 Apr 08; 25 Apr 08; 30 May 08; 16 Dec 08
; 17 Dec 08; 18 Dec 08; 19 Dec 08; 22 Dec 08; 23 Dec 08; 24 Dec 08; 26 Dec 08
; 12 Jan 09; 24 Dec 09; 08 Mar 11

; initially derived from makev.lsp, then from vip.lsp
; Example:     (dag '((c circle) (ls line-segment)))
; find intersection of circle and line, then arc-short - chord-length
; glisp/examples/supermanb.gif
;    (DAGFN146 (a circle center '(3 3) radius 2) '((1 1) (5 6)))  ;  = 1.705

; (dag '((p1 vector) (p2 vector) (poles (listof circle)))) ; superman, Skiena p. 300

; DAG could check type correspondence of two quantities that are connected
; by the user, use a red line to indicate mismatch.
; mismatch could cause generic to be specialized to fit the type,
; or cause a type conversion to be inserted.

; a connection could be interrupted by a variable, file, or tracing function


(defvar *dag-window-width*    700)
(defvar *dag-window-height*   600)
(defvar *dag-perm-window*     nil)  ; t to leave window displayed permanently
(defvar *dag-group*           nil)
(defvar *dag-save-group*      nil)
(defvar *dag-code*            nil)
(defvar *dag-fnname*          nil)
(defvar *dag-fntype*          nil)
(defvar *dag-trace*           nil)
(defvar *dag-let-allowed*     nil)  ; t if allowed to introduce let vars
(defvar *dag-let-size*        2)    ; size of expression to make let var
(defvar *dag-types*           nil)
(defvar *dag-data-urls*
        '(("CD Catalog" . "https://www.w3schools.com/xml/cd_catalog.xml")))
; (defvar *dag-fns*             nil)  ; fns created using dag

(glispglobals
 (*dag-group*          conn-group)
 (*dag-save-group*     conn-group)
 (*dag-perm-window*    boolean)
    )  ; glispglobals

; 24 Dec 08
; Make a window to draw in.
(setf (glfnresulttype 'dag-window) 'window)
(defun dag-window ()
  (let (w)
    (setq w (window-create *dag-window-width* *dag-window-height* "Dag window"))
    (if (boundp '*wio-font*) (window-set-font w *wio-font*))
    w))

; Make a program by graphical connections
; sources = ((name type) ...)
; outputs = ((name type) ...)
; fnname  = name of function to be created
; globals = vars to be treated as globals rather than args
(defun dag (&optional sources outputs fnname globals)
  (dagfn (vip-listify sources)
	 (or (vip-listify outputs) (list (list 'output nil)))
	 (or fnname (glgensym 'dagfn))
         globals) )

; 15 Nov 07; 12 Dec 07; 27 Dec 07; 09 Jan 08; 08 Apr 08; 15 Apr 08; 30 May 08
; 18 Dec 08; 22 Dec 08; 24 Dec 08; 12 Jan 09
(gldefun dagfn ((sources (listof glnametype)) (outputs (listof glnametype))
                fnname (globals (listof symbol)))
  (let (res resb w group)
    (*dag-fnname* = fnname)
    (w = (dag-window))
    (group = (conn-init sources outputs w 'props nil
                        '(("Quit" . quit)
                          ("Done" . done)
                          ("Redo" . redo)
                          ("Redraw" . redraw)
                          ("Move" . move)
                          ("Expand" . expand)
                          ("Minimize" . minimize)
                          ("Delete" . delete)
                          ("Geometry" . geometry)
                          ("Physics" . physics)
                          ("User Law" . userlaw)
                          ("Function From" . dag-function-from)
                          ("Function To"   . dag-function-to)
                          ("VIP" . dag-vip)
                          ("GP" . dag-gp)
                          ("DAG" . dag-dag)
                          ("Program" . program)
                          ("Prop" . prop)
                          ("Make Var" . makevar)
                 ;               ("Grok Data" . dag-grok-data)
                          ("Specify Type" . specifytype)
                          ("OP" . op)
                          ("Type-in" . type-in)
                          ("Constant" . constant)
                          ("Language"      . language)
                          ) nil))
    (*dag-group* = group)      ; for debugging
    (glcc? 'dag-connfn)
    ((connfn group) = 'dag-connfn)
    ((verticalp group) = t)
    (for source in sources
         when (and (symbolp (type source))
                   (not (glbasictypep (type source))))
         (pushnew (type source) *dag-types*) )
    (res = (conn-edit-conns group))
    (if (res != 'quit)
	(setq *dag-save-group* (copy-tree group)) )
    (if (res == 'done)
	(resb = (dag-make-program group fnname globals)))
    (unless *dag-perm-window*
      (destroy (window group))
      ((window group) = nil))
    (or fnname resb res) ))

(defun dag-remake ()
  (dag-make-program (copy-tree *dag-save-group*) *dag-fnname* nil) )

; ---------------------------------------------------------------------------
; Code to make a program from the diagram follows.
; ---------------------------------------------------------------------------

; 06 Nov 07
(defun dag-trace ()
  (trace dag-make-program )
  (setq *dag-trace* t))

; 06 Nov 07
(defun dag-untrace ()
  (untrace dag-make-program )
  (setq *dag-trace* nil))

; 06 Nov 07; 08 Nov 07; 15 Nov 07; 24 Dec 07; 27 Dec 07; 31 Dec 07; 09 Jan 08
; 10 Jan 08; 18 Jan 08; 09 Apr 08; 10 Apr 08; 18 Dec 08; 19 Dec 08; 22 Dec 08
; Make a program from a set of boxes and connections
(gldefun dag-make-program ((group conn-group) (fnname symbol)
                           (globals (listof symbol)))
  (let ((datas (listof cdatas)) comps fromb tob bfrom bto tmp pfrom pto ptmp
         newvar existing (cpr cprogs) newcomp nm source)
    (for box in (boxes group)
      (if (fnp box)
          (push (a ccomps with name  = (boxname box)
                               cprog = (dag-make-cprogs (boxname box)
                                                        (name box)
                                                        (portvals box)))
                comps)
          (if (varp box)
              (push (a cdatas with name = (name box)
                                   type = (type box)  kind = 'var)
                    datas)
              (if (constp box)
                  (push (a cdatas with name = (boxname box)
                                       value = (portcode (first (portvals box)))
                                       type = (type box)  kind = 'const)
                        datas)) ) ) )
    (for conn in (connections (mconns group))
      (if *dag-trace* (format t "conn = ~A~%" conn))
      (pfrom = (from conn))
      (pto   = (to conn))
      (bfrom = (box group (menu-name pfrom)))
      (bto   = (box group (menu-name pto)))
  ; if the connection is known to be backwards, reverse it
      (if (or (dag-port-inputp pfrom bfrom)
              (dag-port-outputp pto bto))
          (progn (tmp  = bfrom) (bfrom = bto) (bto = tmp)
                 (ptmp = pfrom) (pfrom = pto) (pto = ptmp)))
      (if *dag-trace*
          (format t "bfrom = ~A   pfrom = ~A~%bto   = ~A   pto   = ~A~%"
                    bfrom pfrom bto pto) )
      (if (fnp bfrom)
          (if (datap bto)
              (dag-add-conn comps (boxname bfrom)
                            (if (and (fnp bfrom)
                                     (eq (port pfrom) (name bfrom)))
                                'out
                                (port pfrom))
                            (bnm bto) nil)
              (if (fnp bto)       ; insert or reuse a var in the middle
                  (progn
                    (if (existing = (some #'(glambda (d)
                                              (and (eq (type d) (type bfrom))
                                                   (eq (kind d) 'var)
                                                   (dag-output-of bfrom d comps)
                                                   d) )
                                          datas))
                        (newvar = (name existing))
                        (progn
                          (newvar = (glgensym (port pfrom)))
                          (push (a cdatas with name = newvar
                                               type = (type bfrom)
                                               kind = 'var)
                                datas)
                          (dag-add-conn comps (boxname bfrom) (port pfrom)
                                        newvar nil)))
                    (dag-add-conn comps (boxname bto) (port pto) newvar t))
                  (error "dag-make-program ... 1~%")))
          (if (datap bfrom)
              (if (fnp bto)
                  (dag-add-conn comps (boxname bto) (port pto)
                                (if (varp bfrom)
                                    (name bfrom)
                                    (if (constp bfrom)
                                        (boxname bfrom)
                                        (name bfrom))) t)
                  (if (varp bto)
                      (progn (nm = (glmkatom 'id))
                             (push (a ccomps with name = nm
                                      cprog = (dag-make-cprogs nm 'identity
                                                   (list (a conn-port
                                                            portname = 'out
                                                            type = (type bfrom)
                                                            filled = t)
                                                         (a conn-port
                                                            portname = 'in
                                                            type = (type bfrom)
                                                            filled = t))))
                                    comps)
                             (dag-add-conn comps nm 'in (bnm bfrom) t)
                             (dag-add-conn comps nm 'out (bnm bto) nil) )
                      (error "dag-make-program ... 3~%")))
              (error "dag-make-program ... 2~%"))) )
    (defcdag fnname (sources group) (goals group) datas comps)
    (dag2lispfn fnname nil t) ))

; 31 Dec 07
; Make a cprogs data structure for a box.
; First of args is result
(gldefun dag-make-cprogs ((cpr cprogs) (fn symbol) (args (listof conn-port)) )
  (let ()
    ((storedkind cpr)   = 'lisp)
    ((storedfnname cpr) = fn)
    ((storedinputs cpr) = (for inp in (rest args) collect
                               (a glnametype with
                                    name = (portname inp)
                                    type = (type inp))))
    ((storedoutputs cpr) = (list (a glnametype with
                                    name = (portname (first args))
                                    type = (type (first args)))))
    cpr))

; 08 Nov 07; 15 Nov 07; 08 Apr 08; 19 Dec 08
; add a connection between a ccomp and a var in the cdag
(gldefun dag-add-conn ((comps (listof ccomps)) (boxname symbol)
                       (portname symbol) (varname symbol) (inputp boolean))
  (let (cmp)
    (or (setq cmp (assoc boxname comps))
        (error "dag-add-conn undef cmp ~A~%" boxname))
    (if inputp
        (pushnew (list portname varname) (inputs cmp) :test #'equal)
        (pushnew (list portname varname) (outputs cmp) :test #'equal) )
    comps   ))  ; output value used only to debug via trace

; ????? this is a flakey function.  Get rid of it.
; used by conn-draw-conn
; problem: fn might be a prop name, in which case this doesn't work.
(defun dag-inputp (fn name)
  (or (assoc name (or (get fn 'storedinputs) (glargs fn)))
      (member name '(in in1 in2 in3 in4 in5 in6 in7))))

; 09 Jan 08; 19 Dec 08; 31 Dec 08
; test whether a port of a box is an input of that box
; reliable if this returns true, possibly not if false
(gldefun dag-port-inputp ((prt menu-port) (box conn-box))
  (if (opp box)
      (member (port prt) '(in in2 in3))
      (if (fnp box)
          (inputp (port box (port prt)))
          (if (varp box)
              (eq (direction box) 'output) ) ) ) )

; 09 Jan 08; 19 Dec 08; 31 Dec 08
; test whether a port of a box is an output of that box
; reliable if this returns true, possibly not if false
(gldefun dag-port-outputp ((prt menu-port) (box conn-box))
  (if (opp box)
      (member (port prt) '(out))
      (if (fnp box)
          (outputp (port box (port prt)))
          (or (and (varp box)
                   (or (eq (direction box) 'input)
                       (not (eq (port prt) (name box)))))
              (constp box)) ) ) )

(defun dag-input-name (fn name)
  (let (n names)
    (setq names (or (get fn 'storedinputs) (glargs fn)))
    (if (assoc name names)
        name
        (if (setq n (cadr (assoc name '((in 0) (in1 0) (in2 1)
                                        (in3 2) (in4 3)
                                        (in5 4) (in6 5) (in7 6)))))
            (if names
                (car (nth n names))
                name))) ))

(defun dag-output-name (fn name)
  (let (n names)
    (setq names (or (get fn 'storedoutputs)
                    (and (glfnresulttype fn)
                         (list (list 'out (glfnresulttype fn))))))
    (if (assoc name names)
        name
        (if (setq n (cadr (assoc name '((out 0) (output 0) (out1 0)
                                        (out2 1) (out3 2) (out4 3)
                                        (out5 4) (out6 5) (out7 6)))))
            (if names
                (car (nth n names))
                name))) ))

; 18 Dec 08
; test whether data d is the output of box bfrom in comps
(gldefun dag-output-of ((bfrom conn-box) (d cdatas) (comps (listof ccomps)))
  (let (comp)
    (setq comp (assoc (boxname bfrom) comps))
    (and comp (eq (name d) (dest (first (outputs comp))))) ))

; 26 Sep 06
; Use DAG to make a prop for a type
(defun dagprop (type name)
  (let (fn)
    (setq fn (dag (list (list 'self type))))
    (when fn
      (gladdprop type 'prop
                 (cons name (cons (list fn)
                                  (if *dag-fntype*
                                      (list 'result *dag-fntype*))))) ) ))

; 27 Dec 07; 09 Jan 08; 10 Jan 08; 11 Jan 08; 22 Jan 08; 30 May 08
; This is called by conn-add-conn when a graphical connection is made between
; boxes.  Update type of destination if possible.
(gldefun dag-connfn ((group conn-group) (from menu-port) (to menu-port))
  (let (frombox tobox goal tmp tmpbox fnbox convs sel newbox fromtype totype)
    (frombox = (conn-box group (menu-name from)))
    (tobox = (conn-box group (menu-name to)))
    (if (or (dag-port-inputp from frombox)          ; if backwards, switch
            (dag-port-outputp  to tobox))
        (progn (tmp = from) (from = to) (to = tmp)
               (tmpbox = frombox) (frombox = tobox) (tobox = tmpbox)) )
    (if (and (fnp frombox)
             (dag-port-outputp from frombox)
             (type frombox)
             (varp tobox)
             (not (type tobox)))
        (progn ((type tobox) = (type frombox))
               (if (and (goal = (assoc (name tobox) (goals group)))
                        (null (type goal)))
                   ((type goal) = (type frombox)) ) )
        (if (and (varp frombox)
                 (port from)
                 (not (eq (name frombox) (port from)))  ; field or prop
                 (type frombox))
            (progn
              (pop (connections (mconns group)))   ; remove direct conn
              (fnbox = (dag-add-function-from group frombox (port from)))
              (conn-add-conn group    ; note this will call us again
                   (a menu-port with  port = 'out
                                      menu-name = (boxname fnbox))
                   to)
              (redraw group) )
            (if (and (fromtype = (type (port frombox (port from))))
                     (totype = (type (port tobox (port to))))
                     (not (gltypematch fromtype totype))
                     (convs = (glfindpropgoal fromtype totype 'prop)))
                (progn (pop (connections (mconns group)))   ; remove direct conn
                       (sel = (dag-menu (mapcar #'caddr convs) "Conversion:"))
                       (newbox = (dag-add-function-from group frombox sel))
                       (conn-add-conn group    ; note this will call us again
                                      (a menu-port with  port = 'out
                                                         menu-name = (boxname newbox))
                                      to)) ) ) ) ))

; 06 Dec 07; 13 Dec 07; 24 Dec 07; 26 Dec 07; 28 Dec 07; 31 Dec 07; 02 Jan 08
; 09 Jan 08; 10 Jan 08; 18 Dec 08
; Find a function from a specified type; if found, add to the design
(gldefun dag-function-from ((group conn-group)) (result conn-box)
  (let (selbox sel box typ propm picm)
    (selbox = (select (menu-set group)))
    (when (and (box = (assoc (menu-name selbox) (boxes group)))
               (typ = (type box))
               (symbolp typ)
               (gltypep typ))
      (if (propm = (glpropmenu typ))
          (progn (picm = (picmenu-create-from-spec (picmenu-spec propm)))
                 (sel = (picmenu-select picm))
                 (picmenu-destroy picm)))
      (if (null sel)
          (sel = (menu
                  (union (mapcar #'car (gldatanames typ))
                         (union (mapcar #'car (glpropnametypes typ 'prop))
                                (union (mapcar #'car (glpropnametypes typ 'adj))
                                       (mapcar #'car (glpropnametypes typ 'msg))))))))
      (redraw group)
      (when sel (dag-add-function-from group box sel)) ) ))

; 10 Jan 08; 11 Jan 08; 15 Jan 08; 29 Feb 08; 02 Apr 08; 17 Dec 08; 24 Dec 08
; Make a function from a box, add to group.
; returns the new function box.
(gldefun dag-add-function-from ((group conn-group) (box conn-box) (sel symbol))
  (result conn-box)
  (let (typ seltp restype proptype boxname newbox fn propl
            (args (listof glnametype)) sourceport)
    (typ = (type box))
    (when (seltp = (gltypeofprop typ sel))
      (proptype = (car seltp))           ; str prop adj msg
      (restype = (caddr seltp))          ; result type
      (if (eq proptype 'str)
          (progn (fn = sel)
                 (args = (list (list 'in typ))) )
          (if (propl = (glpropdef typ proptype sel))
              (if (and (consp propl)
                       (symbolp (cadr propl))
                       (fboundp (cadr propl)))
                  (progn (fn = (cadr propl))
                         (args = (or (glargs fn)
                                     (list (list 'in typ)))))
                  (if (and (consp (cadr propl))
                           (member (caadr propl) '(lambda glambda)))
                      (args = (dag-prop-args typ propl))
                      (args = (list (list 'in typ))) ) ) ) )
      (boxname = (conn-make-fn-box group fn sel proptype args restype))
      (newbox = (assoc boxname (boxes group)))
      (sourceport = (if (port box (name box))
                        (name box)
                        (if (port box 'out)
                            'out
                            (name box))))
      (conn-add-conn group
                     (a menu-port with  port = sourceport
                                        menu-name = (boxname box))
                     (a menu-port with  port = (name (first args))
                                        menu-name = (boxname newbox)))
      newbox) ))

; 29 Feb 08
; Get args and types from a glambda form in a prop/msg list;
; using argtypes if available.
(defun dag-prop-args (typ propl)
  (let (argtypes res tp)
    (setq argtypes (cons typ (getf (cddr propl) 'argtypes)))
    (dolist (arg (cadadr propl))
      (setq tp (pop argtypes))
      (push (if (consp arg) arg (list arg tp)) res) )
    (reverse res) ))

; 06 Dec 07; 03 Jan 08; 08 Jan 08; 15 Apr 08; 26 Dec 08
; Find a function to a specified type
(gldefun dag-function-to ((group conn-group))
  (let (sel goal choices)
    (sel = (menu (cons '("Type-in" . type-in) *dag-types*)))
    (if (eq sel 'type-in)
        (sel = (get-input group "Enter goal type: ")) )
    (when (and sel (gltypep sel) (not (glbasictypep sel)))
      (pushnew sel *dag-types*)
      (goal = sel)
      (choices = (glresultof goal))     ; change to: (glfindgoal 'circle)
      (if (and (picmenu-spec goal) (equations goal))
          (push '("VIP" . dag-vip) choices))
      (push '("Make from Parts" . dag-make-parts) choices)
      (sel = (if (cdr choices)
                 (menu choices)
                 'dag-make-parts))
      (if (eq sel 'dag-make-parts)
          (dag-make-parts group goal)
          (if (eq sel 'dag-vip)
              (dag-vip group goal)
              (conn-call-fn group sel) ) ) ) ))

; 08 Jan 08
(gldefun dag-make-parts ((group conn-group) (goal gltype))
  (conn-call-fn group (or (glmakefn goal) (glmakebuildfn goal)) ) )

; 08 Jan 08; 11 Apr 08; 15 Apr 08; 30 May 08; 23 Dec 08; 23 Dec 09
(gldefun dag-vip ((group conn-group) &optional (goal gltype))
  (let (inputs fn restype args boxname newbox)
    (inputs = (dag-select-inputs group))
    (if (symbolp inputs)
        (case inputs
          (quit done)
          (redo (dag-vip group goal)) )
        (progn (setq args (mapcar #'cdr inputs))
               (setq fn (vip args (and goal (list 'output goal))
                             (glgensym 'dagfn)))
               (restype = (glfnresulttype fn))
               (boxname = (conn-make-fn-box group fn fn 'fn args restype))
               (newbox = (assoc boxname (boxes group)))
               (dolist (input inputs)
                 (conn-add-conn group (first input)   ; input selection
                                (a menu-port with  port      = (second input)
                                                   menu-name = (boxname newbox))) )
               newbox) ) ))

; 30 May 08
; Get inputs to a sub-function.
; Note this only allows a whole box to be selected, not buttons inside it
; result is a list of (selection var type), or quit, done, redo
(gldefun dag-select-inputs ((group conn-group))
  (let (done sel nm prt bx inputs)
    (conn-msg group "Select inputs, then Done.")
    (while (not done)
      (sel = (select (menu-set group)))
      (nm = (menu-name sel))
      (prt = (port sel))
      (if (eq nm 'command)
          (if (member prt '(quit done redo))
              (setq done prt))
          (progn (bx = (box group nm))
                 (push (list sel (name bx) (type (port bx (name bx))))
                       inputs) ) ) )
    (or (reverse inputs) done) ))

; 30 May 08
(gldefun dag-gp ((group conn-group))
  (let (inputs fn restype args boxname newbox)
    (inputs = (dag-select-inputs group))
    (if (symbolp inputs)
        (case inputs
          (quit done)
          (redo (dag-gp group)) )
        (when (and (setq args (mapcar #'cdr inputs))
                   (setq fn (gp args))
                   (fboundp fn))
          (restype = (glfnresulttype fn))
          (boxname = (conn-make-fn-box group fn fn 'fn args restype))
          (newbox = (assoc boxname (boxes group)))
          (dolist (input inputs)
            (conn-add-conn group (first input)   ; input selection
                           (a menu-port with  port      = (second input)
                                              menu-name = (boxname newbox))) )
          newbox) ) ))

; 12 Jan 09
; make a sub-program using dag
(gldefun dag-dag ((group conn-group))
  (let (inputs fn restype args boxname newbox)
    (inputs = (dag-select-inputs group))
    (if (symbolp inputs)
        (case inputs
          (quit done)
          (redo (dag-dag group)) )
        (when (and (setq args (mapcar #'cdr inputs))
                   (setq fn (dag args))
                   (fboundp fn))
          (restype = (glfnresulttype fn))
          (boxname = (conn-make-fn-box group fn fn 'fn args restype))
          (newbox = (assoc boxname (boxes group)))
          (dolist (input inputs)
            (conn-add-conn group (first input)   ; input selection
                           (a menu-port with  port      = (second input)
                                              menu-name = (boxname newbox))) )
          newbox) ) ))

(defun dag-menu (choices title)
  (if (cdr choices)
      (menu choices title)
      (car choices) ) )

; 08 Apr 08; 08 Mar 11
(gldefun dag-grok-data ((group conn-group))
  (let (url choice)
    (if *dag-data-urls*
        (choice = (menu (cons '("Type-In" . typein) *dag-data-urls*))) )
    (if (or (null choice)
            (eq choice 'typein))
        (url = (get-input group "Enter data URL: "))
        (url = choice))
    (if (and (stringp url) (> (length url) 10))
        (groknroll url)) ))

; 24 Dec 09
; Make a comparator to compare two items of same type, e.g. for sorting
(gldefun dag-make-comparator ((type gltype))
  (let (var1 var2 code sigs sel fn nm name)
    (var1 = (glgensym 'glvar))
    (var2 = (glgensym 'glvar))
    (sigs = (for x in (glpropnamesigs type 'msg)
                 when (and (eq (third x) 'boolean)
                           (= (length (second x)) 2)
                           (eq (cadar (second x)) type)
                           (eq (cadadr (second x)) type))
                 collect (car x)))
    (if sigs (sel = (menu (cons '("New" . zzzznew) sigs))))
    (if (and sel (not (eq sel 'zzzznew)))
        sel
      (progn
        (code = (dag-make-comparatorb type var1 var2 (glallnames type)))
            ; optionally name the new fn and add it to the type
        (sel = (menu '(yes no) "Name this Method?"))
        (if (sel == 'yes)
            (progn (nm = (conn-group-tty-input "Enter method name: "))
                   (name = (intern (string-upcase nm)))
                   (fn = (intern (concatenate 'string (symbol-name type) "-"
                                              (symbol-name name))))
                   (gladdprop type 'msg (list name fn 'result 'boolean)))
            (fn = (glgensym 'glfn)) )
        (eval (list 'gldefun fn (list (list var1 type) (list var2 type)) code))
        fn)) ))

; 24 Dec 09
; Make a chain comparator to compare two items of same type, e.g. for sorting
(gldefun dag-make-comparatorb ((type gltype) (var1 symbol) (var2 symbol)
                               (fields (listof glnametype)))
  (let (propcode code sel newcode dir)
    (propcode = (dag-prop-chain type fields))
    (dir = (menu '(("< (default)" . <) (">" . >)) "Comparison"))
    (code = (list dir (subst var1 '?var propcode) (subst var2 '?var propcode)))
    (sel = (menu '(("Done" . done) ("Add more tests" . add))))
    (if (sel != 'add)
        code
      (progn (newcode = (dag-make-comparatorb type var1 var2 fields))
             (list 'or code
                   (list 'and (cons '= (cdr code))
                         newcode)))) ))

; 24 Dec 09
(gldefun dag-prop-chain ((type gltype) (fields (listof glnametype)))
  (let (names nt sel)
    (names = (for x in fields collect (name x)))
    (if (glpropdef type 'msg '<)
        (names +_ '("Done" . zzzzdone)))
    (sel = (if (glbasictypep type)
               'zzzzdone
             (menu names)))
    (if (sel == 'zzzzdone)
        '?var
        (progn (nt = (assoc sel fields))
               (subst (list sel '?var) '?var
                      (dag-prop-chain (type nt) (glallnames (type nt)))))) ))



; 03 May 94
; Compile the dag.lsp and conn.lsp files into a plain Lisp file
(defun compile-dag ()
  (glcompfiles *directory*
	       '("glisp/vector.lsp"          ; auxiliary files
                 "X/dwindow.lsp"
		 "glisp/menu-set.lsp")
	       '("glisp/conn.lsp"        ; translated files
		 "glisp/dag.lsp")
	       "glisp/dagtrans.lsp"         ; output file
	       "glisp/dag-header.lsp")      ; header file
  (cf dagtrans) )
