; dispm.lsp            Gordon S. Novak Jr.              ; 16 Feb 04

; Copyright (c) 2007 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; System to create graphical display functions for user data.

; Examples: (make-display-method 'box mybox '(50 50) 'display)
;           (make-display-method 'box mybox '(45 20) 'short)
;           (make-display-method 'bl  mybl  '(20 20) 'display)
;           (make-display-method 'elf happy '(50 50) 'display)
;           (make-display-method 'circle c '(50 50) 'display)
;           (make-display-method 'twobox mytwo '(100 50) 'twoboxdis)
;           (make-display-method 'fourbox myfour '(100 100) 'fourboxdis)
;           (make-display-method 'person gsn '(50 50) 'display)

; 29 Jul 94; 05 Jan 95; 02 Jan 97; 16 Feb 99; 19 Feb 99; 23 Feb 99; 28 Feb 02
; 08 Jan 04

(defvar *dispm-window* nil)
(defvar *dispm-window-size* '(600 400))
(defvar *dispm-sources*)
(defvar *dispm-menu-set*)
(defvar *dispm-specs*)
(defvar *dispm-offset* '(10 10))
(defvar *dispm-bounds*)
(defvar *dispm-name*)
(defvar *dispm-example*)
(defvar *dispm-basic-methods* nil)
(defvar *dispm-label-size* nil)
(defvar *dispm-anything-print* nil)
(defvar *dispm-thermom-edit* nil)

(glispglobals
 (*dispm-window*         window)
 (*dispm-window-size*    vector)
 (*dispm-sources*        (listof glnametype))
 (*dispm-menu-set*       menu-set)
 (*dispm-specs*          (listof dispm-item-spec))
 (*dispm-offset*         vector)
 (*dispm-bounds*         vector)
 (*dispm-name*           symbol)
 (*dispm-basic-methods*  (listof dispm-method-spec))
 (*dispm-label-size*     integer)
 (*dispm-anything-print* dispm-method-spec)
 (*dispm-thermom-edit*   dispm-method-spec)
    )  ; glispglobals

(defmacro gldisplays         (x) `(get ,x 'gldisplays))

(setf (glfnresulttype 'gldisplays) '(listof dispm-method-spec))

(glispobjects

; test data types
  (twobox (list (left box) (right box)))
  (fourbox (list (top twobox) (bottom twobox)))

(dispm-method-spec (list (name symbol)        ; name of the display method
			 (type gltype)        ; type of item that is displayed
			 (list (drawfn         symbol)
			       (editfn         symbol)
			       (highlightfn    symbol)
			       (unhighlightfn  symbol)
			       (params         anything)
			       (sizefn         symbol)
			       (direct-subedit boolean))
			 (size vector)
			 (scalable symbol)
			 (example anything)   ; example input data value
			 (canned-code (listof anything))
			 (item-specs (listof dispm-item-spec)))
                                ; items that make up this display
  prop ((code     (canned-code or
		    (list (if (params self)
			      (list (drawfn self) 'params 'object 'w
				    'offsetx 'offsety 'sizex 'sizey)
                              (list (drawfn self) 'object 'w
				    'offsetx 'offsety 'sizex 'sizey))))) )
  msg  ((draw     dispm-method-spec-draw)
	(erase    dispm-method-spec-erase))  )

(dispm-item-spec (list (display-spec dispm-display-spec)
		       (offset       vector)
		       (size         vector)
		       (sourcetype   gltype)
		       (example      anything)
		       (pathcode     anything)
		       (pathtype     gltype)
		       (getfncache   symbol)
		       (putfncache   symbol)
		       (args         (listof dispm-arg-spec))
		       (subdisplay   dispm-display-spec))
  prop ((region     ((virtual region with start = (offset self)
                                          size  = (size self))))
	(display-method  ((display-method display-spec)))
	(subdisplay-method  ((display-method subdisplay)))
	(getfn       (getfncache or (dispm-item-spec-getfn self)))
	(putfn       (putfncache or (dispm-item-spec-putfn self))) )
  adj  ((basic       ( ~ (item-specs (display-method self)))))
  msg  ((contains?   (glambda (self pt) (contains? (region self) pt)))
	(contains-xy (glambda (self x y)
			      ( ~ (offset self) or ~ (size self) or
				  (contains-xy (region self) x y))))
	(draw        dispm-item-draw)
	(item-eval   dispm-item-eval)
	(item-store  dispm-item-store)
	(highlight   dispm-item-highlight)
	(unhighlight dispm-item-highlight) ) )

(dispm-arg-spec (list (var symbol) (code anything)) )

; 06 Nov 91
; Spec for an argument to a function                ; not used ********
(glargspec (list (name     symbol)
		 (type     gltype)
		 (default  anything)
		 (optional boolean)) )

(glcodetypeexample (list (code anything)
			 (type gltype)
			 (example anything)))

(dispm-display-spec (list (type gltype)
			  (display-name symbol))
 default ((self nil))
  prop   ((display-method  ((dispm-display-spec-method type display-name))
			     result dispm-method-spec))
  msg    ((draw             dispm-display-spec-draw)
	  (edit             dispm-display-spec-edit)
	  (size             ((size (display-method self)))) ))

 ) ; glispobjects

; test data
(setq mytwo (list mybox myboxb))
(setq mytwob '((blue 2 (yellow 4 nil)) (violet 7 nil)))
(setq myfour (list mytwo mytwob))

; 22 Nov 91; 27 Nov 91; 06 Dec 91; 09 Dec 91; 12 Dec 91
; Get a display method spec from a type and display name (possibly nil).
(defun dispm-display-spec-method (type name)
  (let (ds)
    (or (and type
	     (or (and (eq type 'integer) (eq name 'thermom)
		      *dispm-thermom-edit*)
	         (and name (assoc name (gldisplays type)))
		 (and (setq ds (dispm-type-editor type name))
		      (dispm-display-spec-method (car ds) (cadr ds)))
		 (and name (some #'(lambda (super)
				     (dispm-display-spec-method super name))
				 (glsupers type)))))
	*dispm-anything-print*) ))

; 15 Oct 91; 04 Nov 91; 16 Feb 04
; Make a display method given a type and example of the data
(gldefun make-display-method
    ((type gltype) &optional example (size vector) (name symbol))
  (let ((spec dispm-method-spec))
    (dispm-reset nil size name)
    (if size
	(if (spec = (assoc name (gldisplays type)))
	    ((size spec) = size)))
    (dispm-edit-display-method type name example)  ))

; 15 Oct 91; 12 Dec 91; 19 Feb 99
; Clear data for creation of a new display method
(gldefun dispm-reset
     ((specs (listof dispm-item-spec)) (size vector) (name symbol))
  (*dispm-specs*  = specs)
  (*dispm-bounds* = (or size (a vector with x = 0 y = 0)))
  (*dispm-name*   = name) )


; 15 Oct 91; 17 Oct 91; 18 Oct 91; 04 Nov 91; 06 Nov 91; 25 Nov 91; 27 Nov 91
; 19 Dec 91; 02 Dec 93; 24 Dec 93; 19 Feb 99; 28 Feb 02; 16 Feb 04
(gldefun dispm-edit-display-method ((type gltype) &optional (name symbol) exmp)
  (prog (sel (spec dispm-method-spec) (source glnametype) item pos region siz
	     (redraw t))
top (dispm-open-window type)
    (*dispm-example* = exmp)
    (spec = (assoc name (gldisplays type)))
    (if spec
	(progn (dispm-reset (item-specs spec) (size spec) (name spec))
	       (*dispm-example* = (or exmp (example spec))) ))
    (if (zerop (x *dispm-bounds*))
	(*dispm-offset* =
	    (a vector with x = ((x *dispm-window-size*) - 200) / 2
	                   y = (y *dispm-window-size*) / 2))
        (*dispm-offset* =
            (a vector with x = ((x *dispm-window-size*)
				   - (x *dispm-bounds*) - 200) / 2
                              y = ((y *dispm-window-size*)
			            - (y *dispm-bounds*)) / 2)) )
    (for item in *dispm-specs* do
	 (dispm-item-draw item *dispm-window* (example item)
		          (x *dispm-offset*) (y *dispm-offset*)) )
    (if (zerop (x *dispm-bounds*))
	(draw-crosshairs-xy *dispm-window*
			    (x *dispm-offset*) (y *dispm-offset*))
        (draw-box-xy *dispm-window* (x *dispm-offset*) (y *dispm-offset*)
			  (x *dispm-bounds*) (y *dispm-bounds*)))
    (force-output *dispm-window*)
 lp (sel = (menu-set-select *dispm-menu-set* redraw))
    (redraw = nil)
    (case (menu-name sel)
      (command
        (case (port sel)
	  (done (close *dispm-window*)
		(return (dispm-make-result type name)))
	  (quit (close *dispm-window*) (return))
	  (move (if (item = (dispm-find-item))
		    (progn (pos = (get-box-position *dispm-window*
				     (x (size item)) (y (size item))))
			   (dispm-erase-item item)
			   ((offset item) = (pos - *dispm-offset*)
			                    + (a vector x 0 y 2))
			   (dispm-item-draw item *dispm-window*
					     (example item)
					     (x *dispm-offset*)
					     (y *dispm-offset*)) ) ))
	  (resize (if (item = (dispm-find-item))
		      (progn (region = (get-region *dispm-window*
						   (x (size item))
						   (y (size item)) ) )
		           (dispm-erase-item item)
			   ((offset item) = ((start region)
					       - *dispm-offset*))
			   ((size item) = (size region))
			   (dispm-item-draw item *dispm-window*
					       (example item)
					       (x *dispm-offset*)
					       (y *dispm-offset*)) ) ))
	  (delete (if (item = (dispm-find-item))
		      (progn (*dispm-specs*  = (delete item *dispm-specs*))
			     (if spec (item-specs spec) = *dispm-specs*)
			     (go top) )))
	  (restart (*dispm-specs*  = nil)
		   (if spec (item-specs spec) = nil)
		   (go top))
	  (boundary (xor-box-xy *dispm-window*
				   (x *dispm-offset*) (y *dispm-offset*)
				   (x *dispm-bounds*) (y *dispm-bounds*))
		    (siz = (get-box-size *dispm-window* (x *dispm-offset*)
					   (y *dispm-offset*) ))
		    (if spec ((size spec) = siz))
		    (*dispm-bounds* = siz)
		    (go top))
	  (if ) ) )
      (source
        (setq source (assoc (port sel) *dispm-sources*))
	(if (((first source) == 'self) and ((second source) == type))
	    (dispm-get-option source type *dispm-example*)
	    (dispm-get-option source type
			      (glsendb *dispm-example* type
				       (port sel) 'msg) ) ) ))
    (go lp)   ))


; 15 Oct 91; 17 Oct 91; 18 Oct 91; 04 Nov 91; 22 Nov 91; 25 Nov 91; 04 Dec 91
; 19 Dec 91; 20 Dec 91; 24 Jan 92; 01 May 92; 02 Dec 93; 25 Dec 93; 28 Feb 02
; 16 Feb 04
(gldefun dispm-get-option ((source glnametype) (toptype gltype) example)
  (let ((path glcodetypeexample) choice basics methods method-names
        (method dispm-method-spec) off (siz vector) reg
	item-spec var-bindings)
    (basics = '(("Print" . print)
		  ("Print w/Label" . print-w/label)
		  ("Print in Box" . print-in-box)
		  (" ... w/Label" . box-w/label)) )
    (path = (dispm-get-path source example))
    (if path
	(progn (methods = (dispm-get-display-methods (type path)))
      (method-names = (for x in methods collect (name x)))
      (choice = (menu (cons '("Quit" . quit)
			      (if ((type path) == 'string)
				  basics
			          (append basics method-names)))))
      (if choice <> 'quit
	(progn
	  (method = (or (assoc choice methods)
			  (assoc choice *dispm-basic-methods*)))
	  (siz = (or (size method)
		       (funcall (sizefn method)
				(or (example path) (example method))
				*dispm-window*)))
	  (off = (get-box-position *dispm-window* (x siz) (y siz)) )
	  (if ((scalable method) == 'x)
	    (progn (reg = (adjust-box-side *dispm-window* (x off) (y off)
					     (x siz) (y siz) 'right) )
	           (off = (start reg))
		   (siz = (size reg))))
	  (item-spec =
	    (a dispm-item-spec with
	       display-spec = (a dispm-display-spec with
				 type = (type method)
				 display-name = (name method) )
	       offset = (off - *dispm-offset*)
	       size = siz
	       sourcetype = toptype
	       example = (example path)
	       pathcode = (code path)
	       pathtype = (glreptype (type path))
	       args = (nconc var-bindings
			     (list (cons 'name
					 (if (consp source)
					     (stringify (name source))
					     "self") ) ))
	       subdisplay = nil ))
	  (push item-spec *dispm-specs*)
	  (dispm-item-draw item-spec *dispm-window* (example path)
			   (x *dispm-offset*) (y *dispm-offset*)) )) ))))


; 15 Oct 91; 16 Oct 91; 18 Oct 91; 02 Dec 93; 19 Feb 99; 23 Feb 99
(gldefun dispm-init-window ()
  (result window)
  (let ()
    (dispm-init-basic-methods)
    (*dispm-window* = (window-create (car *dispm-window-size*)
				       (cadr *dispm-window-size*)
				       "Make Display Method")) ))

; 18 Oct 91
; Find an existing item by mouse selection.
(gldefun dispm-find-item ()
  (result dispm-item-spec)
  (let (pos pt)
    (pos = (window-get-point *dispm-window*))
    (pt = (pos - *dispm-offset*))
    (that *dispm-specs* with (contains? (that dispm-item-spec) pt)) ))

; 18 Oct 91
; Erase an existing item.
(gldefun dispm-erase-item ((item dispm-item-spec))
  (erase-area *dispm-window* ((offset item) + *dispm-offset*)
	                     (size item)) )

; 15 Oct 91; 02 Dec 93; 07 Jan 94
(gldefun dispm-open-window ((source gltype))
  (let ()
    (or *dispm-window* (dispm-init-window))
    (open *dispm-window*)
    (clear *dispm-window*)
    (*dispm-menu-set* = (menu-set-create *dispm-window* nil))
    (add-menu *dispm-menu-set* 'command nil "Commands"
	      '(("Done" . done) ("Quit" . quit) ("Move" . move)
		("Resize Item" . resize) ("Delete Item" . delete)
		("Restart" . restart) ("Boundary" . boundary) ("If" . if))
	      '(0 0))
    (adjust *dispm-menu-set* 'command 'top nil 1)
    (adjust *dispm-menu-set* 'command 'right nil 2)
    (dispm-source-menu source)
    (adjust *dispm-menu-set* 'source 'top nil 1)
    (adjust *dispm-menu-set* 'source 'right 'command 10) ))

; 15 Oct 91; 19 Dec 91; 02 Dec 93; 07 Jun 94
; Make a source menu for a type.  A source menu includes computed properties.
(gldefun dispm-source-menu ((source gltype))
  (let (items offs siz)
    (*dispm-sources* = (dispm-get-sources source))
    (items = (for pair in *dispm-sources*
		    when (not (member (name pair) '(displayprops)))
		    collect (name pair)))
    (add-menu *dispm-menu-set* 'source nil source items '(0 0)) ))

; 19 Dec 91; 23 Feb 99
(gldefun dispm-get-sources ((source gltype))
  (result (listof glnametype))
  (or (append (gevgetnames source t) (glviewnames source))
      (list (a glnametype with name = 'self type = source))))

; 16 Oct 91; 07 Jun 94; 28 Feb 02; 16 Feb 04
; Make a menu to get properties of TYPE with filter FILTER.
; Derived from gevpropmenu
(gldefun dispm-prop-menu ((type gltype) (filter symbol))
  (let ((props (listof glnametype)) sel pnames)
    (props = (gevgetnames type filter))
    (if props
	(progn
	  (pnames = (for x in props
			 when (not (member (name x)
					   '(shortvalue displayprops)))
			 collect (name x)))
	  (sel = (menu (cons '("Quit" . quit)
			     (cons '("Done" . done) pnames))))
	  (if (sel == 'quit or sel == 'done or sel == nil)
	      sel
	      (assoc sel props)) ) )))


; 16 Oct 91; 17 Oct 91; 24 Dec 93; 28 Feb 02; 16 Feb 04
(gldefun dispm-get-path ((source glnametype) example)
  (result glcodetypeexample)
  (let (path done type (sel glnametype))
    (if (and (consp source) (not (eq (name source) 'self)))
        (push (name source) path))
    (type = (type source))
    (while ~done do
      (sel = (dispm-prop-menu type t))
      (if (sel == 'quit or sel == 'done or sel == nil)
	  (done = t)
	  (progn (push (name sel) path)
		 (if example
		   (progn
		     (example = (glsendb example type (name sel) 'msg))
		     (if (example == 'glsendfailure) (example = nil))))
		 (type = (type sel)) ) ))
    (if (sel <> 'quit)
        (list (dispm-path-to-code path) type example)) ))

; 16 Oct 91; 16 Feb 04
(gldefun dispm-path-to-code ((lst (listof symbol)))
  (if lst
      (list (first lst) (dispm-path-to-code (rest lst)))
      'object))

; 16 Oct 91
(gldefun dispm-get-display-methods ((type gltype))
  (result (listof dispm-method-spec))
  (let (methods)
    (methods = (gldisplays type))
    (for super in (glsupers type) do
	 (methods = (append methods (dispm-get-display-methods super))))
    methods))

; 16 Oct 91; 17 Oct 91; 18 Oct 91; 21 Oct 91; 04 Nov 91; 22 Nov 91; 03 Dec 91
; 04 Dec 91; 16 Feb 04
(gldefun dispm-make-result ((type gltype) (name symbol))
  (let (code (display dispm-method-spec) fnname fncode binds
	     newmethod diss olddis)
    (for spec in *dispm-specs* do
	 (display = (assoc (display-name (display-spec spec))
			     (gldisplays (type (display-spec spec)))))
	 (binds = (nconc (list (cons 'object (pathcode spec))
				 (cons 'params (if (params display)
						   (list 'quote
							 (params display))))
				 (cons 'offsetx
				       (list '+ (x (offset spec)) 'offsetx))
				 (cons 'offsety
				       (list '+ (y (offset spec)) 'offsety))
				 (cons 'sizex	 (x (size spec)))
				 (cons 'sizey	 (y (size spec))))
			   (args spec)))
	 (code = (nconc code (sublis binds (code display))))
;        following code "shrink-wraps" the items -- might use elsewhere
;	 (sizex = (max sizex ((x (offset spec)) + (x (size spec)))))
;	 (sizey = (max sizey ((y (offset spec)) + (y (size spec)))))
           )
    (code = (nconc code (list (list 'force-output 'w))))
    (fnname = (if name
		  (dispm-make-symbol type "-" name)
		  (gensym (dispm-make-symbol type "-DISPLAY-"))))
    (fncode = (list 'gldefun fnname
		      (list (dispm-make-symbol '(object  type)) '(w window)
			    '(offsetx integer) '(offsety integer)
			    '(sizex integer) '(sizey integer))
		      (cons 'let (cons nil code))))
    (eval fncode)
    (newmethod = (a dispm-method-spec with
		      name = name
		      type = type
		      drawfn = fnname
		      sizefn = nil
		      size = (a vector with x = (x *dispm-bounds*)
				            y = (y *dispm-bounds*))
		      example = *dispm-example*
		      canned-code = nil
		      item-specs = *dispm-specs*))
    (diss = (gldisplays type))
    (if (olddis = (assoc name diss))
	(diss = (delete olddis diss)))
    (setf (gldisplays type) (nconc diss (list newmethod)))
    fnname ))

(defun dispm-make-symbol (&rest l)
  (let (lstr)
    (dolist (x l)
      (setq lstr (if lstr (concatenate 'string lstr (stringify x))
		          (stringify x))))
    (intern lstr)))

; 18 Oct 91; 21 Oct 91; 06 Nov 91; 25 Nov 91; 26 Nov 91; 04 Dec 91; 13 Dec 91
; 23 Feb 99; 16 Feb 04
(gldefun dispm-item-draw ((item dispm-item-spec) (w window) data
			     (offsetx integer) (offsety integer))
  (let (method)
    (if (method = (display-method item))
	(if ((drawfn method) and ~ (canned-code method))
	    (if (params method)
		(funcall (drawfn method) data w
				 (params method)
				 (offsetx + (x (offset item)))
				 (offsety + (y (offset item)))
				 (x (size item)) (y (size item)))
	        (funcall (drawfn method) data w
				 (offsetx + (x (offset item)))
				 (offsety + (y (offset item)))
				 (x (size item)) (y (size item))))
	    (apply (caar (code method))
		   (mapcar #'(lambda (x)
			       (case x
				 (object data)
				 (offsetx (offsetx + (x (offset item))))
				 (offsety (offsety + (y (offset item))))
				 (w w)
				 (sizex (x (size item)))
				 (sizey (y (size item)))
				 (t (if (symbolp x)
					(cdr (assoc x (args item)))
				        (if (and (consp x)
						 (consp (cdr x))
						 (eq (cadr x) 'object))
					    (glsendb data
						     (pathtype item)
						     (car x) 'msg))))))
			      (cdar (code method)))) ) ) ))

; 16 Oct 91; 17 Oct 91; 03 Dec 91; 04 Dec 91; 09 Dec 91; 10 Dec 91; 12 Dec 91;
; 13 Dec 91; 17 Dec 91; 23 Dec 91; 24 Dec 91; 23 Feb 99
(gldefun dispm-init-basic-methods (&optional (doit boolean))
  (if (doit or (null *dispm-basic-methods*))
      (*dispm-basic-methods* =
        (list
	  (setq *dispm-anything-print*
	    (a dispm-method-spec with name = 'print type = 'anything
	       drawfn = 'dispm-anything-print
	       sizefn = 'dispm-print-size
	       size = '(42 14)
	       scalable = 'x
	       example = "String" ) )
	  (a dispm-method-spec with name = 'print type = 'string
	     drawfn = 'dispm-print
	     sizefn = 'dispm-print-size
	     size = nil
	     scalable = 'x
	     example = "String" )
	  (a dispm-method-spec with name = 'print-w/label type = 'string
	     drawfn = 'dispm-print-w/label
	     sizefn = 'dispm-print-w/label-size
	     size = nil
	     scalable = 'x
	     example = "String"
	     canned-code = 
	   '((dispm-print-w/label object w offsetx offsety sizex sizey name)) )
	  (a dispm-method-spec with name = 'print-in-box type = 'string
	     drawfn = 'dispm-print-in-box
	     size = nil
	     sizefn = 'dispm-print-in-box-size
	     scalable = 'x
	     example = "String" )
	  (a dispm-method-spec with name = 'box-w/label type = 'string
	     drawfn = 'dispm-box-w/label
	     size = nil
	     sizefn = 'dispm-box-w/label-size
	     scalable = 'x
	     example = "String"
	     canned-code = 
	   '((dispm-box-w/label object w offsetx offsety sizex sizey name)))
	  (a dispm-method-spec with name = 'arrow type = 'vector
	     drawfn = 'dispm-arrow
	     size = nil
	     sizefn = 'dispm-arrow-size
	     example = '(4 3)
	     canned-code = 
	   '((dispm-arrow object w offsetx offsety (x object) (y object))))
	  (a dispm-method-spec with name = 't/f type = 'boolean
	     size = '(20 20)
	     drawfn = 'draw-pick-one
	     editfn = 'edit-pick-one
	     direct-subedit = t
	     params = '(("T" . t) ("F" . nil))
	     example = t )
	  (a dispm-method-spec with name = 't/nil type = 'boolean
	     size = '(34 20)
	     drawfn = 'draw-pick-one
	     editfn = 'edit-pick-one
	     direct-subedit = t
	     params = '(("T" . t) ("NIL" . nil))
	     example = t )
	  (a dispm-method-spec with name = 'y/n type = 'boolean
	     size = '(20 20)
	     drawfn = 'draw-pick-one
	     editfn = 'edit-pick-one
	     direct-subedit = t
	     params = '(("Y" . t) ("N" . nil))
	     example = t )
	  (a dispm-method-spec with name = 'yes/no type = 'boolean
	     size = '(30 20)
	     drawfn = 'draw-pick-one
	     editfn = 'edit-pick-one
	     direct-subedit = t
	     params = '(("Yes" . t) ("No" . nil))
	     example = t )
	  (a dispm-method-spec with name = 'b/w type = 'boolean
	     size = '(20 20)
	     drawfn = 'draw-black-white
	     editfn = 'edit-black-white
	     direct-subedit = t
	     params = '((1 . t) (0 . nil))
	     example = t)
	  (a dispm-method-spec with name = 'w/b type = 'boolean
	     size = '(20 20)
	     drawfn = 'draw-black-white
	     editfn = 'edit-black-white
	     direct-subedit = t
	     params = '((0 . t) (1 . nil))
	     example = t)
	  (a dispm-method-spec with name = 'short type = 'integer
	     size = '(50 20)
	     drawfn = 'draw-integer
	     direct-subedit = t
	     params = nil
	     example = 123)
	  (a dispm-method-spec with name = 'short type = 'real
	     size = '(60 20)
	     drawfn = 'draw-real
	     direct-subedit = t
	     params = nil
	     example = 123.456)
	  ))
      (for spec in *dispm-basic-methods* do (dispm-def-method spec))
      (setq *dispm-thermom-edit*
	    (a dispm-method-spec with name = 'thermom type = 'integer
	       size = '(150 250)
	       editfn = 'edit-thermom
	       example = 72 ))     ))

(defun dispm-def-method (spec) (pushnew spec (gldisplays (cadr spec))
					:test #'equal))

(gldefun dispm-label-size ((w window))
  (or *dispm-label-size*
      (*dispm-label-size* = (string-width w "LABELXX: ")) ) )

(gldefun dispm-print (obj (w window) offsetx offsety sizex sizey)
  (printat-xy w (dispm-string-limit w obj sizex) offsetx offsety)   )

; 09 Dec 91; 10 Dec 91; 23 Dec 91
; Display function for use when a more specific one is not found.
(gldefun dispm-anything-print (obj (w window) offsetx offsety sizex sizey)
  (let ((s (stringify obj)) swidth smax dx dy)
    (erase-area-xy w offsetx offsety sizex sizey)
    (swidth = (string-width w s))
    (smax = (min swidth sizex))
    (dx = (sizex - smax) / 2)
    (dy = (max 0 ((sizey - 10) / 2)))
    (printat-xy w (dispm-string-limit w obj smax)
		(offsetx + dx) (offsety + dy))
   ))

(gldefun dispm-print-size (obj (w window))
  (a vector with x = (string-width w obj) y = 13))

(gldefun dispm-print-w/label
	 (obj (w window) offsetx offsety sizex sizey (label string))
  (printat-xy w ((displ-string-limit label 7) + ":") offsetx offsety)
  (printat-xy w (dispm-string-limit w obj (sizex - (dispm-label-size w)))
	        (offsetx + (dispm-label-size w)) offsety)   )

(gldefun dispm-print-w/label-size (obj (w window))
  (a vector with x = ((string-width w obj) + (dispm-label-size w))  y = 11))

; 24 Dec 93
(gldefun dispm-print-in-box (obj (w window) offsetx offsety sizex sizey)
  (printat-xy w (dispm-string-limit w obj (sizex - 4))
	      (offsetx + 4) (offsety + (sizey - 10) / 2))
  (draw-box-xy w offsetx offsety sizex sizey)  )

; 24 Dec 93
(gldefun dispm-update-in-box (obj (w window) offsetx offsety sizex sizey)
  (erase-area-xy w (offsetx + 3) (offsety + 3) (sizex - 6) (sizey - 6))
  (printat-xy w (dispm-string-limit w obj (sizex - 4))
	      (offsetx + 4) (offsety + (sizey - 10) / 2)) )

(gldefun dispm-print-in-box-size (obj (w window))
  (a vector with x = ((string-width w obj) + 8) y = 21))

(gldefun dispm-box-w/label
	 (obj (w window) offsetx offsety sizex sizey (label string))
  (printat-xy w ((displ-string-limit label 7) + ":")
	        (offsetx + 4) (offsety + 5))
  (printat-xy w (dispm-string-limit w obj (sizex - (dispm-label-size w) - 4))
	        (offsetx + 4 + (dispm-label-size w)) (offsety + 5))
  (draw-box-xy w offsetx offsety sizex sizey)  )

(gldefun dispm-box-w/label-size (obj (w window))
  (a vector with x = ((string-width w obj) + (dispm-label-size w) + 8)
                 y = 21))

; 23 Feb 99
(gldefun dispm-arrow (obj (w window) offsetx offsety dx dy)
  (draw-arrow-xy w offsetx offsety (offsetx + dx) (offsety + dy) 1 10) )

(gldefun dispm-arrow-size ((obj vector) (w window)) obj)

; 29 Jul 94
(gldefun dispm-color-bar ((color integer) (w window) offsetx offsety sizex sizey)
  (let (half)
    (half = (sizey / 2))
    (window-set-color w (dispm-rgb color))
    (window-draw-line-xy w offsetx (offsety + half)
			   (offsetx + sizex) (offsety + half) sizey)
    (window-reset-color w) ))

; 29 Jul 94
(gldefun dispm-rgb ((color integer))
  (let (red green blue)
    (red = color / 65536)
    (color = color - red * 65536)
    (green = color / 256)
    (blue = color - green * 256)
    (list (* red 256) (* green 256) (* blue 256)) ))

; 16 Feb 04
; Limit string to a specified number of characters
(gldefun displ-string-limit ((s string) (max integer))
  (result string)
  (if (> (length s) max) (subseq s 0 max) s))

; 28 Oct 91; 24 Dec 93; 16 Feb 99; 16 Feb 04
; Limit string to a specified number of pixels
(gldefun dispm-string-limit ((w window) (s string) (max integer))
  (result string)
  (let ((str (stringify s)) (lng integer) (nc integer))
    (lng = (string-width w str))
    (if (lng > max)
	(progn (nc = (truncate ((length str) * max)  lng))
	       (setq str (subseq str 0 nc))
	       (if (and (numberp s) (> nc 0))
		   (setf (char str (1- nc)) #\*))))
    str) )

; 22 Nov 91
(gldefun dispm-items-display (data (w window)
			      (offsetx integer) (offsety integer)
			      (sizex integer) (sizey integer)
			      (items (listof dispm-item-spec)))
  (let ()
    (for item in items do (draw item w (item-eval item data)
				   offsetx offsety))
 ))

; 22 Nov 91; 12 Dec 91; 13 Dec 91; 16 Feb 04
(gldefun dispm-item-eval ((item dispm-item-spec) data)
  (let ((path (pathcode item)))
    (if (not (consp path))
	data
        (if (eq (cadr path) 'object)
	    (if (symbolp (sourcetype item))
		(glsendd data (sourcetype item) (car path))
	        (if (and (consp (sourcetype item))
			 (eq (first (sourcetype item)) 'listof))
		    (funcall (car path) data)) )
	    (progn (if (not (getfncache item))
		       (dispm-item-make-getfn item))
		   (funcall (getfncache item) data) ) ) )))

; 22 Nov 91; 13 Dec 91; 16 Feb 04
(gldefun dispm-item-store ((item dispm-item-spec) data value)
  (let ((path (pathcode item)))
    (if (consp path)
	(if (eq (cadr path) 'object)
	    (if (symbolp (sourcetype item))
		(glsendd data (sourcetype item)
			 (dispm-add-colon (car path)) value)
	        (if (and (consp (sourcetype item))
			 (eq (first (sourcetype item)) 'listof))
		    (eval (list 'setf (list (car path) data)
				value))) )
	    (progn (if (not (putfncache item))
		       (dispm-item-make-putfn item))
		   (funcall (putfncache item) data value) ) ) )))

; 22 Nov 91
(gldefun dispm-item-make-getfn ((item dispm-item-spec))
  (let (fnname fncode)
    (fnname = (glmkatom 'glfn)) 
    (fncode = (list 'gldefun fnname
		      (list (dispm-make-symbol '(object  (sourcetype item))))
		      (pathcode item)))
    (eval fncode)
    ((getfncache item) = fnname)
    fnname ))

; 22 Nov 91
(gldefun dispm-item-make-putfn ((item dispm-item-spec))
  (let (fnname fncode)
    (fnname = (glmkatom 'glfn)) 
    (fncode = (list 'gldefun fnname
		      (list (dispm-make-symbol 'object\: (sourcetype item))
			    (dispm-make-symbol 'val\: (pathtype item)) )
		      (list (pathcode item) '= 'val)))
    (eval fncode)
    ((putfncache item) = fnname)
    fnname ))

(defun dispm-add-colon (sym)
  (intern (concatenate 'string (symbol-name sym) ":")))

; 04 Nov 91; 06 Nov 91; 16 Feb 04
; Select from multiple edit items within a window.
(gldefun edit-group-select ((eg edit-group) (w window) )
  (prog ((res edit-item) resb)
    (draw eg w)
 lp (setq res
	  (window-track-mouse
	   w
	   #'(lambda (x y code)
	       (or (and (> code 0) code)
		   (that item with (contains-xy (that edit-item) x y))))))
    (if (numberp res)
	(case (menu '(("Quit" . quit)))
	  (quit (return))
	  (t (go lp)))
        (if (setq resb (funcall (selectfn res) (menu-object res)))
	    (return (list (name res) resb))
	    (go lp)) ) ))

; 27 Nov 91; 09 Dec 91; 24 Dec 93
(gldefun dispm-display-spec-draw ((d dispm-display-spec) data (w window)
				 (offsetx integer) (offsety integer)
				 &optional (sizex integer) (sizey integer))
  (draw (display-method d) data w offsetx offsety sizex sizey))

; 26 Nov 91; 09 Dec 91
; Erase an existing item.
(gldefun dispm-method-spec-erase ((d dispm-method-spec) (w window)
				    (offsetx integer) (offsety integer)
				 &optional (sizex integer) (sizey integer))
  (let (szx szy)
    (szx = (min (or sizex 9999) (or (x (size d)) 40)))
    (szy = (min (or sizey 9999) (or (y (size d)) 20)))
    (erase-area-xy w offsetx offsety szx szy) ))

; 26 Nov 91; 03 Dec 91; 05 Dec 91; 09 Dec 91; 19 Dec 91; 24 Dec 93; 05 Jan 94
; 28 Jul 94; 16 Feb 04
; Draw a display consisting of multiple items.
(gldefun dispm-method-spec-draw ((d dispm-method-spec) data (w window)
				 (offsetx integer) (offsety integer)
				 &optional (sizex integer) (sizey integer))
  (let (szx szy)
    (szx = (min (or sizex 9999) (or (x (size d)) 40)))
    (szy = (min (or sizey 9999) (or (y (size d)) 20)))
  ;  (erase d w offsetx offsety szx szy)
    (if (drawfn d)
	(if (params d)
	    (funcall (drawfn d) (params d) data w  ; params moved 7/94
		     offsetx offsety szx szy)
	    (funcall (drawfn d) data w offsetx offsety szx szy))
        (for item in (item-specs d) do
	     (draw item (item-eval item data) w offsetx offsety))) ))

; 03 Dec 91; 04 Dec 91; 09 Dec 91; 19 Dec 91; 23 Dec 91; 24 Dec 93; 26 Dec 93
; 05 Jan 94; 22 Jul 94; 16 Feb 04
(gldefun dispm-display-spec-edit ((d dispm-display-spec) data (w window)
				  (offsetx integer) (offsety integer)
				  &optional (sizex integer) (sizey integer))
  (let (dm res szx szy)
    (dm = (display-method d))
    (szx = (or sizex (x (size dm))))
    (szy = (or sizey (y (size dm))))
    (if (editfn dm)
	(if (or (res = (if (params dm)
			   (funcall (editfn dm) (params dm) data w
                                      ; put (params dm) first 7/22/94
				    offsetx offsety szx szy)
			   (funcall (editfn dm) data w
				    offsetx offsety szx szy)))
		(direct-subedit dm))
	    (list res))
        (dispm-composite-edit d data w offsetx offsety) ) ))

; 27 Nov 91; 02 Dec 91; 12 Dec 91; 13 Dec 91; 19 Dec 91; 24 Dec 93
; 16 Feb 04
; Draw and edit a display consisting of multiple items.
(gldefun dispm-composite-edit ((d dispm-display-spec) data (w window)
				 (offsetx integer) (offsety integer))
  (prog (dm button highlighted inside outside subtrack (res anything) subed
	 selected-(item dispm-item-spec) items)
    (dm = (display-method d))
    (draw dm data w offsetx offsety)
 lp (button = nil)
    (window-track-mouse w
      #'(lambda (x y code)
	  (if selected-item
	      (if ~ (contains-xy selected-item
				 (x - offsetx) (y - offsety))
		  (progn (if highlighted
			     (progn
			       (unhighlight selected-item w offsetx offsety)
			       (highlighted = nil)))
			 (selected-item = nil))))
	  (if ~ selected-item
	      (progn (items = (item-specs dm))
		     (if (selected-item =
			   (that item with (contains-xy
					   (that dispm-item-spec)
					   (x - offsetx) (y - offsety))))
		      (progn (highlight selected-item w offsetx offsety)
			     (highlighted = t)
			     (subtrack = (not (basic selected-item))) ) )))
	  (outside = (   (x < offsetx)
			or (x > (offsetx + (or (x (size dm)) (width w))))
			or (y < offsety)
			or (y > (offsety + (or (y (size dm)) (height w)))) ))
	  (inside = inside or ~ outside)
	  (if (> code 0) (button = code))
	  (or button (and inside outside) subtrack) ))
    (if outside (go out))
    (if (eql button 3)
	(case (menu '(pop quit))
	  (quit (setq res 'quit) (go out))
	  (pop (setq res nil) (go out))
	  (t (go lp))))
    (if button 
	(if selected-item
	    (if (subed = (or (subdisplay selected-item)
			     (dispm-type-editor (pathtype selected-item)
						'edit)))
	        (res = (dispm-subed subed
				    (item-eval selected-item data)
				    w
				    (offsetx + (x (offset selected-item)))
				    (offsety + (y (offset selected-item)))
				    d button
				    (x (size selected-item))
				    (y (size selected-item)) ) )
	        (if (glbasictypep (pathtype selected-item))
		    (if (eql button 2)
			(res = (dispm-typein
				 (item-eval selected-item data))))
		    (res = (gev-edit (item-eval selected-item data)
				     (pathtype selected-item))) ) )
	      (if ((subed = (dispm-type-editor (type dm) 'edit))
		      and (subed <> d))
		  (res = (dispm-subed subed data w offsetx offsety
						d button
						(x (size dm))
						(y (size dm)) ) )
		  (res = (gev-edit data (type dm))))) )
    (if (selected-item and subtrack)
	(res = (edit (display-spec selected-item)
			    (item-eval selected-item data)
			    w
			    (offsetx + (x (offset selected-item)))
			    (offsety + (y (offset selected-item)))) ) )
    (if res
	(if (consp res)
	    (if selected-item
		(progn (item-store selected-item data (car res))
		       (res = (list data)) )))
        (go lp))
 out
    (if highlighted
	(progn (unhighlight selected-item w offsetx offsety)
	       (highlighted = nil)))
    (if (consp res)
	(progn (draw dm data w offsetx offsety)
	       (if selected-item
		   (progn (selected-item = nil)
			  (inside = nil)
			  (go lp)))))
    (return res) ))

; 06 Dec 91; 07 Dec 91; 10 Dec 91; 15 Dec 91; 19 Dec 91; 20 Dec 91; 24 Dec 93
; 29 Jul 94; 28 Feb 02; 16 Feb 04
(gldefun dispm-subed ((subd dispm-display-spec) data (oldw window)
		      (offsetx integer) (offsety integer)
		      (d dispm-display-spec) button
		      (sizex integer) (sizey integer))
  (let ((res anything) dm newsizex newsizey xoff yoff oldsiz w)
    (if (dm = (display-method subd))
	(if ((subd == d) or (direct-subedit dm))
	    (res = (dispm-display-spec-edit subd data oldw
					offsetx offsety sizex sizey))
	    (progn (newsizex = (max (x (size dm)) 20))
		      (newsizey = (max (y (size dm)) 20))
		      (xoff = (max 0 ((offsetx + sizex / 2) - newsizex / 2)))
		      (yoff = (max 0 ((offsety + sizey / 2) - newsizey / 2)))
		      (w = (if (and (member subd '((integer thermom))
					      :test #'equal)
				      ((newsizex + xoff) <= (width oldw))
				      ((newsizey + yoff) <= (height oldw)))
			       oldw
			       (dispm-make-subwindow oldw xoff yoff
						   newsizex newsizey)))
		      (res = (dispm-display-spec-edit subd data w
				 (max 0 ((sizex - (x (size dm))) / 2))
				 (max 0 ((sizey - (y (size dm))) / 2))))
		      (if (w <> oldw) (destroy w))))
        (if (glbasictypep (type subd))
	    (if (eql button 2)
		(res = (list (dispm-typein data))))
	    (gev-edit data (type subd))))
    res ))

; 27 Nov 91
; Highlight an item by drawing a temporary box outside its border.
(gldefun dispm-item-highlight ((ds dispm-item-spec) (w window)
			       (offsetx integer) (offsety integer))
  (window-xor-box-xy w (offsetx + (x (offset ds)) - 4)
		       (offsety + (y (offset ds)) - 4)
		       ((x (size ds)) + 8)   ((y (size ds)) + 8))
  (force-output w) )

; 02 Dec 91; 03 Dec 91; 11 Dec 91; 12 Dec 91; 13 Dec 91; 20 Dec 91; 23 Dec 91
; 16 Feb 04
; Find the display editor associated with a given type
(gldefun dispm-type-editor ((type gltype) &optional (version symbol))
  (result dispm-display-spec)
  (let (disps (ed dispm-method-spec) tmp)
    (or (and version
	     (setq tmp (assoc type '( (boolean ((short (boolean t/f))
						(edit  (boolean t/f)) ))
				      (number  ((edit (integer thermom)) ))
				      (integer ((edit (integer thermom)) ))
				      (real    ((edit (integer thermom)) )) )))
	     (cadr (assoc version (cadr tmp))))
	(progn (disps = (gldisplays type))
	       (if disps and
		   (ed = (or (assoc version disps)
			       (assoc 'display disps)
			       (and (not (glbasictypep type))
				    (find-if #'(lambda (x)
						 (not (member (car x)
							 '(listof-display))))
					     disps))))
		   (a dispm-display-spec with type = type
		      display-name = (name ed)) ) ) ) ))

; 02 Dec 91
; Trivial editor: type in a new value
(defun dispm-typein (oldval)
  (let (val)
    (format t "~%Old value is: ~A    Enter new value: " oldval)
    (setq val (read))
    (list val)))

; 07 Dec 91; 20 Dec 91; 29 Jul 94; 16 Feb 04
; Make a subwindow, attempting to put it at offsetx, offsety within
; the parent window oldw.
; If parent window is too small, a new window is created.
(gldefun dispm-make-subwindow ((oldw window) (offsetx integer) (offsety integer)
					    (sizex integer)   (sizey integer))
  (result window)
  (let (w xoff yoff)
    (xoff = (max 0 (min offsetx ((width oldw) - sizex))))
    (yoff = (max 0 (min offsety ((height oldw) - sizey))))
    (if (and ((sizex + xoff) <= (width oldw))
	     ((sizey + yoff) <= (height oldw)))
	(w = (window-create sizex sizey nil (parent oldw) xoff yoff))
        (w = (window-create sizex sizey)))
    w ))

; 12 Dec 91; 16 Feb 04
; Make a display method for an element of a listof type
(gldefun dispm-listof-display ((type gltype) (example (listof anything)))
  (result dispm-display-spec)
  (let (ds dm siz item-spec)
    (if (not (assoc 'listof-display (gldisplays type)))
	(progn (if (ds = (dispm-type-editor type 'short))
		 (progn (dm = (display-method ds))
			(siz = (size dm)))
		 (progn (ds = (a dispm-display-spec with type = 'anything
			                            display-name = 'print))
			(siz = (dispm-estimate-listof-size example))))
	     (item-spec =
			(a dispm-item-spec with
			   display-spec = ds
			   offset = (a vector with x = 0 y = 0)
			   size = siz
			   sourcetype = (list 'listof type)
			   example = (first example)
			   pathcode = (list 'car 'object)
			   pathtype = type))
	     (*dispm-specs* = (list item-spec))
	     (*dispm-example* = (list (first example)))
	     (*dispm-bounds* = siz)
	     (dispm-make-result type 'listof-display) ))
    (a dispm-display-spec with type = type
                               display-name = 'listof-display) ))

; 12 Dec 91
; Estimate size of print display for a list of items
(gldefun dispm-estimate-listof-size (lst)
  (let ((est 3) lng (n 10) item)
    (while (lst and (n > 0) and (est < 10))
      do (n _- 1)
         (item -_ lst)
         (est = (min 10 (max est (length (stringify item))))) )
    (list (est * 7) 14) ))

; 19 Dec 91
(defun dispm-define-display (dm)
  (let (dis old)
    (setq dis (gldisplays (cadr dm)))
    (setf (gldisplays (cadr dm))
	  (cons dm (if (setq old (assoc (first dm) dis))
		       (delete old dis)
		       dis))) ))
