; equations.lsp            Gordon S. Novak Jr.           ; 20 Oct 10

; Copyright (c) 2010 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Process equations

; 26 Sep 95; 01 Mar 96; 02 Jan 97; 18 Mar 98; 29 May 98; 22 Dec 98; 30 Dec 98
; 13 Jan 99; 14 Jan 99; 02 Feb 99; 04 Feb 99; 09 Feb 99; 11 Feb 99; 18 Feb 99
; 11 Mar 99; 16 Mar 99; 19 Mar 99; 25 Mar 99; 02 Apr 99; 16 May 01; 28 Feb 02
; 03 Jan 03; 08 Jan 03; 09 Jan 03; 22 Oct 03; 15 Jan 04; 11 Feb 04; 12 Feb 04
; 16 Feb 04; 18 Feb 04; 19 Feb 04; 02 Mar 04; 11 Mar 04; 19 Mar 04; 26 Mar 04
; 29 Mar 04; 01 Apr 04; 08 Apr 04; 12 Apr 04; 14 Apr 04; 16 Apr 04; 20 Apr 04
; 23 Apr 04; 28 Apr 04; 29 May 04; 07 Jun 04; 08 Jun 04; 07 Apr 05; 19 Apr 05
; 11 May 05; 06 Dec 05; 08 Dec 05; 14 Dec 05; 28 Dec 05; 23 Jan 06; 26 Jan 06
; 27 Jan 06; 16 May 06; 29 Jun 06; 06 Jul 06; 02 Aug 06; 09 Aug 06; 12 Sep 06
; 09 Oct 06; 30 Oct 06; 13 Feb 07; 15 Feb 07; 19 Feb 07; 21 Feb 07; 01 Mar 07
; 02 Apr 07; 11 Apr 07; 12 Apr 07; 16 Apr 07; 22 May 07; 01 Jun 07; 11 Oct 07
; 16 Oct 07; 23 Jan 08; 30 Jan 08; 26 Jul 08; 22 Sep 08; 07 Nov 08; 10 Nov 08
; 21 Nov 08; 24 Nov 08; 12 Jan 09; 30 Oct 09; 02 Nov 09; 16 Jul 10

(defvar *eqns-trace* nil)  ; set to t to trace equation processing.
(defvar *equations-constants* nil)
(defvar *equations-solved* nil)
(defvar *equations-units* nil)
(defvar *equations-history* nil)

(glispobjects

(equation-set
  (list (equations        (listof anything))  ; remaining unsolved equations 
        (solved-vars      (listof symbol))    ; vars that are solved or defined
        (solved-equations (listof anything))  ; equations that have been solved
        (defined-vars     (listof symbol))    ; vars that are defined
        (deleted-tuples   (listof symbol))    ; deleted tuple vars
        (all-equations    (listof anything))) ; all equations
  prop  ((delete gleqns-delete-eqn open t) )
  adj   ((empty  ((null all-equations)))) )

) ; glispobjects

(pushnew '(mercator mercator)  *conn-user-laws*)

; 12 Apr 07
; Test for constants, omitting T (often used for time in equations).
(defun gleqnconstantp (x) (and (not (eq x t)) (glconstantp x)))

; Process equations to create views for variants of a type

; 21 Apr 92; 22 Apr 92; 05 Dec 92; 27 Feb 94
; Example: (glispobjects (mycir (listobject (diameter real))))
;          (gleqnstoprops 'mycir (get 'circle 'equations))
; Produce a list of PROP's for a GLISP type based on a set of equations
(defun gleqnstoprops (type equations)
  (let (solved newprops (progress t) vars unsolved var neweqn)
    (setq solved
          (nconc (mapcar #'car (gevdatanames type t))
                 (mapcar #'car (gevpropnames type 'prop t))))
    (while (and progress equations)
      (if (consp progress)
          (setq equations (set-difference equations progress)))
      (setq progress nil)
      (dolist (eqn equations)
        (setq vars (remove-if #'gleqnconstantp (glvarsin eqn)))
        (setq unsolved (set-difference vars solved))
        (when (and (consp unsolved) (null (cdr unsolved)))  ; 1 var unsolved
          (setq var (first unsolved))
          (when (setq neweqn (glsolvefor eqn var))
            (push eqn progress)
            (push var solved)
            (push (list var (glptmatch (last neweqn) 'glpatterns))
                  newprops))) ) )
    (nreverse newprops)))

; 29 Jun 06
; Combine equation sets to derive new equations for an object
; vars is a list of variables that are defined
; supers is a list of principles that apply to this object
;    each super is a symbol, or a list (symbol bindings)
;    where each binding is (var-in-target var)
; Example: (glcombineeqns '(circumference weight) '(physob sphere))
(defun glcombineeqns (vars supers)
  (let ((progress t) alleqns eqns bindings)
    (while progress
      (setq progress nil)
      (dolist (super supers)
        (if (consp super)
            (progn (setq bindings (cadr super))
                   (setq super (car super)))
            (setq bindings nil))
        (setq eqns (gleqns-solveeqns
                     (gleqns-renamevars vars bindings)
                     (gleqns-equations super)))
        (when eqns
          (setq progress t)
          (setq eqns (sublis (gleqns-tosublis bindings) eqns))
          (setq alleqns (append alleqns eqns))
          (setq vars (union vars (mapcar #'cadr eqns))) ) ) )
    alleqns ))

; 29 Jun 06
; Get equations defined for a concept
(defun gleqns-equations (x) (equations x))

; 29 Jun 06
; Rename variables given a bindinng list
(defun gleqns-renamevars (vars bindings)
  (let ()
    (mapcar #'(lambda (var)
                (or (car (find var bindings :key #'cadr))
                    var))
            vars)))

; 29 Jun 06
; Convert a list of lists to list of conses for sublis
(defun gleqns-tosublis (lst)
  (mapcar #'(lambda (x) (cons (car x) (cadr x))) lst))

; 29 Jun 06
; Convert a list of equations to list of GLisp PROP's
(defun gleqns-toprops (eqns)
  (mapcar #'(lambda (eqn) (list (cadr eqn) (list (caddr eqn))))
          eqns))

; 29 Jun 06; 12 Apr 07
; Solve a set of equations given a set of known variables
; Result is a list of equations
(defun gleqns-solveeqns (vars eqns)
  (let ((progress t) result dvars eqn eqnb)
    (setq eqns (subset #'(lambda (x) (not (gleqns-tuplep x))) eqns))
    (while (and progress eqns)
      (setq progress nil)
      (when (setq eqn
                  (some #'(lambda (x)
                            (and (setq dvars
                                       (subset #'(lambda (x)
                                                   (not (gleqnconstantp x)))
                                               (set-difference (varsin x)
                                                               vars)))
                                 (= (length dvars) 1)
                                 x))
                            eqns))
        (setq eqns (remove eqn eqns))
        (setq eqnb (glsolvefor eqn (car dvars)))
        (push eqnb result)
        (push (car dvars) vars)
        (setq progress t) ))
    (nreverse result) ))

; 29 Jun 06
; Test for a tuple equation, (= var (tuple ...))
(defun gleqns-tuplep (eqn)
  (and (consp eqn) (eq (car eqn) '=)
       (cdr eqn) (consp (cddr eqn))
       (eq (caaddr eqn) 'tuple)))

; 06 Jul 06
; vars = variables that are to be externally visible
; eqnsets = equation sets
; conns = connections between variables of equation sets
(defun glcombineeqnsets (vars eqnsets conns)
  (let ()
  ))

; 21 Apr 92; 27 Apr 94; 07 Oct 94; 11 Apr 07; 16 Apr 07
; Solve for a given variable in a lisp formula.  cf. solvefor in ISAAC
(defun glsolvefor (form var)
  (let ()
    (setq form (glfixequation form))
    (or (glsimplesolvefor form var)
        (glsimplesolvefor (gleqns-simplify form) var)
    ; or try to regroup terms, simplify, and try again
 ) ))

; 27 Apr 94
(defun glsimplesolvefor (form var)
  (let (res)
    (if (eq (car form) '=)
        (progn
          (setq res (if (eq (cadr form) var)
                        form
                        (if (gloccurs var (caddr form))
                            (glinvert (cadr form) (caddr form) var))))
          (if (not (gloccurs var (caddr res)))
              res)))))

; 22 Apr 92; 11 Nov 92; 28 Feb 94; 29 Apr 94; 02 Mar 04; 02 Apr 04; 19 Apr 05
; 06 Dec 05; 12 Sep 06
; Invert an equation, lhs = rhs, to find a specified var
(defun glinvert (lhs rhs var)
  (if (atom rhs)
      (if (eq rhs var) (list '= rhs lhs))
      (case (first rhs)
        (+ (or (glinvert (list '- lhs (second rhs)) (third rhs)  var)
               (glinvert (list '- lhs (third rhs))  (second rhs) var)))
        (* (if (equal (second rhs) (third rhs))
               (glinvert lhs (list 'expt (second rhs) 2) var)
               (or (glinvert (list '/ lhs (second rhs)) (third rhs) var)
                   (glinvert (list '/ lhs (third rhs))  (second rhs) var))))
        (- (if (cddr rhs)
               (or (glinvert (list '- (second rhs) lhs) (third rhs)  var)
                   (glinvert (list '+ lhs (third rhs))  (second rhs) var))
               (glinvert (list '- lhs) (second rhs) var)))
        (/ (or (glinvert (list '/ (second rhs) lhs) (third rhs)  var)
               (glinvert (list '* lhs (third rhs))  (second rhs) var)))
        ((sqrt |sqrt|)  (glinvert (list 'expt lhs 2) (second rhs) var))
        ((cbrt |cbrt|)  (glinvert (list 'expt lhs 3) (second rhs) var))
        ((log2 |log2|)  (glinvert (list 'expt 2 lhs) (second rhs) var))
        ((log10 |log10|)  (glinvert (list 'expt 10 lhs) (second rhs) var))
        ((log |log|)  (if (and (cddr rhs) (numberp (caddr rhs)))
                          (glinvert (list 'expt (caddr rhs) lhs)
                                    (second rhs) var)
                          (glinvert (list 'exp lhs) (second rhs) var)))
        ((float |float|) (glinvert lhs (second rhs) var))
        ((atan |atan|)
          (if (cddr rhs)
              (or (glinvert (list '/ (second rhs) (list 'tan lhs))
                            (third rhs) var)
                  (glinvert (list '* (third rhs) (list 'tan lhs))
                            (second rhs) var))
              (glinvert (list 'tan lhs) (second rhs) var)))
        ((sin |sin| cos |cos| tan |tan| asin |asin| acos |acos|
              exp |exp|)
          (glinvert (list (cadr (assoc (first rhs)
                                       '((sin asin) (asin sin)
                                         (|sin| asin) (|asin| sin)
                                         (cos acos) (acos cos)
                                         (|cos| acos) (|acos| cos)
                                         (tan atan) (|tan| atan)
                                         (|exp| log) (exp log) ) ))
                          lhs)
                    (second rhs) var))
        ((expt |expt|) 
               (if (eql (third rhs) 2)
                   (glinvert (list 'sqrt lhs) (second rhs) var)
                   (if (eql (third rhs) 3)
                       (glinvert (list 'cbrt lhs) (second rhs) var)
                       (if (eql (second rhs) 2)
                           (glinvert (list 'log2 lhs) (third rhs) var)
                           (if (numberp (second rhs))
                               (if (eql (second rhs) 10)
                                   (glinvert (list 'log10 lhs)
                                             (third rhs) var)
                                   (glinvert (list 'log lhs (second rhs))
                                             (third rhs) var)))))) )) ))

; 02 Oct 92; 20 Oct 92; 03 Nov 92; 03 Dec 92; 19 Mar 93; 22 Feb 94; 12 Apr 07
; Make a list of the unsolved variables in a formula.
(defun glunsolvedvars (form solved &optional unsolved)
  (if (atom form)
      (if (and form (symbolp form))
          (if (or (member form solved)
                  (member form unsolved)
                  (gleqnconstantp form))
              unsolved
              (cons form unsolved))
          unsolved)
      (if (eq (car form) 'quote)
          unsolved
          (dolist (subexp (cdr form) unsolved)
            (setq unsolved (glunsolvedvars subexp solved unsolved))) ) ) )

;------------------------------------------------------------------------------
; Code to solve sets of simultaneous equations

; Example: cannonball problem
(setq *gleqns-cannon*
      '((= fulltime (* time 2))
        (= time (/ yvel 9.81))         ; time to peak altitude
        (= xvel (* vel (cos elev)))
        (= yvel (* vel (sin elev)))    ; initial y velocity
        (= dist (* xvel fulltime))
        (= vel 300)
        (= dist 8000)))
; (gleqns-solve *gleqns-cannon*)
; (gleqns-solve '((= s (+ x y)) (= d (- x y))) '(s d))
; (gleqns-solve '((= volume (* (* pi (expt radius 2)) length))   ; cylinder
;                 (= area (* (* pi (* 2 radius)) length))) '(volume area))

; 22 Feb 94; 27 Feb 94; 02 Nov 09
; Attempt to solve a set of simultaneous equations
; eqns    = list of equations
; solved  = variables considered to be given or already solved
; desired = variables whose solutions are desired
; Result is a list of equations for computing desired variables
(defun gleqns-solve  (eqns &optional solved)
  (let (eqn res)
    (if (and eqns (null (rest eqns))
             (setq eqn (gleqns-solve-eqn (first eqns) solved)))
        (if (null (set-difference (glvarsin (caddr eqn)) solved))
            (list (gleqns-betternumify eqn)))
        (dolist (eqn eqns)
          (if (setq res (gleqns-solve-reduce eqn eqns solved))
              (return-from gleqns-solve res) ) ) ) ))

; 27 Feb 94; 02 Nov 09
; Try to solve an equation set by solving eqn, a member of eqns, for a
; single variable, substituting in the remaining equations, and
; solving the reduced equation set.
(defun gleqns-solve-reduce (eqn eqns solved)
  (let (neweq neweqns newset)
    (when (setq neweq (gleqns-solve-eqn eqn solved))
      (dolist (eq eqns)
        (unless (eq eq eqn) (push (subst (caddr neweq) (cadr neweq) eq)
                                  neweqns)))
      (if (setq newset (gleqns-solve neweqns solved))
          (cons (gleqns-betternumify
                  (gleqns-simplify
                    (sublis (mapcar #'(lambda (eq) (cons (cadr eq) (caddr eq)))
                                    newset)
                            neweq)))
                newset)) ) ))

; 27 Feb 94; 01 Mar 94; 07 Oct 94
; Try to solve a single equation for a desired variable
(defun gleqns-solve-eqn (eqn solved)
  (let (seqn)
    (or (and (symbolp (cadr eqn))
             (not (member (cadr eqn) solved))
             (not (gloccurs (cadr eqn) (caddr eqn)))
             eqn)
        (and (symbolp (caddr eqn))
             (not (member (caddr eqn) solved))
             (not (gloccurs (caddr eqn) (cadr eqn)))
             (list '= (caddr eqn) (cadr eqn)))
        (progn (setq seqn (gleqns-simplify eqn))
               (some #'(lambda (var) (gleqns-solvefor seqn var))
                     (set-difference (glvarsin seqn) solved))) ) ))

; 27 Feb 94; 23 Apr 04
; Try to solve an equation for a variable
(defun gleqns-solvefor (eqn var)
  (let (neweqn)
    (if (or (setq neweqn (glsolvefor eqn var))
            (setq neweqn (glsolvefor
                          (list '= 0 (gleqns-simplify (list '- (cadr eqn)
                                                            (caddr eqn))))
                          var)))
        (list '= (cadr neweqn)
                 (gleqns-simplify (eqn-optprod (caddr neweqn)))) ) ))
       

; 22 Feb 94; 24 Feb 94; 26 Apr 94
; Attempt to simplify a formula
(defun gleqns-simplify (form) (gleqns-trans form 'math))

; 02 Nov 09
; Improve the final form of an equation by making a numeric factor more readable
(defun gleqns-betternumify (form)
  (let (e)
    (setq e (rhs form))
    (if (and (consp form)
             (eq (op form) '=)
             (consp e))
        (if (and (eq (op e) '*)
                 (numberp (lhs e)))
            (list (op form) (lhs form) (list (op e) (betternum (lhs e)) (rhs e)))
            form)
        form) ))
      

;------------------------------------------------------------------------------

; 28 Dec 05
; attempt a numerical solution for a variable in a formula ; see mathfns.lsp
; (glsolveform '(= (expt x 2) 2) 'x)
; (glsolveform '(= (sin x) (/ x 2)) 'x)
(defun glsolveform (form var)
  (regula-falsib (list '- (cadr form) (caddr form)) var))

; 14 Dec 05; 28 Dec 05; Won 19 Jan 06
; Attempt to solve a quadratic equation for variable
; (glsolvequad '(= (* 1/2 (* a (expt t 2))) s) 't)
(defun glsolvequad (form var &optional minus)
  (let (factors newform)
    (setq newform
          (if (and (eq (car form) '=)
                   (eql (cadr form) 0))
              (caddr form)
              (if (and (eq (car form) '=)
                       (eql (caddr form) 0))
                  (cadr form)
                  (list '- (cadr form) (caddr form)))))
    (setq factors (mapcar #'gleqns-simplify (polyfact newform var)))
    (and factors (cdr factors) (cddr factors)
         (not (and (numberp (caddr factors)) (zerop (caddr factors))))
         (list '= var (glquadsol factors minus))) ))

; 14 Dec 05; 28 Dec 05; 16 Apr 07
; construct a solution to quadratic given factors
; (glquadsol '(c b a)) ; for a*x^2 + b*x + c = 0
(defun glquadsol (factors &optional minus)
  (let ((a (caddr factors)) (b (cadr factors)) (c (car factors))
        (op (if minus '- '+)))
    (if (or (and (numberp a) (minusp a))
            (and (consp a) (eq (car a) '-) (null (cddr a)))
            (and (consp a) (member (car a) '(* /))
                 (numberp (cadr a)) (minusp (cadr a))))
        (progn (setq a (gleqns-simplify (list '- a)))
               (setq b (gleqns-simplify (list '- b)))
               (setq c (gleqns-simplify (list '- c)))) )
    (gleqns-simplify
      `(/ (,op (- ,b) (sqrt (- (expt ,b 2) (* 4 (* ,a ,c))))) (* 2 ,a)) ) ))

; 15 Oct 07
; Try to solve a quadratic, preferring a solution that is
; likely to be positive.
(defun glsolvequadb (form var alist &optional minus)
  (let (sola solb rhsa rhsb preferred other denomvar denomval)
    (setq sola (glsolvequad form var))
    (setq solb (glsolvequad form var t))
    (if (and sola solb)
        (progn
          (setq preferred sola)
          (setq rhsa (third sola))
          (setq rhsb (third solb))
          (if (and (numberp rhsa) (numberp rhsb))
              (if (> rhsb rhsa) (setq preferred solb))
              (if (and (consp rhsa)
                       (eq (car rhsa) '/)
                       (setq denomvar (glsolvequadbxtr (caddr rhsa)))
                       (setq denomval (assoc denomvar alist))
                       (numberp (cdr denomval)))
                   (if (< (cdr denomval) 0) (setq preferred solb))))
          (setq other (if (eq preferred sola) solb sola))
          (if minus other preferred))
        (if minus solb sola) ) ))

; 15 Oct 07
; extract a symbol multiplied by constant
(defun glsolvequadbxtr (expr)
  (if (symbolp expr)
      expr
      (if (consp expr)
          (if (eq (car expr) '*)
              (if (numberp (cadr expr))
                  (glsolvequadbxtr (caddr expr))
                  (if (numberp (caddr expr))
                      (glsolvequadbxtr (cadr expr))))
              (if (eq (car expr) '/)
                  (if (numberp (caddr expr))
                      (glsolvequadbxtr (cadr expr))))))))

; 14 Dec 05; Won 19 Jan 06; 23 Jan 06; 26 Jan 06
; find polynomial factors (c x x^2) of a quadratic expression in var
; (polyfact '(+ (* a (expt x 2)) (+ (* b x) c)) 'x)  ; = (C B A)
(defun polyfact (exp var)
  (let (tmp)
    (if (eq exp var)
        (list 0 1 0)
        (if (or (atom exp) (not (gloccurs var exp)))
            (list exp 0 0)
          (case (car exp)
            (+ (polyfact+ (polyfact (cadr exp) var)
                          (polyfact (caddr exp) var)))
            (- (if (cddr exp)
                   (polyfact- (polyfact (cadr exp) var)
                              (polyfact (caddr exp) var))
                   (polyfactminus (polyfact (cadr exp) var))))
            (* (polyfact* (polyfact (cadr exp) var)
                          (polyfact (caddr exp) var)))
            (/ (if (gloccurs var (caddr exp))
                   (error "Bad expr in polyfact ~A~%" exp)
                   (polyfact* (polyfact (cadr exp) var)
                              (list (list '/ 1 (caddr exp)) 0 0))))
            ((expt |expt|)
              (if (eql (caddr exp) 2)
                  (progn (setq tmp (polyfact (cadr exp) var))
                         (polyfact* tmp tmp))
                (error "Bad expr in polyfact ~A~%" exp)))
            (t (error "Unrecognized operator expr in polyfact ~A~%" exp)))))))

; 14 Dec 05
(defun polyfact+ (facts factsb)
  (mapcar #'(lambda (x y)
              (if (eql x 0)
                  y
                  (if (eql y 0)
                      x
                      (list '+ x y)))) facts factsb))

; 14 Dec 05
(defun polyfact- (facts factsb)
  (mapcar #'(lambda (x y)
              (if (eql x 0)
                  (list '- y)
                  (if (eql y 0)
                      x
                      (list '- x y)))) facts factsb))

; 14 Dec 05
(defun polyfactminus (facts)
  (mapcar #'(lambda (x) (if (eql x 0) 0 (list '- x))) facts))

; 14 Dec 05
(defun polyfact* (facts factsb)
  (let (pow1 pow2)
    (setq pow1 (polyfact*1 (cadr facts) factsb))
    (setq pow2 (polyfact*1 (caddr facts) factsb))
    (polyfact+ (polyfact*1 (car facts) factsb)
               (polyfact+ (list 0 (car pow1) (cadr pow1))
                          (list 0 0 (car pow2)))) ))

; 14 Dec 05
(defun polyfact*1 (fact facts)
  (mapcar #'(lambda (x)
              (if (or (eql fact 0)
                      (eql x 0))
                  0
                  (if (eql fact 1)
                      x
                      (if (eql x 1)
                          fact
                          (list '* fact x)))))
          facts))

;------------------------------------------------------------------------------

; Code for manipulating sets of equations for use by makev / vip

; Define parts of an equation set:
(defmacro eqns-equations        (eqns) `(first ,eqns))
(defmacro eqns-solved-vars      (eqns) `(second ,eqns))
(defmacro eqns-solved-equations (eqns) `(third ,eqns))
(defmacro eqns-defined-vars     (eqns) `(fourth ,eqns))
(defmacro eqns-deleted-tuples   (eqns) `(fifth ,eqns))
(defmacro eqns-all-equations    (eqns) `(sixth ,eqns))

; Example:      (setq *eqns-trace* t)
; (setq eqset (gleqns-init-equations (get 'line-segment 'equations)))
; (gleqns-var-defined eqset 'p1x)
; (gleqns-var-defined eqset 'length)
; (gleqns-var-defined eqset 'theta)
; (gleqns-var-defined eqset 'p2y)
; eqset

; 03 Oct 92; 12 Oct 92; 30 Oct 92; 18 Nov 92; 19 Nov 92
; Initialize equation set for use with gleqns-var-defined
(defun gleqns-init-equations (eqns)
  (list (copy-list eqns) nil nil nil nil eqns) )

; 02 Nov 92; 19 Nov 92
(defun gleqns-delete-eqn (eqset eqn)
  (setf (eqns-equations eqset) (delete eqn (eqns-equations eqset))) )

; 19 Mar 93
; Find equation whose left-hand side defines var
(defmacro gleqns-findeq (var eqnlist)
  `(find-if #'(lambda (x) (eq ,var (cadr x))) ,eqnlist))

; 19 Mar 93
; Find the definition of var among solved equations
(defun gleqns-def (var eqns)
  (caddr (gleqns-findeq var (eqns-solved-equations eqns))))

; 19 Mar 93
; Find the definition of var among all equations
(defun gleqns-alldef (var eqns)
  (caddr (gleqns-findeq var (eqns-all-equations eqns))))

; 03 Oct 92; 07 Oct 92; 12 Oct 92; 20 Oct 92; 30 Oct 92; 02 Nov 92; 16 Nov 92
; 05 Dec 92; 05 Mar 93; 14 Mar 93; 15 Mar 93; 16 Mar 93; 28 Mar 93; 01 Apr 93
; 22 Feb 94; 27 Feb 94; 11 Mar 99
; When a var becomes defined, examine the equation set to see if any
; properties defined by the equations have become defined.
; Returns list of vars that have become defined, solved, or invalid,
; and therefore can no longer be independently specified.
; (A tuple becomes invalid if one of its components is defined.)
(defun gleqns-var-defined (eqns var)
  (let ((vars (list var)) (progress t) uns neweqn newvars)
    (if *eqns-trace*
        (format t "1. Enter var-defined, var = ~A ~%" var))
    (pushnew var (eqns-solved-vars eqns))
    (pushnew var (eqns-defined-vars eqns))
    (while progress
      (setq progress nil)
      (setq newvars nil)
      (dolist (eqn (eqns-equations eqns))
        (setq uns (glunsolvedvars eqn (eqns-solved-vars eqns)))
        (if (null uns)
            (progn (if *eqns-trace*
                       (format t "  2b. deleting eqn   ~A ~%" eqn))
                   (gleqns-delete-eqn eqns eqn)))
        (if (intersection uns (eqns-deleted-tuples eqns))
            (progn (if *eqns-trace*
                       (format t "  2d. deleting eqn   ~A ~%" eqn))
                   (gleqns-delete-eqn eqns eqn)))
        (if (and uns
                 (null (cdr uns))                   ; exactly one unsolved
                 (not (member (first uns) newvars))
                 (not (and (consp (caddr eqn))
                           (eq (caaddr eqn) 'tuple)))
                 (setq neweqn (glsolvefor eqn (first uns))))
            (progn (setq neweqn (list (first neweqn) (second neweqn)
                                      (glptmatch (third neweqn) 'glpatterns)))
                   (if *eqns-trace*
                       (if (equal eqn neweqn)
                         (format t "  2a. solved eqn     ~A~%" eqn)
                         (format t
                           "  2a. solved eqn     ~A~%          giving     ~A~%"
                           eqn neweqn)))
                   (push (second neweqn) newvars)
                   (push (second neweqn) vars)
                   (gleqns-delete-eqn eqns eqn)
                   (push neweqn (eqns-solved-equations eqns))
                   (setq progress t)))
        (if (and uns
                 (consp (caddr eqn))
                 (eq (caaddr eqn) 'tuple)
                 (intersection (glvarsin (caddr eqn))
                               (eqns-solved-vars eqns)))
            (progn (if *eqns-trace* (format t "  2c. deleting tuple ~A~%" eqn))
                   (push (cadr eqn) vars)
                   (push (cadr eqn) (eqns-deleted-tuples eqns))
                   (gleqns-delete-eqn eqns eqn))) )
      (when progress
        (if *eqns-trace* (format t "  3. repeating step 2.~%" ))
        (setf (eqns-solved-vars eqns)
              (nconc newvars (eqns-solved-vars eqns))) ) )
    (if *eqns-trace* (format t "4. exit, vars ~A ~%" vars))
    vars))


; 05 Mar 93; 11 Mar 99; 16 Mar 99
; Make an equation set assuming basis variables are the ones that are stored.
(defun gleqns-basis-eqns (goal)
  (let (eqns)
    (when (get goal 'equations)
      (setq eqns (gleqns-init-equations (get goal 'equations)))
      (dolist (var (gleqns-basis goal)) (gleqns-var-defined eqns var))
      eqns) ))

; 09 Feb 94; 10 Feb 94; 22 Dec 98; 30 Dec 98
; Make equations for a box consisting of an operator
(defun gleqns-op-equations (op)
  (let (nargs)
    (setq nargs (if (gleqns-unaryp op) 1 (or (conn-nargs op) 2)))
    (list (list '= 'out
                (cons op
                      (if (= nargs 1)
                          (list 'in)
                          (butlast '(in in2 in3 in4 in5 in6 in7)
                                   (- 7 nargs)))))) ))

; 10 Feb 94
(defun gleqns-prop-equations (prop) (list (list '= 'out (list prop 'in))))

; 09 Feb 94; 29 Apr 94; 26 Sep 95
(defun gleqns-unaryp (fn) (member fn '(sin cos tan sqrt cbrt exp log not)))

; 23 Oct 92; 06 Nov 92; 10 Nov 92; 17 Nov 92; 18 Nov 92; 05 Mar 93; 16 Mar 93
; 19 Mar 93; 25 Mar 93; 16 Mar 99; 28 Feb 02; 03 Jan 03; 08 Jan 03; 09 Jan 03
; 30 Jan 08
; Produce inverse code to store var, a basis variable of
; the goal type, into the approprate forms in a view type.
(defun gleqns-store-var (goal viewtype var eqns undef)
  (let (basis nocc xfer done (progress t) code letvars pair res sourcetype
        lhs sourcename newv newva tmp dep xfers prop)
    (if *eqns-trace* (format t "0. Entering gleqns-store-var, var ~A~%" var))
    (setq sourcename (caar (glstr viewtype)))
    (setq sourcetype (cadar (glstr viewtype)))
    (setq basis (gleqns-basis goal))
    (setq xfer (cadr (caadr (glgetprop viewtype 'prop 'gltransfernames))))
  ; determine which basis vars, other than the var to be stored, are used
  ; in computing the transfer vars
    (dolist (xvar xfer)
      (setq tmp (gleqns-depends-on eqns xvar basis nil))
      (if *eqns-trace* (format t "  1. var ~A depends on ~A~%" xvar tmp))
      (when (member var tmp)
        (push xvar xfers)
        (setq dep (union tmp dep)) ) )
    (when (and (null (intersection dep undef))
               (not (member var undef))
               (or (setq dep (delete var dep))
                   (cdr xfers)
                        ; if a plain store will work, return nil.
                   (not (and (setq prop (glgetprop viewtype 'prop (car xfers)))
                             (setq tmp (cadr prop))
                             (consp tmp)
                             (consp (car tmp))
                             (assoc (caar tmp) (gldatanames sourcetype))))))
      (setq newv (intern (concatenate 'string "VAR-"
                                      (symbol-name sourcename))))
      (setq newva (list sourcename newv))      ; code to access the var
  ; make let vars for other basis vars used in computing transfer vars
      (dolist (v dep) (push (list v (list v newva)) letvars))
      (setq done xfers)
      (dolist (var done)
        (unless (member var basis)
          (setq nocc (gleqns-countocc (gleqns-def var eqns)
                                      basis (gleqns-incocc var nocc)))))
      (while progress
        (setq progress nil)
        (dolist (pair nocc)
          (unless (member (car pair) done)
            (push (car pair) done)
            (setq progress t)
            (setq nocc (gleqns-countocc (gleqns-def (car pair) eqns)
                                        basis nocc))) ) )
      (dolist (prop (eqns-solved-equations eqns))
        (when (and (setq pair (assoc (cadr prop) nocc))
                   (> (cdr pair) 1))
          (push (cadr prop) letvars)
          (push (list (cadr prop) '=
                      (gleqns-fixcode eqns (caddr prop) nocc))
                code) ) )
      (dolist (xvar xfers)
        (setq tmp (assoc xvar (glget viewtype 'prop)))
        (setq lhs (if tmp
                      (subst newva sourcename (caadr tmp))
                      (list xvar newva)))
        (if (and (member xvar (eqns-deleted-tuples eqns))
                 (setq prop (gleqns-alldef xvar eqns))
                 (consp prop)
                 (eq (car prop) 'tuple))
            (dolist (pair (cdr prop))
              (if (member var (gleqns-depends-on eqns (cadr pair) basis nil))
                  (push (list (list (car pair) lhs) '=
                              (gleqns-fixcode eqns (cadr pair) nocc))
                        code)))
            (push (list lhs '= (gleqns-fixcode eqns xvar nocc))
                  code) ) )
      (when code
        (push var code)
        (setq res (list 'glambda (list newv var)
                        (cons 'let
                              (cons (nreverse letvars)
                                    (nreverse code)))))))  ))

; 16 Mar 99
; Get the nmaes of basis variables of a goal type
(defun gleqns-basis (goal)
  (or (get goal 'basis-vars)
      (mapcar #'car (gldatanames goal)) ) )

; 22 Oct 92; 16 Nov 92; 17 Nov 92; 21 Jan 93; 12 Apr 07
; Count occurrences of variables in an equation.  basis vars are not counted.
; nocc is an alist of (var . count).
(defun gleqns-countocc (form basis nocc)
  (let ()
    (if (atom form)
        (if (and form (symbolp form))
            (if (or (member form basis)
                    (gleqnconstantp form))
                nocc
                (gleqns-incocc form nocc))
            nocc)
        (dolist (subexp (cdr form) nocc)
          (setq nocc (gleqns-countocc subexp basis nocc)) ) ) ))

; 17 Nov 92
; Increment occurrence count
(defun gleqns-incocc (var nocc)
  (let (pair)
    (if (setq pair (assoc var nocc))
        (progn (incf (cdr pair))
               nocc)
        (cons (cons var 1) nocc)) ))

; 22 Oct 92; 30 Oct 92; 16 Mar 93
; Fix code to incorporate other equations that are used only once
(defun gleqns-fixcode (eqns form nocc)
  (let (pair)
    (if (atom form)
        (if (and (symbolp form)
                 (setq pair (assoc form nocc)))
            (if (> (cdr pair) 1)
                form
                (gleqns-fixcode eqns (gleqns-def form eqns) nocc))
            form)
        (cons (car form)
              (mapcar #'(lambda (x) (gleqns-fixcode eqns x nocc))
                      (cdr form))) ) ))

; 22 Oct 92; 30 Oct 92; 18 Nov 92; 19 Mar 93
; Find what basis vars a given property depends on
(defun gleqns-depends-on (eqns form basis dep)
  (let (prop)
    (if (atom form)
        (if (and form (symbolp form))
            (if (member form basis)
                (if (member form dep) dep (cons form dep))
                (if (setq prop (gleqns-def form eqns))
                    (gleqns-depends-on eqns prop basis dep)
                    (if (and (setq prop (gleqns-alldef form eqns))
                             (consp prop)
                             (eq (car prop) 'tuple))
                        (gleqns-depends-on eqns prop basis dep))))
            dep)
        (dolist (subexp (cdr form) dep)
          (setq dep (gleqns-depends-on eqns subexp basis dep)) ) ) ))


; 08 Nov 92; 09 Nov 92; 11 Nov 92; 17 Nov 92; 05 Mar 93; 16 Mar 93; 19 Mar 93
; 20 Dec 93; 16 Mar 99; 18 Mar 99; 19 Mar 99; 30 Jan 08
; Produce code to make an instance of the source type of a view
; from a set of values for the basis variables of the goal type.
; Result is (basis auxcode code)
;   where  basis   = basis vars of goal type
;          auxcode = (var code) to compute each auxiliary var
;          code    = (var code) for each var of source type to be stored
(defun gleqns-bfv (goal viewtype eqns)
  (let (basis sourcename sourcetype xfers done nocc progress auxcode tmp
        datanames rhscode proprhs field parts tupleqn code pair (okay t))
    (setq sourcename (caar (glstr viewtype)))
    (setq sourcetype (cadar (glstr viewtype)))
    (setq basis (gleqns-basis goal))
  ;   get transfer names = names of abstract type to be transferred
    (setq xfers (cadr (caadr (glgetprop viewtype 'prop 'gltransfernames))))
    (setq done xfers)
  ;   count occurrences of vars used in computing transfer set
    (dolist (var done)
      (unless (member var basis)
        (setq nocc (gleqns-countocc (gleqns-def var eqns)
                                    basis (gleqns-incocc var nocc)))))
  ;   recursively count occurrences of all vars used
    (setq progress t)
    (while progress
      (setq progress nil)
      (dolist (pair nocc)
        (unless (member (car pair) done)
          (push (car pair) done)
          (setq progress t)
          (setq nocc (gleqns-countocc (gleqns-def (car pair) eqns)
                                      basis nocc))) ) )
  ;   make let vars and assignment statements for intermediate vars that
  ;   are used more than once
    (dolist (prop (eqns-solved-equations eqns))
      (when (and (setq pair (assoc (cadr prop) nocc))
                 (> (cdr pair) 1))
        (push (list (cadr prop)
                    (gleqns-fixcode eqns (caddr prop) nocc))
              auxcode) ) )
  ;   get the names of data fields in the structure to be built
    (setq datanames (gldatanames sourcetype))
  ;   for each transfer var, if it corresponds to a stored field of the
  ;   goal type, make a pair for use in constructing the 'A' function.
    (dolist (var xfers)
      (setq rhscode nil)
   ;    find the PROP entry that computes this transfer var
      (setq tmp (assoc var (glget viewtype 'prop)))
   ;    if it is legit, proprhs is how to compute it, e.g. (RIGHT LS124)
      (if (and (cdr tmp)
               (consp (cadr tmp))
               (consp (setq proprhs (caadr tmp))))
     ;    if it is a stored field, set field to name and type
        (progn
          (setq field (and (eq (cadr proprhs) sourcename)
                           (assoc (car proprhs) datanames)))
     ;    make code to compute this transfer var
     ;    if it is a tuple, build the substructure
          (if (and (setq tupleqn (gleqns-alldef var eqns))
                   (eq (first tupleqn) 'tuple))
              (progn (setq parts (mapcan #'(lambda (x) (list (car x) (cadr x)))
                                       (cdr tupleqn)))
                     (if field
                   ;    if it is a stored field, make 'A' code
                         (setq rhscode (cons 'a (cons (cadr field) parts)))
                   ;    if it is a view, make special 'A' code
                       (if (and (consp (cadr proprhs))
                                (setq field
                                      (and (eq (cadadr proprhs) sourcename)
                                           (assoc (caadr proprhs) datanames))))
                           (setq rhscode
                             (cons 'a (cons (list 'typeof
                                                  (list (car proprhs)
                                                        (list 'a
                                                              (cadr field))))
                                            parts))))))
              (setq rhscode (or (gleqns-fixcode eqns var nocc)
                                var))) )
          (progn (setq field (list var))
                 (setq rhscode (gleqns-fixcode eqns var nocc)) ) )
      (if (gleqns-bfv-ok rhscode basis auxcode)
          (if field
              ;   if it is stored, make it part of the 'A' function
              (push (list (car field) rhscode) code)
              ;   else do an assignment afterwards.         ; 12/20/93
              (push (list (subst (glmkatom 'glbuildresult) sourcename proprhs)
                          rhscode) auxcode))
          (setq okay nil) ) )
    (and okay (list basis (nreverse auxcode) (nreverse code))) ))

; 30 Jan 08
; Make sure that the code to compute something is legitimate,
; i.e. that every var used is a basis var or defined as a let var
(defun gleqns-bfv-ok (rhscode basis auxcode)
  (and rhscode
       (every #'(lambda (var) (or (member var basis) (assoc var auxcode)))
              (glvarsin rhscode)) ) )

; 19 Mar 99; 02 Apr 99; 09 Jan 03; 30 Jan 08
; Generate code to build an instance of the source type of a view
; from a set of values for the basis variables of the goal type.
(defun gleqns-build-from-view (goal viewtype eqns)
  (let (bfv sourcetype auxcode code letvars acode after aftervar datanames
            basis)
    (when (setq bfv (gleqns-bfv goal viewtype eqns))
      (setq sourcetype (cadar (glstr viewtype)))
      (setq basis (car bfv))
      (setq auxcode (cadr bfv))
      (setq code (caddr bfv))
      (setq datanames (gldatanames sourcetype))
      (setq letvars (mapcar #'car auxcode))
      (dolist (pair code)
        (if (assoc (car pair) datanames)
            (progn (push (car pair) acode)
                   (push (cadr pair) acode))
            (progn (or aftervar (setq aftervar (glmkatom 'glbuildresult)))
                   (push (list (list (car pair) aftervar) '= (cadr pair))
                         after)) ) )
      (setq code (nconc (mapcar #'(lambda (x) (list (car x) '= (cadr x)))
                                auxcode)
                        (cons (if aftervar
                                  (list aftervar '=
                                        (cons 'a (cons sourcetype
                                                       (nreverse acode))))
                                  (cons 'a (cons sourcetype (nreverse acode))))
                              (nreverse after))))
      (cons 'glambda
            (cons (cons 'self basis)
                  (if letvars
                      (list (cons 'let (cons (nreverse letvars) code)))
                      code))))  ))

; 19 Mar 99; 02 Apr 99; 28 Feb 02; 03 Jan 03; 09 Jan 03; 30 Jan 08
; Generate code to store into an instance of the source type of a view
; from a variable of the goal type.
(defun gleqns-store-from-view (goal viewtype eqns undef)
  (let (bfv auxcode code letvars acode goalvar sourcename basis
            (sourcevar 'self))
    (when (setq bfv (gleqns-bfv goal viewtype eqns))
      (setq basis (car bfv))
      (setq auxcode (cadr bfv))
      (setq code (caddr bfv))
      (setq sourcename (caar (glstr viewtype)))
      (setq goalvar (glmkatom goal))
      (dolist (v basis)
        (unless (member v undef)
          (push (list v (list v goalvar)) letvars)))
      (setq letvars (nconc (nreverse letvars) (mapcar #'car auxcode)))
      (setq acode
            (nconc (mapcar #'(lambda (x)
                               (list (list (car x) (list sourcename sourcevar))
                                     '= (cadr x)))
                           auxcode)
                   (mapcar #'(lambda (x)
                               (list (list (car x) (list sourcename sourcevar))
                                     '= (cadr x)))
                           code)
                   (list sourcevar)))
      (cons 'glambda
            (cons (list sourcevar goalvar)
                  (if letvars
                      (list (cons 'let (cons (nreverse letvars) acode)))
                      code))))  ))


; 21 Dec 93; 11 Feb 94; 03 Jan 03
; Materialize a data structure of the goal type from an instance of the
; viewed type.
(defun gleqns-materialize-view (goal viewtype undef)
  (let (datanames props items eqns tmp)
    (setq datanames (gldatanames goal))
    (setq props (glget viewtype 'prop))
    (setq eqns (get goal 'equations))
    (setq items
          (mapcan #'(lambda (pair)
                      (if (and (assoc (car pair) props)
                               (not (member (car pair) undef)))
                          (list (car pair) (list (car pair) 'self))
                          (if (setq tmp
                                    (gleqns-materialize-tuple pair eqns props))
                              (list (car pair) tmp))))
                        datanames))
    (if items (list 'glambda (list 'self) (cons 'a (cons goal items)))) ))

; 11 Feb 94; 07 Oct 94
; Materialize a substructure based on TUPLE specification
; e.g. nametype = (P1 VECTOR)
;      eqns     = ((= P1 (TUPLE (X P1X) (Y P1Y))) ...)
;         returns (A VECTOR X (P1X SELF) Y (P1Y SELF))
(defun gleqns-materialize-tuple (nametype eqns props)
  (let (tupleqn tuple items tmp)
    (setq tupleqn (find-if #'(lambda (eqn) (and (eq (cadr eqn) (car nametype))
                                                (consp (caddr eqn))
                                                (eq (caaddr eqn) 'tuple)))
                           eqns))
    (when tupleqn
      (setq tuple (cdaddr tupleqn))
      (setq items
            (mapcan #'(lambda (pair)
                        (if (assoc (cadr pair) props)
                            (list (car pair) (list (cadr pair) 'self))
                            (if (setq tmp
                                    (gleqns-materialize-tuple pair eqns props))
                                (list (car pair) tmp))))
                    tuple))
      (if items (cons 'a (cons (cadr nametype) items))) ) ))

; 13 Nov 92; 16 Nov 92; 17 Nov 92; 16 Mar 93
; Create a function to transfer from one data representation to another
; according to a view shared by both representations.
(defun gleqns-transfer-by-view (goal source
                                     &optional goalviewname sourceviewname)
  (let (sourcevs goalvs sourcev goalv viewtype done datanames vartype parts
        goalvar newv newvt newvv code xfers goalview tmp)
; find views that the source and goal types have in common
    (setq sourcevs (glget source 'view-choices))
    (setq goalvs (glget goal 'view-choices))
    (if (and goalviewname (null sourceviewname))
        (setq sourceviewname goalviewname))
    (if (and goalviewname
             (setq sourcev (assoc sourceviewname sourcevs))
             (setq goalv (assoc goalviewname goalvs)))
        (if (not (eq (second sourcev) (second goalv)))
            (error "Incompatible views  between types ~A and ~A~%"
                   goal source))
        (dolist (gv goalvs)
          (if (and (not done)
                   (setq sourcev (find-if #'(lambda (x) (eq (second x)
                                                            (second gv)))
                                          sourcevs)))
              (progn (setq goalv gv) (setq done t)))) )
    (or goalv (error "No view in common between types ~A and ~A~%"
                     goal source))
    (setq datanames (gldatanames goal))
    (setq viewtype (second goalv))
    (setq goalview (or (caddr (assoc (first goalv) (glget goal 'views)))
                       (mkv goal source nil goalviewname)))
    (setq xfers (cadr (caadr (glgetprop goalview 'prop 'gltransfernames))))
    (setq newv  (intern (concatenate 'string "VAR-" (symbol-name source))))
    (setq newvt (intern (concatenate 'string (symbol-name newv)
                                             ":" (symbol-name source))))
    (setq newvv (intern (concatenate 'string "VAR-" (symbol-name source)
                                             "-VIEW")))
    (dolist (var xfers)
      (setq tmp (find-if #'(lambda (x) (eq (caar x) var))
                         (fifth goalv)))
      (setq goalvar (caadr tmp))
      (setq vartype (cadr (assoc goalvar datanames)))
      (if (glbasictypep vartype)
          (push (list var newvv) code)
          (progn (setq parts (caddr (find-if #'(lambda (x) (eq var (cadr x)))
                                             (get viewtype 'equations))))
                 (push (if (eq (car parts) 'tuple)
                           (cons 'a
                             (cons vartype
                                   (mapcan #'(lambda (x)
                                               (list (car x)
                                                     (list (cadr x) newvv)))
                                           (cdr parts)))))
                       code)))
      (push goalvar code))
    (list 'glambda (list newvt)
          (list 'let (list (list newvv (list viewtype newv)))
                (cons 'a (cons goal code)))) ))

; 19 Nov 94
; make equations from a view-choices form.
; vc = (goals sources correspondences)
(defun gleqns-from-view-choices (vc)
  (let (subs)
    (setq subs (gleqns-vc-vars vc))
    (mapcar #'(lambda (corr)
              (list '= (sublis subs (first corr) :test #'equal)
                       (sublis subs (second corr) :test #'equal)))
          (third vc) ) ))

; 19 Nov 94
; Make a substitution list for variables.
; a variable form is (var source).  If var is unique it is used,
; else it is changed to type-var.
(defun gleqns-vc-vars (vc)
  (let (sources subs)
    (setq sources (append (car vc) (cadr vc)))  ; combine sources and goals
    (dolist (corr (third vc))
      (setq subs (gleqns-vc-vars-subs-l sources corr subs)))
    subs))

(defun gleqns-vc-vars-subs-l (sources items subs)
  (if items
      (gleqns-vc-vars-subs-l sources (cdr items)
                             (gleqns-vc-vars-subs sources (car items) subs))
      subs))

(defun gleqns-vc-vars-subs (sources item subs)
  (let (pair)
    (if (consp item)
        (if (and (consp (cdr item))
                 (symbolp (cadr item))
                 (null (cddr item))
                 (setq pair (assoc (cadr item) sources))
                 (not (assoc item subs :test #'equal)))
            (cons (cons item
                        (if (some #'(lambda (x) (eq (cdr x) (car item)))
                                  subs)
                            (intern (concatenate 'string (cadr pair) "-"
                                                 (car item)))
                            (car item)))
                  subs)
            (gleqns-vc-vars-subs-l sources (rest item) subs))
        subs) ))

; 13 Jan 99; 14 Jan 99; 09 Jan 03
; simple primality test for examples
(gldefun primep ((n integer)) (result boolean)
  (and (> n 1) (integerp n)
       (or (= n 2)
           (and (oddp n)
                (let ((ndiv (floor (sqrt n))) (i 3) (prime t))
                  (while (and prime (<= i ndiv))
                    (if (= 0 (mod n i))
                        (setq prime nil)
                        (i = (+ i 2))))
                  prime)))))

(defun log10 (x) (/ (log x) (log 10.0)))
(defun log2 (x) (/ (log x) (log 2.0)))
(defun atand (y x) (* (/ 180.0 pi) (atan y x)))  ; atan in degrees
(defun flip (&optional (n 1000000)) (> (random n) (/ n 2)))
(defun choose (n k) (chooseb n 1 k 1))
(defun chooseb (n den steps answer)
  (if (zerop steps)
      answer
      (chooseb (1- n) (1+ den) (1- steps) (/ (* answer n) den)) ))
(defun factorial (n)  (if (<= n 1) 1 (* n (factorial (1- n))) ) )

; 28 Apr 94
; Cube root -- defined here so patterns using it can be tested.
; returns a negative real root for a negative argument.
(defun cbrt (x)
  (and (numberp x) (if (>= x 0) (expt x 1/3) (- (expt (- x) 1/3)))))

; 13 Feb 07
; simulate a random coin flip
(defun flipcoin (&rest stuff)
  (declare (ignore stuff))
  (< (random 1.0) 0.5))

; 20 Oct 10
; degrees to radians
(defun radians (degrees) (* degrees (/ pi 180.0)))

; 20 Oct 10
; radians to degrees
(defun degrees (radians) (* radians (/ 180.0 pi)))

(dolist (pair '((sin |sin|) (asin |asin|) (cos |cos|) (acos |acos|)
                (tan |tan|) (log |log|) (exp |exp|) (sqrt |sqrt|)
                (cbrt |cbrt|) (log2 |log2|) (log10 |log10|)
                (float |float|) (atan |atan|) (expt |expt|) ) )
  (setf (symbol-function (cadr pair)) (symbol-function (car pair))) )

; 22 Oct 03; 12 Apr 07
; based on equation eqn, how does the value of variable dvar
; change based on new values newvals (alist).  oldvals default to 1.
; The answer is adjusted for the oldval of dvar.
; "doubling the thickness of a given wire and making it 10 times longer
; will cause its resistance to be... "
; (eqnchanged '(= r (* k (/ l (* pi (expt rad 2))))) 'r '((l . 10) (rad . 2)))
; "what change to the radius will make the resistance double?"
; (eqnchanged '(= r (* k (/ l (* pi (expt rad 2))))) 'rad '((r . 2)))
(defun eqnchanged (eqn dvar newvals &optional oldvals)
  (let ((neweqn (glsolvefor eqn dvar)))
    (when neweqn
      (dolist (var (glvarsin neweqn))
        (or (assoc var oldvals)
            (gleqnconstantp var)
            (push (cons var 1) oldvals)))
      (* (cdr (assoc dvar oldvals))
         (/ (eval (sublis (append newvals oldvals) (caddr neweqn)))
            (eval (sublis oldvals (caddr neweqn))))) ) ))

; 07 Jun 04; 12 Apr 07
; determine how expr changes if dvar is doubled
; only powers of 2 are meaningful results; ignore others
(defun exprchanged (expr dvar)
  (let (vals)
    (dolist (var (glvarsin expr))
      (or (and (gleqnconstantp var) (numberp (eval var)))
          (push (cons var (random 1.0)) vals)))
    (safe-eval
      (list '/ (safe-eval (sublis (cons (cons dvar
                                              (* 2 (cdr (assoc dvar vals))))
                                        vals)
                                  expr))
                     (safe-eval (sublis vals expr)))) ))
 
; 07 Jun 04; 14 Dec 05
; Eval that avoids an error for divide by zero
(defun safe-eval (x &optional alist)
  (if (consp x)
      (let ((lhs (safe-eval (cadr x) alist))
            (rhs (safe-eval (caddr x) alist)))
        (case (car x)
          (/ (if (zerop rhs) 9999999 (/ lhs rhs)))
          (t (if (cddr x)
                 (funcall (car x) lhs rhs)
                 (funcall (car x) lhs)))))
      (if (symbolp x)
          (or (cdr (assoc x alist)) (eval x))
          (eval x) ) ) )

; 15 Jan 04
; make sure that an equation uses only binary operators,
; e.g. (+ a b c) -> (+ (+ a b) c)
(defun glbinarize (eqn)
  (if (consp eqn)
      (if (cdddr eqn)
          (glbinarize (cons (car eqn)
                            (cons (list (car eqn)
                                        (glbinarize (cadr eqn))
                                        (glbinarize (caddr eqn)))
                                  (cdddr eqn))))
          (list (car eqn)
                (glbinarize (cadr eqn))
                (glbinarize (caddr eqn))) )
      eqn))

; 13 Jan 04
; evaluate an expression given an alist of variable values
(defun evalexp (form alist)  (eval (sublis alist form)))

; 06 Feb 04; 12 Feb 04; 13 Feb 04; 18 Feb 04; 18 Mar 04; 26 Mar 04; 01 Apr 04
; 22 Apr 04; 16 May 06; 06 Jul 06; 12 Sep 06; 20 Feb 07; 21 Feb 07; 16 Apr 07
; 22 May 07; 16 Oct 07; 24 Nov 08
; Solve for desired var of a kind of object given defined
; and desired vars.
; defined is a list of items, (var (value units)) or (var value)
; e.g. (solvobjvar '((= DIAMETER (* 2 RADIUS)) (= AREA (* PI (EXPT RADIUS 2))))
;                  'radius '((area a)) )
;      (solvobjvar (get 'fall 'equations) 'time '((height (125 meter))))
;      (solvobjvar (get 'fall 'equations) 'time '((height (h meter))) nil t)
; codeflag is set to include the '(Q number unit) form in output
; quadflag can be set to use the negative solution of a quadratic
(defun solvobjvar (eqns goalvar defined &optional objtype codeflag quadflag)
  (let (eqn eqnb vars solved alist ulist sol (progress t) done tmp quad)
    (setq *equations-history* nil)
    (dolist (var defined)
      (if (consp (cadr var))              ; (<number> <units>)
          (progn (push (cons (car var) (caadr var)) alist)
                 (if (cadadr var)
                     (push (cons (car var) (cadadr var)) ulist)
                     (if (setq tmp (assumedvartype (car var) objtype))
                         (push (cons (car var) tmp) ulist))))
          (progn (push (cons (car var) (cadr var)) alist)
                 (if (setq tmp (assumedvartype (car var) objtype))
                     (push (cons (car var) tmp) ulist)) ) ) )
    (setq done (assoc goalvar alist))
    (while (and progress (not done))
      (setq progress nil)
      (setq solved (mapcar #'car alist))
      (when (setq eqn
                  (some #'(lambda (x)
                            (and (setq vars (set-difference
                                              (varsin x) solved))
                                 (= (length vars) 1)
                                 (not (and (consp (caddr x))
                                           (eq (caaddr x) 'tuple)))
                                 (not (solvboguseqn x (car vars) alist ulist))
                                 x))
                            eqns))
        (setq eqns (remove eqn eqns))
    ; try to change expr to isolate var on one side of the eqn
        (setq eqnb (glsolvefor eqn (car vars)))
        (if (null eqnb)        ; could not isolate the var
            (setq eqnb (glsolvequadb (glfixequation eqn) (car vars)
                                     alist  quadflag)))
        (if (and (consp (rhs eqnb))
                 (eq (car (rhs eqnb)) 'quote)
                 (consp (cadr (rhs eqnb)))
                 (eq (caadr (rhs eqnb)) 'q))
            (if (not (eq (lhs eqnb) goalvar))
                (progn (push (cons (car vars)
                                   (if codeflag
                                       (rhs eqnb)
                                     (cadadr (rhs eqnb))))
                             alist)
                       (push (cons (car vars) (caddr (cadr (rhs eqnb))))
                             ulist)
                       (solveeqhist (car vars) (cadadr (rhs eqnb))
                                    (caddr (cadr (rhs eqnb))) eqn eqnb) ))
            (progn (setq sol (eqn-eval (caddr eqnb) alist ulist))
                   (push (cons (car vars) (car sol)) alist)
                   (push (cons (car vars) (cadr sol)) ulist)
                   (solveeqhist (car vars) (car sol) (cadr sol) eqn eqnb) ))
        (setq progress t)
        (setq done (assoc goalvar alist)) ))
    (setq *equations-solved* alist)
    (setq *equations-units* ulist)
    (list (cdr done) (or (cdr (assoc goalvar ulist)) 1)) ))


; 07 Feb 04; 20 Apr 04; 11 Apr 07; 12 Apr 07
; vars in a formula, omitting constants (except t) and things multiplied by 0
(defun varsin (x) (varsinb x nil))
(defun varsinb (x vars)
  (if (null x)
      vars
      (if (symbolp x)
          (if (or (member x vars)
                  (gleqnconstantp x)
                  (member x *equations-constants*))
              vars
              (cons x vars))
          (if (and (consp x)
                   (not (member (car x) '(quote function))))
              (if (or (and (eq (car x) '*)
                           (or (and (numberp (cadr x)) (zerop (cadr x)))
                               (and (numberp (caddr x)) (zerop (caddr x)))))
                      (and (eq (car x) '/)
                           (numberp (cadr x)) (zerop (cadr x))))
                  vars
                  (dolist (z (cdr x) vars) (setq vars (varsinb z vars))))
              vars) )))

; 18 Feb 04; 05 Apr 04; 16 May 06
; calculate the change in goalvar for specified changes in other vars
; as specified by the alist defined.
; defined = vars that have defined values
; changed = vars that are changed: value is the proportional change.
; Returns a list (number constvars) or nil, where constvars are held constant
; e.g. (changeobjvar 'circle nil '((radius 2)) 'area)
(defun changeobjvar (objtype defined changed goalvar)
  (let (origval newval val lst solved constvars sol origvars newvars tmp)
  ; construct a set of vars with changes made
    (dolist (item defined)
      (push (if (assoc (car item) changed)
                (list (car item) (cons (* (caadr item) (cadr changed))
                                       (cdadr item)))
                item)
            newvars) )
    (dolist (item changed)
      (unless (assoc (car item) defined)
        (if (setq tmp (caddr (assoc (car item) (vars-units objtype))))
            (progn (push (list (car item) (* tmp (cadr item))) newvars)
                   (push (list (car item) tmp) origvars))
            (progn (push item newvars)
                   (push (list (car item) 1) origvars)))))
    (setq newval (solvobjvar (get objtype 'equations) goalvar newvars objtype))
  ; see if the given info is enough to solve it
    (if (and (consp newval)
             (numberp (car newval)))
        (progn (setq origval
                     (solvobjvar (get objtype 'equations) goalvar
                                 (append defined origvars) objtype))
               (setq val (/ (car newval) (car origval)) ))
  ; see if defining one still-undefined basis var will allow solving it
        (progn (setq lst (basis-vars objtype))
               (setq solved *equations-solved*)
               (while (and (not val) lst)
                 (if (not (or (eq (car lst) goalvar)
                              (assoc (car lst) solved)))
                     (progn (setq sol (changeobjvar objtype defined
                                                    (cons (list (car lst) 1)
                                                          changed)
                                                    goalvar))
                            (setq val (car sol))
                            (if val
                                (setq constvars (cons (car lst) (cadr sol))))))
                 (setq lst (cdr lst)) ) ) )
    (list val constvars) ))

; 14 Apr 04; 22 Apr 04
; Identify the form of a change in a variable
; e.g. (meta-change 'circle nil 'radius 'area)
(defun meta-change (objtype defined changevar goalvar)
  (let (chg)
    (setq chg (changeobjvar objtype defined (list (list changevar 2)) goalvar))
    (and (consp chg)
         (numberp (car chg))
         (some #'(lambda (x)
                   (if (< (abs (- (car x) (car chg))) 0.001) (cadr x)))
               '((1/8 inverse-cube) (1/4 inverse-square) (1/2 inverse)
                 (0.70710678118654746 inverse-square-root)
                 (0.79370052598409968 inverse-cube-root)(1 invariant)
                 (1.2599210498948732 cube-root)
                 (1.4142135623730951 square-root) (2 linear) (4 square)
                 (8 cube)) ) )))

; 11 Mar 04
; calculate transitions between energy band levels
; given a list of energy band levels, returns a list of possible transitions
; e.g. 1993 #34, Princeton #12
(defun energybands (lst)
  (let (res)
    (while lst
      (dolist (x (cdr lst))
        (pushnew (abs (- (car lst) x)) res))
      (setq lst (cdr lst)) )
    res))

; 11 Mar 04; 12 Apr 04
; calculate relative differences between two quantities, each of which is
; of the form (number units)
; Returns NIL if failure, else numeric ratio
(defun reldiff (q1 q2)
  (let ((unitconv 1) n1 maxq diff tempconv)
    (when (and (consp q1) (consp q2)
               (numberp (car q1))
               (numberp (car q2)))
        (if (and (cadr q1) (cadr q2))
            (if (and (basictempunit (cadr q1))
                     (basictempunit (cadr q2))
                     (setq tempconv (tempconvert (cons '* q2) (cadr q1))))
                (setq q2 (list tempconv (cadr q1)))
                (if (and (glunitp (cadr q1)) (glunitp (cadr q2)))
                    (setq unitconv (glconvertunit (cadr q1) (cadr q2))))))
      (when unitconv
        (setq n1 (* (car q1) unitconv))
        (setq diff (- n1 (car q2)))
        (setq maxq (if (or (zerop n1) (zerop (car q2)))
                       1
                       (max (abs n1) (abs (car q2)))))
        (abs (/ diff maxq)))) ))


; 24 Mar 04; 29 Mar 04; 20 Apr 04; 19 Feb 07; 20 Feb 07; 12 Apr 07
; evaluate an expression, given alists of numeric values and units of variables
; returns a list, (<value> <units>)
; old:   returns a (Q <number> <units>) form, or a symbolic expression
(defun eqn-eval (expr vals units)
  (let (res un)
    (if (consp expr)
        (if (eq (op expr) 'q)
            (list expr (caddr expr))
            (if (and (eq (op expr) 'quote)
                     (consp (cadr expr))
                     (eq (caadr expr) 'q))
                (list (cadr expr) (caddr (cadr expr)))
                (eqn-evalq expr (eqn-eval (lhs expr) vals units)
                                (and (cddr expr)
                                     (eqn-eval (rhs expr) vals units)))))
        (if (symbolp expr)
            (if (setq res (or (assoc expr vals)
                              (and (gleqnconstantp expr)
                                   (cons expr (eval expr)))))
                (if (numberp (cdr res))
                    (list (cdr res)
                          (if (setq un (assoc expr units))
                              (cdr un)
                              'unity))
                    (list (cdr res) (cdr (assoc expr units))))
                (list expr 'unity))
            (and (numberp expr) (list expr 'unity)))) ))

; 25 Mar 04; 26 Mar 04; 02 Apr 04; 05 Apr 04; 16 Apr 04; 19 Apr 05; 16 May 06
; 30 Oct 06; 19 Feb 07; 20 Feb 07; 02 Apr 07; 01 Jun 07; 26 Jul 08
; Evaluate/create an expression
; lhsv, rhsv are (<value> <units>); returns the same form.
(defun eqn-evalq (expr lhsv rhsv)
  (let (uconv un (lhsn (car lhsv)) (lhsu (cadr lhsv))
                 (rhsn (car rhsv)) (rhsu (cadr rhsv)))
    (case (op expr)
          (* (eqn-makeq (eqn-make '* lhsn rhsn)
                        (eqn-simu (list (op expr) lhsu rhsu))))
          (/ (eqn-makeq (if (and (numberp rhsn)
                                 (= 0 rhsn))
                            1.0e33               ; in case of divide by 0
                            (eqn-make '/ lhsn rhsn))
                        (eqn-simu (list (op expr) lhsu rhsu))))
          ((+ -) (if (cddr expr)   ; test for binary op
                     (progn
                       (setq uconv
                             (if (or (equal lhsu rhsu)
                                     (equal 0 lhsn)
                                     (equal 0 rhsn))
                                 1
                                 (or (glconvertunit rhsu lhsu)
                                     (error "incompatible units ~A ~A~%"
                                            rhsu lhsu))))
                       (eqn-makeq (list (op expr) lhsn
                                        (if (= uconv 1)
                                            rhsn
                                            (eqn-make '* uconv rhsn)))
                                  (if (equal 0 lhsn) rhsu lhsu)))
                     (eqn-makeq (list (op expr) lhsn) lhsu)))
          ((expt |expt|)
             (setq un 1)
             (dotimes (i (abs rhsn))
               (setq un (glsimplifyunit (list '* un lhsu))))
             (if (< rhsn 0)
                 (setq un (list '/ 1 un)))
             (eqn-makeq (eqn-make 'expt lhsn rhsn) un))
          ((sin cos asin acos tan exp log log2 log10
                |sin| |cos| |asin| |acos| |tan| |exp| |log| |log2| |log10|)
            (if (setq un (glconvertunit lhsu 'unity))
                (eqn-makeq (list (op expr) (if (= un 1)
                                               lhsn (eqn-make '* un lhsn))) ; Won
                           (if (member (op expr) '(asin acos |asin| |acos|))
                               'radian 'unity))                ; Won/GSN
                (error "incompatible units ~A ~A~%" (op expr) lhsu)))
          ((atan |atan|)
            (if rhsv
                (if (glconvertunit rhsu lhsu)
                    (eqn-makeq (eqn-make (op expr) lhsn rhsn)
                               'radian)                            ; Won
                    (error "incompatible units ~A ~A~%"
                           rhsu lhsu))
                (eqn-makeq (list (op expr) lhsn) 'radian)) )       ; Won
          ((sqrt |sqrt|) (eqn-makeq (list 'sqrt lhsn)
                                    (glsqrtunit lhsu)))
          ((cbrt |cbrt|) (eqn-makeq (list 'cbrt lhsn)
                                    (glcbrtunit lhsu)))
          (t (error "eqn-evalq: unknown op ~A ~A ~A~%" expr lhsv rhsv)) ) ))

; 12 Apr 07
; Make an expression (value unit)
; possibly removing multiplicative constant from unit
(defun eqn-makeq (value unit)
  (let (num mul)
    (if (or (numberp value)
            (gleqnconstantp value)
            (and (consp value)
                 (every #'numberp (cdr value))))
        (progn
          (setq num (eval value))
          (if (and (consp unit)
                   (eq (car unit) '*)
                   (setq mul (eqn-numval (cadr unit))))
              (eqn-makeq (* mul (eqn-numval num))
                         (if (cdddr unit)
                             (cons '* (cddr unit))
                             (caddr unit)))
              (if (and (consp unit)
                       (eq (car unit) '/)
                       (setq mul (eqn-numval (cadr unit)))
                       (not (equal mul 1)))
                  (eqn-makeq (* mul (eqn-numval num))
                             (glsimplifyunit (list '/ 1 (caddr unit))))
                  (list num unit))))
        (list value unit) ) ))

; 25 Mar 04; 26 Mar 04; 12 Apr 07
; get the numeric value of a unit multiplier such as kilo, else nil
(defun eqn-numval (unit)
  (if (numberp unit)
      unit
      (if (symbolp unit)
          (if (gleqnconstantp unit)
              (eval unit)
              (and (eq (get unit 'glunittype) 'dimensionless)
                   (get unit 'glsiconversion))))))

; 20 Apr 04; 19 Apr 04
; make a binary op expression from components, doing partial evaluation
(defun eqn-make (op lhs rhs)
  (if (and (member op '(* / expt |expt|)) (numberp lhs) (zerop lhs))
      0
    (if (and (eq op '*) (numberp rhs) (zerop rhs))
        0
      (if (and (member op '(+ -)) (numberp rhs) (zerop rhs))
          lhs
        (if (and (eq op '+) (numberp lhs) (zerop lhs))
            rhs
          (if (and (eq op '-) (numberp lhs) (zerop lhs))
              (list '- rhs)
            (if (and (numberp lhs) (numberp rhs))
                (funcall op lhs rhs)
              (list op lhs rhs) )))))) )

; 26 Jul 08
; simplify a unit expression
(defun eqn-simu (unit)
  (if (consp unit)
      (case (car unit)
        (* (if (eq (lhs unit) 'unity)
               (if (cdddr unit)
                   (eqn-simu (cons (first unit) (cddr unit)))
                   (rhs unit))
               (if (eq (rhs unit) 'unity)
                   (if (cdddr unit)
                       (eqn-simu (cons (first unit) (cons (lhs unit)
                                                          (cddr unit))))
                       (lhs unit))
                   (glsimplifyunit unit))))
        (/ (if (eq (rhs unit) 'unity)
               (lhs unit)
               (glsimplifyunit unit)))
        (t (glsimplifyunit unit))) ) )

; 12 Apr 04; 07 Jun 04; 08 Jun 04
; choose the best answer from a multiple-choice answer set
; choices is coded
(defun bestans (answer choices)
  (let ((bestval 99999) (answers '(a b c d e)) diff ans exp bestans)
    (case (car choices)
      (num (dolist (c (cdr choices))
             (if (and (consp c) (numberp (car c))
                      (consp answer) (numberp (car answer)))
                 (progn (setq diff (reldiff answer c))
                        (when (and (numberp diff)
                                   (< diff bestval))
                          (setq bestval diff)
                          (setq ans (car answers))))
               )
             (setq answers (cdr answers)) )
           (and (< bestval 0.1) ans) )
      (expr (setq exp (unvar-copy answer))
            (dolist (c (cdr choices))
              (if (random-equal exp c)
                  (progn (setq bestans c)
                         (setq ans (car answers))))
              (setq answers (cdr answers)))
            (if (and ans (not (expr-equal exp bestans)))
                (format t "Failed to verify ~A = ~A~%" exp bestans))
            ans)
      (t nil)) ))

(defvar *eqn-powers* (make-array 20 :initial-element 0))
(defvar *eqn-vars* (make-array 20))
(defvar *eqn-nvars* 0)
(defvar *eqn-multiplier* 1)

; 22 Apr 04; 23 Apr 04
; optimize an expression composed of product, quotient, sqrt
(defun eqn-optprod (expr)
  (let (newex tmp)
    (setq *eqn-nvars* 0)
    (setq *eqn-multiplier* 1)
    (eqn-optpr expr 1)
    (dotimes (i (truncate *eqn-nvars* 2))
      (setq tmp (aref *eqn-vars* i))
      (setf (aref *eqn-vars* i) (aref *eqn-vars* (- (1- *eqn-nvars*) i)))
      (setf (aref *eqn-vars* (- (1- *eqn-nvars*) i)) tmp)
      (setq tmp (aref *eqn-powers* i))
      (setf (aref *eqn-powers* i) (aref *eqn-powers* (- (1- *eqn-nvars*) i)))
      (setf (aref *eqn-powers* (- (1- *eqn-nvars*) i)) tmp) )
    (setq newex (eqn-formexpr))
    (if (not (equal *eqn-multiplier* 1))
        (if (numberp newex)
            (* newex *eqn-multiplier*)
            (list '* newex *eqn-multiplier*))
        newex)))

; 19 Apr 05
(defun eqn-optpr (expr power)
  (if (numberp expr)
      (setq *eqn-multiplier* (* *eqn-multiplier* (expt expr power)))
    (if (symbolp expr)
        (eqn-accumpower expr power)
      (if (consp expr)
          (case (car expr)
            (* (dolist (x (cdr expr)) (eqn-optpr x power)))
            (/ (eqn-optpr (cadr expr) power)
               (eqn-optpr (caddr expr) (- power)))
            ((expt |expt|)
              (if (numberp (caddr expr))
                  (eqn-optpr (cadr expr) (* power (caddr expr)))
                  (error "expt expression ~A~%" expr)))
            ((sqrt |sqrt|) (eqn-optpr (cadr expr) (/ power 2)))
            (- (if (null (cddr expr))
                   (setq *eqn-multiplier* (* *eqn-multiplier* (expt -1 power)))
                   (eqn-accumpower expr power)))
            (t (eqn-accumpower expr power)))))))

(defun eqn-accumpower (expr power)
  (let (n)
    (dotimes (i *eqn-nvars*) (if (equal expr (aref *eqn-vars* i)) (setq n i)))
    (if (null n)
        (progn (setq n *eqn-nvars*)
               (incf *eqn-nvars*)
               (setf (aref *eqn-powers* n) 0)
               (setf (aref *eqn-vars* n) expr)))
    (incf (aref *eqn-powers* n) power) ))

; form a new expression from powers of terms
(defun eqn-formexpr ()
  (let ((num 1) (den 1) (newn 0) pow var)
    (dotimes (i *eqn-nvars*)
      (setq pow (aref *eqn-powers* i))
      (setq var (aref *eqn-vars* i))
      (if (not (zerop pow))
          (if (integerp pow)
              (if (minusp pow)
                  (setq den (eqn-formmul (if (> (abs pow) 1)
                                             (list 'expt var (abs pow))
                                             var)
                                         den))
                  (setq num (eqn-formmul (if (> pow 1)
                                             (list 'expt var pow)
                                             var)
                                         num)))
              (progn (setf (aref *eqn-powers* newn) (* pow 2))
                     (setf (aref *eqn-vars* newn) var)
                     (incf newn)) ) ) )
    (if (> newn 0)
        (progn (setq *eqn-nvars* newn)
               (setq num (eqn-formmul num (list 'sqrt (eqn-formexpr))))))
    (if (equal den 1)
        num
        (list '/ num den)) ))

(defun eqn-formmul (x y)
  (if (equal x 1)
      y
      (if (equal y 1)
          x
          (list '* x y))))

; 07 Jun 04; 08 Jun 04; 12 Apr 07
; Compare two expressions for algebraic equality
; search is used to try different methods for harder cases.
(defun expr-equal (x y)
  (let (simx simy res)
    (or (equal x y)
        (if (or (consp x) (consp y))
            (or (equal (setq simx (gleqns-simplify x))
                       (setq simy (gleqns-simplify y)))
                (and (consp x) (consp y) (eq (car x) (car y))
                     (every #'expr-equal (cdr x) (cdr y)))
                (and (not (or (safe-zerop simy)
                              (and (gleqnconstantp simy)
                                   (safe-zerop (eval simy)))))
                     (setq res (gleqns-simplify (list '/ x y)))
                     (or (equal res 1)
                         (and (numberp res) (nearly-equal res 1))))
                (and (setq res (gleqns-simplify (list '- simx simy)))
                     (or (equal res 0)
                         (and (numberp res) (nearly-equal res 0))))
                (expr-equal-factor simx simy)
                (expr-equal-perm simx simy) ) ) ) ))

; 07 Jun 04; 08 Jun 04
; Try to remove factors from expressions, test for equality
(defun expr-equal-factor (x y)
  (let ((vars (glvarsin x)) fact new)
    (some #'(lambda (var)
              (setq fact (exprchanged x var))
              (or (and (or (nearly-equal fact 2) (nearly-equal fact 4)
                           (nearly-equal fact 8))
                       (setq new (expr-pushfact '/ x var))
                       (< (expr-size new) (expr-size x))
                       (expr-equal new (expr-pushfact '/ y var)))
                  (and (or (nearly-equal fact 1/2) (nearly-equal fact 1/4)
                           (nearly-equal fact 1/8))
                       (setq new (expr-pushfact '* x var))
                       (< (expr-size new) (expr-size x))
                       (expr-equal new (expr-pushfact '* y var))) ))
          vars)))

; 07 Jun 04; 08 Jun 04
; Try to test permutations of expressions, test for equality
(defun expr-equal-perm (x y)
  (and (consp x) (consp y)
       (if (and (eq (op x) '+) (eq (op y) '+))
           (or (and (expr-equal (lhs x) (lhs y))
                    (expr-equal (rhs x) (rhs y)))
               (and (expr-equal (lhs x) (rhs y))
                    (expr-equal (rhs x) (lhs y))))
         (if (and (eq (op x) '-) (eq (op y) '-))
             (and (expr-equal (lhs x) (lhs y))
                  (expr-equal (rhs x) (rhs y)))
           (if (and (member (op x) '(+ -)) (member (op y) '(* /)))
               (expr-equal-perm x (pushinfact y))
               (if (and (member (op y) '(+ -)) (member (op x) '(* /)))
                   (expr-equal-perm (pushinfact x) y)))))))

(defun nearly-equal (x y) (< (abs (- x y)) 0.000001))

(defun set-equal (x y) (and (subsetp x y :test #'equal)
                            (subsetp y x :test #'equal)))

(defun safe-zerop (x) (and (numberp x) (zerop x)) )

; 07 Jun 04; 08 Jun 04; 12 Apr 07
; Compare two expressions for numeric equality using random variable values
(defun random-equal (x y)
  (let (vars vals)
    (setq vars (varsin x))
    (if (set-equal vars (varsin y))
        (progn
          (dolist (var vars)
            (or (gleqnconstantp var)
                (push (cons var (random 1.0)) vals)))
          (< (reldiff (list (safe-eval (sublis vals x)) 'unity)
                      (list (safe-eval (sublis vals y)) 'unity))
             0.0001)) ) ))

; 07 Jun 04; 12 Apr 07
; size of a binary expression: operators + variables
(defun expr-size (x)
  (if (consp x)
      (+ 1 (expr-size (cadr x)) (expr-size (caddr x)))
      (if (gleqnconstantp x) 0 1)))

; 07 Jun 04
; push a factor into an expression
; we expect this factor to cancel out
(defun expr-pushfact (op x var)
  (let (newx newarg)
    (setq newx (gleqns-simplify (list op x var)))
    (if (< (expr-size newx) (expr-size x))
        newx
      (if (consp x)
          (case (op x)
            ((+ -) (gleqns-simplify
                     (cons (op x)
                           (mapcar #'(lambda (arg) (expr-pushfact op arg var))
                                   (cdr x)))))
            ((* /)
              (setq newarg (expr-pushfact op (lhs x) var))
              (if (< (expr-size newarg) (expr-size (lhs x)))
                  (gleqns-simplify (list (op x) newarg (rhs x)))
                (if (eq (op x) '*)
                    (progn (setq newarg (expr-pushfact op (rhs x) var))
                           (if (< (expr-size newarg) (expr-size (rhs x)))
                               (gleqns-simplify (list (op x) (lhs x) newarg))
                               newx))
                    (progn (setq newarg
                                 (expr-pushfact (if (eq op '*) '/ '*)
                                                (rhs x) var))
                           (if (< (expr-size newarg) (expr-size (rhs x)))
                               (gleqns-simplify (list (op x) (lhs x) newarg))
                               newx)))))
            (t newx))
          newx)) ))

; 07 Jun 04
; Copy an expression, removing VAR- prefix of variables
(defun unvar-copy (x)
  (if (consp x)
      (cons (unvar-copy (car x)) (unvar-copy (cdr x)))
      (if (and x (symbolp x))
          (let ((str (symbol-name x)))
            (if (and (> (length str) 4)
                     (string= str "VAR-" :end1 4))
                (intern (subseq str 4))
                x))
          x)))

; 08 Jun 04
; try to push in multiplicative factors until a + or - is reached.
(defun pushinfact (x) (pushinfactb x 1))
(defun pushinfactb (x factor)
  (let ()
    (and (consp x)
         (if (eq (car x) '*)
             (if (factorp (cadr x))
                 (pushinfactb (caddr x) (makefactor factor (cadr x)))
                 (if (factorp (caddr x))
                     (pushinfactb (cadr x) (makefactor factor (caddr x)))))
             (if (eq (car x) '/)
                 (if (factorp (caddr x))
                     (pushinfactb (cadr x) (makequot factor (caddr x))))
                 (if (and (member (car x) '(+ -)) (cddr x))
                     (list (car x)
                           (gleqns-simplify (makefactor factor (cadr x)))
                           (gleqns-simplify (makefactor factor (caddr x)))
                           )))))))

; 08 Jun 04; 19 Apr 05
; Test whether x is a factor
(defun factorp (x)
  (or (atom x)
      (and (consp x)
           (member (car x) '(* / sqrt expt sin cos tan
                               |sqrt| |expt| |sin| |cos| |tan|))
           (factorp (cadr x))
           (factorp (caddr x)))))

; 08 Jun 04
; Make a product
(defun makefactor (x y)
  (if (equal x 1)
      y
      (if (and (consp x)
               (eq (car x) '/)
               (equal (cadr x) 1))
          (list '/ y (caddr x))
          (list '* x y))))

; 08 Jun 04
; Make a quotient
(defun makequot (x y) (list '/ x y))

; 07 Apr 04
; Find all variables of a kind of object
(defun eqn-allvars (objtype)
  (let ((vars (basis-vars objtype)))
    (dolist (eqn (equations objtype))
      (setq vars (union vars (varsin eqn))) )
    vars))

; 08 Dec 05; 11 Apr 07; 24 Sep 08
; Fix an equation so that the lhs is a single variable
; (glfixequation '(= (* f time) (* m v)))
(defun glfixequation (eqn)
  (let ((op (car eqn)) (lhs (cadr eqn)) (rhs (caddr eqn)) new lhsvars rhsvars)
    (if (symbolp lhs)
        eqn
        (if (symbolp rhs)
            (list op rhs lhs)
            (progn (setq lhsvars (varsin lhs))
                   (setq rhsvars (varsin rhs))
                   (if (and lhsvars
                            (or (null rhsvars)
                                (<= (length lhsvars) (length rhsvars))))
                       (glsimplesolvefor (list '= rhs lhs)
                                         (first (last lhsvars)))
                       (if rhsvars
                           (glsimplesolvefor eqn (first (last rhsvars)))))))) ))

; 08 Dec 05
; Fix an equation so that the lhs is a single variable
; (glfixequation '(= (* f t) (* m v)))
(defun glfixequation-old (eqn)
  (let ((op (car eqn)) (lhs (cadr eqn)) (rhs (caddr eqn)) new)
    (if (symbolp lhs)
        eqn
        (if (symbolp rhs)
            (list op rhs lhs)
          (if (and (consp lhs) (consp rhs)
                   (or (eq (car lhs) '+)
                       (and (eq (car lhs) '-) (cddr lhs)))
                   (or (eq (car rhs) '+)
                       (and (eq (car rhs) '-) (cddr rhs))))
              (progn
                (setq new (glsolvefor (list '= 0 (list '- lhs rhs))
                                      (first (last (glvarsin eqn)))))
                (list '= (cadr new) (gleqns-simplify (caddr new))))
              (progn
                (setq new (glsolvefor (list '= 1 (list '/ lhs rhs))
                                      (first (last (glvarsin eqn)))))
                (list '= (cadr new) (gleqns-simplify (caddr new)))) ) ) ) ))

; 16 May 06; 02 Aug 06
; Record equation solving history (backwards)
(defun solveeqhist (var value units orig-equation new-equation)
  (push (list var value units orig-equation new-equation)
        *equations-history*) )

; 16 May 06; 02 Aug 06
; Filter equation solving history, leaving only relevant steps.
(defun filtereqhist (history goalvar)
  (let (newhist needed)
    (setq needed (list goalvar))
    (dolist (histitem history)
      (if (member (car histitem) needed)     ; if this var was needed
          (progn (push histitem newhist)
                 (setq needed (union needed
                                     (varsin (third (fifth histitem))))))))
    newhist))

; 02 Aug 06; 09 Oct 06; 27 Dec 06; 21 Nov 08
; Explain solution process for var given history
; (phys '(what is the volume of a cone with circumference 3 m
;          and height = 4 m))
; (explaineqhist)
(defun explaineqhist (&optional infix html history goalvar)
  (let (newhist)
    (or history (setq history *equations-history*))
    (or goalvar (setq goalvar (caar history)))
    (setq newhist (filtereqhist history goalvar))
    (dolist (item newhist)
      (if (fourth item)
          (progn
            ; if already solved for desired var, don't explain
            (if (not (equal (fourth item) (fifth item)))
                (progn
                  (format t "Solved equation ~A~%   for ~A~%   giving ~A~%"
                        (if infix (infixstr (fourth item) html) (fourth item))
                        (first item)
                        (if infix (infixstr (fifth item) html) (fifth item)))
                  (if html (format t "<BR>"))))
            ; if a constant such as gravity, don't explain
            (if (not (and (consp (third (fifth item)))
                          (eq (car (third (fifth item))) 'quote)))
                (progn
                  (format t "Evaluated ~A~%   giving ~A = ~A ~A~%"
                        (if infix (infixstr (fifth item) html) (fifth item))
                        (first item)
                        (if infix (infixstr (second item) html) (second item))
                        (or (third item) " "))
                  (if html (format t "<BR>")) ) ) ) ) ) ))

; 22 May 07; 24 Nov 08
; Test for a bogus equation, e.g. one that will divide by zero
(defun solvboguseqn (eqn var alist ulist)
  (let (eqnb)
    (and (setq eqnb (glsolvefor eqn var))
         (eq (lhs eqnb) var)
         (solvbogusexpr (rhs eqnb) alist ulist)) ) )

; 22 May 07; 01 Jun 07; 10 Nov 08; 24 Nov 08
; Test for a bogus expression, e.g. one that includes divide by zero
(defun solvbogusexpr (exp alist ulist)
  (and (consp exp)
       (case (op exp)
         ((+ - * expt |expt| atan |atan|)
           (or (solvbogusexpr (lhs exp) alist ulist)
               (solvbogusexpr (rhs exp) alist ulist)))
         (/ (or (solvbogusexpr (lhs exp) alist ulist)
                (solvbogusexpr (rhs exp) alist ulist)
                (solvzeroexpr (rhs exp) alist ulist)))
         ((sin |sin| cos |cos| exp |exp| log |log|
               log2 |log2| log10 |log10| sqrt |sqrt| cbrt |cbrt|)
           (solvbogusexpr (lhs exp) alist ulist))
         ((tan |tan|)
           (or (solvbogusexpr (lhs exp) alist ulist)
               (solvzeroexpr (list 'cos (lhs exp)) alist ulist)))
         ((acos |acos| asin |asin|)
           (or (solvbogusexpr (lhs exp) alist ulist)
               (solvzeroexpr (lhs exp) alist ulist)))
         (t nil))))

; 22 May 07; 01 Jun 07; 20 Nov 08; 21 Nov 08; 24 Nov 08
; Test for a zero expression
(defun solvzeroexpr (exp alist ulist)
  (let (val)
    (or (and (numberp exp) (zerop exp))
        (and (symbolp exp)
             (setq val (assoc exp alist))
             (numberp (cdr val))
             (zerop (cdr val)))
        (and (consp exp)
             (case (op exp)
               (* (or (solvzeroexpr (lhs exp) alist ulist)
                      (solvzeroexpr (rhs exp) alist ulist)))
               ((/ sqrt |sqrt| cbrt |cbrt|)
                 (solvzeroexpr (lhs exp) alist ulist))
               ((sin |sin| tan |tan|)
                 (setq val (eqn-eval (lhs exp) alist ulist))
                 (if (and (numberp (car val))
                          (member (cadr val) '(degree |*degree|)))
                     (setq val (list (* (car val) (/ pi 180))
                                     'radian)))
                 (or (solvzeroexpr (lhs exp) alist ulist)
                     (and (numberp (car val))
                          (or (approx= (car val) 0)
                              (approx= (car val) pi)
                              (approx= (car val) (- pi))))))
               ((cos |cos|)
                 (setq val (eqn-eval (lhs exp) alist ulist))
                 (if (and (numberp (car val))
                          (member (cadr val) '(degree |*degree|)))
                     (setq val (list (* (car val) (/ pi 180))
                                     'radian)))
                 (and (numberp (car val))
                      (or (approx= (car val) (/ pi 2))
                          (approx= (car val) (- (/ pi 2)))
                          (approx= (car val) (/ (* 3 pi) 2)))))
               ((log |log| log2 |log2| log10 |log10|)
                 (and (setq val (eqn-eval (lhs exp) alist ulist))
                      (numberp (car val))
                      (approx= (car val) 1.0)))
               (t nil))) ) ))

; 23 Jan 08
; Find vars needed given equation, known, and desired vars
(defun eqn-needed (equation known desired)
  (let ((vars (glvarsin equation)))
    (set-difference (set-difference vars known) desired) ))

; 07 Nov 08; 01 Apr 13
; used by conprop.lsp
; Solve a set of equations given an alist (var value unit)* of values.
; returns alist of solved variables
(defun solveqnset (equations values)
  (let ((progress t) known vars diffvars newvals neweqn val)
    (while progress
      (setq progress nil)
      (dolist (equation equations)
        (setq vars (varsin equation))
        (setq diffvars (set-difference vars (mapcar #'car values)))
        (when (and diffvars (null (cdr diffvars)))  ; exactly 1
          (setq neweqn (glsolvefor equation (first diffvars)))
          (setq newval (eqn-eval (rhs neweqn) (sublisify values)
                                 (mapcan #'(lambda (x)
                                             (if (third x)
                                                 (list (cons (first x)
                                                             (third x)))))
                                         values)))
          (setq val (cons (first diffvars) newval))
          (push val newvals)
          (push val values)
          (setq progress t) ) ) )
    newvals))

; 21 Nov 08
; test approximate equality
(defun approx= (x y &optional (tolerance 0.00001))
  (or (equal x y)
      (and (numberp x) (numberp y)
           (< (abs (- x y)) (* tolerance (max (abs x) (abs y)))))))

; 30 Oct 09
; if x is approximately an integer, return that integer, else nil
(defun approxint (x)
  (let ( (n (round x)) )
    (if (< (abs (- x n)) (* 0.000001 (abs n)) )
        n) ))

; 30 Oct 09
; try to rationalize the number x as approximately quotient of small integers
(defun rationalizeb (x)
  (let (r p n q)
    (or (approxint x)
        (if (approxint (/ 1 x))
            (/ 1 (approxint (/ 1 x))))
        (and (setq r (rationalize x))
             (< (numerator r) 1000)
             (< (denominator r) 1000)
             r)
        (and (setq p (* x 8640))     ;  2^7 * 3^3 * 5 * 7 * 11
             (setq n (approxint p))
             (/ n 8640)) ) ))

; 30 Oct 09
; try to find a better representation of the number x
(defun betternum (x)
  (let (z)
    (or (rationalizeb x)
        (if (setq z (rationalizeb (* pi x)))
            (list '/ z 'pi)
            (if (setq z (rationalizeb (/ x pi)))
                (if (= z 1) 'pi (list '* z 'pi))
                x) ) ) ))
