; genll.lsp                Gordon S. Novak Jr.            ; 22 Dec 10

; Copyright (c) 2010 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Generic linked-list abstract data types and functions

; 01 Nov 94; 05 Jan 95; 05 May 95; 09 May 95; 31 May 95; 08 Aug 95; 18 Aug 95
; 23 Aug 95; 24 Aug 95; 01 Feb 96; 06 Feb 96; 16 Jan 97; 21 Jan 97; 13 Feb 97
; 11 Mar 97; 25 Mar 97; 04 Apr 00; 31 Jul 01; 02 Aug 01; 28 Feb 02; 27 Nov 02
; 03 Dec 02; 01 Dec 03; 07 Jan 04; 03 Mar 04; 28 May 04; 01 Jun 04; 10 Jun 04
; 14 Jun 04; 25 Jun 04; 26 Aug 04; 27 Aug 04; 28 Oct 04; 18 Nov 04; 12 Jul 06
; 01 Mar 07; 27 Mar 07; 03 Apr 07; 06 Aug 07

; 07 Oct 91
(gldefclusterc
  'record-with-pointer
  '((pointer (pointer (^ record)
       adj     ((null          (self == (null-value self)) ))
       prop    ((dereference   (self) result (clustertype (typeof self)
							  'record))
	        (null-value    (*glnull*)) )
       default ((self nil))  ))

    (record (record anything
        prop   ((implementation (self) result (strof (typeof self)) ))
	msg    ((new            ((a (typeof self)))
				  result (clustertype (typeof self)
						      'pointer)) ) )) )
  '()
 )

; 27 Apr 90; 14 May 90; 31 May 90; 26 Nov 90; 19 Feb 91; 28 Feb 91; 11 Mar 91
; 07 Oct 91; 27 Oct 94; 01 Feb 96; 14 Jun 04
(gldefclusterc
  'linked-list
  '((pointer (linked-list-pointer (^ linked-list-record)
       prop    ((first             generic-first        open t)
	        (rest              generic-rest         open t)
		(cddr              generic-cddr         open t)
	        (length            generic-length       specialize t)
	        (length-b          generic-length-b     specialize t)
	        (last              generic-last         specialize t)
	        (copy-list         generic-copy-list    specialize t)
		(reverse           generic-reverse      specialize t)
		(size              (length))
		(creation-size     (0))
		(initial-pointer   (self)) )
       msg     ((nth               generic-nth          open t)
	        (nthcdr            generic-nthcdr       specialize t)
	        (index             generic-nth          open t)
		(set-nth           generic-set-nth      specialize t)
		(pop               generic-pop          open t)
	        (length-up-to      generic-length-up-to specialize t)
	        (copy-list-to      generic-copy-list-to specialize t)
		(subset            generic-subset       specialize t)
		(some              generic-some         specialize t)
		(every             generic-every        specialize t)
		(member            generic-member       specialize t)
		(delete            generic-delete       specialize t)
		(nconc             generic-nconc        specialize t)
		(nreverse          generic-nreverse     specialize t)
		(nfirstn           generic-nfirstn      specialize t)
		(new               generic-list-new     specialize t)
		(append            generic-append       specialize t)
		(union             generic-union        specialize t)
		(intersection      generic-intersection specialize t)
		(set-difference    generic-set-difference specialize t)
		(push              generic-push         open t)
		(+_                generic-push         open t)
		(push-item         generic-push-item    open t)
		(splice-in         generic-splice-in    open t)
		(sort-before       generic-sort-before  open t)
		(iterator          linked-list-iterator)
		(collector         linked-list-collector)
		(set-contents      generic-set-contents open t)
		(merge             ll-merge             specialize t)
		(sort              ll-merge-sort        specialize t)
		(sort-element-view ('identity))
		(sort-before-fn    ('<))
		(sort-direction    ('ascending))
		(transfer-contents generic-transfer-contents open t)
		(deref             (glambda (self ptr)
				     (implementation (dereference ptr))))
		(next              (glambda (self ptr) (rest ptr)))
		(endp              (glambda (self ptr) (null ptr)))
		(max               generic-max         specialize t)
		) ))

    (record (linked-list-record (tuple (contents anything)
                                       (link linked-list-pointer))
        prop   ((copy-contents        (contents))
		(contents             (self))
		(copy-contents-names  (nil))
		(equality-names       (copy-contents-names)) )
	msg    ((equal             generic-equal-element   open t)
		(==                generic-equal-element   open t)
		(equal-contents    generic-equal-contents  open t) ) )))
                    ; was ((components (typeof (implementation self))))
  '(record-with-pointer)
 )

(gldefclusterc
  'alist
  '((pointer (alist-pointer (^ alist-record)
       prop    ((order-one-access  (nil)))
       msg     ((init       (glambda (self) ))
		(insert-key        alist-insert-key  specialize t)
		(member            alist-member      specialize t)
		(newempty          (nil) result (typeof self)) )))

    (record  (alist-record
	       (tuple (contents anything)
		      (link     alist-pointer))
       prop    ((sort-value     (contents))) ) ) )
  '(linked-list) )

(setf (get 'linked-list 'glnpointers) 1)
(setf (get 'sorted-linked-list 'glnpointers) 1)

; ap-omit: omit the function from offerings of the automatic programming server
(setf (get 'linked-list-pointer 'ap-omit)
      '(first rest length-b size creation-size initial-pointer index pop +_
	sort-before iterator collector set-contents sort-element-view
	sort-before-fn sort-direction transfer-contents deref next endp
	push-item push null null-value dereference sort merge splice-in
	new every some subset copy-list-to delete member nth set-nth cddr))

; java-static is t for functions that could be called with a null pointer arg
(mapc #'(lambda (x) (setf (get x 'java-static) t))
      '(generic-append generic-copy-list generic-last generic-length
        generic-length-up-to generic-nconc generic-nreverse generic-nthcdr
	generic-reverse alist-insert-key alist-member
	sll-insert sll-insert-key sll-remove-key sll-merge ll-merge-sort))

(setf (glcarrier 'linked-list)
      '((cons (contents contents-type)
	      (link pointer))
	nil) )

(setf (glcarrier 'sorted-linked-list)
      '((cons (contents contents-type)
	      (link pointer))
	nil) )

; 19 Sep 91
(gldefclusterc
  'ext-linked-list
  '((pointer (ext-list-pointer (^ ext-list-record)
	       msg ((collector      external-linked-list-collector)
		    (iterator       external-linked-list-iterator))))
    (record  (ext-list-record  (tuple (contents anything)
				      (link ext-list-pointer)) )) )
  '(linked-list))

; 30 Dec 91; 06 Jan 92; 07 Jan 92
(gldefclusterc
  'circular-linked-list
  '((pointer (circular-linked-list-pointer (^ circular-linked-list-record)
               prop ((length         generic-length-b       specialize t)
		     (length-up-to   generic-length-up-to-b specialize t))
               msg  ((iterator       circular-linked-list-iterator))))
    (record  (circular-linked-list-record
	       (tuple (contents anything)
		      (link circular-linked-list-pointer)) )) )
  '(linked-list))

; 14 Mar 91
(gldefclusterc
  'lisp-linked-list
  '((pointer (lisp-linked-list-pointer (^ lisp-linked-list-record)
      #| ;  commented out: result types do not yet work
             prop ((length    length    result integer)
	           (last      last      result (typeof self))
		   (copy-list copy-list result (clustertype (typeof self)
                                                            'pointer))) |#
	     ))
    (record  (lisp-linked-list-record
	       (cons (contents (clustertype (typeof self) 'contents))
		     (link lisp-linked-list-pointer))
               prop ((contents ((first self))))
               msg  ((new ((cons nil nil))
			  result (clustertype (typeof self) 'pointer))))))
  '(linked-list))

; 19 May 92; 26 Oct 93
(gldefclusterc 'linked-list-in-array
  '((pointer (linked-list-in-array-pointer integer
      prop    ((null-value  (-1)   result (clustertype (typeof self) 'pointer))
	       (dereference (self) result (clustertype (typeof self) 'record)))
      adj     ((null        (self < 0)))
      supers  (linked-list-pointer) ))
    (record  (linked-list-in-array-record integer
      prop    ((link           ((aref (link-array self) self))
			          result (clustertype (typeof self) 'pointer))
	       (contents       ((aref (contents-array self) self)))
	       (implementation (contents)) )
      supers  (linked-list-record) ) ))
  '(linked-list))

(gldefclusterc
  'lisp-linked-list-of-integer
  '((pointer  (lisp-linked-list-of-integer-pointer 
	        (^ lisp-linked-list-of-integer-record)))
    (record   (lisp-linked-list-of-integer-record
                (cons (contents integer)
		      (link lisp-linked-list-of-integer-pointer))
               msg  ((new ((cons 0 nil))
			  result lisp-linked-list-of-integer-pointer)))))
  '(lisp-linked-list))

; 03 Apr 07
(gldefclusterc
  'two-pointer-queue
  '((record (two-pointer-queue-record
               (tuple (qstart linked-list-pointer)
		      (qend   linked-list-pointer))
               adj  ((empty   (glambda (self) (null (qstart self)))))
	       msg  ((insert      two-pointer-queue-insert      open t)
		     (insert-item two-pointer-queue-insert-item open t)
		     (remove      two-pointer-queue-remove      specialize t)
		     (remove-item two-pointer-queue-remove-item specialize t)
                     (initialize  (glambda (self) 
                                    ((qstart self) = (null-value (qstart self)))
                                    ((qend self)   = (null-value (qstart self))))))))
     )
  '() )

(gldefclusterc
  'front-pointer-queue
  '((record (front-pointer-queue-record
	     (tuple (qstart linked-list-pointer))
	    msg     ((insert      front-pointer-queue-insert      open t)
		     (insert-item front-pointer-queue-insert-item open t)
		     (remove      two-pointer-queue-remove        open t)
		     (remove-item two-pointer-queue-remove-item   open t)
                        ))))
  '() )

(gldefclusterc
  'end-pointer-queue
  '((record (end-pointer-queue-record
	     (tuple (qend linked-list-pointer))
	     msg     ((insert      end-pointer-queue-insert      open t)
		      (insert-item end-pointer-queue-insert-item open t)
		      (remove      end-pointer-queue-remove      open t)
		      (remove-item end-pointer-queue-remove-item open t)
                           ))))
  '() )

(gldefclusterc
  'priority-queue
  '((record (priority-queue-record
	      (tuple (indexing-method anything))
	     msg     ((insert      priority-queue-insert      open t)
		      (insert-item priority-queue-insert-item open t)
		      (remove      priority-queue-remove      open t)
		      (remove-item priority-queue-remove-item open t)
                          ))))
  '()
  'parameters '((size 1 integer))  )

; 24 Apr 90; 10 Jan 97
; sorted linked list
(gldefclusterc
  'sll
  '((pointer (sll-pointer (^ sll-record)
               msg ((insert      sll-insert      specialize t)
		    (insert-key  sll-insert-key  specialize t)
		    (remove-key  sll-remove-key  specialize t)
		    (nreverse    error)
		    (merge       sll-merge       specialize t) )) )

    (record  (sll-record (tuple (contents anything)
				(link sll-pointer))
               prop ((sort-value (implementation))
		     (sort-direction ('ascending)))
               msg  ((sort-before (glambda (x y)
				    (< (sort-value x) (sort-value y))))))) )
  '(linked-list))

(setf (get 'sorted-linked-list 'ap-type) 'sll-pointer)
(setf (get 'sll-pointer 'ap-omit)
      '(splice-in append new nreverse nconc every some subset copy-list-to
	set-nth reverse))
(setf (get 'sll-pointer 'ap-include) '(sort merge))

(glispobjects

; 10 Jan 97
  (sorted-linked-list anything
  msg     ((insert      sorted-linked-list-insert      specialize t)
	   (insert-key  sorted-linked-list-insert-key  specialize t)
	   (sort-before (glambda (l y) (< (contents (^. l)) y)) ) ) )

; 26 Sep 91; 02 Oct 91; 30 Jan 92; 02 Oct 93; 12 Nov 93; 22 Nov 93; 27 Oct 94
; 12 Jul 06; 22 Dec 10
(array anything
  prop    ((array             (self))
	   (creation-size     (1))
	   (size              ((array-total-size (array self))) result integer)
	   (filled-size       (size))
	   (lower-bound       (0))
	   (increment         (1))
	   (implementation    (self) result (strof (typeof self)))
           (first             ((aref self 0)))
	   (element-type      ((typeof (aref self 0))))
	   (initial-element   ((a (typeof (aref self 0)))))
	   (sort-element-view ('identity))
	   (sort-before-fn    ('<))
	   (sort-direction    ('ascending))
	   (element-components ((components (typeof (index self 0)))))
	   (initial-pointer   (lower-bound)) )
  msg     ((index   (glambda (self n) (aref (array self) n)) )
           (set     (glambda (self n val) (setf (index self n) val)))
     ; Note: set-all only works for 1-dimensional arrays
	   (set-all (glambda (self val)
		      (dotimes (i (size self)) (setf (index self i) val))))
	   (new     ((make-array (creation-size self)
			     :initial-element (initial-element self)))
		         result (typeof self))
	   (length-up-to (glambda (self n) (size self)))
	   (next-entry   array-next-entry open t)
	   (sort         array-quicksort  specialize t)
	   (sortb        array-quicksortb specialize t)
	   (nreverse     array-nreverse specialize t)
	   (iterator     array-iterator)
	   (deref        (glambda (self n) (aref (array self) n)) )
	   (next         (glambda (self n) (+ n (increment self))))
	   (endp         (glambda (self n) (>= (- n (lower-bound self))
					       (size self)))) ) )

 ) ; glispobjects

(setf (get 'array 'ap-omit)
      '(array creation-size size filled-size lower-bound increment
        implementation initial-element sort-element-view sort-before-fn
	sort-direction element-components initial-pointer index set new
	length-up-to next-entry sortb iterator deref next endp))

; 08 Nov 93; 09 Nov 93; 11 Nov 93; 12 Nov 93; 15 Nov 93; 16 Nov 93; 17 Nov 93
; 27 Oct 94
(gldefclusterc
  'array-with-pointer
  '((pointer (array-pointer integer
      prop    ((null-value  ((lower-bound self) - 1)
                              result (clustertype (typeof self) 'pointer))
	       (lower-bound (0))
	       (dereference     (self) result (clustertype (typeof self)
							   'record))
	       (initial-pointer (lower-bound))
	       (increment       (1)) )
      msg     ((next            (glambda (self n) (+ n (increment self))))
	       (endp            (glambda (self n) (>= (- n (lower-bound self))
						      (size self)))))
      adj     ((null (self < 0)) )
      default ((self -1)) ) )

    (record  (array-record integer
      prop   ((contents       ((index (array self) self)))
              ; array property must be defined by the instance
              ; as the array variable, e.g. (array  (*myarray*))
	      (implementation (contents)) )
      msg    ((new            array-next-entry
			      result (clustertype (typeof self)
						  'pointer)) ) )) )
  '(record-with-pointer)
 )

; 12 Nov 93
(gldefun array-next-entry ((arr array))  (result integer)
  (let (n)
    (n = (filled-size arr))
    (if (n < (size arr))
	(progn ((filled-size arr) _+ 1)
	       n)
        (error "Array overflow~%") ) ))

; 10 May 93; 22 Nov 93; 12 Jul 06
(gldefun array-quicksort (arr)
  (sortb arr (lower-bound arr) ((filled-size arr) - 1))
  arr)

; 10 May 93; 11 May 93; 13 May 93; 22 Nov 93; 23 Nov 93; 07 Jan 04; 12 Jul 06
; cf. Sedgewick, Algorithms, p. 118; Sedgewick's version, however, requires
; a "sentinel" value less than any data value at the left of the array,
; and it is O(n^2) for an array that is already almost sorted.
(gldefun array-quicksortb (arr (l integer) (r integer))
  (let (val tmp i j mid)
    (if (r > l)
     (progn
      (mid = (truncate (l + r) 2))
      (val = (index arr mid))
      (i = l - 1)
      (j = r + 1)
      (while (j > i) do
        (repeat (i = i + 1)
	  until (not (funcall (sort-before-fn arr)       ; not ( a[i] < val )
			      (funcall (sort-element-view arr)
				       (index arr i))
			      (funcall (sort-element-view arr) val)) ) )
	(repeat (j = j - 1)
          until (or (i >= j)
		    (not (funcall (sort-before-fn arr)   ; not ( val < a[j] )
				  (funcall (sort-element-view arr) val)
				  (funcall (sort-element-view arr)
					   (index arr j)) ) ) ) )
	(if (j > i)
	 (progn
	  (tmp = (index arr i))
	  ((index arr i) = (index arr j))
	  ((index arr j) = tmp) ) ))
      (if (funcall (sort-before-fn arr)                  ; val < a[i]
		   (funcall (sort-element-view arr) val)
		   (funcall (sort-element-view arr)
			    (index arr i)) )
	  (i = (i - 1)))
      (sortb arr l i)
      (sortb arr (i + 1) r) )  )) )

; 27 Oct 94
; destructively reverse the order of elements in an array
(gldefun array-nreverse (arr)
  (let (tmp siz)
    (siz = (filled-size arr))
    (dotimes (i ((siz - (lower-bound arr)) / 2))
      (tmp = (index arr (+ i (lower-bound arr))))
      ((index arr (+ i (lower-bound arr))) =
         (index arr (siz - (i + 1))))
      ((index arr (siz - (i + 1))) = tmp) )
    arr))

; edited: 14-Apr-89 15:57 
(gldefun generic-first ((l linked-list-pointer))
  (contents (^. l)))

; 19 Mar 91
(gldefun generic-rest ((l linked-list-pointer))
  (result (typeof l))
  (link (^. l)) )

; 01 Feb 96
(gldefun generic-cddr ((l linked-list)) (rest (rest l)))

; edited: 13-Mar-89; 21 May 90; 05 Mar 92; 19 May 92; 21 May 92
(gldefun generic-length ((l linked-list-pointer))
  (let (n)
    (n = 0)
    (while (not (null l)) do
      (n = n + 1)
      (l = (rest l)))
    n))

; 08 Oct 91; 05 Mar 92; 27 Sep 94
; Find length, up to a given bound
(gldefun generic-length-up-to ((l linked-list-pointer) (max integer))
  (let ((n 0))
    (while (and (not (null l)) (n < max)) do
      (n = n + 1)
      (l = (rest l)))
    n))

; version for plain Lisp lists.  30 Jan 92
(defun length-up-to (l max)
  (let ((n 0))
    (while (and l (< n max)) (incf n) (pop l))
    n))

; 25 Sep 91; 27 Sep 94
; This version relies on the iterator, so it works for circular linked lists.
(gldefun generic-length-b ((l linked-list-pointer))
  (let ((n 0))
    (for x in l do (n = n + 1))
    n))

; 07 Jan 92; 27 Sep 94
; Find length, up to a given bound
(gldefun generic-length-up-to-b ((l linked-list-pointer) (max integer))
  (let ((n 0))
    (for x in l ; while (n < max)       ; while not implemented in for loop yet
                do (n = n + 1))
    n))

; edited:  8-Mar-89; 28 Dec 89; 23 Feb 90; 21 May 92
(gldefun generic-nreverse ((l linked-list-pointer))
  (result (typeof l))
  (let (prev next)
    (prev = (null-value l))
    (while (not (null l)) do
      (next = (rest l))
      ((rest l) = prev)
      (prev = l)
      (l = next))
    prev))

; edited:  7-Apr-89; 14 May 90; 03 Dec 02
(gldefun generic-nth ((l linked-list-pointer) (n integer))
  (implementation (first (nthcdr l n))) )

; 31 May 90; 18 Aug 95
(gldefun generic-nthcdr ((l linked-list-pointer) (n integer))
  (result (typeof l))
;;;  (if (n < 0) (error "Index out of range"))  ; removed
  (while l is not null and (n > 0) do (n _- 1) (l = (rest l)))
  l)

; 25 Mar 97
; Destrunctively truncate a list after the first n elements
(gldefun generic-nfirstn ((l linked-list-pointer) (n integer))
  (result (typeof l))
  (let ((ll (nthcdr l (1- n))))
    (if (not (null ll))
	((rest ll) = (null-value l)))
    l))

; 22 May 90; 11 Mar 91; 10 Jan 97; 28 Feb 02; 07 Jan 04
; last arg type should be: (or (typeof (^. l)) (typeof (contents (^. l))))
; but at present GLDECL doesn't accept that.
(gldefun generic-set-nth ((l linked-list-pointer)
			    (n integer)
			    (newrec (typeof (^. l))))
  (if (n < 0) (error "Index out of range"))
  (while l is not null and (n > 0) do (n _- 1) (l = (rest l)))
  (if l is null
      (error "Index out of range")
      (if (typeof (^. l)) == (typeof newrec)
	  (set-contents l newrec)
	  ((first l) = newrec))) )

; 08 Aug 95; 21 Jan 97
(gldefun generic-last ((l linked-list-pointer))
  (result (typeof l))
  (while (and (not (null l))
	      (not (null (rest l))))
    (l = (rest l)) )
  l)

; 14-Apr-89; 27 Nov 02
(gldefun generic-pop ((l linked-list-pointer))
  (let (tmp)
    (tmp = (first l))
    (l __ (rest l))
    tmp) )

; 30 Dec 91
; Splice in linked-list element m following l.
(gldefun generic-splice-in ((l linked-list-pointer) (m linked-list-pointer))
  ((rest m) = (rest l))
  ((rest l) = m) )

; 11 Oct 89; 31 May 90; 07 Jan 04; 14 Jun 04
(gldefun generic-member ((l linked-list-pointer)
			 (item (typeof (first l))))
  (while l is not null do
    (if (equal (first l) item)
        (return l)
        (l = (rest l))) )
  l)

; 07 Jan 04
(gldefun generic-delete ((l linked-list-pointer)
			 (item (typeof (first l))))
  (let ((origl l) (prev (typeof l)))
    (while l is not null do
      (if (equal (first l) item)
	  (progn (l = (rest l))
		 (if prev
		     ((rest prev) = l)
		     (origl = l)))
	  (progn (prev = l)
		 (l = (rest l)) ) ))
    origl))


; 19 Apr 90; 21 Jan 97; 07 Jan 04
(gldefun generic-nconc ((l linked-list-pointer) (m (typeof l)))
  (result (typeof l))
  (let (ll)
    (if l is null
	m
	(progn (ll = l)
	       (while (rest l) is not null do (l = (rest l)))
	       ((rest l) = m)
	       ll) )))


; 09 Apr 90; 31 May 90; 12 Feb 91; 14 Feb 91; 11 Mar 91; 26 Dec 91; 10 Jan 97
; 21 Jan 97; 07 Jan 04; 01 Mar 07
(gldefun generic-copy-list ((l linked-list-pointer))
  (result (typeof l))
  (let ((cpy (typeof l)) (r (typeof l)) newrec)
    (cpy = (null-value l))
    (r = (null-value l))
    (if (null l)
	l
        (progn (while (not (null l))
		    (newrec = (new (^. l)))
		    (transfer-contents newrec l)
		    (if (not (null cpy))
		        ((link (^. r)) = newrec)
		        ( cpy = newrec))
		    (r = newrec)
		    (l = (rest l)) )
	       ((link (^. r)) = (null-value l))
	       cpy))))

; 01 Nov 94; 10 Jan 97
(gldefun generic-reverse ((l linked-list-pointer))
  (result (typeof l))
  (let ((r (typeof l)) newrec)
    (r = (null-value l))
    (while l is not null do
      (newrec = (new (^. l)))
      (transfer-contents newrec l)
      ((link (^. newrec)) = r)
      (r = newrec)
      (l = (rest l)) )
    r))

; 04 Oct 91; 10 Jan 87; 07 Jan 04
; Subset of a linked-list that satisfies a predicate p
(gldefun generic-subset ((l linked-list-pointer) p)
  (result (typeof l))
  (let (found cpy (r (typeof (^. l))) newrec)
    (while l is not null do
	   (if (funcall p l)
	       (progn (newrec = (new (^. l)))
		      (transfer-contents newrec l)
		      (if found
			  ((link r) = newrec)
		          (progn (cpy = newrec)
				 ( found = t))
			  (r = (^. newrec)))))
	   (l = (rest l)) )
    (if found
	(progn ((link r) = (null-value l))
	       cpy)
        (null-value l)) ))


; 04 Oct 91; 07 Jan 04; 01 Jun 04
; Find some element of a linked-list that satisfies a predicate p
(gldefun generic-some ((l linked-list-pointer) p)
  (result (typeof l))
  (let (found res)
    (while (and (not (null l)) (not found)) do
	   (if (funcall p l)
	       (progn (res = l)
		      (found = t)))
	   (l = (rest l)) )
    (if found res (null-value l)) ))


; 04 Oct 91; 07 Jan 04; 01 Jun 04
; Find some element of a linked-list that satisfies a predicate p
(gldefun generic-every ((l linked-list-pointer) p)
  (result boolean)
  (let ((res t))
    (while (and (not (null l)) res) do
	   (if (not (funcall p l))
	       (res = nil)
	       (l = (rest l)) ) )
    res))


; 12 Feb 91; 11 Mar 91; 15 Mar 91; 07 Jan 04
; Implement (l := m) by transferring contents.
(gldefun generic-transfer-contents (l m)
  (if (copy-contents-names (^. l))
      (for nm in (copy-contents-names (^. l)) do
	   ((funcall nm (implementation (^. l))) =
	      (funcall nm (implementation (^. m))) ) )
      ((copy-contents (^. l)) = (copy-contents (^. m))) ) )


; 02 Apr 91; 07 Jan 04
; Implement (l := m) by setting contents of l to be m
(gldefun generic-set-contents (l m)
  (if (copy-contents-names (^. l))
      ((funcall (first (copy-contents-names (^. l)))
		     (implementation (^. l)))
	       = m)
      ((copy-contents (^. l)) = m) ) )

; 10 Jun 04; 14 Jun 04
; Implement equality as equality of contents
(gldefun generic-equal-element (l m)
  (and (equal (typeof l) (typeof m))
       (equal-contents l m (equality-names l)) ) )

(gldefun generic-equal-contents (l m names)
  (or (null names)
      (and ((funcall (car names) (implementation l)) ==
	    (funcall (car names) (implementation m)) )
	   (equal-contents l m (cdr names)) ) ) )

; 03 Jan 91; 12 Feb 91; 11 Mar 91; 19 Jul 91; 10 Jan 97; 07 Jan 04
; 03 Mar 04
; Copy one kind of list, l, to another, m.
; m is a dummy argument that serves only to provide the type.
(gldefun generic-copy-list-to ((l linked-list-pointer) (m linked-list-pointer))
  (result (typeof m))
  (let (cpy (r (typeof m)) newrec)
    (if l is null
	(null-value m)
        (progn (while l is not null do
		    (newrec = (new (^. m)))
		    (transfer-contents newrec l)
		    (if cpy
			((link (^. r)) = newrec)
		        (cpy = newrec))
		    (r = newrec)
		    (l = (rest l)) )
	       ((link (^. r)) = (null-value m))
	       cpy))))


; 19 Jul 90; 26 Nov 90; 26 Oct 94; 10 Jan 97; 07 Jan 04; 03 Mar 04
(gldefun generic-list-new ((l linked-list-pointer))
  (result (typeof l))
  (if ((creation-size l) > 0)
    (let (cpy (newrec (typeof (^. l))) n)
      (n = (creation-size l))
      (while ((n _- 1) >= 0) do
        (newrec = (new (^. l)))
        (if cpy
	    ((link newrec) = cpy)
	    ((link newrec) = (null-value l)))
        (cpy = newrec) )
      cpy)
    (null-value l)))


; 26 Nov 90; 11 Mar 91; 03 Jan 97; 07 Jan 97; 07 Jan 04; 03 Mar 04
(gldefun generic-append ((l linked-list-pointer) (m (typeof l)))
  (result (typeof l))
  (let ((cpy (typeof l)) (r (typeof l)) (newrec (typeof l)))
    (cpy = *glnull*)       ; Java demands an init
    (r = *glnull*)
    (if l is null
	m
        (progn (while l is not null do
		    (newrec = (new (^. l)))
		    (transfer-contents newrec l)
		    (if (null cpy)
			(cpy = newrec)
		        ((rest r) = newrec))
		    (r = newrec)
		    (l = (rest l)) )
	       ((rest r) = m)
	       cpy))))

; 14 Jun 04
(gldefun generic-union ((l linked-list-pointer) (m (typeof l)))
  (result (typeof l))
  (let ((cpy (typeof l)) (r (typeof l)) (newrec (typeof l)))
    (cpy = *glnull*)       ; Java demands an init
    (r = *glnull*)
    (while (not (null l))
      (if (not (member m (first l)))
	  (progn (newrec = (new (^. l)))
		 (transfer-contents newrec l)
		 (if (null cpy)
		     (cpy = newrec)
		     ((rest r) = newrec))
		 (r = newrec)))
      (l = (rest l)) )
    (if (null cpy)
	m
        (progn ((rest r) = m)
	       cpy) ) ))

; 14 Jun 04
(gldefun generic-intersection ((l linked-list-pointer) (m (typeof l)))
  (result (typeof l))
  (let ((cpy (typeof l)) (r (typeof l)) (newrec (typeof l)))
    (cpy = *glnull*)       ; Java demands an init
    (r = *glnull*)
    (while (not (null l))
      (if (member m (first l))
	  (progn (newrec = (new (^. l)))
		 (transfer-contents newrec l)
		 (if (null cpy)
		     (cpy = newrec)
		     ((rest r) = newrec))
		 (r = newrec)))
      (l = (rest l)) )
    cpy ))

; 14 Jun 04
(gldefun generic-set-difference ((l linked-list-pointer) (m (typeof l)))
  (result (typeof l))
  (let ((cpy (typeof l)) (r (typeof l)) (newrec (typeof l)))
    (cpy = *glnull*)       ; Java demands an init
    (r = *glnull*)
    (while (not (null l))
      (if (not (member m (first l)))
	  (progn (newrec = (new (^. l)))
		 (transfer-contents newrec l)
		 (if (null cpy)
		     (cpy = newrec)
		     ((rest r) = newrec))
		 (r = newrec)))
      (l = (rest l)) )
    cpy ))

; 27 Nov 90; 19 Mar 92; 20 May 92; 10 Jan 97
; Push -- note that args are backwards compared to Common Lisp
(gldefun generic-push ((l linked-list-pointer) (newrec (typeof l)))
  ((rest (coercetype newrec (typeof l))) = l)
  (l __ newrec) )


; 27 Nov 90; 11 Mar 91; 10 Jan 97; 01 Dec 03
; Push -- note that args are backwards compared to Common Lisp
(gldefun generic-push-item ((l linked-list-pointer)
			    (item (typeof (contents (^. l)))) )
  (let (newrec)
    (newrec = (new (^. l)))
    (set-contents newrec item)
    ((rest newrec) = l)
    (l __ newrec) ))

; 11 Nov 93
(gldefun generic-sort-before ((l linked-list-pointer) (m linked-list-pointer))
  (< (contents (^. l)) (contents (^. m))) )

; 6-Feb-87 14:59 ; 11 Oct 89; 14 Dec 89; 04 Jan 90; 20 Apr 90; 10 Jan 97
; 07 Jan 04
; Destructively insert the new element NEWREC into list LST 
(gldefun sorted-linked-list-insert ((lst linked-list-pointer)
				    (newrec (typeof lst)))
  (let ((ptr lst) prev)
     (while ptr is not null and (sort-before ptr newrec)
         do (prev = ptr)
	    (ptr = (rest ptr)))
     ((link newrec) = ptr)
     (if prev
	 (progn ((rest prev) = newrec)
		lst)
         newrec)))


; 05 Jan 90; 20 Apr 90; 11 Mar 91; 10 Jan 97; 07 Jan 04
(gldefun sorted-linked-list-insert-key
	 ((lst linked-list-pointer)
	  (newrec (typeof (contents (^. lst)))) )    ; ??? vs. a key value
  (let ((ptr lst) prev newcell)
     (while ptr is not null and (sort-before ptr newrec)     ; ???
         do (prev = ptr)
	    (ptr = (rest ptr)))
     (newcell = (new (^. lst)))
     (set-contents newcell newrec)
     ((link (^. newcell)) = ptr)
     (if prev
	 (progn ((rest prev) = newcell)
		lst)
         newcell)))

; 22 Apr 90; 24 Apr 90; 11 Jun 90; 23 Jan 92; 23 Aug 95; 10 Jan 97; 21 Jan 97
; 28 Feb 02; 07 Jan 04; 03 Mar 04; 01 Jun 04
; Destructively insert the new element NEWREC into list LST 
(gldefun sll-insert ((lst sll-pointer) (newrec (typeof lst)))
  (let ((ptr lst) prev)
     (prev = (null-value lst))
     (while (not (null ptr)) and
            (if (sort-direction (^. lst)) == 'ascending
	        (sort-before (^. ptr) (^. newrec))
	        (not (sort-before (^. ptr) (^. newrec))))
         do (prev = ptr)
	    (ptr = (rest ptr)))
     ((rest newrec) = ptr)
     (if (not (null prev))
	 (progn ((rest prev) = newrec)
		lst)
	 newrec)))

; 22 Apr 90; 24 Apr 90; 08 June 90; 11 Mar 91; 10 Jan 97; 16 Jan 97; 21 Jan 97
; 28 Feb 02; 07 Jan 04; 03 Mar 04; 01 Jun 04
(gldefun sll-insert-key
	 ((lst sll-pointer) (key (typeof (sort-value (^. lst)))) )
  (let ((ptr lst) prev newcell)
     (prev = (null-value lst))
     (newcell = (new (^. lst)))
     (set-contents newcell key)
     (while ptr is not null and
            (if (sort-direction (^. lst)) == 'ascending
	        (sort-before (^. ptr) (^. newcell))
	        (not (sort-before (^. ptr) (^. newcell))))
         do (prev = ptr)
	    (ptr = (rest ptr)))
     ((link (^. newcell)) = ptr)
     (if (not (null prev))
	 (progn ((rest prev) = newcell)
		lst)
	 newcell)))

; 10 Oct 91; 10 Jan 97; 21 Jan 97; 28 Feb 02; 07 Jan 04
; This removes the item whose sort-value equals the supplied value.
(gldefun sll-remove-key
	 ((lst sll-pointer) (key (typeof (sort-value (^. lst)))) )
  (let ((ptr lst) (prev (typeof lst)))
    (prev = (null-value lst))
    (while (not (null ptr)) do
           (if ((sort-value (^. ptr)) == key)
	       (if (not (null prev))
		   ((link (^. prev)) = (link (^. ptr)))
		 (lst = (link (^. ptr))))
	     (prev = ptr))
	   (ptr = (rest ptr)) )
    lst))

; 20 Sep 91; 04 Oct 91; 31 May 95; 21 Jan 97; 28 Feb 02; 07 Jan 04; 01 Jun 04
; Destructively merge the sorted linked lists LST and LSTB
(gldefun sll-merge ((lst sll-pointer) (lstb (typeof lst)))
  (let (front ptr ptrb (prev (typeof lst)))
    (front = (null-value lst))
    (ptr = lst)
    (ptrb = lstb)
    (prev = (null-value lst))
    (while (and (not (null ptr))
		(not (null ptrb)))
      (if (if (sort-direction (^. lst)) == 'ascending
	      (sort-before (^. ptr) (^. ptrb))
	      (not (sort-before (^. ptr) (^. ptrb))))
         (progn (if (not (null prev))
		    ((link (^. prev)) = ptr)
		    (front = ptr))
		(prev = ptr)
		(ptr = (rest ptr)))
         (progn (if (not (null prev))
		    ((link (^. prev)) = ptrb)
		    (front = ptrb))
		(prev = ptrb)
		(ptrb = (rest ptrb))) ))
    (if (null prev)
        (if (null ptr) ptrb ptr)
	(progn (if (null ptr)
		   ((rest prev) = ptrb)
		   ((rest prev) = ptr))
	       front) )))

; 20 Sep 91; 04 Oct 91; 23 Nov 93; 07 Jan 04
; Destructively merge the sorted linked lists LST and LSTB
(gldefun ll-merge ((lst linked-list-pointer) (lstb (typeof lst)))
  (result (typeof lst))
  (let (front (ptr lst) (ptrb lstb) (prev (typeof lst)))
    (if (null lst)
	lstb
        (if (null lstb)
	    lst
	    (progn (while (and (not (null ptr)) (not (null ptrb)))
		    (if (xor ((sort-direction lst) <> 'ascending)
			     (funcall (sort-before-fn lst)
			       (funcall (sort-element-view lst)
					(contents (^. ptr)))
			       (funcall (sort-element-view lst)
					(contents (^. ptrb)))))
			(progn (if prev
				   ((link (^. prev)) = ptr)
			           (front = ptr))
			       (prev = ptr)
			       (ptr = (rest ptr)))
		        (progn (if prev
				   ((link (^. prev)) = ptrb)
			            (front = ptrb))
			       (prev = ptrb)
			       (ptrb = (rest ptrb)) ) ))
	          (if (null ptr)
		      ((rest prev) = ptrb)
	              ((rest prev) = ptr) )
		  front )) )))

; 23 Nov 93; 24 Nov 93; 07 Jan 04
(gldefun ll-merge-sort ((l linked-list-pointer))
  (result (typeof l))
  (let ((front l) mid end tmp)
    (if (null l)
	l
        (if (null (rest l))
	    l
	    (progn (mid = l)
		   (end = (rest l))
		   (while (not (null end))
		     (end = (rest end))
		     (if (not (null end))
			 (progn (end = (rest end))
				(mid = (rest mid)) ) ))
		   (tmp = end)
		   (end = (rest mid))
		   ((rest mid) = tmp)
		   (merge (sort front) (sort end)) ) ) )))

; 25 Jun 04
; Returns the list element that has the max value according to sort order
; ? should this be called max ?
(gldefun generic-max ((l linked-list-pointer))
   (let (found winner)
     (while l is not null do
       (if (not found)
	   (progn (found = t)
		  (winner = l))
	   (if found
	       (if (sort-before winner l)
		   (winner = l))))
       (l = (rest l)))
     winner))

; edited: 12-Jun-87; 11 Oct 89; 21 Nov 89; 20 Dec 90; 10 Jan 97; 04 Apr 00
; 07 Jan 04
; Insert a new element in a two-pointer queue 
(gldefun two-pointer-queue-insert
	 ((self two-pointer-queue) (newrec (typeof (qstart self))))
  (if (qstart self) is null
      ((qstart self) = newrec)
      ((rest (qend self)) = newrec))
  ((qend self) = newrec)
  ((link (^. newrec)) = (null-value (qstart self)))
  self)


; edited: 12-Jun-87; 11 Oct 89; 21 Nov 89; 11 Mar 91; 04 Apr 00; 07 Jan 04
; 28 May 04
; Insert a new element in a two-pointer queue 
(gldefun two-pointer-queue-insert-item
	 ((tpq two-pointer-queue-record)
	  (item (typeof (contents (^. (qstart tpq))))))
  (let (newcell)
    (newcell = (new (^. (qstart tpq))))
    (set-contents newcell item)
    (if (qstart tpq) is null
        ((qstart tpq) = newcell)
        ((rest (qend tpq)) = newcell))
    ((qend tpq) = newcell)
    ((link (^. newcell)) = (null-value (qstart tpq)))
    tpq))


; edited: 15-Jun-87; 11 Oct 89; 21 Nov 89; 13 Jul 90; 26 Nov 90; 19 Feb 91
; 07 Jan 04; 27 Mar 07
; Remove the top entry from a two-pointer queue. 
(gldefun two-pointer-queue-remove ((self two-pointer-queue))
  (result (typeof (implementation (^. (qstart self)))))
  (let (tmp)
    (if (qstart self) is null
        (qstart self)
        (progn (tmp = (qstart self))
	       ((qstart self) = (link (^. tmp)))
               (if ((qstart self) == (null-value (qstart self)))
                   ((qend self) = (null-value (qstart self))))
	       ((link (^. tmp)) = (null-value (qstart self)))
	       tmp) )))


; edited: 13 Jul 90; 07 Jan 04; 27 Mar 07
; Remove the top entry from a two-pointer queue. 
(gldefun two-pointer-queue-remove-item ((self two-pointer-queue))
  (let (tmp)
    (if (qstart self) is null
        nil
        (progn (tmp = (qstart self))
	       ((qstart self) = (link (^. tmp)))
               (if ((qstart self) == (null-value (qstart self)))
                   ((qend self) = (null-value (qstart self))))
	       ((link (^. tmp)) = (null-value (qstart self)))
	       (copy-contents (^. tmp)) ) )))


; 22 Nov 89; 05 Dec 89; 22 May 90; 19 Sep 91; 10 Jan 97; 07 Jan 04
; Insert a new record at the end of a front-pointer-queue
(gldefun front-pointer-queue-insert
	 ((fpq front-pointer-queue) (newrec (typeof (qstart fpq))))
  (if (qstart fpq) is null
      ((qstart fpq) = newrec)
      ((rest (last (qstart fpq))) = newrec))
  ((link (^. newrec)) = (null-value (qstart fpq)))
  fpq)

; 22 Nov 89; 05 Dec 89; 22 May 90; 11 Mar 91; 07 Jan 04
; Insert a new item at the end of a front-pointer queue,
; making a new list cell for the item.
(gldefun front-pointer-queue-insert-item
	 ((fpq front-pointer-queue)
          (item (typeof (contents (^. (qstart fpq))))))
  (let (newcell)
    (newcell = (new (^. (qstart fpq))))
    (set-contents newcell item)
    (if (qstart fpq) is null
        ((qstart fpq) = newcell)
        ((rest (last (qstart fpq))) = newcell))
    ((link (^. newcell)) = (null-value (qstart fpq)))
    fpq))


; edited: 13 Jul 90; 16 Jul 90; 19 Sep 91; 10 Jan 97; 07 Jan 04
; Insert a new element in an end-pointer circular queue 
(gldefun end-pointer-queue-insert
	 ((self end-pointer-queue) (newrec (typeof (qend self))))
  (if (qend self) is null
      (progn ((qend self) = newrec)
	     ((link (^. newrec)) = newrec))
      (progn ((link (^. newrec)) = (link (^. (qend self))))
	     ((link (^. (qend self))) = newrec)
	     ((qend self) = newrec) ))
  nil )


; edited: 13 Jul 90; 16 Jul 90; 11 Mar 91; 07 Jan 04
; Insert a new element in an end-pointer circular queue
(gldefun end-pointer-queue-insert-item
	 ((self end-pointer-queue)
	  (item (typeof (contents (^. (qend self))))))
  (let (newcell)
    (newcell = (new (^. (qend self))))
    (set-contents newcell item)
    (if (qend self) is null
        (progn ((qend self) = newcell)
	       ((link (^. newcell)) = newcell))
        (progn ((link (^. newcell)) = (link (^. (qend self))))
	       ((link (^. (qend self))) = newcell)
	       ((qend self) = newcell) ))
    nil ))


; edited: 13 Jul 90; 16 Jul 90; 26 Nov 90; 19 Feb 91; 07 Oct 91; 28 Feb 02
; 07 Jan 04
; Remove the top entry from an end-pointer circular queue 
(gldefun end-pointer-queue-remove ((self end-pointer-queue))
  (result (typeof (implementation (^. (qend self)))))
  (let (tmp)
    (if (qend self) is null
        (qend self)
        (progn (tmp = (link (^. (qend self))))
	       (if (tmp == (qend self))
		   ((qend self) = (null-value (qend self)))
                   ((link (^. (qend self))) = (link (^. tmp))))
	       ((link (^. tmp)) = (null-value (qend self)))
	       tmp) )))


; edited: 13 Jul 90; 16 Jul 90; 28 Feb 02; 07 Jan 04
; Remove the top entry from an end-pointer circular queue 
(gldefun end-pointer-queue-remove-item ((self end-pointer-queue))
  (let (tmp)
    (if (qend self) is null
        nil
        (progn (tmp = (link (^. (qend self))))
	       (if (tmp == (qend self))
		   ((qend self) = (null-value (qend self)))
                   ((link (^. (qend self))) = (link (^. tmp))))
	       ((link (^. tmp)) = (null-value (qend self)))
	       (copy-contents (^. tmp)) ) )))

; 22 Nov 89; 10 Jan 97
(gldefun priority-queue-insert ((q priority-queue) (n integer) newrec)
  (insert (index q n) newrec) )

; 28 Nov 89; 10 Jan 97
(gldefun priority-queue-insert-item ((q priority-queue) (n integer) newrec)
  (insert-item (index q n) newrec) )

; 22 Nov 89
(gldefun priority-queue-remove ((q priority-queue) (n integer))
  (remove (index q n)) )

; 13 Jul 90
(gldefun priority-queue-remove-item ((q priority-queue) (n integer))
  (remove-item (index q n)) )

; 31 Jul 01; 28 Feb 02; 07 Jan 04; 06 Aug 07
(setf (glinstancename 'alist-member) 'member)
(gldefun alist-member ((l alist-pointer) (key (typeof (sort-value (^. l)))))
  (result (typeof l))
  (let (done)
    (done = *glfalse*)
    (while (not done)
      (if (null l)
          (done = *gltrue*)
          (if (key == (sort-value (^. l)))
              (done = *gltrue*)
              (l = (rest l)) ) ) )
    l))

; 02 Aug 01
(setf (glinstancename 'alist-insert-key) 'insert-key)
(gldefun alist-insert-key ((head alist-pointer)
			      (key (typeof (sort-value (^. head)))))
  (result (typeof head))
  (let ((newrec (typeof head)))
    (newrec = (new (^. head)))
    ((sort-value (^. newrec)) = key)
    ((rest newrec) = head)
    newrec))
