; gentr.lsp               Gordon S. Novak Jr.            ; 02 Apr 09

; Copyright (c) 2009 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Generic tree abstract data types and functions

; 03 Dec 93; 05 Jan 95; 08 Aug 95; 10 Aug 95; 28 Sep 95; 12 Nov 96; 30 Dec 96
; 10 Jan 97; 23 Jan 97; 27 Jan 97; 28 Jan 97; 13 Feb 97; 06 Mar 97; 14 Mar 97
; 28 Feb 02; 18 Mar 03; 18 Nov 03; 19 Nov 03; 08 Jan 04; 03 Mar 04; 29 May 04
; 28 May 04; 01 Jun 04; 21 Sep 04; 09 Aug 06; 11 Aug 06; 12 Feb 07; 13 Feb 07

; 07 Sep 90; 10 Sep 90; 20 Sep 90; 14 Mar 91; 15 Mar 91; 26 Mar 93; 01 Mar 07
(gldefclusterc
  'tree
  '((pointer (tree-pointer (^ tree-record)
			   ; (or (^ tree-record) (^ tree-terminal))
       adj     ((terminal        ( (null (descendants (^. self))))))
       prop    ((dereference     (self) result (clustertype (typeof self)
							    'record))
		(size            generic-tree-size specialize t)
		(null-value      (*glnull*)))
       msg     ((new             generic-tree-new  specialize t
				 result (typeof self))
		(transfer-contents generic-transfer-contents open t))
      default  ((self nil))  ))

    (record (tree-record (tuple (contents anything)
                                (descendants (setof tree-pointer)))
        prop   ((copy-contents (contents))
		(implementation       (self) result (strof (typeof self)))
		(copy-contents-names  (nil))
		(contents      (self)))
	msg    ((new ((a (typeof self))) result (clustertype (typeof self)
							     'pointer)))))

    (terminal (tree-terminal anything
                prop ((contents (self))))) )
  '()
 )


; 15 Mar 91; 19 Mar 91; 11 Apr 91; 23 Jan 97
; A uniform tree is one in which all nodes have the same type.
(gldefclusterc
  'uniform-tree
  '((pointer (uniform-tree-pointer (^ uniform-tree-record)
       adj     ((terminal        (self == (null-value self))))
       msg     ((member          uniform-tree-member  specialize t)
		(copy            uniform-tree-copy    specialize t)
		(copy-to         uniform-tree-copy-to specialize t)
		(iterator        uniform-tree-iterator) ) ))

    (record  (uniform-tree-record
	       (tuple (contents anything)
		      (descendants (setof uniform-tree-pointer))) )))
			            ; ^ should be sequence of
  '(record-with-pointer tree) )


; 30 Sep 93; 05 Oct 93; 18 Nov 93; 19 Nov 93; 28 Sep 95; 23 Jan 97; 18 Nov 03
; 19 Nov 03; 28 May 04; 21 Sep 04; 01 Mar 07
; Binary tree with info in the records
(gldefclusterc
  'binary-tree
  '((pointer (binary-tree-pointer (^ binary-tree-record)
       prop    ((size              binary-tree-size    specialize t)
		(order-one-access  (nil)) )
       msg     ((set-contents      generic-set-contents open t)
		(transfer-contents generic-transfer-contents open t)
		(member            binary-tree-member  specialize t)
		(minval            binary-tree-minval  specialize t)
		(maxval            binary-tree-maxval  specialize t)
		(copy              binary-tree-copy    specialize t)
		(copy-to           binary-tree-copy-to specialize t)
		(flatten           binary-tree-flatten specialize t)
		(flattenb          binary-tree-flattenb specialize t)
		(iterator          binary-tree-iterator)
		(newempty          (nil) result (typeof self))
		(init              (glambda (self) )) ) ))
    (record  (binary-tree-record
	       (tuple (contents anything)
		      (left     binary-tree-pointer)
		      (right    binary-tree-pointer) )
               prop ((sort-value (implementation))
		     (sort-direction ('ascending))
		     (descendants binary-tree-descendants open t
				  result
				   (listof (clustertype (typeof self)
							'pointer))))
               msg  ((sort-before (glambda (x y)
				    (< (sort-value x) (sort-value y))))
		     (new ((a (typeof self))) result (typeof (left self)))))) )
  '(uniform-tree) )

(setf (get 'binary-tree 'glnpointers) 2)

; 08 Aug 95; 10 Jan 97
(setf (get 'binary-tree-pointer 'ap-omit)
      '(new iterator copy-to transfer-contents set-contents null terminal
	    null-value dereference flattenb))

; 23 Jan 97
; java-static = t for functions that could be called with a null pointer arg
(mapc #'(lambda (x) (setf (get x 'java-static) t))
      '(binary-tree-size binary-tree-member binary-tree-copy
        binary-tree-copy-to binary-tree-flatten binary-tree-flattenb
	avl-tree-insert	avl-tree-insert-key))

(setf (glcarrier 'binary-tree)
      '((list (contents contents-type)
	      (left pointer)
	      (right pointer))
	((left left) (right right))) )

; 05 Oct 93; 15 Nov 93; 02 Apr 09
; AVL (Adelson-Velskii & Landis) balanced binary tree
(gldefclusterc
  'avl-tree
  '((pointer (avl-tree-pointer (^ avl-tree-record)
       msg     ((insert            avl-tree-insert  specialize t)
                (insert-item       avl-tree-insert  specialize t)
		(insert-key        avl-tree-insert-key  specialize t))))

    (record  (avl-tree-record
	       (tuple (contents anything)
		      (balance  integer)               ; -1, 0, +1
		      (left     avl-tree-pointer)
		      (right    avl-tree-pointer) ) ) ) )
  '(binary-tree) )

(setf (get 'avl-tree 'glnpointers) 2)

(setf (glcarrier 'avl-tree)
      '((list (contents contents-type)
	      (left pointer)
	      (right pointer)
	      (balance integer))
	((left left) (right right) (balance balance))) )

; 15 Mar 91; 19 Mar 91; 26 Mar 93; 10 Jan 97; 23 Jan 97; 18 Mar 03
; A fringe tree has the data nodes at the fringes only
(gldefclusterc
  'fringe-tree
  '((pointer (fringe-tree-pointer (^ fringe-tree-record)
				  ; (or (^ fringe-tree-record)
				  ;     (^ fringe-tree-terminal))
       prop    ((size            generic-tree-size specialize t)
		(terminals-size  generic-tree-terminals-size specialize t))
       msg     ((member          generic-tree-member specialize t)
		(copy            generic-tree-copy specialize t)
		(copy-to         generic-tree-copy-to specialize t)
		(iterator        tree-iterator) ) ) )

    (record  (fringe-tree-record (tuple (contents anything)
                                 (descendants (setof fringe-tree-pointer))) ))

    (terminal (fringe-tree-terminal anything)) )
  '(tree)
 )


; 20 Sep 90; 15 Mar 91; 16 Apr 91; 03 Dec 93
(gldefclusterc
  'lisp-tree
  '((pointer (lisp-tree-pointer (^ lisp-tree-record)
		adj  ((terminal atom))
                prop ((dereference ((if self is terminal
					(deref-t self)
				        (deref-n self))))
		      (deref-t (self) result (clustertype (typeof self)
							  'terminal))
		      (deref-n (self) result (clustertype (typeof self)
							  'record)))))
  ;  (^.t      (self) result (clustertype (typeof self) 'terminal))
    (record  (lisp-tree-record (cons (contents anything)
			   	     (desc (listof lisp-tree-pointer)))
		prop ((descendants ((if self is not atomic
					desc))))
                msg  ((new ((cons nil nil))
			   result (clustertype (typeof self) 'pointer)))))
    (terminal (lisp-tree-terminal atom
                prop ((contents (self))) ) ) )
  '(fringe-tree))


; 18 Sep 90; 20 Sep 90
(gldefclusterc
  'lisp-cons-tree
  '((pointer  (lisp-cons-tree-pointer (^ lisp-cons-tree-record)
		adj  ((terminal atom))
                prop ((^.t      (self) result (clustertype (typeof self)
							   'terminal)))))
    (record   (lisp-cons-tree-record
	        (cons (left  lisp-cons-tree-pointer)
		      (right lisp-cons-tree-pointer))
	        prop ((descendants ((list left right))))    ; tuple ?
                msg  ((new ((cons nil nil))
			   result (clustertype (typeof self) 'pointer)))))
    (terminal (lisp-cons-tree-terminal atom
                prop ((contents (self))))) )
  '(tree))


(gldefclusterc
  'elf-tree
  '((pointer (elf-tree-pointer (^ elf-tree-record)
		prop ((null-value ('nobody)))
		adj  ((terminal (self == 'nobody))) ) )
    (record  (elf-tree-record  (z123 elf)
	       prop ((descendants ((friends z123)) result elf-tree-ll-pointer)
		     (copy-contents-names    ('(name age))))
               msg  ((new ((an elf)) result (clustertype (typeof self)
							 'pointer))))))
  '(uniform-tree))

(pushnew '(tree elf-tree-pointer) (glviews 'elf))


(gldefclusterc
  'elf-tree-ll
  '((pointer  (elf-tree-ll-pointer (^ elf-tree-ll-record)))
    (record   (elf-tree-ll-record (z elf)
		prop ((link ((buddy z)) result elf-tree-ll-pointer))
	        msg  ((new ((an elf)) result elf-tree-ll-pointer) ) )) )
  '(elf-ll))

(glispobjects 
  (token (list (type symbol)
	       (subtype anything)
	       (numval number)
	       (operands (^ token))
	       (side-link (^ token)))
  prop ((op ((nth subtype '(nil + - * / = = <> < <= >= > ^ \. and or not))))
	(id ((intern (string-trim '(#\Space) subtype))))
	(tokval ((if type == 'operator
		     op
		     (if type == 'identifier
			 id
		         (if type == 'number
			     numval))))))
  adj  ((terminal (type <> 'operator)))
 views ((tree token-tree-pointer))  )
 )  ; glispobjects 


; 15 Feb 91
(gldefclusterc
  'token-tree
  '((pointer (token-tree-pointer (^ token-tree-record)))
    (record  (token-tree-record (zz token)
	       prop ((descendants ((operands zz))
				  result TOKEN-AS-LL-POINTER)
		     (contents ((tokval zz))))
               msg  ((new ((a token))
			  result (clustertype (typeof self) 'pointer))))))
  '(tree))

; 09 Aug 06; 12 Feb 07; 13 Feb 07
(gldefclusterc
  'heap-tree
  '((pointer (heap-tree-pointer (^ heap-tree-record)
       prop    ((left-depth     heap-tree-left-depth  specialize t)
                (right-depth    heap-tree-right-depth  specialize t))
       msg     ((insert         heap-tree-insert  specialize t)
		(insert-key     heap-tree-insert-key  specialize t)
                (insert-bottom  heap-tree-insert-bottom  specialize t)
		(okay           ((not (null self))))
		(left           (glambda (self) (leftptr (^. self))))
		(right          (glambda (self) (rightptr (^. self))))
		(parent         (glambda (self) (parentptr (^. self))))
		(sortp          (glambda (i j)
			          (if (sort-direction (^. i)) == 'ascending
				      (funcall (sort-fn (^. i))
					       (contents (^. i))
					       (contents (^. j)))
				      (not (funcall (sort-fn (^. i))
						    (contents (^. i))
						    (contents (^. j)))))))
		(heapify        heap-tree-heapify specialize t)
	     ) ))

    (record  (heap-tree-record
	       (tuple (contents anything)
		      (parentptr heap-tree-pointer)
		      (leftptr   heap-tree-pointer)
		      (rightptr  heap-tree-pointer) )
	       prop ((sort-value (contents))
		     (sort-direction ('ascending))
		     (sort-fn   ('<)) ) ) ) )
  '(binary-tree) )

(gldefclusterc
  'my-heap
  '((pointer (my-heap-pointer (^ my-heap-record)))
    (record  (my-heap-record
	       (list (contents integer)
		      (parent   my-heap-pointer)
		      (leftptr  my-heap-pointer)
		      (rightptr my-heap-pointer) ) ) ) )
  '(heap-tree))

; 13 Feb 07
(gldefclusterc
  'heapz
  '((pointer (heap-pointer integer
      prop  ((parent   ((truncate self 2)))
             (left     ((* self 2)))
             (right    ((+ 1 (* self 2)))) ) ))
    (record  (heap-record (arrayof anything)
      prop  ((maxsize  array-total-size)) ) ) )
  '())

; 07 Sep 90; 09 Sep 90; 14 Mar 91; 08 Jan 04; 01 Jun 04
; Iterate to produce all nodes of tree
(gldefmacro tree-iterator (tree) (item condition code)
  (prog ( (ptr (a (typeof tree))) item stack)
    (ptr = tree)
 lp (item =  ptr)               ; vs.  (implementation (^. ptr)) for list?
    (if condition (progn . code))
    (if (not (terminal ptr))
        (stack +_ (descendants (^. ptr))) )
lpb (if (null (first stack))
	(if (null stack)
	    (return)
	    (progn (stack = (rest stack))
		   (go lpb)))
        (progn (ptr -_ (first stack))
	       (go lp) ) )))


; 15 Mar 91; 08 Jan 04; 01 Jun 04
; Iterate to produce all nodes of uniform tree, including interior nodes.
(gldefmacro uniform-tree-iterator (tree) (item condition code)
  (prog ( (ptr (a (typeof tree))) item stack)
    (ptr = tree)
 lp (item = (implementation (^. ptr)))
    (if (not (terminal ptr))
        (progn (if condition (progn . code))
	       (stack +_ (descendants (^. ptr))) ))
lpb (if (null stack)
        (return)
        (if (null (first stack))
	    (progn (stack = (rest stack))
		   (go lpb))
            (progn (ptr = (pop (first stack)))
		   (go lp))) )))


; 01 Nov 93; 11 Nov 93; 08 Jan 04
; Iterate to produce all nodes of tree
(gldefmacro binary-tree-iterator (tree) (item condition code)
  (prog ( (ptr (a (typeof tree))) item stack)
    (ptr = tree)
    (stack = nil)
 lp (if (null ptr)
        (if (null stack)
	    (return)
	    (progn (ptr -_ stack)
		   (item = (implementation (^. ptr)))
		   (if condition (progn . code))
		   (ptr = (right (^. ptr)))
		   (go lp)))
        (progn (stack +_ ptr)
	       (ptr = (left (^. ptr)))
	       (go lp)) )))


; 14 Mar 91; 08 Jan 04
     ; ***** must be edited -- just a copy of linked-list-collector
(gldefmacro tree-collector (tree m) (item condition code)
  (let (val cpy (end (typeof m)) new)
    (for item in seq when condition do
	 (val = (progn . code))
	 (new = (new (^. m)))
	 ((copy-contents (^. new)) = val)     ; needs a transfer method
	 (if cpy
	     ((link (^. end)) = new)
	     (cpy = new))
	 (end = new))
    ((link (^. end)) = (null-value m))
    cpy))


; 09 Sep 90
(gldefun generic-tree-size ((tr tree-pointer))
  (let ((n 0)) (for x in tr do (n _+ 1)) n))

; 03 Mar 04
(gldefun generic-tree-terminals-size ((tr tree-pointer))
  (let ((n 0))
    (for x in tr do
	 (if x is terminal (n _+ 1)))
    n))


; 18 Sep 90; 20 Sep 90; 23 Jan 97; 08 Jan 04
(gldefun generic-tree-copy ((l tree-pointer))
  (result (typeof l))
    (if (terminal l)
	(^.t l)
        (a (typeof (^. l)) with contents = (contents (^. l))
		descendants = (for x in (descendants (^. l))
				   collect (copy x)))))


; 15 Mar 91; 02 Apr 91; 11 Apr 91; 23 Jan 97; 08 Jan 04
(gldefun uniform-tree-copy ((l uniform-tree-pointer))
  (result (typeof l))
  (let (m)
    (if (terminal l)
	l
        (progn (m = (new (^. l)))
	       (transfer-contents m l)
	       ((descendants (^. m)) =
		  (for x in (descendants (^. l))
		     collect as (typeof (descendants (^. l)))
		     (copy (tree x))))
	       m) )))


; 11 Apr 91; 08 Jan 04
; Needs work.  Another challenge: equality of diffferent kinds of trees.
(gldefun uniform-tree-equal ((l uniform-tree-pointer) (m uniform-tree-pointer))
  (result boolean)
  (if (terminal l)
      (terminal m)
      (progn (m = (new (^. l)))
	     (for x in (descendants (^. l))     ; or every ??
		  as y in (descendants (^. m))
		  (equal (tree x) (tree y)))) ))


; 15 Feb 91; 19 Feb 91; 14 Mar 91; 08 Jan 04
; Copy one type of tree to another
(gldefun generic-tree-copy-to ((l tree-pointer) (m tree-pointer))
  (result (typeof m))
  (let (node)
    (if (terminal l)
	(^.t m)
        (progn (node = (new (^. m)))
	       ((contents (^. node)) = (contents (^. l)))
	       ((descendants (^. node)) = 
		 (for x in (descendants (^. l))
		    collect 
		  ;  as (typeof (descendants (^. m))      ; 14 Mar 91
		    (copy-to (tree x) m)))
	       node) )))

; 19 Nov 93; 30 Nov 93; 08 Jan 04
(gldefun binary-tree-descendants ((r binary-tree-record))
  (if (not (null (left r)))
      (cons (left r) (if (not (null (right r))) (list (right r))))
      (if (not (null (right r))) (list (right r)))) )


; ! Depth-first search = member(solution) in a virtual tree!
; 18 Sep 90; 20 Sep 90; 08 Jan 04
(gldefun generic-tree-member ((l tree-pointer)
			      (item (typeof (contents (^.t l)))))
  (if l is terminal
      (equal (contents (^.t l)) item)
      (for d in (descendants (^. l)) do
	   (if (member d item) (return t)))) )

; 30 Sep 93; 01 Oct 93; 28 Feb 02; 08 Jan 04
(setf (glinstancename 'binary-tree-member) 'member)
(gldefun binary-tree-member ((l binary-tree-pointer)
			     (key (typeof (sort-value (^. l)))))
  (result (typeof l))
  (if (null l)
      l
      (if (key == (sort-value (^. l)))
	  l
	  (if (key < (sort-value (^. l)))
	      (member (left (^. l)) key)
	      (member (right (^. l)) key) ) ) ) )

; 01 Mar 07
; (setf (glinstancename 'binary-tree-minval) 'minval)
(gldefun binary-tree-minval ((l binary-tree-pointer))
  (result (typeof l))
  (if (null l)
      l
      (if (null (left (^. l)))
	  l
          (minval (left (^. l))) ) ) )

; 01 Mar 07
; (setf (glinstancename 'binary-tree-maxval) 'maxval)
(gldefun binary-tree-maxval ((l binary-tree-pointer))
  (result (typeof l))
  (if (null l)
      l
      (if (null (right (^. l)))
	  l
          (maxval (right (^. l))) ) ) )

; 28 Sep 95
; Flatten a binary tree destructively into a linked list using right pointer.
; The result is sorted the same way as the tree.
(setf (glinstancename 'binary-tree-flatten) 'flatten)
(gldefun binary-tree-flatten ((l binary-tree-pointer))
  (result (typeof l))
  (flattenb l nil) )

; 28 Sep 95; 10 Jan 97; 08 Jan 04
(setf (glinstancename 'binary-tree-flattenb) 'flattenb)
(gldefun binary-tree-flattenb ((l binary-tree-pointer) (lst (typeof l)))
  (result (typeof l))
  (let (res)
    (if (not (null l))
        (progn ((right (^. l)) = (flattenb (right (^. l)) lst))
	       (res = (flattenb (left (^. l)) l))
	       ((left (^. l)) = (null-value l)))
	     res
        lst) ))

; 23 Jan 97; 08 Jan 04
; size (number of elements) of a binary tree.
(setf (glinstancename 'binary-tree-size) 'size)
(gldefun binary-tree-size ((l binary-tree-pointer))
  (result integer)
    (if (null l)
	0
        (+ 1 (size (left (^. l))) (size (right (^. l))) ) ) )

; 23 Jan 97; 08 Jan 04
; cf. generic-copy-list
(setf (glinstancename 'binary-tree-copy) 'copy)
(gldefun  binary-tree-copy((l binary-tree-pointer))
  (result (typeof l))
  (let ((newrec (typeof l)))
    (if (null l)
	l
        (progn (newrec = (new (^. l)))
	       (transfer-contents newrec l)
	       ((left (^. newrec)) = (copy (left (^. l))))
	       ((right (^. newrec)) = (copy (right (^. l))))
	       newrec) )))

; 23 Jan 97; 08 Jan 04
; cf. generic-copy-list-to
(gldefun  binary-tree-copy-to((l binary-tree-pointer) (m binary-tree-pointer))
  (result (typeof m))
  (let ((newrec (typeof m)))
    (if (null l)
	(null-value m)
        (progn (newrec = (new (^. m)))
	       (transfer-contents newrec l)
	       ((left (^. newrec)) = (copy (left (^. l))))
	       ((right (^. newrec)) = (copy (right (^. l))))
	       newrec) )))

(setf (glinfo 'avl-tree-insert)
      '("This function inserts a new record into a (possibly empty)"
	"AVL tree, an approximately height-balanced binary tree,"
	"rebalancing the tree as needed."
	"If the value is already in the tree, the tree is left unchanged."
	"The new root of the tree is returned; use an assignment:"
	"   tree = insert(tree, newrec)"
	"The algorithm is adapted from Knuth vol. 3, p. 455."))

; 04 Oct 93; 05 Oct 93; 06 Oct 93; 16 Nov 93; 17 Nov 93; 26 Dec 96; 30 Jan 96
; 23 Jan 97; 28 Feb 02; 08 Jan 04; 29 May 04; 01 Jun 04
; Insert a new element into a binary AVL tree.
; Adapted from Knuth vol. 3, p. 455.
; Returns the root of the resulting tree (possibly different from old root).
; newtr is a new record to be inserted; see also avl-tree-insert-key,
; whose argument is a key value rather than a record.
; head may be null, in which case newtr will become the head.
; Note that key values are assumed to be unique, so that an attempt to
; insert a record whose key value is already in the tree will do nothing.
(setf (glinstancename 'avl-tree-insert) 'insert)
(gldefun avl-tree-insert ((head binary-tree-pointer) (newtr (typeof head)))
  (result (typeof head))
  (let (tt s p q r aa found done)
    (if (null head)
      (progn ( (left (^. newtr)) = (null-value head) )  ; special case: empty
	     ( (right (^. newtr)) = (null-value head) )
	     ( (balance (^. newtr)) = 0)
	     newtr)
      (progn (tt = head)                                    ; step a1
           (s = head)
           (p = head)
	   (q = (null-value head))                        ; or Java complains
	   (found = *glfalse*)
	   (done = *glfalse*)
	   (while (not done) do
	     (if ( (sort-value (^. newtr)) == (sort-value (^. p)) )   ; step a2
		 (progn (found = t) (done = t))
		 (if ( (sort-value (^. newtr)) < (sort-value (^. p)) )
		     (if (null (left (^. p)))     ; step a3: move L
			 (progn (q = newtr)
				((left (^. p)) = q)
				(done = t))
		         (progn (q = (left (^. p)))
				(if ( (balance (^. q)) <> 0 )
				    (progn (tt = p)
					   (s = q)))
				(p = q)))
		     (if (null (right (^. p)))    ; step a4: move R
			 (progn (q = newtr)
				((right (^. p)) = q)
				(done = t))
		         (progn (q = (right (^. p)))
				(if ( (balance (^. q)) <> 0 )
				    (progn (tt = p)
					   (s = q)))
				(p = q)) ) ) ))
	   (if found
	       head
	     (progn ( (left (^. q)) = (null-value head) )             ; step a5
	          ( (right (^. q)) = (null-value head) )
		  ( (balance (^. q)) = 0)
		  (if ((sort-value (^. newtr)) < (sort-value (^. s)))
		      (r = (left (^. s)))
		      (r = (right (^. s))) )
		  (p = r)
		  (while (p <> q) do                                  ; step a6
		    (if ((sort-value (^. newtr)) < (sort-value (^. p)))
			(progn ( (balance (^. p)) = -1)
			       (p = (left (^. p))))
			(progn ( (balance (^. p)) = 1)
			       (p = (right (^. p))) ) ))
		  (if ((sort-value (^. newtr)) < (sort-value (^. s))) ; step a7
		      (aa = -1)
		      (aa = 1))
		  (if ( (balance (^. s)) == 0 )
		      (progn ( (balance (^. s)) = aa )
			     head)                            ; done
		      (if ( (balance (^. s)) == (- aa) )
			  (progn ( (balance (^. s)) = 0 )
				 head )                  ; done
			  (progn (if ( (balance (^. r)) == aa ) ; rebalance
				     (progn (p = r)     ; step a8
					    (if (aa == 1)
						(progn ((right (^. s)) = 
							 (left (^. r)))
						       ((left (^. r)) = s))
					        (progn ((left (^. s)) = 
						         (right (^. r)))
						       ((right (^. r)) = s) ))
					    ( (balance (^. s)) = 0)
					    ( (balance (^. r)) = 0))
				     (progn (if (aa == 1)  ; step a9
						(progn (p = (left (^. r)))
						       ((left (^. r)) = 
						        (right (^. p)))
						       ((right (^. p)) = r)
						       ((right (^. s)) =
						         (left (^. p)))
						       ((left (^. p)) = s))
					        (progn (p = (right (^. r)))
						       ((right (^. r)) = 
						        (left (^. p)))
						       ((left (^. p)) = r)
						       ((left (^. s)) =
						         (right (^. p)))
						       ((right (^. p)) = s) ))
					    (if ((balance (^. p)) == 0)
						(progn ((balance (^. s)) = 0)
						       ((balance (^. r)) = 0))
					        (progn (if ((balance (^. p)) == aa)
							   (progn ((balance (^. s)) =
								   (- aa))
								  ((balance (^. r)) = 0))
							   (progn ((balance (^. s)) = 0)
								  ((balance (^. r)) = aa)
						      ))))
					    ((balance (^. p)) = 0)))
				 (if ( s == (right (^. tt)))  ; modified a10
				     (progn ( (right (^. tt)) = p)
					    head)
				   (if ( s == (left (^. tt)))
				       (progn ( (left (^. tt)) = p)
					      head)`
				       p)))))) )))))

(setf (glinfo 'avl-tree-insert-key)
      '("This function makes a new record containing a specified key value"
	"and inserts it into a (possibly empty) AVL tree, an approximately"
	"height-balanced binary tree, rebalancing the tree as needed."
	"If the value is already in the tree, the tree is left unchanged."
	"The new root of the tree is returned; use an assignment:"
	"   tree = insert-key(tree, newkey)"
	"The algorithm is adapted from Knuth vol. 3, p. 455."))

; 04 Oct 93; 05 Oct 93; 06 Oct 93; 15 Nov 93; 17 Nov 93; 26 Dec 96; 30 Dec 96
; 28 Feb 02; 08 Jan 04; 01 Jun 04
; Insert a (possibly) new element into a binary AVL tree (Knuth vol. 3 p. 455)
; Returns the root of the resulting tree (possibly different from old root).
(setf (glinstancename 'avl-tree-insert-key) 'insert-key)
(gldefun avl-tree-insert-key ((head binary-tree-pointer)
			      (key (typeof (sort-value (^. head)))))
  (result (typeof head))
  (let (tt s p q r aa found done newtr)
    (if (null head)
	(progn (newtr = (new (^. head)))              ; special case: empty tree
	       ( (sort-value (^. newtr)) = key)
	       newtr)
      (progn (tt = head)                                    ; step a1
           (s = head)
           (p = head)
	   (q = (null-value head))                        ; or Java complains
	   (found = *glfalse*)
	   (done = *glfalse*)
	   (while (not done) do
	     (if ( key == (sort-value (^. p)) )              ; step a2
		 (progn (found = t) (done = t))
		 (if ( key < (sort-value (^. p)) )
		     (if (null (left (^. p)))     ; step a3: move L
			 (progn (q = (new (^. head)))
				((left (^. p)) = q)
				(done = t))
		         (progn (q = (left (^. p)))
				(if ( (balance (^. q)) <> 0 )
				    (progn (tt = p)
					   (s = q)))
				(p = q)))
		     (if (null (right (^. p)))    ; step a4: move R
			 (progn (q = (new (^. head)))
				((right (^. p)) = q)
				(done = t))
		         (progn (q = (right (^. p)))
				(if ( (balance (^. q)) <> 0 )
				    (progn (tt = p)
					   (s = q)))
				(p = q)) ) ) ))
	   (if found
	       head
	     (progn ( (sort-value (^. q)) = key )           ; step a5
	          ( (left (^. q)) = (null-value head) )
	          ( (right (^. q)) = (null-value head) )
		  ( (balance (^. q)) = 0)
		  (if (key < (sort-value (^. s)))
		      (r = (left (^. s)))
		      (r = (right (^. s))) )
		  (p = r)
		  (while (p <> q) do                        ; step a6
		    (if (key < (sort-value (^. p)))
			(progn ( (balance (^. p)) = -1)
			       (p = (left (^. p))))
			(progn ( (balance (^. p)) = 1)
			       (p = (right (^. p))) ) ))
		  (if (key < (sort-value (^. s)))          ; step a7
		      (aa = -1)
		      (aa = 1))
		  (if ( (balance (^. s)) == 0 )
		      (progn ( (balance (^. s)) = aa )
			     head  )                          ; done
		      (if ( (balance (^. s)) == (- aa) )
			  (progn ( (balance (^. s)) = 0 )
				 head )                  ; done
			  (progn (if ( (balance (^. r)) == aa ) ; rebalance
				     (progn (p = r)     ; step a8
					    (if (aa == 1)
						(progn ((right (^. s)) = 
						         (left (^. r)))
						       ((left (^. r)) = s))
					      (progn ((left (^. s)) = 
						         (right (^. r)))
						     ((right (^. r)) = s) ))
					    ( (balance (^. s)) = 0)
					    ( (balance (^. r)) = 0))
				     (progn (if (aa == 1)  ; step a9
						(progn (p = (left (^. r)))
					              ((left (^. r)) = 
						        (right (^. p)))
						      ((right (^. p)) = r)
						      ((right (^. s)) =
						         (left (^. p)))
						      ((left (^. p)) = s))
					        (progn (p = (right (^. r)))
					              ((right (^. r)) = 
						        (left (^. p)))
						      ((left (^. p)) = r)
						      ((left (^. s)) =
						         (right (^. p)))
						      ((right (^. p)) = s) ))
					    (if ((balance (^. p)) == 0)
						 (progn ((balance (^. s)) = 0)
							((balance (^. r)) = 0))
					        (if ((balance (^. p)) == aa)
						    (progn ((balance (^. s)) =
							    (- aa))
							   ((balance (^. r)) = 0))
						    (progn ((balance (^. s)) = 0)
							   ((balance (^. r)) = aa)
						      )))
					    ((balance (^. p)) = 0)))
				 (if ( s == (right (^. tt)))  ; modified a10
				     (progn ( (right (^. tt)) = p)
					    head)
				     (if ( s == (left (^. tt)))
					 (progn ( (left (^. tt)) = p)
						head)
				         p)))))) )))))

; AVL tree deletion: http://geocities.com/wkaras/gen_c/cavl_impl_h.txt

; Following is some attempt at implementing heap as trees.
; Maybe this is silly; just use AVL trees instead.

; 11 Aug 06
(gldefun heap-heapify (i)
  (let (l r largest tmp)
    (l = (left i))
    (r = (right i))
    (if (valid l)
	(if (sortp l i)
	    (largest = l)
	    (largest = i)))
    (if (valid r)
	(if (sortp r largest)
	    (largest = r)))
    (if (largest != i)
	(progn (tmp = (contents (^. i)))
	       ((contents (^. i)) = (contents (^. largest)))
	       ((contents (^. largest)) = tmp)
	       (heapify largest))) ))

; 12 Feb 07
; maintain the heap property: parent is >= its children
; cf. Cormen, Leiserson & Rivest  7.2
(gldefun heap-tree-heapify ((self heap-tree))
  (let (l r largest)
    (l = (left self))
    (r = (right self))
    (if (and (okay l) (sortp self l))
        (largest = l)
        (largest = self))
    (if (and (okay r) (sortp self r))
        (largest = r))
    (if (largest != self)
        (progn (exchange self largest)
               (heapify largest) ) ) ))

; 13 Feb 07
; exchange two nodes in a heap-tree by swapping pointers
(gldefun heap-tree-exchange ((self heap-tree) (other heap-tree))
   ; ...
)

; 13 Feb 07
; (not used) exchange two nodes in a heap-tree by swapping pointers
; advantage: moving pointers is constant work, may be less than
; moving a large contents.
; problem: possibly losing the pointer to the top of the tree.
(gldefun heap-tree-exchange-pointers ((self heap-tree) (other heap-tree))
  (let (parentself parentother tmp)
    (parentself = (parent self))
    (parentother = (parent other))
    (if (not (null parentself))
        (if (eq (left parentself) self)
            ((left parentself) = other)
            (if (eq (right parentself) self)
                ((right parentself) = other))))
    (if (not (null parentother))
        (if (eq (left parentother) other)
            ((left parentother) = self)
            (if (eq (right parentother) other)
                ((right parentother) = self))))
    ((parent self) = parentother)
    ((parent other) = parentself)
    (tmp = (left self))
    ((left self) = (left other))
    ((left other) = tmp)
    (tmp = (right self))
    ((right self) = (right other))
    ((right other) = tmp) ))

; 13 Feb 07
; Insert a new node in the next available slot at bottom of a heap-tree
; Assumes heap is non-empty.
(gldefun heap-tree-insert-bottom ((heap heap-tree) (new heap-tree))
  (let (leftdepth)
    (if (null (left heap))
        ((left heap) = new)
      (if (null (right heap))
          ((right heap) = new)
        (progn
          (leftdepth = (left-depth heap))
          (if (= leftdepth (right-depth heap))
              (insert-bottom (left heap) new)
              (if (= (1- leftdepth) (right-depth (left heap)))
                  (insert-bottom (right heap) new)
                  (insert-bottom (left heap) new) ) ) ) ) ) ))

; 13 Feb 07
(gldefun heap-tree-left-depth ((heap heap-tree))
  (if (null heap)
      0
      (1+ (left-depth (left heap)))))

; 13 Feb 07
(gldefun heap-tree-right-depth ((heap heap-tree))
  (if (null heap)
      0
      (1+ (right-depth (right heap)))))

; 12 Feb 07
(gldefun heap-tree-insert-key ((self heap-tree) (key (typeof (contents (^. self)))))
  (let (newtr)
    (newtr = (new (^. self)))
    ((sort-value (^. newtr)) = key)
    ; ...
  ))


(gldefun ww ((tr lisp-tree-pointer))
  (for x in tr do (print (if x is terminal
			     x
			     (contents (^. x))))))
(setq mytr '(= y (+ (* m x) b)))

(gldefun wv ((tr lisp-tree-pointer)) (size tr))

(gldefun wvt ((tr lisp-tree-pointer)) (terminals-size tr))

(gldefun wu ((tr elf-tree-pointer)) (for x in tr do (print (name x))))

(gldefun wub ((tr elf-tree-pointer)) (for x in tr average (age x)))

(gldefun wt ((tr elf-tree-pointer)) (size tr))

(setq mytrb '((a . b) . ((c . d) . (e . f))))
(gldefun wvc ((tr lisp-cons-tree-pointer)) (size tr))
; (wvc mytrb) ; = 11
(gldefun wvct ((tr lisp-cons-tree-pointer)) (terminals-size tr))
; (wvct mytrb) ; = 6

(gldefun ws ((tr lisp-tree-pointer) (s atom)) (member tr s))

(gldefun wsc ((tr lisp-cons-tree-pointer) (s atom)) (member tr s))

(gldefun wr ((tr lisp-tree-pointer)) (copy tr))

(gldefun wrb ((tr lisp-cons-tree-pointer) (trb lisp-tree-pointer))
  (copy-to tr trb))

(glispobjects

(intbintr (list (l (^ intbintr)) (n integer) (r (^ intbintr))))
(intavltr (list (l (^ intavltr)) (n integer) (bal integer) (r (^ intavltr))))
(symentb (list (name string) (type symbol)))
(symbintr (list (less (^ symbintr)) (is symentb) (more (^ symbintr))))
(symavltr (list (bal integer)
		(less (^ symavltr))
		(is   symentb)
		(more (^ symavltr))))
 ) ; glispobjects

; (viewas 'binary-tree 'intbintr)
(gldefun pa ((tr intbintr) (i integer)) (member (binary-tree tr) i))
(setq mybintr '( ((nil 1 nil) 2 (nil 3 nil)) 4 ((nil 5 nil) 6 (nil 7 nil)) ))
; (pa mybintr 2)
(gldefun pf ((tr intbintr)) (for x in (binary-tree tr) (print (n x))))
; (pf mybintr)
; (viewas 'avl-tree 'intavltr)
(gldefun pdb ((n integer))
  (let ((tr intavltr))
    (dotimes (i n) (tr = (insert-key (avl-tree tr) (random (* 2 n)))) )
    tr))
; (setq myinttr (pdb 50))   ; make example tree for testing

; (viewas 'binary-tree 'symbintr)
(gldefun pb ((tr symbintr) (str string)) (member (binary-tree tr) str))
(setq mystr
      '( ((nil ("ape" real) nil) ("bat" integer) (nil ("bee" string) nil))
	 ("cat" real)
	 ((nil ("dog" real) nil) ("elf" integer) (nil ("fox" string) nil)) ) )
; (pb mystr "elf")
(gldefun pg ((tr symbintr)) (for x in (binary-tree tr) (print (name (is x)))))
; (pg mystr)

; (viewas 'avl-tree 'symavltr)
(gldefun pc ((tr symavltr) (str string)) (member (avl-tree tr) str))
(gldefun pd ((tr symavltr) (str string)) (insert-key (avl-tree tr) str))
; (setq myavltr (pe))  ; Make an avl tree for testing
(gldefun pe ()
  (let ((avltr symavltr))
    (for word in
	 '(now is the time for all good men to come to the aid of their party)
	 do (avltr = (insert-key (avl-tree avltr) (symbol-name word))) ) ))
; (ph myavltr)   ; print the words of the test tree in order
(gldefun ph ((tr symavltr)) (for x in (avl-tree tr) (print (name (is x)))))
; (setq myavlll (pj myavltr))   ; destructively flatten the tree to linked list
(gldefun pj ((tr symavltr)) (flatten (avl-tree tr)))
; (pk myavlll)   ; print the linked list in order
(gldefun pk ((l symavltr)) (for x in (linked-list l) (print (name (is x)))))
