; geometry.lsp                  Gordon S. Novak Jr.           ; 20 Jan 11

; Copyright (c) 2011 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Geometric definitions

; 06 Jun 95; 02 Jan 97; 15 Jan 99; 07 Jan 04; 03 Mar 05; 19 Jul 06; 21 Jul 06
; 04 Aug 06; 15 Sep 06; 18 Sep 06; 19 Sep 06; 20 Sep 06; 22 Sep 06; 26 Sep 06
; 29 Sep 06; 07 Feb 07; 26 Feb 07; 27 Feb 07; 23 Mar 07; 20 Apr 07; 06 Dec 07
; 02 Jan 08; 14 Jan 08; 18 Jan 08; 23 Jan 08; 24 Jan 08; 05 Feb 08; 14 Feb 08
; 18 Feb 08; 20 Feb 08; 28 Feb 08; 29 Feb 08; 03 Oct 08; 22 Dec 08; 24 Dec 08
; 26 Dec 08; 09 Jan 09; 12 Jan 09; 13 Jan 09; 15 Jan 09; 31 Mar 09; 20 Apr 09
; 01 Sep 09; 30 Dec 10; 10 Jan 11; 12 Jan 11

(glispobjects

;    a*x + b*y + c = 0, b = 1 or (b = 0 and a = 1)
;    y = (- a) * x + (- c)   when b = 1

(line (crecord line (aa real) (b real) (c real))    ; cf. Skiena & Revilla
 prop ((slope         ((if (b != 0) (- (/ aa b)) most-positive-long-float)))
       (intercept     ((if (b != 0) (- (/ c b)))))
       (angle         ((atan (- aa) b))) )
 adj  ((vertical      ((b == 0)))
       (horizontal    ((aa == 0))) )
 msg  ((parallel      line-parallel open t)
       (==            line-equal open t)
       (equal         line-equal open t)
       (intersect     line-intersect     specialize t)
       (closest-point line-closest-point specialize t)
       (distance-to   line-distance-to   specialize t)
       (y             line-y open t)
       (x             line-x open t)
       (point-along   line-point-along specialize t)
       (tangent-circle line-tangent-circle specialize t)
       (tangent-circleb line-tangent-circleb specialize t)
 ) )

; 18 Sep 06; 26 Sep 06; 18 Feb 08
; line given point and angle
(linepta (crecord linepta (pt cvector) (ang real))
  prop ((aa        ((if (vertical self)
                        1
                        (- (tan (ang self))))))
        (b         ((if (vertical self) 0 1)))
        (c         ((if (vertical self)
                        (- (x pt))
                        (- (* (tan (ang self)) (x pt)) (y pt))))) )
 adj  ((vertical      ((< (abs (cos (ang self))) 1.0e-6)))
       (horizontal    ((< (abs (ang self)) 1.0e-6))) )
 msg  ((closest-point linepta-closest-point specialize t)
       (distance-to   linepta-distance-to specialize t) )
       ; (glambda (self pt) (distance (closest-point self pt) pt)) ) )
 supers (line))

; ??? shouldn't his be merged with line-segment ?
(points-as-line (list (pt1 nvector) (pt2 nvector))
  prop ((dx        ((- (x pt2) (x pt1))))
        (dy        ((- (y pt2) (y pt1))))
        (slope     ((/ (dy self) (dx self))))
        (line      ((if (vertical self)
                        (virtual line aa = 1 b = 0 c = (- (x pt1)))
                        (virtual line aa = (- (slope self)) b = 0
                                 c = (- (* slope (x pt1)) (y pt1))))))
        (aa        ((if (vertical self) 1 (- (slope self)))))
        (b         ((if (vertical self) 0 1)))
        (c         ((if (vertical self)
                        (- (x pt1))
                        (- (* slope (x pt1)) (y pt1))))) )
  adj  ((vertical  ((= dx 0))))
  msg  ((materialize ((a line aa = (aa self) b = (b self) c = (c self)))))
  supers (line))

(vector3d (list (x number) (y number) (z number))
  prop ((magnitude    ((sqrt (+ (expt x 2) (expt y 2) (expt z 2)))))
        (unitvector (glambda (self)
                      (let ((m (magnitude self)))
                        (a (typeof self) with x = (/ x m)
                                              y = (/ y m)
                                              z = (/ z m))))
                    result (typeof self))
        )
msg     ((+           vector3dplus         open t argtypes (vector3d))
         (-           vector3ddiff         open t argtypes (vector3d))
         (distance    vector3d-distance    open t argtypes (vector3d))
         )
)

; 3D direction using azimuth, elevation
; azimuth is degrees clockwise from north (north = 0, east = 90)
; elevation is degrees above horizon (x-y plane)
(azel (list (azimuth   (units real degrees))
            (elevation (units real degrees)) )
  prop ((unitvector   (glambda (self)      ; cf Wolfram Mathworld
                        (let ( (theta (- '(q 90 degree) (azimuth self)))
                               (phi   (- '(q 90 degree) (elevation self))))
                          (a vector3d with x = (* (cos theta) (sin phi))
                                           y = (* (sin theta) (sin phi))
                                           z = (cos phi)) ) )
                      result vector3d)
        )
  msg  ((angleto       (glambda (self other)
                        (* 2 (asin (* 0.5
                                      (magnitude (- (unitvector self)
                                                    (unitvector other)))))))
                      result (units real radians))
        )
)

; (radius real) isn't right, should indicate length units
(circle (crecord circle (center nvector)
              (radius real))
     ; props commented out because equations do it now.    12 Jan 11
     ; combination of props and equations caused a loop.
     ;    (diameter      (radius * 2))
     ;    (circumference (pi * diameter))
     ;    (area          (pi * radius ^ 2))
prop    ( (displayprops  ('(diameter circumference area))) )
msg     ((contains           circle-contains open t)
         (intersect-line-pt circle-intersect-line specialize t
                                                argtypes (line))
         (intersects?     circle-intersects-line? specialize t
                                                  argtypes (line))
         (intersects-line? circle-intersects-line? specialize t
                                                  argtypes (line))
         (intersect       (glambda (self ln)
                            (virtual circle-intersect-line c = self
                                     ln = ln)) argtypes (line))
         (intersect-line  (glambda (self ln)
                            (virtual circle-intersect-line c = self
                                     ln = ln)) argtypes (line))
         (intersect-line-ptb circle-intersect-lineb specialize t)
; need to fix intersection with circle to be like int with line above
         (intersects?     circle-intersects-circle? specialize t
                                                    argtypes (circle))
         (intersects-circle? circle-intersects-circle? specialize t
                                                    argtypes (circle))
         (intersect       circle-intersect-circle specialize t
                                                  argtypes (circle))
         (intersect-circle circle-intersect-circle specialize t
                   argtypes (circle &optional boolean))
         (intersect-chord circle-intersect-chord specialize t)
         (chord-point     circle-chord-point specialize t)
         (tangent-point   circle-tangent-point argtypes (vector real)
                          specialize t)
         (tangent-pointa  (glambda (self v) (tangent-point self v 1.0)))
         (tangent-pointb  (glambda (self v) (tangent-point self v -1.0)))
         (angle-to-point  (glambda (self v) (angle (- v (center self)))) )
         (angle-between-points circle-angle-between-points specialize t)
         (arc-between-points circle-arc-between-points specialize t)
 ) )

(circlersq (list (center vector)
                 (rsq real))
 prop  ((radius ((sqrt rsq))))
 supers (circle))

; ???   O'Rourke p. 201
(circle3 (list (pta vector) (ptb vector) (ptc vector))
 prop  ((d    (2 * (a1 * c0 + b1 * a0 - b1 * c0 - a1 * b0 - c1 * a0 + c1 * b0)))
        (p0   ((b1 * (expt a0 2) - c1 * (expt a0 2) - (expt b1 2) * a1
                 + (expt c1 2) * a1 + (expt b0 2) * c1 + (expt a1 2) * b1
                 + (expt c0 2) * a1 - (expt c1 2) * b1 - (expt c0 2) * b1
                 - (expt b0 2) * a1 + (expt b1 2) * c1 - (expt a1 2) * c1)
                 / d))
        (p1   (((expt a0 2) * c0 + (expt a1 2) * c0 + (expt b0 2) * a0
                - (expt b0 2) * c0  + (expt b1 2) * a0 - (expt b1 2) * c0
                - (expt a0 2) * b0 - (expt a1 2) * b0 - (expt c0 2) * a0
                + (expt c0 2) * b0 - (expt c1 2) * a0 + (expt c1 2) * b0)
                 / d))
        (a0   ((x pta)))
        (a1   ((y pta)))
        (b0   ((x ptb)))
        (b1   ((y ptb)))
        (c0   ((x ptc)))
        (c1   ((y ptc)))
        (rsq  ((expt (a0 - p0) 2) + (expt (a1 - p1) 2)))
        (center ((virtual vector with x = p0 y = p1))) )
 supers (circlersq))

; 23 Mar 07
(circle3b (list (pta vector) (ptb vector) (ptc vector))
  prop ((lab ((virtual line-segment with p1 = (pta self) p2 = (ptb self))))
        (lbc ((virtual line-segment with p1 = (ptb self) p2 = (ptc self))))
        (center ((intersect (perpendicular-bisector lab)
                            (perpendicular-bisector lbc))))
        (radius ((magnitude (- (pta self) center)))))
  supers (circle))

(sphere anything
 ; prop   ((volume ((4.0 / 3.0) * pi * radius ^ 3))
 ;         (area   (4.0 * pi * radius ^ 2)))
 supers (circle))


; 14 Sep 06; 19 Sep 06; 13 Jan 09; 15 Jan 09
(triangle (list (p1 vector) (p2 vector) (p3 vector))
  prop ((line1 ((virtual line-segment with p1 = (p2 self) p2 = (p3 self))))
        (line1b ((virtual line-segment with p1 = (p3 self) p2 = (p2 self))))
        (line2 ((virtual line-segment with p1 = (p3 self) p2 = (p1 self))))
        (line2b ((virtual line-segment with p1 = (p1 self) p2 = (p3 self))))
        (line3 ((virtual line-segment with p1 = (p1 self) p2 = (p2 self))))
        (line3b ((virtual line-segment with p1 = (p2 self) p2 = (p1 self))))
; signed-area will be positive if points are in counter-clockwise order
        (signed-area  ((/ (- (* (- (x p2) (x p1))
                                (- (y p3) (y p1)))
                             (* (- (y p2) (y p1))
                                (- (x p3) (x p1))))
                          2)))
    ;    (signed-area  ((+ (area-under (line1b self))
    ;                      (+ (area-under (line2b self))
    ;                         (area-under (line3b self))))))
        (area   ((abs (signed-area self))))
        (perimeter ((+ (length (line1 self))
                       (+ (length (line2 self)) (length (line3 self))))))
        (angle1 ((abs (- (angle (line3 self)) (angle (line2b self))))))
        (angle2 ((abs (- (angle (line3b self)) (angle (line1 self))))))
        (angle3 ((abs (- (angle (line1b self)) (angle (line2 self))))))
        (circumcenter ((intersect (perpendicular-bisector line1)
                                     (perpendicular-bisector line2))))
        (circumradius ((distance (p1 self) (circumcenter self))))
        (circumcircle ((virtual circle radius = (circumradius self)
                                       center = (circumcenter self))))
        (bisector1    ((line-from-point-angle p1
                         (/ (+ (angle (line2b self)) (angle (line3 self))) 2))))
        (bisector2    ((line-from-point-angle p2
                         (/ (+ (angle (line1 self)) (angle (line3b self))) 2))))
        (bisector3    ((line-from-point-angle p3
                         (/ (+ (angle (line2 self)) (angle (line1b self))) 2))))
        (incenter     ((intersect (bisector1 self) (bisector2 self))))
        (inradius     ((distance (line1 self) (incenter self))))
        (incircle     ((virtual circle radius = (inradius self)
                                       center = (incenter self))))
        )
   adj (  ; (ccw          ((> (signed-area self) 0.0)))
        (ccw          ((leftof (virtual line-segment p1 (p1 self)
                                    p2 (p2 self)) (p3 self))))
                      ; true if p1, p2, p3 are in counterclockwise order
        )
 )
; things to add:
; inscribed circle, angles, center point
; make triangle from points, point and line segment, 3 lines
; use a view of another rep. as line segment, make a triangle

(triangle3lns (list (l1 line) (l2 line) (l3 line))
  prop ((p1 ((intersect l2 l3)))
        (p2 ((intersect l1 l3)))
        (p3 ((intersect l1 l2))))
  supers (triangle))

(equilateral-triangle (side number)
  prop ((height    ((* side (/ (sqrt 3) 2))))
        (area      ((* 1/2 (* (side self) (height self)))))))

; angle where p2 is the center point
(angle (list (p1 vector) (p2 vector) (p3 vector))
  prop ((angle     angle-from-points)
        (absangle  ((abs (angle self)))) ) )

(hexagon (list (side number))
  prop  ((circumradius   (side))
         (circumdiameter ((* circumradius 2)))
         (inradius       ((* side (* 1/2 (sqrt 3)))))
         (indiameter     ((* inradius 2)))
         (sagitta        ((- circumradius inradius)))
         (area           ((* (sqrt 27/4) (expt side 2))))
         (circumference  ((* side 6)))) )

(square (list (side number))
  prop  ((area          (side ^ 2))) )

(quadratic (list (aa number) (b number) (c number))
  prop  ((rad   ((sqrt (- (expt b 2) (* 4 aa c)))))
         (root  ((/ (+ (- b) rad) (* 2 aa))))
         (rootb ((/ (- (- b) rad) (* 2 aa))))))

(cylinder anything
  prop  ((volume   ((* (area (cross-section self)) (height self))))) )

(circular-cylinder (list (r real) (height real))
  prop  ((cross-section ((virtual circle with radius = r))) )
  supers (cylinder) )

(shape-with-hole anything
  prop  ((area    ((- (area (shape self)) (area (hole self))))) ) )

(square-nut (list (width real) (d real) (thickness real))
  prop  ((cross-section  ((virtual shape-with-hole with
                            shape = (virtual square with side = width)
                            hole  = (virtual dcircle with diameter = d))))
         (height         (thickness)) )
  supers (cylinder) )

(hex-nut (list (width real) (d real) (thickness real))
  prop  ((cross-section  ((virtual shape-with-hole with
                            shape = (virtual hexagon with
                                             side = (/ width (sqrt 3)))
                            hole  = (virtual dcircle with diameter = d))))
         (height         (thickness)) )
  supers (cylinder) )

(foo-nut (list (width real) (d real) (thickness real))
  prop  ((cross-section  ((virtual shape-with-hole with
                            shape = (virtual square with side = width)
                            hole  = (virtual shape-with-hole with
                                      shape = (virtual dcircle
                                                  with diameter = d)
                                      hole  = (virtual square
                                             with side = d / (sqrt 2))))))
         (height         (thickness)) )
  supers (cylinder) )

(sierpinski-triangle (list (side real) (n integer))
  prop  ((area       ((area (shape self))))
         (shape      ((virtual shape-with-hole with
                               shape = (virtual equilateral-triangle
                                            with side = side)
                               hole  = (if (> n 0)
                                           (virtual sierpinski-triangle
                                               side = (/ side 2)
                                               n    = (1- n))
                                           (virtual equilateral-triangle
                                                    with side = 0)))))))

; (gldefun t104 ((s real) (th real))
;  (volume (virtual sierpinski-nut with side = s  thickness = th  n = 3)))
(sierpinski-nut (list (side real) (n integer) (thickness real))
  prop  ((cross-section  ((virtual sierpinski-triangle with
                                  side = side   n = n)))
         (height         (thickness)) )
  supers (cylinder) )

; 22 Dec 08
(circle-intersect-line (list (c circle) (ln line))
  prop ((d     ((distance-to ln (center c))))
        (chord-half ((sqrt (- (expt (radius c) 2) (expt (d self) 2))))
                    result (typeof (radius c)) )
        (chord-length ((* chord-half 2)) result (typeof (radius c)))
        (angle-half ((acos (/ (d self) (radius c)))))
        (angle      ((* angle-half 2)))
        (arc-short  ((* (radius c) (* 2 angle-half)))
                    result (typeof (radius c)) )
        (arc-long   ((* (radius c) (- (* 2 pi) (* 2 angle-half))))
                    result (typeof (radius c)) )
        (p1         (glambda (self)
                      (intersect-line-pt (c self) (ln self))))
        (p2         (glambda (self)
                      (intersect-line-ptb (c self) (ln self))))
        (chord      ((virtual line-segment with p1 = p1 p2 = p2))) ) )

) ; glispobjects

(setf (get 'circle 'ap-omit)    '(contains))      ; should fix this! *******
(setf (get 'circle 'ap-include) '(radius))

(setf (get 'sphere 'ap-omit)    '())
(setf (get 'sphere 'ap-include) '(radius))

(setf (glpropmenu 'circle) 'circleprops)

; 06 Jun 95
; Test whether a circle contains a point
(gldefun circle-contains ((c circle) (pt vector))
  (< (distance (center c) pt) (radius c)))

; 19 Jul 06
(gldefun line-parallel ((self line) (other line))  (result boolean)
  (and (<= (abs (- (aa self) (aa other))) long-float-epsilon)
       (<= (abs (- (b self)  (b other)))  long-float-epsilon)))

(gldefun line-equal ((self line) (other line))  (result boolean)
  (and (parallel self other)
       (<= (abs (- (c self) (c other))) long-float-epsilon) ) )

; 19 Jul 06; 18 Feb 08
(gldefun line-intersect ((self line) (other line) (omit (pt rvector)))
  (result (typeof pt))
  (let (xval)
    (if (and (not (parallel self other))
             (not (equal self other)))
        (progn
          (xval = (/ (- (* (b other) (c self)) (* (b self) (c other)))
                     (- (* (aa other) (b self)) (* (aa self) (b other))) ))
          (a (typeof pt)
             x xval
             y (if (> (abs (b self)) long-float-epsilon)
                   (- (/ (+ (* (aa self) xval) (c self)) (b self)))
                   (- (/ (+ (* (aa other) xval) (c other)) (b other)))))) ) ))

; 19 Jul 06; 02 Jan 08; 18 Feb 08
; point on a line closest to a given point
(gldefun line-closest-point ((self line) (pt vector))
  (result (typeof pt))
  (if (vertical self)
      (a (typeof pt) x (- (c self)) y (y pt))
      (if (horizontal self)
          (a (typeof pt) x (x pt) y (- (c self)))
          (intersect self (line-slope pt (/ 1.0 (aa self)))) ) ) )

; 19 Feb 08
(gldefun linepta-closest-point ((self linepta) (ptb cvector))
  (result (typeof ptb))
  (let (m xx yy)
    (if (vertical self)
        (a (typeof ptb) x = (x (pt self))  y = (y ptb))
        (if (horizontal self)
            (a (typeof ptb) x = (x ptb)  y = (y (pt self)))
            (progn (m = (tan (ang self)))
                   (xx = (/ (+ (- (* m (x (pt self)))
                                  (y (pt self)))
                               (+ (y ptb) (/ (x ptb) m)))
                            (+ m (/ 1 m))))
                   (yy = (+ (y (pt self)) (* m (- xx (x (pt self))))))
                   (a (typeof ptb) x = xx y = yy) ) ) ) ))

; 19 Feb 08
(gldefun linepta-distance-to ((self linepta) (ptb cvector))
  (result real)
  (let (m xx yy)
    (if (vertical self)
        (abs (- (x (pt self)) (x ptb)))
        (if (horizontal self)
            (abs (- (y (pt self)) (y ptb)))
            (progn (m = (tan (ang self)))
                   (xx = (/ (+ (- (* m (x (pt self)))
                                  (y (pt self)))
                               (+ (y ptb) (/ (x ptb) m)))
                            (+ m (/ 1 m))))
                   (yy = (+ (y (pt self)) (* m (- xx (x (pt self))))))
                   (xydistance xx yy (x ptb) (y ptb)) ) ) ) ))

; 19 Feb 08
(gldefun xydistance ((x1 real) (y1 real) (x2 real) (y2 real))
  (let (dx dy)
    (dx = (- x2 x1))
    (dy = (- y2 y1))
    (sqrt (+ (* dx dx) (* dy dy))) ))    

; 21 Jul 06
(gldefun line-distance-to ((self line) (pt vector)) (result real)
  (distance (closest-point self pt) pt))

; 23 Jan 08
; Create circle with center at a point, tangent to a line
(gldefun line-tangent-circle ((self line) (pt vector)) (result circle)
  (a circle with center = pt  radius = (distance-to self pt)))

; 24 Jan 08
(gldefun line-tangent-circleb ((self line) (pt vector) (omit (c circle)))
  (result (typeof c))
  (a (typeof c) with center = pt  radius = (distance-to self pt)))

; 21 Jul 06
; find the y value on a line for a given x
(gldefun line-y ((self line) (x real)) (result real)
  (if ((b line) == 0) 0 (- (/ (+ (* (aa self) x) (c self)) (b self)))))

; 21 Jul 06
; find the x value on a line for a given y
(gldefun line-x ((self line) (y real)) (result real)
  (if ((aa line) == 0)
      0
    (- (/ (+ (* (b self) y) (c self)) (aa self)))))

; 21 Jul 06; 26 Dec 08
; make a line given a point and angle
(gldefun line-from-point-angle ((pt vector) (angle (units real radians)))
         (result line)
  (let (n)
    (n = (truncate angle (* 2 pi)))
    (if (n != 0) (angle = (- angle (* (* 2 pi) n))))
    (if (< (abs (- (abs angle) (/ pi 2))) long-float-epsilon)
        (a line aa = 1 b = 0 c = (- (x pt)))
        (progn (tanangle = (tan angle))
               (a line aa = (- tanangle) b = 1
                       c = (- (* tanangle (x pt)) (y pt))) ) ) ))

; 21 Jul 06
; find the point a distance d in direction of a line from given point
(gldefun line-point-along ((self line) (pt vector) (d real)) (result vector)
  (dist-angle pt d (angle self)) )

; 21 Jul 06
; find line from given point tangent to a circle
(gldefun circle-tangent-point ((self circle) (pt vector) (sign real))
  (result vector)
  (let (d ang)
    (d = (distance pt (center self)))
    (if (> d (radius self))
        (progn (ang = (angle-to pt (center self)))
               (dist-angle pt (sqrt (- (expt d 2) (expt (radius self) 2)))
                           (+ ang (* sign (asin (/ (radius self) d))))))) ))

;    a*x + b*y + c = 0, b = 1 or b = 0 and a = 1
;    y = (- a) * x + (- c)   when b = 1
; 04 Aug 06
(gldefun line-from-points ((pt1 vector) (pt2 vector))
  (let (dx dy slope)
    (dx = (x pt2) - (x pt1))
    (dy = (y pt2) - (y pt1))
    (if (not (= dx 0))
        (progn (slope = (/ dy dx))
               (a line aa = (- slope)   b = 1
                       c = (- (* slope (x pt1)) (y pt1))))
        (a line aa = 1  b = 0  c = (- (x pt1))) ) ))

; 12 Jan 09
; Angle in radians of an angle whose center is pt2: value 0 .. pi
(gldefun angle-from-points ((pt1 vector) (pt2 vector) (pt3 vector))
  (let (ang)
    (ang = (abs (- (atan (- (y pt1) (y pt2)) (- (x pt1) (x pt2)))
                   (atan (- (y pt3) (y pt2)) (- (x pt3) (x pt2))) ) ) )
    (if (> ang pi)
        (- (* 2.0 pi) ang)
        ang) ))

; 19 Sep 06
(gldefun triangle-from-lines ((l1 line) (l2 line) (l3 line))
  (a triangle p1 = (intersect l2 l3) p2 = (intersect l1 l3)
              p3 = (intersect l1 l2)))

; 19 Sep 06; 12 Jan 09
(gldefun triangle-from-lengths ((la number) (lb number) (lc number))
  (result triangle)
  (let (xp yp tmp)
    (if (< la lb)
        (progn (tmp = la) (la = lb) (lb = tmp)))
    (if (< la lc)
        (progn (tmp = la) (la = lc) (lc = tmp)))
    (if (< lb lc)
        (progn (tmp = lb) (lb = lc) (lc = tmp)))
    (if (<= la (+ lb lc))
        (progn
          (xp = (/ (+ (expt la 2) (- (expt lb 2) (expt lc 2))) (* 2 la)))
          (yp = (sqrt (- (expt lb 2) (expt xp 2))))
          (a triangle p1 = (a rvector x = 0 y = 0)
                      p2 = (a rvector x = la y = 0)
                      p3 = (a rvector x = xp y = yp)) )) ))

; 07 Feb 07
(gldefun circle-intersect-line ((c circle) (l line))
  (let (pt d)
    (pt = (closest-point l (center c)))
    (d = (distance pt (center c)))
    (if (d <= (radius c))
        (point-along l pt (sqrt (- (expt (radius c) 2) (expt d 2))))) ))

(gldefun circle-intersect-lineb ((c circle) (l line))
  (let (pt d)
    (pt = (closest-point l (center c)))
    (d = (distance pt (center c)))
    (if (d <= (radius c))
        (point-along l pt (- (sqrt (- (expt (radius c) 2) (expt d 2)))))) ))

(gldefun circle-intersects-line? ((c circle) (l line))
  (<= (distance (closest-point l (center c)) (center c)) (radius c)) )

; 26 Feb 07
; test whether two circles intersect
(gldefun circle-intersects-circle? ((c circle) (d circle))
  (let (dst)
    (dst = (length (virtual line-segment p1 = (center c) p2 = (center d))))
    (and (<= dst (+ (radius c) (radius d)))
         (>= (+ dst (radius c)) (radius d))
         (>= (+ dst (radius d)) (radius c))) ))


; 27 Feb 07; 31 Aug 09
; compute the center of the chord connecting circle intersection points
; cf. http://mathworld.wolfram.com/Circle-CircleIntersection.html
(gldefun circle-intersect-chord ((c circle) (d circle))
  (let (dist)
    (dist = (length (virtual line-segment p1 = (center c) p2 = (center d))))
    (if (zerop dist)
        (center c)
    (xdist = 
    ((center c) + (- (center d) (center c))
                  * ( ( (dist ^ 2 - (radius d) ^ 2 + (radius c) ^ 2)
                        / (* 2 dist) )
                      / dist)) ) ) ))

; 27 Feb 07; 20 Apr 09; 31 Aug 09
; compute the chord point from circle intersection point
; set other = T to get the other point
(gldefun circle-chord-point ((c circle) (d circle) (pt nvector)
                             &optional other)
  (let (dx dy lsq l dist odist)
    (dx = ( (x pt) - (x (center c)) ) )
    (dy = ( (y pt) - (y (center c)) ) )
    (lsq = ( dx ^ 2 + dy ^ 2))
    (l = (sqrt lsq))
    (dist = (sqrt ( (radius c) ^ 2 - lsq)))    ; 1/2 chord length
    (odist = (if other (- dist) dist))
    (if (zerop lsq)
        (if (equal (center c) (center d))
            (a nvector x = (x pt) y = (+ (y pt) odist))
            (circle-chord-point d c pt other))
        (a nvector x = (- (x pt) (* (/ dy l) odist))
                   y = (+ (y pt) (* (/ dx l) odist))) ) ))

; 27 Feb 07; 31 Aug 09
; compute the center of the chord connecting circle intersection points
(gldefun circle-intersect-circle ((c circle) (d circle) &optional other)
  (result vector)
  (chord-point c d (intersect-chord c d) other))


; 09 Jan 09
; smaller angle between two points from center of a circle: value 0 .. pi
(gldefun circle-angle-between-points ((c circle) (p1 nvector) (p2 nvector))
  (angle-from-points p1 (center c) p2))

; 09 Jan 09
; smaller arc distance on circle from arc defined by two points vs. center
(gldefun circle-arc-between-points ((c circle) (p1 nvector) (p2 nvector))
  (* (radius c) (angle-between-points c p1 p2)) )


; 27 Feb 07
; test circle intersection
; do (load "/u/novak/X/dwtest.lsp") and (wtesta) to define window myw.
; (tci myw 150 150 30 170 160 20)
(gldefun tci ((w window) x1 y1 r1 x2 y2 r2)
  (let (c d p1 p2)
    (c = (a circle center = (a vector x x1 y y1) radius r1))
    (d = (a circle center = (a vector x x2 y y2) radius r2))
    (p1 = (intersect-circle c d))
    (p2 = (intersect-circle c d t))
    (clear w)
    (set-color w '(0 0 0))
    (draw-circle w (center c) (radius c))
    (draw-circle w (center d) (radius d))
    (draw-dot-xy w (x p1) (y p1))
    (set-color w '(65000 0 0))
    (draw-dot-xy w (x p2) (y p2))
    (force-output w) ))

; 20 Apr 09
; do (ldwt) and (wtesta) first
(gldefun test-circle-intersect-circle ()
  (let (x y)
    (dotimes (i 24)
      (x = (* 80 (cos (* i (/ pi 12)))))
      (y = (* 80 (sin (* i (/ pi 12)))))
      (tci myw 150 150 100 (+ x 150) (+ y 150) 30)
      (sleep 1) ) ))

; 19 Nov 07
; smallest circle passing through 2 points
(gldefun circle-thru-two-points ((p1 vector) (p2 vector)) (result circle)
  (a circle with center = (midpoint p1 p2)
                 radius = (/ (distance p1 p2) 2)) )

; 19 Nov 07
; circle passing through 3 points
(gldefun circle-thru-points ((p1 vector) (p2 vector) (p3 vector))
  (result circle)
  (let (ctr)
    (ctr = (circumcenter (virtual triangle with p1 = p1  p2 = p2  p3 = p3)))
    (a circle with center = ctr
                   radius = (distance p1 ctr)) ))

; 30 Dec 10
(gldefun vector3dplus ((v1 vector3d) (v2 vector3d))
  (a (typeof v1) with x = (x v1) + (x v2)
                      y = (y v1) + (y v2)
                      z = (z v1) + (z v2)  ))

(gldefun vector3ddiff ((v1 vector3d) (v2 vector3d))
  (a (typeof v1) with x = (x v1) - (x v2)
                      y = (y v1) - (y v2)
                      z = (z v1) - (z v2) ))

(gldefun vector3d-distance ((u vector3d) (v vector3d))
  (let ((dx ((x u) - (x v))) (dy ((y u) - (y v))) (dz ((z u) - (z v))) )
    (sqrt dx * dx + dy * dy + dz * dz)))
