; gltest.lsp             Gordon S. Novak Jr.             ; 10 Oct 11

; Copyright (c) 2011 Gordon S. Novak Jr. and The University of Texas at Austin
; All rights reserved.

; Common Lisp version of GLISP test file

; 12 Oct 92; 08 Jun 94; 23 Dec 94; 05 Jan 95; 08 May 95; 26 May 95; 06 Jun 95
; 03 Oct 95; 25 Mar 97; 28 Dec 98; 28 Feb 02; 09 Aug 02; 07 Jan 04; 30 Sep 05
; 23 Jul 07

;   Note: much of the data in this file is pure fiction, especially the 
;         personnel data. 

; (load "/glisp/vector.lsp") and (load "/glisp/geometry.lsp") are needed.

(glispobjects

(employee (listobject
                (name       string)
		(date-hired shdate)
		(salary     real)
		(jobtitle   symbol)
		(trainee    boolean))
prop    ((seniority ((year (currentdate)) - (year date-hired)))
	 (monthly-salary (salary * 174))
	 (shortvalue (name))
	 (displayprops (t)))
adj     ((high-paid (monthly-salary > 2000)))
isa     ((trainee (trainee))
	 (greenhorn (trainee and seniority < 2)))
msg     ((youre-fired (salary = 0))))

(date (list (year  integer)
            (month integer)
	    (day   integer))
  msg   ((< date-< open t)))

(shdate (list (month     integer)
	      (day       integer)
	      (shortyear integer))
prop    ((monthname ((nth (1- month)
			  '(january february march april may
				    june july august september
				    october november december)))
		    result symbol)
	 (year ((if (> shortyear 20) (+ shortyear 1900) (+ shortyear 2000))))
	 (shortvalue ((concatenate 'string (pname monthname)
			       " "  (stringify day) ", " (stringify year))))
	 (prettyform (shortvalue)))
 supers (date))

(company (atomobject (president employee)
		     (employees (listof employee)))
prop    ((electricians ((those employees with jobtitle == 'electrician)))))


(project (atomobject (title         string)
		     (abbreviation  symbol)
		     (administrator person)
		     (contracts     (listof contract))
		     (executives    (listof person)))
prop    ((shortvalue (abbreviation))
	 (displayprops (t))
	 (budget total-budget)))


(contract (atomobject (title   string)
		      (leader  person)
		      (sponsor agency)
		      (budget  budget))
prop    ((shortvalue (title))))


(agency (atomobject (name         string)
		    (abbreviation symbol)
		    (address      address)
		    (phone        phone-number))
prop    ((shortvalue (abbreviation))))


(person (atomobject (name         string)
		    (initials     symbol)
		    (title        symbol)
		    (project      project)
		    (salary       real)
		    (ssno         integer)
		    (birthdate    shdate)
		    (phone        phone-number)
		    (office       campus-address)
		    (home-address address)
		    (home-phone   phone-number)
		    (friend       (^ person))
		    (picture      picture))
prop    ((shortvalue (initials))
	 (contracts ((those contracts of project with leader == self)))
	 (age ((year (currentdate)) - (year birthdate)))
	 (monthly-salary (salary / 12))
	 (displayprops (t)))
adj     ((faculty (title <= '(prof assoc-prof asst-prof))))
views   ((pasv vector (x (age)) (y (salary / 1000.0)))) )


(budget (listobject (labor    real)
	            (computer real))
prop    ((overhead (labor * .59))
	 (total (labor + overhead + computer))
	 (shortvalue (total))
	 (displayprops (t))))


(address (listobject (street string)
	             (city   string)
		     (state  symbol)
		     (zip    integer))
prop    ((shortvalue ((concatenate 'string city ", " (pname state))))))


(phone-number (listobject (area   integer)
		          (number integer))
prop    ((number-split ((multiple-value-list (truncate number 10000)))
	       result (list (prefix integer) (rest integer)))
         (prefix ((prefix number-split)))
	 (rest   ((rest number-split)))
         (shortvalue ((format nil "(~3D) ~3D-~4,'0D"
			      area (prefix number-split)
			      (rest number-split)))) )
adj     ((local (area == 512))))


(picture anything
msg     ((edit       paintw)
	 (gevdisplay picture-gevdisplay)))


(campus-address (list (building building)
		      (room     atom))
prop    ((shortvalue ((format nil "~A ~A" (pname (abbreviation building))
				          room)))))


(building (atomobject (abbreviation symbol)
		      (name         string)
		      (number       integer))
prop    ((shortvalue (name))))

)

; 14 Jul 94
(gldefun date-< ((x shdate) (y shdate))
  (or (< (year x) (year y))
      (and (= (year x) (year y))
	   (or (< (month x) (month y))
	       (and (= (month x) (month y))
		    (< (day x) (day y)))))))


; Program to give raises to the electricians. 
(gldefun give-raise ((c company))
  (for x in electricians when x is not a trainee do
     (salary _+ (if (seniority > 1) 2.5 1.5))
     (format t "~A ~A ~A~%" (name x)
	                    (prettyform date-hired)
			    monthly-salary)))


(gldefun currentdate ()
  (result shdate)
  (let ((timelist (multiple-value-list (get-decoded-time))))
    (a shdate with shortyear = (sixth timelist) - 1900
                 month = (fifth timelist) day = (fourth timelist)) ))


; 22-OCT-82; 28 Mar 91
(gldefun total-budget ((p project))
  (let ((sum 0.0))
    (for c in (contracts p) do (sum _+ (total (budget c))))
    sum))

; 23 Dec 94
; (total-budget-b hpp)
(gldefun total-budget-b ((p project)) (for c in (contracts p) sum (total budget)))

; GSN  7-AUG-83 15:58 ; 30 Sep 05
; Some test data for the above functions. 
(gldefun init-company ()
  (setq company1
	(a company with president =
	 (an employee with name = "OSCAR THE GROUCH"
	     salary = 88.0 jobtitle = 'president
	     date-hired =
	     (a shdate with month = 3 day = 15 shortyear = 7))
	 employees =
	 (list (an employee with name = "COOKIE MONSTER" salary = 12.5 
		   jobtitle = 'electrician date-hired =
		   (a shdate with month = 4 day = 1 shortyear = 65))
	       (an employee with name = "BETTY LOU" salary = 9.0
		   jobtitle = 'electrician date-hired =
		   (a shdate with month = 8 day = 03 shortyear = 80))
	       (an employee with name = "GROVER" salary = 3.0 jobtitle = 
		   'electrician trainee = t date-hired =
		   (a shdate with month = 10 day = 14 shortyear = 78))))))


; GSN  9-FEB-83 11:24 
; Initialize data structures for GEV demo. 
(gldefun gevdemo-init ()
(prog nil (hpp = (a project with title = "Heuristic Programming Project" 
		    abbreviation = 'hpp))
      (mjh = (a building with abbreviation = 'mjh
		name = "Margaret Jacks Hall" number = 460))
      (aro = (an agency with name = 
		  "US Army Research Office"
		  abbreviation = 'aro
		  address =
		  (an address with street = "P.O. Box 12211" city = 
		      "Research Triangle Park"
		      state = 'nc
		      zip = 27709)
		  phone = (a phone-number with area = 919
			     number = 5490641)))
      (darpa = (an agency with name = 
		  "Defense Advanced Research Projects Agency"
		  abbreviation = 'darpa
		  address =
		  (an address with street = "1400 Wilson Blvd." city = 
		      "Arlington"
		      state = 'va
		      zip = 22209)
		  phone = (a phone-number with area = 202
			     number = 6944349)))
      (nsf = (an agency with name = "National Science Foundation"
		 abbreviation = 'nsf
		 address =
		 (an address with street = "1800 G STREET N.W." city = 
		     "Washington"
		     state = 'dc
		     zip = 20550)
		 phone = (a phone-number with area = 202
			    number = 6327346)))
      (nih = (an agency with name = "National Institutes of Health" 
		 abbreviation = 'nih
		 address =
		 (an address with street = "9000 Rockville Pike" city = 
		     "Bethesda"
		     state = 'md
		     zip = 20001)
		 phone = (a phone-number with area = 301
			    number = 4964000)))
      (gsn =
	   (a person with name = "Gordon S. Novak Jr." initials =
	      'gsn
	      title = 'visitor
	      project = hpp salary = 30000.0 ssno = 454123456 birthdate =
	      (a shdate with day = 1 month = 4 shortyear = 48)
	      phone = (a phone-number with area = 415 number = 4974532)
	      office = (a campus-address with building = mjh room = 244)
	      home-phone = (a phone-number with area = 415
			      number = 4935807)
	      home-address =
	      (an address with street = "3857 Ross Road"
		  city = "Palo Alto" 
		  state = 'ca
		  zip = 94303)))
      (jca =
	   (a person with name = "Jon C. Admin" initials = 'jca
	      title = 'administrator
	      project = hpp salary = 30000.0 ssno = 452123456 birthdate =
	      (a shdate with day = 2 month = 1 shortyear = 47)
	      phone = (a phone-number with area = 415 number = 4972780)
	      home-phone = (a phone-number with area = 415
			      number = 4324321)
	      office = (a campus-address with building = mjh room = 236)
	      home-address = (an address)))
      (mxe =
	   (a person with name = "Max Enchilada" initials =
	      'mxe
	      title = 'prof
	      project = hpp salary = 99999.0 ssno = 123123456 birthdate =
	      (a shdate with day = 4 month = 1 shortyear = 42)
	      phone = (a phone-number with area = 415 number = 4971234)
	      office = (a campus-address with building = mjh room = 226)
	      home-phone = (a phone-number with area = 415
			      number = 4931234)
	      home-address =
	      (an address with street = " " city = "Stanford" state =
		  'ca
		  zip = 94305)))
      (erw =
	   (a person with name = "Ernie R. Whiz" initials =
	      'erw
	      title = 'asst-prof
	      project = hpp salary = 31234.0 ssno = 177123456 birthdate =
	      (a shdate with day = 2 month = 1 shortyear = 50)
	      phone = (a phone-number with area = 415 number = 4970324)
	      office = (a campus-address with building = mjh room = 234)
	      home-phone = (a phone-number with area = 415
			      number = 4324321)
	      home-address = (an address)))
      (j5 =
	  (a contract with title = "Advanced A.I. Architectures"
	     leader = mxe 
	     sponsor = darpa budget =
	     (a budget with labor = 50000.0 computer = 10000.0)))
      (ia =
	  (a contract with title = "Intelligent Agents"
	     leader = erw sponsor = darpa
	     budget = (a budget with labor = 70000.0
             computer = 50000.0)))
      (dart =
	    (a contract with title = "Diagnosis and Repair Techniques"
	       leader = erw sponsor = darpa budget =
	       (a budget with labor = 100000.0 computer = 150000.0)))
      (glisp =
	     (a contract with title = "GLISP" leader = gsn sponsor = aro 
		budget = (a budget with labor = 50000.0
			    computer = 20000.0)))
      (cm =
	  (a person with name = "Cookie Monster" initials = 'cm
	     title = 'monster
	     project = hpp salary = 1.0 ssno = 123456789 birthdate =
	     (a shdate with month = 4 day = 1 shortyear = 65)
	     phone = (a phone-number with area = 415 number = 4971234)
	     office = (a campus-address with building = mjh room = 252)
	     home-phone = (a phone-number with area = 415
			     number = 4561234)
	     home-address =
	     (an address with street = "123 Sesame Street"
		 city = "Palo Alto" 
		 state = 'ca
		 zip = 94303)))
      (carbm =
	     (a contract with title = 
		"Carbohydrate Metabolism in Atypical Hominids"
		leader = cm sponsor = nih budget =
		(a budget with labor = 1.39 computer = 5.0)))
      ((administrator hpp) = jca)
      ((contracts hpp) = (list j5 ia dart glisp carbm))
      ((executives hpp) = (list mxe erw gsn jca))
      (c = (a circle with center = (a vector with x = 1 y = 1)
	      radius = 5.0))))


(glispobjects

(consv (cons (x integer) (y integer))
  supers  (vector))

(fvector (cons (y string)
	       (x boolean))
supers  (vector)
doc     (* "A FVECTOR is a very different kind of VECTOR: it has a" 
	   "different storage structure and different element types."
	   "However, it can still inherit some vector properties," 
	   "e.g., addition."))

(vofv (list (x vector)
	    (y vector))
supers  (vector)
doc     ("A VOFV is a vector of vectors, that is, a vector whose" 
	   "components are VECTORs"))


(radians real
prop    ((degrees (self * (180.0 / 3.141593))
		  result degrees)
	 (displayprops (t))))


(degrees real
prop    ((radians (self * (3.141593 / 180.0))
		  result radians)
	 (displayprops (t))))

)


; Now we define some test functions that use the above definitions. 
; First there are some simple functions which test vector operations. 

(gldefun tvplus ((u vector) (v vector)) u + v)

(gldefun tfvplus ((u fvector) (v fvector)) u + v)

(gldefun tvofvplus ((u vofv) (v vofv)) u + v)

(gldefun tvmove ((u vector) (v vector)) u _+ v)

(gldefun tvtimesv ((u vector) (v vector)) (dotproduct u v))   ; 03 Oct 95

(gldefun tvtimesn ((u vector) (v number)) u * v)


; Next we define some graphics objects that use VECTOR operations in 
;   their definitions. 

(glispobjects

(graphicsobject (list (shape atom)
		      (start vector)
		      (size vector))
prop    ((left   ((x start)))
	 (bottom ((y start)))
	 (right  (left + width))
	 (top    (bottom + height))
	 (width  ((x size)))
	 (height ((y size)))
	 (center (start + size / 2))
	 (area   (width * height)))
msg     ((draw   ((funcall (get shape 'drawfn) self 'paint)))
	 (erase  ((funcall (get shape 'drawfn) self 'erase)))
	 (move   graphicsobjectmove open t)))


(movinggraphicsobject (list (transparent graphicsobject)
			    (velocity vector))
msg     ((accelerate mgo-accelerate open t)
	 (step ((glsend self move velocity)))))

)


; 11-JAN-82
(gldefun graphicsobjectmove ((self graphicsobject) (delta vector))
  (glsend self erase)
  (start _+ delta)
  (glsend self draw))


(gldefun mgo-accelerate ((self movinggraphicsobject) (acceleration vector))
  velocity _+ acceleration)


; This test function creates a MovingGraphicsObject and then moves it 
;   across the screen by sending it MOVE messages. Everything in this 
;   example is compiled open; the STEP message involves a great deal 
;   of message inheritance. 
(gldefun mgo-test nil
  (let (mgo n)
    (mgo = (a movinggraphicsobject with shape = 'rectangle
		size = (a vector with x = 4 y = 3)
		velocity = (a vector with x = 3 y = 4)))
      (n = 0)
      (while ((n _+ 1) < 100) (step mgo))
      (print (start mgo)) ))


; This function tests the properties of a graphicsobject. 
(gldefun testfn2 ((g graphicsobject))
(list shape start size left bottom right top width height center area))


; 11-JAN-82
(gldefun drawrect ((g graphicsobject) (dspop atom))
  (prog (oldds)
    (oldds = (currentdisplaystream dsps))
    (dspoperation dspop)
    (moveto left bottom)
    (drawto left top)
    (drawto right top)
    (drawto right bottom)
    (drawto left bottom)
    (currentdisplaystream oldds)))


(glispobjects

(lisptree (cons (car lisptree)
		(cdr lisptree))
prop    ((leftson ((if (atom self) nil car)))
	 (rightson ((if (atom self) nil cdr))))
adj     ((empty (~self)))
doc     ("The LispTree and PreorderSearchRecord objects illustrate" 
	   "how generators can be written. In defining a LispTree,"
	   "which can actually be of multiple types (atom or dotted pair)," 
	   "we define it as the more complex dotted-pair type and"
	   "take care of the simpler case in the PROPerty definitions."))


(preordersearchrecord (cons (node lisptree)
			    (previousnodes (listof lisptree)))
msg     ((next ((let (tmp)
		  (if (tmp = (leftson node))
		      (progn (if (rightson node)
			         (previousnodes +_ node))
			     (node = tmp))
		      (progn (tmp -_ previousnodes)
			     (node = (rightson tmp)))) )) ))
doc     (* "PreorderSearchRecord is defined to be a generator. Its" 
	   "data structure holds the current node and a stack of"
	   "previous nodes, and its NEXT message is defined as code to" 
	   "step through the preorder search."))
)

; (printleaves '(((a (b (c d (e)) f) g) h) i))
; PRINTLEAVES prints the leaves of the tree, using a 
; PreorderSearchRecord as the generator for searching the tree. 
(gldefun printleaves ((lt lisptree))
  (let (psr)
      (psr = (a preordersearchrecord with node = lt))
      (while node (if (atom node) (progn (prin1 node) (terpri)))
	          (glsend psr next))))


#| The definition for circle is now in the file geometry.lsp

(circle (listobject (center vector)
	            (radius real))
prop    (      ;   (pi (3.1415926535897931))
	 (diameter (radius * 2))
	 (circumference (pi * diameter))
	 (area (pi * radius ^ 2))
	 (squareside ((sqrt area)))
	 (displayprops ('(diameter circumference area))))
msg     ((grow (area _+ 100))
	 (shrink (area = area / 2))
	 (standard (area = 100.0)))
adj     ((big (area > 100))
	 (small (area < 80))))
|#

(glispobjects

(dcircle (listobject (center   vector)
		     (diameter real))
prop    ((radius (diameter / 2)))
supers  (circle)
doc     (* "A DCIRCLE is implemented differently from a circle. The" 
	   "data structure is different, and DIAMETER is stored"
	   "instead of RADIUS. By defining RADIUS as a PROPerty, all" 
	   "of the CIRCLE properties defined in terms of radius can be"
	   "inherited."))

)


; Make some CIRCLE objets for testing 
; Since DCIRCLE is an Object type, it can be used with interpreted 
;   messages, e.g., (glsend dc area) to get the area property, 
;   (glsend dc standard) to set the area to the standard value, 
;   (glsend dc diameter) to get the stored diameter value. 
(gldefun init-circles nil
  (setq mycircle (a circle))
  (setq dc (a dcircle with diameter = 10.0)))


(gldefun growcircle ((c circle))
  ((area c) _+ 100)
  c)


; A simple version of SQRT 
(gldefun sqrtb ((x number))
  (let ((s x))
    (if (x >= 0)
	(progn
	  (while ((abs s * s - x) > 1.0E-6) do (s = (s + x / s) * .5))
	  s)) ))


; Function SQUASH illustrates elimination of compile-time constants. 
;   Of course, nobody would write such a function directly. However, 
;   such forms can arise when inherited properties are compiled. 
;   Conditional compilation occurs automatically when appropriate 
;   variables are defined to the GLISP compiler as compile-time 
;   constants because the post-optimization phase of the compiler 
;   makes the unwanted code disappear. 
(gldefun squash nil
  (if (> 1 3)
      `amaazing
      (if (sqrt 7.2) < 2
	  'incredible
	  (if 2 + 2 == 4 'okay 'jeez))))


; The following object definitions describe a student records 
;   database. 

(glispobjects

(student (atomobject (name   string)
		     (sex    symbol)
		     (major  symbol)
		     (grades (listof integer)))
prop    ((average student-average)
	 (grade-average student-grade-average)
	 (shortvalue (name))
	 (displayprops (t)))
adj     ((male (sex == 'male))
	 (female (sex == 'female))
	 (winning (average >= 95))
	 (losing (average < 60)))
isa     ((winner (self is winning))))


(student-group (listof student)
prop    ((n-students length)
	 (average student-group-average)
	 (shortvalue ((for x in self collect (shortvalue x))))))


(class (atomobject (department symbol)
		   (number     integer)
		   (instructor string)
		   (students   student-group))
prop    ((n-students ((n-students students)))
	 (men ((those students who are male)))
	 (women ((those students who are female)))
	 (winners ((those students who are winning)))
	 (losers ((those students who are losing)))
	 (class-average ((average students)))))
) ; glispobjects


(gldefun student-average ((s student))
  (result real)
  (let ((sum 0.0) (n 0))
    (for g in grades do (n _+ 1) (sum _+ g))
    (sum / n) ))


(gldefun student-grade-average ((s student))
  (result symbol)
  (let ((av (average s)))
    (if av >= 90.0
	'a
        (if av >= 80.0
	    'b
	    (if av >= 70.0
		'c
		(if av >= 60.0 'd 'f) )))))


(gldefun student-group-average ((sg student-group))
  (let ((sum 0.0))
    (for s in sg do (sum _+ (average s)))
    (sum / (n-students sg)) ))

(gldefun student-group-average-b ((sg student-group))
  (for s in sg average (average s)))

; (test1 class1)
; Print name and grade average for each student 
(gldefun test1 ((c class))
  (for s in (students c) (format t "~A ~A~%" (name s) (grade-average s))))

; Print name and average of the winners in the class 
(gldefun test2 ((c class))
  (for s in (winners c) (format t "~A ~A~%" (name s) (average s))))

; The average of all the male students' grades 
(gldefun test3 ((c class)) (average (men c)) )

; The name and average of the winning women 
(gldefun test4 ((c class))
  (for s in (women c) when s is winning
             (format t "~A ~A~%" (name s) (average s))))


; Another version of the above function. The * operator in this case 
;   denotes the intersection of the sets of women and winners. The 
;   GLISP compiler optimizes the code so that these intermediate sets 
;   are not actually constructed. 
(gldefun test4b ((c class))
  (for s in ((women c) * (winners c))
       (format t "~A ~A~%" (name s) (average s)) ) )


; Make a list of the easy professors. 
(gldefun easy-profs ((classes  (listof class)))
  (for c in classes when (class-average c) > 90.0 collect instructor))


; Another version of EASY-PROFS. 
(gldefun easy-profs-b ((classes  (listof class)))
  (for c in classes when ((class-average c) > 90.0) collect (instructor c)))


; Some test data for testing the above functions. 
(gldefun init-class nil
(setq class1
      (a class with instructor = "A. PROF" department = 'cs number = 102 
	 students =
	 (list (a student with name = "JOHN DOE" sex = 'male
		  major = 'cs  grades = ' (99 98 97 93))
	       (a student with name = "FRED FAILURE" sex = 'male
		  major = 'cs  grades = ' (52 54 43 27))
	       (a student with name = "MARY STAR" sex = 'female
		  major = 'cs  grades = ' (100 100 99 98))
	       (a student with name = "DORIS DUMMY" sex = 'female
		  major = 'cs  grades = ' (73 52 46 28))
	       (a student with name = "JANE AVERAGE" sex = 'female
		  major = 'cs  grades = ' (75 82 87 78))
	       (a student with name = "LOIS LANE" sex = 'female
		  major = 'cs  grades = ' (98 95 97 96))))))


; The following object definitions illustrate inheritance of 
;   properties from multiple parent classes. The three bottom classes 
;   Planet, Brick, and Bowling-Ball all inherit the same definition of 
;   the property Density, although they are represented in very 
;   different ways. 


(glispobjects

(physical-object anything
prop    ((density (mass / volume))))

(ordinary-object anything
prop    ((mass (weight / 9.88)))
supers  (physical-object))

; (sphere anything
;   prop    ((volume ((4.0 / 3.0) * 3.141593 * radius ^ 3))))

(parallelepiped anything
prop    ((volume (length * width * height))))

(planet (listobject (mass   real)
		    (radius real))
supers  (physical-object sphere))

(brick (object (length real)
	       (width  real)
	       (height real)
	       (weight real))
supers  (ordinary-object parallelepiped))

(bowling-ball (atomobject (type   symbol)
			  (weight real))
prop    ((radius ((if type == 'adult .1 .07))))
supers  (ordinary-object sphere))

)

; Three test functions to demonstrate inheritance of the Density property.

; (dplanet earth)
(gldefun dplanet ((p planet)) density)

; (dbrick brick1)
(gldefun dbrick ((b brick)) density)

; (dbb bb1)
(gldefun dbb ((b bowling-ball)) density)


; Some objects to test the functions on. 
; Since the object types Planet, Brick, and Bowling-Ball are defined 
;   as Object types , messages can be sent to them directly from the 
;   keyboard for interactive examination of the objects. For example, 
;   the following messages could be used: (GLSEND EARTH DENSITY) 
;   (GLSEND BRICK1 WEIGHT\: 25.0) (GLSEND BRICK1 MASS\: 2.0) 
;   (GLSEND BB1 RADIUS) (GLSEND BB1 TYPE\: 'CHILD) 
(gldefun init-objects nil
 (setq earth  (a planet with mass = 5.98E24 radius = 6.37E6))
 (setq brick1 (a brick with weight = 20.0 width = .1
		 height = .05 length = .2))
 (setq bb1 (a bowling-ball with type = 'adult weight = 60.0)))

; Compile and execute functions to set up test data.
(init-company)
(init-circles)
(init-class)
(init-objects)
(gevdemo-init)
