; glutil.lsp               Utility Functions for GCL         ; 12 Jan 18

; Copyright (c) 2018 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; 10 Apr 97; 27 Feb 98; 05 Mar 02; 30 Sep 02; 28 Oct 04; 11 May 05; 07 Sep 06
; 05 Dec 06; 09 Jan 07; 18 Oct 07; 18 Aug 08; 29 Jan 09; 12 Aug 09

; Small utility functions for use with GLISP
(or (boundp '*proplist-hide*) (setq *proplist-hide* nil))
(or (boundp '*directory*) (setq *directory* "/u/novak/"))
(defvar *file-list* nil)
(defvar *gldirectory* (concatenate 'string *directory* "glisp/"))
(defvar *gloutdirectory* "")
(defvar *glxdirectory* (concatenate 'string *directory* "X/"))
(defvar *glbdirectory* *gldirectory*)

(defmacro fb  (l)     `(fboundp (unq ',l)))

(defun lgl () (lb glisp))

(defun lt  () (ld tmp))

(defun ldt () (loadr (file-name "genll.test" *gldirectory* "")))

(defun ldw () (ldws) (ldwb))

(defun ldwt () (load (file-name "X/dwtest" *directory* "")))

(defun fo () (window-force-output))

(defun compile-glisp () (ld compile-glisp))

(defmacro ld (&rest files) `(mapc #'ld-expr ',files))
(defun ld-expr (file)
  (loadr (file-name file *gldirectory* ".lsp")) file)

(defmacro ldx (&rest files) `(mapc #'ldx-expr ',files))
(defun ldx-expr (file)
  (loadr (file-name file *glxdirectory* ".lsp")) file)

; load a file from a specified directory
(defmacro ldd (dir &rest files) `(mapc #'(lambda (file) (ldd-expr ,dir file))
				       ',files))
(defun ldd-expr (dir file)
  (load (concatenate 'string dir (string-downcase (symbol-name file)) ".lsp")))

(defmacro lds (&rest files) `(mapc #'lds-expr ',files))
(defun lds-expr (file)
  (loadr (file-name file (concatenate 'string *gldirectory* "sun/" ".lsp")))
  file)

(defmacro ld? (&rest files) `(mapc #'ld?-expr ',files))
(defun ld?-expr (file)
  (load? (file-name file *gldirectory* ".lsp")) file)

(defmacro lp (&rest files) `(mapc #'lp-expr ',files))
(defun lp-expr (file)
  (loadr (file-name file (concatenate 'string *directory* "/physics/"))) file)

(defmacro lb (&rest files) `(mapc #'lb-expr ',files))
(defun lb-expr (file)
  (loadr (file-name file *glbdirectory* ".o")) file)

; Compile-file macro: (cf <file>)
; cf adds a .lsp suffix and puts the output in the *gloutdirectory* subdirectory
(defmacro cf (&rest files) `(mapc #'cf-expr ',files))
(defun cf-expr (file)
  (compile-file (concatenate 'string *gldirectory*
		     (string-downcase (symbol-name file)) ".lsp")
		:output-file
		(concatenate 'string *gldirectory* *gloutdirectory*
			     (string-downcase (symbol-name file)) ".o") ) )

(defun ut  ()         (untrace))

(defun tr (&rest l)   (eval (cons 'trace l)))

(defmacro str (symbol) `(car (glstr (unq ',symbol))))

(defmacro pl (symbol) `(symbol-plist (unq ',symbol)))

(defmacro ppl (symbol) `(pprint (spl (unq ',symbol))))

(defmacro res (symbol) `(glfnresulttype (unq ',symbol)))

(defmacro types (symbol) `(gltypesdefined (unq ',symbol)))

(defun unq (sym)
  (if (and (consp sym) (eq (car sym) 'quote))
      (cadr sym)
      (if (eq sym '*) (eval sym) sym)))

; "unquote" a function name
(defun unqf (sym)
  (let ((symb (unq sym)))
    (if (symbolp symb)
	(if (fboundp symb)
	    symb
	    (and (boundp symb)
		 (fboundp (eval symb))
		 (eval symb)))) ))

(defmacro ppd (fn) `(pprint (glgetd (unq ',fn))))

(defun clr (&optional w) (window-clear (or w myw)))

(defun spl (sym)
  (let ((pl (and (symbolp sym) (symbol-plist sym))))
    (if (and (boundp sym)
	     (or (null pl)
		 (and (consp pl)
		      (member (car pl) *proplist-hide*)
		      (null (cddr pl)))))
	(symbol-plist (eval sym))
        pl) ))

(defmacro glp (&rest fn)
  (if fn `(glprintcompiled (unqf ',(car fn)))
    `(glprintcompiled *gllastfncompiled*)))

(defmacro glc (fn) `(glcc (unqf ',fn)))

(defmacro glcp (fn) `(glcpfn (unqf ',fn)))

; print the result of glisp compilation.
(defun glprintcompiled (fn)
  (when fn
    (format t "result type: ~A" (glfnresulttype fn))
    (pprint (glcompiled fn))
    (terpri)
    fn))

(defun glps () (glp *glspecfncompiled*))

(defun glcps (fn)
  (let ((lastspec *glspecfncompiled*))
    (glcp fn)
    (unless (eq lastspec *glspecfncompiled*) (glps)) ))

; circular print
(defun cp (x)
   (let ((old-print-circle *print-circle*))
     (unwind-protect
       (progn (setq *print-circle* t)
	      (print x)
	      nil)
       (setq *print-circle* old-print-circle)) ))

(defmacro pp (symbol) `(pp-expr (unq ',symbol)))
(defun pp-expr (symbol)
  (let ((def (and (symbolp symbol)
		  (or (get symbol 'gloriginalexpr)
		      (and (fboundp symbol)
			   (not (special-form-p symbol))
			   (not (compiled-function-p (symbol-function symbol)))
			   (symbol-function symbol))
		      (and (boundp symbol)
			   (eval symbol))
		      (symbol-plist symbol)))))
    (if def (pprint def)) ))

(defmacro gls (symbol) `(pprint (get (unq ',symbol) 'glstructure)))

; Find x in y
(defun fnd (x y)
  (if (consp y)
      (if (eq (car y) x)
	  y
	  (or (fnd x (car y))
	      (fnd x (cdr y))))))

(defun fndgc (x fn) (fnd x (glcompiled fn)))

; Make x into a file name
(defun file-name (x &optional (prefix *gldirectory*) (suffix ".lsp"))
  (concatenate 'string prefix
	       (if (symbolp x)
		   (string-downcase (symbol-name x))
		   (if (stringp x) x (error "Bad file name ~A~%" fn)))
		 suffix))

; Load a file if it is not already loaded
(defun load? (filename)
  (unless (member filename *file-list* :test #'string=)
    (load filename)
    (push filename *file-list*)
    filename))

; Load a file if it is not already loaded
(defun loadr (filename)
  (load filename)
  (unless (member filename *file-list* :test #'string=)
          (push filename *file-list*)
	  filename))

; 09 Sep 92; 06 Nov 92; 23 Dec 96; 10 Apr 97
(defun get-time-string ()
  (let (second minute hour date month year day dst zone str)
    (multiple-value-setq (second minute hour date month year day dst zone)
			 (get-decoded-time))
; Hack for GCL, which does not seem to do daylight savings time
    (if (and (> month 3) (< month 11))
	(progn (setq dst t) (incf hour)))
    (setq str (format nil "~2D ~A ~4D ~2D:~2D:~2D ~3A"
		      date (nth (1- month) '("Jan" "Feb" "Mar" "Apr" "May"
					     "Jun" "Jul" "Aug" "Sep" "Oct"
					     "Nov" "Dec"))
		      year hour minute second
		      (if (and (>= zone 5) (<= zone 8))
			  (nth (- zone 5)
			       (if dst '("EDT" "CDT" "MDT" "PDT")
				       '("EST" "CST" "MST" "PST")))
			  "") ))
    (if (char= (char str 0) #\Space) (setf (char str 0) #\0))
    (if (char= (char str 12) #\Space) (setf (char str 12) #\0))
    (if (char= (char str 15) #\Space) (setf (char str 15) #\0))
    (if (char= (char str 18) #\Space) (setf (char str 18) #\0))
    str))

; 30 Sep 02
(defun get-time-list ()
  (let (second minute hour date month year day dst zone str)
    (multiple-value-setq (second minute hour date month year day dst zone)
			 (get-decoded-time))
    (list second minute hour date month year day dst zone str) ))

; 05 Mar 93
; Make a (glambda ...) code expression into a function
(defun glfn (code fnname) (eval (cons 'gldefun (cons fnname (cdr code)))))

(defun wrc () (dotimes (i 4) (window-reset-cursor)))

(defun cu (x y) (glconvertunit x y))

(defun gltypeof (obj)
  (or (and (consp obj)
           (eq (car obj) 'crecord)
           (consp (cdr obj))
           (gltypep (cadr obj))
           (cadr obj))
      (and (consp obj)
           (gltypep (car obj))
           (car obj))
      (and (symbolp obj)
           (gltypep (get obj 'class))
           (get obj 'class)) ) )
(setf (get 'gltypeof 'glfnresulttype) 'gltype)

; 29 Jan 09
; make a list of all glisp functions
(defun allglfns ()
  (let (fns)
    (dolist (file *file-list*)
      (setq fns (union (fnsinfile file) fns)) )
    fns))

(defmacro ty (fn) `(glfntypes (unq ',fn)))

(defun glfntypes (fn)
  (list fn (glarguments fn) '-> (glfnresulttype fn) (gltypesdefined fn)) )
