; iter.lsp             Gordon S. Novak Jr.             ; 22 Dec 10

; Copyright (c) 2010 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Experimental macro code for iterators

; 10 Nov 94; 05 Jan 95; 17 Mar 95; 29 Mar 96; 29 Oct 96; 02 Feb 99; 29 Dec 00
; 28 Feb 02; 29 Oct 02; 04 Dec 03; 09 Dec 03; 08 Jan 04; 19 Oct 04; 15 May 07
; 30 Dec 08; 19 Oct 09

(glispobjects
  (stats (listobject (n integer) (sum real) (sumsq real))
    prop ((mean ((float sum) / n))
	  (mean-square ((float sumsq) / n))
	  (variance ( mean-square - mean ^ 2))
	  (std-dev  ((sqrt variance)))
	  (displayprops (t))) )

  (filter anything
    prop   ((predicate (t))
	    (sequence  ('prog1)))
    msg    ((iterator  filter-iterator)))

  (chain-code anything
    prop   ((delta-view ('prog1))
	    (delta-op   ('+)))
    msg    ((next-state chain-code-next-state open t)
	    (iterator   enumeration-iterator)) )

 ) ; glispobjects

; 31 Aug 90
; (gldefmacro <macro-name> (<evaluated-args>) (<textual-args>) <code>)
(defmacro gldefmacro (&rest args) `(gldefmacro-expr ',args))
(defun gldefmacro-expr (l) (setf (glmacro (first l)) (rest l)) )


; 31 Aug 90; 02 Jun 93; 17 Mar 95; 29 Oct 96
; Make a unique name relative to current context.  Result is name or name-n.
(defun gliteruniquename (name context vars)
  (let ((nm name) (n 0))
    (while (or (glcodetype nm context)
	       (assoc nm *gltypesdefined*)
	       (member nm vars))
       (incf n)
       (setq nm (intern (concatenate 'string (symbol-name name)
				             (princ-to-string n)))) )
    nm ))

; 31 Aug 90; 09 Sep 90; 26 Feb 91; 02 Jun 93; 20 Aug 93; 18 Oct 93; 10 Nov 94
; 29 Oct 96; 29 Oct 02
; Macro-expand a macro definition into in-line code.
(defun glmacroexpand (macroname eval-args text-args)
  (let (macrodef subs newform tmp code ovar var localvars targ textvars)
    (or (setq macrodef (glmacro macroname))             ; macro definition
	(error "Undefined glisp macro ~A~%" macroname))
    (setq code (third macrodef)) 
; Make text-arg pairing for substitution
    (dolist (x (second macrodef))
      (setq targ (pop text-args))
      (push (cons x targ) subs)
      (if (symbolp targ) (push targ textvars)) )
 ; Make local-var pairs to insure uniqueness of local vars
    (setq localvars (append (first macrodef)
			    (if (member (first code) '(let prog))
				(second code))))         ; let / prog vars
    (dolist (x localvars)
      (setq ovar (if (consp x) (first x) x))
      (setq var (if (char= (lastchar ovar) #\:)
		    (intern (subseq (symbol-name ovar) 0
				    (1- (length (symbol-name ovar)))))
		    ovar))
      (unless (or (member var (second macrodef))
		  (member var *gltypenames*)
		  (member var '(typeof arrayof listof)))
	(setq tmp (gliteruniquename var *glcontext* textvars))
	(unless (eq tmp var)
	  (push (cons var tmp) subs)
	  (if (char= (lastchar ovar) #\:)
	      (push (cons ovar (intern (concatenate 'string tmp ":")))
		    subs)) ) ) )
    (setq newform (cons 'glambda (list (sublis subs (first macrodef))
				       (sublis subs code))))
    (glcompopen newform eval-args (mapcar #'cadr eval-args) nil	*glcontext*) ))


; 13 Mar 91; 19 Mar 91; 27 May 93; 18 Feb 97; 09 Dec 03
; Find the appropriate macro definition for a specified type.
(defun glfindmacro (type macrotype)
  (let (tmp)
    (and type
	 (if (and (consp type)
		  (setq tmp (assoc (first type)
				   '((listof linked-list) (arrayof array)))))
	     (glfindclmacro (second tmp) macrotype)
	     (if (symbolp type)
		 (or (glfindclmacro (glcluster type) macrotype)
		     (and (setq tmp (glstrprop type 'msg macrotype nil))
			  (glmacro (cadr tmp))
			  (cadr tmp))
		     (and (setq tmp (assoc macrotype (glmacrodefs type)))
			  (cadr tmp))
		     (some #'(lambda (x) (glfindclmacro (first x) macrotype))
			   (glviews type))
		     (and (setq tmp (glxtrtype (glgetstr type)))
			  (not (equal tmp type))
			  (glfindmacro tmp macrotype)) ) ) ) ) ))

; 19 Mar 91; 09 Dec 03
; Find a macro definition associated with a cluster.
(defun glfindclmacro (cluster macrotype)
  (let (tmp)
    (and cluster
	 (if (setq tmp (assoc macrotype (glmacrodefs cluster)))
	     (cadr tmp)
	     (or (some #'(lambda (x) (glfindclmacro x macrotype))
		       (glclustersupers cluster))
		 (some #'(lambda (x) (glfindclmacro x macrotype))
		       (glclusterviewsupers cluster)) ) ) ) ))


(setf (glmacrodefs 'linked-list) '((iterator linked-list-iterator)
				   (collect  linked-list-collector)
				   (append   linked-list-append)))

(setf (glmacrodefs 'lisp-linked-list)
      '((collect lisp-linked-list-collector) ))

(setf (glmacrodefs 'ext-linked-list)
      '((iterator external-linked-list-iterator)
	(collect external-linked-list-collector) ))

(setf (glmacrodefs 'array) '((iterator array-iterator)
			     (collect  array-collector)))

(setf (glmacrodefs 'string) '((iterator string-iterator)))

(setf (glmacrodefs 'integer) '((iterator integer-iterator)))

(setf (glmacrodefs 'enumeration) '((iterator enumeration-iterator)))

(setf (glmacrodefs 'generator) '((iterator generator-iterator)))

(setf (glmacrodefs 'filter) '((iterator filter-iterator)))

; 31 Aug 90; 19 Feb 91; 22 Feb 91; 20 May 93; 30 Dec 08
(gldefmacro linked-list-iterator (list) (item condition code stopcond)
  (let ( (ptr (a (typeof list))) item ptrnext)
    (ptr = list)
    (while (not (null ptr))
	   (ptrnext = (rest ptr))
	   (item = (implementation (^. ptr)))
	   (if condition (progn . code))
           (if stopcond (return))
	   (ptr = ptrnext) ) ))

; 30 Dec 91; 31 Dec 91; 28 May 93; 28 Feb 02; 08 Jan 04; 30 Dec 08
(gldefmacro circular-linked-list-iterator (list) (item condition code stopcond)
  (let (start ptr item done)
    (start = list)
    (ptr = start)
    (done = nil)
    (while (and (not (null ptr)) (not done))
	   (item = (implementation (^. ptr)))
	   (if condition (progn . code))
	   (ptr = (rest ptr))
           (if stopcond (return))
	   (if (ptr == start) (done = t)) ) ))

; 19 Sep 91; 20 May 93; 30 Dec 08
(gldefmacro external-linked-list-iterator (list) (item condition code stopcond)
  (let ( (ptr (a (typeof list))) item ptrnext)
    (ptr = list)
    (while (not (null ptr))
	   (ptrnext = (rest ptr))
	   (item = (contents (^. ptr)))
	   (if condition (progn . code))
           (if stopcond (return))
	   (ptr = ptrnext) ) ))

; 21 Sep 90; 26 Feb 91; 01 Oct 93; 02 Oct 93; 04 Oct 93; 12 Nov 93; 08 Jan 04
; 30 Dec 08
(gldefmacro array-iterator (arr) (item condition code stopcond)
  (let (item)
    (if (= (lower-bound arr) 0)
	(dotimes (indx (filled-size arr))
	  (item = (index arr indx))
	  (if condition (progn . code))
          (if stopcond (return)) )
        (progn (indx = (lower-bound arr))
	       (while (and (< indx (filled-size arr)) (not done))
		      (item = (index arr indx))
		      (if condition (progn . code))
                      (if stopcond (return))
		      (indx _+ 1)) ) )))

; 19 Oct 93; 19 Oct 04; 30 Dec 08
(gldefmacro string-iterator (str) (item condition code stopcond)
  (let (i item)
    (dotimes (i (string-length str))
      (item = (char str i))
      (if condition (progn . code))
      (if stopcond (return)) ) ))

(defun string-length (s) (length s))

; 18 Feb 97; 30 Dec 08
(gldefmacro integer-iterator (int) (item condition code stopcond)
  (dotimes (item int)
    (if condition (progn . code))
    (if stopcond (return)) ) )

; 02 Oct 91; 30 Dec 08
(gldefmacro double-iterator (seq) (item condition code stopcond)
  (let (item)
    (for subseq in (implementation seq) do
      (for item in subseq do
        (if condition (progn . code))
        (if stopcond (return)) )
      (if stopcond (return)) ) ))

; 19 Feb 91; 28 Feb 91; 02 Apr 91; 28 May 92; 08 Jan 04; 30 Dec 08
; This is an 'external-linked-list' collector.  The item that is presented
; is a 'contents' item; new records are created to hold each item.
(gldefmacro external-linked-list-collector (seq m) (item condition code stopcond)
  (let (val cpy (end (typeof m)) new)
    (cpy = nil)
    (for item in seq when condition until stopcond do
	 (val = (progn . code))
	 (new = (new (^. m)))
	 (set-contents new val)     ; needs a transfer method
	 (if cpy
	     ((link (^. end)) = new)
	     (cpy = new))
	 (end = new))
    ((link (^. end)) = (null-value m))
    cpy))

; 04 Apr 91; 11 Apr 91; 28 May 92; 08 Jan 04; 30 Dec 08
; Collect items as an internal linked list.  The items are records
; that are simply linked together as the type of seq.
(gldefmacro linked-list-collector (seq outseq) (item condition code stopcond)
  (let (val cpy (end (typeof outseq)))
    (cpy = nil)
    (if (null seq)
	seq
        (progn
	  (for item in seq when condition until stopcond do
	     (val = (progn . code))
	     (if cpy
		 ((link (^. end)) = val)
	         (cpy = val))
	     (end = val))
	  ((link (^. end)) = (null-value end))
	  cpy))))

; 11 Apr 91; 28 May 92; 08 Jan 04; 30 Dec 08
; Collect items as a Lisp linked list.
(gldefmacro lisp-linked-list-collector (seq m) (item condition code stopcond)
  (let (val cpy end)
    (cpy = nil)
    (if (null seq)
	nil
        (progn
	  (for item in seq when condition until stopcond do
	     (val = (list (progn . code)))
	     (if cpy
		 ((rest end) = val)
	         (cpy = val))
	     (end = val))
	  cpy))))


; 19 Mar 91; 28 May 92; 08 Jan 04; 30 Dec 08
; Collect an internal linked list, where the code generates a record
; containing the pointer.
(gldefmacro linked-list-append (seq m) (item condition code stopcond)
  (let (val cpy (end (typeof m)) (new (typeof m)))
    (cpy = nil)
    (for item in seq when condition until stopcond do
	 (new = (progn . code))
	 (if cpy
	     ((link (^. end)) = new)
	     (cpy = new))
	 (end = new))
    ((link (^. end)) = (null-value m))
    cpy))


; 13 Mar 91; 20 Aug 93; 30 Dec 08
; If type of m is just ARRAY it would be nice to make it (ARRAYOF (typeof code))
(gldefmacro array-collector (seq m) (item condition code stopcond)
  (let (val (newarr (typeof m)) (i 0))
    (newarr = (make-array (length seq)))
    (for item in seq when condition until stopcond do
	 (val = (progn . code))
         (set newarr i val)
	 (i _+ 1))
    newarr))

(setf (glmacrodef 'average) 'average-iterator)

; 26 Feb 91; 28 Feb 91; 21 Oct 93; 28 Feb 02; 08 Jan 04; 30 Dec 08
(gldefmacro average-iterator (seq) (item condition code stopcond)
  (let (sum val (n 0))
    (for item in seq when condition until stopcond do
      (val = (progn . code))
      (if (n == 0)
	  (sum = val)  ; must occur first so sum has type of val
          (sum = sum + val))
      (n _+ 1))
    (if (n > 0)
	(sum / n)
        (a (typeof val))) ))


(setf (glmacrodef 'rms) 'rms-iterator)

; 10 Apr 91; 30 Dec 08
(gldefmacro rms-iterator (seq) (item condition code stopcond)
  (let (avg val)
    (avg = (for item in seq when condition until stopcond average
		  (progn (val = (progn . code))
			 (float (val ^ 2)))))
    (sqrt avg) ))



(setf (glmacrodef 'stats) 'stats-iterator)

; 10 Apr 91; 24 Sep 96; 28 Feb 02; 08 Jan 04; 30 Dec 08; 19 Oct 09
(gldefmacro stats-iterator (seq) (item condition code stopcond)
  (let (sumvar sumsqvar val (nvar 0))
    (for item in seq when condition until stopcond do
      (val = (progn . code))
      (if (nvar == 0)
	  (progn (sumvar = val)  ; must occur first so sumvar has type of val
		 (sumsqvar = val * val))
          (progn (sumvar = (+ sumvar val))
		 (sumsqvar = (+ sumsqvar (* val val)))))
      (nvar _+ 1))
    (a stats with n = nvar sum = sumvar sumsq = sumsqvar) ))

; 24 May 94
(gldefun chain-code-next-state ((cc chain-code) state delta)
  (funcall (delta-op cc) state delta) )

(setf (glmacrodef 'sum) 'sum-iterator)

; 28 Feb 91; 19 May 93; 21 Oct 93; 08 Jan 04; 30 Dec 08; 22 Dec 10
(gldefmacro sum-iterator (seq) (item condition code stopcond)
  (let (sum)
    (sum = (zero (a (typeof (let ((item (first seq))) (progn . code))))))
    (for item in seq when condition until stopcond do
         (sum = sum + (progn . code)))
    sum ))

(setf (glmacrodef 'max) 'max-iterator)

; 30 Dec 91; 19 May 93; 19 Oct 93; 04 Dec 03; 08 Jan 04; 30 Dec 08
; Might be nice to return both the max value and the element with the max.
(gldefmacro max-iterator (seq) (item condition code stopcond)
  (let (val itm flg maxval)
    (flg = nil)
    (for item in seq when condition until stopcond do
      (cast maxval (typeof (progn . code)))
      (val = (progn . code))
      (if ~ flg or (val > maxval)
	  (progn (flg = t)
		 (itm = item)
		 (maxval = val)) ))
    (if flg maxval)  ))

(setf (glmacrodef 'min) 'min-iterator)

; 30 Dec 91; 19 May 93; 19 Oct 93; 07 Jun 94; 04 Dec 03; 08 Jan 04; 30 Dec 08
(gldefmacro min-iterator (seq) (item condition code stopcond)
  (let (val itm flg minval)
    (flg = nil)
    (for item in seq when condition until stopcond do
      (cast minval (typeof (progn . code)))
      (val = (progn . code))
      (if ~ flg or (val < minval)
	  (progn (flg = t)
		 (itm = item)
		 (minval = val)) ))
    (if flg minval)  ))

(setf (glmacrodef 'argmax) 'argmax-iterator)

; 04 Dec 03; 08 Jan 04; 30 Dec 08
; Might be nice to return both the max value and the element with the max.
(gldefmacro argmax-iterator (seq) (item condition code stopcond)
  (let (val itm flg maxval)
    (flg = nil)
    (for item in seq when condition until stopcond do
      (cast maxval (typeof (progn . code)))
      (val = (progn . code))
      (if ~ flg or (val > maxval)
	  (progn (flg = t)
		 (itm = item)
		 (maxval = val)) ))
    (if flg itm)  ))

(setf (glmacrodef 'argmin) 'argmin-iterator)

; 04 Dec 03; 08 Jan 04; 30 Dec 08
(gldefmacro argmin-iterator (seq) (item condition code stopcond)
  (let (val itm flg minval)
    (flg = nil)
    (for item in seq when condition until stopcond do
      (cast minval (typeof (progn . code)))
      (val = (progn . code))
      (if ~ flg or (val < minval)
	  (progn (flg = t)
		 (itm = item)
		 (minval = val)) ))
    (if flg itm)  ))

(setf (glmacrodef 'every) 'every-iterator)

; 15 May 07; 30 Dec 08
(gldefmacro every-iterator (seq) (item condition code stopcond)
  (let (sum start val)
    (start = nil)
    (for item in seq when condition until stopcond do
      (val = (progn . code))
      (if ~ start
	  (progn (sum = val)  ; must occur first so sum has type of val
		 (start = t))
          (sum = (and sum val))))
    (if start
	sum
        (a (typeof val))) ))

(setf (glmacrodef 'find) 'find-iterator)

; 02 Feb 99; 08 Jan 04; 30 Dec 08
; (for x in (linked-list l) when (eq (color x) 'blue) find (name x))
(gldefmacro find-iterator (seq) (item condition code stopcond)
  (let (itm)
    (for item in seq until stopcond
      (if condition
	  (progn (itm = (progn . code))
                 (return))))
    itm  ))

; 18 Oct 93; 19 Oct 93; 30 Dec 08
; Macro for enumerations, including functions of previous state.
; This version assumes that the initial state and (n - 1) deltas
; form a total of n items.
(gldefmacro enumeration-iterator (enum) (state condition code stopcond)
  (let (item state)
    (state = (initial-state enum))
    (if condition (progn . code))
    (for item in (sequence enum) until stopcond do
      (state = (next-state enum state (funcall (delta-view enum) item)))
      (if condition (progn . code)) ) ))

; 21 Oct 93; 30 Dec 08
; Macro for filters, i.e., subsequences of another sequence.
(gldefmacro filter-iterator (filter) (item condition code stopcond)
  (let (item)
    (for item in (sequence filter) until stopcond do
      (if (and (funcall (predicate filter) item)
	       condition)
	  (progn . code)) ) ))

; 02 Nov 93; 03 Nov 93; 31 Mar 96; 30 Dec 08
; Macro for generators, including functions of previous state.
; Note: state must be a "container" that can be modified by the generator.
(gldefmacro generator-iterator (gen) (item condition code stopcond)
  (let (item state)
    (state = (initial-state gen))
    (while (not (done state)) do
      (setq item (generate state))
      (if condition (progn . code))
      (if stopcond (return)) ) ))

; 18 Oct 93
; Copy a data structure.
; This is defined as shown here for Lisp, but should have a different
; definition for C: just move the fields, but don't allocate storage.
(defun copy (x) (copy-tree x))


; some test code:

(gldefun wa ((e elf-ll-pointer)) (for x in e do (print (age x))))
(gldefun wb ((b box)) (for x in (linked-list b) do (print (color x))))
; (glviewas 'linked-list 'bl)
(gldefun wc ((b bl)) (for x in (linked-list b) do (print (val x))))
(gldefun wd ((e elf)) (for x in (linked-list e) do (print (name x))))
(gldefun we ((cb cboard-llp)) (for x in cb do (print (piece x))))
(gldefun wf ((ar (arrayof person))) (for p in ar (print (age p))))
(gldefun wj ((l (listof integer))) (for z in l average z))
(gldefun wk ((c class)) (for s in (women c) average (first (grades s))))
(gldefun wl ((b box)) (for z in (linked-list b) average (wavelength (color z))))
(gldefun wx ((cb cboard-llp)) (for x in cb average (value x)))
(gldefun wy ((b box)) (for z in (linked-list b) sum (size z)))
(gldefun wz ((cb cboard-llp)) (for x in cb sum (pname (piece x))))
(gldefun zb ((b box) (l bl))
  (for x in (linked-list b) collect as (typeof (linked-list l))
     (a bl with val = (size x))))
; (zb mybox nil)  ; --> (((NIL . 3) . 2) . 1)

; 12 Mar 91; 13 Mar 91; 04 Apr 91
(gldefun rn  ((ar (arrayof integer))) (for x in ar collect x))
(gldefun rna ((ar (arrayof integer))) (for x in ar average x ^ 2))
(gldefun rnb ((ar (arrayof integer)))
  (for x in ar collect as (linked-list bl) x))
(gldefun rnc ((l (arrayof integer)))
  (for x in l when (x > 0) collect as bl-as-ll-pointer x))
(gldefun rnd ((l (listof integer))) (for x in l collect as (arrayof integer) x))
(gldefun rne ((ar array)) (for x in ar (print x)))
(gldefun rnf ((l box)) (for x in (linked-list l)
			   collect as (arrayof color) (color x)))
; 10 Apr 91; 11 Apr 91
(gldefun rng ((c class))  (for x in (women * winners) stats (midterm x)))
(gldefun rnh ((l (listof person)))
   (std-dev (for x in l when ((age x) > 30) stats (salary x))))
(gldefun rni ((l (arrayof integer)))
  (std-dev (for x in l when (x > 0) stats x)))
(gldefun rnj ((b box)) (for z in (linked-list b) stats (size z)))
(gldefun rnk ((cb cboard-llp)) (for x in cb stats (length (pname (piece x)))))
(gldefun rnl ((e elf)) (for x in (linked-list e) stats (age x)))
(gldefun rnm ((e elf)) (for x in (tree e) stats (age x)))
(gldefun va ((b box))
  (for x in (linked-list b) collect as (linked-list box)
       (a box with color = (color x) size = (size x) ^ 2)))
(gldefun vb ((b bl)) (length (ext-linked-list b)))
(gldefun vc ((l (arrayof integer)))
  (for x in l collect as (ext-linked-list bl) x^2))
