; lldisp.lsp                  Gordon S. Novak Jr.      ; 27 May 04

; Copyright (c) 2007 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Generic linked-list display functions

; (dispm-display-spec-edit '(box linked-list) mybox *myw* 20 50 280 70)
; (dispm-display-spec-edit '(bl linked-list) mybl *myw* 20 50 250 40)

; 29 Jul 94; 05 Jan 95; 03 Nov 95; 02 Jan 97; 28 Feb 02; 03 Mar 04

(defvar *default-editor-window* nil)

(glispobjects

(lldisplayspec (listobject (viewtype            gltype)
			   (arrow-length        integer)
			   (line-width          integer)
			   (options
			    (list
			     (box-style        symbol)
			     (draw-pointer-cell boolean)
			     (draw-arrow        boolean)
			     (draw-line         boolean) ))
			   (item-display        dispm-display-spec)
			   (subdisplay          dispm-display-spec))
 default ((box-style  'rectangle)
	  (draw-arrow t)
	  (draw-line  t)
	  (line-width 1))
 prop    ((arroww          ((if arrow-length and (arrow-length > 0)
			        arrow-length
			        (max (box-width / 2) 20))))
	  (deltax          (box-width + arroww))
	  (arrowup         ((if draw-pointer-cell
			        (box-height - 8 - line-width / 2)
			        (box-height / 2))))
	  (contents-size   ((size (display-method item-display))))
	  (box-width       ((x contents-size) + line-width * 2))
	  (box-height      ((y contents-size) + line-width * 2
			     + (if draw-pointer-cell
				   (line-width + 14)
				   0)))  )
 adj     ((cons-list       (viewtype == 'lisp-linked-list-pointer)) )
 msg     ((draw            ll-display-draw))    )

(arrdisplayspec (listobject (viewtype          gltype)
			    (line-width        integer)
			    (box-size          vector)
			   (options
			    (list
			     (draw-box         boolean)
			     (draw-index-x     boolean)
			     (draw-index-y     boolean)
			     (vertical         boolean) ))
			    (item-display      dispm-display-spec)
			    (subdisplay        dispm-display-spec))
 default ((draw-box   t)
	  (box-size nil)
	  (line-width 1))
 prop    ((deltax          ((x contents-size) + line-width))
	  (deltay          ((y contents-size) + line-width))
	  (contents-size   (box-size or (size (display-method item-display)))))
 msg     ((draw            arr-display-draw))    )

(dispm-arr-spec (list (offs     integer)      ; offset before items start
		      (itemsize integer)      ; size of one item
		      (gapsize  integer)      ; gap between items
		      (nitems   integer))     ; number of items
 prop   ((total-size  (itemsize + gapsize))) )

 ) ; glispobjects


; 01 Nov 91; 02 Dec 91
; Display/edit data
(gldefun gev-edit (data type)
  (geva nil data (glbasetype type)) )

; 11 Oct 91; 25 Oct 91; 28 Oct 91; 29 Oct 91; 06 Nov 91; 04 Dec 91; 05 Dec 91;
; 06 Dec 91; 10 Dec 91; 17 Dec 91; 19 Dec 91; 24 Dec 93; 26 Dec 93; 03 Mar 04
(gldefun ll-display-draw
  ((dspec lldisplayspec) (lst linked-list) (w window)
	     (offsetx integer) (offsety integer)
	     &optional (sizex integer) (sizey integer))
  (let (lng maxboxes ptr n lastx boxx boxw boxh lw lw2 lw2b arrowy
	    totalx (llvt (viewtype dspec)) enddots)
    (boxw = (box-width dspec))
    (boxh = (box-height dspec))
    (lw = (line-width dspec))
    (lw2 = lw / 2)
    (lw2b = lw - lw2)
    (lastx = offsetx + boxw / 2)
    (boxx = offsetx + (deltax dspec))
    (totalx = (min (or sizex 9999) ((width w) - offsetx) ) )
    (maxboxes = ((totalx - boxw) / (deltax dspec)))       ; if no ...
    (lng = (glsendd lst llvt 'length-up-to (1+ maxboxes)))
    (if (lng > maxboxes)
	(lng = ((totalx - boxw - lw - 26) / (deltax dspec))))
    (arrowy = offsety + (arrowup dspec))
    (erase-area-xy w boxx offsety ((or sizex totalx) - (boxx - offsetx))
		                  (or sizey boxh))
    (ptr = lst)
    (n = 0)
    (while (n < lng) do
      (if (box-style dspec) and (draw-pointer-cell dspec) 
	  (progn (draw-circle-xy w lastx arrowy 1 3)
	       (draw-arrow-xy w lastx arrowy boxx arrowy (min lw 4))
	       (draw-line-xy w boxx (offsety + boxh - lw - 13)
			       (boxx + boxw) (offsety + boxh - lw - 13) lw)
	       (if (glsendd (glsendd ptr llvt 'rest) llvt 'null)
		   (draw-line-xy w boxx (arrowy - 7)
				        (boxx + boxw) (arrowy + 7) (min 4 lw))))
	  (if (draw-arrow dspec)
	      (draw-arrow-xy w lastx arrowy boxx arrowy (min 4 lw))
	      (if (draw-line dspec)
		  (draw-line-xy w lastx arrowy boxx arrowy (min 4 lw)))))
      (dispm-draw-box (box-style dspec) w (boxx + lw2) (offsety + lw2)
		      (boxw - lw) (boxh - lw) lw)
      (draw (item-display dspec)
	    (if (cons-list dspec) (car ptr) ptr)
	    w (boxx + lw) (offsety + lw))
      (ptr = (glsendd ptr llvt 'rest))
      (if (draw-pointer-cell dspec) 
	  (lastx = boxx + boxw / 2)
	  (lastx = boxx + boxw))
      (boxx  = boxx + (deltax dspec))
      (n _+ 1) )
    (unless (glsendd ptr llvt 'null)
      (enddots = t)
      (printat-xy w "..." (boxx - (deltax dspec) + boxw + lw + 5) arrowy))
    (force-output w)
    (list lng enddots) ))

; 06 Dec 91; 07 Dec 91; 10 Dec 91; 11 Dec 91; 17 Dec 91; 23 Dec 91; 24 Dec 93
; 26 Dec 93; 28 Feb 02; 08 Jan 04; 03 Mar 04
; Draw and edit a linked-list display.
(gldefun ll-display-edit
  ((dspec lldisplayspec) (lst linked-list) (w window)
	     (offsetx integer) (offsety integer)
	     &optional (sizex integer) (sizey integer))
  (prog (current previous sel xspec yspec res resb resc button nx lw
		 xspec yspec (llvt (viewtype dspec)) enddots changed)
    (current = lst)
    (lw = (line-width dspec))
    (xspec = (a dispm-arr-spec with offs     = (deltax dspec)
		                      itemsize = (box-width dspec)
				      gapsize  = (arroww dspec) ) )
    (yspec = (a dispm-arr-spec with offs     = 0
		                      itemsize = (box-height dspec)
				      nitems   = 1))
 lp (resb = (ll-display-draw dspec current w offsetx offsety sizex sizey))
    (if (current <> lst)
	(printat-xy w "..." (offsetx + (offs xspec)
					   - (max (gapsize xspec) 21))
			          (offsety + (arrowup dspec) - 2)) )
    ((nitems xspec) = (first resb))
    (enddots = (second resb))
 lb (res = (dispm-track-array w 'outside offsetx offsety
				  sizex (box-height dspec) xspec yspec))
    (if ~ res (go out))
    (button = (car res))
    (nx = (cadr res))
    (if ((nx == 'low) and (current <> lst))    ; initial ... selected
	(progn (printat-xy w "   " (offsetx + (offs xspec)
					   - (max (gapsize xspec) 21))
			          (offsety + (arrowup dspec) - 2))
             (current = (pop previous))
             (go lp)))
    (if ((nx == 'high) and enddots)            ; final ... selected
	(progn (push current previous)
             (current = (ll-skip-n current llvt (nitems xspec)))
	     (go lp)))
    (if (not (numberp nx)) (go lp))
    (sel = (ll-skip-n current llvt nx))
    (if (eql button 3)
        (case (menu '(pop quit delete))
	       (quit (setq res 'quit) (go out))
	       (pop (setq res nil) (go out))
	       (delete (if (sel == lst)
			   (lst = sel)
			   (progn (prev = (ll-find-prev lst sel llvt))
			        (glsendd prev llvt '(rest 
					 (glsendd sel llvt 'rest)))))
		       (changed = t)
		       (go lp))
	       (t (go lb)) ) )
    (resc = (dispm-subed (subdisplay dspec)
			   (if (cons-list dspec)
			       (car sel)
			       sel)
			   w
			   (offsetx + (offs xspec) + lw
				    + nx * (total-size xspec))
			   (offsety + lw) (item-display dspec) button
			   (itemsize xspec) (itemsize yspec) ) )
    (if (consp resc)
	(progn (changed = t)
             (if (cons-list dspec) (rplaca sel (car resc)))
	     (draw (item-display dspec)
		   (if (cons-list dspec)
		       (car sel)
		       sel)
		   w
		   (nx * ((itemsize xspec) + (gapsize xspec))
		       + (offs xspec) + offsetx + lw)
		   (offsety + lw)) ))
    (go lb)
out (if changed (return (list lst))
                (return res))  ))

; 06 Dec 91
; Skip n items in a linked list described by view type llvt.
(gldefun ll-skip-n ((lst linked-list) (llvt gltype) (n integer))
  (while ((n > 0) and (not (glsendd lst llvt 'null))) do
	 (lst = (glsendd lst llvt 'rest))
	 (n _- 1) )
  lst)

; 06 Dec 91; 08 Jan 04
; Find predecessor in a linked list described by view type llvt.
(gldefun ll-find-prev ((lst linked-list) (item linked-list) (llvt gltype))
  (let (found nxt)
    (while ((not found) and (not (glsendd lst llvt 'null))) do
      (nxt = (glsendd lst llvt 'rest))
      (if (item == nxt)
	  (found = t)
	  (lst = nxt)) )
    lst))

; 12 Dec 91; 13 Dec 91; 17 Dec 91; 20 Dec 91; 22 Jul 94; 08 Jan 04
; Make a linked-list display editor spec for a given type descriptor.
; type might be e.g. (bl linked-list), (integer listof), cir-llp.
(gldefun ll-make-edit (type &optional data (sizex integer) (sizey integer))
  (let (vt tmp itemtype disname itemdis subdis llds dms itemsize)
    (if (consp type)
	(progn (itemtype = (first type))
             (disname = (second type))
             (if (disname == 'listof)
		 (vt = 'lisp-linked-list-pointer)
	         (if (tmp = (assoc disname (glviews itemtype)))
			  (vt = (second tmp))
			  (vt = 'lisp-linked-list-pointer))))
	(progn (vt = type)
	     (disname = 'linked-list)
             (itemtype = type) ))
    (itemdis = (dispm-type-editor itemtype 'short))
    (itemsize = (size itemdis))
    (subdis = (dispm-type-editor itemtype 'edit))
    (llds = (a lldisplayspec with viewtype = vt
		                    item-display = itemdis
				    subdisplay = subdis))
    (dms =	 (a dispm-method-spec with name = disname
		                       type = itemtype
				       drawfn = 'll-display-draw
				       editfn = 'll-display-edit
				       params = llds
				       size = (a vector with
		  x = (or sizex ((x itemsize) * 4) 200)
		  y = (min (or sizey 9999) (or (y itemsize) 30)))))
    (dispm-def-method dms)
    (a dispm-display-spec with display-name = disname type = itemtype)  ))

; 12 Dec 91; 19 Dec 91; 20 Dec 91; 08 Jan 04
(gldefun ll-find-or-make-edit (type &optional data
				    (sizex integer) (sizey integer))
  (if (and (consp type)
	   (assoc (second type) (gldisplays (first type))))
      type
      (if (and (symbolp type) (assoc 'linked-list (gldisplays type)))
	  (a dispm-display-spec with display-name = 'linked-list type = type)
	  (ll-make-edit type data sizex sizey))))

; 12 Dec 91; 13 Dec 91; 19 Dec 91; 20 Dec 91; 23 Dec 91; 24 Dec 91; 27 Dec 91
; 29 Jul 94; 28 Feb 02; 08 Jan 04
; Edit data in a window.
; distype may be a view type, e.g. (linked-list box)
(gldefun edit (data &optional distype (w window)
			      (offsetx integer) (offsety integer)
			      (sizex integer) (sizey integer))
  (let (type (ds dispm-display-spec) tmp)
    (or w (w = *default-editor-window*))
    (type = (glconstanttype data))
    (if ~ offsetx (offsetx = 10))
    (if ~ offsety (offsety = 10))
    (if ~ sizex (sizex = (width w) - offsetx))
    (if ~ sizey (sizey = (height w) - offsety))
    (if (and (consp distype) (member (car distype) '(linked-list ll-short)))
        (ds = (ll-find-or-make-edit
		       (list (cadr distype) (car distype)) data sizex sizey))
        (if (consp type)
	    (if (((car type) == 'listof) and (symbolp (cadr type)))
		(ds = (ll-find-or-make-edit (list (cadr type) 'listof)
						    data sizex sizey))
		 (if (((car type) == 'arrayof) and (symbolp (cadr type)))
		     (ds = (arr-find-or-make-edit (cadr type)
						     sizex sizey))))
	(if (consp distype)
	    (ds = distype)
	    (if (and (symbolp type) distype (assoc distype (gldisplays type)))
		(ds = (list type distype))
	        (if (symbolp type)
		    (ds = (dispm-type-editor type 'edit)) )))))
    (if ds
	(progn (clear w)
	       (dispm-display-spec-edit ds data w offsetx offsety sizex sizey)))
    data ))

; 23 Dec 91; 08 Jan 04
(gldefun arr-find-or-make-edit (type &optional (sizex integer) (sizey integer))
  (if (assoc 'arrayof (gldisplays type))
      (list type 'arrayof)
      (arr-make-edit type 'arrayof nil nil sizex sizey) ) )

; 19 Dec 91; 23 Dec 91; 29 Dec 93; 22 Jul 94; 08 Jan 04
; Make an array display editor spec for a given type descriptor.
; itemtype is the array element type.
(gldefun arr-make-edit (itemtype &optional disname itemdis
			     (itemsize vector) (sizex integer) (sizey integer))
  (let (subdis arrds dms)
    (if ~ disname (disname = 'arrayof))
    (if ~ itemdis
        (itemdis = (or (dispm-type-editor itemtype 'short)
		       (dispm-type-editor itemtype 'edit))))
    (if ~ itemsize (itemsize = (size itemdis)))
    (subdis = (dispm-type-editor itemtype 'edit))
    (arrds = (an arrdisplayspec with viewtype = itemtype
		                       draw-index-x = t
		                       draw-index-y = t
		                       item-display = itemdis
				       subdisplay = subdis))
    (dms = (a dispm-method-spec with name = disname
		                       type = itemtype
				       drawfn = 'arr-display-draw
				       editfn = 'arr-display-edit
				       params = arrds
				       size = (a vector with
		  x = (or sizex (min ((x itemsize) * 10) 200))
		  y = (or sizey (min ((y itemsize) * 10) 200)) ) ))
    (dispm-def-method dms)
    (a dispm-display-spec with display-name = disname type = itemtype)  ))

; 19 Dec 91; 20 Dec 91; 29 Dec 93
; Draw a box in specified style
(gldefun dispm-draw-box ((style symbol) (w window)
			 (offsetx integer) (offsety integer)
			 (sizex integer) (sizey integer) &optional lw)
  (let ((rx (sizex / 2)) (ry (sizey / 2)))
    (case style
      ((rectangle t) (draw-box-xy w offsetx offsety sizex sizey lw))
      (rcbox     (draw-rcbox-xy w offsetx offsety sizex sizey 8 lw))
      (circle    (draw-circle-xy w (offsetx + rx) (offsety + ry) rx lw))
      (ellipse   (draw-ellipse-xy w (offsetx + rx) (offsety + ry) rx ry lw)) )
    ))

; 06 Dec 91; 28 Feb 02; 08 Jan 04
; Find the item selected by a given numeric position.
; Returns LOW if less than the offset, HIGH if greater than nth item,
; number ( 0 .. n-1 ) if within an item, NIL if in a gap.
(gldefun dispm-arr-spec-n ((spec dispm-arr-spec) (x integer))
  (let (n left)
    (n = (x - (offs spec)) / ((itemsize spec) + (gapsize spec)))
    (left = (offs spec) + n * ((itemsize spec) + (gapsize spec)))
    (if ((n < 0) or (x < left))
        'low
        (if ((n >= (nitems spec)) or
		  ((n == ((nitems spec) - 1))
		     and (x > (left + (itemsize spec)))))
	    'high
	    (if (x <= (left + (itemsize spec))) n))) ))

; 06 Dec 91; 11 Dec 91; 02 Dec 93; 24 Dec 93
; Highlight a selected item in an array-like display
(gldefun dispm-arr-highlight ((highl symbol) (w window)
				        (offsetx integer) (topy integer)
					(xspec dispm-arr-spec) (nx integer)
					(yspec dispm-arr-spec) (ny integer))
  (let (left sizex top bottom sizey)
    (sizex = (itemsize xspec))
    (sizey = (itemsize yspec))
    (left = offsetx + (offs xspec) + nx * (total-size xspec))
    (top   = (topy - (offs yspec)) - ny * (total-size yspec))
    (bottom = top - sizey)
    (case highl
      (invert (window-invert-area-xy w left bottom sizex sizey))
      (inside (window-xor-box-xy w  (left + 2)  (bottom + 2)
				    (sizex - 4) (sizey - 4)))
      (outside (window-xor-box-xy w (left - 3)  (bottom - 3)
				    (sizex + 5) (sizey + 5))))
    (force-output w) ))
    

; 06 Dec 91; 08 Jan 04
; Track mouse in an array-like area in a window.
(gldefun dispm-track-array ((w window) (highl symbol)
			    (offsetx integer) (offsety integer)
			    (sizex integer) (sizey integer)
			    (xspec dispm-arr-spec) (yspec dispm-arr-spec))
  (let (highlighted button inside outside dx dy nx ny selx sely)
    (button = nil)
    (window-track-mouse w
      #'(lambda (x y code)
	  (dx = (x - offsetx))
	  (dy = ((offsety + sizey) - y))
	  (nx = (dispm-arr-spec-n xspec dx))
	  (ny = (dispm-arr-spec-n yspec dy))
	  (if selx
	      (if (highlighted and ((selx <> nx) or (sely <> ny)))
		  (progn (dispm-arr-highlight highl w
						 offsetx (offsety + sizey)
						 xspec selx yspec sely)
		            (highlighted = nil)
		            (selx = nil))))
	  (if ~ selx and (numberp nx) and (numberp ny)
	      (progn (selx = nx)
	           (sely = ny)
		   (dispm-arr-highlight highl w offsetx (offsety + sizey)
						xspec nx yspec ny)
		   (highlighted = t) ))
	  (outside = ((dx < 0) or (dx > sizex) or (dy < 0) or (dy > sizey)))
	  (inside = inside or ~ outside)
	  (if (> code 0) (button = code))
	  (or button (and inside outside)) ))
    (if highlighted
        (dispm-arr-highlight highl w offsetx (offsety + sizey)
				  xspec selx yspec sely))
    (if button (list button nx ny)) ))

; 08 Dec 91; 09 Dec 91; 10 Dec 91; 15 Dec 91; 24 Dec 91; 24 Dec 93; 26 Dec 93
; 28 Feb 02; 08 Jan 04
; Draw an array display.  nx, ny are offsets of display from 0 index.
(gldefun arr-display-draw
  ((dspec arrdisplayspec) (arr array) (w window)
	     (offsetx integer) (offsety integer)
	     &optional (sizex integer) (sizey integer)
	     (ncol integer) (nrow integer))
  (result (list (xxspec dispm-arr-spec) (yyspec dispm-arr-spec)
		(flgs (list (leftdots boolean) (rightdots boolean)
			    (topdots boolean)  (bottomdots boolean)))))
  (let (rank dims rightdots bottomdots (xoff 1) (yoff 1) nx ny lw lw2 lw2b item
	     xleft xright ytop ybottom
	     (maxx 1) (maxy 1) totalx totaly (llvt (viewtype dspec)) xx yy)
    (erase-area-xy w offsetx offsety sizex sizey)
    (lw = (if (draw-box dspec)
	      (line-width dspec)
	      0))
    (lw2 = lw / 2)
    (lw2b = lw - lw2)
    (rank = (array-rank arr))
    (dims = (array-dimensions arr))
    (or ncol (ncol = 0))
    (or nrow (nrow = 0))
    (if (draw-index-y dspec) (xoff _+ 15))
    (if (draw-index-x dspec) (yoff _+ 16))             
    (if (vertical dspec)
	(progn (nx = nrow)
             (ny = ncol)
	     (if (rank == 1)
		 (maxy = (first dims) - ncol)
	         (progn (maxy = (second dims) - ncol)
			(maxx = (first dims) - nrow))))
	(progn (nx = ncol)
	     (ny = nrow)
	     (if (rank == 1)
		 (maxx = (first dims) - ncol)
	         (progn (maxx = (second dims) - ncol)
			(maxy = (first dims) - nrow)) )))
    (if (nx > 0) (xoff _+ 12))
    (if (ny > 0) (yoff _+ 12))
    (totalx = (max 0 ((min (or sizex 9999) ((width w) - offsetx) )
			- xoff - lw)) )
    (totaly = (max 0 ((min (or sizey 9999) ((height w) - offsety) )
			- yoff - lw)) )
    (if ((totalx / (deltax dspec)) < maxx)
	(progn (maxx = ((totalx - 12) / (deltax dspec)))
	       (rightdots = t)))
    (if ((totaly / (deltay dspec)) < maxy)
	(progn (maxy = ((totaly - 12) / (deltay dspec)))
	       (bottomdots = t)))
    (xleft = offsetx + xoff + lw)
    (xright = xleft + maxx * (deltax dspec))
    (ytop = (offsety + sizey - yoff))
    (ybottom = ytop - maxy * (deltay dspec) - lw)
    (if (ny > 0)
        (printat-xy w "..." (xleft + (maxx * deltax) / 2 - 11 - lw2)
			         (ytop + 5)))
    (if (nx > 0)
	(dotimes (i 3)
	  (printat-xy w "." (xleft - lw - 12)
		   (ytop - ((maxy * deltay + lw) / 2 + (i - 1) * 12)))))
    (if bottomdots
	 (printat-xy w "..." (xleft + (maxx * deltax) / 2 - 11 - lw2)
			         (ybottom - 12)))
    (if rightdots
	(dotimes (i 3)
	  (printat-xy w "." (xright + 4)
		   (ytop - ((maxy * deltay + lw) / 2 + (i - 1) * 12)))))
    (if (draw-index-x dspec)
	(dotimes (i maxx)
	  (printat-xy w (+ i nx) (xleft + (* i deltax) + deltax / 2 - 4 - lw2)
		                 (offsety + sizey - 12)) ) )
    (if (draw-index-y dspec)
        (dotimes (i maxy)
	  (printat-xy w (+ i ny) (offsetx + 3)
		      (ytop - ((* i deltay) + deltay / 2 + 6) - lw2))) )
    (yy = ytop - lw2)
    (if (draw-box dspec)
	(progn
	  (draw-line-xy w xleft yy xright yy lw)
	  (draw-line-xy w (xleft - lw2b) ytop (xleft - lw2b) ybottom lw)))
    (dotimes (i maxy)
      (yy = yy - (deltay dspec))
      (xx = xleft)
      (dotimes (j maxx)
	(item = (if (> rank 1)
		    (if (vertical dspec)
			(aref arr (j + nx) (i + ny))
		        (aref arr (i + ny) (j + nx)))
		    (if (vertical dspec)
			(aref arr (i + ny))
		        (aref arr (j + nx)))))
	(draw (item-display dspec) item w xx (yy + lw2b)
	      (x (contents-size dspec)) (y (contents-size dspec)))
	(xx _+ (deltax dspec))
	(if ((i == 0) and (draw-box dspec))
	    (draw-line-xy w (xx - lw2b) ytop (xx - lw2b) ybottom lw)) )
      (if (draw-box dspec)
	  (draw-line-xy w xleft yy xright yy lw)) )
    (force-output w)
; Make parameters for use in tracking by arr-display-edit.
    (list (a dispm-arr-spec with offs     = (xoff + lw)
		                 itemsize = (x (contents-size dspec))
				 gapsize  = lw
				 nitems   = maxx)
	  (a dispm-arr-spec with offs     = (yoff + lw)
		                 itemsize = (y (contents-size dspec))
				 gapsize  = lw
				 nitems   = maxy)
;               leftdots rightdots topdots  bottomdots
	  (list (nx > 0) rightdots (ny > 0) bottomdots))    ))


; 10 Dec 91; 11 Dec 91; 15 Dec 91; 24 Dec 91; 24 Dec 93; 26 Dec 93; 28 Feb 02
; 08 Jan 04; 03 Mar 04
; Draw and edit an array display.
(gldefun arr-display-edit
  ((dspec arrdisplayspec) (arr array) (w window)
	     (offsetx integer) (offsety integer)
	     &optional (sizex integer) (sizey integer))
  (prog (lw res resb resc resd (ncol 0) (nrow 0) (llvt (viewtype dspec))
	 changed rank flags xspec yspec nxx nyy nx ny button sel prevx
	 prevy lw2 lw2b)
    (lw = (line-width dspec))
    (lw2 = lw / 2)
    (lw2b = lw - lw2)
    (rank = (array-rank arr))
; Draw display and get parameters of drawing
 lp (resb = (arr-display-draw dspec arr w offsetx offsety sizex sizey
				ncol nrow))
    (xspec = (xxspec resb))
    (yspec = (yyspec resb))
    (flags = (flgs resb))     ;  leftdots rightdots topdots bottomdots
 lb (res = (dispm-track-array w 'inside offsetx offsety sizex sizey
				  xspec yspec))
    (if ~ res (return res))
    (button = (first res))
    (nxx = (second res))
    (nyy = (third res))
    (if (eql button 3)
	(case (menu '(pop quit))
	       (quit (setq resd 'quit) (go out))
	       (pop (setq resd nil) (go out))
	       (t (go lp)) ) )
    (if ((nxx == 'low) and (leftdots flags))             ; left ... selected
	(progn (ncol = (pop prevx))
	       (go lp)))
    (if ((nxx == 'high) and (rightdots flags))           ; right ... selected
	(progn (push ncol prevx)
	       (ncol _+ (nitems xspec))
	       (go lp)))
    (if ((nyy == 'low) and (topdots flags))              ; top ... selected
	(progn (nrow = (pop prevy))
	       (go lp)))
    (if ((nyy == 'high) and (bottomdots flags))          ; bottom ... selected
	(progn (push nrow prevy)
	       (nrow _+ (nitems yspec))
	       (go lp)))
    (if (not (and (numberp nxx) (numberp nyy))) (go lp))
    (nx = nxx + ncol)
    (ny = nyy + nrow)
    (sel = (if (> rank 1)
	       (if (vertical dspec)
		   (aref arr nx ny)
		   (aref arr ny nx))
	       (if (vertical dspec)
		   (aref arr ny)
		   (aref arr nx))))
    (resc = (dispm-subed (subdisplay dspec) sel w
			   (offsetx + (offs xspec)
				    + nxx * (total-size xspec))
			   (offsety + sizey - (offs yspec) + lw
				    - (nyy + 1) * (total-size yspec) + lw2b)
			   (item-display dspec) button
			   (itemsize xspec) (itemsize yspec) ) )
    (if (consp resc)
	(progn (changed = t)
             (if (> rank 1)
		 (if (vertical dspec)
		     (setf (aref arr nx ny) (car resc))
		     (setf (aref arr ny nx) (car resc)))
	         (if (vertical dspec)
		     (setf (aref arr ny) (car resc))
		     (setf (aref arr nx) (car resc)) ) )
	     (draw (item-display dspec) (car resc) w
		   (offsetx + (offs xspec) + nxx * (total-size xspec))
		   (offsety + sizey - (offs yspec) + lw
			    - (nyy + 1) * (total-size yspec) + lw2b)
		   (x (contents-size dspec)) (y (contents-size dspec))) ))

    (go lb)      ; to go to lb, must redisplay the changed item.
out (if changed
	(return (list arr))
        (return resd))  ))

; (make-display-method 'box mybox '(50 50) 'display)
; (llta mybox)
(gldefun llta ((b box))
  (ll-display-draw (a lldisplayspec viewtype        = 'box-as-ll-pointer
			            line-width        = 1
				    draw-pointer-cell = t
				    item-display      = '(box display))
		   (linked-list b) *myw* 10 20 ))

; (make-display-method 'box mybox '(45 20) 'short)
; (lltb mybox)
(gldefun lltb ((b box))
  (ll-display-draw (a lldisplayspec viewtype        = 'box-as-ll-pointer
		                    line-width        = 1
				    draw-pointer-cell = t
				    item-display      = '(box short))
		   (linked-list b) *myw* 10 100 ))

; (lltc mybox)
(gldefun lltc ((b box))
  (ll-display-draw (a lldisplayspec viewtype = 'box-as-ll-pointer
			            line-width = 1
				    item-display = '(box short))
		   (linked-list b) *myw* 10 160 ))

; see llte
(gldefun lltd ((b bl) (xoff integer) (yoff integer) &optional bs da dl lw)
  (ll-display-draw (a lldisplayspec viewtype = 'bl-as-ll-pointer
			            line-width = (or lw 1)
				    box-style  = bs
				    draw-arrow = da
				    draw-line  = dl
				    item-display  = '(bl display))
		   (linked-list b) *myw* xoff yoff ))

; (make-display-method 'bl  mybl  '(20 20) 'display)
; (viewas 'linked-list 'bl)
; (llte mybl)
(gldefun llte ((b bl))
  (lltd b 10 150 'rectangle t)
  (lltd b 10 175 'circle t)
  (lltd b 10 200 'rcbox t)
  (lltd b 10 225 nil t)
  (lltd b 10 250 nil nil t)
  (lltd b 10 275 ) )

; 05 Dec 91; 11 Dec 91
(pushnew '(linked-list box
		       (ll-display-draw ll-display-edit nil nil
			 (lldisplayspec box-as-ll-pointer 0 1 (t t t t)
					(box display) (box display))
			 nil nil)
		       (280 70) nil (red 1 (green 2 (blue 3 nil)))
		       nil nil nil)
	 (gldisplays 'box) :test #'equal)

(pushnew '(ll-short box
		    (ll-display-draw ll-display-edit nil nil
		      (lldisplayspec box-as-ll-pointer 0 1 (t nil t t)
					(box short) (box display))
			 nil nil)
		       (280 22) nil (red 1 (green 2 (blue 3 nil)))
		       nil nil nil)
	 (gldisplays 'box) :test #'equal)

(pushnew '(linked-list bl
		       (ll-display-draw ll-display-edit nil nil
			 (lldisplayspec bl-as-ll-pointer 0 1
					(t nil t t)
					(bl display) (bl display))
			 nil nil)
		       (280 22) nil '((nil . 7) . 6)
		       nil nil nil)
	 (gldisplays 'bl) :test #'equal)

; 11 Dec 91
; Test data for array displays
(setq myarr (make-array 4 :initial-contents '(1 2 3 4)))
(setq my2arr (make-array '(3 4)
  :initial-contents '((1 2 3 4) (5 6 7 8) (9 10 11 12))))
(setq my2arrb (make-array 12 :displaced-to my2arr))
(setq my2b (make-array '(6 6)
  :initial-contents '((1 2 3 4 5 6) (7 8 9 10 11 12) (13 14 15 16 17 18)
	      (19 20 21 22 23 24) (25 26 27 28 29 30) (31 32 33 34 35 36))))
(setq myard (an arrdisplayspec viewtype 'integer
  line-width 1 draw-box t draw-index-x t draw-index-y t
  item-display '(integer display) subdisplay '(integer thermom)))
(setq myarb '(ARRDISPLAYSPEC box 1 nil (T T nil NIL) (box short)
			     (box display)))
(setq myardb (copy-tree myard)) (setf (third myardb) 3)
(setq myardc (copy-tree myard)) (setf (third (fifth myardc)) t)
(setq myarrc (make-array 8 :initial-element nil))
(setq myboxc '(blue 2 (red 3 (green 4 nil))))
(setq myboxd '(yellow 5 (red 3 (blue 7 nil))))
(setq myarrb (make-array 3 :initial-contents (list mybox myboxb myboxc)))
(setq myardcc '(ARRDISPLAYSPEC boolean 1 nil (T T NIL NIL) (boolean t/f)
			        (boolean t/f)))
(setq myardd '(arrdisplayspec boolean 1 (10 10) (T nil nil nil) (boolean b/w)
			        (boolean b/w)))
(setq myl '(1 2 3 4 5 6 7 8 9 10 11 12 13 14 15 16))
(setq myarbl (make-array '(8 8) :initial-element nil))
; (progn (clr) (arr-display-edit myard my2b *myw* 50 50 150 100))
; (progn (clr) (arr-display-edit myarb myarrb *myw* 20 20 200 90))
; (progn (clr) (arr-display-edit myardcc myarrc *myw* 20 20 200 90))
; (edit mybox '(linked-list box))
; (edit mybl '(linked-list bl))
; (edit myl)
; (edit (list happy dopey grumpy))
; (edit dopey '(linked-list elf) nil 50 100 250 100)
; (edit (list mxe erw gsn jca))
; (progn (clr) (arr-display-edit myardd myarbl *myw* 50 50 100 100))
(setq myarde '(ARRDISPLAYSPEC box 1 nil (T T T NIL) (box display)
			      (box display)))
(setq myarrb2 (make-array '(2 2) :initial-contents
			  (list (list mybox myboxb) (list myboxc myboxd))))
; (progn (clr) (arr-display-edit myarde myarrb2 *myw* 50 50 120 120))
