; mathsimp.lsp                  Gordon S. Novak Jr.         ; 30 Sep 06

; Copyright (c) 2007 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; 23 Dec 94; 05 Jan 95; 02 Jan 97; 28 Feb 02; 27 Apr 04; 07 Jun 04; 08 Jun 04
; 30 Jun 06

(defvar *gleqnsmatch-trace* nil)

; 26 Apr 94; 07 Oct 94; 27 Apr 04; 07 Jun 04
; eval a form that may include constants with units.
; cf. trans in ~/autop/patm.lsp
(defun gleqns-trans (x patwd)
  (let (xp tail)
    (if (and (consp x)            ; if certain fns are applied to constants
	     (not (member (car x) '(quote function))))
	(progn
	  (setq tail (gleqns-transl (rest x) patwd))  ; translate args first
	  (unless (eq tail (rest x)) (setq x (cons (first x) tail)))
	  (if (and (member (first x) '(+ - * / expt sqrt sin cos tan
					 asin acos atan 1+ 1-
					 = /= > >= < <=))
		   (every #'glnumberp (rest x)))
	      (if (and (eq (first x) '/)
		       (cddr x)
		       (zerop (glqn (caddr x))))
		  x
		  (glevalunits x))
	      (progn (setq xp (gleqnsmatch x patwd))   ; was glptmatch
		     (if (eq x xp)               ; if it changed, hit it again
			 x
		         (gleqns-trans xp patwd))) ) )
        x) ))

; 26 Jan 93; 26 Apr 94
; Translate a list of arguments.  Avoids conses if no changes are made.
; cf. pttransl in patm.lsp
(defun gleqns-transl (l patwd)
  (let (new tail)
    (when l
      (setq new (gleqns-trans (first l) patwd))
      (setq tail (gleqns-transl (rest l) patwd))
      (if (or (not (eq tail (rest l)))
	      (not (eq new (first l))))
	  (cons new tail)
	  l)) ))

; 26 Apr 94; 27 Apr 94; 28 Apr 94; 29 Apr 94; 07 Oct 94; 28 Feb 02; 30 Jun 06
; Perform an operation on constants, coercing units as necessary
; e.g. convert feet to meters.  Args may be numbers or '(Q n unit) constants.
; cf. glcoerceunits
(defun glevalunits (form)
  (let (op lhs lhsu rhs rhsu exponent newunit factor expunit)
    (setq op (car form))
    (setq lhs (glqn (cadr form)))
    (setq lhsu (glqunit (cadr form)))
    (when (cddr form)
      (setq rhs (glqn (caddr form)))
      (setq rhsu (glqunit (caddr form))) )

    (case op
      ((* /) (glqres (funcall op lhs rhs)
		     (if (eq op '*)
			 (glsimplifyunit (glmultunits lhsu rhsu))
		         (glsimplifyunit (gldivunits lhsu rhsu)))))
      ((\:= + - < <= = == <> != >= >)
       (if (and (eq op '-) (null (cddr form)))
	   (glqres (funcall op lhs) lhsu)
	   (progn (if (or (equal lhsu rhsu)
			  (zerop lhs)           ; got to be good lookin'
			  (zerop rhs))          ; 'cause it's so hard to see
		      (setq factor 1)
		      (or (setq factor (glconvertunit rhsu lhsu))
			  (progn (glerror 'glevalunits
					  "Cannot apply op ~A to ~A and ~A"
					  op lhsu rhsu)
				 (setq factor 1))))
		  (glqres (funcall op lhs (* factor rhs)) lhsu) )))
      ((^ expt)
       (if (and (integerp rhs)
		(< (setq exponent (abs rhs)) 6)
		(eq rhsu 'unity))
	   (progn
	     (setq expunit 'unity)
	     (while (> exponent 0)
	       (decf exponent)
	       (setq expunit
		     (if (minusp rhs)
			 (glsimplifyunit (gldivunits expunit lhsu))
		         (glsimplifyunit (glmultunits expunit lhsu)))))
	     (glqres (funcall op lhs rhs) expunit))
	 form))
      (sqrt (if (setq newunit (glsqrtunit lhsu nil t))
		(glqres (funcall op lhs) newunit)
	        form))
      (cbrt (if (setq newunit (glcbrtunit lhsu nil t))
		(glqres (funcall op lhs) newunit)
	        form))
      ((sin cos tan) (if (setq factor (glconvertunit lhsu 'radian))
			 (funcall op (* factor lhs))
		         form))
      (atan (if (and rhsu lhsu (setq factor (glconvertunit rhsu lhsu)))
		(glqres (atan lhs (* factor rhs)) 'radian)
	        form))		
      (t (if (every #'glevalnumberp (cdr form))
	     (eval form)
	     form) ) ) ))

; 27 Apr 94
(defun glevalnumberp (x)
  (or (numberp x)
      (and (constantp x) (numberp (eval x)))))
 
; 26 Apr 94; 24 May 94
; Get the unit from a Q constant form
(defun glqunit (x)
  (if (numberp x)
      'unity
      (if (consp x)
	  (if (eq (car x) 'q)
	      (caddr x)
	      (if (and (eq (car x) 'quote)
		       (consp (cadr x))
		       (eq (caadr x) 'q))
		  (caddr (cadr x))))
	  (if (symbolp x)
	      (if (constantp x)
		  'unity
		  (if (glispconstantflg x)
		      (glispconstanttype x)))) ) ) )

; 26 Apr 94; 24 May 94
; Get the number from a Q constant form
(defun glqn (x)
  (if (numberp x)
      x
      (if (consp x)
	  (if (eq (car x) 'q)
	      (cadr x)
	      (if (and (eq (car x) 'quote)
		       (consp (cadr x))
		       (eq (caadr x) 'q))
		  (cadadr x)))
	  (if (symbolp x)
	      (if (constantp x)
		  (eval x)
		  (if (glispconstantflg x)
		      (glispconstantval x)))) ) ) )

; 26 Apr 94; 30 Sep 06
; Put a result back into Q form
(defun glqres (x unit)
  (if (numberp unit)
      (* x unit)
      (if (and (consp unit)
	       (eq (car unit) '*)
	       (numberp (cadr unit)))
	  (glqres (* x (cadr unit))
                  (if (cddr unit)
                      (cons '* (cddr unit))
                      (caddr unit)))
	  (if (eq unit 'unity)
	      x
	      (list 'quote (list 'q x unit))))))

; 28 Apr 94
; Relational operators that work for numbers or q constants
(defun glq>0  (x) (>  (glqn x) 0))
(defun glq>=0 (x) (>= (glqn x) 0))
(defun glq/=0 (x) (/= (glqn x) 0))

; 28 Apr 94
; Test for a Q constant form, (quote (q <n> <unit>))
(defmacro glqconstp (x)
  `(and (consp ,x) (eq (car ,x) 'quote) (consp (cdr ,x))
	(consp (cadr ,x)) (eq (caadr ,x) 'q)
	(consp (cdadr ,x))
	(numberp (cadadr ,x))))

; 28 Apr 94
; Test if it is legal to take sqrt of a constant
(defun glsqrtable (x)
  (or (and (numberp x) (>= x 0))
      (and (glqconstp x)
	   (glsqrtunit (glqunit x) nil t)
	   (>= (glqn x) 0))) )

; 29 Apr 94
; Test if it is legal to take cbrt of a constant
(defun glcbrtable (x)
  (or (numberp x)
      (and (glqconstp x)
	   (glcbrtunit (glqunit x) nil t) ) ) )

; 23 Dec 94
; Test whether a code form x can absorb a multiplicative constant
(defun glabsorbconstant (x)
  (if (atom x)
      (numberp x)
      (case (first x)
	(* (or (glabsorbconstant (cadr x))
	       (glabsorbconstant (caddr x))) )
	(/ (glabsorbconstant (cadr x)) )
	((+ - > >= = /= <= <)
	   (and (glabsorbconstant (cadr x))
		(glabsorbconstant (caddr x))) )
	(if (and (glabsorbconstant (caddr x))
		 (glabsorbconstant (cadddr x))) )
	(sqrt (glabsorbconstant (cadr x)))
	(t nil) ) ) )

; 23 Dec 94
; Push a multiplicative constant value C into an expression X
; Returns a new expression if success, NIL if failure.
(defun glpushconstant (c x)
  (let (tmp tmpb)
    (if (atom x)
	(if (numberp x)
	    (* c x)
	    nil)
        (case (first x)
	  (* (if (setq tmp (glpushconstant c (second x)))
		 (list (first x) tmp (third x))
	         (if (setq tmp (glpushconstant c (third x)))
		     (list (first x) (second x) tmp)
		     nil)))
	  (/ (if (setq tmp (glpushconstant c (second x)))
		 (list (first x) tmp (third x))
	         nil))
	  ((+ - > >= = /= <= <)
	     (if (and (setq tmp (glpushconstant c (second x)))
		      (setq tmpb (glpushconstant c (third x))))
		 (list (first x) tmp tmpb)))
	  (if (if (and (setq tmp (glpushconstant c (third x)))
		      (setq tmpb (glpushconstant c (fourth x))))
		  (list (first x) (second x) tmp tmpb)))
	  (sqrt (if (and (> c 0)
			 (setq tmp (glpushconstant (* c c) (second x))))
		    (list (first x) tmp)))
	  (t nil) ) ) ))

; 27 Apr 04; 08 Jun 04
; try to transform input using patterns
(defun gleqnsmatch (inp patwd)
  (let (patterns done res pat)
    (if (and (consp inp)
	     (symbolp (car inp)))
	(setq patterns (get (car inp) patwd)) )
    (while (and patterns (not done))
      (setq pat (car patterns))
      (setq res (transf pat inp))
      (if (not (eq res 'match-failure))
	  (setq done t))
      (setq patterns (cdr patterns)) )
    (if (and done *gleqnsmatch-trace*)
	(format t "~A ^ ~A --> ~A~%" inp pat res))
    (if done res inp) ))

