;  parsed.lsp        Gordon S. Novak Jr.        ; 22 Feb 12

; Copyright (c) 2012 Gordon S. Novak Jr. and The University of Texas at Austin.

; Functions to parse data lines and analyze web accesses
; 09 Sep 02; 26 Sep 02; 27 Sep 02; 02 Oct 02; 07 Oct 02; 08 Oct 02; 25 Oct 02
; 15 Jan 03; 29 Apr 03; 30 Apr 03; 04 Sep 03; 30 Oct 03; 24 Mar 04; 25 Mar 04
; 27 Mar 04; 12 Apr 04; 13 Apr 04; 23 Apr 04; 03 Jun 04; 31 Oct 04; 02 Dec 05
; 08 Feb 06; 07 Mar 06; 13 Mar 06; 07 Jul 06; 09 Oct 06; 10 Oct 06; 12 Oct 06
; 17 Oct 06; 18 Oct 06; 19 Oct 06; 24 Oct 06; 25 Oct 06; 17 Dec 08; 05 Jan 11
; 07 Jan 11; 07 Oct 11

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software Foundation,
; Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


; Instructions for using this program:

; 1. Modify the lists *papers*, *classes*, and *ignore* for your use.
;    *papers* lists names of .html files that will be counted as
;    paper accesses.
;    *classes* lists names that will be counted as class accesses;
;    any file name beginning with a name on *classes* will be counted
;    as an access for that class.
;    *ignore* is a list of accessors to be ignored.  Change the first
;    line to your own machine path to ignore self-accesses.

; 2. Do the following on Unix (change 'novak' to your login name).
;    This will capture the log information at the end of the day.
;
;    cat >whonet
;    ( cd /var/spool/net/www/logs ; grep 'novak' * )
;    ^D       (control-D)
;    chmod +x whonet
;    crontab
;    50 23 * * * whonet >who.net
;    ^D       (control-D)

; 3. To process the previous day's statistics, do the following.
;    Change the variable *directory* for your directory.
;    Change the load to access your copy of the program.
;    It may take a few minutes to process all the data.
;
;    gcl
;    (load "/u/novak/glisp/parsed.lsp")    ; or parsed.o (compiled)
;    (netstats)
;    (bye)       ; to exit from gcl
;
;    You can get totals with (netstats t) or averages with (netstats 'a)
;    You can reset the totals with (netstats t t) or (netstats 'a t) .
;
;    The program will run faster if compiled.  To compile it:
;
;    gcl
;    (compile-file "parsed.lsp")

(defvar *directory*)
(setq *directory* "/u/novak/")

; html names that should be counted as paper accesses
(defvar *papers*)
(setq *papers* '(aaai90 aaai91 aimag83 ajcl76 caia90 caia94 chap94
		 ijcai77 jvlc93 kbse92 kbse94 sara00 symp92
		 tkde91 tose92 tose95 tose97 units95 wisr99))

; files beginning with these names are counted as class accesses
(defvar *classes*)
(setq *classes* '(cs307 cs315 cs343 cs375 cs381k cs394p))

; ignore my own machine and web crawlers in making counts
(defvar *ignore*)
(setq *ignore* '((loch cs utexas edu)
		 (chasm csres utexas edu)
		 (hobart csres utexas edu)
		 (google cc utexas edu)
		 (* googlebot com)
		 (* picsearch com)
		 (* inktomisearch com)
		 (* search mud yahoo com)
		 (* image search mud yahoo net)
		 (* fastsearch net)
		 (crawlers looksmart com)
		 (dnaspider01 mia lycos com)
		 (* * fastsearch net)
		 (* streaming cesnet cz)
		 (* betaspider com)
		 (msnbot msn com)
		 (* msnbot msn com)
		 (* 220 16 91)    ))

(proclaim '(optimize (compilation-speed 0) (safety 3) (speed 3)))

(defmacro while (test &rest forms)
  `(loop (unless ,test (return)) ,@forms) )

(defvar *prefix* "")
(defvar *line*)
(defvar *ptr*)
(defvar *lng*)
(defvar *eof* nil)
(defvar *done* nil)
(defvar *linenumber* 0)
(defvar *symchars*)               ; chars that can be part of symbol
(setq  *symchars* '(#\- #\_ #\$))
(defvar *xmlsymchars*)               ; chars that can be part of XML symbol
(setq  *xmlsymchars* '(#\- #\_ #\$ #\. #\:))
(defvar *parsedfile* nil)
(defvar *tracexml* nil)
(defvar *xmlomits*)      ; tags to omit when doing html
(setq *xmlomits* '(valign align))

(defvar *cstrcons* (cons 'string nil))
(defun cstr (&rest strings)
  (setf (cdr *cstrcons*) strings)
  (apply #'concatenate *cstrcons*))

; initialize the line for parsing to be the specified string
(defun initline (string)
  (setq *line* string)
  (setq *ptr* 0)
  (setq *lng* (length *line*)))

; read a line and increment line number
(defun parseread (infile)
  (incf *linenumber*)
  (if infile
      (read-line infile nil 'zzeofvalue)
      'zzeofvalue) )

; parse successive lines from a file.
; returns a list of results
(defun parselines (filenm gram)
  (let (line res)
    (setq *parsedfile* nil)
    (with-open-file (infile filenm :direction :input
				:if-does-not-exist nil)
      (while (not (eq (setq line (parseread infile))
                      'zzeofvalue))
	(push (parsed line gram) res) ) )
    (nreverse res) ))

; parse successive lines from a file, call a function on them.
(defun parselinesfn (filenm gram fn)
  (let (line)
    (setq *parsedfile* nil)
    (with-open-file (infile filenm :direction :input
				:if-does-not-exist nil)
      (while (not (eq (setq line (parseread infile))
                      'zzeofvalue))
	(funcall fn (parsed line gram) line) ) ) ))

; 24 Mar 04; 06 Jan 11
; parse successive lines from a file, filter, write to output file
; for each line of input,
;    for each test of testacts that is satisfied, call the action
;    if pred is true, write to the output file
(defun filterlines (filenm outfilenm pred testacts)
  (let (line res)
    (setq *parsedfile* nil)
    (with-open-file (infile filenm :direction :input
				:if-does-not-exist nil)
      (with-open-file (outfile outfilenm :direction :output
				:if-does-not-exist :create)
	(while (not (eq (setq line (parseread infile))
                      'zzeofvalue))
	  (dolist (testact testacts)
	    (if (setq res (funcall (car testact) line))
		(funcall (cadr testact) line res outfile)) )
	  (if (funcall pred line)
	      (write-line line outfile)) ) ) )
    ))

; 10 Oct 06; 17 Oct 06; 06 Jan 11
; parse all lines from a file.
; returns a list of results
(defun parsefile (filenm gram)
  (let (line res)
    (with-open-file (infile filenm :direction :input
				:if-does-not-exist nil)
      (setq *parsedfile* infile)
      (setq *linenumber* 0)
      (setq *ptr* 0)
      (setq *lng* 0)
      (setq res (parseg gram)) )
    (setq *parsedfile* nil)
    res ))

; 13 Sep 02
; parse a line according to a grammar spec
(defun parsed (line gram)
  (initline line)
  (setq *parsedfile* nil)
  (parseg gram) )

; 13 Sep 02; 10 Oct 06
; parse from current line and pointer according to grammar
; The convention is that a grammar part returns nil if failure,
; 'zzomit if it succeeded but the result should be omitted.
; If failure, *ptr* must be left as before.
(defun parseg (gram)
  (let (line)
    (while (and *parsedfile* (not (> *lng* *ptr*)))
      (setq line (parseread *parsedfile*))
      (if (eq line 'zzeofvalue)
          (setq *parsedfile* nil)
          (initline line)))
    (if (< *ptr* *lng*)
        (if (consp gram)
            (case (car gram)
              (seq (parseseq gram))              ; sequence of items
              (* (parse* gram))                  ; 0 or more repetitions
              (int (parseint gram))              ; integer
              (num (parsenum gram))              ; number
              (sym (parsesym gram))              ; symbol
              (xmlsym (parsexmlsym gram))        ; XML symbol
              (alphanum (parsealphanum gram))    ; alphanumeric string
              (xmlattr (parsexmlattr gram))      ; xml alphanumeric-. string
              (skip (parseskip gram))
              (skipto (parseskipto gram))
              (skiptoline (parseskiptoline gram))
              (skipws (parseskipws gram))
              (stringto (parsestringto gram))
              (stringupto (parsestringupto gram))
              (xmltext  (parsexmltext gram))
              (delim (parsedelim gram))
              (or (parseor gram))
              (col (parsecol gram))
              (opt (parseopt gram))              ; optional
              (car (car (parseg (cadr gram))))   ; car of a list result
              (append (parseappend gram))        ; append results
              (fn (parsefn gram))                ; apply fn to result
              (xml (parsexml gram))
              (xmlparms (parsexmlparms gram))
              (echo (parseecho gram))
              (omit (and (parseg (cadr gram)) 'zzomit)) )
            (if (stringp gram)
                (parsestring gram)        ; exactly the specified string
	        (if (and (symbolp gram) (get gram 'grammar))
                    (parseg (get gram 'grammar))
	            nil)))
        nil) ))

(defun defgram (name gram) (setf (get name 'grammar) gram))
(defmacro listify (x) `(if (consp ,x) ,x (if ,x (list ,x))))

; 13 Sep 02
; parse a sequence of specified items
; grammar is (seq <grammar1> ... <grammarn>)
; result is a list of sub-results, with nil sub-results omitted
(defun parseseq (gram)
  (let (res item fail (save *ptr*))
    (setq gram (cdr gram))
    (while (and (not fail) gram)
      (setq item (parseg (pop gram)))
      (if item
	  (if (not (eq item 'zzomit))
	      (push item res))
	  (setq fail t)) )
    (if fail
	(progn (setq *ptr* save) nil)
        (nreverse res)) ))

; 13 Sep 02
; parse a sequence of specified items, appending the results
; grammar is (append <grammar1> ... <grammarn>)
; result is a list of sub-results, with nil sub-results omitted
(defun parseappend (gram)
  (let (res item fail (save *ptr*))
    (setq gram (cdr gram))
    (while (and (not fail) gram)
      (setq item (parseg (pop gram)))
      (if item
	  (if (not (eq item 'zzomit))
	      (setq res (append res (listify item))))
	  (setq fail t)) )
    (if fail
	(progn (setq *ptr* save) nil)
        res) ))

; 13 Sep 02
; parse a sequence of 0 or more items
; grammar is (* <grammar>)
(defun parse* (gram)
  (let (res item)
    (while (setq item (parseg (cadr gram)))
      (if (not (eq item 'zzomit)) (push item res) ) )
    (nreverse res) ))

; 13 Sep 02
; set column to specified column number, starting with 1
; grammar is (col <n>)
(defun parsecol (gram)
  (if (and (numberp (cadr gram))
	   (<= (cadr gram) *lng*))
      (progn (setq *lng* (1- (cadr gram))) 'zzomit) ) )

; 13 Sep 02; 10 Oct 06; 08 Oct 07; 22 Feb 12
; skip to a specified string
; grammar is (skipto "str")
(defun parseskipto (gram)
  (prog (c i found line)
top (while (and (< *ptr* *lng*)
		(not (char= (setq c (char *line* *ptr*))
                            (char (cadr gram) 0))))
      (incf *ptr*))
    (if (char= c (char (cadr gram) 0))
        (progn (incf *ptr*)
               (setq i 0)
               (setq found t)
               (while (and found
                           (< (1+ i) (length (cadr gram)))
                           (< (+ i *ptr*) *lng*))
                 (if (char= (setq c (char *line* (+ i *ptr*)))
                            (char (cadr gram) (1+ i)))
                     (incf i)
                     (setq found nil)))
               (if (< (1+ i) (length (cadr gram))) (setq found nil))
               (if found (incf *ptr* (1- (length (cadr gram)))))))
    (if found
        (return 'zzomit)
        (progn
          (while (and *parsedfile* (not (> *lng* *ptr*)))
            (setq line (parseread *parsedfile*))
            (if (eq line 'zzeofvalue)
                (setq *parsedfile* nil)
              (initline line)))
          (if (< *ptr* *lng*)
              (go top)
              (return nil)))) ))

; 06 Jan 11
; skip to a specified line number
; grammar is (skiptoline n)
(defun parseskiptoline (gram)
  (let (n)
    (setq n (cadr gram))
    (while (and *parsedfile* (not (>= *linenumber* n)))
      (setq line (parseread *parsedfile*))
      (if (eq line 'zzeofvalue)
          (setq *parsedfile* nil)
          (initline line)))
    (and (not (eq line 'zzeofvalue)) 'zzomit) ))

(defun parsewhitespacep (ch)
  (or (char= ch #\Space) (not (graphic-char-p ch)) ) )

; skip white space
; grammar is (skipto "str")
(defun parseskipws (gram)
  (let ()
    (while (and (< *ptr* *lng*)
                (parsewhitespacep (char *line* *ptr*)) )
      (incf *ptr*))
    (if (< *ptr* *lng*) 'zzomit (parseg gram)) ))

; 09 Sep 02
; parse according to any of specified sub-grammars
; grammar is (or <grammar1> ... <grammarn>)
; result is first non-nil result
(defun parseor (gram)
  (let (res)
    (setq gram (cdr gram))
    (while (and gram (not res))
      (setq res (parseg (car gram)))
      (setq gram (cdr gram)) )
    res))

; 27 Sep 02
; parse optional item.  Returns 'zzomit if not found
(defun parseopt (gram)
  (or (parseg (cadr gram)) 'zzomit) ))

; 09 Sep 02
; parse delimited set of characters
; grammar is (delim "[" "]") specifying start and end chars
; result is a string of chars between delimiters
(defun parsedelim (gram)
  (let (start end c res)
    (setq start (char (cadr gram) 0))
    (if (cddr gram)
	(setq end (char (caddr gram) 0))
        (setq end start))
    (if (char= (safe-char) start)
	(progn (while (and (parse-not-end) (not (char= (setq c (safe-char)) end)))
		 (push c res)
		 (incf *ptr*))
	       (incf *ptr*)
	       (coerce (nreverse res) 'string))
        nil) ))

; 09 Sep 02; 17 Oct 06; 18 Oct 06; 19 Oct 06; 07 Oct 11
; parse delimited set of characters
; grammar is (stringto " ")                specifying end char
;            (stringto '(#\< #\Newline))    specifying end chars
; result is a string of chars not including end char, which is consumed
(defun parsestringto (gram)
  (let (end c res done)
    (setq res (third gram))
    (setq end (if (consp (cadr gram))
                  (cadr gram)
                  (list (if (characterp (cadr gram))
                            (cadr gram)
                            (char (cadr gram) 0)))))
    (while (and (parse-not-end)
                (setq c (safe-char))
                (not (setq done (member c end :test #'char=))))
      (push (if (char= c #\Return) #\Newline c) res)
      (incf *ptr*))
    (if (or done (not (parse-not-end)))
        (progn (incf *ptr*)
               (coerce (nreverse res) 'string))
        (parseg (list 'stringto end res))) ))

; 17 Oct 06; 18 Oct 06; 19 Oct 06
; parse delimited set of characters
; grammar is (stringupto " ")                specifying end char
;            (stringupto '(#\< #\Newline))    specifying end chars
; result is a string of chars not including end char, which is consumed
(defun parsestringupto (gram)
  (let (end c res done)
    (setq res (third gram))
    (setq end (if (consp (cadr gram))
                  (cadr gram)
                  (list (if (characterp (cadr gram))
                            (cadr gram)
                            (char (cadr gram) 0)))))
    (while (and (parse-not-end)
                (setq c (safe-char))
                (not (setq done (member c end :test #'char=))))
      (push (if (char= c #\Return) #\Newline c) res)
      (incf *ptr*))
    (if done
        (coerce (reverse res) 'string)
        (parseg (list 'stringupto end res))) ))

; 13 Sep 02; 02 Oct 02; 10 Oct 06
; skip any number of a specified character.  Result is omitted.
; grammar is (skip " ")
(defun parseskip (gram)
  (let (done c (skipped (char (cadr gram) 0)))
    (while (and (< *ptr* *lng*) (not done))
      (setq c (char *line* *ptr*))
      (if (char= c skipped)
	  (incf *ptr*)
	  (setq done t)) )
    (if *parsedfile* (parseg gram) 'zzomit) ))

; 22 Feb 12
; echo current position for debugging
(defun parseecho (gram)
  (let (n)
    (format t "*ptr* = ~A " *ptr*)
    (dotimes (i 20)
      (setq n (+ *ptr* (- i 19)))
      (if (>= n 0) (princ (char *line* n))))
    (format t " ^ ")
    (dotimes (i 20)
      (setq n (+ *ptr* i))
      (if (< n *lng*) (princ (char *line* n))))
    (terpri)
    'zzomit))

; 13 Sep 02
; apply a function to result of a grammar
; grammar is (fn <fn> <gram>)
(defun parsefn (gram)
  (let ((res (parseg (caddr gram))))
    (and res (funcall (cadr gram) res)) ))

; 13 Sep 02
; parse exactly the specified string.  Grammar is (string ".")
(defun parsestring (str)
  (let ((ptr 0) fail (n (length str)) (save *ptr*))
    (while (and (not fail) (< *ptr* *lng*) (< ptr n))
      (if (not (char= (char *line* *ptr*)
		      (char str ptr)))
	  (setq fail t)
	  (progn (incf *ptr*) (incf ptr)) ) )
    (if fail
	(progn (setq *ptr* save) nil)
        'zzomit)))

(defun safe-char ()
  (if (< *ptr* *lng*)
      (char *line* *ptr*)
      #\Space))

(defun safe-next-char ()
  (if (< (1+ *ptr*) *lng*)
      (char *line* (1+ *ptr*))
      #\Space))

(defun parse-not-end () (< *ptr* *lng*))

; 13 Sep 02; 24 Mar 04
; Parse a symbol of alphanumeric characters
; grammar is (sym <n>) where <n> is max columns, optional
(defun parsesym (gram)
  (let (c res (n 0))
    (and (or (alpha-char-p (safe-char)) (digit-char-p (safe-char)))
         (progn
           (while (and (< *ptr* *lng*)
                       (if (cadr gram) (< n (cadr gram)) t)
                       (or (alpha-char-p (setq c (char *line* *ptr*)))
                           (digit-char-p c)
                           (member c *symchars* :test #'char=)))
             (push c res)
             (incf n)
             (incf *ptr*))
           (if res
               (intern (string-upcase (coerce (nreverse res) 'string)))) ) ) ))

; 17 Oct 06; 24 Oct 06; 25 Oct 06
; Parse an XML symbol of alphanumeric characters and *symchars*
; grammar is (xmlsym)
(defun parsexmlsym (gram)
  (let ((str (parsealphanum gram)))
    (intern (string-upcase str)) ))

; 17 Oct 06; 25 Oct 06
(defun parsealphanum (gram)
  (let (c res (n 0))
    (and (or (alpha-char-p (safe-char)) (digit-char-p (safe-char)))
	 (progn
	   (while (and (< *ptr* *lng*)
		       (if (cadr gram) (< n (cadr gram)) t)
		       (or (alpha-char-p (setq c (char *line* *ptr*)))
			   (digit-char-p c)
			   (member c *xmlsymchars* :test #'char=)))
	     (push c res)
	     (incf n)
	     (incf *ptr*))
	   (if res (coerce (nreverse res) 'string))) ) ))

; 17 Oct 06
; Parse an XML attribute name
(defun parsexmlattr (gram)
  (let (c res (n 0))
    (and (setq c (safe-char))
         (or (alpha-char-p c) (digit-char-p c))
	 (progn
	   (while (and (< *ptr* *lng*)
		       (if (cadr gram) (< n (cadr gram)) t)
		       (or (alpha-char-p (setq c (char *line* *ptr*)))
			   (digit-char-p c)
			   (member c *symchars* :test #'char=)))
	     (push c res)
	     (incf n)
	     (incf *ptr*))
	   (if res (coerce (nreverse res) 'string))) ) ))

(defun stringify (x)
  (cond ((stringp x) x)
        ((symbolp x) (copy-seq (symbol-name x)))   ; avoids Sun bug
	(t (princ-to-string x))))

; Parse an integer
; grammar is (int <n>) where <n> is max columns, optional
(defun parseint (gram)
  (let (c (num 0) (n 0) digit found)
    (while (and (< *ptr* *lng*)
		(if (cadr gram) (< n (cadr gram)) t)
		(setq digit (digit-char-p
			     (setq c (char *line* *ptr*)))))
      (setq found (or found digit))
      (setq num (+ (* num 10) digit))
      (incf *ptr*)
      (incf n))
    (if found num nil) ))

; 10 Oct 06
; Parse a number string, returning the characters
; grammar is (nstring)
(defun parsenstring (gram)
  (let (c res (ok t))
    (if (or (digit-char-p (safe-char))
            (char= (safe-char) #\+)
            (char= (safe-char) #\-))
        (progn
          (while (and (< *ptr* *lng*)
                      (setq c (char *line* *ptr*))
                      (or ok (digit-char-p c)))
	     (push c res)
	     (setq ok nil)
	     (incf *ptr*))
          (if res (nreverse res)) ) ) ))

; 10 Oct 06
; Parse a number
; grammar is (num)
(defun parsenum (gram)
  (let (c res (n 0))
    (setq res (parsenstring '(nstring)))
    (if (char= (safe-char) #\.)
        (progn (setq res (append res (list #\.)))
               (incf *ptr*)
               (setq res (append res (parsenstring '(nstring))))
               (if (or (char= (safe-char) #\e) (char= (safe-char) #\E))
                   (progn (setq res (append res (list (safe-char))))
                          (incf *ptr*)
                          (setq res (append res (parsenstring '(nstring))))))))
    (if res (read-from-string (coerce res 'string))) ))

; 12 Oct 06; 17 Oct 06; 18 Oct 06; 19 Oct 06; 05 Jan 11
; Parse XML into Lisp lists (tag contents)
(defun parsexml (gram)
  (let (tag inttag parms)
    (parseg '(skipto "<"))
    (if *tracexml* (format t "xml   n = ~A  char = ~S~%" *ptr* (safe-char)))
    (if (and (< *ptr* *lng*)
             (not (alpha-char-p (char *line* *ptr*))))
        (progn (parseg '(skipto ">"))         ; skip non-named stuff for now
               (parseg gram))
        (progn
          (setq tag (parseg '(xmlattr)))
          (if (and (stringp tag) (> (length tag) 0))
              (progn
                (setq inttag (intern (string-upcase tag)))
                (parseg '(skipws))
                (if (alpha-char-p (safe-char))
                    (setq parms (parseg '(xmlparms))))
                (parseg '(skipws))
                (if (and (char= (safe-char) #\/)
                         (char= (safe-next-char) #\>))
                    (progn (incf *ptr* 2)
                           (cons inttag parms))
                    (progn
                      (if (char= (safe-char) #\>) (incf *ptr*))
                      (cons inttag (append parms (parseg '(xmltext))
                                                 (parsexmls tag))))))
              (progn (parseg '(skipto ">"))
                     (parseg gram)) ) )) ))

; 17 Oct 06; 19 Oct 06; 24 Oct 06; 05 Jan 11
; Parse a sequence of XML parameters tag="value" up to >
(defun parsexmlparms (gram)
  (let (tag inttag parms val endch)
    (parseg '(skipws))
    (if *tracexml* (format t "parms n = ~A  char = ~S~%" *ptr* (safe-char)))
    (if (< *ptr* *lng*)
        (if (alpha-char-p (char *line* *ptr*))
            (progn (setq tag (intern (string-upcase (parseg '(xmlattr)))))
                   (if (char= (safe-char) #\=)
                       (progn (incf *ptr*)
                              (if (or (char= (safe-char) #\")
                                      (char= (safe-char) #\'))
                                  (progn (setq endch (safe-char))
                                         (incf *ptr*)
                                         (setq val
                                               (parseg (list 'stringto endch)))
                                         (parsexmladdparms tag val
                                               (parseg '(xmlparms))))
                                  (if (alpha-char-p (safe-char))
                                      (parsexmladdparms tag
                                                  (parseg '(alphanum))
                                                  (parseg '(xmlparms))))))))
            (if (or (char= (safe-char) #\>)
                    (char= (safe-char) #\/))
                nil
                (parseg '(xmlparms))))
        (parseg '(xmlparms)) ) ))

; 05 Jan 11
; add (tag val) unless tag should be omitted
(defun parsexmladdparms (tag val rest)
  (if (member tag *xmlomits*)
      rest
      (cons (list tag val) rest) ) )

; 12 Oct 06; 17 Oct 06; 19 Oct 06; 24 Oct 06; 06 Jan 11
; Parse a sequence of XML items to closing tag
(defun parsexmls (tag)
  (let (res done str saveptr)
    (while (not done)
      (parseg '(skipws))
      (if *tracexml* (format t "xmls  n = ~A  char = ~S~%" *ptr* (safe-char)))
      (setq saveptr *ptr*)
      (if (and (< *ptr* *lng*)
               (char= #\< (char *line* *ptr*)))
          (if (and (< (1+ *ptr*) *lng*)
                   (char= #\/ (char *line* (1+ *ptr*))))
              (progn (incf *ptr*) (incf *ptr*)
                     (setq str (parseg '(stringto ">")))
                     (if (and (stringp str)
                              (string= str tag))
                         (setq done t)
                         (progn (setq done t)
                                (setq *ptr* saveptr))))
              (if (and (< (1+ *ptr*) *lng*)
                       (alpha-char-p (char *line* (1+ *ptr*))))
                  (push (parseg '(xml)) res)
                  (parseg '(skipto ">"))))
          (if (and (< *ptr* *lng*)
                   (char= #\/ (char *line* *ptr*))
                   (incf *ptr*)
                   (setq str (parseg '(stringto ">")))
                   (stringp str)
                   (string= str tag))
              (setq done t)
              (if (and (< *ptr* *lng*)
                       (member (char *line* *ptr*) '(#\& #\; #\:)
                               :test #'char=))
                  (skipamper)
                  (progn (setq done t) (parseg '(skipto "<")))))))
    (nreverse res)))

; 06 Jan 11
; skip ampersand stuff in html
(defun skipamper ()
  (while (and (< *ptr* *lng*)
              (or (alpha-char-p (char *line* *ptr*))
                  (member (char *line* *ptr*) '(#\& #\; #\:)
                          :test #'char=)))
    (incf *ptr*) ) )

; 17 Oct 06; 19 Oct 06
(defun parsexmltext (gram)
    (parseg '(skipws))
    (if *tracexml* (format t "text  n = ~A  char = ~S~%" *ptr* (safe-char)))
    (if (not (member (safe-char) '(#\< #\/ #\>)
                     :test #'char=))
        (list (list 'text (parseg '(stringupto "<")) ) ) ) )

(defun tracexml ()
  (setq *tracexml* t)
  (trace parsexml parsexmlparms parsexmls parsexmltext parsexmlattr))

(defun untracexml ()
  (setq *tracexml* nil)
  (untrace parsexml parsexmlparms parsexmls parsexmltext parsexmlattr))

; 13 Sep 02
; Convert a month symbol to month number
(defun monthconv (sym)
  (1+ (position sym '(jan feb mar apr may jun jul aug sep oct nov dec)
		:test #'(lambda (x y) (string-equal (symbol-name x)
						    (symbol-name y))))))

; 27 Sep 02
; destructively remove a character sequence, replacing with blanks
(defun remchar (string removestring)
  (let (found)
    (dotimes (i (length string))
      (if (char= (char string i) (char removestring 0))
	  (progn (setq found t)
		 (dotimes (j (length removestring))
		   (if (not (char= (char string (+ i j))
				   (char removestring j)))
		       (setq found nil)))
		 (if found
		     (dotimes (j (length removestring))
		       (setf (char string (+ i j)) #\Space)))) ) )
    string))

(defun googlefix (arg)
  (if (stringp arg)
      (remchar arg "+")
      nil))

(defvar *testlines*)
(setq *testlines*
 '(
"access:eglt407.directhit.com - - [11/Sep/2002:23:59:42 -0500] \"GET /users/novak/rmnp/c012.html HTTP/1.0\" 200 442 \"-\" \"Mozilla/2.0 (compatible; Ask Jeeves)\""
"access:infinity.ucsd.edu - - [12/Sep/2002:00:00:19 -0500] \"GET /users/novak/dellc610.html HTTP/1.1\" 200 5784 \"http://www.linux-on-laptops.com/dell.html\" \"Mozilla/4.0 (compatible; MSIE 5.21; Mac_PowerPC)\""
"access:phnxdslgw4poolb186.phnx.uswest.net - - [12/Sep/2002:00:03:55 -0500] \"GET /users/novak/dellc610.html HTTP/1.1\" 200 5784 \"http://search.msn.com/results.asp?q=dell+c610&origq=dvd+to+tv+dell+c610&RS=CHECKED&FORM=SMCRT&v=1&cfg=SMCINITIAL&nosp=0&thr=\" \"Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1)\""
"access:62.32.60.50 - - [12/Sep/2002:00:21:49 -0500] \"GET /users/novak/cs375contents.html HTTP/1.0\" 200 19125 \"http://www.google.com/search?hl=ar&ie=UTF-8&oe=UTF-8&q=data+structure%2Bstack%2Bpascal%2Bexample&btnG=%D8%A8%D8%AD%D8%AB+Google&lr=\" \"Mozilla/4.0 (compatible; MSIE 5.5; Windows 98; Win 9x 4.90)\""
"access:130.160.47.33 - - [12/Sep/2002:01:15:25 -0500] \"GET /users/novak/cs307exam2ans.html HTTP/1.1\" 200 10384 \"http://www.google.com/search?q=weight%2Bbalanced%2Bbinary%2Btree&hl=en&lr=&ie=UTF-8&oe=UTF-8&start=40&sa=N\" \"Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.0; .NET CLR 1.0.3705)\""
"access:rdu57-210-115.nc.rr.com - - [12/Sep/2002:01:17:14 -0500] \"GET /users/novak/talk97/img5.gif HTTP/1.1\" 304 - \"http://www.cs.utexas.edu/users/novak/talk97/\" \"Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; .NET CLR 1.0.3705)\""
"access:209.106.200.142 - - [12/Sep/2002:01:56:00 -0500] \"GET /users/novak/ijcai77.html HTTP/1.1\" 200 32394 \"http://www.google.com/search?hl=en&lr=&ie=UTF-8&oe=UTF-8&q=Solving+Physics+Equations&btnG=Google+Search\" \"Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1; YComp 5.0.0.0)\""
"access:pc080.ohsuga.info.waseda.ac.jp - - [12/Sep/2002:02:43:47 -0500] \"GET /users/novak/autop.html HTTP/1.1\" 200 2311 \"http://www.google.co.jp/search?q=%22automatic+programming%22&ie=Shift_JIS&hl=ja&lr=\" \"Mozilla/5.0 (compatible; Konqueror/3.0.0)\""
"access:1cust79.tnt1.echuca.au.da.uu.net - - [24/Sep/2002:23:57:51 -0500] \"GET /users/novak/cs375108.html HTTP/1.1\" 200 886 \"http://www.google.com/search?q=resolving+shift/reduce+conflicts&hl=en&lr=&ie=UTF-8&start=30&sa=N\" \"Mozilla/4.0 (compatible; MSIE 5.01; Windows NT 5.0)\""
"access:202.71.148.136 - - [25/Sep/2002:00:11:39 -0500] \"GET /users/novak/cs375.html HTTP/1.0\" 200 2777 \"http://www.google.com/search?hl=en&lr=&ie=ISO-8859-1&q=lecture+notes++and+questions+in+design+of+compilers&btnG=Google+Search\" \"Mozilla/4.0 (compatible; MSIE 5.01; Windows NT; YComp 5.0.0.0)\""
"access:student4856.student.nau.edu - - [25/Sep/2002:00:16:12 -0500] \"GET /users/novak/caia90.html HTTP/1.1\" 200 1587 \"http://www.google.com/search?hl=en&ie=UTF-8&oe=UTF-8&q=real+world+physics+problems\" \"Mozilla/4.0 (compatible; MSIE 6.0; Windows NT 5.1)\""
"access:scf4.wc.optusnet.com.au - - [25/Sep/2002:00:28:40 -0500] \"GET /users/novak/units95.html HTTP/1.0\" 200 64827 \"http://search.msn.com/results.asp?ba=(11.9)0.....&co=(0.10)200.2.4.10.3.&FORM=IE4&aq=si+units+conversion+charts&cfg=SMCSP&nosp=0&origq=conversion+charts&q=SI+units+conversion+charts&rd=0&thr&v=1&pn=2\" \"Mozilla/4.0 (compatible; MSIE 6.0; Windows 98; Win 9x 4.90; MSOCD; AtHomeEN191)\""
"access:65.209.67.243 - - [25/Sep/2002:16:42:49 -0500] \"GET /users/novak/units95.html HTTP/1.1\" 200 64827 \"http://www.askjeeves.com/main/metaAnswer.asp?t=ai&s=a&MetaEngine=directhit&en=te&eo=6&o=0&frames=True&url=http%3A%2F%2Fask%2Edirecthit%2Ecom%2Ffcgi%2Dbin%2Fredirurl%2Efcg%3Furl%3Dhttp%3A%2F%2Fwww%2Ecs%2Eutexas%2Eedu%2Fusers%2Fnovak%2Funits95%2Ehtml%26qry%3Dmoney%2Bconversion%2Bcharts%26rnk%3D8%26cz%3D349d4fef415c217c%26src%3DDH%5FASK%5FSRCH%26uid%3D051ab9d26ae6417b3%26sid%3D1892bde88dab229d3%26u%3D&ac=9&adcat=fin&pt=Conversion+of+Units+of+Measurement&dm=http%3A%2F%2Fwww%2Ecs%2Eutexas%2Eedu%2Fusers%2Fnovak%2Funits95%2Ehtml&io=7&qid=F34EDE314EACF34AB810DB4647A8D9D2&back=ask%3DMoney%2BConversion%2BCharts%26qsrc%3D6&ask=Money+Conversion+Charts&dt=020925144226&amt=\" \"Mozilla/4.0 (compatible; MSIE 5.01; Windows 95)\""
))

(defgram 'pathpart '(* (car (seq (or (sym) (int)) "."))))
(defgram 'path '(append pathpart (or (sym) (int))))
(defgram 'datime '(seq (int) "/" (fn monthconv (sym)) "/" (int) ":"
		       (int) ":" (int) ":" (int)))
(defgram 'directory '(* (car (seq (sym) "/"))))
(defgram 'filename '(seq (sym) "." (sym)))
(defgram 'access
  '(seq "access:" path (skipto "-") (skipto "-") (skip " ")
        "[" datime (skipto "]") (skipto "/") (or "users/" "~")
        directory filename
	(opt (fn googlefix (car (car (* googlesearch)))))))
(defgram 'googlesearch '(seq (skipto ":") "//www.google.com/search?q="
			     (stringto "&")))

(defun init (n) (initline (nth n *testlines*)))

; functions to analyze web logs

(defvar *nettimestamp* nil)
(defvar *nethtml* nil)
(defvar *netpdf* nil)
(defvar *netjpg* nil)
(defvar *netcgi* nil)
(defvar *neterrs* nil)
(defvar *netothers* nil)
(defvar *netignored* nil)
(defvar *netall* nil)
(defvar *nstats* 0)
(defvar *instats* nil)
(defvar *nstatdate* 0)
(defvar *ngif* 0)
(defvar *netlines* 0)

(defun netinit ()
  (setq *nettimestamp* (get-universal-time))
  (setq *nethtml* nil)
  (setq *netpdf* nil)
  (setq *netjpg* nil)
  (setq *netcgi* nil)
  (setq *neterrs* nil)
  (setq *netothers* nil)
  (setq *netignored* nil)
  (setq *ngif* 0)
  (setq *netlines* 0)
)

(defun symbol<  (x y)
  (and (symbolp x) (symbolp y) (string<  (symbol-name x) (symbol-name y))))

; 16 Sep 02; 30 Sep 02
(defun netsyminit (sym kind timestamp search)
  (setf (get sym timestamp) *nettimestamp*)
  (setf (get sym kind) 0)
  (setf (get sym search) nil) )

; 13 Sep 02; 27 Sep 02; 30 Sep 02; 02 Oct 02
; process a net access.  data is (who date directory (filename ext))
(defun netfn (data line)
  (let ((file (fourth data)) time)
   (incf *netlines*)
   (if (not (netignore data))
    (if data
	(case (cadr file)
	  ((html htm)
	    (setq time (get (car file) 'htmltimestamp))
	    (if (or (null time) (not (= time *nettimestamp*)))
		(progn (netsyminit (car file) 'html
				   'htmltimestamp 'htmlsearches)
		       (push (car file) *nethtml*)) )
	    (incf (get (car file) 'html))
	    (if (fifth data)
		(push (fifth data) (get (car file) 'htmlsearches))) )
	  ((ps pdf)
	    (setq time (get (car file) 'pdftimestamp))
	    (if (or (null time) (not (= time *nettimestamp*)))
		(progn (netsyminit (car file) 'pdf 'pdftimestamp 'pdfsearches)
		       (push (car file) *netpdf*)) )
	    (incf (get (car file) 'pdf))
	    (if (fifth data)
		(push (fifth data) (get (car file) 'pdfsearches))) )
	  (jpg
	    (setq time (get (car file) 'jpgtimestamp))
	    (if (or (null time) (not (= time *nettimestamp*)))
		(progn (netsyminit (car file) 'jpg 'jpgtimestamp 'jpgsearches)
		       (push (car file) *netjpg*)) )
	    (incf (get (car file) 'jpg))
	    (if (fifth data)
		(push (fifth data) (get (car file) 'jpgsearches))) )
	  (cgi
	    (setq time (get (car file) 'cgitimestamp))
	    (if (or (null time) (not (= time *nettimestamp*)))
		(progn (netsyminit (car file) 'cgi 'cgitimestamp 'cgisearches)
		       (push (car file) *netcgi*)) )
	    (incf (get (car file) 'cgi))
	    (if (fifth data)
		(push (fifth data) (get (car file) 'cgisearches))) )
	  (gif (incf *ngif*))     ; don't want to count gif's
	  (class)                 ; don't count Java classes
	  (t (push data *netothers*)))	  
        (push line *neterrs*) ) ) ))

; 26 Sep 02; 27 Sep 02; 02 Oct 02; 08 Oct 02; 24 Oct 02; 15 Jan 03
; what = nil (today), t (total), or a (average)
(defun netres (&optional what)
  (let (html jpg cgi pdf n (nhtml 0) (njpg 0) (njpga 0) (njpgb 0) (njpgc 0) 
	     (ncgi 0) (npdf 0)
	(npapers 0) (nclasses (nzeros (length *classes*)))
	str clstr nm nm4 desflag)
    (format t "Total lines processed = ~A~%" *netlines*)
    (if (eq what 'a) (format t "Averages over ~A days:~%" *nstats*))
    (if what
	(progn (setq html
		     (union
		       (subset #'(lambda (x)
				   (let ((z (get x 'totalhtml)))
				     (and z (> z 0))))
			       *netall*)
		       *nethtml*))
	       (setq jpg
		     (union
		       (subset #'(lambda (x)
				   (let ((z (get x 'totaljpg)))
				     (and z (> z 0))))
			       *netall*)
		       *netjpg*))
	       (setq cgi
		     (union
		       (subset #'(lambda (x)
				   (let ((z (get x 'totalcgi)))
				     (and z (> z 0))))
			       *netall*)
		       *netcgi*))
	       (setq pdf
		     (union
		       (subset #'(lambda (x)
				   (let ((z (get x 'totalpdf)))
				     (and z (> z 0))))
			       *netall*)
		       *netpdf*)) )
        (progn (setq html *nethtml*)
	       (setq pdf *netpdf*)
	       (setq jpg *netjpg*)
	       (setq cgi *netcgi*)) )
    (setq html (sort (copy-list html) #'symbol<))
    (setq pdf (sort (copy-list pdf) #'symbol<))
    (setq jpg (sort (copy-list jpg) #'symbol<))
    (setq cgi (sort (copy-list cgi) #'symbol<))
    (format t "~%Html:~%")
    (dolist (item html)
      (setq n (if what
		  (if (eq what 'a)
		      (* 1.0 (/ (get item 'totalhtml) *nstats*))
		      (get item 'totalhtml) )
		  (get item 'html)))
      (incf nhtml n)
      (if (member item *papers*) (incf npapers n))
      (setq str (symbol-name item))
      (dotimes (i (length *classes*))
	(setq clstr (symbol-name (nth i *classes*)))
	(if (and (>= (length str) (length clstr))
		 (string= str clstr :end1 (length clstr)))
	    (incf (nth i nclasses) n) ) )
      (princ item)
      (if (or (> n 1) (eq what 'a))
	  (if (floatp n)
	      (format t "~14T~6,2F" n)
	      (format t "~15T~3D" n)))
      (if (get item 'htmlsearches)
	  (dolist (x (get item 'htmlsearches)) (format t "~21T~A~%" x))
	  (terpri)) )
    (format t "Total papers = ~6,2F~%" npapers)
    (dotimes (i (length *classes*))
      (if (> (nth i nclasses) 0)
	  (format t "Total for class ~A = ~6,2F~%" (nth i *classes*)
		    (nth i nclasses)) ) )
    (format t "Total html = ~6,2F~%" nhtml)
    (format t "~%Pdf/PS:~%")
    (dolist (item pdf)
      (setq n (if what
		  (if (eq what 'a)
		      (* 1.0 (/ (get item 'totalpdf) *nstats*))
		      (get item 'totalpdf) )
		  (get item 'pdf)))
      (incf npdf n)
      (princ item)
      (if (or (> n 1) (eq what 'a))
	  (if (floatp n)
	      (format t "~14T~6,2F" n)
	      (format t "~15T~3D" n)))
      (if (get item 'pdfsearches)
	  (dolist (x (get item 'pdfsearches)) (format t "~21T~A~%" x))
	  (terpri)) )
    (format t "Total pdf/ps = ~6,2F~%" npdf)
    (format t "~%Jpg:~%")
    (dolist (item jpg)
      (setq n (if what
		  (if (eq what 'a)
		      (* 1.0 (/ (get item 'totaljpg) *nstats*))
		      (get item 'totaljpg) )
		  (get item 'jpg)))
      (incf njpg n)
      (setq desflag nil)
      (setq nm (symbol-name item))
      (if (>= (length nm) 4)
	  (setq nmb (intern (subseq nm 0 4)))
	  (setq nmb 'none))
      (if (and (= (length nm) 4) (digit-char-p (char nm 3)))
	  (progn (incf njpga n) (setq desflag t)))
      (if (and (= (length nm) 5) (digit-char-p (char nm 3))
	       (char= (char nm 4) #\B))
	  (progn (incf njpgb n) (if (> n 1) (setq desflag t))))
      (if (and (= (length nm) 5) (digit-char-p (char nm 3))
	       (char= (char nm 4) #\C))
	  (incf njpgc n))
      (when (not (and (= (length nm) 5) (digit-char-p (char nm 3))
		      (char= (char nm 4) #\C)))
	(princ item)
	(if (or (> n 1) (eq what 'a))
	    (if (floatp n)
		(format t "~14T~6,2F" n)
	      (format t "~15T~3D" n)))
	(if desflag (format t "~21T~A"
			    (or (get nmb 'description) "no description")))
	(if (get item 'jpgsearches)
	    (dolist (x (get item 'jpgsearches)) (format t "~21T~A~%" x))
	    (terpri)) ) )
    (If (> njpga 0) (format t "    jpg A = ~6,2F~%" njpga))
    (if (> njpgb 0) (format t "    jpg B = ~6,2F~%" njpgb))
    (if (> njpgc 0) (format t "    jpg C = ~6,2F~%" njpgc))
    (format t "Total jpg = ~6,2F~%" njpg)
    (format t "~%Cgi:~%")
    (dolist (item cgi)
      (setq n (if what
		  (if (eq what 'a)
		      (* 1.0 (/ (get item 'totalcgi) *nstats*))
		      (get item 'totalcgi) )
		  (get item 'cgi)))
      (incf ncgi n)
      (princ item)
      (if (or (> n 1) (eq what 'a))
	  (if (floatp n)
	      (format t "~14T~6,2F" n)
	      (format t "~15T~3D" n)))
      (if (get item 'cgisearches)
	  (dolist (x (get item 'cgisearches)) (format t "~21T~A~%" x))
	  (terpri)) )
    (format t "Total cgi = ~6,2F~%" ncgi)
    ))

(defun netignore (data)
  (let ((reader (first data)))
    (if (member reader *ignore* :test #'list=*)
	(progn (push data *netignored*) t) )  ))

(defun list=* (lst pattern)
  (or (and (null lst) (null pattern))
      (and (consp lst) (consp pattern)
	   (or (eq (car lst) (car pattern))
	       (eq (car pattern) '*))
	   (list=* (cdr lst) (cdr pattern)))))

; make a list of n 0's
(defun nzeros (n)
  (let (lst)
    (dotimes (i n lst) (push 0 lst)) ))

; 27 Sep 02; 30 Sep 02; 07 Oct 02; 29 Apr 03; 31 Oct 05
(defun netstats (&optional what reset (file (cstr *directory* "who.net")))
  (if reset
      (netreset)
      (load (cstr *directory* "net.stats") :if-does-not-exist nil))
  (if (not (equal *nstatdate* (dateint)))
      (progn (netinit)
	     (parselinesfn file 'access #'netfn)
	     (outstats)) )
  (if (fboundp 'get-time-string)
      (progn (print (get-time-string)) (terpri)))
  (netres what) )

; define statistics for a symbol
; input is (sym . (totals))
(defun defstats (inp)
  (setf (get (car inp) 'totals) (cdr inp)) )

; 30 Sep 02
; make one item for output to statistics file
(defun statline (sym)
  (list sym
	(get sym 'totalhtml)
	(get sym 'totalpdf)
	(get sym 'totaljpg)
	(get sym 'totalcgi)))

; 25 Sep 02; 30 Sep 02; 02 Oct 02; 24 Oct 02; 31 Oct 05
; output statistics
(defun outstats (&optional file)
  (let ()
    (netall)
    (when (< *nstatdate* (dateint))  ; update totals only if new date
	(dolist (item *netall*)
	  (setf (get item 'totalhtml)
		(+ (or (get item 'totalhtml) 0) (or (get item 'html) 0)))
	  (setf (get item 'totalpdf)
		(+ (or (get item 'totalpdf) 0) (or (get item 'pdf) 0)))
	  (setf (get item 'totaljpg)
		(+ (or (get item 'totaljpg) 0) (or (get item 'jpg) 0)))
	  (setf (get item 'totalcgi)
		(+ (or (get item 'totalcgi) 0) (or (get item 'cgi) 0))) )
	(incf *nstats*)
	(setq *nstatdate* (dateint)) )
    (with-open-file (outfile (or file (cstr *directory* "net.stats"))
			     :direction :output
			     :if-exists :supersede)
      (format outfile "(setq *nstats* ~A)~%" *nstats*)
      (format outfile "(setq *nstatdate* ~A)~%" (dateint))
      (format outfile "(instats (quote~%")
      (print (mapcar #'statline
		     (sort (copy-list *netall*) #'symbol<))
	     outfile)
      (format outfile "))~%")
      )
    ))

; compute list of all file names
(defun netall ()
  (setq *netall*
    (union *netall* (union (union *nethtml* *netpdf*)
			   (union  *netjpg* *netcgi*)))) )

; 24 Oct 02
; input statistics
(defun instats (lst)
  (dolist (item lst)
    (pushnew (car item) *netall*)
    (setf (get (car item) 'totalhtml) (second item))
    (setf (get (car item) 'totalpdf)  (third item))
    (setf (get (car item) 'totaljpg)  (fourth item))
    (setf (get (car item) 'totalcgi)  (fifth item)) ) )

; reset totals
(defun netreset ()
  (setq *nstats* 0)
  (netall)
  (dolist (item *netall*)
    (setf (get item 'totalhtml) 0)
    (setf (get item 'totalpdf) 0)
    (setf (get item 'totaljpg) 0)
    (setf (get item 'totalcgi) 0)) )

(defun get-time-list ()
  (let (second minute hour date month year day dst zone str)
    (multiple-value-setq (second minute hour date month year day dst zone)
			 (get-decoded-time))
    (list second minute hour date month year day dst zone str) ))

; get current date as an integer
(defun dateint ()
  (let ((timelist (get-time-list)))
    (+ (* (sixth timelist) 10000) (* (fifth timelist) 100)
       (fourth timelist)) ))

; -------------------------------------------------------------------
; functions to add info to html files for physics analysis

; Usage:

; (filterlines "physap.html" "physapnew.html" 'true
;              '((testhtmlcomment1 htmlproblem) (testhtmleof htmleof)))

(defvar *lastproblem* nil)
(defvar *lasttask* nil)
(defvar *tasksdef* nil)     ; tasks defined
(defvar *tasksref* nil)     ; tasks references
(defvar *nprobs* 0)         ; number of problems
(defvar *multch* 0)         ; multiple choice problems
(defvar *solved* 0)         ; problems marked 'solved'
(defvar *probstatus* nil)   ; solution status
(defvar *matched* 0)        ; problems matched with correct answer
(defvar *unmatched* nil)    ; problems marked 'solved' but not matched
(defvar *matcherrs* nil)    ; problems matched to wrong answer

(defmacro prob-diagram   (x) `(cadr ,x))     ; kind of diagram
(defmacro prob-diagused  (x) `(third ,x))    ; is diagram needed
(defmacro prob-status    (x) `(fourth ,x))   ; solution status
(defmacro prob-tasks     (x) `(fifth ,x))    ; list of tasks used
(defmacro prob-comments  (x) `(sixth ,x))    ; comments (strings)

; 24 Mar 04
; Test for special line containing desired symbol
(defun testhtmlcomment1 (line)
  (let ()
    (initline line)
    (and (> *lng* 12)
	 (string-equal line "<LI> <!-- " :end1 10)
	 (setq *ptr* 10)
	 (parseskip '(skip " ")) ) ))

; 24 Mar 04
; Test for special line containing desired symbol
(defun testhtmlitem1 (line)
  (let ()
    (initline line)
    (and (> *lng* 9)
	 (string-equal line "<LI> <B>" :end1 8)
	 (setq *ptr* 8)
	 (parseskip '(skip " ")) ) ))

; 25 Mar 04
; Test for special line containing desired symbol
(defun testhtmleof (line)
  (let ()
    (initline line)
    (and (> *lng* 11)
	 (string-equal line "<!-- eof -->" :end1 12)
	 (setq *ptr* 12) ) ))

; 24 Mar 04
; Process a problem in an html file
(defun htmlproblem (line res outfile)
  (let (probn)
    (setq probn (parsesym '(sym)))
    (if (and *lastproblem* (not (eq *lastproblem* 'eof)))
	(outproblem *lastproblem* outfile))
    (setq *lastproblem* probn) ))

; 25 Mar 04
; Process an eof in an html file
(defun htmleof (line res outfile)
  (let ()
    (if (and *lastproblem* (not (eq *lastproblem* 'eof)))
	(outproblem *lastproblem* outfile))
    (setq *lastproblem* nil)
    (if (and *lasttask* (not (eq *lasttask* 'eof)))
	(outtask *lasttask* outfile))
    (setq *lasttask* nil)

; 25 Mar 04
; Process a problem in an html file
(defun htmltask (line res outfile)
  (let (probn)
    (setq probn (parsesym '(sym)))
    (if (and *lasttask* (not (eq *lasttask* 'eof)))
	(outtask *lasttask* outfile))
    (setq *lasttask* probn) ))

 ))

; 24 Mar 04; 25 Mar 04; 27 Mar 04; 12 Apr 04; 13 Apr 04; 23 Apr 04; 03 Jun 04
; Output info about a problem to an html file
(defun outproblem (probn outfile)
  (let (data diag tasks status comments sents res codes answ programans)
    (when (setq data (assoc probn *probdata*))
      (incf *nprobs*)
      (if (eq (prob-status data) t) (push probn *solved*))
      (format outfile "<P>~%")
      (setq diag (prob-diagram data))
      (when (not (eq diag '-))
	(format outfile "<BR><B>Diagram:</B> ~A"
		(cadr (assoc diag '((c "Component") (m "Metric")
				    (n "Network")
				    (r "R-rated") (x "X-rated")))))
	(if (not (prob-diagused data))
	    (format outfile ", not necessary."))
	(terpri outfile))
      (format outfile "<BR><B>Tasks Used:</B> ")
      (setq tasks (prob-tasks data))
      (while tasks
	(outxref 'phystasks (car tasks) outfile)
	(pushnew (car tasks) *tasksref*)
	(setq tasks (cdr tasks))
	(if tasks
	    (format outfile ", &nbsp~%")
	    (terpri outfile)))
      (if (setq answ (assoc probn *probanswers*)) (push probn *multch*))
      (when (setq sents (assoc probn *probsents*))
	(format outfile "<BR><B>Sentences:</B>~%<PRE>~%")
	(dolist (sent (cdr sents))
	  (setq res (phys sent))
	  (format outfile "~A~%" sent)
	  (format outfile "~A~%" res))
	(format outfile "</PRE>~%")
	(if (and (eq (prob-status data) t) ; marked as solved
		 (eql (length (cdr sents)) 1)
		 answ)
	    (progn (setq programans (bestans res (caddr answ)))
		   (if (and (null programans)           ; an Oppenheimer!
			    (eq (caaddr answ) 'num)
			    (consp res) (numberp (car res)))
		       (setq programans
			     (bestans (cons (- (car res)) (cdr res))
				      (caddr answ))))
		   (if (eq programans (cadr answ))
		       (progn (format outfile
			         "<BR><B>Program's answer:</B> ~A (correct)~%"
				 programans)
			      (push probn *matched*))
		       (if programans (push probn *matcherrs*))))
	    (push probn *unmatched*)) )	    
      (when (setq codes (assoc probn *probcode*))
	(format outfile "<BR><B>Code:</B>~%<PRE>~%")
	(dolist (code (cdr codes))
	  (setq res (eval code))
	  (format outfile "~A~%" code)
	  (format outfile "~A~%" res))
	(format outfile "</PRE>~%"))
      (setq status (prob-status data))
      (when status
	  (push probn (cadr (assoc status *probstatus*)))
	  (format outfile "<BR><B>Status:</B> ~A~%"
		  (cadr (assoc status
			     '((t "<font color=\"#0000FF\">Solved.</font>")
			       (c "Covered.")
			       (g "<font color=\"#00A000\">Green.</font>")
			       (y "<font color=\"#FFFF00\">Yellow.</font>")
			       (r "<font color=\"#FF0000\">Red.</font>") )))))
      (setq comments (prob-comments data))
      (when comments
	(format outfile "<BR><B>Comments:</B>~%")
	(dolist (comment comments)
	  (write-line comment outfile)))
      (format outfile "<P><P>~%")
      ) ))

; 24 Mar 04
; Output a link
(defun outxref (filename name outfile)
  (format outfile "<A href=\"~A.html#~A\">~A</A>~%"
	  (string-downcase (symbol-name filename))
	  (string-downcase (symbol-name name))
	  (string-downcase (symbol-name name))))

; 25 Mar 04; 27 Mar 04
; Output info about a problem to an html file
(defun outtask (task outfile)
  (let (probs nm)
    (when (or (get task 'equations)
	      (get task 'approblems))
      (pushnew task *tasksdef*)
      (format outfile "<P>~%")
      (when (get task 'equations)
	(format outfile "<BR><B>Equations:</B>~%<PRE>~%")
	(dolist (eqn (get task 'equations))
	  (format outfile "~A~%" eqn))
	(format outfile "</PRE>~%"))
      (when (get task 'approblems)
	(format outfile "<BR><B>Used in Problems:</B>~%~%")
	(setf (get task 'approblems)
	      (sort (get task 'approblems)
		    #'(lambda (x y) (symbol< (car x) (car y)))))
	(setq probs (get task 'approblems))
	(while probs
	  (outxref (intern (subseq (symbol-name (caar probs)) 0
				   (- (length (symbol-name (caar probs))) 2)))
		   (caar probs) outfile)
	  (setq probs (cdr probs))
	  (if probs
	      (format outfile ", &nbsp~%")
	    (terpri outfile))) )
      (format outfile "<P><P>~%") ) ))


; 25 Mar 04
; Process and fix an AP file
; name is a string, without .html
(defun fixapfile (name)
  (let (infilename outfilename)
    (setq *lastproblem* nil)
    (setq infilename (concatenate 'string name "z.html"))
    (setq outfilename (concatenate 'string name ".html"))
    (filterlines infilename outfilename 'true
		 '((testhtmlcomment1 htmlproblem) (testhtmleof htmleof))) ))

; 25 Mar 04
; Process and fix a task file
; name is a string, without .html
(defun fixtaskfile (name)
  (let (infilename outfilename)
    (setq *lasttask* nil)
    (setq infilename (concatenate 'string name "z.html"))
    (setq outfilename (concatenate 'string name ".html"))
    (dolist (prob *probdata*)
       (dolist (task (prob-tasks prob))
	 (setf (get task 'approblems)
	       (cons prob (remove-if #'(lambda (x) (eq (car x) (car prob)))
				     (get task 'approblems))))))
    (filterlines infilename outfilename 'true
		 '((testhtmlitem1 htmltask) (testhtmleof htmleof))) ))

; 25 Mar 04; 27 Mar 04; 12 Apr 04; 13 Apr 04; 23 Apr 04; 03 Jun 04
; Process and fix all files
(defun fixallfiles ()
  (let (undef unref)
    (setq *tasksdef* nil)
    (setq *tasksref* nil)
    (setq *nprobs* 0)         ; number of problems
    (setq *multch* nil)       ; number of multiple choice problems
    (setq *solved* nil)       ; number of problems marked 'solved'
    (setq *probstatus* (copy-tree '((t nil) (c nil) (g nil) (y nil) (r nil))))
    (setq *matched* nil)      ; problems matched with correct answer
    (setq *unmatched* nil)    ; problems marked 'solved' but not matched
    (setq *matcherrs* nil)    ; problems matched to wrong answer
    (fixapfile "aura/phys/physap")
    (fixapfile "aura/phys/phys98b")
    (fixapfile "aura/phys/phys93b")
    (fixapfile "aura/phys/prin")
    (fixtaskfile "aura/phys/phystasks")
    (setq undef (set-difference *tasksref* *tasksdef*))
    (setq unref (set-difference *tasksdef* *tasksref*))
    (if undef (format t "Tasks not defined: ~A~%" undef))
    (if undef (format t "Tasks not referenced: ~A~%" unref))
    (format t "Total problems: ~A~%" *nprobs*)
    (format t "Multiple choice problems: ~A ~A~%" (length *multch*) *multch*)
    (format t "Problems marked 'solved': ~A ~A~%" (length *solved*) *solved*)
    (dolist (pair *probstatus*)
      (format t "Problem difficulty ~A: ~A ~A~%"
	        (cadr (assoc (car pair) '((t "Solved") (c "Covered")
					  (g "Green") (y "Yellow") (r "Red"))))
		(length (cadr pair)) (reverse (cadr pair))))
    (format t "Problems matched to correct answer: ~A ~A~%"
	      (length *matched*) (reverse *matched*))
    (format t "Problems not matched: ~A ~A~%" (length *unmatched*)
	                                      (reverse *unmatched*))
    (if *matcherrs*
	(format t "Problems matched to wrong answer: ~A~%"
		(reverse *matcherrs*)))
 ))
