; polygon.lsp               Gordon S. Novak Jr.        ; 03 Mar 04

; Copyright (c) 2007 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Generic functions for polygons

; 22 Nov 94; 05 Jan 95; 02 Jan 97; 09 Apr 98; 23 Apr 98; 10 May 99; 04 Jan 02
; 28 Feb 02; 03 Oct 03; 06 Oct 03; 07 Oct 03; 08 Jan 04

(gldefclusterc
  'polygon-cluster
  '((point      (polygon-point (transparent vector)))
    (collection (polygon-collection (listof polygon-point)
		   prop ((point-view ('prog1))
			 (area       polygon-area       specialize t)
		         (perimeter  polygon-perimeter  specialize t))
		   msg  ((contains   polygon-contains   specialize t)) ) ) )
  '())

(gldefclusterc
  'poly-cluster
  '((point        (poly-point (transparent consv)))
    (collection   (poly-collection (listof poly-point))) )
  '(polygon-cluster))

(gldefclusterc
  'poly-cluster-b
  '((point        (poly-point-b (transparent vector)))
    (collection   (poly-collection-b (transparent (arrayof poly-point-b)))) )
  '(polygon-cluster))

; 15 Oct 92
(gldefclusterc
  'poly-cluster-e
  '((point        (poly-point-e (z123 vector)
		    prop  ((x    ((x z123)))
			   (y    ((y z123))))
		    supers (vector)))
    (collection   (poly-collection-e (arrayof poly-point-e))) )
  '(polygon-cluster))

(glispobjects

(mypt (list (row integer) (col integer) (nxt (^ mypt)))
  views ((vector mypt-as-vector)
	 (polygon mypt-as-polygon) ))

(mypt-as-vector (z mypt)
  prop   ((y ((row z))) (x ((col z))) )
  supers (vector))

; 04 Jan 02
; definitions of points as used by the weather bureau,
; e.g. "40 miles SSE of Pflugerville"
(wbcity  (list (name symbol) (location utm-vector)))
(wbpoint (list (distance  (units integer miles))
	       (direction symbol)
	       (from      wbcity))
  prop  ((angle  ((position direction '(e ene ne nne n nnw nw wnw
					w wsw sw ssw s sse se ese))
		   * (pi / 8))
		 result (units real radians))
	 (deltax (distance * (cos angle)))
	 (deltay (distance * (sin angle))))
  views ((vector wbpoint-as-vector)))

(wbpoint-as-vector (z wbpoint)
  prop   ((x      ((east (location (from z))) + (deltax z)))
	  (y      ((north (location (from z))) + (deltay z))))
  supers (vector))


) ; glispobjects

(gldefclusterc
  'poly-cluster-c
  '((point        (poly-point-c mypt-as-vector))
    (collection   (mypt-as-polygon   mypt-as-ll-pointer
		    prop ((point-view      ('vector))) ) ) )
  '(polygon-cluster))

(gldefclusterc
  'poly-cluster-f
  '((point        (poly-point-f      wbpoint-as-vector))
    (collection   (poly-collection-f (listof wbpoint)
		    prop ((point-view      ('vector))) ) ) )
  '(polygon-cluster))

(gldefclusterc
  'poly-cluster-d
  '((point        (poly-point-d (transparent vector)))
    (collection   (poly-collection-d (listof poly-point-d))) )
  '(polygon-cluster))

; 13 Jan 92; 14 Jan 92; 17 Jan 92; 28 Sep 93; 10 May 99; 08 Jan 04
(gldefun polygon-area ((points (listof vector)))
  (result real)
  (let (area flg pt firstpt lastpt)
    (area = 0.0)
    (flg = t)         ; for first point only
    (for p in points do
      (pt = (funcall (point-view points) p))
      (if flg (progn (firstpt = pt) (lastpt = pt) (flg = nil))
              (progn (area _+ (area-under2 (virtual line-segment
					     with p1 = lastpt p2 = pt)))
	           (lastpt = pt) ) ))
    (area _+ (area-under2 (virtual line-segment with p1 = lastpt p2 = firstpt)))
    (/ (abs area) 2.0) ))

; 28 Sep 93; 08 Jan 04
(gldefun polygon-perimeter ((points (listof vector)))
  (result real)
  (let (perimeter flg pt firstpt lastpt)
    (perimeter = 0.0)
    (flg = t)         ; for first point only
    (for p in points do
      (pt = (funcall (point-view points) p))
      (if flg (progn (firstpt = pt) (lastpt = pt) (flg = nil))
              (progn (perimeter _+ (length (virtual line-segment
					       with p1 = lastpt p2 = pt)))
	           (lastpt = pt) ) ))
    (perimeter _+ (length (virtual line-segment with
				      p1 = lastpt p2 = firstpt)))
    perimeter))

; 13 Jan 92; 14 Jan 92; 17 Jan 92; 29 Sep 93; 08 Jan 04
(gldefun polygon-contains ((points (listof vector)) (pt vector))
  (result boolean)
  (let (inside flg ptv firstpt lastpt pview)
    (flg = t)         ; for first point only
    (for p in points do
      (ptv = (funcall (point-view points) p))
      (if flg
	  (progn (firstpt = ptv) (lastpt = ptv) (flg = nil))
          (progn (if (leftof-x (virtual line-segment with p1 = lastpt p2 = ptv)
			     pt)
		     (inside = (not inside)) )
	       (lastpt = ptv)) ))
    (if (leftof-x (virtual line-segment with p1 = lastpt p2 = firstpt)
		  pt)
	(inside = (not inside)) )
    inside))

(gldefun pta ((l poly-collection)) (area l))
; (pta '((1 . 1) (3 . 3) (5 . 1)))
(gldefun ptb ((l poly-collection-b)) (area l))
; (ptb (make-array 3 :initial-contents '((1 1) (3 3) (5 1))))
(gldefun ptc ((l poly-collection) (v consv)) (contains l v))
; (ptc '((1 . 1) (3 . 3) (5 . 1)) '(3 . 2))
(gldefun ptd ((q mypt)) (area (polygon q)))
; (viewas 'linked-list 'mypt)
; (ptd '(1 1 (3 3 (1 5 nil)))) ; = 4.0
(gldefun pte ((l poly-collection-c) (v poly-point-c)) (contains l v))
; (pte '(1 1 (3 3 (1 5 nil))) '(2 3 nil))
(gldefun ptf ((l poly-collection-b) (v vector)) (contains l v))
; (ptf (make-array 3 :initial-contents '((1 1) (3 3) (5 1))) '(3 2))
(gldefun ptg ((l poly-collection-d)) (area l))
; (ptg '((1 1) (3 3) (5 1)))
(gldefun pth ((l poly-collection-d) (v vector)) (contains l v))
; (pth '((1 1) (3 3) (5 1)) '(3 2))
(gldefun pti ((l poly-collection-e)) (area l))
; (pti (make-array 3 :initial-contents '((1 1) (3 3) (5 1))))
(gldefun ptj ((l poly-collection-f)) (area l))
(gldefun ptk ((l poly-collection-f)) (perimeter l))

(glispobjects

(polygon (listof nvector)
  prop   ((seqview   ('prog1))           ; defaults to given sequence
	  (pointview ('prog1))           ; defaults to item of sequence
          (pointseq  ((funcall (seqview self) self)))
	  (sides     polygon-sides       specialize t)
	  (vectortype (nil) result nvector) )
  msg    ((area      polygon-area-b      specialize t)
	  (perimeter polygon-perimeter-b specialize t)
          (contains  polygon-contains-b  specialize t)
          (cofg      polygon-cofg-b      specialize t)
	  (point     (glambda (self pt) (funcall (pointview self) pt)))
	  (order     polygon-order       specialize t)  ; file lookc.lsp
	  (draw      polygon-draw        specialize t)
	  (pointlist polygon-pointlist-b specialize t)
 ) )

(mypoly  (listof city)
  prop   ((pointview ('point)))
  supers (polygon))

(mypolyb (arrayof city)
  prop   ((pointview ('point)))
  supers (polygon))

(mypolyc (listof consv) supers (polygon))

(mypolyd (pts (listof consv))
  prop   ((seqview   ('evens))
	  (evens     ((for p in pts when (evenp (x p)) collect p))))
 supers (polygon))

(mypolye (list (link (^ mypolye))
	       (x integer)
	       (y integer)))

(mypolyf (list (next (^ mypolyf))
	       (r real)
	       (theta real)))

(mypolyg (listof vector) supers (polygon))

(mypolyh (arrayof vector) supers (polygon))

(mypolyi (arrayof cvector) supers (polygon))

(mypolyj (list (base vector)
	       (offsets (listof vector)))
  prop   ((seqview   ('offsets)))
  msg    ((point     (glambda (self pt) (base self) + pt)))
; (virtual vector with x = (x (base self)) + (x pt)
;		       y = (y (base self)) + (y pt))
; still materialized and worked badly for pointlist
  supers (polygon))

(character-as-point (z character)
  prop ((x char-keyboard-x)
	(y char-keyboard-y)) )

(stringpoly (s string)
  prop   ((seqview   ('chars))
	  (pointview ('point))
	  (chars     ((chars s))))
  supers (polygon))

(stringpolyb (s string)
  prop   ((pointview ('point))
	  (predicate ('consonantp))
	  (sequence  ((chars s))))
  supers (filter polygon))

(personnamepoly (p person)
  prop   ((seqview   ('g0017))
	  (pointview ('point))
	  (g0017     ((chars (name p)))))
  supers (polygon))

(boxpoly (b box)
  prop   ((seqview   ('g0018))
	  (pointview ('vector))
	  (g0018     ((linked-list b))) )
  supers (polygon) )

(box-as-vector (b box)
  prop   ((x ((size b)))
	  (y ((wavelength b)))))

(wbpoly (listof wbpoint)
  prop   ((pointview ('vector)))
  supers (polygon) )

(cirpoly (pt mypolye)
  prop ((seqview   ('g0019))
	(pointview ('vector))
	(g0019     ((circular-linked-list pt))))
  supers (polygon) )

(cirpolyf (pt mypolyf)
  prop ((seqview   ('g0020))
	(pointview ('vector))
	(g0020     ((circular-linked-list pt))))
  supers (polygon) )

  ) ; glispobjects


; 13 Jan 92; 14 Jan 92; 17 Jan 92; 29 Jan 92; 13 Oct 93; 22 Mar 94; 08 Nov 94
; 23 Apr 98; 10 May 99; 08 Jan 04
; what the code *should* look like:
; (for line in (edges poly) sum (area-under2 line))
(gldefun polygon-area-b ((poly polygon))
  (result real)
  (let ((area 0.0) (flg boolean) p firstpt lastpt)
    (flg = nil)
    (for item in (pointseq poly) do
      (p = item)
      (if (not flg)
	  (progn (firstpt = p)
	       (lastpt  = p)
	       (flg = t))
          (progn (area _+ (area-under2
			  (virtual line-segment with p1 = (point poly lastpt)
				   p2 = (point poly p))))
	       (lastpt = p)) ))
    (area _+ (area-under2 (virtual line-segment with p1 = (point poly lastpt)
				  p2 = (point poly firstpt))))
    (/ (abs area) 2.0) ))

; 13 Oct 93; 22 Mar 94; 17 Nov 94; 23 Apr 98; 08 Jan 04
; Desired code: (for line in (edges poly) sum (length line))
(gldefun polygon-perimeter-b ((poly polygon))
  (result real)
  (let ((perimeter 0.0) (flg boolean) p firstpt lastpt)
    (flg = nil)
    (for item in (pointseq poly) do
      (p = item)
      (if (not flg)
	  (progn (firstpt = p)
	       (lastpt  = p)
	       (flg = t))
          (progn (perimeter _+ (length
			  (virtual line-segment with p1 = (point poly lastpt)
				   p2 = (point poly p))))
	       (lastpt = p)) ))
    (perimeter _+ (length
		    (virtual line-segment with p1 = (point poly lastpt)
			     p2 = (point poly firstpt))))
    perimeter ))

; 13 Jan 92; 14 Jan 92; 17 Jan 92; 29 Jan 92; 13 Oct 93; 22 Mar 94; 24 Mar 94
; 17 Nov 94; 23 Apr 98; 28 Feb 02; 08 Jan 04
(gldefun polygon-contains-b ((poly polygon) (pt vector))
  (result boolean)
  (let ((inside boolean) (flg boolean) p itm firstpt lastpt)
    (flg = nil)
    (for item in (pointseq poly) do
      (itm = item)
      (p = item)
      (if (not flg)
	  (progn (firstpt = p)
	       (lastpt  = p)
	       (flg = t))
          (progn (if (leftof-x (virtual line-segment
					with p1 = (point poly lastpt)
				      p2 = (point poly p))
			       (if ((typeof pt) == (typeof itm))
				   (point poly pt)
			         pt))
		     (inside = (not inside)) )
	       (lastpt = p)) ))
    (if (leftof-x (virtual line-segment with p1 = (point poly lastpt)
			   p2 = (point poly firstpt))
		  (if ((typeof pt) == (typeof itm))
		      (point poly pt)
		      pt))
	(inside = (not inside)) )
    inside))

; 22 Oct 93; 22 Mar 94; 17 Nov 94; 23 Apr 98; 10 May 99; 07 Oct 03; 08 Jan 04
; Center of gravity of a polygon
; Note:  the result should be of the same type as original points!!!
;          (assuming this is possible)
(gldefun polygon-cofg-b ((poly polygon))
  (result nvector)
  (let ((area 0.0) (flg boolean) p firstpt lastpt (areax 0.0) (areay 0.0))
    (flg = nil)
    (for item in (pointseq poly) do
      (p = item)
      (if (not flg)
	  (progn (firstpt = p)
	       (lastpt  = p)
	       (flg = t))
          (progn (area _+ (area-under2
			  (virtual line-segment with p1 = (point poly lastpt)
				   p2 = (point poly p))))
	       (areax _+ (cofg-area-x
			   (virtual line-segment with p1 = (point poly lastpt)
				    p2 = (point poly p))))
	       (areay _+ (cofg-area-y
			   (virtual line-segment with p1 = (point poly lastpt)
				    p2 = (point poly p))))
	       (lastpt = p)) ))
    (area _+ (area-under2 (virtual line-segment with p1 = (point poly lastpt)
				  p2 = (point poly firstpt))))
    (areax _+ (cofg-area-x (virtual line-segment
				    with p1 = (point poly lastpt)
				    p2 = (point poly firstpt))))
    (areay _+ (cofg-area-y (virtual line-segment
				    with p1 = (point poly lastpt)
				    p2 = (point poly firstpt))))
    (a (typeof (vectortype poly))
       with x = (2.0 * areax / area)  y = (2.0 * areay / area) ) ))

; 03 Nov 94; 08 Nov 94; 17 Nov 94; 08 Jan 04
; Draw a polygon on a substrate
(gldefun polygon-draw ((poly polygon) (w window)
		       &optional (xoff number) (yoff number) (scale number))
  (let ((flg boolean) p firstpt lastpt)
    (for item in (pointseq poly) do
      (p = (point poly item))
      (if (not flg)
	  (progn (firstpt = p)
	       (lastpt  = p)
	       (flg = t))
          (progn (draw (virtual line-segment with
		       p1 = (virtual vector with x (xoff + (x lastpt) * scale)
				                 y (yoff + (y lastpt) * scale))
		       p2 = (virtual vector with x (xoff + (x p) * scale)
			                         y (yoff + (y p) * scale)))
		     w)
	       (lastpt = p)) ))
      (draw (virtual line-segment with
		     p1 = (virtual vector with x (xoff + (x lastpt) * scale)
				               y (yoff + (y lastpt) * scale))
		     p2 = (virtual vector with x (xoff + (x firstpt) * scale)
			                       y (yoff + (y firstpt) * scale)))
		     w) ))

; 08 Nov 94; 17 Nov 94
(gldefun polygon-pointlist-b ((poly polygon))
  (result (listof nvector))
  (for item in (pointseq poly) collect
    (virtual nvector with x = (x (point poly item))
	                  y = (y (point poly item)))  ) )    ; causes problems!

; must also load cvhe.lsp for these examples
(gldefun gf ((c mypoly))  (area c))
(gldefun gg ((c mypolyb)) (area c))
(gldefun gj ((c mypoly))  (perimeter c))
(gldefun gk ((c mypolyb)) (perimeter c))
(gldefun gh ((c mypoly)  (d city)) (contains c d))
; (gh myptsc (third myptsc))               ; = T   (Waco)
; (gh myptsc '(fooville 31.55 50 -97.5))   ; = NIL (32.5 miles west of Waco)
(gldefun gi ((c mypolyb) (d city)) (contains c d))
(gldefun gm ((c mypolyc))  (area c))
(gldefun gn ((c mypolyc))  (perimeter c))
(gldefun gpz ((c mypolyc) (d consv)) (contains c d))

; 08 Jan 04
(gldefun char-keyboard-x ((c character))
  (let (pos)
    (if (setq pos (position c '(#\z #\x #\c #\v #\b #\n #\m)))
	(pos + 0.75)
        (if (setq pos (position c '(#\a #\s #\d #\f #\g #\h #\j #\k #\l)))
	    (pos + 0.25)
	    (if (setq pos (position c
				    '(#\q #\w #\e #\r #\t #\y #\u #\i #\o #\p)))
		pos
	        0.0) ))))

; 08 Jan 04
(gldefun char-keyboard-y ((c character))
  (if (member c '(#\z #\x #\c #\v #\b #\n #\m))
      0.0
      (if (member c '(#\a #\s #\d #\f #\g #\h #\j #\k #\l))
	  1.0
          (if (member c '(#\q #\w #\e #\r #\t #\y #\u #\i #\o #\p))
	      2.0
	      0.0))))

(gldefun consonantp ((c character))
  (member c '(#\b #\c #\d #\f #\g #\h #\j #\k #\l #\m #\n
	      #\p #\q #\r #\s #\t #\v #\w #\x #\z)))

; (gladdprop 'character 'views '(point character-as-point)) ; do this first
(gldefun gq ((c stringpoly))  (area c))
(gldefun gr ((c stringpoly))  (perimeter c))
(gldefun gs ((c stringpoly) (d character)) (contains c d))
; (gq "fred")     ; = 1.0
; (gr "fred")     ; = 4.06
; (gs "bart" #\g) ; = T

(gldefun gt ((c personnamepoly))  (area c))
(gldefun gu ((c personnamepoly))  (perimeter c))
(gldefun gv ((c personnamepoly) (d character)) (contains c d))

; (gladdprop 'box 'views '(vector box-as-vector)) ; do this first
(gldefun gw ((c boxpoly))  (area c))
(gldefun gx ((c boxpoly))  (perimeter c))
(gldefun gy ((c boxpoly) (d box)) (contains c d))

(gldefun ha ((c wbpoly))  (area c))
(gldefun hb ((c wbpoly))  (perimeter c))
(gldefun hc ((c wbpoly) (d wbpoint)) (contains c d))

(glispobjects

(chain-polygon (listobject (start  vector)
			   (deltas (listof vector)))
  prop   ((initial-state   (start))
	  (sequence        (deltas)))
  supers (chain-code polygon)  )

(chain-rth-polygon (listobject (start  rvector)
			       (deltas (listof rthvector)))
  prop   ((initial-state   (start))
	  (sequence        (deltas))
	  (delta-view      ('rvector)))
  supers (chain-code polygon)  )

(chain-rth-c-polygon (crecord ccrth (start  cvector)
			            (deltas polyb))
  prop   ((initial-state   (start))
	  (sequence        (deltas))
	  (delta-view      ('rvector)))
  supers (chain-code polygon)  )

(chars-polygon (listobject (start vector)
			   (str   string))
  prop   ((initial-state   (start))
	  (sequence        ((chars str)))
	  (delta-view      ('char-delta-view)))
  supers (chain-code polygon)  )

 ) ; glispobjects

(setq myccpoly (a chain-polygon with start = '(10 10)
		  deltas = '((0 4) (3 0) (0 -4)) ))

(gldefun hd ((p chain-polygon)) (area p))
(gldefun he ((p chain-polygon)) (perimeter p))
(gldefun hf ((p chain-polygon) (pt consv)) (contains p pt))

(setq myccrthpoly (a chain-rth-polygon with start = '(10 10)
                       deltas = (list (list 4.0 (/ pi 2))
				      (list 3.0 0.0)
				      (list 4.0 (- (/ pi 2.0))))))

(setq myccrthpolyb (a chain-rth-polygon with start = '(10 10)
                       deltas = (list (list 4.0 (/ pi 4))
				      (list 3.0 0.0)
				      (list 4.0 (- (/ pi 4))))))

; (mkv 'rvector 'rthvector)
(gldefun hdb ((p chain-rth-polygon)) (area p))
(gldefun heb ((p chain-rth-polygon)) (perimeter p))
(gldefun hfb ((p chain-rth-polygon) (pt consv)) (contains p pt))
(gldefun hgb ((p chain-rth-polygon)) (cofg p))
(gldefun hhb ((p chain-rth-polygon)) (pointlist p))

; (hdb myccrthpolyb) ; = 16.485281

(gldefun hdc ((p chain-rth-c-polygon)) (area p))
(gldefun hec ((p chain-rth-c-polygon)) (perimeter p))
(gldefun hfc ((p chain-rth-c-polygon) (pt consv)) (contains p pt))
(gldefun hgc ((p chain-rth-c-polygon)) (cofg p))     ;  ??? not working
(gldefun hhc ((p chain-rth-c-polygon)) (pointlist p))

(gldefun char-delta-view (ch)
  (result vector)
  (case ch of (#\N '(0 1)) (#\S '(0 -1)) (#\E '(1 0)) (#\W '(-1 0))
    (t '(0 0))))

(gldefun hg ((p chars-polygon)) (area p))
(gldefun hh ((p chars-polygon)) (perimeter p))
(gldefun hi ((p chars-polygon) (pt consv)) (contains p pt))

(setq mychpoly (a chars-polygon with start = '(10 10)
		  str = "NNENEENNESSSSSWWWW"))
; (hg mychpoly)            ; = 13.0
; (hh mychpoly)            ; = 18.0
; (hi mychpoly '(12 . 12)) ; = T

(gldefun hj ((p mypolyd)) (area p))

(gldefun hk ((c stringpolyb))  (area c))
; (hk "carla") ; = 5.375

(gldefun makecirpoly ()   ; make a circular linked-list polygon for testing
  (let (p plast)
    (p = (a mypolye x 1 y 1 link (a mypolye x 3 y 3 link
				      (setq plast (a mypolye x 5 y 1)))))
    ((link plast) = p)
    (mycirpoly = p)
    nil))

(gldefun makecirpolyf ()   ; make a circular linked-list polygon for testing
  (let (p plast p1 p2 p3)
    (setq p1 (torth 1 1))
    (setq p2 (torth 3 3))
    (setq p3 (torth 5 1))
    (p = (a mypolyf r (car p1) theta (cadr p1)
	      next (a mypolyf r (car p2) theta (cadr p2)
		     next (setq plast (a mypolyf r (car p3) theta (cadr p3))))))
    ((next plast) = p)
    (mycirpolyf = p)
    nil))

(gldefun hl ((p cirpoly)) (area p))
; (viewas 'circular-linked-list 'mypolye)
; (mkv 'vector 'mypolye)
; (makecirpoly)
; (hl mycirpoly)  ;  = 4.0

(gldefun hm ((p cirpolyf)) (area p))
(gldefun hn ((p cirpolyf)) (perimeter p))
; (viewas 'circular-linked-list 'mypolyf)
; (mkv 'vector 'mypolyf)
; (makecirpolyf)
; (hm mycirpolyf)  ;  = 3.9999999999999973
; (hn mycirpolyf)  ;  = 9.6568542494923779

(glispobjects

(poly (crecord poly              ; polygon record
		(x     integer)
		(y     integer)
		(next  (^ poly)) )
  supers (vector))

(polyv   (p poly)
  prop   ((seqview   ('g0021))
	  (g0021     ((linked-list p))) )
  supers (polygon) )

(polyb (crecord polyb              ; polygon record
		(r     real)
		(theta real)
		(next  (^ polyb)) ) )

(polybv   (p polyb)
  prop   ((seqview   ('g0022))
	  (pointview ('vector))
	  (g0022     ((linked-list p))) )
  supers (polygon) )

(polybc   (p polyb)
  prop   ((seqview   ('g0023))
	  (pointview ('vector))
	  (g0023     ((circular-linked-list p))) )
  supers (polygon) )

 ) ; glispobjects

(gldefun hp ((p polyv)) (area p))
; (viewas 'linked-list 'poly)
; (glcp 'hp)
; (gltoc 'polygon-area-b-1)

(gldefun hq ((p polybv)) (area p))
; (viewas 'linked-list 'polyb)
; (mkv 'vector 'polyb)
; (glcp 'hq)
; (gltolang 'polygon-area-b0 'c)
(gldefun hr ((p polybv) (q polyb)) (contains p q))
(gldefun hs ((p polybv) (q vector)) (contains p q))

(gldefun hqc ((p polybc)) (area p))
; (viewas 'circular-linked-list 'polyb)

; polygon as list of vectors
(setq polya '((6 3) (6 15) (18 8)))
(setq polyb '((5 5) (5 10) (16 10) (16 5)))
(setq polyc '((5 5) (5 10) (10 17) (13 17) (16 12) (16 5) (10 3)))

(gldefun ht ((p mypolyg)) (area p))
(gldefun htb ((p mypolyg)) (perimeter p))
(gldefun htc ((p mypolyg)) (cofg p))
(gldefun hu ((p mypolyg) (v vector)) (contains p v))

(gldefun hv ((p mypolyh)) (area p))
(gldefun hw ((p mypolyh) (v vector)) (contains p v))

(setq polyca (make-array 7 :initial-contents
		  '((5 5) (5 10) (10 17) (13 17) (16 12) (16 5) (10 3))))
; (hv polyca)  ; = 118.0

(gldefun hx ((p mypolyi)) (area p))                    ; ??? problems
(gldefun hy ((p mypolyi) (v vector)) (contains p v))

; 06 Oct 03; 08 Jan 04
; derived from polygon-area-b
; Desired code: (for line in (edges poly) sum (area-under2 line))
(gldefun poly-area ((poly anything))
  (result real)
  (let ((area 0.0) (flg boolean) firstpt lastpt)
    (flg = nil)
    (for item in (seq poly) do
      (if (not flg)
	  (progn (firstpt = item)
	       (lastpt  = item)
	       (flg = t))
          (progn (area _+ (area-under2
			  (virtual line-segment with
			    p1 = (vecview (cast lastpt (dataview poly)))
			    p2 = (vecview (cast item (dataview poly))) )))
	       (lastpt = item)) ))
    (area _+ (area-under2 (virtual line-segment with
			    p1 = (vecview (cast lastpt (dataview poly)))
			    p2 = (vecview (cast firstpt (dataview poly))) )))
    (/ (abs area) 2.0) ))

; 06 Oct 03; 08 Jan 04
; Desired code: (for line in (edges poly) sum (length line))
(gldefun poly-perimeter ((poly polygon))
  (result real)
  (let ((perimeter 0.0) (flg boolean) firstpt lastpt)
    (flg = nil)
    (for item in (seq poly) do
      (if (not flg)
	  (progn (firstpt = item)
	       (lastpt  = item)
	       (flg = t))
          (progn (perimeter _+ (length
			  (virtual line-segment with
			    p1 = (vecview (cast lastpt (dataview poly)))
			    p2 = (vecview (cast item (dataview poly))) )))
	       (lastpt = item)) ))
    (perimeter _+ (length
		    (virtual line-segment with
			    p1 = (vecview (cast lastpt (dataview poly)))
			    p2 = (vecview (cast firstpt (dataview poly))) )))
    perimeter ))

; 06 Oct 03; 07 Oct 03; 08 Jan 04; 03 Mar 04
(gldefun poly-contains ((poly polygon) (pt vector))
  (result boolean)
  (let ((inside boolean) (flg boolean) itm firstpt lastpt)
    (flg = nil)
    (inside = nil)
    (for item in (seq poly) do
      (itm = item)
      (if (not flg)
	  (progn (firstpt = item)
	       (lastpt  = item)
	       (flg = t))
          (progn (if (leftof-x (virtual line-segment with
			       p1 = (vecview (cast lastpt (dataview poly)))
			       p2 = (vecview (cast item (dataview poly))) )
			     (vecview (cast pt (dataview poly))) )
		     (inside = (not inside)) )
	       (lastpt = item)) ))
    (if (leftof-x (virtual line-segment with
			   p1 = (vecview (cast lastpt (dataview poly)))
			   p2 = (vecview (cast firstpt (dataview poly))) )
		  (vecview (cast pt (dataview poly))) )
	(inside = (not inside)) )
    inside))

; 06 Oct 03; 07 Oct 03; 08 Jan 04
; Center of gravity of a polygon
; Note:  the result should be of the same type as original points!!!
;          (assuming this is possible)
(gldefun poly-cofg ((poly polygon))
  (result nvector)
  (let ((area 0.0) (flg boolean) firstpt lastpt (areax 0.0) (areay 0.0))
    (flg = nil)
    (for item in (seq poly) do
      (if (not flg)
	  (progn (firstpt = item)
	       (lastpt  = item)
	       (flg = t))
          (progn (area _+ (area-under2
			  (virtual line-segment with
			     p1 = (vecview (cast lastpt (dataview poly)))
			     p2 = (vecview (cast item (dataview poly))) )))
	       (areax _+ (cofg-area-x
			   (virtual line-segment with
			     p1 = (vecview (cast lastpt (dataview poly)))
			     p2 = (vecview (cast item (dataview poly))) )))
	       (areay _+ (cofg-area-y
			   (virtual line-segment with 
			     p1 = (vecview (cast lastpt (dataview poly)))
			     p2 = (vecview (cast item (dataview poly))) )))
	       (lastpt = item)) ))
    (area _+ (area-under2 (virtual line-segment with
			     p1 = (vecview (cast lastpt (dataview poly)))
			     p2 = (vecview (cast firstpt (dataview poly))) )))
    (areax _+ (cofg-area-x (virtual line-segment
			     p1 = (vecview (cast lastpt (dataview poly)))
			     p2 = (vecview (cast firstpt (dataview poly))) )))
    (areay _+ (cofg-area-y (virtual line-segment
			     p1 = (vecview (cast lastpt (dataview poly)))
			     p2 = (vecview (cast firstpt (dataview poly))) )))
    (a (typeof (vector lastpt))
       with x = (2.0 * areax / area)  y = (2.0 * areay / area) ) ))
