; viewas.lsp                 Gordon S. Novak Jr.              ; 10 Oct 11

; Copyright (c) 2011 Gordon S. Novak Jr. and The University of Texas at Austin.
; All rights reserved.

; Functions to make a view of a user type as an abstract type.

; 10 Jan 97; 14 Mar 97; 25 Mar 97; 03 Dec 98; 03 Jan 00; 31 Oct 00; 31 Jul 01
; 24 Jan 02; 31 Jan 02; 28 May 02; 10 Sep 02; 13 Mar 03; 13 Aug 03; 14 Aug 03
; 26 Nov 03; 05 Jan 04; 09 Nov 04; 14 Nov 06; 15 Feb 07; 27 May 09; 05 Apr 10
; 16 Jul 10

(defvar *glviewas-tty*     nil)
(defvar *glviewas-debug*   nil)
(defvar *glviewas-parms*   nil)
(defvar *glviewas-choices* nil)
(defvar *glviewas-echo*    nil)

; 18 Jul 90; 09 Aug 90; 10 Aug 90; 21 Nov 90; 03 Dec 90; 27 Sep 93; 28 Sep 93
; 22 Dec 95; 10 Sep 02
; Functions to view a set of types as a type cluster,
; e.g., view a user type as a linked list: (glviewas 'linked-list 'box)
; goal-cluster may be a specific goal cluster, e.g. front-pointer-queue,
; or the name of a class of goal clusters, e.g., queue.
; An existing view that matches will be returned unless redoflg is t.
(defun glviewas (goal-cluster source-types &optional redoflg viewname echo)
  (let (view-spec tmp clus)
    (setq *glviewas-echo* echo)
    (setq source-types (listify source-types))
    (if (and (not redoflg)
	     (null (cdr source-types))       ; see if existing view matches
	     (some #'(lambda (x)
		       (if (if viewname
			       (eq viewname (car x))
			       (eq goal-cluster (car x)))
			   (setq clus (glcluster (cadr x)))))
		   (glviews (first source-types))))
	clus
	(if (or (setq view-spec (glclusterviewspec goal-cluster))
		(and (setq tmp (glclusterviews goal-cluster))
		     (setq goal-cluster
			   (glvmenu "Specify type of ~A for ~A"
				    (list goal-cluster source-types) tmp nil))
		     (setq view-spec (glclusterviewspec goal-cluster))))
	    (glviewasb goal-cluster view-spec source-types redoflg viewname)
            (progn (format t "~A ?~%" goal-cluster) nil)) ) ))

; 15 Dec 93
(defun viewas (goal-cluster source-types &optional redoflg viewname)
  (glviewas goal-cluster source-types redoflg viewname t) )

; 24 May 90; 25 May 90; 30 May 90; 06 June 90; 08 June 90; 09 Jul 90;
; 12 Jul 90; 13 Jul 90; 17 Jul 90; 18 Jul 90; 20 Jul 90; 07 Aug 90; 09 Aug 90;
; 10 Aug 90; 16 Nov 90; 19 Nov 90; 03 Dec 90; 04 Dec 90; 04 Feb 91; 05 Feb 91;
; 12 Feb 91; 26 Dec 91; 28 Sep 93; 30 Sep 93; 01 Oct 93; 05 Oct 93; 18 Nov 93
; 15 Dec 93; 28 Feb 94; 18 Oct 94; 11 Aug 95; 22 Dec 95; 06 Feb 96; 15 Aug 96
; 25 Mar 97
(defun glviewasb (goal-cluster view-spec source-types redoflg viewname)
  (prog (typesubs parms subparms cluster-spec cluster-supers tmpb restp
	 viewstoragesubs viewtypesubs tmp propsubs part parts cltypespecs
	 errflg clus clustersubs view-name str recnames subpname subpspec
	 partsb zaps choices choice new-choices)
    (declare (special recnames))
    (setq parms          (third view-spec))      ; parameter specs
    (setq subparms       (fourth view-spec))     ; derived from parameters
    (setq cluster-spec   (fifth view-spec))      ; type specs of new cluster
    (setq cluster-supers (sixth view-spec))      ; super-clusters
    (unless redoflg
      (setq choices  ; choices made previously in making this view
	    (glgetviewspecs (first source-types) viewname goal-cluster)))
; Make a name for the view
    (setq view-name
	  (glviewname source-types
		      (or viewname (first view-spec) goal-cluster) ; abbrev
		      (second view-spec)))     ; t if "pure" view
                                               ; i.e. data str is unchanged

; Make names for the new types to be created as part of the view,
; put on viewtypesubs, e.g. (RECORD . BOX-AS-LL-RECORD)
    (dolist (cltypespec cluster-spec)
      (push (cons (car cltypespec)
		  (glhyphenate (symbol-name view-name) (car cltypespec)))
	    viewtypesubs))

; Match parameters with source-types.
; typesubs        = ((internal type name    . actual argument type name) ...)
; viewstoragesubs = ((internal type name    . stored type used in cluster) ...)
; clustersubs     = ((internal cluster name . actual cluster) ...)
    (mapc
      #'(lambda (parm source-type)
	  (setq source-type (viewas-expand-listof source-type))
	  (cond  ; match source type against a type specification
	        ((gltypematch source-type (sublis typesubs (cadr parm)))
	          (push (cons (car parm) source-type) typesubs)
		  (push (cons (car parm) source-type) viewstoragesubs))
                 ; match source type from a cluster against a cluster spec
		((and (symbolp (cadr parm))
		      (glclusterp (cadr parm))
		      (setq clus (glcluster source-type))
		      (glsubclusterp clus (cadr parm)))
		  (push (cons (car parm) clus) clustersubs))
                 ; try to make a cluster view automatically to match the spec
		((and (symbolp (cadr parm))
		      (or (glclusterp (cadr parm))
			  (glclusterviews (cadr parm)))
		      (setq clus (glviewas (cadr parm) source-type redoflg)))
		  (push (cons (car parm) clus) clustersubs))
		(t (format t "Parameter ~A doesn't match desired type ~A~%"
			   source-type parm)
		   (setq errflg t)) ) )
	  parms source-types)
    (if errflg (return))

; Try to identify sub-parameters, which typically are derived from the
; primary parameters.
    (dolist (subparm subparms)
      (setq subpname (cadr subparm))
      (setq subpspec (caddr subparm))
      (if *glviewas-debug* (format t "viewas: parameter ~a  ~A~%"
				   subpname subpspec))
      (setq choice (cdr (assoc subpname (cddr choices))))
					; previously specified choice sequence
      (setq *glviewas-choices* nil)
      (case (car subparm)
;         type spec might be (^ rec) or (role cluster)
	(type (when (setq tmp (or (glvfindcltype subpspec clustersubs)
				  (glvfindtype (sublis typesubs subpspec))))
		    (push (cons subpname tmp) typesubs)
		    (push (cons subpname (if (symbolp tmp)
					     tmp
					     (sublis viewtypesubs subpspec)))
			  viewstoragesubs) ) )
	(cluster (if (setq clus (glviewas (car subpspec)
					  (sublis (nconc (glclustertypesubs
							  clustersubs)
							 typesubs)
						  (cdr subpspec))
					  redoflg))
		     (push (cons subpname clus) clustersubs)))
	(prop (case (car subpspec)
		(partof
		  (setq tmp (sublis typesubs subpspec))
		  (if (setq part (glvfindpart (cadr tmp) (caddr tmp)
					      subpname choice
					      (glvzaps (cadddr subpspec)
						       propsubs)))
		      (push (cons subpname
				  (cons subpname
				        (cons (list (list (car part)
							  (glvrecname
							    (cadr subpspec)
							    viewtypesubs)))
					      (sublis viewtypesubs
						      (cdddr subparm)))))
			    propsubs)
		      (progn (setq errflg t)
			     (format t "The parameter ~A could not be found.~%"
				       subpname))))
		(oneof
		  (when (setq part (glvmenu "Specify choice for ~A"
					    (list subpname)
					    (cadr subpspec) choice))
		    (glvechochoice subpname part nil)
		      (push (kwote part) *glviewas-choices*)
		      (push (cons subpname
				  (cons subpname
				        (cons (list (kwote part))
					      (sublis viewtypesubs
						      (cdddr subparm)))))
			    propsubs)))
		(input
		  (setq part (glvmenu "Specify value for ~A"
				      (list subpname) nil choice))
		  (glvechochoice subpname part nil)
		  (push (kwote part) *glviewas-choices*)
		  (push (cons subpname
			      (cons subpname
				    (cons (list (kwote part))
					  (sublis viewtypesubs
						  (cdddr subparm)))))
			propsubs))
		((tuple-except names-except choose-except choose-prop-except)
		  (setq str (sublis typesubs (cadr subpspec)))
		  (setq parts (gldatanames str))
		  (if (eq (car subpspec) 'choose-prop-except)
		      (setq parts (nconc parts (glpropnametypes str 'prop))))
		  (setq zaps (glvzaps (caddr subpspec) propsubs))
		  (setq parts (subset #'(lambda (x)
					  (not (member (car x) zaps)))
				      parts))
		  (unless (eq (car subpspec) 'names-except)
		    (setq partsb
			  (if (rest parts)
			      (if (eq (car subpspec) 'tuple-except)
				  (progn
				    (setq tmpb
					  (mapcar #'(lambda (x)
						      (list (car x)
							    (glvrecname
							     (cadr subpspec)
							     viewtypesubs)))
						  parts))
				    (setq restp
					  (cons 'result
						(list (cons 'tuple
							(mapcar #'(lambda (x)
							   (list (car x)
							     (list 'typeof x)))
								tmpb)))))
				    (cons 'tuple tmpb))
				  (glvchoice parts subpname
					     (glvrecname (cadr subpspec)
							 viewtypesubs) choice))
			      (glvchoice parts subpname
					 (glvrecname (cadr subpspec)
						     viewtypesubs) choice) )))
		  (push (cons subpname
			      (cons subpname
				    (cons (if (eq (car subpspec) 'names-except)
					      (list (kwote
						     (mapcar #'car parts)))
					      (list partsb))
					  (or (sublis typesubs (cdddr subparm))
					      (if (eq (car subpspec)
						      'tuple-except)
						  restp)))))
			propsubs))) ) )
      (if *glviewas-choices*
	  (push (cons subpname (nreverse *glviewas-choices*))
		new-choices)) )

; Make args for a call to gldefclusterc
    (setq cltypespecs
	  (mapcar #'(lambda (spec)
		      (glviewastypespec spec viewtypesubs
					viewstoragesubs propsubs recnames))
		  cluster-spec))
; Save debugging info
    (setq *glviewas-parms*
	  (list 'typesubs typesubs 'parms parms 'subparms subparms
	        'cluster-spec cluster-spec 'cluster-supers cluster-supers
		'viewstoragesubs viewstoragesubs 'zaps zaps
		'viewtypesubs viewtypesubs 'tmp tmp 'propsubs propsubs
		'part part 'parts parts 'cltypespecs cltypespecs
		'errflg errflg 'clus clus 'clustersubs clustersubs
		'view-name view-name 'str str 'recnames recnames
		'subpname subpname 'subpspec subpspec 'partsb partsb))

    (if errflg (return))

    (gldefclusterc view-name cltypespecs cluster-supers)
    (when (and source-types
	       (null (cdr source-types))
	       (symbolp (car source-types)))
      (pushnew (list (or viewname goal-cluster) (caadar cltypespecs))
	       (getf (cdr (or (glstructure (car source-types))
			      (and (glclusterp (car source-types))
				   (glstructure (glclmaintype
						 (car source-types))))))
		     'views)
	       :test #'equal)
      (if new-choices
	  (if choices
	      (progn (if (null (cdr choices))
			 (setf (cdr choices) (list goal-cluster)))
		     (setf (cddr choices) new-choices))
	      (push (cons (or viewname goal-cluster)
			  (cons goal-cluster new-choices))
		     (glget (car source-types) 'viewspecs) ) ) ) )
    (return view-name) ))


; 18 July 90; 16 Nov 90
; Convert type specifications as needed for instantiated cluster
(defun glviewastypespec (typespec viewtypesubs viewstoragesubs propsubs
				  recnames)
  (let (viewtypename viewstr tmp)
    (setq viewtypename (cdr (assoc (car typespec) viewtypesubs)))
    (setq viewstr (sublis viewtypesubs
			  (sublis viewstoragesubs (cadr typespec))))
    (list (car typespec)
	  (cons viewtypename
		(cons (if (and (consp viewstr)
			       (eq (car viewstr) 'tuple))
			  (glmakeds (cdr viewstr))
			  (if (setq tmp (assoc (car typespec) recnames))
			      (list (cdr tmp) viewstr)
			      viewstr))
		      (sublis propsubs (cddr typespec)) ))) ))
  
; 14 Sep 95; 02 Aug 01
; Try to find a type that matches a pointer typespec, e.g. (^ type)
; returns a special pointer type if there is one, else (^ type)
(defun glvfindtype (typespec)
  (let (ptr str)
    (or (and (consp typespec)
	     (eq (car typespec) '^)
	     (setq ptr (glgetpointer (cadr typespec)))
	     (setq str (car (glstr ptr)))
	     (not (and (consp str) (eq (car str) '^)
		       (eq (cadr str) (cadr typespec))))
	     ptr)
      typespec) ))

; 12 Jul 90; 20 Jul 90; 07 Aug 90
; Try to find a type spec (role cluster) from cluster specs
(defun glvfindcltype (typespec clsubs)
  (let (cl)
    (and (consp typespec)
	 (consp (cdr typespec))
	 (symbolp (cadr typespec))
	 (setq cl (cdr (assoc (cadr typespec) clsubs)))
	 (or (glclusterrole (if (glclusterp cl) cl (glcluster cl))
			    (car typespec))
	     (if (eq (car typespec) 'maintype)
		 (glclmaintype cl)))) ))

; 25 Mar 97; 03 Dec 98
; If a (listof x) type is specified, expand it into a data structure
; using cons records.
(defun viewas-expand-listof (type)
  (let ((str (glxtrtypeb type)) newtype)
    (if (and (consp str)
	     (eq (car str) 'listof))
	(progn (setq newtype (gentemp "GLTYPE"))
	       (setf (glstructure newtype)
		     (list (list 'cons (list 'contents (cadr str))
				       (list 'link (list '^ newtype)))
			   'supers (list 'lisp-linked-list)))
	       newtype)
        type) ))


; 28 Sep 93
; Make a name for a view
(defun glviewname (source-types clabbrev viewflg)
  (let (view-name-string)
    (if viewflg
	(progn (dolist (source-type source-types)
		 (setq view-name-string
		       (if view-name-string
			   (concatenate 'string view-name-string "-AND-"
					(symbol-name source-type))
			   (symbol-name source-type))) )
	       (setq view-name-string
		     (concatenate 'string view-name-string "-AS-"
				  (symbol-name clabbrev))))
	(progn (setq view-name-string
		     (concatenate 'string (symbol-name clabbrev) "-OF-"))
	       (dolist (source-type source-types)
		 (setq view-name-string
		       (if (eq source-type (first source-types))
			   (concatenate 'string view-name-string
					(symbol-name source-type))
			   (concatenate 'string view-name-string "-AND-"
					(symbol-name source-type)))))))
    (intern view-name-string) ))


; 28 Sep 93; 11 Nov 94; 21 Oct 97
; Test if a type is a sequence; if so, returns the item type.
; e.g. (glsequencep '(listof consv))
(defun glsequencep (type)
  (if type
      (if (symbolp type)
	  (glsequencep (glxtrtype (glgetstr type)))
	  (if (member (car type) '(listof arrayof))
	      (cadr type)))))

; 21 Oct 97
; Find an iterator macro for a type if one is defined.
; Returns the name of the iterator macro.
(defun gliteratormacro (type)
  (let (tmp dtmp)
    (or (and (setq tmp (glstrprop type 'msg 'iterator nil))
	     (glmacro (cadr tmp))
	     (cadr tmp))
	(and (setq dtmp (glxtrtype (glgetstr type)))
	     (not (eq dtmp type))
	     (gliteratormacro dtmp)))))


; 21 Oct 97
; Find view types that have iterator macros if any are defined.
; Returns the name of the view.
; e.g. (gliteratorviews 'box)  =  (SORTED-LINKED-LIST LINKED-LIST)
(defun gliteratorviews (type)
  (remove-duplicates
    (nconc (mapcan #'(lambda (x)
		       (if (glstrprop (cadr x) 'msg 'iterator nil)
			   (list (car x))))
		   (glviews type) )
	   (mapcan #'(lambda (x)
		       (if (assoc 'iterator
				  (glmacrodefs (car x)))
			   (list (car x))))
		   (glviewspecs type)) ) ) )


; 20 Jul 90; 18 Jan 96
; Get the first type defined for a cluster as the "main" type.
; e.g. (glclmaintype 'box-as-ll)   = BOX-AS-LL-POINTER
;      (glclmaintype 'linked-list) = LINKED-LIST-POINTER
(defun glclmaintype (cl) (cadr (first (glclusterroles cl))))

; 10 Sep 90; 27 Sep 93; 05 Oct 93; 15 Dec 93; 11 Aug 95
; Try to find a stored field of a type that matches a typespec.
; e.g. (glvfindpart 'box '(^ box) 'link nil nil) = (NEXT (^ BOX))
; ch, if non-nil, is a previous choice sequence
(defun glvfindpart (type partspec role ch except)
  (let (parts choice)
    (setq parts (subset #'(lambda (x)
			    (and (not (member (car x) except))
				 (gltypematch (cadr x) partspec)))
			(gldatanames type)))
    (setq choice (glvchoose parts role ch))
    (if *glviewas-echo* (terpri))
    choice ))

; 11 Aug 95
; Given a set of exceptions (names of view spec properties that cannot
; be legal selections), find the corresponding user names to eliminate.
; (GLVZAPS '(LINK) '((LINK LINK ((NEXT Z2)) RESULT BOX-AS-SLL-POINTER)))
;    => (NEXT)
(defun glvzaps (excepts propsubs)
  (let (tmp)
    (mapcan #'(lambda (prop)
		(if (setq tmp (assoc prop propsubs))
		    (list (if (consp (caaddr tmp))
			      (if (eq (car (caaddr tmp)) 'tuple)
				  (cadr (caaddr tmp))
				  (car (caaddr tmp)))
			      (caaddr tmp)))))
	    excepts) ))


; 27 Sep 93; 15 Dec 93; 06 Feb 96
(defun glvchoose (parts role ch)
  (let (choice)
    (setq choice (if (cdr parts)
		     (glvmenu "Specify choice for ~A" (list role) parts ch)
		     (car parts)) )
    (push (if (consp choice) (first choice) choice) *glviewas-choices*)
    (glvechochoice role (if (consp choice) (first choice) choice)
		   (null (cdr parts)))
    choice))

; 06 Feb 96
(defun glvechochoice (role choice always)
  (if (and *glviewas-echo*
	   (or always (not *glviewas-tty*)))
      (format t "  Choice for ~A = ~A~%" role choice)) )

; 30 Sep 93; 01 Oct 93; 05 Oct 93; 15 Dec 93
(defun glvchoice (parts role source ch)
  (prog (choice path)
    (setq choice (glvchoose parts role ch))
    (if (if (consp choice)
	    (eq (first choice) (first ch))
	    (eq choice (first ch)))
	(if (null (rest ch))
	    (return (list (first choice) source))
	    (pop ch))
	(setq ch nil))
    (setq path (glvdatapath (list (if (consp choice) (first choice) choice)
				  source)
			    (if (consp choice) (cadr choice))
			    ch))
    (return (first path)) ))

; 01 Oct 93; 05 Oct 93; 15 Dec 93; 26 Sep 95
; Find a path to the desired data from starting code and type.
; ch is a previous choice list
; Result is (<code> <type>)
(defun glvdatapath (code type ch)
  (let (propnames sel)
    (if (and (consp type)          ; automatically dereference a pointer
	     (eq (car type) '^))
	(progn (format t " ^.")
	       (glvdatapath (list '^. code) (cadr type) ch))
	(if (and (not (glbasictypep type))
		 (setq propnames (gevgetnames type t)))
	    (progn (setq sel (glvmenu "" nil
				      (cons "Done" (mapcar #'car propnames))
				      ch))
		   (if (eq sel (first ch))
		       (pop ch)
		       (setq ch nil))
		   (if (equalp sel "Done")
		       (list code type)
		       (progn
			 (push sel *glviewas-choices*)
			 (if *glviewas-echo* (format t " ~A" sel))
			 (glvdatapath (list sel code)
				      (cadr (assoc sel propnames)) ch))) )
	    (progn (if *glviewas-echo* (terpri))
		   (list code type)) ) ) ))

; 16 Nov 90; 19 Nov 90
; Find a record name to be used for a record within a cluster instance
(defun glvrecname (record viewtypesubs)
  (let (pair name)
    (declare (special recnames))
    (if (setq pair (assoc record recnames))
	(cdr pair)
	(progn (setq name (glmkatom 'z))
	       (push (cons record name) recnames)
	       name)) ))

; 27 Sep 93; 30 Sep 93; 15 Dec 93; 28 May 02; 10 Sep 02; 13 Aug 03; 14 Aug 03
; 09 Nov 04; 27 May 09; 16 Jul 10
; Use window menu if available, or simulate with tty.
; Choices is a list of (name type) pairs.
; ch is previous choice or choice list.
(defun glvmenu (prompt-string prompt-parms choices ch)
  (let (choice)
    (or (and ch (glvmenuchoice (if (consp ch) (first ch) ch) choices))
	(if (or *glviewas-tty* (null choices))
	    (ttymenu prompt-string prompt-parms choices)
	    (progn 
	      (glusermsg
	        (apply #'format (cons nil (cons prompt-string prompt-parms)))
		t)
	      (setq choice
		    (menu (mapcar #'(lambda (x)
                                      (if (consp x)
                                          (cons (stringify (first x)) x)
                                          x))
				  choices)
			  (stringify (if (cdr prompt-parms)
					 prompt-parms
				       (car prompt-parms)))))
	      (glusermsg (format nil "   = ~A" choice))
	      choice) ) ) ))

; 15 Dec 93
(defun glvmenuchoice (resp choices)
  (some #'(lambda (x) (if (consp x)
			  (if (equal (car x) resp) x)
			  (if (equalp x resp) resp)))
	choices) )

; 10 July 90; 18 Jul 90; 09 Nov 04
; Simple menu function for typein
; A choice can be an atom or (displayed-choice . result) .
(defun ttymenu (prompt-string prompt-parms choices)
  (prog (resp choice)
lp  (apply #'format (cons t (cons prompt-string prompt-parms)))
    (if (and (consp choices) (null (rest choices)))
	(progn (format t " -- Choice is: ~A~%" (first choices))
	       (return (first choices))))
    (if choices 
	(format t "~%Choices are: ~A"
		(mapcar #'(lambda (x) (if (consp x) (car x) x))
			choices)))
    (setq resp (read))
    (if (null choices) (return resp))
    (if (setq choice (some #'(lambda (x)
			       (if (consp x)
				   (if (equal (car x) resp) x)
				   (if (equal x resp) resp)))
			   choices))
	(return choice)
	(if (and (numberp resp) (> resp 0) (<= resp (length choices)))
	    (return (nth (1- resp) choices))
	    (progn (format t "Improper response~%") (go lp)) )) ))

; 03 Dec 90
; (gldefviewspec <name> <supers> <abbrev> <viewflg>
;                       <parms> <subparms> <cluster-spec> <cluster>)
(defun gldefviewspec (def)
  (setf (glclusterviewspec (car def)) (cddr def))
  (setf (glclusterviewsupers (car def)) (cadr def))
  (dolist (s (cadr def)) (pushnew (car def) (glclusterviews s)) ) )

; 20 Jul 90
; Make a set of type substitutions corresponding to main types of
; the given cluster substitutions.
(defun glclustertypesubs (clsubs)
  (mapcar #'(lambda (sub) (cons (car sub) (glclmaintype (cdr sub))))
	  clsubs) )

; 27 Sep 93
; Make a symbol by hyphenating a prefix and suffix.
(defun glhyphenate (prefix suffix)
  (intern (concatenate 'string (stringify prefix) "-" (stringify suffix))) )

; 05 Feb 91; 18 Jan 96
; Undo types associated with a view.  The argument may be a single type,
; in which case all views are undone, or a list (viewname type).
(defun glclchanged (arg)
  (let (viewlist tmp clus)
    (setq viewlist (if (symbolp arg)
		       (glviews arg)
		       (if (setq tmp (assoc (first arg) (glviews (second arg))))
			   (list tmp))))
    (dolist (view viewlist)
      (setq clus (glcluster (cadr view)))
      (dolist (role (glclusterroles clus))
	(glstrchanged (cadr role)))
      (setf (glviews arg) (delete view (glviews arg))) ) ))

; 20 Jul 90
(gldefviewspec
  '(lisp-linked-list (sequence) lll nil ((rec anything))
     ()
     ((pointer (^ record))
      (record (cons (contents rec) (link pointer))))
     (linked-list)) )

; Linked list where pointer is internal to record
(gldefviewspec
  '(linked-list (sequence) ll t ((record anything))
     ((type pointer             (^ record))
      (prop link                (partof record pointer) result pointer)
      (prop copy-contents-names (names-except record (link)) ) )
     ((pointer pointer)
      (record record
         prop (link copy-contents-names)))
     (linked-list)) )

; Linked list where link pointer is internal to record
(gldefviewspec
  '(linked-list-b (sequence) llb t ((rec anything))
     ((type ptr           (^ rec))
      (prop link          (partof rec ptr) result pointer)
      (prop copy-contents-names (names-except rec (link)) ) )
     ((pointer ptr)
      (record  rec
         prop (link copy-contents-names)))
     (linked-list)) )

; 04 Apr 91; 19 Sep 91
; Linked list where pointer is external to record
(gldefviewspec
  '(ext-linked-list (sequence) xll t ((record anything))
     ((type pointer             (^ record))
      (prop link                (partof record pointer) result pointer)
      (prop copy-contents-names (names-except record (link)))
      (prop contents            (choose-prop-except record (link))) )
     ((pointer pointer)
      (record record
         prop (link contents copy-contents-names)))
     (ext-linked-list)) )

; 30 Dec 91
(gldefviewspec
  '(circular-linked-list (sequence) cll t ((record anything))
     ((type pointer             (^ record))
      (prop link                (partof record pointer) result pointer)
      (prop copy-contents-names (names-except record (link)) ) )
     ((pointer pointer)
      (record record
         prop (link copy-contents-names)))
     (circular-linked-list)) )

; Sorted linked list, sorted on a value, ascending or descending
(gldefviewspec
  '(sorted-linked-list (sorted-sequence) sll t ((record anything))
     ((type pointer        (^ record))
      (prop link           (partof record pointer) result pointer)
      (prop copy-contents-names  (names-except record (link)) )
      (prop sort-value     (choose-prop-except record (link)))
      (prop sort-direction (oneof (ascending descending))) )
     ((pointer pointer)
      (record  record
         prop (link copy-contents-names sort-value sort-direction)))
     (sll)) )

; 31 Jul 01
; Association list, comparison on a value, unordered
(gldefviewspec
  '(alist (sequence) alist t ((record anything))
     ((type pointer        (^ record))
      (prop link           (partof record pointer) result pointer)
      (prop copy-contents-names  (names-except record (link)) )
      (prop sort-value     (choose-prop-except record (link))) )
     ((pointer pointer)
      (record  record
         prop (link copy-contents-names sort-value)))
     (alist)) )

; 20 Sep 91
; Experimental version of sorted-linked-list built on linked-list
(gldefviewspec
  '(sordid-linked-list (sorted-sequence) sdll t ((ll linked-list))
     ((type pointer        (pointer ll))
      (type record         (record ll))
      (prop sort-value     (choose-prop-except record (link)))
      (prop sort-direction (oneof (ascending descending))) )
     ((pointer pointer)
      (record  record
         prop (sort-value sort-direction)))
     (sdll)) )

; 12 Jul 90
; Two-pointer queue built from a linked list
(gldefviewspec
  '(two-pointer-queue (queue) tpq nil ((ll linked-list))
     ((type pointer (pointer ll)) )
     ((record (tuple (qstart pointer) (qend pointer))))
     (two-pointer-queue)) )

; 16 Jul 90
; Front-pointer queue built from a linked list
(gldefviewspec
  '(front-pointer-queue (queue) fpq nil ((ll linked-list))
     ((type pointer (pointer ll)) )
     ((record (tuple (qstart pointer))))
     (front-pointer-queue)) )

; 16 Jul 90
; Circular queue with end pointer, built from a linked list
(gldefviewspec
  '(end-pointer-queue (queue) epq nil ((ll linked-list))
     ((type pointer (pointer ll)) )
     ((record (tuple (qend pointer))))
     (end-pointer-queue)) )

; 18 Jul 90; 19 Jul 90
(gldefviewspec
  '(list-priority-queue (priority-queue) lpq nil ((q queue))
     ((type qrec (record q))
      (prop size (input integer)))
     ((record (cons (contents qrec) (link pointer)))
      (pointer (^ record) prop (size) supers (priority-queue-record)))
     (priority-queue linked-list)) )

; 20 Jul 90
(gldefviewspec
  '(priority-queue () pq nil ((q queue))
     ((cluster qseq (sequence q))
      (type qseqrec (maintype qseq))
      (prop size (input integer)))
     ((record (transparent qseqrec)              ; ******** elim. transparent?
	      prop (size) supers (priority-queue-record)))
     (priority-queue)) )

; 21 Nov 90
; Priority queue starting from record spec
(gldefviewspec
  '(priq () priq nil ((record anything))
     ((cluster q       (queue record))
      (cluster qseq    (sequence q))
      (type    qseqrec (maintype qseq))
      (prop    size    (input integer)))
     ((record (transparent qseqrec)              ; ******** elim. transparent?
	      prop (size) supers (priority-queue-record)))
     (priority-queue)) )

; 10 Sep 90
(gldefviewspec
  '(tree () tree t ((record anything))
     ((type pointer  (^ record))
      (prop descendants (partof record (setof pointer)))
       ; (adj  terminal    (partof pointer boolean))
      (prop copy-contents (tuple-except record (descendants))) )
     ((pointer pointer)
      (record (transparent record)              ; ******** elim. transparent?
         prop (descendants copy-contents)))
     (tree)) )

; 28 Dec 92
(gldefviewspec
  '(convex-hull (sequence) cvh t ((record anything))
     ()
     ((source-point record)
      (source-collection (listof record))
      (internal-record (list (pt record)
			     (next (^ internal-record))
			     (points (listof record)))))
     (convex-hull-cluster)) )

; 30 Sep 93; 04 Oct 93
; Binary tree, sorted on a value
(gldefviewspec
  '(binary-tree (tree) bintr t ((record anything))
     ((type pointer        (^ record))
      (prop left           (partof record pointer) result pointer)
      (prop right          (partof record pointer (left)) result pointer)
      (prop copy-contents-names  (names-except record (left right)) )
      (prop sort-value     (choose-prop-except record (left right))) )
     ((pointer pointer)
      (record  record
         prop (left right copy-contents-names sort-value)))
     (binary-tree)) )

; 05 Oct 93; 09 Mar 94; 14 Mar 97
; Binary tree, sorted on a value
(gldefviewspec
  '(avl-tree (tree) avltr t ((record anything))
     ((type pointer        (^ record))
      (prop left           (partof record pointer) result pointer)
      (prop right          (partof record pointer (left)) result pointer)
      (prop copy-contents-names (names-except record (left right)) )
      (prop balance        (choose-prop-except record (left right)))
      (prop sort-value     (choose-prop-except record (left right balance))) )
     ((pointer pointer)
      (record  record
         prop (left right balance copy-contents-names sort-value)))
     (avl-tree)) )

; Examples of objects that can be viewed as a linked list:

(glispobjects (box (list (color color)
			 (name string)
			 (size integer)
			 (next (^ box)))
		   prop ((volume ((size ^ 3))) 
			 (wavelength ((wavelength color))))
		   viewspecs ((linked-list)))
              (color symbol
		   prop ((wavelength
			  ((or (cadr (assoc self '((red 10)(orange 8)
						   (yellow 6)(green 4)
						   (blue 2)(violet 1))))
			       5))))
		   msg  ((< (glambda (x (y (typeof x)))
				     (< (wavelength y) (wavelength x)))))
		   ) )

; currently boxb does not work due to name conflict on 'link'
(glispobjects (boxb (list (color color)
			  (size integer)
			  (link (^ boxb)))      ; conflict with abstract name
		   prop ((volume ((size ^ 3))) 
			 (wavelength ((wavelength color)))) )
	      )

; C version of box structure
(glispobjects (cbox (crecord cbox
			     (color color)
			     (name string)
			     (next (^ cbox))
			     (size integer))
		    supers (box)))

(glispobjects (myrec (crecord myrec
			      (color integer)
			      (name string)
			      (size integer)
			      (next (^ myrec)))))


(glispobjects (foo (atomobject (name string)
			       (birthday date)
			       (salary number)
			       (relatives (list (friend (^ foo))
						(mother (^ foo))
						(father (^ foo))))
			       (weight number)))
 (foob (list (birthday date) (name string) (friend (^ foob)) ) ) )

(glispobjects (bl (cons (nxt (^ bl)) (val integer))     ; "backward list"
		  viewspecs ((ext-linked-list))) )      ; 04 Apr 91

(gldefun xa ((b box))
  (let (tmp)
    (format t "Original: ~A~%" b)
    (format t "Length:   ~A~%" (length (linked-list b)))
    (format t "Copied:   ~A~%" (setq tmp (copy-list (linked-list b))))
    (format t "Reversed: ~A~%" (nreverse tmp))))
; [now automatic] (glviewas 'linked-list 'box) 
(defvar mybox)
(setq mybox
  (a box with color = 'red name = "widget" size = 1
     next = (a box with color = 'green name = "grommet" size 2
	      next  = (a box with color = 'blue name = "gadget" size 3))))
; (xa mybox)
(defvar mycbox)
(setq mycbox
  (a cbox with color = 'red name = "widget" size = 1
     next = (a cbox with color = 'green name = "grommet" size 2
	      next  = (a cbox with color = 'blue name = "gadget" size 3))))

(gldefun xb ((b foo))
  (let (tmp)
    (format t "Original: ~A~%" b)
    (format t "Length:   ~A~%" (length (linked-list b)))
    (format t "Copied:   ~A~%" (setq tmp (copy-list (linked-list b))))
    (format t "Reversed: ~A~%" (nreverse tmp))))
; (glviewas 'linked-list 'foo) ; and choose FRIEND as link.
(defvar tom (a foo with name = "Tom"
	       birthday = (a date with month = 3  day = 23  year = 1951)
	     friend = (a foo with name = "Dick"
		    birthday = (a date with month = 4  day = 3  year = 1978)
			 friend = (a foo with name = "Harry"
		    birthday = (a date with month = 8  day = 17  year = 1980)
     ))))
; (xb tom)

(gldefun xc ((b bl))                   ; 04 Apr 91
  (let (tmp)
    (format t "Original: ~A~%" b)
    (format t "Length:   ~A~%" (length (ext-linked-list b)))
    (format t "Copied:   ~A~%" (setq tmp (copy-list (ext-linked-list b))))
    (format t "Reversed: ~A~%" (nreverse tmp))))
(defvar mybl '(((((((nil . 7) . 6) . 5) . 4) . 3) . 2) . 1))
; (xc mybl)

; 10 Jan 97
; (glviewas 'sorted-linked-list 'bl)   ; and specify 1 for ASCENDING
(gldefun xd ((b bl) (n integer)) (insert-key (sorted-linked-list b) n))
; (xd mybl 3.5)
(gldefun xdd ((b bl) (bb bl))
  (merge (sorted-linked-list b) (sorted-linked-list bb)))
(setq myblb '(((((((nil . 17) . 6) . 3.5) . 3.25) . 3) . 1.5) . 0.5))
; (xdd mybl myblb)

; (glviewas 'sorted-linked-list 'box)    ; select SIZE or COLOR to sort on,
;                                                 ASCENDING
(gldefun xe ((b box) new) (insert (sorted-linked-list b) new))
; (xe mybox '(yellow 2.5 nil))
(gldefun zl ((l box-as-sll-pointer) (b box-as-sll-pointer)) (insert l b))
(gldefun zll ((l box) (b box)) (insert (sorted-linked-list l)
				     (sorted-linked-list b)))
; (zl  mybox (a box with color = 'yellow size = 2.5))
; (zll mybox (a box with color = 'violet size = 2.25))

; Must select COLOR, ASCENDING for the following example:
(defvar myboxb '(orange 7 (yellow 5 (violet 9 nil))))
(gldefun xee ((b box) (bb box))
  (merge (sorted-linked-list b) (sorted-linked-list bb)))
; (setq myboxc (xee mybox myboxb))

; (glviewas 'two-pointer-queue 'box)
(gldefun zu () (a tpq-of-box-record))
(gldefun zt ((q tpq-of-box-record) (b box)) (insert q (linked-list b)))
(gldefun zs ((q tpq-of-box-record)) (remove q))
; (setq mybq (zu))
; (zt mybq (a box with color 'yellow size 3))
; (zt mybq (a box with color 'orange size 4))
; (zt mybq (a box with color 'red    size 5))
; (zt mybq (a box with color 'green  size 6))
; (zs mybq)

; (glviewas 'front-pointer-queue 'box)
(gldefun zo () (a fpq-of-box-record))
(gldefun zn ((q fpq-of-box-record) (b box)) (insert q (linked-list b)))
(gldefun zm ((q fpq-of-box-record)) (remove q))
; (setq myfq (zo))
; (zn myfq (a box with color 'yellow size 3))
; (zn myfq (a box with color 'orange size 4))
; (zn myfq (a box with color 'red    size 5))
; (zn myfq (a box with color 'green  size 6))
; (zm myfq)

; (glviewas 'queue 'box t) ; and choose end-pointer-queue
(gldefun zr () (a epq-of-box-record))
(gldefun zq ((q epq-of-box-record) (b box)) (insert q (linked-list b)))
(gldefun zp ((q epq-of-box-record)) (remove q))
; (setq myeq (zr))
; (zq myeq (a box with color 'yellow size 3))
; (zq myeq (a box with color 'orange size 4))
; (zq myeq (a box with color 'red    size 5))
; (zq myeq (a box with color 'green  size 6))
; (zp myeq)

; (glviewas 'queue 'foo) ; and choose end-pointer-queue
(gldefun yr () (a epq-of-foo-record))
(gldefun yq ((q epq-of-foo-record) (b foo))                 ; 07 Oct 91
  (insert q (linked-list b)))
(gldefun yp ((q epq-of-foo-record))                        ; 07 Oct 91
  (let (tmp) (while (tmp =  (remove q)) (print (name tmp)))))
; (setq feq (yr))
; (yq feq (a foo with name 'marvin))
; (yq feq (a foo with name 'fred))
; (yp feq)

; (glviewas 'priority-queue 'box)
; choose e.g. FRONT-POINTER-QUEUE, LISP-LINKED-LIST
(gldefun ym () (a pq-of-box-record))
(gldefun yn ((pq pq-of-box-record) (n integer) (new box))
  (insert pq n (linked-list new)) )    ; removed (^. )  25 Sep 91
(gldefun yo ((pq pq-of-box-record) (n integer)) (remove pq n))
; (setq pqq '((nil)(nil)(nil)(nil))) ; a crock -- should use (a ...)
; (yn pqq 2 (a box with color 'yellow size 3))
; (yn pqq 2 (a box with color 'orange size 4))
; (yn pqq 2 (a box with color 'red    size 5))
; (yn pqq 1 (a box with color 'green  size 6))
; (yo pqq 1)
; (yo pqq 2)

; (glviewas 'queue 'elf) ; and choose end-pointer-queue
(gldefun xr () (an epq-of-elf-record))
(gldefun xq ((q epq-of-elf-record) (e elf))                 ; 07 Oct 91
  (insert q (linked-list e)))
(gldefun xp ((q epq-of-elf-record))                        ; 07 Oct 91
  (let (tmp) (while ((tmp =  (remove q)) <> 'nobody)
	       (print (name tmp)))))
; (setq eeq (xr))
; (xq eeq (an elf with name 'marvin))
; (xq eeq (an elf with name 'fred))
; (xp eeq)

; (glviewas 'tree 'elf)
(gldefun wdd ((e elf)) (for x in (tree elf) do (print (name x)))) ; 18 Sep 91
(gldefun wee ((e elf)) (size (tree e)))
; (wdd happy)  ; data defined in genll.test
; (wee happy)

#|       ; commented out since the pointer property of fum generates an error.
; 28 Nov 90; 30 Nov 90
(glispobjects

(fum-pointer integer
prop    ((null-value (-1) result fum-pointer)
         (dereference ((aref *fumarray* self)) result fum))
adj     ((null minusp)) )

(fum (list (name symbol) (next fum-pointer))
msg  ((new fum-new result fum-pointer))
pointer (fum-pointer) )            ; () added 06 May 91

  ) ; glispobjects     |#

; 03 Jan 91
; Copy one kind of linked list to another
; 07 Oct 91: presently not working because of mismatch in names:
; bl uses the name VAL for contents, listof-integer-pointer uses CONTENTS.
; bl-as-xll-record defines both copy-contents-names and copy-contents,
; and the former has precedence in generic-transfer-contents.
; But wh, going the other direction, works.
(gldefun wg ((l listof-integer-pointer) (m bl))
  (glsend l copy-list-to (ext-linked-list m)))
; (wg '(1 2 3 4) nil)

(gldefun wh ((l bl) (m listof-integer-pointer))
  (glsend (ext-linked-list l) copy-list-to m))
; (wh mybl nil)

; 05 Feb 91
; Copy one kind of linked list to another.
; foo and foob do not match exactly, but stored values of foob are a subset
; of the stored values of foo.
(gldefun wi ((l foo) (m foob))
  (glsend (linked-list l) copy-list-to (linked-list m)) )
; (wi tom nil)

; 12 Feb 91; 04 Sep 92
(glispobjects (cirl (list (radius real) (center vector)
			  (color color) (nxt (^ cirl)))
		    supers (circle)
		    viewspecs ((linked-list)))
	      (dcirl (list (color color) (nxtb (^ dcirl)) (area real)
			   (center vector))     ; challenge: consv instead
		     prop ((radius ((sqrt (/ area pi)))))       ; 08 Oct 91
		     supers (circle)
		    viewspecs ((linked-list))) )
(setq mycirl (a cirl with radius 3.0 center '(3 4) color 'red nxt 
		(a cirl with radius 4.0 center '(5 6) color 'yellow nxt
		   (a cirl with radius 5.0 center '(7 8) color 'blue))))
; wjb, wjc  demonstrate copying with different format and different data --
; area vs. radius.
(gldefun wjb ((l cirl) (m dcirl))
  (copy-list-to (linked-list l) (linked-list m)) )
(gldefun wjc ((l dcirl) (m cirl))
  (copy-list-to (linked-list l) (linked-list m)) )
; (glviewas 'linked-list 'cirl)
; (glviewas 'linked-list 'dcirl)
; (setq mydcirl (wjb mycirl nil))
; (setq mycirlb (wjc mydcirl nil))
