; webrpc.lsp                  Gordon S. Novak Jr.            ; 14 Jan 13

; Copyright (c) 2012 Gordon S. Novak Jr. and The University of Texas at Austin.

; 11 Oct 06; 23 Oct 06; 23 Oct 06; 03 Jan 07; 18 Jul 07; 18 Sep 07 ; 10 Mar 08
; 17 Dec 08; 22 Feb 12

; This program is free software; you can redistribute it and/or modify
; it under the terms of the GNU General Public License as published by
; the Free Software Foundation; either version 2 of the License, or
; (at your option) any later version.

; This program is distributed in the hope that it will be useful,
; but WITHOUT ANY WARRANTY; without even the implied warranty of
; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
; GNU General Public License for more details.

; You should have received a copy of the GNU General Public License
; along with this program; if not, write to the Free Software Foundation,
; Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


; Implement a web remote procedure call from Gnu Common Lisp

; The basic idea is to use (system ...) from GCL to issue a command
; to curl to transmit parameters to a web page and retrieve as a file
; the results returned by the web page.  These results can then
; be parsed to get the desired answer(s).

; Generate a gensym file name
(defun genfilename (&optional string)
  (cstr (symbol-name (if string (gensym string) (gensym))) ".TXT"))

; 17 Dec 08
; Make a call to curl, with args as -d options
(defun makecurl (url args resultfile)
  (let (res)
    (push (cstr " >" resultfile) res)
    (push url res) (push " " res)
    (while args
      (push (cstr " -d " (stringify (pop args)) "=\""
                         (stringify (pop args)) "\"")
            res))
    (push "curl -s" res)
    (apply #'concatenate (cons 'string res)) ))

; 17 Dec 08
; Make a call to curl, with args in the URL
(defun makecurlb (url args resultfile)
  (let (res sep)
    (push (cstr " >" resultfile) res)
    (push "\"" res)
    (while args
      (setq sep (if (cddr args) "&" "?"))
      (push (cstr sep (stringify (pop args)) "=" (stringify (pop args)))
            res))
    (push url res) (push "\"" res)
    (push "curl -s " res)
    (apply #'concatenate (cons 'string res)) ))

; 10 Oct 06; 17 Dec 08
; Perform a "remote procedure call" to a web site
; url     = url of the web site (the "action" address, not the top page)
; args    = list of  (arg1 value1 ...)
; grammar = grammar for parsing the result
; inurl   = true to include args in the URL rather than post
(defun webrpc (url args grammar &optional inurl)
  (let ((file (genfilename)) command (n 0) okay fileptr res)
    (setq command (if inurl
                      (makecurlb url args file)
                      (makecurl url args file)))
    (system command)
    (while (and (not okay) (< n 100))
      (setq fileptr (open file :direction :input :if-does-not-exist nil))
      (if fileptr
          (progn (close fileptr) (setq okay t))
          (progn (incf n) (sleep 0.1))) )
    (if okay
        (setq res (parsefile file grammar)))
    (system (cstr "rm " file))
    res))

; 11 Oct 06; 23 Oct 06
; Use curl to do an ftp.  All args are strings.
; path should begin and end with /
; dir should end with /
; rename is new file name if desired
; returns "dir/file" if successful
; (ftpcurl "ftp.cs.utexas.edu" "/pub/novak/cs381k/" "sunify.lsp")
(defun ftpcurl (site path file &optional (dir "") rename)
  (let (command newfile okay (n 0) fileptr)
    (setq newfile (concatenate 'string dir (or rename file)))
    (setq command (if (and site path)
                      (concatenate 'string "curl -s ftp://" site path file
                                           " >" newfile)
                      (concatenate 'string "curl -s " file " >" newfile)))
    (system command)
    (while (and (not okay) (< n 100))
      (setq fileptr (open newfile :direction :input :if-does-not-exist nil))
      (if fileptr
          (progn (close fileptr) (setq okay t))
          (progn (incf n) (sleep 0.1))) )
    (and okay newfile)))

; 10 Oct 06; 18 Jul 07; 20 Sep 07
; Look up a zip code to get (city state zip lat long)
; cf. http://zipinfo.com/search/zipcode.htm
(setf (glfnresulttype 'zipcode)
      '(cons (city string) (cons (statecode statecode) (cons (zip number)
             (lat-long lat-long)))))
(defun zipcode (zip)
  (let ((res
    (webrpc "http://zipinfo.com/cgi-local/zipsrch.exe"
            (list "zip" (zipstring zip) "ll" "ll")
            '(seq (skipto "Longitude")
                  (skipto "<td") (skipto ">") (stringto "<")
                  (skipto "<td") (skipto ">") (sym)
                  (skipto "<td") (skipto ">") (num)
                  (skipto "<td") (skipto ">") (num)
                  (skipto "<td") (skipto ">") (num)) )))
    (if res (setf (fifth res) (- (fifth res))))   ; make - for West longitude
    res))

(defun zipstring (zip)
  (if (and (numberp zip) (> zip 0) (< zip 100000))
      (subseq (princ-to-string (+ zip 100000)) 1) ) )

; 10 Oct 06; 11 Oct 06; 06 Nov 09; 22 Feb 12
; Look up a stock price on Yahoo Finance
(defun stockprice (symbol)
  (webrpc "http://finance.yahoo.com/q"
          (list "s" symbol)
          '(car (seq (skipto "time_rtq_ticker") (skipto "<span ")
                     (skipto ">")(num))) ) )

; 31 Oct 06; 17 Dec 08; 06 Nov 09
; Look up a stock price on Google Finance
(defun stockpriceb (symbol)
  (webrpc "http://finance.google.com/finance"
          (list "q" symbol)
          '(car (seq (skipto "<title>") (skipto ":") (skipws) (num)))
          t) )

; 14 Jan 13
; Look up an IP address
(defun ipaddress (urlstring)
  (webrpc "http://tejji.com/ip/url-to-ip-address.aspx"
          (list "domain" urlstring)
          '(seq (skipto "IP Address(es)") (skipto "<tr>") (skipto "<span>")
                (int) (skipto ".") (int) (skipto ".") (int)
                (skipto ".") (int) )
          t) )

; 17 Dec 08
; Convert currencies: see currencies.lsp
; (currency 'usd 'jpy)
; ? this one doesn't quite work -- web site gives same old page
(defun currencyc (from to &optional (amount 1))
  (webrpc "http://www.gocurrency.com/v2/dorate.php"
          (list "inV" amount "from" from "to" to "Calculate" "Convert")
          '(car (seq (skipto "Currency Converter Results")
                     (skipto "<b") (skipto "=") (skipws) (num))) ) )

; 03 Jan 07
; not working: old Yahoo version
(defun currencyb (from to &optional (amount 1))
  (webrpc "http://finance.yahoo.com/currency/convert"
          (list "amt" amount "from" from "to" to "submit" "Convert")
          '(car (seq (skipto "Currency Conversion Results")
                     (skipto "Ask")
                     (skipto "<tr") (skipto "<td") (skipto "<td")
                     (skipto "<td") (skipto "<td") (skipto ">") (num))) ) )

; 17 Dec 08
(defun currency (from to &optional (amount 1))
  (webrpc "http://finance.google.com/finance/converter"
          (list "a" amount "from" from "to" to)
          '(car (seq (skipto "currency_converter_result")
                     (skipto "span")(skipto "bld")
                     (skipto ">") (num)))
          t) )

; 18 Sep 07
(defun citylatlong (city state)
  (webrpc (concatenate 'string "http://www.citytowninfo.com/places/"
                       (stringify state) "/" (stringify city))
          nil
          '(seq (skipto "Latitude") (skipto "<td") (skipto ">") (num)
                (skipto "Longitude") (skipto "<td") (skipto ">") (num)) ) )

(defun citystatelatlong (citystate)
  (citylatlong (car citystate) (cadr citystate)) )

; 08 Oct 07
(defun elementdata (symbol)
  (webrpc (concatenate 'string
             "http://www.webelements.com/webelements/elements/text/"
             (stringify symbol) "/key.html")
          nil
          '(seq (skipto "essentials")
                (skipto "Name") (skipto ":")          (skipws) (sym)
                (skipto "Atomic number") (skipto ":") (skipws) (num)
                (skipto "Atomic weight") (skipto ":") (skipws) (num)) ) )
