/* picoGraf.c
 * 31may91abu
 */

#include "pico.h"
#include <malloc.h>

/* Globals */
static pico drawFoo;
static point onPt;
static long drawN, crossN, onN;
static long crLin, crCnt, crSiz, *crLst=NULL;

/* Prototypes */
static void bounds(pico,point*,point*);
static void buildCross(pico);
static void cross(number,number,number,number);
static void drawBez(long,long,long,long,long,long,long,long);
static void crossBez(long,long,long,long,long,long,long,long);
static pico graf(pico,pico);
static bool onBez(int,long,long,long,long,long,long,long,long);
static bool onLine(long,long,long,long);
static long bezRoute(int,long,long,long,long,long,long,long,long);


/* Lines */
static bool onLine(h1,v1,h2,v2)
number h1,v1,h2,v2;
{
   register number dh,dv;

   dh = h2 - h1;
   dv = v2 - v1;
   if (dh || dv) {
      if (abs(dh) > abs(dv)) {
         if (dh > 0) {
            if (h1 <= onPt.h  &&  onPt.h <= h2  &&
                 abs(v1 + muldiv(onPt.h-h1,dv,dh) - onPt.v) < onN)
               return YES;
         }
         else {
            if (h2 <= onPt.h  &&  onPt.h <= h1  &&
                 abs(v2 + muldiv(onPt.h-h2,dv,dh) - onPt.v) < onN)
               return YES;
         }
      }
      else {
         if (dv > 0) {
            if (v1 <= onPt.v  &&  onPt.v <= v2  &&
                 abs(h1 + muldiv(onPt.v-v1,dh,dv) - onPt.h) < onN)
               return YES;
         }
         else {
            if (v2 <= onPt.v  &&  onPt.v <= v1  &&
                 abs(h2 + muldiv(onPt.v-v2,dh,dv) - onPt.h) < onN)
               return YES;
         }
      }
   }
   return NO;
}

static void cross(h1,v1,h2,v2)
number h1,v1,h2,v2;
{
   register number c, *p;
   int i;
   number temp;

   if (v1 <= crLin && crLin < v2  ||  v2 <= crLin && crLin < v1) {
      c = h1 + muldiv(crLin-v1, h2-h1, v2-v1);
      if (crCnt == crSiz)
         crLst = (number*)realloc(crLst, ++crSiz * sizeof(number));
      p = crLst;
      i = 0;
      while (i < crCnt  &&  c > *p)
         ++i, ++p;
      while (i++ < crCnt)
         temp = *p,  *p++ = c,  c = temp;
      *p = c;
      ++crCnt;
   }
}

/* Points */
void needPoint(x)
register pico x;
{
   if (!isCell(x) || !isNum(car(x)) || !isNum(cdr(x)))
      errObj(x, "Point expected");
}

void unBoxPoint(x,pt)
register pico x;
register point *pt;
{
   needPoint(x);
   pt->h = unBox(car(x));
   pt->v = unBox(cdr(x));
}

pico boxPoint(pt)
point *pt;
{
   return newCell(boxNum(pt->h),boxNum(pt->v));
}

pico AccPoint(x)
pico x;
{
   register point *adr;

   adr = (point*)EVAL1(x);
   NEEDNUM(adr);
   adr = (point*)unBox(adr);
   if (isNil(x = EVAL1((cdr(x)))))
      return newCell(boxNum(adr->h), boxNum(adr->v));
   unBoxPoint(x,(point*)adr);
   return x;
}

/* Rectangles */
bool isRect(x)
register pico x;
{
   register pico y;

   return (isCell(x) && isCell(y = car(x)) && isCell(cdr(x)) &&
            isNum(car(y)) && isNum(cdr(y)) && isNum(cdr(car(x))) &&
                                                isNum(cdr(cdr(x))) );
}

void needRect(x)
pico x;
{
   if (!isRect(x))
      errObj(x, "Rectangle expected");
}

pico boxRect(r)
register rect *r;
{
   return newCell2(boxNum(r->left), boxNum(r->top),
                  boxNum(r->right), boxNum(r->bottom));
}

void unBoxRect(x,r)
register pico x;
register rect *r;
{
   register pico y;

   needRect(x);
   y = car(x);
   x = cdr(x);
   r->left = (integer)unBox(car(y));
   r->top = (integer)unBox(cdr(y));
   r->right = (integer)unBox(car(x));
   r->bottom = (integer)unBox(cdr(x));
}

pico AccRect(x)
pico x;
{
   register rect *adr;

   adr = (rect*)EVAL1(x);
   NEEDNUM(adr);
   adr = (rect*)unBox(adr);
   if (isNil(x = EVAL1((cdr(x)))))
      return boxRect(adr);
   unBoxRect(x,(rect*)adr);
   return x;
}

static void bounds(x,p1,p2)
register pico x;
register point *p1, *p2;
{
   register number n;

   if (isCell(x)) {
      if (isNum(car(x))) {
         if ((n = unBox(car(x))) < p1->h)
            p1->h = n;
         if (n > p2->h)
            p2->h = n;
         if ((n = unBox(cdr(x))) < p1->v)
            p1->v = n;
         if (n > p2->v)
            p2->v = n;
      }
      else {
         bounds(car(x),p1,p2);
         bounds(cdr(x),p1,p2);
      }
   }
}

pico Bounds(x)
register pico x;
{
   register pico y;
   point pt1,pt2;

   pt1.h = pt1.v = MAXNUM;
   pt2.h = pt2.v = MAXNEG;
   NEEDCELL(x);
   do
      bounds(EVAL1(x), &pt1, &pt2);
   while (isCell(x = cdr(x)));
   return newCell2(boxNum(pt1.h), boxNum(pt1.v),
                        boxNum(pt2.h), boxNum(pt2.v));
}

/* Beziers */
static void drawBez(px,py,qx,qy,rx,ry,sx,sy)
long px,py,qx,qy,rx,ry,sx,sy;
{
   long d,ax,ay,bx,by,cx,cy,dx,dy,ex,ey;

   ax = ry - qy;
   ay = qx - rx;
   if (d = dist(px,py,sx,sy)) {
      if (abs(muldiv(qx-px,ax,d) + muldiv(qy-py,ay,d)) +
                  abs(muldiv(rx-sx,ax,d) + muldiv(ry-sy,ay,d)) < drawN)
         apply4(drawFoo,boxNum(px),boxNum(py),boxNum(sx),boxNum(sy));
      else {
         dx = ((ax = (px+qx)/2) + (bx = (qx+rx)/2)) / 2;
         dy = ((ay = (py+qy)/2) + (by = (qy+ry)/2)) / 2;
         ex = (bx + (cx = (rx+sx)/2)) / 2;
         ey = (by + (cy = (ry+sy)/2)) / 2;
         drawBez(px,py,ax,ay,dx,dy,(dx+ex)/2,(dy+ey)/2);
         drawBez((dx+ex)/2,(dy+ey)/2,ex,ey,cx,cy,sx,sy);
      }
   }
}

static void crossBez(px,py,qx,qy,rx,ry,sx,sy)
long px,py,qx,qy,rx,ry,sx,sy;
{
   long d,ax,ay,bx,by,cx,cy,dx,dy,ex,ey;

   if (crLin<py && crLin<qy && crLin<ry && crLin<sy ||
                      py<crLin && qy<crLin && ry<crLin && sy<crLin)
      return;
   ax = ry - qy;
   ay = qx - rx;
   if (!(d = dist(px,py,sx,sy)))
      return;
   ax = ry - qy;
   if (abs(muldiv(qx-px,ax,d) + muldiv(qy-py,ay,d)) +
         abs(muldiv(rx-sx,ax,d) + muldiv(ry-sy,ay,d)) < crossN) {
      cross(px,py,qx,qy);
      cross(qx,qy,rx,ry);
      cross(rx,ry,sx,sy);
   }
   else {
      dx = ((ax = (px+qx)/2) + (bx = (qx+rx)/2)) / 2;
      dy = ((ay = (py+qy)/2) + (by = (qy+ry)/2)) / 2;
      ex = (bx + (cx = (rx+sx)/2)) / 2;
      ey = (by + (cy = (ry+sy)/2)) / 2;
      crossBez(px,py,ax,ay,dx,dy,(dx+ex)/2,(dy+ey)/2);
      crossBez((dx+ex)/2,(dy+ey)/2,ex,ey,cx,cy,sx,sy);
   }
}

bool onBez(n,px,py,qx,qy,rx,ry,sx,sy)
register int n;
long px,py,qx,qy,rx,ry,sx,sy;
{
   register long a;
   long ax,ay,bx,by,cx,cy,dx,dy,ex,ey;

   a = onPt.h + onN;
   if (a<px && a<qx && a<rx && a<sx)
      return NO;
   a = onPt.h - onN;
   if (a>px && a>qx && a>rx && a>sx)
      return NO;
   a = onPt.v + onN;
   if (a<py && a<qy && a<ry && a<sy)
      return NO;
   a = onPt.v - onN;
   if (a>py && a>qy && a>ry && a>sy)
      return NO;
   if (--n < 0)
      return YES;  /* onLine(px,py,qx,qy) || onLine(qx,qy,rx,ry) ||
                                             onLine(rx,ry,sx,sy); */
   dx = ((ax = (px+qx)/2) + (bx = (qx+rx)/2)) / 2;
   dy = ((ay = (py+qy)/2) + (by = (qy+ry)/2)) / 2;
   ex = (bx + (cx = (rx+sx)/2)) / 2;
   ey = (by + (cy = (ry+sy)/2)) / 2;
   return
      onBez(n,px,py,ax,ay,dx,dy,(dx+ex)/2,(dy+ey)/2) ||
            onBez(n,(dx+ex)/2,(dy+ey)/2,ex,ey,cx,cy,sx,sy);
}

long bezRoute(n,px,py,qx,qy,rx,ry,sx,sy)
register int n;
long px,py,qx,qy,rx,ry,sx,sy;
{
   long ax,ay,bx,by,cx,cy,dx,dy,ex,ey;

   if (--n < 0)
      return pythag(px-qx,py-qy) +
               pythag(qx-rx,qy-ry) + pythag(rx-sx,ry-sy);
   qx = (ax = (qx - px))*2;
   qy = (ay = (qy - py))*2;
   rx = (rx - px)*2;
   ry = (ry - py)*2;
   sx = (sx - px)*2;
   sy = (sy - py)*2;
   dx = (ax + (bx = (qx+rx)/2)) / 2;
   dy = (ay + (by = (qy+ry)/2)) / 2;
   ex = (bx + (cx = (rx+sx)/2)) / 2;
   ey = (by + (cy = (ry+sy)/2)) / 2;
   return (bezRoute(n,0,0,ax,ay,dx,dy,(dx+ex)/2,(dy+ey)/2) +
         bezRoute(n,(dx+ex)/2,(dy+ey)/2,ex,ey,cx,cy,sx,sy)) / 2;
}

pico Bezier(x)
pico x;
{
   number px,py,vx,vy,wx,wy,sx,sy;

   px = nextNum(&x);
   py = nextNum(&x);
   vx = nextNum(&x);
   vy = nextNum(&x);
   wx = nextNum(&x);
   wy = nextNum(&x);
   sx = nextNum(&x);
   sy = nextNum(&x);
   return newCell2(
      boxNum((3*sx - 10*px + 24*vx - 8*wx) / 9),
      boxNum((3*sy - 10*py + 24*vy - 8*wy) / 9),
      boxNum((3*px - 10*sx + 24*wx - 8*vx) / 9),
      boxNum((3*py - 10*sy + 24*wy - 8*vy) / 9) );
}

/* Grafs */
pico graf(x,foo)
register pico x;
pico foo;
{
   register pico y,z;

   if (!isCell(x))
      return x;
   push(y = newCell(apply1(foo,car(x)),nilSym));
   while (isCell(x = cdr(x))) {
      if (isNum(car(z = car(x))))
         cdr(y) = newCell(apply1(foo,z),nilSym);
      else {
         push(z = apply1(foo,car(z)));
         cdr(y) = newCell(newCell(z,apply1(foo,cdr(car(x)))),nilSym);
         drop();
      }
      y = cdr(y);
   }
   return pop();
}

pico Graf(x)
register pico x;
{
   register pico y;

   if (isNil(y = EVAL1(x)))  /* Graf */
      return y;
   push(y);
   x = cdr(x);
   push(EVAL1(x));  /* foo */
   push(x = newCell(graf(car(y),tos),nilSym));
   while (isCell(y = cdr(y))) {
      cdr(x) = newCell(graf(car(y),nos),nilSym);
      x = cdr(x);
   }
   x = pop();
   drop2();
   return x;
}

pico Draw(x)
pico x;
{
   register pico y,z,p;
   pico bezFoo;
   number n;

   n = nextNum(&x); /* Resolution */
   push(EVAL1(x)); /* Graf */
   x = cdr(x);
   push(drawFoo = EVAL1(x)); /* Drawing function */
   bezFoo = isCell(x = cdr(x))? EVAL1(x) : NULL;
   x = nos;
   drawN = n;
   while (isCell(x)) {
      y = car(x),  x = cdr(x);
      p = car(y);
      while (isCell(y = cdr(y))) {
         if (isNum(car(z = car(y)))) {
            apply4(drawFoo, car(p), cdr(p), car(z), cdr(z));
            p = z;
         }
         else {
            y = cdr(y);
            if (!bezFoo || apply4(bezFoo,p,car(z),cdr(z),car(y))==nilSym)
               drawBez(unBox(car(p)), unBox(cdr(p)),
                       unBox(car(car(z))), unBox(cdr(car(z))),
                       unBox(car(cdr(z))), unBox(cdr(cdr(z))),
                       unBox(car(car(y))), unBox(cdr(car(y))) );
            p = car(y);
         }
      }
   }
   drop2();
   return tSym;
}

static void buildCross(x)
pico x;
{
   register pico y,z;
   register number h1,v1,h2,v2;

   crCnt = 0;
   if (!crLst)
      crLst = (long*)malloc((crSiz=32) * sizeof(long));
   while (isCell(x)) {
      y = car(x),  x = cdr(x);
      z = car(y);
      h1 = unBox(car(z));
      v1 = unBox(cdr(z));
      while (isCell(y = cdr(y))) {
         if (isNum(car(z = car(y))))
            cross(h1, v1, h2=unBox(car(z)), v2=unBox(cdr(z)));
         else {
            y = cdr(y);
            crossBez(h1, v1, unBox(car(car(z))), unBox(cdr(car(z))),
                        unBox(car(cdr(z))), unBox(cdr(cdr(z))),
                        h2 = unBox(car(car(y))), v2 = unBox(cdr(car(y))));
         }
         h1 = h2;
         v1 = v2;
      }
   }
}

pico Inside(x)
pico x;
{
   pico y;
   register number h,v,i;
   register bool res;
   number l,n;

   y = EVAL1(x);
   x = cdr(x);
   needPoint(y);
   h = unBox(car(y)); /* Point */
   l = unBox(cdr(y));
   n = nextNum(&x); /* Resolution */
   x = (EVAL1(x)); /* Graf */
   crLin = l;
   crossN = n;
   buildCross(x);
   res = NO;
   for (i = 0; i < crCnt && crLst[i] <= h; ++i)
      res = !res;
   return res? tSym : nilSym;
}

pico Atari(x)
register pico x;
{
   register pico y;
   point pt,pt1,pt2;
   number n;

   unBoxPoint(EVAL1(x),&pt); /* Point */
   x = cdr(x);
   n = num(EVAL1(x)); /* Resolution */
   NEEDNUM(n);
   n = unBox(n);
   x = cdr(x);
   unBoxPoint(EVAL1(x),&pt1);
   if (dist(pt.h,pt.v,pt1.h,pt1.v) <= n)
      return tSym;
   x = cdr(x);
   y = EVAL1(x);
   NEEDCELL(y);
   x = EVAL1(cdr(x));
   onPt = pt;
   onN = n;
   if (isNum(car(y)))
      return boxBool(onLine(pt1.h, pt1.v, unBox(car(y)), unBox(cdr(y))));
   return boxBool(
         onBez(3, pt1.h, pt1.v, unBox(car(car(y))), unBox(cdr(car(y))),
                      unBox(car(cdr(y))), unBox(cdr(cdr(y))),
                      unBox(car(x)), unBox(cdr(x)) ) );
}

pico Onside(x)
register pico x;
{
   register pico y,z;
   register number h1,v1,h2,v2;
   point pt;
   number n;

   unBoxPoint(EVAL1(x),&pt);
   x = cdr(x);
   push(EVAL1(x));
   n = 2;
   x = cdr(x);
   if (!isNil(x = EVAL1(x))) {
      NEEDNUM(x);
      n = unBox(x);
   }
   x = pop();
   onPt = pt;
   onN = n;
   while (isCell(x)) {
      y = car(x),  x = cdr(x);
      z = car(y);
      h1 = unBox(car(z));
      v1 = unBox(cdr(z));
      if (dist(h1,v1,onPt.h,onPt.v) <= onN)
         return tSym;
      while (isCell(y = cdr(y))) {
         if (isNum(car(z = car(y)))) {
            if (onLine(h1, v1, h2 = unBox(car(z)), v2 = unBox(cdr(z))))
               return tSym;
         }
         else {
            y = cdr(y);
            if (onBez(3, h1, v1, unBox(car(car(z))), unBox(cdr(car(z))),
                      unBox(car(cdr(z))), unBox(cdr(cdr(z))),
                      h2 = unBox(car(car(y))), v2 = unBox(cdr(car(y)))))
               return tSym;
         }
         h1 = h2;
         v1 = v2;
      }
   }
   return nilSym;
}

pico Scan(x)
pico x;
{
   register long i;
   pico y;
   number l,n;

   l = nextNum(&x); /* Scan line */
   n = nextNum(&x); /* Resolution */
   y = EVAL1(x); /* Graf */
   crLin = l;
   crossN = n;
   buildCross(y);
   x = cdr(x);
   push(EVAL1(x)); /* Function */
   /* Apply function to pairs of crosspoints */
   x = nilSym;
   for (i = 0; i < crCnt; i += 2)
      x = apply2(tos, boxNum(crLst[i]), boxNum(crLst[i+1]));
   drop();
   return x;
}

pico Route(x)
register pico x;
{
   register pico y,z;
   register number h1,v1,h2,v2;
   number n;

   x = EVAL1(x);
   n = 0;
   while (isCell(x)) {
      y = car(x),  x = cdr(x);
      z = car(y);
      h1 = unBox(car(z));
      v1 = unBox(cdr(z));
      while (isCell(y = cdr(y))) {
         if (isNum(car(z = car(y))))
            n += pythag(h1 - (h2=unBox(car(z))), v1 - (v2=unBox(cdr(z))));
         else {
            y = cdr(y);
            n += bezRoute(5,h1,v1,unBox(car(car(z))),unBox(cdr(car(z))),
                        unBox(car(cdr(z))), unBox(cdr(cdr(z))),
                        h2 = unBox(car(car(y))), v2 = unBox(cdr(car(y))));
         }
         h1 = h2;
         v1 = v2;
      }
   }
   return boxNum(n);
}

pico Plot(x)
register pico x;
{
#if 1
   err("Plot not implemented yet");
#else
   register pico y,z;
   register bool xchg;
   register number i, e, sh,sv, dh,dv, h,v, h0,v0, temp;

   push(z = EVAL1(x)); /* Graf */
   NEEDCELL(z);
   needRect(y = car(z));
   push(EVAL1(cdr(x))); /* Foo */
   y = car(y);
   h0 = unBox(car(y));
   v0 = unBox(cdr(y));
   z = cdr(z);
   do {
      NEEDCELL(z);
      y = car(z);
      NEEDCELL(y);
      while (isCell(cdr(y))) {
         x = car(y);
         y = cdr(y);
         h = h0 + ptH(x);
         v = v0 + ptV(x);
         dh = h0 + ptH(car(y)) - h;
         dv = v0 + ptV(car(y)) - v;
         sh = 0;
         if (dh > 0)
            sh = 1;
         else if (dh < 0) {
            dh = -dh;
            sh = -1;
         }
         sv = 0;
         if (dv > 0)
            sv = 1;
         else if (dv < 0) {
            dv = -dv;
            sv = -1;
         }
         xchg = NO;
         if (dv > dh) {
            xchg = YES;
            temp = dh,  dh = dv,  dv = temp;
         }
         e = 2*dv - dh;
         for (i = 0; i < dh; ++i) {
            x = apply2(tos, boxNum(h), boxNum(v));
            if (e >= 0) {
               if (xchg)
                  h += sh;
               else
                  v += sv;
               e -= 2*dh;
            }
            if (xchg)
               v += sv;
            else
               h += sh;
            e += 2*dv;
         }
      }
   } while (isCell(z = cdr(z)));
   drop2();
   return x;
#endif
}
