/* picoView.c
 * 16mar90abu
 */

#include "pico.h"

#define VIEWSIZE 6000.0
#define CLIP 16383.0
#define SCALE 100.0

/* Camera parameters */
static vector camLoc;
static vector spot;
static double focLen;
static int orgH;
static int orgV;

/* Line drawing arguments */
static number viewLength,lastX,lastY;

/* Transformation coefficients */
static double sinP,cosP, sinY,cosY;
static double coeff1,coeff2,coeff4,coeff5,coeff6,coeff7,coeff8,coeff9;

/* Prototypes */
static void qdClip(double,double);

/* Set Camera parameters */
pico Camera(x)
register pico x;
{
#if MADA
	register pico y;
	double dx,dy,dz, dist2,dist3;

	y = EVAL1(x);
	NEEDNUM(y);
	focLen = (double)unBox(y);		/* Focal length */
	x = cdr(x);
	unBoxVector(EVAL1(x),&camLoc);	/* Camera location */
	x = cdr(x);
	unBoxVector(EVAL1(x),&spot);	/* View spot */

	orgH = (thePort->portRect.left + thePort->portRect.right) / 2;
	orgV = (thePort->portRect.top + thePort->portRect.bottom) / 2;

	dx = spot.x - camLoc.x;
	dy = spot.y - camLoc.y;
	dz = spot.z - camLoc.z;
	dist2 = sqrt(dx*dx + dy*dy);
	dist3 = sqrt(dx*dx + dy*dy + dz*dz);
	if (dist2 == 0.0) {
		sinY = 0.0;
		cosY = 1.0;
	}
	else {
		sinY = dy / dist2;
		cosY = dx / dist2;
	}
	sinP = dz / dist3;
	cosP = dist2 / dist3;

	coeff1 = -sinY;
	coeff2 = cosY;
	coeff4 = cosY*sinP;
	coeff5 = sinY*sinP;
	coeff6 = -cosP;
	coeff7 = cosY*cosP;
	coeff8 = sinY*cosP;
	coeff9 = sinP;
#endif
	return tSym;
}

/* Poor man's clipping */
static void qdClip(x,y)
register double x,y;
{
	register number px,py,pz;

	if (x < -CLIP) {
		y = y * -CLIP/x;
		x = -CLIP;
	}
	else if (x > CLIP) {
		y = y * CLIP/x;
		x = CLIP;
	}
	if (y < -CLIP) {
		x = x * -CLIP/y;
		y = -CLIP;
	}
	else if (y > CLIP) {
		x = x * CLIP/y;
		y = CLIP;
	}
	px = orgH + (number)x;
	py = orgV + (number)y;
	if (px != lastX || py != lastY) {
		nos = newCell(boxNum(py),nos);
		nos = newCell(boxNum(px),nos);
		++viewLength;
	}
	lastX = px;
	lastY = py;
}

pico View(x)
register pico x;
{
	register pico y,rgb;
	register double vx,vy,vz,n;
	double px1,py1,pz1,px2,py2,pz2,zMax,zMin;

	push(x = EVAL1(x));
	push(nilSym);
	while (isCell(x)) {
		y = car(x), x = cdr(x);
		viewLength = 0;
		lastX = lastY = MAXNUM; /* Init with illegal value */
		rgb = car(y);
		y = cdr(y);
		vx = (double)unBox(car(y)) - camLoc.x;
		y = cdr(y);
		vy = (double)unBox(car(y)) - camLoc.y;
		y = cdr(y);
		vz = (double)unBox(car(y)) - camLoc.z;
		y = cdr(y);
		px1 = coeff1*vx + coeff2*vy;
		py1 = coeff4*vx + coeff5*vy + coeff6*vz;
		pz1 = coeff7*vx + coeff8*vy + coeff9*vz;
		push(nilSym); /* Screen list */
		push(newCell(boxDouble(pz1*SCALE),nilSym)); /* Eye list */
		tos = newCell(boxDouble(py1*SCALE),tos);
		tos = newCell(boxDouble(px1*SCALE),tos);
		zMax = zMin = pz1;
		while (isCell(y)) {
			vx = (double)unBox(car(y)) - camLoc.x;
			y = cdr(y);
			vy = (double)unBox(car(y)) - camLoc.y;
			y = cdr(y);
			vz = (double)unBox(car(y)) - camLoc.z;
			y = cdr(y);
			px2 = coeff1*vx + coeff2*vy;
			py2 = coeff4*vx + coeff5*vy + coeff6*vz;
			pz2 = coeff7*vx + coeff8*vy + coeff9*vz;
			tos = newCell(boxDouble(pz2*SCALE),tos);
			tos = newCell(boxDouble(py2*SCALE),tos);
			tos = newCell(boxDouble(px2*SCALE),tos);
			if (pz2 > zMax)
				zMax = pz2;
			else if (pz2 < zMin)
				zMin = pz2;
			if (pz1 > 1.0  ||  pz2 > 1.0) {
				if (pz1 <= 1.0) {
					n = (1 - pz1) / (pz2 - pz1);
					qdClip(focLen * (px1 + n * (px2-px1)), focLen * (py1 + n * (py2-py1)));
					qdClip(focLen*px2/pz2, focLen*py2/pz2);
				}
				else if (pz2 <= 1.0) {
					n = (1 - pz2) / (pz1 - pz2);
					qdClip(focLen*px1/pz1, focLen*py1/pz1);
					qdClip(focLen * (px2 + n * (px1-px2)), focLen * (py2 + n * (py1-py2)));
				}
				else {
					qdClip(focLen*px1/pz1, focLen*py1/pz1);
					qdClip(focLen*px2/pz2, focLen*py2/pz2);
				}
			}
			px1 = px2;
			py1 = py2;
			pz1 = pz2;
		}
		if (viewLength < 3) {
			drop2();
		}
		else {
			nos = newCell(rgb,nos);
			tos = newCell(boxDouble(zMin*SCALE),tos);
			y = newCell(boxDouble(zMax*SCALE),pop());
			y = newCell(xpop(),y); /* Cons screen and eye lists */
			y = newCell(tSym,y); /* Flag */
			tos = newCell(y,tos);
		}
	}
	x = pop();
	drop();
	return x;
}

pico Paint(x)
register pico x;
{
#if MADA
	register number n,h,v;
	bool transparent;
	RGBColor rgb;
	PolyHandle p;

	PenNormal();
	x = car(cdr(EVAL1(x)));
	if (isNil(car(x)))
		transparent = YES;
	else {
		transparent = NO;
		unBoxColor(car(x),&rgb);
	}
	x = cdr(x);
	p = OpenPoly();
	h = unBox(car(x));
	x = cdr(x);
	MoveTo(h, v = unBox(car(x)));
	while (isCell(x = cdr(x))) {
		n = unBox(car(x));
		x = cdr(x);
		LineTo(n,unBox(car(x)));
	}
	LineTo(h,v);
	ClosePoly();
	if (!transparent) {
		RGBForeColor(&rgb);
		PaintPoly(p);
	}
	rgb.red = rgb.green = rgb.blue = 0;
	RGBForeColor(&rgb);
	FramePoly(p);
	KillPoly(p);
#endif
	return tSym;
}

pico Hiding(x)
register pico x;
{
	register pico face1,face2;
	pico xMax1,xMin1,yMax1,yMin1,xMax2,xMin2,yMax2,yMin2;
	register double n,a,b,c,d;
	double x1,y1,z1,x2,y2,z2,x3,y3,z3;

	push(face1 = cdr(EVAL1(x)));
	x = cdr(x);
	face2 = cdr(EVAL1(x));
	drop();

	x = cdr(car(face1));
	xMax1 = xMin1 = car(x);
	x = cdr(x);
	yMax1 = yMin1 = car(x);
	x = cdr(x);
	do {
		if (num(car(x)) > num(xMax1))
			xMax1 = car(x);
		else if (num(car(x)) < num(xMin1))
			xMin1 = car(x);
		x = cdr(x);
		if (num(car(x)) > num(yMax1))
			yMax1 = car(x);
		else if (num(car(x)) < num(yMin1))
			yMin1 = car(x);
	} while (isCell(x = cdr(x)));
	x = cdr(car(face2));
	xMax2 = xMin2 = car(x);
	x = cdr(x);
	yMax2 = yMin2 = car(x);
	x = cdr(x);
	do {
		if (num(car(x)) > num(xMax2))
			xMax2 = car(x);
		else if (num(car(x)) < num(xMin2))
			xMin2 = car(x);
		x = cdr(x);
		if (num(car(x)) > num(yMax2))
			yMax2 = car(x);
		else if (num(car(x)) < num(yMin2))
			yMin2 = car(x);
	} while (isCell(x = cdr(x)));
	if (num(xMin1) >= num(xMax2)  ||  num(xMin2) >= num(xMax1)  ||
							num(yMin1) >= num(yMax2)  ||  num(yMin2) >= num(yMax1))
		return nilSym; /* Bounding boxes disjoint in x or y */
/* prString("Not disjoint\n"); */
	x = cdr(cdr(cdr(face2))), x1 = (double)unBox(car(x));
	x = cdr(x), y1 = (double)unBox(car(x));
	x = cdr(x), z1 = (double)unBox(car(x));
	x = cdr(x), x2 = (double)unBox(car(x));
	x = cdr(x), y2 = (double)unBox(car(x));
	x = cdr(x), z2 = (double)unBox(car(x));
	x = cdr(x), x3 = (double)unBox(car(x));
	x = cdr(x), y3 = (double)unBox(car(x));
	x = cdr(x), z3 = (double)unBox(car(x));
	a = y1*(z2-z3) + y2*(z3-z1) + y3*(z1-z2);
	b = x1*(z3-z2) + x2*(z1-z3) + x3*(z2-z1);
	c = x1*(y2-y3) + x2*(y3-y1) + x3*(y1-y2);
	n = sqrt(a*a + b*b + c*c);
	a /= n;
	b /= n;
	c /= n;
	d = (x1*y3*z2 + x2*y1*z3 + x3*y2*z1 - x1*y2*z3 - x2*y3*z1 - x3*y1*z2) / n;
	x = cdr(cdr(cdr(face1)));
	do {
		n = a * (double)unBox(car(x)), x = cdr(x);
		n += b * (double)unBox(car(x)), x = cdr(x);
		n += c * (double)unBox(car(x)) + d;
		if (n>SCALE && d>SCALE || n<-SCALE && d<-SCALE) /* n and d same sign? */
			goto fail1;
	} while (isCell(x = cdr(x)));
	return nilSym; /* First wholly behind plane of second */
fail1:
/* prString("First not behind second\n"); */
	x = cdr(cdr(cdr(face1))), x1 = (double)unBox(car(x));
	x = cdr(x), y1 = (double)unBox(car(x));
	x = cdr(x), z1 = (double)unBox(car(x));
	x = cdr(x), x2 = (double)unBox(car(x));
	x = cdr(x), y2 = (double)unBox(car(x));
	x = cdr(x), z2 = (double)unBox(car(x));
	x = cdr(x), x3 = (double)unBox(car(x));
	x = cdr(x), y3 = (double)unBox(car(x));
	x = cdr(x), z3 = (double)unBox(car(x));
	a = y1*(z2-z3) + y2*(z3-z1) + y3*(z1-z2);
	b = x1*(z3-z2) + x2*(z1-z3) + x3*(z2-z1);
	c = x1*(y2-y3) + x2*(y3-y1) + x3*(y1-y2);
	n = sqrt(a*a + b*b + c*c);
	a /= n;
	b /= n;
	c /= n;
	d = (x1*y3*z2 + x2*y1*z3 + x3*y2*z1 - x1*y2*z3 - x2*y3*z1 - x3*y1*z2) / n;
	x = cdr(cdr(cdr(face2)));
	do {
		n = a * (double)unBox(car(x)), x = cdr(x);
		n += b * (double)unBox(car(x)), x = cdr(x);
		n += c * (double)unBox(car(x)) + d;
		if (n>SCALE && d<-SCALE || n<-SCALE && d>SCALE) /* n and d different sign? */
			goto fail2;
	} while (isCell(x = cdr(x)));
	return nilSym; /* Second wholly before plane of first */
fail2:
/* prString("Second not before first\n"); */
	return tSym;
}

pico Horizon(x)
pico x;
{
#if MADA
	RGBColor rgb;
	rect r;
	PolyHandle p;
	register integer n;

	nextColor(&x,&rgb);
	nextRect(&x,&r);
	p = OpenPoly();
	n = orgV + (integer)(sinP*focLen / cosP);
	MoveTo(r.left-1, n);
	LineTo(r.right+1,n);
	LineTo(r.right+1,r.bottom+1);
	LineTo(r.left-1, r.bottom+1);
	LineTo(r.left-1, n);
	ClosePoly();
	RGBForeColor(&rgb);
	PaintPoly(p);
	KillPoly(p);
	return tSym;
#endif
}

#if 0
/* Draw WireFrame */
pico View(x)
pico x;
{
	register pico y;
	register matHandle h;
	register number n;
	vector pos,v;
	double dx,dy,dz,cx,cy,cz,scale;
	register double px1,py1,pz1,px2,py2,pz2,d;

	push(EVAL1(x));					/* WireObject */
	x = cdr(x);
	scale = nextDouble(&x,VIEWSIZE);/* Scale */
	nextVector(&x,&pos,X_SCALE);	/* Position */
	h  =  (isCell(x))? (matHandle)nextDynamo(&x) : NULL; /* Optional matrix */
	x = pop();

	dx = pos.x - camLoc.x;
	dy = pos.y - camLoc.y;
	dz = pos.z - camLoc.z;

	/* Simple "behind check" */
	if (coeff7*dx + coeff8*dy + coeff9*dz + 1024.0*scale  <=  1.0)
		return nilSym;

	n = unBox(car(x));
	cx = (double)(n >> 20 & 0x3FF);
	cy = (double)(n >> 10 & 0x3FF);
	cz = (double)(n & 0x3FF);
	while (isCell(x = cdr(x))) {
		y = car(x);
		n = unBox(car(y));
		v.x = ((double)(n >> 20 & 0x3FF) - cx) * scale;
		v.y = ((double)(n >> 10 & 0x3FF) - cy) * scale;
		v.z = ((double)(n & 0x3FF) - cz) * scale;
		if (h) {
			vector tmp;
			tmp = v;
			transform(*h,&tmp,&v);
		}
		v.x += dx;
		v.y += dy;
		v.z += dz;
		px1  =  (coeff1*v.x + coeff2*v.y);
		py1  =  (coeff4*v.x + coeff5*v.y + coeff6*v.z);
		pz1  =  (coeff7*v.x + coeff8*v.y + coeff9*v.z);
		while (isCell(y = cdr(y))) {
			n = unBox(car(y));
			v.x = ((double)(n >> 20 & 0x3FF) - cx) * scale;
			v.y = ((double)(n >> 10 & 0x3FF) - cy) * scale;
			v.z = ((double)(n & 0x3FF) - cz) * scale;
			if (h) {
				vector tmp;
				tmp = v;
				transform(*h,&tmp,&v);
			}
			v.x += dx;
			v.y += dy;
			v.z += dz;
			px2  =  (coeff1*v.x + coeff2*v.y);
			py2  =  (coeff4*v.x + coeff5*v.y + coeff6*v.z);
			pz2  =  (coeff7*v.x + coeff8*v.y + coeff9*v.z);
			if (pz1 > 1.0  ||  pz2 > 1.0) {
				if (pz1 <= 1.0) {
					d = (1 - pz1) / (pz2 - pz1);
					qdClip(focLen * (px1 + d * (px2-px1)), focLen * (py1 + d * (py2-py1)));
					MoveTo(qdX,qdY);
					qdClip(focLen*px2/pz2, focLen*py2/pz2);
					LineTo(qdX,qdY);
				}
				else if (pz2 <= 1.0) {
					d = (1 - pz2) / (pz1 - pz2);
					qdClip(focLen*px1/pz1, focLen*py1/pz1);
					MoveTo(qdX, qdY);
					qdClip(focLen * (px2 + d * (px1-px2)), focLen * (py2 + d * (py1-py2)));
					LineTo(qdX,qdY);
				}
				else {
					qdClip(focLen*px1/pz1, focLen*py1/pz1);
					MoveTo(qdX,qdY);
					qdClip(focLen*px2/pz2, focLen*py2/pz2);
					LineTo(qdX,qdY);
				}
			}
			px1 = px2;
			py1 = py2;
			pz1 = pz2;
		}
	}
	return tSym;
}
#endif
