/* picoBig.c
 * 21dec92abu
 */

#pragma segment picoBig

#include "pico.h"

typedef unsigned long bigDig;

#define setCarry(n) (carry = ((long)(n) << 1) < 0)
#define unBig(n)    ((bigDig)(n) >> 2)

static bigDig carry;

static void bigUnder(void);
static void bigAdd(pico,pico);
static void bigSub(pico,pico);
static void bigInc(pico,bigDig);
static void bigDec(pico,bigDig);
static void bigMul2(pico);
static void bigMul5(pico);
static void bigDiv2(pico);
static bool bigLeq(pico,pico);

static void bigUnder()
{
   err("BigNum underflow");
}

pico Big(x)
register pico x;
{
	cell c1;

   x = EVAL1(x);
   NEEDSTRING(x);
   push(newCell(boxNum(unBox(car(x)) - '0'),nilSym), c1);
   if (car(x) == boxNum('0'))
      while (isCell(x = cdr(x))) {
         bigMul2(tos(c1)); /* Times 16 */
         bigMul2(tos(c1));
         bigMul2(tos(c1));
         bigMul2(tos(c1));
         bigInc(tos(c1),
				car(x)>boxNum(9)? unBox(car(x))-'7' : unBox(car(x))-'0' );
      }
   else
      while (isCell(x = cdr(x))) {
         bigMul5(tos(c1)); /* Times 10 */
         bigMul2(tos(c1));
         bigInc(tos(c1), unBox(car(x)) - '0');
      }
   return pop(c1);
}

static void bigAdd(src,dst)
register pico src,dst;
{
   register pico x;
   register bigDig n;

   setCarry(n = unBig(car(src)) + unBig(car(dst)));
   car(dst) = boxNum(n);
   src = cdr(src);
   dst = cdr(x = dst);
   loop {
      if (!isCell(src)) {
         while (isCell(dst)) {
            if (!carry)
               return;
            setCarry(n = unBig(car(dst)) + carry);
            car(dst) = boxNum(n);
            dst = cdr(x = dst);
         }
         break;
      }
      if (!isCell(dst)) {
         do {
            setCarry(n = unBig(car(src)) + carry);
            cdr(x) = newCell(boxNum(n),nilSym);
            x = cdr(x);
         } while (isCell(src = cdr(src)));
         break;
      }
      setCarry(n = unBig(car(src)) + unBig(car(dst)) + carry);
      car(dst) = boxNum(n);
      src = cdr(src);
      dst = cdr(x = dst);
   }
   if (carry)
      cdr(x) = newCell(ONE,nilSym);
}

static void bigSub(src,dst)
register pico src,dst;
{
   register pico x,y;
   register bigDig n;

   setCarry(n = unBig(car(dst)) - unBig(car(src)));
   car(dst) = boxNum(n);
   src = cdr(src);
   dst = cdr(x = y = dst);
   loop {
      if (!isCell(src)) {
         while (isCell(dst)) {
            if (!carry)
               return;
            setCarry(n = unBig(car(dst)) - carry);
            car(dst) = boxNum(n);
            dst = cdr(x = dst);
         }
         break;
      }
      if (!isCell(dst))
         bigUnder();
      setCarry(n = unBig(car(dst)) - unBig(car(src)) - carry);
      car(dst) = boxNum(n);
      src = cdr(src);
      dst = cdr(x = dst);
   }
   if (carry)
      bigUnder();
   while (car(x) == ZERO &&  x != y) {
      src = y;
      while (cdr(src) != x)
         src = cdr(src);
      cdr(x = src) = nilSym;
   }
}

static void bigInc(x,n)
register pico x;
bigDig n;
{
   register pico y;

   setCarry(n = unBig(car(x)) + n);
   car(x) = boxNum(n);
   while (carry) {
      if (isCell(x = cdr(y = x))) {
         setCarry(n = unBig(car(x)) + 1);
         car(x) = boxNum(n);
      }
      else {
         cdr(y) = newCell(ONE,nilSym);
         break;
      }
   }
}

static void bigDec(x,n)
register pico x;
bigDig n;
{
   register pico y,z;

   setCarry(n = unBig(car(z = x)) - n);
   car(x) = boxNum(n);
   while (isCell(x = cdr(y = x))) {
      if (!carry)
         return;
      setCarry(n = unBig(car(x)) - carry);
      car(x) = boxNum(n);
   }
   if (carry)
      bigUnder();
   while (car(y) == ZERO &&  y != z) {
      x = z;
      while (cdr(x) != y)
         x = cdr(x);
      cdr(y = x) = nilSym;
   }
}

static void bigMul2(x)
register pico x;
{
   register pico y;
   bigDig n;

   setCarry(n = 2*unBig(car(x)));
   car(x) = boxNum(n);
   while (isCell(x = cdr(y = x))) {
      setCarry(n = 2*unBig(car(x)) + carry);
      car(x) = boxNum(n);
   }
   if (carry)
      cdr(y) = newCell(ONE,nilSym);
}

static void bigMul5(x)
register pico x;
{
   register bigDig n,m,ov1,ov2;
   register pico y;

   n = unBig(car(x));
   ov1 = ((long)(m = 4 * n) << 1) < 0;
   if ((long)m < 0)
      ov1 += 2;
   if ((m ^ m+n) & 0xC0000000)
      ++ov1;
   car(x) = boxNum(m+n);
   while (isCell(x = cdr(y = x))) {
      n = unBig(car(x));
      ov2 = ((long)(m = 4 * n) << 1) < 0;
      if ((long)m < 0)
         ov2 += 2;
      if ((m ^ m+n+ov1) & 0xC0000000)
         ++ov2;
      car(x) = boxNum(m+n+ov1);
      ov1 = ov2;
   }
   if (ov1)
      cdr(y) = newCell(boxNum(ov1),nilSym);
}

static void bigDiv2(x)
register pico x;
{
   register pico y,z;

   car(x) = boxNum(unBig(car(z = x)) / 2);
   while (isCell(x = cdr(y = x))) {
      if ((bigDig)car(x) & 4)
         car(y) = (pico)((bigDig)car(y) | 0x80000000);
      car(x) = boxNum(unBig(car(x)) / 2);
   }
   if (car(y) == ZERO &&  y != z) {
      while (cdr(z) != y)
         z = cdr(z);
      cdr(z) = nilSym;
   }
}

static bool bigLeq(x,y)
register pico x,y;
{
   register bool res;
   register pico x1,y1,x2,y2;

   x1 = y1 = nilSym;
   while (isCell(x2 = cdr(x)) && isCell(y2 = cdr(y))) {
      cdr(x) = x1;
      x1 = x;
      x = x2;
      cdr(y) = y1;
      y1 = y;
      y = y2;
   }
   if (isCell(cdr(x)))
      res = NO;
   else if (isCell(cdr(y)))
      res = YES;
   else loop {
      if ((bigDig)car(x) < (bigDig)car(y)) {
         res = YES;
         break;
      }
      if ((bigDig)car(x) > (bigDig)car(y)) {
         res = NO;
         break;
      }
      if (!isCell(x1))
         return YES;
      x2 = cdr(x1);
      cdr(x1) = x;
      x = x1;
      x1 = x2;
      y2 = cdr(y1);
      cdr(y1) = y;
      y = y1;
      y1 = y2;
   }
   while (isCell(x1)) {
      x2 = cdr(x1);
      cdr(x1) = x;
      x = x1;
      x1 = x2;
      y2 = cdr(y1);
      cdr(y1) = y;
      y = y1;
      y1 = y2;
   }
   return res;
}

pico BigAdd(x)
register pico x;
{
	cell c1,c2;

   push(copy1(EVAL1(x)), c1);
   NEEDCELL(tos(c1));
   NEEDNUM(car(tos(c1)));
   while (isCell(x = cdr(x))) {
      push(EVAL1(x),c2);
      if (isNum(tos(c2)))
         bigInc(tos(c1),unBox(tos(c2)));
      else {
         NEEDCELL(tos(c2));
         NEEDNUM(car(tos(c2)));
         bigAdd(tos(c2),tos(c1));
      }
      drop(c2);
   }
   return pop(c1);
}

pico BigSub(x)
register pico x;
{
	cell c1,c2;

   push(copy1(EVAL1(x)), c1);
   NEEDCELL(tos(c1));
   NEEDNUM(car(tos(c1)));
   while (isCell(x = cdr(x))) {
      push(EVAL1(x),c2);
      if (isNum(tos(c2)))
         bigDec(tos(c1),unBox(tos(c2)));
      else {
         NEEDCELL(tos(c2));
         NEEDNUM(car(tos(c2)));
         bigSub(tos(c2),tos(c1));
      }
      drop(c2);
   }
   return pop(c1);
}

pico BigMul(x)
register pico x;
{
   register bigDig n;
	cell c1,c2,c3;

   push(copy1(EVAL1(x)), c1);
   NEEDCELL(tos(c1));
   NEEDNUM(car(tos(c1)));
   while (isCell(x = cdr(x))) {
      push(copy1(EVAL1(x)), c2);
      push(newCell(ZERO,nilSym), c3);
      if (isNum(tos(c2))) {
         n = unBox(tos(c2));
         while (n) {
            if (n & 1)
               bigAdd(tos(c1),tos(c3));
            bigMul2(tos(c1));
            n /= 2;
         }
      }
      else {
         NEEDCELL(tos(c2));
         NEEDNUM(car(tos(c2)));
         while (car(tos(c2)) != ZERO || isCell(cdr(tos(c2)))) {
            if ((bigDig)car(tos(c2)) & 4)
               bigAdd(tos(c1),tos(c3));
            bigMul2(tos(c1));
            bigDiv2(tos(c2));
         }
      }
      tos(c1) = tos(c3);
      drop(c2);
   }
   return pop(c1);
}

pico BigDiv(x)
register pico x;
{
   register long i;
	cell c1,c2,c3;

   push(copy1(EVAL1(x)), c1);
   NEEDCELL(tos(c1));
   NEEDNUM(car(tos(c1)));
   x = cdr(x);
   push(copy1(EVAL1(x)), c2);
   NEEDCELL(tos(c2));
   NEEDNUM(car(tos(c2)));
   x = cdr(x);
   x = EVAL1(x); /* Mod-Flag */
   push(newCell(ZERO,nilSym), c3);
   i = 0;
   while (bigLeq(tos(c2),tos(c1))) {
      bigMul2(tos(c2));
      ++i;
   }
   while (--i >= 0) {
      bigMul2(tos(c3));
      bigDiv2(tos(c2));
      if (bigLeq(tos(c2),tos(c1))) {
         bigSub(tos(c2),tos(c1));
         bigInc(tos(c3),(bigDig)1);
      }
   }
   x = isNil(x)? tos(c3) : newCell(tos(c3),tos(c1));
   drop(c1);
   return x;
}

pico BigSqrt(x)
register pico x;
{
	cell c1,c2,c3;

   push(copy1(EVAL1(x)), c1);
   NEEDCELL(tos(c1));
   NEEDNUM(car(tos(c1)));
   x = cdr(x);
   x = EVAL1(x); /* Round-Flag */
   push(newCell(ONE,nilSym), c2);
   while (bigLeq(tos(c2),tos(c1))) {
      bigMul2(tos(c2));
      bigMul2(tos(c2));
   }
   push(newCell(ZERO,nilSym), c3);
   do {
      bigAdd(tos(c2),tos(c3));
      if (bigLeq(tos(c3),tos(c1))) {
         bigSub(tos(c3),tos(c1));
         bigAdd(tos(c2),tos(c3));
      }
      else
         bigSub(tos(c2),tos(c3));
      bigDiv2(tos(c3));
      bigDiv2(tos(c2));
      bigDiv2(tos(c2));
   } while (car(tos(c2)) != ZERO || isCell(cdr(tos(c2))));
   if (!isNil(x) && !bigLeq(tos(c1),tos(c3)))
      bigInc(tos(c3),(bigDig)1);
   drop(c1);
   return tos(c3);
}

pico BigLeq(x)
register pico x;
{
   register pico y,z;

   y = EVAL1(x);
   NEEDCELL(y);
   NEEDNUM(car(y));
   while (isCell(x = cdr(x))) {
      z = EVAL1(x);
      NEEDCELL(z);
      NEEDNUM(car(z));
      if (!bigLeq(y,z))
         return nilSym;
      y = z;
   }
   return tSym;
}
