/* picoIO.c
 * 12jan92abu
 */

#pragma segment picoIO

#include "pico.h"

/* Prototypes */
static picoFile *getStream(pico);
static void readError(void);
static void writeError(void);
static void closeError(void);
static void picoFlush(picoFile*);
static void picoFill(picoFile*);
static void fileRead(picoFile*);
static void fileWrite(int,picoFile*);
static int nextc(picoFile*);
static int getc(picoFile*);
static void pipeError(void);
static picoFile *picoOpen(pico,int);
static void cmprWhite(int*);
static void doComment(void);
static void prLine(pico,bool);
static pico rdList(void);
static int skipWhite(void);
static bool symChar(int);
static pico readName(int);
static void testEsc(int*);
static bool isAsciiList(pico);
static void prAsciiList(pico);


void setStream(f)
picoFile *f;
{
   stream = f;
}

picoFile *getStream(x)
register pico x;
{
   if (isNil(x))
      return (picoFile*)NULL;
   NEEDNUM(x);
   return (picoFile*)unBoxPtr(x);
}

void readError()
{
	err("File read error");
}

void writeError()
{
	err("Write error");
}

void closeError()
{
   err("File close error");
}

void picoFlush(f)
register picoFile *f;
{
   if (f->dirty) {
      if (!libSeek(f->fd, f->pos)  ||
               libWrite(f->fd, f->buf, (long)f->max) != f->max )
			writeError();
      f->dirty = NO;
   }
}

void picoFill(f)
register picoFile *f;
{
   if ((f->pos = libFPos(f->fd)) < 0 ||
            (f->max = libRead(f->fd, f->buf, (long)BUFSIZE)) < 0) {
      f->max = 0;
		readError();
   }
   f->cnt = 0;
}

void fileRead(f)
register picoFile *f;
{
   if (++f->cnt == f->max) {
      picoFlush(f);
      picoFill(f);
   }
}

void fileWrite(c,f)
int c;
register picoFile *f;
{
   f->buf[f->cnt++] = (uchar)c;
   f->dirty = YES;
   if (f->cnt > f->max)
      f->max = f->cnt;
   if (f->cnt == BUFSIZE) {
      picoFlush(f);
      if ((f->pos = libFPos(f->fd)) < 0)
         err("Write seek error");
      f->cnt = 0;
      f->max = 0;
   }
}

int nextc(f)
register picoFile *f;
{
   if (!f) {
      if (!*lbp) {
         prompt('?');
         getLine(lbp=lBuff, lBuff+LBSIZE-1);
         if (revaLevel  &&  *lbp == EOL)
            lBuff[0] = 'T', lBuff[1] = EOL;
         crlf();
      }
      return (int)*lbp;
   }
	if (!f->max)
		return -1;
   return (int)f->buf[f->cnt];
}

int getc(f)
register picoFile *f;
{
   register int c;

   c = nextc(f);
   if (f)
      fileRead(f);
   else
      ++lbp;
   return c;
}

int chrIn()
{
   register int c;

   if ((c = getc(stream)) < 0)
      err("Read past End of file");
   if (c == EOL  &&  isCell(car(val(loadSym))))
      car(car(val(loadSym))) = (pico)(num(car(car(val(loadSym)))) + 4);
   return c;
}

picoFile *picoOpen(x,flg)
register pico x;
{
   uchar fName[FILENAME];
   int fd;
   picoFile *f;

   bufString(x, fName, (long)FILENAME);
   switch (flg) {
      case 1:
         if ((fd = libRdOpen(fName)) < 0)
            return (picoFile*)NULL;
         break;
      case 2:
         if ((fd = libWrOpen(fName,0)) < 0)
            return (picoFile*)NULL;
         break;
      case 3:
         if ((fd = libRdWrOpen(fName)) < 0)
            return (picoFile*)NULL;
         break;
   }
   if (!(f = (picoFile*)libAlloc((long)sizeof(picoFile))))
		return (picoFile*)NULL;
   f->fd = fd;
   f->dirty = NO;
   picoFill(f);
   val(fileSym) = newCell(newCell(boxPtr(num(f)), x), val(fileSym));
   return f;
}

void closeFile(f)
picoFile *f;
{
   register pico x,y;

   x = boxPtr(num(f));
   y = val(fileSym);
   if (x == car(car(y)))
      val(fileSym) = cdr(y);
   else  {
      while (isCell(cdr(y))) {
         if (x == car(car(cdr(y)))) {
            cdr(y) = cdr(cdr(y));
            goto doClose;
         }
         y = cdr(y);
      }
      errObj(x, "Can't close");
   }
doClose:
   picoFlush(f);
   if (!libClose(f->fd))
		closeError();
   libFree((char *)f);
}

void pipeError()
{
	err("Can't pipe");
}

pico Pipe(x)
register pico x;
{
	register long cnt;
   int c;
   picoFile *src,*dst;
	cell c1;

   if (!(src = picoOpen(EVAL1(x),1)))
		pipeError();
   x = cdr(x);
   if (!(dst = picoOpen(EVAL1(x),2)))
		pipeError();
   x = cdr(x);
	push(EVAL1(x),c1);
	cnt = 0;
   if (isNil(tos(c1))) {
		while (src->max) {
			if (libWrite(dst->fd, src->buf, (long)src->max) != src->max)
				writeError();
			cnt += src->max;
			if ((src->max = libRead(src->fd, src->buf, (long)BUFSIZE)) < 0) {
				src->max = 0;
				readError();
			}
		}
	}
	else {
		while ((c=getc(src))>=0 && (x=apply1(tos(c1),boxNum(c)))!=tSym) {
			if (isNum(x)) {
				++cnt;
				fileWrite((int)unBox(x),dst);
			}
			else {
				while (isCell(x)) {
					++cnt;
					fileWrite((int)unBox(car(x)),dst);
					x = cdr(x);
				}
			}
		}
	}
   drop(c1);
   closeFile(src);
   closeFile(dst);
   return boxNum(cnt);
}

pico Raw(x)
pico x;
{
   uchar fName[FILENAME];
	long pos,cnt;
	char *buf;
   int fd;

	nextString(&x,fName,FILENAME);
	pos = nextNum(&x);
	buf = (char*)nextNum(&x);
	cnt = nextNum(&x);
   if ((fd = libRdOpen(fName)) < 0 || !libSeek(fd,pos))
		return nilSym;
   cnt = libRead(fd,buf,cnt);
   if (!libClose(fd))
		closeError();
	return boxNum(cnt);
}

pico Open(x)
pico x;
{
   picoFile *f;

   if (!(f = picoOpen(EVAL1(x),3)))
      return nilSym;
   return boxPtr(num(f));
}

pico Creat(x)
register pico x;
{
   picoFile *f;

   if (!(f = picoOpen(EVAL1(x),2)))
      return nilSym;
   return boxPtr(num(f));
}

pico Close(x)
register pico x;
{
   x = EVAL1(x);
   NEEDNUM(x);
   closeFile((picoFile*)unBoxPtr(x));
   return x;
}

pico Load(x)
register pico x;
{
   picoFile *str, *streamSave;
   pico fName, f2Name;

   fName  =  isCell(x)?  EVAL1(x) : car(val(editSym));
   if (!isNum(fName) && isSym(fName)) {
      if (isNil(f2Name = EVAL1(cdr(x))))
         x = get(fName,srcFlg);
      else
         x = get(f2Name,fName);
      if (isNil(x))
         errObj(fName, "Can't load");
      fName = cdr(x);
   }
   if (!(str = picoOpen(fName,1)))
      errStrObj(fName, "File open error");
   streamSave = stream;
   setStream(str);
   val(loadSym) = newCell(newCell(ONE, fName), val(loadSym));
   revalo(nilSym);
   val(loadSym) = cdr(val(loadSym));
   closeFile(stream);
   setStream(streamSave);
   return tSym;
}

pico Erase(x)
pico x;
{
   uchar buf[FILENAME];

   bufString(EVAL1(x), buf, (long)FILENAME);
   return boxBool(libUnlink(buf));
}

pico fRename(x)
register pico x;
{
   uchar old[FILENAME];
   uchar new[FILENAME];

   bufString(EVAL1(x), old, (long)FILENAME);
   bufString(EVAL1(cdr(x)), new, (long)FILENAME);
   return boxBool(libLink(old,new));
}

pico Where(x)
pico x;
{
   register picoFile *f;

   f = getStream(EVAL1(x));
   return boxNum(f->pos + f->cnt);
}

pico Seek(x)
register pico x;
{
   register pico y;
   register picoFile *f;

   y = EVAL1(x);
   NEEDNUM(y);
   x = cdr(x);
   f = getStream(EVAL1(x));
   picoFlush(f);
   f->pos = unBox(y);
   if (!libSeek(f->fd, f->pos))
      return nilSym;
   picoFill(f);
   return y;
}

pico FSize(x)
register pico x;
{
	long pos;
   register picoFile *f;

   f = getStream(EVAL1(x));
   picoFlush(f);
   if ((pos = libFSize(f->fd)) < 0)
      return nilSym;
   return boxNum(pos);
}

pico Read(x)
register pico x;
{
   picoFile *streamSave;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   x = read0(YES);
   setStream(streamSave);
   return x;
}

void prLine(x,suppress)
register pico x;
bool suppress;
{
   register int c;

   keyBreak();
   NEEDSTRING(x);
   while (isCell(x)) {
      c = (int)unBox(car(x));
      x = cdr(x);
      if (c >= 0)
         chrOut(c);
      else if (suppress && !isCell(x))
         return;
      else do
         space();
      while (++c);
   }
}

pico Prin2(x)
register pico x;
{
   picoFile *streamSave;
	cell c1;

   push(EVAL1(x),c1);
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   x = pop(c1);
   if (isCell(x) || isNil(x))
      prLine(x,NO);
   else
      prin0(x);
   setStream(streamSave);
   return x;
}

pico PrLine(x)
register pico x;
{
   picoFile *streamSave;
	cell c1;

   push(EVAL1(x),c1);
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   prLine(x = pop(c1),YES);
   crlf();
   setStream(streamSave);
   return x;
}

pico Prin1(x)
register pico x;
{
   picoFile *streamSave;
	cell c1;

   push(EVAL1(x),c1);
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   prin0(x = pop(c1));
   setStream(streamSave);
   return x;
}

pico Print(x)
register pico x;
{
   picoFile *streamSave;
	cell c1;

   push(EVAL1(x),c1);
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   prin0(x = pop(c1));
   crlf();
   setStream(streamSave);
   return x;
}

pico PrHex(x)
register pico x;
{
   picoFile *streamSave;
	cell c1;

   push(EVAL1(x),c1);
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   x = pop(c1);
   NEEDNUM(x);
   prHexNum((unsigned long)(unBox(x)));
   setStream(streamSave);
   return x;
}

pico CutPr(x)
register pico x;
{
   picoFile *streamSave;
	cell c1;

   push(EVAL1(x),c1);
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   cutPr(x = pop(c1));
   setStream(streamSave);
   return x;
}

pico Comment(x)
register pico x;
{
   picoFile *streamSave;
	cell c1;

   push(EVAL1(x),c1);
   x = cdr(x);
   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   chrOut('[');
   prLine(tos(c1),NO);
   chrOut(']');
   space();
   setStream(streamSave);
   return pop(c1);
}

pico Terpri(x)
pico x;
{
   picoFile *streamSave;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   crlf();
   setStream(streamSave);
   return tSym;
}

pico Space(x)
pico x;
{
   picoFile *streamSave;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   space();
   setStream(streamSave);
   return tSym;
}

pico Tab(x)
pico x;
{
   picoFile *streamSave;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   tab();
   setStream(streamSave);
   return tSym;
}

pico Bell(x)
pico x;
{
   picoFile *streamSave;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   chrOut(7);
   setStream(streamSave);
   return tSym;
}

pico Backsp(x)
pico x;
{
   picoFile *streamSave;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   chrOut(8);
   setStream(streamSave);
   return tSym;
}

pico Putc(x)
register pico x;
{
   picoFile *streamSave;
   register int c;
	cell c1;

   push(EVAL1(x),c1);
   streamSave = stream;
   setStream(getStream(EVAL1(cdr(x))));
   if (isNum(x = pop(c1))) {
      if ((c = (int)unBox(x)) >= 0)
         chrOut(c);
      else do
         space();
      while (++c);
   }
   setStream(streamSave);
   return x;
}

pico Nextc(x)
register pico x;
{
   picoFile *streamSave;
   int c;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   if ((c = nextc(stream)) < 0)
      x = nilSym;
   else
      x = boxNum(c);
   setStream(streamSave);
   return x;
}

pico Getc(x)
register pico x;
{
   picoFile *streamSave;
   int c;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   if ((c = getc(stream)) < 0)
      x = nilSym;
   else
      x = boxNum(c);
   setStream(streamSave);
   return x;
}

void cmprWhite(p)
register int *p;
{
   if (*p == ' ')
      *p = -1;
   else if (*p == 9)
      *p = -TABLEN;
}

pico unBufString(s)
uchar *s;
{
   register pico x;
   int c;
	cell c1;

   if (!*s)
      return nilSym;
   c = *s++;
   cmprWhite(&c);
   push(x = newCell(boxNum(c), nilSym), c1);
   while (*s) {
      c = *s++;
      cmprWhite(&c);
      if (c < 0  &&  num(car(x)) < 0)
         car(x) = boxNum(c + unBox(car(x)));
      else {
         cdr(x) = newCell(boxNum(c), nilSym);
         x = cdr(x);
      }
   }
   return pop(c1);
}

pico unBufCntString(cnt,s)
register int cnt;
uchar *s;
{
   register pico x;
   int c;
	cell c1;

   if (!cnt)
      return nilSym;
   c = *s++;
   cmprWhite(&c);
   push(x = newCell(boxNum(c), nilSym), c1);
   while (--cnt > 0) {
      c = *s++;
      cmprWhite(&c);
      if (c < 0  &&  num(car(x)) < 0)
         car(x) = boxNum(c + unBox(car(x)));
      else {
         cdr(x) = newCell(boxNum(c), nilSym);
         x = cdr(x);
      }
   }
   return pop(c1);
}

pico GetLine(x)
register pico x;
{
   picoFile *streamSave;
   int c;
	cell c1;

   streamSave = stream;
   setStream(getStream(EVAL1(x)));
   if ((c = getc(stream)) < 0)
      x = nilSym;
   else if (c == EOL)
      x  =  stream? newCell(boxNum(-1),nilSym) : nilSym;
   else {
      cmprWhite(&c);
      push(x = newCell(boxNum(c), nilSym), c1);
      while ((c = getc(stream)) >= 0 && c != EOL) {
         cmprWhite(&c);
         if (c < 0  &&  num(car(x)) < 0)
            car(x) = boxNum(c + unBox(car(x)));
         else {
            cdr(x) = newCell(boxNum(c), nilSym);
            x = cdr(x);
         }
      }
      x = shareList(pop(c1));
   }
   setStream(streamSave);
   return x;
}

/* Character I/O */
void chrOut(c)
int c;
{
   if (stream)
      fileWrite(c,stream);
   else {
      if (c == EOL) {
         ttyOut(CR);
         ttyOut(NL);
      }
      else
         ttyOut(c);
   }
}

void space()
{
   chrOut(' ');
}

void crlf()
{
   chrOut(EOL);
}

void tab()
{
   prString("   ");
}

void prString(s)
register uchar *s;
{
   while (*s)
      chrOut(*s++);
}

void prNumber(n)
register number n;
{
   char buf[16];
   register int i = 0;

   if (n < 0){
      chrOut('-');
      n = -n;
   }
   do {
      buf[i++] = (char)(n % 10 + '0');
   } while (n /= 10);
   while (--i >= 0)
      chrOut(buf[i]);
}

void hexChar(c)
register int c;
{
   if ((c += '0') > '9')
      c += 7;
   chrOut(c);
}

void prHexNum(n)
register unsigned long n;
{
   char buf[16];
   register int i = 0;

   keyBreak();
   chrOut('0');
   do {
      buf[i++] = (char)(n % 16);
   } while (n /= 16);
   while (--i >= 0)
      hexChar(buf[i]);
}

void prompt(c)
int c;
{
   if (revaLevel)
      chrOut(revaLevel + '0');
   chrOut(c);
   chrOut(' ');
}

/* Skip nested Comments */
void doComment()
{
   int c;

   chrIn();
   if (nextc(stream) == '#') {
      register pico x,y;

      chrIn();
      y = read0(NO);
		if ((x = EVAL(y)) == voidSym)
			errObj(y,"Undefined Conditional");
      if (!isNil(x))
         return;
   }
   while ((c = nextc(stream)) != ']')
      if (c == '[')
         doComment();
      else
         chrIn();
   chrIn();
}

/* Skip White Space */
int skipWhite()
{
   int c;

   loop {
      while ((c = nextc(stream)) <= ' ' || c == ']')
         chrIn();
      if ((c = nextc(stream)) != '[')
         return (char)c;
      doComment();
   }
}

void testEsc(p)
register int *p;
{
   if (*p == '\\')
      *p = chrIn();
   else if (*p == '^')
      *p = chrIn() & 0x1F;
	else
   	cmprWhite(p);
}

bool symChar(c)
register int c;
{
   return (c>='A' && c<='Z' || c=='-' || c>='a' && c<='z' || c>='0' && c<='9');
}

pico readName(c)
register int c;
{
	register pico x;
   register int i;
   bool first;
   number n;
	cell c1;
	static uchar badChar[] = "  -- Bad character";

   i = 4;
   first = YES;
   if (!symChar(c)) {
		badChar[0] = (char)c;
      err(badChar);
	}
   accumulate(c,&n);
   while (symChar(c = nextc(stream))) {
      if (--i < 0) {
         if (first) {
            first = NO;
            push(x = newCell(boxNum(n),nilSym), c1);
         }
         else {
            cdr(x) = newCell(boxNum(n),nilSym);
            x = cdr(x);
         }
         i = 4;
      }
      accumulate(c,&n);
      chrIn();
   }
   while (--i >= 0)
      accumulate(0,&n);
   if (first)
      return boxNum(n);
   cdr(x) = boxNum(n);
   return pop(c1);
}

/* Read one Expression */
pico read0(top)
bool top;
{
   register pico x,y;
   number n;
   int c, sign;
	cell c1;

   do
      skipWhite();
   while ((c = chrIn()) == ')'  ||  c == '>');
   if (top) {
      x = car(val(loadSym));
      loadPos = car(x);
      loadName = cdr(x);
   }
   if (c == '(') {
      x =  rdList();
      if (top &&  nextc(stream) == '>')
         chrIn();
		return x;
   }
   if (c == '<') {
      x = rdList();
      if (chrIn() != '>')
         err("Super parentheses mismatch");
		return x;
   }
   if (c == '\'')
      return share(quoteSym, read0(NO));
   if (c == '\\')
      return boxNum(chrIn());
   if (c == '^')
      return boxNum(chrIn() & 0x1F);
   if (c == '#') {
      y = read0(NO);
      if ((x = EVAL(y)) == voidSym)
         errObj(y,"Undefined Read Macro");
		return x;
   }
   if (c == '"') {
      if ((c = chrIn()) == '"')
         return nilSym;
      testEsc(&c);
      push(x = newCell(boxNum(c), nilSym), c1);
      while ((c = chrIn()) != '"') {
         testEsc(&c);
         if (c < 0  &&  num(car(x)) < 0)
            car(x) = boxNum(c + unBox(car(x)));
         else {
            cdr(x) = newCell(boxNum(c), nilSym);
            x = cdr(x);
         }
      }
      return shareList(pop(c1));
   }
   if (c>='0' && c<='9' ||
				(c=='+' || c=='-') &&
            nextc(stream) >= '0' && nextc(stream) <= '9' ) {
      sign = 0;
      if (c=='+' || c=='-' && ++sign)
         c = chrIn();
      if (n = c - '0')
         while ((c = nextc(stream)) >= '0' && c <= '9') {
            n  =  n * 10 + c - '0';
            chrIn();
         }
      else
         while ((c = nextc(stream))>='0' && c<='9' || c>='A' && c<='F') {
            if ((c -= '0') > 9)
               c -= 7;
            n  =  (n << 4) + c;
            chrIn();
         }
      return boxNum(sign? -n : n);
   }
   if (c == ':') {
   	if (!(x = find(y = readName(chrIn()), NO)))
      	return intern(newSym(y,voidSym), NO);
		return x;
   }
   if (c == '.')
		return nilSym;
	x = val(metaSym);
	while (isCell(x)) {
		if (car(y = car(x)) == boxNum(c))
			return cdr(y);
		x = cdr(x);
	}
	if (!(x = find(y = readName(c), YES)))
      return intern(newSym(y,voidSym), YES);
   return x;
}

/* Read a List */
pico rdList()
{
   register pico x;
   register int c;
	cell c1;

   if ((c = skipWhite()) == ')') {
      chrIn();
      return nilSym;
   }
   if (c == '>')
      return nilSym;
   if (c == '.') {
      chrIn();
      x = read0(NO);
      if (skipWhite() != '>'  &&  chrIn() != ')')
         err("Reading bad dotted pair");
      return x;
   }
   push(read0(NO),c1);
   x = rdList();
   return share(pop(c1), x);
}

bool isAsciiList(x)
register pico x;
{
	bool sp = NO;

	do {
		if (!isNum(car(x)))
			return NO;
		if (num(car(x)) < 0) {
			if (sp || num(car(x))<boxNum(-6))
				return NO;
			else
				sp = YES;
		}
		else {
			sp = NO;
			if (num(car(x))<=boxNum(32) || num(car(x))>=boxNum(127))
				return NO;
		}
	} while (isCell(x = cdr(x)));
	return isNil(x);
}

void prAsciiList(x)
register pico x;
{
   register int c;

	chrOut('"');
   keyBreak();
   do {
      c = (int)unBox(car(x));
      if (c < 0)
			do
				space();
			while (++c);
		else {
			if (c == '"')
				chrOut('\\');
         chrOut(c);
		}
   } while (isCell(x = cdr(x)));
	chrOut('"');
}

/* Print one expression */
void prin0(x)
register pico x;
{
   keyBreak();
   if (isNum(x))
      prNumber(unBox(x));
   else if (isSym(x)) {
		if (hidden(x))
			chrOut(':');
      prName(getPname(x));
	}
   else if (isAsciiList(x))
		prAsciiList(x);
	else {
      /* List or dotted pair */
      chrOut('(');
      loop {
         prin0(car(x));
         if (isNil(x = cdr(x)))
            break;
         if (!isCell(x)) {
            prString(" . ");
            prin0(x);
            break;
         }
         space();
      }
      chrOut(')');
   }
}

void cutPr(x)
register pico x;
{
   if (val(printFlg) != nilSym  ||  isNum(x)  ||  isSym(x))
      prin0(x);
   else {
      chrOut('(');
      cutPr(car(x));
      if (!isNil(x = cdr(x))) {
         if (!isCell(x)) {
            prString(" . ");
            prin0(x);
         }
         else {
            space();
            cutPr(car(x));
            if (!isNil(cdr(x)))
               prString(" ..");
         }
      }
      chrOut(')');
   }
}
