/* picoDebug.c
 * 09jul90abu
 */

#include "pico.h"
#include "stack.h"

/* Prototypes */
static pico doCrash(void);
static void dots(long);
static void how(pico,pico);
static bool nestMatch(pico,pico);
static bool nestp(pico,pico);
static void showRegs(void);

static unsigned long pcReg,srReg;
static long cpuRegs[16];

void showRegs()
{
	register int i;

	prString("PC ");
	prHexNum(pcReg);
	prString("    SR ");
	prHexNum(srReg);
	prString("\rData ");
	i = 0;
	do {
		prHexNum(cpuRegs[i]);
		space();
	} while (++i < 8);
	prString("\rAddr ");
	do {
		prHexNum(cpuRegs[i]);
		space();
	} while (++i < 16);
	crlf();
}

pico doCrash()
{
	asm {
		move.l a5,-(a7)
		movea.l CurrentA5,a5
		move.w 4(a7),srReg+2
		move.w 4(a7),sr
		move.l 6(a7),pcReg
		movem.l d0-d7/a0-a7,cpuRegs
		move.l (a7)+,cpuRegs[13];
	}
	reset();
	prString("System Crash\r");
	showRegs();
	doError();
}

pico Crash(x)
register pico x;
{
	register fun *vect;

	if (!isCell(x))
		showRegs();
	else do {
		vect = (fun*)EVAL1(x);
		NEEDNUM(vect);
		vect = (fun*)unBox(vect);
		*vect = doCrash;
	} while (isCell(x = cdr(x)));
	return tSym;
}

pico Loc(x)
register pico x;
{
	x = EVAL1(x);
	return isNum(x)? nilSym:boxNum(x);
}

bool nestMatch(x,y)
register pico x,y;
{
	loop {
		if (x == star1Sym  ||  x == y)
			return YES;
		if (!isCell(x) || !isCell(y) || !nestMatch(car(x),car(y)))
			return NO;
		x = cdr(x);
		y = cdr(y);
	}
}

bool nestp(x,y)
register pico x,y;
{
	while (isCell(y)) {
		if (nestMatch(x,car(y)) || isCell(car(y)) && nestp(x,car(y)))
			return YES;
		y = cdr(y);
	}
	return NO;
}

pico Who(x)
register pico x;
{
	register int i;
	register pico y, z1, z2, z3;

	z1 = EVAL1(x);
	z2 = z3 = NULL;
	if (isCell(x = cdr(x))) {
		z2 = EVAL1(x);
		if (isCell(x = cdr(x)))
			z3 = EVAL1(x);
	}
	push(nilSym);
	for (i=0; i<=THREADS; ++i) {
		x = env.threads[i];
		while (isCell(x)) {
			if (isCell(y = val(car(x))) && nestp(z1,y) &&
									(!z2 || nestp(z2,y)) && (!z3 || nestp(z3,y)) )
				tos = newCell(car(x),tos);
			x = cdr(x);
		}
	}
	return pop();
}

void how(x,y)
register pico x,y;
{
	register pico z,expr;

	expr = y;
	while (isCell(y)) {
		z = car(y);
		if (nestMatch(x,z)  ||  isCell(z) && car(z)==quoteSym && car(cdr(z))==x) {
			tab(), prin0((!isCell(z) || (car(z)==quoteSym))? expr:z), crlf();
			break;
		}
		if (isCell(z))
			how(x,z);
		y = cdr(y);
	}
}

pico How(x)
register pico x;
{
	register int i;
	register pico y, z, t;
	file *sSave;

	sSave = stream;
	stream = NULL;
	y = EVAL1(x);
	if (isCell(x = cdr(x))) {
		NEEDSYM(y);
		how(EVAL1(x),val(y));
	}
	else
		for (i=0; i<=THREADS; ++i) {
			t = env.threads[i];
			while (isCell(t)) {
				if (isCell(x = val(car(t)))) {
					z = x;
					do
						z = cdr(z);
					while (isCell(z));
					if (z == objectSym) {
						while (isCell(z = car(x))) {
							if (nestp(y,cdr(z))) {
								prin0(car(z));
								space();
								prin0(car(t));
								crlf();
								how(y,cdr(z));
							}
							x = cdr(x);
						}
					}
					else if (nestp(y,x)) {
						prin0(car(t));
						crlf();
						how(y,x);
					}
				}
				t = cdr(t);
			}
		}
	stream = sSave;
	return nilSym;
}

void doBreak(x)
pico x;
{
	file *sSave;

	push(val(taskSym));
	setVal(taskSym, nilSym);
	push(val(logFlg));
	setVal(logFlg, nilSym);
	push(val(macFlg));
	setVal(macFlg, nilSym);
	sSave = stream;
	stream = NULL;
	prString("$ ");
	prin0(x);
	crlf();
	revalo(x);
	stream = sSave;
	setVal(macFlg, pop());
	setVal(logFlg, pop());
	setVal(taskSym, pop());
}

pico BreakFun(x)
register pico x;
{
	doBreak(x);
	return EVAL(x);
}

void dots(i)
register long i;
{
	i  =  i < 64 ?  i:64;
	while (--i >= 0)
		chrOut('.');
}

pico TraceFun(x)
register pico x;
{
	register long i;
	pico foo, body;
	file *sSave;

	sSave = stream;
	stream = NULL;
	dots(++traceLevel);
	prin0(foo = car(x));
	x = cdr(x);
	body = cdr(x);
	chrOut('[');
	if (isCell(x = car(x)))
		loop {
			prin0(val(car(x)));
			if (!isCell(x = cdr(x)))
				break;
			space();
		}
	chrOut(']');
	crlf();
	x = evalBody(body);
	dots(traceLevel--);
	prin0(foo);
	chrOut('=');
	prin0(x);
	crlf();
	stream = sSave;
	return x;
}

pico Dump(x)
register pico x;
{
	register char c, *adr;
	register int i, j, len;
	file *sSave;

	adr = (char*)EVAL1(x);
	NEEDNUM(adr);
	adr = (char*)(unBox(adr) & 0x3FFFFFFF);
	if (isNil(x = EVAL1(cdr(x))))
		len = 1;
	else
		len = unBox(x);
	sSave = stream;
	stream = NULL;
	for (i=0; i<len; ++i) {
		keyBreak();
		prHexNum((unsigned long)adr + i*16);
		prString("   ");
		j = 0;
		do {
			c = adr[i*16+j];
			hexChar(c >> 4 & 0xF);
			hexChar(c & 0xF);
			space();
		} while (++j < 16);
		prString("  ");
		j = 0;
		do {
			chrOut(((c = adr[i*16+j] & 0x7F) >= ' '  &&  c <= '~') ?  c : '.');
		} while (++j < 16);
		crlf();
	}
	stream = sSave;
	return boxNum(adr+len*16);
}

pico Trail()
{
	register pico x;
	register stkFrame *s;

	if (!(s = stkBase))
		return nilSym;
	push(x = newCell(s->self? s->self:nilSym, nilSym));
	while (s = s->link) {
		setCdr(x, newCell(s->self? s->self:nilSym, nilSym));
		x = cdr(x);
	}
	return pop();
}

pico Track(x)
pico x;
{
	register int dspCnt, fCnt;
	register pico *p;
	register stkFrame *s, *last;

	if (!isCell(x))
		dspCnt = 99;
	else {
		x = EVAL1(x);
		NEEDNUM(x);
		dspCnt = unBox(x);
	}
	s = stkBase;
	last = NULL;
	while (s  &&  dspCnt > 0) {
		if (s->self) {
			cutPr(s->self);
			space();
			--dspCnt;
		}
		fCnt = s->cnt;
		p = s->sp + 4*fCnt - 3;
		while (--fCnt >= 0) {
			if (s->self)
				space(), prin0(*p), prString(":"), prin0(val(*p));
			x = val(*p);
			setVal(*p, *(p+2));
			*(p+2) = x;
			p -= 4;
		}
		if (s->self)
			crlf();
		p = (pico*)s;
		s = s->link;
		((stkFrame*)p)->link = last;
		last = (stkFrame*)p;
	}
	while (last) {
		fCnt = last->cnt;
		p = last->sp + 4*fCnt - 3;
		while (--fCnt >= 0) {
			x = val(*p);
			setVal(*p, *(p+2));
			*(p+2) = x;
			p -= 4;
		}
		p = (pico*)last;
		last = last->link;
		((stkFrame*)p)->link = s;
		s = (stkFrame*)p;
	}
	return tSym;
}

pico Revalo(x)
register pico x;
{
	revalo(EVAL1(x));
	return tSym;
}
