/* picoGraf.c
 * 13aug90abu
 */

#include "pico.h"
#include "stack.h"

/* Prototypes */
static number cross(number,number,number,number,number);


/* Points */
void needPoint(x)
register pico x;
{
	if (!isCell(x) || !isNum(car(x)) || !isNum(cdr(x)))
		errObj(x, "Point expected");
}

void unBoxPoint(x,pt)
register pico x;
register Point *pt;
{
	needPoint(x);
	pt->h = (integer)unBox(car(x));
	pt->v = (integer)unBox(cdr(x));
}

pico boxPoint(pt)
Point pt;
{
	return newCell(boxNum(pt.h),boxNum(pt.v));
}

pico AccPoint(x)
pico x;
{
	register Point *adr;

	adr = (Point*)EVAL1(x);
	NEEDNUM(adr);
	adr = (Point*)unBox(adr);
	if (isNil(x = EVAL1((cdr(x)))))
		return newCell(boxNum(adr->h), boxNum(adr->v));
	unBoxPoint(x,(Point*)adr);
	return x;
}

/* Rectangles */
bool isRect(x)
register pico x;
{
	register pico y;

	return (isCell(x) && isCell(y = car(x)) && isCell(cdr(x)) &&
				isNum(car(y)) && isNum(cdr(y)) && isNum(cdr(car(x))) && isNum(cdr(cdr(x))) );
}

void needRect(x)
pico x;
{
	if (!isRect(x))
		errObj(x, "Rectangle expected");
}

pico boxRect(r)
register Rect *r;
{
	return newCell2(boxNum(r->left), boxNum(r->top), boxNum(r->right), boxNum(r->bottom));
}

void unBoxRect(x,r)
register pico x;
register Rect *r;
{
	register pico y;

	needRect(x);
	y = car(x);
	x = cdr(x);
	r->left = (integer)unBox(car(y));
	r->top = (integer)unBox(cdr(y));
	r->right = (integer)unBox(car(x));
	r->bottom = (integer)unBox(cdr(x));
}

pico AccRect(x)
pico x;
{
	register Rect *adr;

	adr = (Rect*)EVAL1(x);
	NEEDNUM(adr);
	adr = (Rect*)unBox(adr);
	if (isNil(x = EVAL1((cdr(x)))))
		return boxRect(adr);
	unBoxRect(x,(Rect*)adr);
	return x;
}

pico Bounds(x)
register pico x;
{
	register pico y;
	register number h1,v1,h2,v2;

	y = EVAL1(x);
	needPoint(y);
	h1 = h2 = unBox(car(y));
	v1 = v2 = unBox(cdr(y));
	while (isCell(x = cdr(x))) {
		y = EVAL1(x);
		needPoint(y);
		if (unBox(car(y)) < h1)
			h1 = unBox(car(y));
		else if (unBox(car(y)) > h2)
			h2 = unBox(car(y));
		if (unBox(cdr(y)) < v1)
			v1 = unBox(cdr(y));
		else if (unBox(cdr(y)) > v2)
			v2 = unBox(cdr(y));
	}
	return newCell2(boxNum(h1), boxNum(v1), boxNum(h2), boxNum(v2));
}

pico Graf(x)
register pico x;
{
	register pico y,z,z1,z2;
	register number h1,v1,h2,v2;

	push(z = EVAL1(x));
	NEEDCELL(z);
	y = car(z);
	NEEDCELL(y);
	needPoint(x = car(y));
	h1 = h2 = unBox(car(x));
	v1 = v2 = unBox(cdr(x));
	while (isCell(y = cdr(y))) {
		needPoint(x = car(y));
		if (unBox(car(x)) < h1)
			h1 = unBox(car(x));
		else if (unBox(car(x)) > h2)
			h2 = unBox(car(x));
		if (unBox(cdr(x)) < v1)
			v1 = unBox(cdr(x));
		else if (unBox(cdr(x)) > v2)
			v2 = unBox(cdr(x));
	}
	while (isCell(z = cdr(z))) {
		y = car(z);
		NEEDCELL(y);
		do {
			needPoint(x = car(y));
			if (unBox(car(x)) < h1)
				h1 = unBox(car(x));
			else if (unBox(car(x)) > h2)
				h2 = unBox(car(x));
			if (unBox(cdr(x)) < v1)
				v1 = unBox(cdr(x));
			else if (unBox(cdr(x)) > v2)
				v2 = unBox(cdr(x));
		} while (isCell(y = cdr(y)));
	}
	z = tos;
	push(z1 = newCell(newCell2(boxNum(h1), boxNum(v1), boxNum(h2), boxNum(v2)), nilSym));
	do {
		x = car(z);
		y = car(x);
		push(z2 = newCell(packPt(unBox(car(y))-h1, unBox(cdr(y))-v1), nilSym));
		while (isCell(x = cdr(x))) {
			y = car(x);
			setCdr(z2, newCell(packPt(unBox(car(y))-h1, unBox(cdr(y))-v1), nilSym));
			z2 = cdr(z2);
		}
		setCdr(z1, newCell(pop(),nilSym));
		z1 = cdr(z1);
	} while isCell(z = cdr(z));
	x = pop();
	drop();
	return x;
}

pico DeGraf(x)
register pico x;
{
	register pico y,z,z1,z2;
	register number h,v;

	push(z = EVAL1(x));
	NEEDCELL(z);
	needRect(y = car(z));
	h = unBox(car(car(y)));
	v = unBox(cdr(car(y)));
	z = cdr(z);
	NEEDCELL(z);
	y = car(z);
	NEEDCELL(y);
	x = car(y);
	NEEDNUM(x);
	push(z2 = newCell(newCell(boxNum(h + ptH(x)), boxNum(v + ptV(x))), nilSym));
	while (isCell(y = cdr(y))) {
		x = car(y);
		NEEDNUM(x);
		setCdr(z2, newCell(newCell(boxNum(h + ptH(x)), boxNum(v + ptV(x))), nilSym));
		z2 = cdr(z2);
	}
	tos = z1 = newCell(tos,nilSym);
	while (isCell(z = cdr(z))) {
		y = car(z);
		NEEDCELL(y);
		x = car(y);
		NEEDNUM(x);
		push(z2 = newCell(newCell(boxNum(h + ptH(x)), boxNum(v + ptV(x))), nilSym));
		while (isCell(y = cdr(y))) {
			x = car(y);
			NEEDNUM(x);
			setCdr(z2, newCell(newCell(boxNum(h + ptH(x)), boxNum(v + ptV(x))), nilSym));
			z2 = cdr(z2);
		}
		setCdr(z1, newCell(pop(),nilSym));
	}
	x = pop();
	drop();
	return x;
}

pico GrafPt(x)
register pico x;
{
	register pico y;
	register number n1, n2, h, v;

	n1 = (number)EVAL1(x);
	NEEDNUM(n1);
	n1 = unBox(n1);
	x = cdr(x);
	n2 = (number)EVAL1(x);
	NEEDNUM(n2);
	n2 = unBox(n2);
	x = cdr(x);
	x = EVAL1(x);
	NEEDCELL(x);
	needRect(y = car(x));
	y = car(y);
	h = unBox(car(y));
	v = unBox(cdr(y));
	x = cdr(x);
	NEEDCELL(x);
	while (--n1 >= 0)
		x = cdr(x);
	NEEDCELL(x);
	x = car(x);
	NEEDCELL(x);
	while (--n2 >= 0)
		x = cdr(x);
	x = car(x);
	NEEDNUM(x);
	return newCell(boxNum(h + ptH(x)), boxNum(v + ptV(x)));
}

pico MvGraf(x)
register pico x;
{
	register pico y;
	register number dh,dv;

	push(y = EVAL1(x));
	NEEDCELL(y);
	x = cdr(x);
	dh = (number)EVAL1(x);
	NEEDNUM(dh);
	dh = unBox(dh);
	x = cdr(x);
	dv = (number)EVAL1(x);
	NEEDNUM(dv);
	dv = unBox(dv);
	needRect(x = car(y));
	tos = newCell(
			newCell2(
				boxNum(unBox(car(car(x))) + dh),
				boxNum(unBox(cdr(car(x))) + dv),
				boxNum(unBox(car(cdr(x))) + dh),
				boxNum(unBox(cdr(cdr(x))) + dv) ),
			cdr(y) );
	return pop();
}

pico Draw(x)
register pico x;
{
	register pico y,z;
	register number h,v,scale,orgH,orgV;

	scale = 1;
	orgH = orgV = 0;
	y = EVAL1(x);
	NEEDCELL(y);
	if (isCell(x = cdr(x))) {
		push(y);
		scale = (number)EVAL1(x);
		NEEDNUM(scale);
		scale = unBox(scale);
		if (isCell(x = cdr(x))) {
			orgH = (number)EVAL1(x);
			NEEDNUM(orgH);
			orgH = unBox(orgH);
			orgV = (number)EVAL1(cdr(x));
			NEEDNUM(orgV);
			orgV = unBox(orgV);
		}
		y = pop();
	}
	needRect(x = car(y));
	z = car(x);
	h = unBox(car(z));
	v = unBox(cdr(z)); 
	z = cdr(y);
	do {
		NEEDCELL(z);
		y = car(z);
		NEEDCELL(y);
		x = car(y);
		NEEDNUM(x);
		MoveTo(orgH + scale*(h + ptH(x)), orgV + scale*(v + ptV(x)));
		while (isCell(y = cdr(y))) {
			x = car(y);
			NEEDNUM(x);
			LineTo(orgH + scale*(h + ptH(x)), orgV + scale*(v + ptV(x)));
		}
	} while (isCell(z = cdr(z)));
	return tSym;
}

number cross(y,h1,v1,h2,v2)
register number y,h1,v1,h2,v2;
{
	if (v1 <= y && y < v2  ||  v2 <= y && y < v1)
		return  h1 + (y - v1)*(h2 - h1)/(v2-v1);
	return MAXNEG;
}

pico InGraf(x)
register pico x;
{
	register pico y,z;
	register number h,v,h1,v1;
	register bool res;
	number scale,orgH,orgV;

	y = EVAL1(x);
	x = cdr(x);
	needPoint(y);
	h = unBox(car(y));
	v = unBox(cdr(y));
	y = EVAL1(x);
	NEEDCELL(y);
	needRect(car(y));
	if (isCell(x = cdr(x))) {
		push(y);
		scale = (number)EVAL1(x);
		NEEDNUM(scale);
		scale = unBox(scale);
		orgH = orgV = 0;
		if (isCell(x = cdr(x))) {
			orgH = (number)EVAL1(x);
			NEEDNUM(orgH);
			orgH = unBox(orgH);
			orgV = (number)EVAL1(cdr(x));
			NEEDNUM(orgV);
			orgV = unBox(orgV);
		}
		h = (h - orgH) / scale;
		v = (v - orgV) / scale;
		y = pop();
	}
	if (h < unBox(car(x = car(car(y)))) || h >= unBox(car(cdr(car(y)))) ||
				v < unBox(cdr(x)) || v >= unBox(cdr(cdr(car(y)))) )
		return nilSym;
	h1 = unBox(car(x));
	v1 = unBox(cdr(x));
	res = NO;
	z = cdr(y);
	do {
		NEEDCELL(z);
		y = car(z);
		NEEDCELL(y);
		while (isCell(cdr(y))) {
			x = car(y);
			y = cdr(y);
			if (h < cross(v, h1+ptH(x), v1+ptV(x), h1+ptH(car(y)), v1+ptV(car(y))))
				res = !res;
		}
	} while (isCell(z = cdr(z)));
	return res? tSym : nilSym;
}

pico OnGraf(x)
register pico x;
{
	register pico y,z;
	register number h,v,dh,dv,h1,v1,fx,fy,gx,gy;
	number scale,orgH,orgV;

	y = EVAL1(x);
	x = cdr(x);
	needPoint(y);
	h = unBox(car(y));
	v = unBox(cdr(y));
	y = EVAL1(x);
	NEEDCELL(y);
	needRect(car(y));
	if (isCell(x = cdr(x))) {
		push(y);
		scale = (number)EVAL1(x);
		NEEDNUM(scale);
		scale = unBox(scale);
		orgH = orgV = 0;
		if (isCell(x = cdr(x))) {
			orgH = (number)EVAL1(x);
			NEEDNUM(orgH);
			orgH = unBox(orgH);
			orgV = (number)EVAL1(cdr(x));
			NEEDNUM(orgV);
			orgV = unBox(orgV);
		}
		h = (h - orgH) / scale;
		v = (v - orgV) / scale;
		y = pop();
	}
	if (h < unBox(car(x = car(car(y)))) || h >= unBox(car(cdr(car(y)))) ||
				v < unBox(cdr(x)) || v >= unBox(cdr(cdr(car(y)))) )
		return nilSym;
	h1 = unBox(car(x));
	v1 = unBox(cdr(x));
	z = cdr(y);
	do {
		NEEDCELL(z);
		y = car(z);
		NEEDCELL(y);
		while (isCell(cdr(y))) {
			x = car(y);
			y = cdr(y);
			fx = h1 + ptH(x);
			fy = v1 + ptV(x);
			gx = h1 + ptH(car(y));
			gy = v1 + ptV(car(y));
			dh = gx - fx;
			dv = gy - fy;
			if (dh || dv) {
				if (abs(dh) > abs(dv)) {
					if (dh > 0) {
						if (fx <= h  &&  h <= gx  &&  abs(fy + (h-fx)*dv/dh - v) <= 2)
							return tSym;
					}
					else {
						if (gx <= h  &&  h <= fx  &&  abs(gy + (h-gx)*dv/dh - v) <= 2)
							return tSym;
					}
				}
				else {
					if (dv > 0) {
						if (fy <= v  &&  v <= gy  &&  abs(fx + (v-fy)*dh/dv - h) <= 2)
							return tSym;
					}
					else {
						if (gy <= v  &&  v <= fy  &&  abs(gx + (v-gy)*dh/dv - h) <= 2)
							return tSym;
					}
				}
			}
		}
	} while (isCell(z = cdr(z)));
	return nilSym;
}

pico Plot(x)
register pico x;
{
	register pico y,z;
	register bool xchg;
	register number i, e, sh,sv, dh,dv, h,v, h0,v0, temp;

	push(z = EVAL1(x)); /* Graf */
	NEEDCELL(z);
	needRect(y = car(z));
	push(EVAL1(cdr(x))); /* Foo */
	y = car(y);
	h0 = unBox(car(y));
	v0 = unBox(cdr(y));
	z = cdr(z);
	do {
		NEEDCELL(z);
		y = car(z);
		NEEDCELL(y);
		while (isCell(cdr(y))) {
			x = car(y);
			y = cdr(y);
			h = h0 + ptH(x);
			v = v0 + ptV(x);
			dh = h0 + ptH(car(y)) - h;
			dv = v0 + ptV(car(y)) - v;
			sh = 0;
			if (dh > 0)
				sh = 1;
			else if (dh < 0) {
				dh = -dh;
				sh = -1;
			}
			sv = 0;
			if (dv > 0)
				sv = 1;
			else if (dv < 0) {
				dv = -dv;
				sv = -1;
			}
			xchg = NO;
			if (dv > dh) {
				xchg = YES;
				temp = dh,  dh = dv,  dv = temp;
			}
			e = 2*dv - dh;
			for (i = 0; i < dh; ++i) {
				x = apply2(tos, boxNum(h), boxNum(v));
				if (e >= 0) {
					if (xchg)
						h += sh;
					else
						v += sv;
					e -= 2*dh;
				}
				if (xchg)
					v += sv;
				else
					h += sh;
				e += 2*dv;
			}
		}
	} while (isCell(z = cdr(z)));
	drop2();
	return x;
}

pico NextPt(x)
register pico x;
{
	register pico y;
	register number h,v,dh,dv,sh,sv,xchg,temp;

	needPoint(y = EVAL1(x));
	h = unBox(car(y));
	v = unBox(cdr(y));
	x = cdr(x);
	needPoint(x = EVAL1(x));
	dh = unBox(car(x)) - h;
	dv = unBox(cdr(x)) - v;
	sh = 0;
	if (dh > 0)
		sh = 1;
	else if (dh < 0) {
		dh = -dh;
		sh = -1;
	}
	sv = 0;
	if (dv > 0)
		sv = 1;
	else if (dv < 0) {
		dv = -dv;
		sv = -1;
	}
	xchg = NO;
	if (dv > dh) {
		xchg = YES;
		temp = dh,  dh = dv,  dv = temp;
	}
	if (2*dv >= dh) {
		if (xchg)
			h += sh;
		else
			v += sv;
	}
	if (xchg)
		v += sv;
	else
		h += sh;
	return newCell(boxNum(h),boxNum(v));
}

pico Raster(x)
register pico x;
{
	register pico y, z, w;
	register number v, v0, v1, v2, h0, h1, h2;

	v = (number)EVAL1(x);	/* Raster scan line number */
	NEEDNUM(v);
	v = unBox(v);
	x = cdr(x);
	push(z = EVAL1(x)); /* Graf */
	NEEDCELL(z);
	needRect(y = car(z));
	y = car(y);
	h0 = unBox(car(y));
	v0 = unBox(cdr(y));

	/* Build sorted list of crossPoints */
	push(nilSym);
	z = cdr(z);
	do {
		NEEDCELL(z);
		y = car(z);
		NEEDCELL(y);
		while (isCell(cdr(y))) {
			x = car(y);
			y = cdr(y);
			v1 = v0 + ptV(x);
			v2 = v0 + ptV(car(y));
			if (v1 <= v && v < v2  ||  v2 <= v && v < v1) {
				h1 = h0 + ptH(x);
				h2 = h0 + ptH(car(y));
				w = boxNum(h1 + (v - v1)*(h2 - h1)/(v2-v1)); /* Crosspoint */
				x = tos;
				if (!isCell(x) || w <= car(x))
					tos = newCell(w,x);
				else {
					while (isCell(cdr(x)) && w > car(cdr(x)))
						x = cdr(x);
					setCdr(x, newCell(w,cdr(x)));
				}
			}
		}
	} while (isCell(z = cdr(z)));
	x = pop();
	drop();
	return x;
}

pico Scan(x)
register pico x;
{
	register pico y,z;
	register number v, v0, v1, v2, lim, h0, temp;
	register long i,n,m;
	number c, h1, h2;
	Handle h;
	number *p;

	push(dynHandle(h = NewHandle((m = 4) * sizeof(number))));
	z = EVAL1(x);	/* Graf */
	NEEDCELL(z);
	push(cdr(z));
	needRect(z = car(z));
	y = car(z);
	h0 = unBox(car(y));
	v0 = unBox(cdr(y));
	lim = unBox(cdr(cdr(z)));
	push(EVAL1(cdr(x))); /* Foo */

	for (v = v0; v < lim; ++v) {
		z = nos;
		/* Build sorted array of crossPoints */
		n = 0;
		do {
			NEEDCELL(z);
			y = car(z);
			NEEDCELL(y);
			while (isCell(cdr(y))) {
				x = car(y);
				y = cdr(y);
				v1 = v0 + ptV(x);
				v2 = v0 + ptV(car(y));
				if (v1 <= v && v < v2  ||  v2 <= v && v < v1) {
					h1 = h0 + ptH(x);
					h2 = h0 + ptH(car(y));
					c = h1 + (v - v1)*(h2 - h1)/(v2-v1);
					if (n == m)
						SetHandleSize(h, ++m * sizeof(number));
					p = *(number**)h;
					i = 0;
					while (i < n  &&  c > *p)
						++i, ++p;
					while (i++ < n)
						temp = *p,  *p++ = c,  c = temp;
					*p = c;
					++n;
				}
			}
		} while (isCell(z = cdr(z)));
		/* Apply function to pairs of crosspoints */
		for (i = 0; i < n; i += 2)
			x = apply3(tos, boxNum(v), boxNum(((number*)*h)[i]), boxNum(((number*)*h)[i+1]));
	}
	drop3();
	return x;
}

/* Displacement for parallel lines */
pico Parall(x)
register pico x;
{
	double d,h,v,dh,dv,a;

	push(EVAL1(x)); /* Line */
	needPoint(tos);
	x = cdr(x);
	push(EVAL1(x));
	needPoint(tos);
	x = cdr(x);
	x = EVAL1(x); /* Displacement */
	NEEDNUM(x);
	d = (double)unBox(x);
	x = pop();
	dh = (double)unBox(car(x));
	dv = (double)unBox(cdr(x));
	x = pop();
	dh -= (double)unBox(car(x));
	dv -= (double)unBox(cdr(x));
	if (dh == 0.0)
		return newCell(boxDouble(dv<0.0? d : -d), boxNum(0));
	if (d < 0.0) {
		dh = -dh;
		dv = -dv;
	}
	a = dv / dh;
	v = sqrt(d*d / (1 + a*a));
	h = -a * v;
	return newCell(boxDouble(dh<0.0? -h : h), boxDouble(dh<0.0? -v : v));
}

/* Intersection of two lines */
pico Intsec(x)
register pico x;
{
	register pico y;
	double ax,ay,bx,by,cx,cy,dx,dy,ex,ey;
	double a1,a2,b1,b2;

	needPoint(y = EVAL1(x)); /* A */
	ax = (double)unBox(car(y));
	ay = (double)unBox(cdr(y));
	x = cdr(x);
	needPoint(y = EVAL1(x)); /* B */
	bx = (double)unBox(car(y));
	by = (double)unBox(cdr(y));
	x = cdr(x);
	needPoint(y = EVAL1(x)); /* D */
	dx = (double)unBox(car(y));
	dy = (double)unBox(cdr(y));
	x = cdr(x);
	needPoint(y = EVAL1(x)); /* E */
	ex = (double)unBox(car(y));
	ey = (double)unBox(cdr(y));
	x = cdr(x);
	x = EVAL1(x); /* Flag */
	
	if (ax==bx && dx==ex || ay==by && dy==ey) { /* Trivially parallel */
parallel:
		if (ax==dx && ay==dy || ax==ex && ay==ey)
			return newCell(boxDouble(ax), boxDouble(ay));		
		if (bx==dx && by==dy || bx==ex && by==ey)
			return newCell(boxDouble(bx), boxDouble(by));		
		return nilSym;
	}
	if (ax == bx) {
		a2 = (ey - dy) / (ex - dx);
		b2 = dy - a2 * dx;
		cx = ax;
		cy = a2 * cx + b2;
	}
	else if (dx == ex) {
		a1 = (by - ay) / (bx - ax);
		b1 = ay - a1 * ax;
		cx = dx;
		cy = a1 * cx + b1;
	}
	else if (dy == ey) {
		a1 = (by - ay) / (bx - ax);
		b1 = ay - a1 * ax;
		cy = dy;
		cx = (cy - b1) / a1;
	}
	else {
		a1 = (by - ay) / (bx - ax);
		a2 = (ey - dy) / (ex - dx);
		b1 = ay - a1 * ax;
		b2 = dy - a2 * dx;
		if (a1 == a2)
			goto parallel; /* Parallel */
		cy = (b1 - a1*b2/a2) / (1 - a1/a2);
		cx = (cy - b2) / a2;
	}
	if (isNil(x)) {
		double temp;

		if (bx < ax)
			temp = bx,  bx = ax,  ax = temp;
		if (ex < dx)
			temp = ex,  ex = dx,  dx = temp;
		if (by < ay)
			temp = by,  by = ay,  ay = temp;
		if (ey < dy)
			temp = ey,  ey = dy,  dy = temp;
		if (cx<ax || cx>bx || cx<dx || cx>ex || cy<ay || cy>by || cy<dy || cy>ey)
			return nilSym;
	}
	return newCell(boxDouble(cx), boxDouble(cy));
}
