/* picoIO.c
 * 27jul90abu
 */

#include "pico.h"
#include "stack.h"

file *streamSave;
static FInfo info;

/* Prototypes */
static void cmprWhite(int*);
static void comment(void);
static int fileRead(file*);
static void fileWrite(char,file*);
static int fillNext(void);
static void flush(file*);
static file *getStream(pico);
static file *picoCreate(pico,long);
static file *picoOpen(pico);
static void prLine(pico,bool);
static pico rdList(void);
static void setStream(pico);
static char skipWhite(void);
static bool symChar(char);
static void testEsc(int*);


pico Pipe(x)
register pico x;
{
	int c;
	file *src,*dst;

	src = picoOpen(EVAL1(x));
	x = cdr(x);
	dst = picoCreate(EVAL1(x),info.fdType);
	if (!src || !dst)
		err("Can't pipe");
	x = cdr(x);
	push(EVAL1(x));
	while ((c = fileRead(src)) >= 0) {
		x = apply1(tos,boxNum(c));
		if (isNum(x))
			fileWrite(unBox(x),dst);
		else {
			while (isCell(x)) {
				fileWrite(unBox(car(x)),dst);
				x = cdr(x);
			}
		}
	}
	drop();
	closeFile(src);
	closeFile(dst);
	return tSym;
}

file *picoOpen(x)
register pico x;
{
	char fName[FILENAME];
	integer refNum, dummy;
	file *f;

	CtoPstr(bufString(x, fName, FILENAME));
	if (FSOpen(fName, 0, &refNum))
		return (file*)NULL;
	if (GetFInfo(fName,0,&info) != noErr)
		return (file*)NULL;
	f = (file*)NewPtr(sizeof(file));
	f->fd = refNum;
	f->cnt = 0;
	f->max = 0;
	f->dirty = NO;
	f->pos = 0;
	setVal(fileSym, newCell(newCell(boxNum(f), x), val(fileSym)));
	return f;
}

file *picoCreate(x,typ)
register pico x;
long typ;
{
	file *f;
	char fName[FILENAME];

	CtoPstr(bufString(x, fName, FILENAME));
	picoErase((StringPtr)fName);
	if (Create(fName, 0, CREATOR, typ))
		return (file*)NULL;
	if (!(f = picoOpen(x)))
		return (file*)NULL;
	return f;
}

void closeFile(f)
file *f;
{
	register pico x,y;

	x = boxNum(f);
	y = val(fileSym);
	if (x == car(car(y)))
		setVal(fileSym, cdr(y));
	else  {
		while (isCell(cdr(y))) {
			if (x == car(car(cdr(y)))) {
				setCdr(y, cdr(cdr(y)));
				goto doClose;
			}
			y = cdr(y);
		}
		errObj(x, "Can't close");
	}
doClose:
	flush(f);
	if (FSClose(f->fd) || FlushVol((StringPtr)NULL,0))
		errObj(x, "File close error");
	DisposPtr((Ptr)f);
}

pico picoErase(s)
StringPtr s;
{
	IOParam pb;

	pb.ioNamePtr = s;
	pb.ioVRefNum = 0;
	pb.ioVersNum = 0;
	return PBDelete(&pb, NO)? nilSym:tSym;
}

void flush(f)
register file *f;
{
	long count = f->max;

	if (f->dirty) {
		if (FSWrite(f->fd, &count, f->buf) || count != f->max)
			err("Write error");
		f->dirty = NO;
	}
}

int fileRead(f)
register file *f;
{
	long count;
	OSErr e;

	if (f->cnt == f->max) {
		flush(f);
		count = BUFSIZE;
		GetFPos(f->fd, &f->pos);
		if ((e = FSRead(f->fd, &count, f->buf)) && e != eofErr)
			err("File read error");
		if (!count)
			return -1;
		f->cnt = 0;
		f->max = count;
		f->dirty = NO;
	}
	return f->buf[f->cnt++] & 0xFF;
}

void fileWrite(c,f)
char c;
register file *f;
{
	if (f->cnt == BUFSIZE) {
		flush(f);
		GetFPos(f->fd, &f->pos);
		f->cnt = 0;
		f->max = 0;
	}
	f->buf[f->cnt++] = c;
	f->dirty = YES;
	if (f->max < f->cnt)
		f->max = f->cnt;
}

pico Load(x)
register pico x;
{
	file *streamSave;
	int nextSave, dummy;
	pico fName, f2Name;

	streamSave = stream;
	nextSave = nextChar;
	fName  =  isCell(x)?  EVAL1(x) : car(val(editSym));
	if (isNum(fName) || isSym(fName)) {
		if (isNil(f2Name = EVAL1(cdr(x))))
			x = get(fName,srcFlg);
		else
			x = get(f2Name,fName);
		if (isNil(x))
			errObj(fName, "Can't load");
		fName = cdr(x);
	}
	NEEDSTRING(fName);
	if (!(stream = picoOpen(fName)))
		errStrObj(fName, "File open error");
	setVal(loadSym, newCell(newCell(boxNum(1), fName), val(loadSym)));
	nextChar = fileRead(stream);
	revalo(nilSym);
	setVal(loadSym,cdr(val(loadSym)));
	closeFile(stream);
	stream = streamSave;
	nextChar = nextSave;
	return tSym;
}

pico Open(x)
pico x;
{
	file *f;

	if (!(f = picoOpen(EVAL1(x))))
		return nilSym;
	return boxNum(f);
}

pico Creat(x)
pico x;
{
	pico y;
	file *f;

	y = EVAL1(x);
	x = cdr(x);
	if (!(f = picoCreate(y, isCell(x)? nextPLong(&x) : 'TEXT')))
		return nilSym;
	return boxNum(f);
}

pico Close(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	closeFile((file*)unBox(x));
	return x;
}

pico Erase(x)
pico x;
{
	char buf[FILENAME];

	CtoPstr(bufString(EVAL1(x), buf, FILENAME));
	return picoErase((StringPtr)buf);
}

pico fRename(x)
register pico x;
{
	char old[FILENAME];
	char new[FILENAME];
	IOParam pb;

	bufString(EVAL1(x), old, FILENAME);
	bufString(EVAL1(cdr(x)), new, FILENAME);
	CtoPstr(old);
	CtoPstr(new);
	pb.ioNamePtr = (StringPtr)old;
	pb.ioVRefNum = 0;
	pb.ioVersNum = 0;
	pb.ioMisc = new; 
	return PBRename(&pb, NO)? nilSym:tSym;
}

pico Where(x)
pico x;
{
	register file *f;

	f = getStream(EVAL1(x));
	return boxNum(f->pos + f->cnt);
}

pico Seek(x)
register pico x;
{
	register pico y;
	register file *f;

	y = EVAL1(x);
	NEEDNUM(y);
	x = cdr(x);
	f = getStream(EVAL1(x));
	flush(f);
	if (SetFPos(f->fd, fsFromStart, f->pos = unBox(y)))
		return nilSym;
	f->cnt = f->max = 0;
	return y;
}

pico Read(x)
register pico x;
{
	int nextSave, dummy;

	setStream(EVAL1(x));
	nextSave = nextChar;
	nextChar = ' ';
	chrIn();
	x = read0(YES);
	nextChar = nextSave;
	stream = streamSave;
	return x;
}

pico ReadBlock(x)
pico x;
{
	file *f;
	long count;
	char *buffer;
	OSErr err;

	f = (file*)nextNum(&x);
	buffer = (char*)nextNum(&x);
	count = nextNum(&x);
	if ((err = FSRead(f->fd, &count, buffer)) && err != eofErr)
		return nilSym;
	return boxNum(count);
}

pico WriteBlock(x)
pico x;
{
	file *f;
	long count;
	char *buffer;

	f = (file*)nextNum(&x);
	buffer = (char*)nextNum(&x);
	count = nextNum(&x);
	if (FSWrite(f->fd, &count, buffer))
		return nilSym;
	return boxNum(count);
}

void prLine(x,suppress)
register pico x;
bool suppress;
{
	register int c, dummy;

	NEEDSTRING(x);
	while (isCell(x)) {
		keyBreak();
		c = unBox(car(x));
		x = cdr(x);
		if (c >= 0)
			chrOut(c);
		else if (suppress && !isCell(x))
			return;
		else do {
			keyBreak();
			space();
		} while (++c);
	}
}

pico Prin2(x)
register pico x;
{
	push(EVAL1(x));
	setStream(EVAL1(cdr(x)));
	x = pop();
	if (isCell(x) || isNil(x))
		prLine(x,NO);
	else
		prin0(x);
	stream = streamSave;
	return x;
}

pico PrLine(x)
register pico x;
{
	push(EVAL1(x));
	setStream(EVAL1(cdr(x)));
	prLine(x = pop(),YES);
	crlf();
	stream = streamSave;
	return x;
}

pico Prin1(x)
register pico x;
{
	push(EVAL1(x));
	setStream(EVAL1(cdr(x)));
	prin0(x = pop());
	stream = streamSave;
	return x;
}

pico Print(x)
register pico x;
{
	push(EVAL1(x));
	setStream(EVAL1(cdr(x)));
	prin0(x = pop());
	crlf();
	stream = streamSave;
	return x;
}

pico PrHex(x)
register pico x;
{
	push(EVAL1(x));
	setStream(EVAL1(cdr(x)));
	x = pop();
	NEEDNUM(x);
	prHexNum((unsigned long)(unBox(x)));
	stream = streamSave;
	return x;
}

pico CutPr(x)
register pico x;
{
	push(EVAL1(x));
	setStream(EVAL1(cdr(x)));
	cutPr(x = pop());
	stream = streamSave;
	return x;
}

pico Comment(x)
register pico x;
{
	push(EVAL1(x));
	x = cdr(x);
	setStream(EVAL1(x));
	chrOut('[');
	prLine(tos,NO);
	chrOut(']');
	space();
	stream = streamSave;
	return pop();
}

pico Terpri(x)
pico x;
{
	setStream(EVAL1(x));
	crlf();
	stream = streamSave;
	return nilSym;
}

pico Space(x)
pico x;
{
	setStream(EVAL1(x));
	space();
	stream = streamSave;
	return nilSym;
}

pico Tab(x)
pico x;
{
	setStream(EVAL1(x));
	tab();
	stream = streamSave;
	return nilSym;
}

pico Bell(x)
pico x;
{
	setStream(EVAL1(x));
	chrOut(7);
	stream = streamSave;
	return nilSym;
}

pico Backsp(x)
pico x;
{
	setStream(EVAL1(x));
	chrOut(8);
	stream = streamSave;
	return nilSym;
}

pico Putc(x)
register pico x;
{
	push(EVAL1(x));
	setStream(EVAL1(cdr(x)));
	if (isNum(x = pop()))
		chrOut(unBox(x));
	stream = streamSave;
	return x;
}

pico Getc(x)
register pico x;
{
	int nextSave, dummy;

	setStream(EVAL1(x));
	nextSave = nextChar;
	fillNext();
	if (nextChar >= 0)
		x = boxNum(nextChar);
	else
		x =  nilSym;
	nextChar = nextSave;
	stream = streamSave;
	return x;
}

void cmprWhite(p)
register int *p;
{
	if (*p == ' ')
		*p = -1;
	else if (*p == 9)
		*p = -TABLEN;
}

pico unBufString(s)
unsigned char *s;
{
	int c;
	register pico x;
	register int cnt;

	if (! (cnt = *s++))
		return nilSym;;
	--cnt;
	c = *s++;
	cmprWhite(&c);
	push(x = newCell(boxNum(c), nilSym));
	while (--cnt >= 0) {
		c = *s++;
		cmprWhite(&c);
		if (c < 0  &&  num(car(x)) < 0)
			setCar(x, boxNum(c + unBox(car(x))));
		else {
			setCdr(x, newCell(boxNum(c), nilSym));
			x = cdr(x);
		}
	}
	return pop();
}

pico unBufCString(s)
char *s;
{
	int c;
	register pico x;

	if (!*s)
		return nilSym;
	c = *s++;
	cmprWhite(&c);
	push(x = newCell(boxNum(c), nilSym));
	while (*s) {
		c = *s++;
		cmprWhite(&c);
		if (c < 0  &&  num(car(x)) < 0)
			setCar(x, boxNum(c + unBox(car(x))));
		else {
			setCdr(x, newCell(boxNum(c), nilSym));
			x = cdr(x);
		}
	}
	return pop();
}

pico GetLine(x)
register pico x;
{
	int nextSave, dummy;

	setStream(EVAL1(x));
	nextSave = nextChar;
	fillNext();
	if (nextChar < 0)
		x = nilSym;
	else if (!nextChar || nextChar==EOL) {
		x = nilSym;
		if (stream)
			x = newCell(boxNum(-1),x);
	}
	else {
		cmprWhite(&nextChar);
		push(x = newCell(boxNum(nextChar), nilSym));
		while ((fillNext(), nextChar > 0) && nextChar != EOL) {
			cmprWhite(&nextChar);
			if (nextChar < 0  &&  num(car(x)) < 0)
				setCar(x, boxNum(nextChar + unBox(car(x))));
			else {
				setCdr(x, newCell(boxNum(nextChar), nilSym));
				x = cdr(x);
			}
		}
		x = shareList(pop());
	}
	nextChar = nextSave;
	stream = streamSave;
	return x;
}

/* File I/O */
file *getStream(x)
register pico x;
{
	if (isNil(x))
		return (file*)NULL;
	NEEDNUM(x);
	return (file*)unBox(x);
}

void setStream(x)
pico x;
{
	streamSave = stream;
	stream = getStream(x);
}

/* Character I/O */
char upc(c)
register char c;
{
	return (c<'a' || c>'z')? c : (c-32);
}

void chrOut(c)
char c;
{
	if (stream)
		fileWrite(c,stream);
	else {
		if (c == EOL) {
			ttyOut(CR);
			ttyOut(NL);
		}
		else
			ttyOut(c);
	}
}

void space()
{
	chrOut(' ');
}

void crlf()
{
	chrOut(EOL);
}

void tab()
{
	prString("   ");
}

void prString(s)
register char *s;
{
	while (*s)
		chrOut(*s++);
}

void prNumber(n)
register number n;
{
	char buf[16];
	register int i = 0;

	if (n < 0){
		chrOut('-');
		n = -n;
	}
	do {
		buf[i++] = n % 10 + '0';
	} while (n /= 10);
	while (--i >= 0)
		chrOut(buf[i]);
}

void hexChar(c)
register char c;
{
	if ((c += '0') > '9')
		c += 7;
	chrOut(c);
}

void prHexNum(n)
register unsigned long n;
{
	char buf[16];
	register int i = 0;

	keyBreak();
	chrOut('0');
	do {
		buf[i++] = n % 16;
	} while (n /= 16);
	while (--i >= 0)
		hexChar(buf[i]);
}

int fillNext()
{
	register int c;
	file *sSave;

	c = nextChar;
	if (!stream) {
		if (!c) {
			if (revaLevel)
				chrOut(revaLevel + '0');
			chrOut(':');
			lbp = getLine(lBuff, lBuff+LBSIZE-1);
			crlf();
			c = EOL;
		}
		nextChar = *lbp++;
	}
	else {
		if (!isNil(val(echoFlg)) && c) {
			sSave = stream;
			stream = NULL;
			chrOut(c);
			stream = sSave;
		}
		if ((nextChar = fileRead(stream)) >= 0) {
			if (nextChar == EOL  &&  isCell(car(val(loadSym))))
				setCar(car(val(loadSym)), (pico)(num(car(car(val(loadSym)))) + 4));
		}
	}
	return c;
}

char chrIn()
{
	int c, dummy;

	if ((c = fillNext()) < 0)
		err("Read past End of file");
	return c;
}

/* Skip nested Comments */
void comment()
{
	chrIn();
	if (nextChar == '#') {
		register pico x;

		chrIn();
		x = read0(NO);
		if (!isNil(EVAL(x)))
			return;
	}
	while (nextChar != ']')
		if (nextChar == '[')
			comment();
		else
			chrIn();
	chrIn();
}

/* Skip White Space */
char skipWhite()
{
	loop {
		while (nextChar <= ' ' || nextChar == ']')
			chrIn();
		if (nextChar != '[')
			return nextChar;
		comment();
	}
}

void testEsc(p)
register int *p;
{
	if (*p == '\\')
		*p = chrIn();
	else if (*p == '^')
		*p = chrIn() & 0x1F;
	cmprWhite(p);
}

bool symChar(c)
register char c;
{
	return (c>='A' && c<='Z' || c=='1' || c=='2' || c=='$' || c=='*' || c=='-');
}

/* Read one Expression */
pico read0(top)
bool top;
{
	register pico x,y;
	register int i;
	number n;
	int c, sign;
	bool first, dummy;

	do
		skipWhite();
	while ((c = chrIn()) == ')'  ||  c == '>');
	if (top) {
		x = car(val(loadSym));
		loadPos = car(x);
		loadName = cdr(x);
	}
	if (c == '(') {
		x =  rdList();
		if (top && nextChar == '>')
			chrIn();
	}
	else if (c == '<') {
		x = rdList();
		if (chrIn() != '>')
			err("Super parentheses mismatch");
	}
	else if (c == '\'')
		x = share(quoteSym, share(read0(NO), nilSym));
	else if (c == '\\') {
		if ((c = chrIn()) == '^')
			c = chrIn() & 0x1F;
		x = boxNum(c);
	}
	else if (c == '#') {
		x = read0(NO);
		if ((x = EVAL(x)) == voidSym)
			err("Undefined Read Macro");
	}
	else if (c=='+' || c=='-' || c>='0' && c<='9') {
		sign = 0;
		if (c=='+' || c=='-' && ++sign)
			c = chrIn();
		if (c < '0' || c > '9')
			err("Reading bad number");
		if (n = c - '0') 
			while ((c = nextChar) >= '0' && c <= '9') {
				n  =  n * 10 + c - '0';
				chrIn();
			}
		else
			while ((c = upc(nextChar))>='0' && c<='9' || c>='A' && c<='F') {
				if ((c -= '0') > 9)
					c -= 7;
				n  =  (n << 4) + c;
				chrIn();
			}
		x = boxNum(sign? -n : n);
	}
	else if (c == '"') {
		if ((c = chrIn()) == '"')
			x = nilSym;
		else {
			testEsc(&c);
			push(x = newCell(boxNum(c), nilSym));
			while ((c = chrIn()) != '"') {
				testEsc(&c);
				if (c < 0  &&  num(car(x)) < 0)
					setCar(x, boxNum(c + unBox(car(x))));
				else {
					setCdr(x, newCell(boxNum(c), nilSym));
					x = cdr(x);
				}
			}
			x = shareList(pop());
		}
	}
	else {
		i = 5;
		first = YES;
		if (!symChar(c = upc(c)))
			err("Bad character");
		accumulate(c,&n);
		while (symChar(c=upc(nextChar))) {
			if (--i < 0) {
				if (first) {
					first = NO;
					push(x = newCell(boxNum(n),nilSym));
				}
				else {
					setCdr(x, newCell(boxNum(n),nilSym));
					x = cdr(x);
				}
				i = 5;
			}
			accumulate(c,&n);
			chrIn();
		}
		while (--i >= 0)
			accumulate(0,&n);
		if (first)
			y = boxNum(n);
		else {
			setCdr(x, boxNum(n));
			y = pop();
		}
		if (!(x = find(y)))
			x = intern(newSym(y,voidSym));
	}
	return x;
}

/* Read a List */
pico rdList()
{
	register pico x;
	register int c;

	if ((c = skipWhite()) == ')') {
		chrIn();
		return nilSym;
	}
	if (c == '>')
		return nilSym;
	if (c == '.') {
		chrIn();
		x = read0(NO);
		if (skipWhite() != '>'  &&  chrIn() != ')')
			err("Reading bad dotted pair");
		return x;
	}
	push(read0(NO));
	x = rdList();
	return share(pop(), x);
}

/* Print one expression */
void prin0(x)
register pico x;
{
	register int strFlg;

	keyBreak();
	if (isNum(x))
		prNumber(unBox(x));
	else if (isSym(x))
		prName(getPname(x));
	else {
		/* List or dotted pair */
		chrOut('(');
		loop {
			prin0(car(x));
			if (isNil(x = cdr(x)))
				break;
			if (!isCell(x)) {
				prString(" . ");
				prin0(x);
				break;
			}
			space();
		}
		chrOut(')');
	}
}

void cutPr(x)
register pico x;
{
	if (isNum(x) || isSym(x))
		prin0(x);
	else {
		chrOut('(');
		cutPr(car(x));
		if (!isNil(x = cdr(x))) {
			if (!isCell(x)) {
				prString(" . ");
				prin0(x);
			}
			else {
				space();
				cutPr(car(x));
				if (!isNil(cdr(x)))
					prString(" ..");
			}
		}
		chrOut(')');
	}
}
