/* picoMath.c
 * 02aug90abu
 */

#include "pico.h"
#include "stack.h"

number cosTab[] = {
	10000, 9998, 9994, 9986, 9976, 9962, 9945, 9925, 9903, 9877,
	 9848, 9816, 9781, 9744, 9703, 9659, 9613, 9563, 9511, 9455,
	 9397, 9336, 9272, 9205, 9135, 9063, 8988, 8910, 8829, 8746,
	 8660, 8572, 8480, 8387, 8290, 8192, 8090, 7986, 7880, 7771,
	 7660, 7547, 7431, 7314, 7193, 7071, 6947, 6820, 6691, 6561,
	 6428, 6293, 6157, 6018, 5878, 5736, 5592, 5446, 5299, 5150,
	 5000, 4848, 4695, 4540, 4384, 4226, 4067, 3907, 3746, 3584,
	 3420, 3256, 3090, 2924, 2756, 2588, 2419, 2250, 2079, 1908,
	 1736, 1564, 1392, 1219, 1045,  872,  698,  523,  349,  175,
	0
};

number tanTab[] = {
	     0,    175,    349,    524,    699,    875,   1051,   1228,   1405,   1584,
	  1763,   1944,   2126,   2309,   2493,   2679,   2867,   3057,   3249,   3443,
	  3640,   3839,   4040,   4245,   4452,   4663,   4877,   5095,   5317,   5543,
	  5774,   6009,   6249,   6494,   6745,   7002,   7265,   7536,   7813,   8098,
	  8391,   8693,   9004,   9325,   9657,  10000,  10355,  10724,  11106,  11504,
	 11918,  12349,  12799,  13270,  13764,  14281,  14826,  15399,  16003,  16643,
	 17321,  18040,  18807,  19626,  20503,  21445,  22460,  23559,  24751,  26051,
	 27475,  29042,  30777,  32709,  34874,  37321,  40108,  43315,  47046,  51446,
	 56713,  63138,  71154,  81443,  95144, 114301, 143007, 190811, 286363, 572900,
	MAXNUM
};

/* Prototypes */
static number fixCos(number);
static number fixTan(number);

void transform(m,src,dst)
register matrix *m;
register vector *src,*dst;
{
	dst->x  =  src->x * m->a.x + src->y * m->b.x + src->z * m->c.x;
	dst->y  =  src->x * m->a.y + src->y * m->b.y + src->z * m->c.y;
	dst->z  =  src->x * m->a.z + src->y * m->b.z + src->z * m->c.z;
}

pico Enum(x)
register pico x;
{
	register number n;

	n = 0;
	while (isCell(x)) {
		if (isNum(car(x)))
			n += unBox(car(x));
		else
			redefine(car(x), boxNum(n++));
		x = cdr(x);
	}
	return tSym;
}

pico Max(x)
register pico x;
{
	register pico y, res;

	res = EVAL1(x);
	NEEDNUM(res);
	while (isCell(x = cdr(x))) {
		y = EVAL1(x);
		NEEDNUM(y);
		if (num(y) > num(res))
			res = y;
	}
	return res;
}

pico Min(x)
register pico x;
{
	register pico y, res;

	res = EVAL1(x);
	NEEDNUM(res);
	while (isCell(x = cdr(x))) {
		y = EVAL1(x);
		NEEDNUM(y);
		if (num(y) < num(res))
			res = y;
	}
	return res;
}

pico Limit(x)
register pico x;
{
	register pico y,z;

	y = EVAL1(x);
	NEEDNUM(y);
	x = cdr(x);
	z = EVAL1(x);
	NEEDNUM(z);
	x = cdr(x);
	x = EVAL1(x);
	NEEDNUM(x);
	if (num(y) < num(z))
		return z;
	if (num(y) > num(x))
		return x;
	return y;
}

pico pBit(x)
register pico x;
{
	register number b;

	b = num(EVAL1(x));
	NEEDNUM(b);
	b = unBox(b);
	x = cdr(x);
	x = EVAL1(x);
	NEEDNUM(x);
	return (unBox(x) & 1 << b)? tSym:nilSym;
}

pico Bitl(x)
register pico x;
{
	register number b;

	b = num(EVAL1(x));
	NEEDNUM(b);
	b = unBox(b);
	x = cdr(x);
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(unBox(x) << b);
}

pico Bitr(x)
register pico x;
{
	register number b;

	b = num(EVAL1(x));
	NEEDNUM(b);
	b = unBox(b);
	x = cdr(x);
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(unBox(x) >> b);
}

pico pBitOn(x)
register pico x;
{
	register number b;

	b = num(EVAL1(x));
	NEEDNUM(b);
	b = unBox(b);
	x = cdr(x);
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(unBox(x) | 1L << b);
}

pico pBitOff(x)
register pico x;
{
	register number b;

	b = num(EVAL1(x));
	NEEDNUM(b);
	b = unBox(b);
	x = cdr(x);
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(unBox(x) & ~(1L << b));
}

pico pBitAnd(x)
register pico x;
{
	register number n = num(boxNum(-1));
	register pico y;

	do {
		y = EVAL1(x);
		NEEDNUM(y);
		n = n & num(y);  /* n &= num(y); */
	} while (isCell(x = cdr(x)));
	return (pico)n;
}

pico pBitNot(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(~unBox(x));
}

pico pBitOr(x)
register pico x;
{
	register number n = 0;
	register pico y;

	do {
		y = EVAL1(x);
		NEEDNUM(y);
		n = n | num(y);  /* n |= num(y); */
	} while (isCell(x = cdr(x)));
	return (pico)n;
}

pico pBitXor(x)
register pico x;
{
	register number n = 0;
	register pico y;

	do {
		y = EVAL1(x);
		NEEDNUM(y);
		n ^= num(y);
	} while (isCell(x = cdr(x)));
	return (pico)(n | 2);
}

pico Ascii(x)
pico x;
{
	x = EVAL1(x);
	NEEDSYM(x);
	return boxNum(firstChar(x));
}

pico Upc(x)
register pico x;
{
	if (isNum(x=EVAL1(x)) && x >= boxNum('a') && x <= boxNum('z'))
		x = (pico)(num(x) & ~0x80);
	return x;
}

pico Lowc(x)
register pico x;
{
	if (isNum(x=EVAL1(x)) && x >= boxNum('A') && x <= boxNum('Z'))
		x = (pico)(num(x) | 0x80);
	return x;
}

pico Digit(x)
register pico x;
{
	x = EVAL1(x);
	if (isNum(x) && boxNum('0')<=x && x<=boxNum('9'))
		return (pico)(num(x) - ('0' << 2));
	return nilSym;
}

pico Letter(x)
register pico x;
{
	x = EVAL1(x);
	if (isNum(x) && (boxNum('a')<=x && x<=boxNum('z')) || (boxNum('A')<=x && x<=boxNum('Z')))
		return (pico)((num(x) & 0xFF7F) - ('@' << 2));
	return nilSym;
}

pico Add1(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)(num(x)+4);
}

pico Sub1(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)(num(x)-4);
}

pico Zero(x)
register pico x;
{
	register pico sym;

	do {
		sym = car(x);
		NEEDSYM(sym);
		CHECKSYM(sym);
		setVal(sym, boxNum(0));
	} while (isCell(x = cdr(x)));
	return boxNum(0);
}

pico Inc(x)
register pico x;
{
	register pico sym;

	sym = car(x);
	NEEDSYM(sym);
	NEEDNUM(val(sym));
	if (isCell(x = cdr(x))) {
		x = EVAL1(x);
		NEEDNUM(x);
		setVal(sym, x = boxNum(unBox(val(sym)) + unBox(x)));
	}
	else
		setVal(sym, x = (pico)(num(val(sym)) + 4));
	return x;
}

pico Dec(x)
register pico x;
{
	register pico sym;

	sym = car(x);
	NEEDSYM(sym);
	NEEDNUM(val(sym));
	if (isCell(x = cdr(x))) {
		x = EVAL1(x);
		NEEDNUM(x);
		setVal(sym, x = boxNum(unBox(val(sym)) - unBox(x)));
	}
	else
		setVal(sym, x = (pico)(num(val(sym)) - 4));
	return x;
}

pico Add2(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)(num(x)+8);
}

pico Sub2(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)(num(x)-8);
}

pico Add(x)
register pico x;
{
	register number n = 0;
	register pico y;

	do {
		y = EVAL1(x);
		NEEDNUM(y);
		n += unBox(y);
	} while (isCell(x = cdr(x)));
	return boxNum(n);
}

pico Sub(x)
register pico x;
{
	register number n;
	register pico y;

	y = EVAL1(x);
	NEEDNUM(y);
	n = unBox(y);
	while (isCell(x = cdr(x))) {
		y = EVAL1(x);
		NEEDNUM(y);
		n -= unBox(y);
	}
	return boxNum(n);
}

pico Minus(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(-unBox(x));
}

pico Abs(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return num(x)>=0 ?  x : boxNum(-unBox(x));
}

pico Mul2(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)((num(x) & ~3) << 1 | 2);
}

pico Div2(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)((num(x) & ~3) >> 1 | 2);
}

pico Mul(x)
register pico x;
{
	register number n = 1;
	register pico y;

	do {
		y = EVAL1(x);
		NEEDNUM(y);
		n *= unBox(y);
	} while (isCell(x = cdr(x)));
	return boxNum(n);
}

pico Div(x)
register pico x;
{
	register number n;
	register pico y;

	y = EVAL1(x);
	NEEDNUM(y);
	n = unBox(y);
	while (isCell(x = cdr(x))) {
		y = EVAL1(x);
		NEEDNUM(y);
		if (isZero(y))
			divError();
		n /= unBox(y);
	}
	return boxNum(n);
}

pico Mod(x)
register pico x;
{
	register number n;
	register pico y;

	y = EVAL1(x);
	NEEDNUM(y);
	n = unBox(y);
	while (isCell(x = cdr(x))) {
		y = EVAL1(x);
		NEEDNUM(y);
		if (isZero(y))
			divError();
		n %= unBox(y);
	}
	return boxNum(n);
}

pico Square(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(unBox(x) * unBox(x));
}

pico Sqrt(x)
pico x;
{
	register number a, b, n, r;

	n = num(EVAL1(x));
	x = EVAL1(cdr(x));
	NEEDNUM(n);
	if (n < 0)
		errObj((pico)n, "Bad square root");
	n = unBox(n);
	r = 0;
	a = 1L << 28;
	do {
		b = r + a;
		r >>= 1;
		if (b <= n) {
			n -= b;
			r += a;
		}
	} while (a >>= 2);
	if (!isNil(x) && r < n)
		++r;
	return boxNum(r);
}

/* Multiply and divide without overflow */
pico Muldiv(x)
register pico x;
{
	register pico y;
	register double n;

	y = EVAL1(x);
	NEEDNUM(y);
	n = (double)unBox(y);
	x = cdr(x);
	y = EVAL1(x);
	NEEDNUM(y);
	n *= (double)unBox(y);
	x = cdr(x);
	y = EVAL1(x);
	NEEDNUM(y);
	return boxNum((number)(n / (double)unBox(y)));
}

/* Return distance between two points */
pico DistPt(x)
register pico x;
{
	register pico y;
	register double h,v;

	push(y = EVAL1(x));
	needPoint(y);
	x = cdr(x);
	needPoint(x = EVAL1(x));
	drop();
	h = (double)(unBox(car(y)) - unBox(car(x)));
	v = (double)(unBox(cdr(y)) - unBox(cdr(x)));
	return boxNum((number)(sqrt(h*h + v*v)));
}

/* Convert string to number */
pico Number(x)
register pico x;
{
	register number n, prec;
	bool sign, frac;

	n = 0;
	prec = 0;
	push(EVAL1(x));
	NEEDSTRING(tos);
	if (isCell(x = cdr(x))) {
		prec = num(EVAL1(x));
		NEEDNUM(prec);
		prec = unBox(prec);
	}
	x = pop();
	while (isCell(x)) {  /* Skip white space */
		if (!isNum(car(x)))
			return nilSym;
		if (car(x) > boxNum(' '))
			break;
		x = cdr(x);
	}
	if (isCell(x)) {
		sign = NO;
		if (car(x) == boxNum('-')  ||  car(x) == boxNum('+')) {
			if (car(x) == boxNum('-'))
				sign = YES;
			x = cdr(x);
		}
		frac = NO;
		while (isCell(x)  &&  (!frac  ||  prec > 0)) {
			if (!isNum(car(x)))
				return nilSym;
			if (car(x) >= boxNum('0')  &&  car(x) <= boxNum('9')) {
				n = n * 10 + unBox(car(x)) - '0';
				if (frac)
					--prec;
			}
			else if (car(x) == boxNum('.'))
				frac = YES;
			else
				break;
			x = cdr(x);
		}
		if (isCell(x)  &&  car(x) >= boxNum('5'))
			++n;
		while (--prec >= 0)
			n *= 10;
		if (sign)
			n = -n;
	}
	return boxNum(n);
}

/* Convert number to string */
pico Format(x)
register pico x;
{
	register number n, prec;
	register bool sign;

	n = num(EVAL1(x));
	NEEDNUM((pico)n);
	n = unBox((pico)n);
	prec = 0;
	if (isCell(x = cdr(x))) {
		prec = num(EVAL1(x));
		NEEDNUM(prec);
		prec = unBox(prec);
	}
	sign = NO;
	if (n < 0) {
		sign = YES;
		n = -n;
	}
	push(nilSym);
	do {
		tos = newCell(boxNum(n%10 + '0'), tos);
		if (--prec == 0)
			tos = newCell(boxNum('.'), tos);
	} while (n /= 10);
	if (prec >= 0) {
		if (prec > 0) {
			do
				tos = newCell(boxNum('0'), tos);
			while (--prec);
			tos = newCell(boxNum('.'), tos);
		}
		tos = newCell(boxNum('0'), tos);
	}
	if (sign)
		tos = newCell(boxNum('-'), tos);
	return pop();
}

pico High(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)(num(x) >> 20 & 0xFFCL | 2);
}

pico Middle(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)(num(x) >> 10 & 0xFFCL | 2);
}

pico Low(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return (pico)(num(x) & 0xFFEL);
}

pico Pack(x)
register pico x;
{
	register pico h,m;

	h = EVAL1(x);
	NEEDNUM(h);
	x = cdr(x);
	m = EVAL1(x);
	NEEDNUM(m);
	x = EVAL1(cdr(x));
	NEEDNUM(x);
	return packNum(unBox(h),unBox(m),unBox(x));
}

pico UnPack(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return newCell((pico)(num(x) >> 20 & 0xFFCL | 2),
				newCell((pico)(num(x) >> 10 & 0xFFCL | 2),
					newCell((pico)(num(x) & 0xFFEL),nilSym) ) );
}

pico pRandom(x)
register pico x;
{
	static bits32 seed = 0;

	if (isNil(x = EVAL1(x)))
		return boxNum(seed  =  seed * 1664525 + 1);
	else {
		NEEDNUM(x);
		seed = unBox(x);
		return x;
	}
}

number fixCos(n)
register number n;
{
	if (n < 0)
		n = -n;
	n %= 360;
	if (n > 180)
		n = 360 - n;
	if (n <= 90)
		return cosTab[n];
	else
		return -cosTab[180-n];
}

number fixTan(n)
register number n;
{
	n %= 180;
	if (n <= 90)
		return tanTab[n];
	else
		return -tanTab[180-n];
}

pico Cos(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(fixCos(unBox(x)));
}

pico Sin(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	return boxNum(fixCos(unBox(x) + 270));
}

pico Tan(x)
register pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	x = (pico)unBox(x);
	return boxNum((number)x < 0 ?  -fixTan(-(number)x) : fixTan((number)x));
}
