/* picoPrim.c
 * 21may90abu
 */

#include "pico.h"
#include "stack.h"

static char charTable[32] = {
	  0,  'A',  'B',  'C',  'D',  'E',  'F',  'G',
	'H',  'I',  'J',  'K',  'L',  'M',  'N',  'O',
	'P',  'Q',  'R',  'S',  'T',  'U',  'V',  'W',
	'X',  'Y',  'Z',  '1',  '2',  '$',  '*',  '-'
};

/* Prototypes */
static pico compressShort(char **p);
static pico *hash(pico);
static void strLenErr(pico);

pico boxDouble(d)
double d;
{
	if (d < (double)MAXNEG)
		return boxNum(MAXNEG);
	if (d > (double)MAXNUM)
		return boxNum(MAXNUM);
	return boxNum(d >= 0.0?  d+ROUND: d-ROUND);
}

pico boxVector(v)
register vector *v;
{
	return newCell(boxDouble(v->x), 
					newCell(boxDouble(v->y),
						newCell(boxDouble(v->z), nilSym) ) );
}

void unBoxVector(x,v)
register pico x;
register vector *v;
{
	NEEDCELL(x);
	v->x = (double)unBox(car(x));
	x = cdr(x);
	v->y = (double)unBox(car(x));
	x = cdr(x);
	v->z = (double)unBox(car(x));
}

double distPt(h1,v1,h2,v2)
double h1,v1,h2,v2;
{
	register double h,v;

	h = h2 - h1;
	v = v2 - v1;
	return sqrt(h*h + v*v);
}

bool sameDef(x,y)
register pico x,y;
{
	loop {
		if (x == y)
			return YES;
		if (isNum(x) || isNum(y))
			return NO;
		if (isSym(x) && isSym(y))
			return equal(getPname(x),getPname(y));
		if (!isCell(x) || !isCell(y) || !sameDef(car(x),car(y)))
			return NO;
		x = cdr(x);
		y = cdr(y);
	}
}

void record(sym,prop)
register pico sym;
pico prop;
{
	if (!isNil(val(srcFlg)) && isNum(loadPos)) {
		push(sym);
		push(prop);
		put(newCell(loadPos,loadName));		
		drop2();
	}
}

void redefMsg(sym1,sym2)
pico sym1,sym2;
{
	file *sSave;

	sSave = stream;
	stream = NULL;
	prin0(sym1);
	if (sym2)
		space(), prin0(sym2);
	prString(" redefined\r");
	stream = sSave;
}

void redefine(sym,x)
register pico sym;
pico x;
{
	NEEDSYM(sym);
	CHECKSYM(sym);
	if (val(sym) != voidSym  &&  !sameDef(x,val(sym)))
		redefMsg(sym,NULL);
	setVal(sym,x);
	record(sym,srcFlg);
}

void *nextDynamo(p)
register pico *p;
{
	register pico x;

	x = EVAL1(*p);
	*p = cdr(*p);
	NEEDDYNAMO(x);
	x = cdr(val(x));
	NEEDNUM(x);
	return (void*)unBox(x);
}

number nextNum(p)
register pico *p;
{
	register pico x;

	x = EVAL1(*p);
	*p = cdr(*p);
	NEEDNUM(x);
	return unBox(x);
}

void *nextPtr(p)
register pico *p;
{
	register pico x;

	x = EVAL1(*p);
	*p = cdr(*p);
	if (isNil(x))
		return NULL;
	NEEDNUM(x);
	return (void*)unBox(x);
}

long nextPLong(p)
register pico *p;
{
	register pico x;
	register ResType retVal = 0;
	register int c;

	x = EVAL1(*p);
	*p = cdr(*p);
	while (isCell(x)) {
		c = unBox(car(x));
		if (c >= 0)
			retVal  =  retVal << 8  |  c & 0xFF;
		else do
			retVal  =  retVal << 8  |  ' ';
		while (++c);
		x = cdr(x);
	}
	return retVal;
}

pico bufPLong(n)
long n;
{
	return
		newCell(boxNum(n >> 24 & 0xFF),
			newCell(boxNum(n >> 16 & 0xFF),
				newCell(boxNum(n >> 8 & 0xFF),
					newCell(boxNum(n & 0xFF), nilSym) ) ) );
}

pico boxBool(b)
bool b;
{
	return  b ? tSym : nilSym;
}

pico boxPtr(p)
void *p;
{
	return  p ?  boxNum(p) : nilSym;
}

pico boxHiLo(n)
long n;
{
	return boxNum(n & 1023  |  (n & 1023L<<16) << 4);
}

pico boxLong(n)
long n;
{
	return newCell(boxNum(n>>16 & 0xFFFF), boxNum(n & 0xFFFF));
}

pico Ref(x)
register pico x;
{
	x = EVAL1(x);
	NEEDDYNAMO(x);
	return cdr(val(x));
}

pico Ref2(x)
register pico x;
{
	x = EVAL1(x);
	NEEDDYNAMO(x);
	x = cdr(val(x));
	NEEDNUM(x);
	return boxNum(*(long*)unBox(x));
}

pico AccPtr(x)
register pico x;
{
	register long *adr;

	adr = (long*)EVAL1(x);
	NEEDNUM(adr);
	adr = (long*)unBox(adr);
	if (isNil(x = EVAL1(cdr(x))))
		return boxNum(*adr);
	NEEDNUM(x);
	*adr = unBox(x);
	return x;
}

pico AccWord(x)
register pico x;
{
	register integer *adr;

	adr = (integer*)EVAL1(x);
	NEEDNUM(adr);
	adr = (integer*)unBox(adr);
	if (isNil(x = EVAL1(cdr(x))))
		return boxNum(*adr);
	NEEDNUM(x);
	*adr = unBox(x);
	return x;
}

pico AccByte(x)
register pico x;
{
	register char *adr;

	adr = (char*)EVAL1(x);
	NEEDNUM(adr);
	adr = (char*)unBox(adr);
	if (isNil(x = EVAL1(cdr(x))))
		return boxNum(*adr);
	NEEDNUM(x);
	*adr = unBox(x);
	return x;
}

number maxNumber(a,b)
number a,b;
{
	return (a > b)? a:b;
}

number minNumber(a,b)
number a,b;
{
	return (a <= b)? a:b;
}

number length(x)
register pico x;
{
	register number n = 0;

	NEEDCELL(x);
	do {
		if (++n > MAXLIST)
			circError();
	} while (isCell(x = cdr(x)));
	return n;
}

pico copy(x)
register pico x;
{
	register pico y;
	register number count;

	if (!isCell(x))
		return x;
	count = MAXLIST;
	push(y = newCell(copy(car(x)), cdr(x)));
	while (isCell(x = cdr(x))) {
		if (--count < 0)
			circError();
		setCdr(y, newCell(copy(car(x)), cdr(x)));
		y = cdr(y);
	}
	return pop();
}

pico append(x,y)
register pico x,y;
{
	register pico z;

	if (!isCell(x))
		return y;
	push(x), push(y);
	push(z = newCell(car(x),nilSym));
	while (isCell(x = cdr(x))) {
		setCdr(z, newCell(car(x),nilSym));
		z = cdr(z);
	}
	setCdr(z,y);
	x = pop();
	drop2();
	return x;
}

pico nconc(x,y)
register pico x,y;
{
	register pico z;
	register number count;

	if (!isCell(x))
		return y;
	z = x;
	count = MAXLIST;
	while (isCell(cdr(z))) {
		if (--count < 0)
			circError();
		z = cdr(z);
	}
	setCdr(z,y);
	return x;
}

pico delete(x,y)
register pico x,y;
{
	register pico z;

	if (!isCell(y))
		return y;
	if (equal(x,car(y)))
		return cdr(y);
	push(x), push(y);
	push(z = newCell(car(y),nilSym));
	while (isCell(y = cdr(y))) {
		if (equal(x,car(y))) {
			setCdr(z, cdr(y));
			x = pop();
			drop2();
			return x;
		}
		setCdr(z, newCell(car(y),nilSym));
		z = cdr(z);
	}
	setCdr(z,y);
	x = pop();
	drop2();
	return x;
}

bool equal(x,y)
register pico x,y;
{
	loop {
		if (x == y)
			return YES;
		if (!isCell(x) || !isCell(y) || !equal(car(x),car(y)))
			return NO;
		x = cdr(x);
		y = cdr(y);
	}
}

bool member(x,y)
register pico x,y;
{
	register number count;

	count = MAXLIST;
	while (isCell(y)) {
		if (equal(x,car(y)))
			return YES;
		if (--count < 0)
			circError();
		y = cdr(y);
	}
	return NO;
}

void accumulate(c,p)
register char c;
register number *p;
{
	register char i;

	for (i=0; i < sizeof(charTable)-1; ++i)
		if (c == charTable[i])
			break;
	*p = *p << 5 | i;
}

pico compressShort(p)
register char **p;
{
	number n;
	register int c, cnt;

	cnt = 6;
	do {
		if (c = **p)
			++(*p);
		accumulate(c,&n);
	} while (--cnt);
	return boxNum(n);
}

pico compressName(s)
char *s;
{
	register pico x, y;

	y = compressShort(&s);
	if (!*s)
		return y;
	push(x = newCell(y, nilSym));
	while (y = compressShort(&s), *s) {
		setCdr(x, newCell(y, nilSym));
		x = cdr(x);
	}
	setCdr(x,y);
	return pop();
}

pico compressString(x)
register pico x;
{
	register pico y;
	register int i;
	number n;
	bool first, dummy;

	NEEDSTRING(x);
	i = 5;
	first = YES;
	NEEDNUM(car(x));
	accumulate(upc(unBox(car(x))),&n);
	while (isCell(x = cdr(x))) {
		NEEDNUM(car(x));
		if (--i < 0) {
			if (first) {
				first = NO;
				push(y = newCell(boxNum(n),nilSym));
			}
			else {
				setCdr(y, newCell(boxNum(n),nilSym));
				y = cdr(y);
			}
			i = 5;
		}
		accumulate(upc(unBox(car(x))),&n);
	}
	while (--i >= 0)
		accumulate(0,&n);
	if (first)
		return boxNum(n);
	setCdr(y, boxNum(n));
	return pop();
}

char *expShort(n,dst)
register number n;
register char *dst;
{
	register int cnt;

	dst[6] = '\0';
	cnt = 5;
	do {
		dst[cnt] = charTable[n & 0x1F];
		n >>= 5;
	} while (--cnt >= 0);
	return dst;
}

void prName(x)
register pico x;
{
	char buf[8];

	while (!isNum(x)) {
		prString(expShort(unBox(car(x)), buf));
		x = cdr(x);
	}
	prString(expShort(unBox(x), buf));
}

void strLenErr(x)
pico x;
{
	errStrObj(x,"String too long");
}

char *bufString(x,buf,len)
register pico x;
char buf[];
long len;
{
	register pico y;
	register int c;
	register char *s;

	s = buf;
	y = x;
	while (isCell(y)) {
		NEEDNUM(car(y));
		c = unBox(car(y));
		if (c >= 0) {
			*s++ = c;
			if (--len < 1)
				strLenErr(x);
		}
		else do {
			*s++ = ' ';
			if (--len < 1)
				strLenErr(x);
		} while (++c);
		y = cdr(y);
	}
	*s++ = '\0';
	return buf;
}

int strLength(x)
register pico x;
{
	register int c, n;

	NEEDSTRING(x);
	n = 0;
	while (isCell(x)) {
		c = unBox(car(x));
		if (c >= 0)
			++n;
		else
			n -= c;
		x = cdr(x);
	}
	return n;
}

pico getPname(x)
register pico x;
{
	x = tail(x);
	while (!isNum(x) && !isNum(car(x)))
		x = cdr(x);
	return x;
}

char firstChar(x)
register pico x;
{
	if (!isNum(x = getPname(x)))
		x = car(x);
	return charTable[num(x)>>27 & 0x1F];
}

pico *hash(name)
pico name;
{
	register unsigned long s;
	register unsigned h = 0;

	if (charTable[num(isNum(name)? name:car(name)) >> 27 & 0x1F] == '$')
		return env.threads+THREADS;
	while (isCell(name)) {
		s = num(car(name));
		h += s & 255, s >>= 8;
		h += s & 255, s >>= 8;
		h += s & 255, s >>= 8;
		h += s & 255;
		name = cdr(name);
	}
	s = num(clr(name));
	h += s & 255, s >>= 8;
	h += s & 255, s >>= 8;
	h += s & 255, s >>= 8;
	h += s & 255;
	return env.threads + (h & THREADS-1);
}

pico find(name)
pico name;
{
	register pico x,y,t;

	t = *hash(name);
	while (isCell(t)) {
		if ((x = getPname(car(t))) == name)
			return car(t);
		y = name;
		while (isCell(x) && isCell(y) && car(x)==car(y))
			if ((x = cdr(x)) == (y = cdr(y)))
				return car(t);
		t = cdr(t);
	}
	return NULL;
}

pico intern(sym)
register pico sym;
{
	register pico *p;

	p = hash(getPname(sym));
	*p = newCell(sym, *p);
	return sym;
}

pico Oblist(x)
register pico x;
{
	register long i;

	if (isNil(EVAL1(x))) {
		x = nilSym;
		i = 0;
		do
			x = append(env.threads[i], x);
		while (++i <= THREADS);
	}
	else {
		x = nilSym;
		i = THREADS;
		do
			x = newCell(env.threads[i], x);
		while (--i >= 0);
	}
	return x;
}

pico gensym(v)
pico v;
{
	register long i;

	i =  5;
	while (i > 1  &&  env.genSeed[i]++ == 'Z')
		env.genSeed[i--] = 'A';
	return newSym(compressName(env.genSeed), v);
}

pico Gensym()
{
	return gensym(voidSym);
}

pico Symbol(x)
register pico x;
{
	register pico y, name;

	if (y = find(name = compressString(EVAL1(x))))
		return y;
	push(y = newSym(name, voidSym));
	x = EVAL1(cdr(x));
	drop();
	if (isNil(x))
		return y;
	return intern(y);
}

pico Remob(x)
register pico x;
{
	register pico s, *t;

	x = EVAL1(x);
	NEEDSYM(x);
	if (x > dolSym) {
		t = hash(getPname(x));
		if (car(s = *t) == x) {
			*t = cdr(s);
			return x;
		}
		while (isCell(cdr(s))) {
			if (car(cdr(s)) == x) {
				setCdr(s,cdr(cdr(s)));
				return x;
			}
			s = cdr(s);
		}
	}
	return nilSym;
}
