/* picoSyst.c
 * 02aug90abu
 */

#include "pico.h"
#include "stack.h"
#include <SerialDvr.h>
#include <Color.h>

extern pico portSym;
static char *inAuxName, *outAuxName;

/* Prototypes */
static int convBaud(pico);
static void doBs(void);
static void echoChar(char);
static bool fKey(char);
static char readKey(void);
static void waitTTY(void);

SerShk serShake = {
	YES,		/* fXOn */
	NO,			/* fCTS */
	ctrl('Q'),	/* xOn */
	ctrl('S'),	/* xOff */
	0,			/* errs */
	0,			/* evts */
	YES,		/* fInX */
	NO			/* fDTR */
};

char charQueue[1];
bool qFull = NO;

void exitPico(flag)
int flag;
{
	stopApplication();
	SysEvtMask = saveMask;
	CloseDriver(inTTY);
	CloseDriver(outTTY);
	if (inAux)
		CloseDriver(inAux);
	if (outAux)
		CloseDriver(outAux);
	ExitToShell();
}

pico Check()
{
	keyBreak();
	return tSym;
}

void backTask()
{
	SystemTask();
	if (!isNil(val(taskSym)))
		apply0(val(taskSym));
}

void initSysSyms()
{
	extern symInit macSyms[];
	extern symInit rsrcSyms[], memSyms[], qdSyms[], fontSyms[], eventSyms[], winSyms[];
	extern symInit menuSyms[], textSyms[], dlgSyms[], deskSyms[], toolSyms[], printSyms[];
	extern symInit osSyms[], macioSyms[], ctlSyms[], colorSyms[];

	initSymTab(macSyms);
	initSymTab(rsrcSyms);
	initSymTab(memSyms);
	initSymTab(qdSyms);
	initSymTab(fontSyms);
	initSymTab(eventSyms);
	initSymTab(winSyms);
	initSymTab(menuSyms);
	initSymTab(textSyms);
	initSymTab(dlgSyms);
	initSymTab(deskSyms);
	initSymTab(toolSyms);
	initSymTab(printSyms);
	initSymTab(osSyms);
	initSymTab(macioSyms);
	initSymTab(ctlSyms);
	initSymTab(colorSyms);
}

void initSysVars()
{
	portSym = init1Sym("*PORT", YES, boxNum(thePort));
	init1Sym("*ARROW", YES, boxNum(&arrow));
	init1Sym("*SCREENBITS", YES, boxNum(&screenBits));
}

typedef struct {
	STANDARD_PBHEADER
	integer		ioFRefNum;
	SignedByte	ioFVersNum;
	SignedByte	filler1;
	integer		ioFDirIndex;
	SignedByte	ioFlAttrib;
	SignedByte	filler2;
	FInfo		ioFlFndrInfo;
	long		ioDirID;
	integer		ioFlStBlk;
	long		ioFlLgLen;
	long		ioFlPyLen;
	integer 	ioFlRStBlk;
	long		ioFlRLgLen;
	long		ioFlRPyLen;
	long		ioFlCrDat;
	long		ioFlMdDat;
	long		ioFlBkDat;
	FXInfo		ioFlXFndrInfo;
	long		ioFlParID;
	long		ioFlClpSiz;
} myCInfoPBRec;

pico Path()
{
	Str255 fName;
	myCInfoPBRec cir;

	if (GetVol(fName, &cir.ioVRefNum))
		return nilSym;
	cir.ioNamePtr = fName;
	cir.ioCompletion = NULL;
	cir.ioFDirIndex = -1;
	cir.ioDirID = 0;
	push(nilSym);
	do {
		if (PBGetCatInfo(&cir,NO))
			return nilSym;
		tos = newCell(boxNum(':'), tos);
		tos = nconc(unBufString(cir.ioNamePtr), tos);
	} while ((cir.ioDirID = cir.ioFlParID) != 1);
	return pop();
}

static pico dir(long);
myCInfoPBRec cir;
Str255 fName;

pico dir(id)
long id;
{
	register pico x;
	int i;

	if (cir.ioFDirIndex=-1, cir.ioDirID=id, PBGetCatInfo(&cir,NO))
		return nilSym;
	push(x = newCell(unBufString(cir.ioNamePtr),nilSym));
	i = 1;
	while (cir.ioFDirIndex=i, cir.ioDirID=id, PBGetCatInfo(&cir,NO)==noErr) {
		if (cir.ioFlAttrib & 0x10)
			setCdr(x, newCell(dir(cir.ioDirID),nilSym));
		else
			setCdr(x, newCell(unBufString(cir.ioNamePtr),nilSym));
		x = cdr(x);
		++i;
	}
	return pop();
}

pico Dir(x)
register pico x;
{
	x = EVAL1(x);
	if (GetVol(fName, &cir.ioVRefNum))
		return nilSym;
	cir.ioNamePtr = fName;
	cir.ioCompletion = NULL;
	if (isNil(x)) {
		cir.ioFDirIndex = 1;
		push(nilSym);
		while (cir.ioDirID=0, PBGetCatInfo(&cir,NO)==noErr) {
			tos = newCell(unBufString(cir.ioNamePtr), tos);
			++cir.ioFDirIndex;
		}
		return pop();
	}
	return dir(0);
}

pico pGetVol()
{
	Str255 volName;
	integer vRefNum;

	if (GetVol(volName, &vRefNum))
		return nilSym;
	return boxNum(vRefNum);
}

pico pSetVol(x)
register pico x;
{
	char volName[FILENAME];
	myCInfoPBRec cir;

	if (isNum(x = EVAL1(x)))
		return SetVol((StringPtr)NULL, unBox(x))? nilSym:tSym;
#if 1
	CtoPstr(bufString(x, volName, FILENAME));
	return boxNum(SetVol(volName, 0));
#else
	cir.ioNamePtr = (unsigned char*)volName;
	cir.ioCompletion = NULL;
	cir.ioVRefNum = 0;
	*(long*)((char*)&cir+48) = 0; /* WDDirID */
	return boxNum(PBHSetVol(&cir,NO));
#endif
}

pico PutFile(x)
pico x;
{
	char prompt[256];
	char origName[256];
	Point where;
	SFReply reply;

	where.h = 100;
	where.v = 100;
	CtoPstr(bufString(EVAL1(x),prompt,256));
	x = cdr(x);
	CtoPstr(bufString(EVAL1(x),origName,256));
	SFPutFile(where, (Ptr)prompt, (Ptr)origName, (ProcPtr)NULL, &reply);
	if (!reply.good  ||  SetVol((StringPtr)NULL, reply.vRefNum))
		return nilSym;
	return unBufString(reply.fName);
}

pico GetFile(x)
pico x;
{
	integer types;
	Point where;
	SFReply reply;
	SFTypeList typeList;

	where.h = 100;
	where.v = 100;
	types = 0;
	while (isCell(x) && types < 4)
		typeList[types++] = nextPLong(&x);
	if (!types)
		types = -1;
	SFGetFile(where, (Ptr)NULL, (ProcPtr)NULL, types, &typeList, (ProcPtr)NULL, &reply);
	if (!reply.good  ||  SetVol((StringPtr)NULL, reply.vRefNum))
		return nilSym;
	return unBufString(reply.fName);
}

pico FType(x)
pico x;
{
	char fName[FILENAME];
	FInfo info;

	CtoPstr(bufString(EVAL1(x), fName, FILENAME));
	if (GetFInfo(fName,0,&info) != noErr)
		return nilSym;
	push(bufPLong(info.fdType));
	if (isCell(x = cdr(x))) {
		info.fdType = (OSType)nextPLong(&x);
		SetFInfo(fName,0,&info);
	}
	return pop();
}

pico pDate()
{
	DateTimeRec t;

	GetTime(&t);
	return packNum(t.year % 100, t.month, t.day);
}

pico pTime()
{
	DateTimeRec t;

	GetTime(&t);
	return packNum(t.hour, t.minute, t.second);
}

pico pBench(x)
register pico x;
{
	register long n;
	file *sSave;

	n = TickCount();
	x = evalBody(x);
	n = (TickCount() - n);
	sSave = stream;
	stream = NULL;
	prNumber(n), prString(" ticks [");
	n /= 6;
	prNumber(n/10), chrOut('.'), prNumber(n%10), prString(" sec]\r");
	stream = sSave;
	return x;
}

pico DelayTicks(x)
register pico x;
{
	register pico y;
	register long n;

	y = EVAL1(x);
	x = EVAL1(cdr(x));
	NEEDNUM(y);
	n = unBox(y) + TickCount();
	while (TickCount() < n) {
		keyBreak();
		if (!isNil(x))
			backTask();
	}
	return tSym;
}

pico AccString(x)
register pico x;
{
	unsigned char *p, *adr;
	register int c;
	pico y;

	adr = (unsigned char*)EVAL1(x);
	NEEDNUM(adr);
	adr = (unsigned char*)unBox(adr);
	if (isNil(x = EVAL1(cdr(x))))
		return unBufString(adr);
	p = adr;
	*adr++ = 0;
	y = x;
	while (isCell(x)) {
		c = unBox(car(x));
		if (c >= 0) {
			*adr++ = c;
			++*p;
		}
		else do {
			*adr++ = ' ';
			++*p;
		} while (++c);
		x = cdr(x);
	}
	return y;
}

pico Alloc(x)
pico x;
{
	register char c, *p, *q;
	register number size;

	size = nextNum(&x);
	q = NewPtr((Size)size);
	if (isCell(x)) {
		c = (char)nextNum(&x);
		p = q;
		while (--size >= 0)
			*p++ = c;
	}
	return boxPtr(q);
}

pico Free(x)
pico x;
{
	DisposPtr((Ptr)nextNum(&x));
	return  MemError() ? nilSym : tSym;
}

pico Stuff(x)
register pico x;
{
	register unsigned char *p;
	register pico y;
	register number cnt;

	p = (unsigned char*)EVAL1(x);
	x = cdr(x);
	NEEDNUM(p);
	p = (unsigned char*)unBox(p);
	y = EVAL1(x);
	if (isNum(y)) {
		if (!(cnt = unBox(y)))
			return nilSym;
		--cnt;
		push(x = newCell(boxNum(*p++), nilSym));
		while (--cnt >= 0) {
			setCdr(x, newCell(boxNum(*p++), nilSym));
			x = cdr(x);
		}
		return pop();
	}
	loop {
		NEEDLIST(y);
		while (isCell(y)) {
			NEEDNUM(car(y));
			*p++ = (char)unBox(car(y));
			y = cdr(y);
		}
		if (!isCell(x = cdr(x)))
			return boxNum(p);
		y = EVAL1(x);
	}
}

pico Block(x)
register pico x;
{
	register pico src,dst;

	src = EVAL1(x);
	NEEDNUM(src);
	x = cdr(x);
	dst = EVAL1(x);
	NEEDNUM(dst);
	x = EVAL1(cdr(x));
	NEEDNUM(x);
	BlockMove(unBox(src),unBox(dst),unBox(x));
	return src;
}

/* Init the serial drivers */
void initSerial(baud,f)
int baud;
bool f;
{
	if (OpenDriver(f? "\p.AIn":"\p.BIn", &inTTY) || OpenDriver(f? "\p.AOut":"\p.BOut", &outTTY))
		exitPico(FAIL);
	SerHShake(outTTY, &serShake);
	SerReset(inTTY, baud+data8+stop10+noParity);
	SerReset(outTTY, baud+data8+stop10+noParity);
	inAuxName = f? "\p.BIn":"\p.AIn";
	outAuxName = f? "\p.BOut":"\p.AOut";
}

int convBaud(x)
pico x;
{
	x = EVAL1(x);
	NEEDNUM(x);
	switch (unBox(x)) {
		case 300:
			return baud300+data8+stop10+noParity;
			break;
		case 600:
			return baud600+data8+stop10+noParity;
			break;
		case 1200:
			return baud1200+data8+stop10+noParity;
			break;
		case 2400:
			return baud2400+data8+stop10+noParity;
			break;
		case 4800:
			return baud4800+data8+stop10+noParity;
			break;
		case 9600:
			return baud9600+data8+stop10+noParity;
			break;
		case 19200:
			return baud19200+data8+stop10+noParity;
			break;
		default:
			errObj(x, "Bad baud rate");
	}
}

pico Baud(x)
pico x;
{
	integer n;

	if (inAuxName) {
		if (OpenDriver(inAuxName, &inAux) || OpenDriver(outAuxName, &outAux))
			err("Can't open Serial driver");
		inAuxName = NULL;
		SerHShake(outAux, &serShake);
	}
	n = convBaud(x);
	return boxBool(!SerReset(inAux, n) & !SerReset(outAux, n));
}

pico Tty(x)
pico x;
{
	integer n;

	n = convBaud(x);
	return boxBool(!SerReset(inTTY, n) & !SerReset(outTTY, n));
}

int serialStat()
{
	long count = 1;
	
	SerGetBuf(inAux, &count);
	return (int)count;
}

int serialIn()
{
	long count = 1;
	char buff[4];
	
	if (FSRead(inAux, &count, buff))
		err("Serial input error");
	return buff[0];
}

void serialOut(c)
int c;
{
	long count = 1;
	char buff[4];
	
	buff[0] = c;
	if (FSWrite(outAux, &count, buff))
		err("Serial output error");
}

pico Serial(x)
pico x;
{
	long count;
	char buff[4];
	
	if (isNil(x = EVAL1(x))) {
		if (!serialStat())
			return nilSym;
		return boxNum(serialIn());
	}
	NEEDNUM(x);
	serialOut(unBox(x));
	return x;
}

void waitTTY()
{
	long count = 1;

	if (FSRead(inTTY, &count, charQueue))
		err("TTY input error");
}

bool keyHit()
{
	long count;

	if (qFull)
		return YES;
	SerGetBuf(inTTY, &count);
	if (count) {
		waitTTY();
		return (qFull = YES);
	}
	return NO;
}

char readKey()
{
	if (qFull)
		qFull = NO;
	else
		waitTTY();
	if (!isNil(val(logFlg)))
		setVal(logFlg, newCell(boxNum(charQueue[0]),val(logFlg)));
	return charQueue[0];
}

pico Key()
{
	backTask();
	if (!keyHit())
		return nilSym;
	return boxNum(readKey());
}

pico HitKey()
{
	register pico c;

	backTask();
	if (isCell(val(macFlg))) {
		c = car(val(macFlg));
		setVal(macFlg, cdr(val(macFlg)));
	}
	else {
		while (!keyHit())
			backTask();
		c = boxNum(readKey());
	}
	return c;
}

void keyBreak()
{
	if (keyHit() && charQueue[0] == BREAK) {
		qFull = NO;
		cBreak();
	}
}

void ttyOut(c)
char c;
{
	long count = 1;
	char buf[1];

	buf[0] = c;
	if (FSWrite(outTTY, &count, buf))
		err("TTY output error");
}

void echoChar(c)
char c;
{
	chrOut(c < ' ' ?  '_' : c);
}

void doBs()
{
	chrOut(8);
	chrOut(' ');
	chrOut(8);
}

bool fKey(c)
char c;
{
	register pico x = val(fkeySym);

	while (isCell(x)) {
		if (unBox(car(car(x))) == c) {
			evalBody(cdr(car(x)));
			return YES;
		}
		x = cdr(x);
	}
	return NO;
}

char *getLine(first,last)
char *first,*last;
{
	register pico x;
	register char c, *s;
	register int i,again;

	s = first;
	again = -1;
	loop {
		backTask();
		if (isCell(val(macFlg))) {
			c = unBox(car(val(macFlg)));
			setVal(macFlg, cdr(val(macFlg)));
		}
		else {
			do {
				while (!keyHit())
					backTask();
				c = readKey();
			} while (c == ESCAPE  &&  fKey(c = readKey()));
		}
		switch (c) {
			case EOL:
				while (s <= last)
					*s++ = 0;
				return first;
			case BREAK:
				break;
			case ctrl('H'):
				if (s > first) {
					--s;
					doBs();
				}
				break;
			case ctrl('X'):
				while (s > first) {
					--s;
					doBs();
				}
				break;
			case ctrl('I'):
				for (i=0; i<TABLEN && s<last; ++i)
					chrOut(*s++ = ' ');
				break;
			case ctrl('A'):
				while (s > first) {
					--s;
					doBs();
				}
				x = val(againSym);
				++again;
				for (i=0; isCell(x) && i<again; ++i)
					x = cdr(x);
				if (isCell(x)) {
					bufString(car(x), lBuff, LBSIZE);
					while (*s)
						echoChar(*s++);
				}
				break;
			case ctrl('Z'):
				while (s > first) {
					--s;
					doBs();
				}
				x = val(againSym);
				if (again)
					--again;
				for (i=0; isCell(x) && i<again; ++i)
					x = cdr(x);
				if (isCell(x)) {
					bufString(car(x), lBuff, LBSIZE);
					while (*s)
						echoChar(*s++);
				}
				break;
			case ctrl('R'):
				while (*s)
					echoChar(*s++);
				break;
			case ctrl('C'):
				reset();
				prString("^C\r");
				unwind();
				closeAll();
				longjmp(errRst,-1);
			default:
				if (s < last)
					echoChar(*s++ = c);
				break;
		}
	}
}
