/* picoView.c
 * 22jun90abu
 */

#include "pico.h"
#include "stack.h"
#include "mac.h"

#define EMAX		16
#define FRAMECOLOR	255
#define CLIP		2000000000.0

typedef struct {
	short cnt;
	short h[EMAX];
	unsigned long z[EMAX];
} edge;

/* Camera parameters */
static vector camLoc;
static vector spot;
static double focLen;

/* Frame- and Z-Buffer */
static long sizeH, sizeV, orgH, orgV;
static char *fBuffer;
static unsigned long *zBuffer;
static edge *edges;

/* Transformation coefficients */
static double sinP,cosP, sinY,cosY;
static double coeff1,coeff2,coeff4,coeff5,coeff6,coeff7,coeff8,coeff9;

/* Prototypes */
static void mkEdge(double,double,double,double,double,double);
static void viewEdge(double,double,double,double,double,double);
static void zLine(int,long,long,unsigned long,long,unsigned long);

/* Set Camera parameters */
pico Camera(x)
pico x;
{
	register number i,n;
	register unsigned long m, *p;
	long c1,c2;
	RGBColor col;
	double dx,dy,dz, dist2,dist3;

	focLen = (double)nextNum(&x);			/* Focal length */
	unBoxVector(EVAL1(x),&camLoc);			/* Camera location */
	x = cdr(x);
	unBoxVector(EVAL1(x),&spot);			/* View spot */
	x = cdr(x);
	orgH = (sizeH = nextNum(&x) & ~7) / 2;	/* Sizes (should be a */
	orgV = (sizeV = nextNum(&x) & ~7) / 2;	/* multiple of 8) */
	fBuffer = (char*)nextNum(&x);			/* Buffers */
	zBuffer = (unsigned long*)nextNum(&x);
	nextColor(&x,&col);						/* Ground color */
	c1 = Color2Index(&col) & 0xFF;
	nextColor(&x,&col);						/* Sky color */
	c2 = Color2Index(&col) & 0xFF;
	/* Init coefficients */
	dx = spot.x - camLoc.x;
	dy = spot.y - camLoc.y;
	dz = spot.z - camLoc.z;
	dist2 = sqrt(dx*dx + dy*dy);
	dist3 = sqrt(dx*dx + dy*dy + dz*dz);
	if (dist2 == 0.0) {
		sinY = 0.0;
		cosY = 1.0;
	}
	else {
		sinY = dy / dist2;
		cosY = dx / dist2;
	}
	sinP = dz / dist3;
	cosP = dist2 / dist3;
	coeff1 = -sinY;
	coeff2 = cosY;
	coeff4 = cosY*sinP;
	coeff5 = sinY*sinP;
	coeff6 = -cosP;
	coeff7 = cosY*cosP;
	coeff8 = sinY*cosP;
	coeff9 = sinP;
	/* Init frame buffer */
	n = maxNumber(0, minNumber(sizeV, orgV + (integer)(sinP*focLen / cosP)));
	p = (unsigned long*)fBuffer;
	m = c2 << 24 | c2 << 16 | c2 << 8 | c2;
	i = sizeH * n;
	while ((i-=8) >= 0) {
		p[0] = m;
		p[1] = m;
		p += 2;
	}
	m = c1 << 24 | c1 << 16 | c1 << 8 | c1;
	i = sizeH * (sizeV-n);
	while ((i-=8) >= 0) {
		p[0] = m;
		p[1] = m;
		p += 2;
	}
	/* Init z-buffer */
	p = zBuffer;
	m = 0xFFFFFFFF;
	i = sizeH * sizeV;
	while ((i-=8) >= 0) {
		p[0] = m;
		p[1] = m;
		p[2] = m;
		p[3] = m;
		p[4] = m;
		p[5] = m;
		p[6] = m;
		p[7] = m;
		p += 8;
	}
	return tSym;
}

void mkEdge(x1,y1,z1,x2,y2,z2)
double x1,y1,z1,x2,y2,z2;
{
	register double a,n1,n2;
	register edge *p;
	register char *fPtr;
	register unsigned long *zPtr;
	short hTmp;
	register short *hp;
	unsigned long zTmp;
	register unsigned long z, *zp;
	register number h,i,j,dh,dv,dz,h1,h2,v1,v2,d1,d2;


	if (y2 < y1) {
		a = x1,  x1 = x2,  x2 = a;
		a = y1,  y1 = y2,  y2 = a;
		a = z1,  z1 = z2,  z2 = a;
	}
	/* Clip vertical */
	if (y1 > (n2 = orgV)  ||  y2 <= (n1 = -n2))
		return;
	if (y1 < n1) {
		a = (n1-y1)/(y2-y1); /* m = (x2-x1)/(y2-y1) */
		x1 += a*(x2-x1); /* x1 = x1 + m * (n1 - y1) */
		z1 += a*(z2-z1);
		y1 = n1;
	}
	if (y2 > n2) {
		a = (y2-n2)/(y2-y1);
		x2 += a*(x1-x2);
		z2 += a*(z1-z2);
		y2 = n2;
	}
	if (x1 < -CLIP)
		x1 = -CLIP;
	else if (x1 > CLIP)
		x1 = CLIP;
	if (x2 < -CLIP)
		x2 = -CLIP;
	else if (x2 > CLIP)
		x2 = CLIP;
	d1 = z1 * 10000.0;
	d2 = z2 * 10000.0;
	/* Digital Differential Analyzer */
	dh = (h2 = num(x2)) - (h1 = num(x1));
	dv = (v2 = num(y2)) - (v1 = num(y1));
	dz = d2 - d1;
	h1 += orgH;
	v1 += orgV;
	if (!dv) {
		h2 += orgH;
		if (h1 < 0)
			h1 = 0;
		else if (h1 > sizeH)
			h1 = sizeH;
		if (h2 < 0)
			h2 = 0;
		else if (h2 > sizeH)
			h2 = sizeH;
		if (h1 < h2)
			zLine(FRAMECOLOR,v1,h1,d1,h2,d2);
		else
			zLine(FRAMECOLOR,v1,h2,d2,h1,d1);
	}
	else {
		p = edges + v1;
		fPtr = fBuffer + v1*sizeH;
		zPtr = zBuffer + v1*sizeH;
		for (i = 0; i < dv; ++i) {
			if ((h = h1 + i*dh/dv) < 0)
				h = 0;
			else if (h > sizeH)
				h = sizeH;
			z = d1 + i*dz/dv; /* z not accurate when clipped in h */
			if (h != sizeH  &&  z < zPtr[h])
				zPtr[h] = z, fPtr[h] = FRAMECOLOR;
			if (p->cnt < EMAX-1) {
				hp = p->h;
				zp = p->z;
				j = 0;
				while (j < p->cnt  &&  h > *hp)
					++j, ++hp, ++zp;
				while (j++ < p->cnt) {
					hTmp = *hp, *hp++ = h, h = hTmp;
					zTmp = *zp, *zp++ = z, z = zTmp;
				}
				*hp = h;
				*zp = z;
				++p->cnt;
			}
			++p;
			fPtr += sizeH;
			zPtr += sizeH;
		}
	}
}

void viewEdge(px1,py1,pz1,px2,py2,pz2)
double px1,py1,pz1,px2,py2,pz2;
{
	double n;

	if (pz1 > 1.0  ||  pz2 > 1.0) {
		if (pz1 <= 1.0) {
			n = (1 - pz1) / (pz2 - pz1);
			mkEdge(
				focLen * (px1 + n * (px2-px1)),
				focLen * (py1 + n * (py2-py1)),
				pz1,
				focLen*px2/pz2,
				focLen*py2/pz2,
				pz2 );
		}
		else if (pz2 <= 1.0) {
			n = (1 - pz2) / (pz1 - pz2);
			mkEdge(
				focLen*px1/pz1,
				focLen*py1/pz1,
				pz1,
				focLen * (px2 + n * (px1-px2)),
				focLen * (py2 + n * (py1-py2)),
				pz2 );
		}
		else
			mkEdge(
				focLen*px1/pz1,
				focLen*py1/pz1,
				pz1,
				focLen*px2/pz2,
				focLen*py2/pz2,
				pz2 );
	}
}

/* Z-buffer processing for one line */
static void zLine(col,v,h1,z1,h2,z2)
int col;
long v,h1,h2;
unsigned long z1,z2;
{
	register unsigned long z;
	register long e,dh,dz,d,sz;
	register char *p;
	register unsigned long *q;

	if (dh = h2 - h1) {
		z = z1;
		dz = z2 - z;
		v = v*sizeH + h1;
		p = fBuffer + v;
		q = zBuffer + v;
		sz = 0;
		if (dz > 0)
			sz = 1;
		else if (dz < 0) {
			dz = -dz;
			sz = -1;
		}
		d = 0;
		if (dz > dh) {
			d = dz/dh;
			dz -= d*dh;
			d *= sz;
		}
		e = (dz *= 2) - dh;
		dh *= 2;
		while ((dh-=2) >= 0) {
			if (z < *q)
				*q = z, *p = col;
			z += d;
			if (e >= 0) {
				z += sz;
				e -= dh;
			}
			++p;
			++q;
			e += dz;
		}
	}
}

/* Paint a poly into the buffers */
pico Paint(x)
register pico x;
{
	register pico y;
	register matHandle h;
	pico rgb;
	register long i;
	register int c;
	register edge *p;
	double dx,dy,dz,px0,py0,pz0,px1,py1,pz1,px2,py2,pz2;
	vector pos,v;
	RGBColor col;

	if (!(edges = (edge*)NewPtr(sizeV*sizeof(edge))))
		return nilSym;
	push(EVAL1(x));				/* Plane list */
	x = cdr(x);
	unBoxVector(EVAL1(x),&pos);	/* Position */
	h = NULL; 					/* Optional matrix */
	if (isCell(x = cdr(x))) {
		x = EVAL1(x);
		NEEDNUM(x);
		h = (matHandle)unBox(x);
	}
	x = pop();

	dx = pos.x - camLoc.x;
	dy = pos.y - camLoc.y;
	dz = pos.z - camLoc.z;

	while (isCell(x)) {
		y = car(x);
		x = cdr(x);
		i = sizeV;
		while (--i >= 0)
			edges[i].cnt = 0;
		if (!isNil(rgb = car(y))) {
			unBoxColor(rgb,&col);
			c = Color2Index(&col) & 0xFF;
		}
		y = cdr(y);
		v.x = (double)unBox(car(y));
		y = cdr(y);
		v.y = (double)unBox(car(y));
		y = cdr(y);
		v.z = (double)unBox(car(y));
		if (h) {
			vector tmp;
			tmp = v;
			transform(*h,&tmp,&v);
		}
		v.x += dx;
		v.y += dy;
		v.z += dz;
		px0 = px1 = coeff1*v.x + coeff2*v.y;
		py0 = py1 = coeff4*v.x + coeff5*v.y + coeff6*v.z;
		pz0 = pz1 = coeff7*v.x + coeff8*v.y + coeff9*v.z;
		while (isCell(y = cdr(y))) {
			v.x = (double)unBox(car(y));
			y = cdr(y);
			v.y = (double)unBox(car(y));
			y = cdr(y);
			v.z = (double)unBox(car(y));
			if (h) {
				vector tmp;
				tmp = v;
				transform(*h,&tmp,&v);
			}
			v.x += dx;
			v.y += dy;
			v.z += dz;
			px2 = coeff1*v.x + coeff2*v.y;
			py2 = coeff4*v.x + coeff5*v.y + coeff6*v.z;
			pz2 = coeff7*v.x + coeff8*v.y + coeff9*v.z;
			viewEdge(px1,py1,pz1,px2,py2,pz2);
			px1 = px2;
			py1 = py2;
			pz1 = pz2;
		}
		viewEdge(px2,py2,pz2,px0,py0,pz0);
		if (!isNil(rgb)) {
			p = edges + sizeV;
			while (--p >= edges)
				if (i = p->cnt)
					while ((i-=2) >= 0)
						zLine(c, p-edges, p->h[i],p->z[i],p->h[i+1],p->z[i+1]);
		}
	}
	DisposPtr(edges);
	return tSym;
}
