/* plugin.c
 * Written originally by Alex Burger.
 * Modified by Kazuhiro ohgi
 */

#pragma segment plugin
 
//#define DEBUG
#ifdef DEBUG
#include <Stdio.h>
#endif DEBUG

#include "retch.h"
#include "AcquireInterface.h"
#include "ExportInterface.h"
#include <Errors.h>

#define	DupNamAlertID		129
#define	DskFulAlertID		130
#define FileNotFndAlertID	131
#define BadModeAlertID		132
#define MemFullAlertID		133
#define CannotAcquireID		134

#define NormAlrt		0
#define StopAlrt		1
#define NoteAlrt		2
#define CautAlrt		3

#define WatchCurStart	600
#define WatchCurNum		8

typedef pascal void (*PluginProc)(integer,Ptr,long*,integer*);

/* Prototypes */
static bool openPlugin(StringPtr,ResType);
static integer plugin(integer,Ptr);
static void closePlugin(void);
static pico Plugin(pico);
static pascal bool testAbort(void);
static pascal void updateProgress (long,long);
static pico AcqCMYK(pico);
static void LinesToTiles(uchar*,uchar*,long,long,long);
static pico ExpCMYK(pico);
static short CenterAlert(short);
static void nextWatchCursor(void);
static long getVolSize(short);

/* Globals */
static integer plugFD;
static Handle plugCode;
static long plugData;

#ifdef DEBUG
char debugString[64];
#endif DEBUG

symInit rtPlugSyms[] = {
	{"plugin",	Plugin},
	{"acqCMYK",	AcqCMYK},
	{"expCMYK",	ExpCMYK},
	NULL
};


bool openPlugin(name,type)
StringPtr name;
ResType type;
{
	if ((plugFD = OpenResFile(name)) < 0)
		return NO;
	if (!(plugCode = Get1IndResource(type,1))) {
		CloseResFile(plugFD);
		return NO;
	}
	return YES;
}

integer plugin(sel,stuff)
integer sel;
Ptr stuff;
{
	integer res;

	LoadResource(plugCode);
	MoveHHi(plugCode);
	HLock(plugCode);
	(*(PluginProc)StripAddress(*plugCode))(sel,stuff,&plugData,&res);
	HUnlock(plugCode);
	return res;
}

void closePlugin()
{
	CloseResFile(plugFD);
}

pico Plugin(x)
pico x;
{
	Str255 name;
	ResType type;
	integer sel;
	Ptr stuff;
	pico sym;
	integer res;

	nextCntString(&x,name);
	type = nextPLong(&x);
	sel = (integer)nextNum(&x);
	stuff = nextPtr(&x);
	plugData = unBox(val(sym = nextVar(&x)));
	if (!openPlugin(name,type))
		return nilSym;
	res = plugin(sel,stuff);
	val(sym) = boxNum(plugData);
	closePlugin();
	return boxNum(res);
}

static bool abortFlag;

pascal bool testAbort()
{
	return abortFlag;
}

pascal void updateProgress (done,total)
long done,total;
{
#pragma unused(done,total)
}


/*  AcqCMYK assumes that acquire plug-in module returns the data where */
/* theRect.left = 0 and theRect.right = imageSize.h . This means that  */
/* plug-in module must return the Rect data which includes whole lines.*/
/* If not so, AcqCMYK does not return the right data.(or does not work */
/* at all.                                                             */

static long horzTiles, vertTiles;
static Handle peepHndl;
static long peepInc, peepScl, peepLim;

pico AcqCMYK(x)
pico x;
{
	Afile *dst;
	register uchar *p, *src;
	register long i,h,v;
	Str255 name, volName;
	ResType type;
	AcquireRecord stuff;
	Size grow;
	uchar *dstBuf;
	long count;
	uchar buf[TFHEAD];
	long TopTileOffset, TopPixOffset;
	long CurrentTopTileOffset, CurrentTopPixOffset;
	long BottomTileOffset, BottomPixOffset, LastBottomPixOffset=0;
	long RightTileOffset, RightPixOffset;
	uchar *restBuf;
	short vRefNum, refNum;
	OSErr result;
	Str255 fileName;
	cell c1,c2;

	abortFlag = NO;
	memFill(&stuff,0,sizeof(AcquireRecord));
	nextCntString(&x,name);
	type = nextPLong(&x);
	push(EVAL1(x),c1); /* Progress function */
	x = cdr(x);
	push(EVAL1(x),c2); /* Progress dialog text */
	NEEDSTRING(tos(c2));
	x = cdr(x);
	dst = (Afile*)nextNum(&x);

	if (!openPlugin(name,type)) {
		drop(c1);
		return nilSym;
	}
	stuff.serialNumber = 0;
	stuff.abortProc = (ProcPtr)testAbort;
	stuff.progressProc = (ProcPtr)updateProgress;
	stuff.maxData = MaxMem(&grow) / 2;
	stuff.data = NULL;
	stuff.vRefNum = -1;
	plugData = NULL;
	if (plugin(acquireSelectorPrepare,&stuff)) {
		drop(c1);
		CenterAlert(CannotAcquireID);
		return nilSym;
	}
	if(GetVol(&volName, &vRefNum) != noErr) {
		drop(c1);
		return nilSym;
	}
	if(FSOpen("\pMPSLaunchParam", vRefNum, &refNum) == noErr) {
		count = 2;
		if(FSRead(refNum, &count, (Ptr)&stuff.vRefNum) != noErr) {
			drop(c1);
			FSClose(refNum);
			return nilSym;
		}
		count = 32;
		if(FSRead(refNum, &count, stuff.filename) != noErr) {
			drop(c1);
			FSClose(refNum);
			return nilSym;
		}
		FSClose(refNum);
	}
	if ((result = plugin(acquireSelectorStart,&stuff)) != noErr) {
		drop(c1);
		if(result == fnfErr) {
			CenterAlert(FileNotFndAlertID);
		}
		else if(result != 1) {
			CenterAlert(CannotAcquireID);
		}
		return nilSym;
	}
	for(i=0;i<32;++i)
		fileName[i] = stuff.filename[i]==':'?'/':stuff.filename[i];
	
	
	/*
	 * Assuming that here should be:
	 *   imageMode: 4 (CMYK)
	 *   depth: 8
	 *   planes: 4
	 *   image?Res: any
	 */
	if(stuff.imageMode != acquireModeCMYKColor  ||  stuff.depth != 8  ||  stuff.planes != 4) {
		drop(c1);
		plugin(acquireSelectorFinish,&stuff);
		CenterAlert(BadModeAlertID);
		return nilSym;
	}

	horzTiles = (long)((stuff.imageSize.h + TILE-1) / TILE);	// number of horizontal tiles
	vertTiles = (long)((stuff.imageSize.v + TILE-1) / TILE);	// number of vertical tiles
	
	peepHndl = NULL;
	peepScl = 0;
	if (isCell(x)) {
		peepInc = maxNumber(1,
				TILE / maxNumber(1,minNumber(PEEP/horzTiles,PEEP/vertTiles)) );
		peepScl = TILE / peepInc;
		peepLim = peepScl * peepInc;
		if (!(peepHndl = NewHandle(horzTiles*vertTiles*4*peepScl*peepScl))) {
			drop(c1);
			plugin(acquireSelectorFinish,&stuff);
			return nilSym;
		}
		val(nextVar(&x)) = boxNum(peepHndl);
		val(nextVar(&x)) = boxNum(peepScl);
		val(nextVar(&x)) = boxNum(horzTiles);
		val(nextVar(&x)) = boxNum(vertTiles);
	}

	/* Enough Space? */
	if(getVolSize(vRefNum) < (horzTiles*vertTiles*TILEBUFF+TFHEAD)) {
		drop(c1);
		if (peepHndl)
			DisposHandle(peepHndl);
		CenterAlert(DskFulAlertID);
		plugin(acquireSelectorFinish,&stuff);
		return nilSym;
	}
	
	/* Write header */
	count = TFHEAD;
	*(word*)buf = (word)horzTiles;
	*(word*)(buf+2) = (word)vertTiles;
	*(word*)(buf+4) = (word)stuff.imageSize.h;
	*(word*)(buf+6) = (word)stuff.imageSize.v;
	*(short*)(buf+8) = stuff.vRefNum;
	p = buf+10;
	src = (uchar*)stuff.filename;
	for(i=0;i<256;++i)
		*p++ = *src++;
	if ((result = FSWrite(dst->fd, &count, buf)) != noErr) {
		drop(c1);
		if (peepHndl)
			DisposHandle(peepHndl);
		plugin(acquireSelectorFinish,&stuff);
		if(result == dskFulErr)
			CenterAlert(DskFulAlertID);
		return nilSym;
	}

	/* Process the area of theRect */
	restBuf = nil;

	x = pop(c2);
	apply2(tos(c1), boxNum(vertTiles), x);

	while(1) {
		nextWatchCursor();
		
		/* Continue */
		if (plugin(acquireSelectorContinue,&stuff)) {
			drop(c1);
			if(dstBuf)
				DisposPtr(dstBuf);
			if(restBuf)
				DisposPtr(restBuf);
			if (peepHndl)
				DisposHandle(peepHndl);
			plugin(acquireSelectorFinish,&stuff);
			InitCursor();
			return nilSym;
		}
		
		/* Checking */
		if(stuff.theRect.left != 0 || stuff.theRect.right != stuff.imageSize.h ||
				stuff.colBytes != 4  ||  stuff.loPlane != 0 || stuff.hiPlane != 3 ) {
			drop(c1);
			if (peepHndl)
				DisposHandle(peepHndl);
			plugin(acquireSelectorFinish,&stuff);
			InitCursor();
			CenterAlert(BadModeAlertID);
			return nilSym;
		}
		
		/* theRect is empty */
		if (!stuff.data) {
			if(restBuf) {
				p = restBuf + stuff.rowBytes * LastBottomPixOffset;
				for(v=LastBottomPixOffset;v<TILE;v++)
					for(h=0;h<stuff.rowBytes;++h)
						*p++ = 0;
				if ((dstBuf = NewPtr(horzTiles * TILEBUFF)) == nil) {
					drop(c1);
					if (peepHndl)
						DisposHandle(peepHndl);
					plugin(acquireSelectorFinish,&stuff);
					DisposPtr(restBuf);
					CenterAlert(MemFullAlertID);
					return nilSym;
				}
				LinesToTiles(restBuf, dstBuf, stuff.imageSize.h, stuff.rowBytes, vertTiles-1);
				nextWatchCursor();
				count = horzTiles * TILEBUFF;
				if((result = FSWrite(dst->fd, &count, dstBuf)) != noErr) {
					drop(c1);
					if (peepHndl)
						DisposHandle(peepHndl);
					plugin(acquireSelectorFinish,&stuff);
					DisposPtr(restBuf);
					DisposPtr(dstBuf);
					if(result == dskFulErr)
						CenterAlert(DskFulAlertID);
					InitCursor();
					return nilSym;
				}
				DisposPtr(restBuf);
				restBuf = nil;
				DisposPtr(dstBuf);
				dstBuf = nil;
			}
			break;
		}

		TopTileOffset = stuff.theRect.top / TILE;
		TopPixOffset = stuff.theRect.top % TILE;
		BottomTileOffset = stuff.theRect.bottom / TILE;
		BottomPixOffset = stuff.theRect.bottom % TILE;
		RightTileOffset = stuff.theRect.right / TILE;
		RightPixOffset = stuff.theRect.right % TILE;
		CurrentTopTileOffset = TopTileOffset;
		
		/* If restBuf contains rest lines, process them */
		if(restBuf) {	//  restBuf exists with contents
			/* if TopTileOffset == BottomTileOffset */
			if(TopTileOffset == BottomTileOffset) {	// both top and bottom are in same tile
				src = stuff.data;
				p = restBuf + stuff.rowBytes*TopPixOffset;
				for(v=TopPixOffset;v<BottomPixOffset;++v) {
					for(h=0;h<stuff.rowBytes;++h) {
						*p++ = *src++;
					}
				}
				BottomPixOffset = 0;
			}
			/* if TopTileOffset < BottomTileOffset */
			else {
				src = stuff.data;
				p = restBuf + stuff.rowBytes*TopPixOffset;
				for(v=TopPixOffset;v<TILE;++v) {
					for(h=0;h<stuff.rowBytes;++h) {
						*p++ = *src++;
					}
				}
				if ((dstBuf = NewPtr(horzTiles * TILEBUFF)) == nil) {
					drop(c1);
					if (peepHndl)
						DisposHandle(peepHndl);
					plugin(acquireSelectorFinish,&stuff);
					DisposPtr(restBuf);
					CenterAlert(MemFullAlertID);
					return nilSym;
				}
				LinesToTiles(restBuf, dstBuf, stuff.imageSize.h, stuff.rowBytes, TopTileOffset);
				nextWatchCursor();
				count = horzTiles * TILEBUFF;
				if((result = FSWrite(dst->fd, &count, dstBuf)) != noErr) {
					drop(c1);
					plugin(acquireSelectorFinish,&stuff);
					if (peepHndl)
						DisposHandle(peepHndl);
					DisposPtr(restBuf);
					DisposPtr(dstBuf);
					if(result == dskFulErr)
						CenterAlert(DskFulAlertID);
					InitCursor();
					return nilSym;
				}
				DisposPtr(restBuf);
				restBuf = nil;
				DisposPtr(dstBuf);
				dstBuf = nil;
				CurrentTopTileOffset++;
			}
		}
		
		for(v=CurrentTopTileOffset;v<BottomTileOffset;++v) {
			if(TopPixOffset == 0)
				CurrentTopPixOffset = 0;
			else
				CurrentTopPixOffset = TILE - TopPixOffset;
			src = stuff.data + CurrentTopPixOffset*stuff.rowBytes
					+ stuff.rowBytes*TILE*(v-CurrentTopTileOffset);
			if ((dstBuf = NewPtr(horzTiles * TILEBUFF)) == nil) {
				drop(c1);
				plugin(acquireSelectorFinish,&stuff);
				if (peepHndl)
					DisposHandle(peepHndl);
				CenterAlert(MemFullAlertID);
				return nilSym;
			}
			LinesToTiles(src, dstBuf, stuff.imageSize.h, stuff.rowBytes ,v);
			nextWatchCursor();
			count = horzTiles * TILEBUFF;
			if((result = FSWrite(dst->fd, &count, dstBuf)) != noErr) {
				drop(c1);
				if (peepHndl)
					DisposHandle(peepHndl);
				plugin(acquireSelectorFinish,&stuff);
				DisposPtr(dstBuf);
				if(result == dskFulErr)
					CenterAlert(DskFulAlertID);
				InitCursor();
				return nilSym;
			}
			DisposPtr(dstBuf);
			dstBuf = nil;
		}
		
		/* Making restBuf */
		if(BottomPixOffset != 0) {
			if ((restBuf = NewPtr(stuff.rowBytes * TILE)) == nil) {
				drop(c1);
				plugin(acquireSelectorFinish,&stuff);
				if (peepHndl)
					DisposHandle(peepHndl);
				InitCursor();
				CenterAlert(MemFullAlertID);
				return nilSym;
			}
			p = restBuf;
			src = stuff.data + (stuff.theRect.bottom - stuff.theRect.top - BottomPixOffset) * stuff.rowBytes;
			for(v=0;v<BottomPixOffset;++v)
				for(h=0;h<stuff.rowBytes;++h)
					*p++ = *src++;
			LastBottomPixOffset = BottomPixOffset;
		}
	}

	drop(c1);
	plugin(acquireSelectorFinish,&stuff);
	closePlugin();
	InitCursor();
	return unBufCntString(fileName[0],fileName+1);
}

void LinesToTiles(src, dst, rowPix, rowBytes, vTiles)
uchar *src, *dst;
long rowPix, rowBytes, vTiles;
{
	register uchar *p, *s;
	long horzTiles = (rowPix + TILE - 1) / TILE;
	long RightTileOffset = rowPix / TILE;
	long RightPixOffset = rowPix % TILE;
	register long i,j,h,v;

	if (isNil(apply1(tos(*stkPtr),nilSym)))
		abortFlag = -1;
	for (v = 0; v < TILE; ++v) {
		for (h = 0; h < RightTileOffset; ++h) {
			p = dst + TILEBUFF*h + TILE*4*v;
			s = src + rowBytes*v + TILE*4*h;
			for (i = 0; i < TILE; ++i) {
				*p = *(s+3);
				*(p+1) = *s;
				*(p+2) = *(s+1);
				*(p+3) = *(s+2);
				p += 4;
				s += 4;
			}
		}
		p = dst + TILEBUFF*RightTileOffset + TILE*4*v;
		s = src + rowBytes*v + TILE*4*RightTileOffset;
		if(RightPixOffset!=0) {
			for(i = 0;i < RightPixOffset; ++i) {
				*p = *(s+3);
				*(p+1) = *s;
				*(p+2) = *(s+1);
				*(p+3) = *(s+2);
				p += 4;
				s += 4;
			}
			for(i = RightPixOffset; i < TILE; ++i) {
				*p = 0;
				*(p+1) = 0;
				*(p+2) = 0;
				*(p+3) = 0;
				p += 4;
			}
		}
	}
	if (peepHndl) {
		register pixel *q;

		q = *(pixel**)peepHndl + vTiles * horzTiles * peepScl * peepScl;
		for (j = 0; j < peepLim; j += peepInc)
			for (h = 0; h < horzTiles; ++h)
				for (i = 0; i < peepLim; i += peepInc)
					*q++ = ((pixel*)dst)[(h*TILE + j)*TILE + i];
	}
}


/*  ExpCMYK assumes that export plug-in module returns the Rect of     */
/* theRect.left = 0 and theRect.right = imageSize.h . This means that  */
/* plug-in module must require the Rect data including whole lines.    */
/* If not so, ExpCMYK does not return the right data.(or does not work */
/* at all.                                                             */

pico ExpCMYK(x)
pico x;
{
	register uchar *s;
	register long i,j,h,v;
	long count;
	Str255 name;
	ResType type;
	Afile *src;
	uchar *srcBuf, *dstBuf;
	ExportRecord stuff;
	Size grow;
	uchar buf[TFHEAD];
	long top, bottom;
	long TopTileOffset, TopPixOffset;
	long BottomTileOffset, BottomPixOffset;
	long horzTiles, vertTiles;
	short horzPixels, vertPixels;
	uchar *tmpBuf;
	OSErr result;
	cell c1,c2;
	
	abortFlag = NO;
	memFill(&stuff,0,sizeof(ExportRecord));
	nextCntString(&x,name);
	type = nextPLong(&x);
	push(EVAL1(x),c1); /* Progress function */
	x = cdr(x);
	push(EVAL1(x),c2); /* Progress dialog text */
	x = cdr(x);
	NEEDSTRING(tos(c2));
	src = (Afile*)nextNum(&x);
	stuff.vRefNum = (integer)nextNum(&x);
	nextCntString(&x,stuff.filename);

	SetFPos(src->fd, fsFromStart, 0);
	count = TFHEAD; /* Read header */
	if (FSRead(src->fd, &count, buf)) {
		drop(c1);
		return nilSym;
	}
	horzTiles = (long)(*(word*)buf);
	vertTiles = (long)(*(word*)(buf+2));
	horzPixels = (short)(*(word*)(buf+4));
	vertPixels = (short)(*(word*)(buf+6));

	if (!openPlugin(name,type)) {
		drop(c1);
		return nilSym;
	}
	
	/* Set stuff.vRefNum */
	stuff.vRefNum = *(short *)(buf+8);
	
	/* Writing header (into stuff.finename) */
	s = buf + 10;
	for(i=0;i<256;++i)
		(stuff.filename)[i] = *s++;
		
	stuff.serialNumber = 0;
	stuff.abortProc = (ProcPtr)testAbort;
	stuff.progressProc = (ProcPtr)updateProgress;
	stuff.maxData = MaxMem(&grow) / 2;
	stuff.imageMode = exportModeCMYKColor;
	stuff.imageSize.h = horzPixels;
	stuff.imageSize.v = vertPixels;
	stuff.depth = 8;
	stuff.planes = 4;
	stuff.imageHRes = 72;
	stuff.imageVRes = 72;
	stuff.data = NULL;
	stuff.rowBytes = horzTiles * TILE * 4;

	plugData = NULL;
	if ( (result = plugin(exportSelectorPrepare,&stuff)) ||
						(result = plugin(exportSelectorStart,&stuff)) ) {
		if(result == memFullErr)
			CenterAlert(MemFullAlertID);
		else if(result == fnfErr)
			CenterAlert(FileNotFndAlertID);
		drop(c1);
		return nilSym;
	}
	if((stuff.loPlane != 0) || (stuff.hiPlane != 3)) {
		drop(c1);
		return nilSym;
	}
		
	tmpBuf = nil;
	
	x = pop(c2);
	apply3(tos(c1), boxNum(vertTiles), x, tSym);

	while(1) {
		top = stuff.theRect.top;
		bottom = stuff.theRect.bottom;
		TopTileOffset = (top + TILE - 1) / TILE;
		TopPixOffset = top % TILE;
		BottomTileOffset = bottom / TILE;
		BottomPixOffset = bottom % TILE;
		
		if(EmptyRect(&stuff.theRect)) {
			if(tmpBuf) {
				DisposPtr(tmpBuf);
			}
			break;
		}
		
		if((stuff.theRect.left != 0) || (stuff.theRect.right != horzPixels)) {
			drop(c1);
			if(tmpBuf)
				DisposPtr(tmpBuf);
			plugin(exportSelectorFinish,&stuff);
			InitCursor();
			return nilSym;
		}
		
		
		count = horzTiles * TILE * (bottom -top) * 4;
		if(!(stuff.data = NewPtr(count))) {
			drop(c1);
			if(tmpBuf)
				DisposPtr(tmpBuf);
			plugin(exportSelectorFinish,&stuff);
			InitCursor();
			CenterAlert(MemFullAlertID);
			return nilSym;
		}
		
		dstBuf = (uchar*)stuff.data;
		
		/* Processing the TopPix tile in tmpBuf */
		if(tmpBuf) {	// tmpBuf exists iff TopPixOffset!=0
			/* TopPix and BottomPix in same TILE */
			if(TopTileOffset>BottomTileOffset) {
				for(v=TopPixOffset;v<BottomPixOffset;++v) {
					for(h=0;h<horzTiles;++h) {
						s = tmpBuf + h*TILEBUFF + TILE*v*4;
						for(i=0;i<TILE;++i) {
							*dstBuf = *(s+1);
							*(dstBuf+1) = *(s+2);
							*(dstBuf+2) = *(s+3);
							*(dstBuf+3) = *s;
							dstBuf += 4;
							s += 4;
						}
					}
				}
				BottomPixOffset = 0;
			}
			/* TopPix and BottomPix in different TILE */
			else {
				for(v=TopPixOffset;v<TILE;++v) {
					for(h=0;h<horzTiles;++h) {
						s = tmpBuf + h*TILEBUFF + TILE*v*4;
						for(i=0;i<TILE;++i) {
							*dstBuf = *(s+1);
							*(dstBuf+1) = *(s+2);
							*(dstBuf+2) = *(s+3);
							*(dstBuf+3) = *s;
							dstBuf += 4;
							s += 4;
						}
					}
				}

				DisposPtr(tmpBuf);
				tmpBuf = nil;
			}
		}
		
		/* Processing ordinary tiles */
		count = horzTiles * TILEBUFF;
		if(!(srcBuf = NewPtr(count))) {
			drop(c1);
			if(tmpBuf)
				DisposPtr(tmpBuf);
			plugin(exportSelectorFinish,&stuff);
			InitCursor();
			CenterAlert(MemFullAlertID);
			return nilSym;
		}
		for(v=TopTileOffset;v<BottomTileOffset;++v) {
			nextWatchCursor();
			FSRead(src->fd, &count, srcBuf);
			apply1(tos(c1),nilSym);
			for(i=0;i<TILE;++i) {
				for(h=0;h<horzTiles;++h) {
					s = srcBuf + TILEBUFF*h + TILE*i*4;
					for(j=0;j<TILE;++j) {
						*dstBuf = *(s+1);
						*(dstBuf+1) = *(s+2);
						*(dstBuf+2) = *(s+3);
						*(dstBuf+3) = *s;
						dstBuf += 4;
						s += 4;
					}
				}
			}
		}
		DisposPtr(srcBuf);
		
		/* Making tmpBuf and Processing the BottomPix tile */
		if(BottomPixOffset != 0) {
			count = horzTiles * TILEBUFF;
			if(!(tmpBuf = NewPtr(count))) {
				drop(c1);
				plugin(exportSelectorFinish,&stuff);
				DisposPtr(stuff.data);
				InitCursor();
				CenterAlert(MemFullAlertID);
				return nilSym;
			}
			FSRead(src->fd, &count, tmpBuf);
			apply1(tos(c1),nilSym);
			nextWatchCursor();
			for(v=0;v<BottomPixOffset;++v) {
				for(h=0;h<horzTiles;++h) {
					s = tmpBuf + TILEBUFF*h + TILE*v*4;
					for(i=0;i<TILE;++i) {
						*dstBuf = *(s+1);
						*(dstBuf+1) = *(s+2);
						*(dstBuf+2) = *(s+3);
						*(dstBuf+3) = *s;
						dstBuf += 4;
						s += 4;
					}
				}
			}
		}
		
		/* Continue with exportSelectorContinue */
		if (plugin(exportSelectorContinue,&stuff)) {
			drop(c1);
			plugin(exportSelectorFinish,&stuff);
			DisposPtr(stuff.data);
			return nilSym;
		}
		nextWatchCursor();
		DisposPtr(stuff.data);
	}

	drop(c1);
	plugin(exportSelectorFinish,&stuff);
	closePlugin();
	InitCursor();
	return tSym;
}

short CenterAlert(ID)
short ID;
{
	short	itemHit;
	Handle	dt;
	Rect	*drt, srt;
	long	h, v;
	
	srt = qd.screenBits.bounds;
	dt = GetResource('ALRT', ID);
	HLock(dt);
	drt = (Rect *)(*dt);
	h = (srt.right - drt->right + drt->left)>>1;
	v = (srt.bottom - drt->bottom + drt->top)>>1;
	h -= drt->left;
	v -= drt->top;
	OffsetRect(drt, h, v);
	HUnlock(dt);
	itemHit = Alert(ID, nil);
	ReleaseResource(dt);
	return itemHit;
}

void nextWatchCursor()
{
	static cursNum=0;
	CursHandle watchCursHndl;
	
	watchCursHndl = GetCursor(WatchCurStart + cursNum);
	
	cursNum = (cursNum + 1) % 8;
	SetCursor(*watchCursHndl);
}

long getVolSize(vref)
short	vref;
{
	VolumeParam vp;

	vp.ioCompletion = nil;
	vp.ioNamePtr = nil;
	vp.ioVRefNum = vref;
	vp.ioVolIndex = -1;
	PBGetVInfo((ParmBlkPtr)&vp,0);
	
	return ((long)(vp.ioVFrBlk * vp.ioVAlBlkSiz));
}

