/* picoPrim.c
 * 12jan93abu
 */

#pragma segment picoPrim

#include "pico.h"

static uchar charTable[64] = {
     0,  'A',  'B',  'C',  'D',  'E',  'F',  'G',  'H',
   'I',  'J',  'K',  'L',  'M',  'N',  'O',  'P',  'Q',
   'R',  'S',  'T',  'U',  'V',  'W',  'X',  'Y',  'Z',
   '-',  'a',  'b',  'c',  'd',  'e',  'f',  'g',  'h',
   'i',  'j',  'k',  'l',  'm',  'n',  'o',  'p',  'q',
   'r',  's',  't',  'u',  'v',  'w',  'x',  'y',  'z',
   '0',  '1',  '2',  '3',  '4',
   '5',  '6',  '7',  '8',  '9'
};

/* Prototypes */
static pico compressShort(uchar**);
static pico *hash(pico);
static void strLenErr(pico);

pico boxReal(d)
real d;
{
   if (d < (real)MAXNEG)
      return boxNum(MAXNEG);
   if (d > (real)MAXNUM)
      return boxNum(MAXNUM);
   return boxNum(d >= 0.0?  d+ROUND: d-ROUND);
}

number compare(x,y)
register pico x,y;
{
	number n,m;

	loop {
		if (isNum(x)) {
			if (isNum(y))
				return unBox(y)-unBox(x);
			return isSym(y)? -1:+1;
		}
		if (isSym(x)) {
			if (isNum(y) || isCell(y))
				return +1;
			x = getPname(x);
			y = getPname(y);
			loop {
				n = (number)((unsigned long)(isNum(x)? x:car(x)) >> 2);
				m = (number)((unsigned long)(isNum(y)? y:car(y)) >> 2);
				if (n != m)
					return m-n;
				if (isNum(x) && isNum(y))
					return 0;
				if (isNum(x))
					return +1;
				if (isNum(y))
					return -1;
				x = cdr(x);
				y = cdr(y);
			}
		}
		if (isNum(y) || isSym(y))
			return -1;
		if (n = compare(car(x),car(y)))
			return n;
		x = cdr(x);
		y = cdr(y);
	}
}

bool sameDef(x,y)
register pico x,y;
{
   loop {
      if (x == y)
         return YES;
      if (isNum(x) || isNum(y))
         return NO;
      if (isSym(x) && isSym(y))
         return equal(getPname(x),getPname(y));
      if (!isCell(x) || !isCell(y) || !sameDef(car(x),car(y)))
         return NO;
      x = cdr(x);
      y = cdr(y);
   }
}

void record(sym,prop)
register pico sym;
pico prop;
{
	cell c1,c2;

   if (!isNil(val(srcFlg)) && isNum(loadPos)) {
      push(sym,c1);
      push(prop,c2);
      put(sym,prop,newCell(loadPos,loadName));
      drop(c1);
   }
}

void redefMsg(sym1,sym2)
pico sym1,sym2;
{
   picoFile *sSave;

   sSave = stream;
   setStream(NULL);
   prin0(sym1);
   if (sym2)
      space(), prin0(sym2);
   prString(" redefined\n");
   stream = sSave;
}

void redefine(sym,x)
register pico sym;
pico x;
{
   NEEDSYM(sym);
   CHECKSYM(sym);
   if (val(sym) != voidSym  &&  !sameDef(x,val(sym)))
      redefMsg(sym,NULL);
   val(sym) = x;
   record(sym,srcFlg);
}

void *nextDynamo(p)
register pico *p;
{
   register pico x;

   x = EVAL1(*p);
   *p = cdr(*p);
   NEEDDYNAMO(x);
   x = ref(x);
   NEEDNUM(x);
   return (void*)unBox(x);
}

number nextNum(p)
register pico *p;
{
   register pico x;

   x = EVAL1(*p);
   *p = cdr(*p);
   NEEDNUM(x);
   return unBox(x);
}

void *nextPtr(p)
register pico *p;
{
   register pico x;

   x = EVAL1(*p);
   *p = cdr(*p);
   if (isNil(x))
      return NULL;
   NEEDNUM(x);
   return (void*)unBoxPtr(x);
}

bool nextBool(p)
register pico *p;
{
   register pico x;

   x = EVAL1(*p);
   *p = cdr(*p);
   return  isNil(x) ?  NO : YES;
}

pico nextVar(p)
register pico *p;
{
   register pico sym;

   sym = car(*p);
   *p = cdr(*p);
   NEEDSYM(sym);
   CHECKSYM(sym);
   return sym;
}

uchar *nextString(p,buf,max)
register pico *p;
register uchar *buf;
int max;
{
   register pico x;
   register uchar *s;
   register int c;

   x = EVAL1(*p);
   *p = cdr(*p);
   NEEDSTRING(x);
   if (strLength(x) > max)
      strLenErr(x);
   s = buf;
   while (isCell(x)) {
      c = (int)unBox(car(x));
      if (c >= 0)
         *s++ = (uchar)c;
      else do
         *s++ = ' ';
      while (++c);
      x = cdr(x);
   }
   *s = '\0';
   return buf;
}

pico boxBool(b)
bool b;
{
   return  b ? tSym : nilSym;
}

pico chkPtr(p)
void *p;
{
   return  p ?  boxPtr(num(p)) : nilSym;
}

pico boxHiLo(n)
long n;
{
   return boxNum(n & 1023  |  (n & 1023L<<16) << 4);
}

pico boxLong(n)
long n;
{
   return newCell(boxNum(n>>16 & 0xFFFF), boxNum(n & 0xFFFF));
}

pico Ref(x)
register pico x;
{
   x = EVAL1(x);
   NEEDDYNAMO(x);
   return ref(x);
}

pico Ref2(x)
register pico x;
{
   x = EVAL1(x);
   NEEDDYNAMO(x);
   x = ref(x);
   NEEDNUM(x);
   return boxNum(*(long*)unBoxPtr(x));
}

pico AccPtr(x)
register pico x;
{
   register long *adr;

   adr = (long*)EVAL1(x);
   NEEDNUM(adr);
   adr = (long*)unBoxPtr((pico)adr);
   if (isNil(x = EVAL1(cdr(x))))
      return boxNum(*adr);
   NEEDNUM(x);
   *adr = unBox(x);
   return x;
}

pico AccWord(x)
register pico x;
{
   register short *adr;

   adr = (short*)EVAL1(x);
   NEEDNUM(adr);
   adr = (short*)unBoxPtr((pico)adr);
   if (isNil(x = EVAL1(cdr(x))))
      return boxNum(*adr);
   NEEDNUM(x);
   *adr = (short)unBox(x);
   return x;
}

pico AccByte(x)
register pico x;
{
   register char *adr;

   adr = (char*)EVAL1(x);
   NEEDNUM(adr);
   adr = (char*)unBoxPtr((pico)adr);
   if (isNil(x = EVAL1(cdr(x))))
      return boxNum(*adr);
   NEEDNUM(x);
   *adr = (char)unBox(x);
   return x;
}

pico AccChar(x)
register pico x;
{
   register uchar *adr;

   adr = (uchar*)EVAL1(x);
   NEEDNUM(adr);
   adr = (uchar*)unBoxPtr((pico)adr);
   if (isNil(x = EVAL1(cdr(x))))
      return boxNum(*adr);
   NEEDNUM(x);
   *adr = (uchar)unBox(x);
   return x;
}

number absNumber(a)
number a;
{
   return (a >= 0)? a : -a;
}

number maxNumber(a,b)
number a,b;
{
   return (a > b)? a:b;
}

number minNumber(a,b)
number a,b;
{
   return (a <= b)? a:b;
}

number length(x)
register pico x;
{
   register number n = 0;

   NEEDCELL(x);
   do {
      if (++n > MAXLIST)
         circError();
   } while (isCell(x = cdr(x)));
   return n;
}

pico copy(x)
register pico x;
{
   register pico y;
   register number count;
	cell c1;

   if (!isCell(x))
      return x;
   count = MAXLIST;
   push(y = newCell(copy(car(x)), cdr(x)), c1);
   while (isCell(x = cdr(x))) {
      if (--count < 0)
         circError();
      cdr(y) = newCell(copy(car(x)), cdr(x));
      y = cdr(y);
   }
   return pop(c1);
}

pico copy1(x)
register pico x;
{
   register pico y;
   register number count;
	cell c1,c2;

   if (!isCell(x))
      return x;
   push(x,c1);
   count = MAXLIST;
   push(y = newCell(car(x), cdr(x)), c2);
   while (isCell(x = cdr(x))) {
      if (--count < 0)
         circError();
      cdr(y) = newCell(car(x), cdr(x));
      y = cdr(y);
   }
   drop(c1);
   return tos(c2);
}

pico append(x,y)
register pico x,y;
{
   register pico z;
	cell c1,c2,c3;

   if (!isCell(x))
      return y;
   push(x,c1);
	push(y,c2);
   push(z = newCell(car(x),nilSym), c3);
   while (isCell(x = cdr(x))) {
      cdr(z) = newCell(car(x),nilSym);
      z = cdr(z);
   }
   cdr(z) = y;
   drop(c1);
   return tos(c3);
}

pico nconc(x,y)
register pico x,y;
{
   register pico z;
   register number count;

   if (!isCell(x))
      return y;
   z = x;
   count = MAXLIST;
   while (isCell(cdr(z))) {
      if (--count < 0)
         circError();
      z = cdr(z);
   }
   cdr(z) = y;
   return x;
}

pico doDelete(x,y)
register pico x,y;
{
   register pico z;
	cell c1,c2,c3;

   if (!isCell(y))
      return y;
   if (equal(x,car(y)))
      return cdr(y);
   push(x,c1);
	push(y,c2);
   push(z = newCell(car(y),nilSym), c3);
   while (isCell(y = cdr(y))) {
      if (equal(x,car(y))) {
         cdr(z) = cdr(y);
         drop(c1);
         return tos(c3);
      }
      cdr(z) = newCell(car(y),nilSym);
      z = cdr(z);
   }
   cdr(z) = y;
   drop(c1);
   return tos(c3);
}

bool equal(x,y)
register pico x,y;
{
   loop {
      if (x == y)
         return YES;
      if (!isCell(x) || !isCell(y) || !equal(car(x),car(y)))
         return NO;
      x = cdr(x);
      y = cdr(y);
   }
}

bool member(x,y)
register pico x,y;
{
   register number count;

   count = MAXLIST;
   while (isCell(y)) {
      if (equal(x,car(y)))
         return YES;
      if (--count < 0)
         circError();
      y = cdr(y);
   }
   return NO;
}

void accumulate(c,p)
register int c;
register number *p;
{
   register int i;

   for (i=0; i < sizeof(charTable)-1; ++i)
      if (c == charTable[i])
         break;
   *p = *p << 6 | i;
}

static pico compressShort(p)
register uchar **p;
{
   number n;
   register int c, cnt;

   cnt = 5;
   do {
      if (c = **p)
         ++(*p);
      accumulate(c,&n);
   } while (--cnt);
   return boxNum(n);
}

pico compressName(s)
uchar *s;
{
   register pico x, y;
	cell c1;

   y = compressShort(&s);
   if (!*s)
      return y;
   push(x = newCell(y, nilSym), c1);
   while (y = compressShort(&s), *s) {
      cdr(x) = newCell(y, nilSym);
      x = cdr(x);
   }
   cdr(x) = y;
   return pop(c1);
}

pico compressString(x)
register pico x;
{
   register pico y;
   register int i;
   number n;
   bool first;
	cell c1;

   NEEDSTRING(x);
   i = 4;
   first = YES;
   NEEDNUM(car(x));
   accumulate((int)unBox(car(x)),&n);
   while (isCell(x = cdr(x))) {
      NEEDNUM(car(x));
      if (--i < 0) {
         if (first) {
            first = NO;
            push(y = newCell(boxNum(n),nilSym), c1);
         }
         else {
            cdr(y) = newCell(boxNum(n),nilSym);
            y = cdr(y);
         }
         i = 4;
      }
      accumulate((int)unBox(car(x)),&n);
   }
   while (--i >= 0)
      accumulate(0,&n);
   if (first)
      return boxNum(n);
   cdr(y) = boxNum(n);
   return pop(c1);
}

char *expShort(n,dst)
register number n;
register uchar *dst;
{
   register int cnt;

   dst[5] = '\0';
   cnt = 4;
   do {
      dst[cnt] = charTable[n & 0x3F];
      n >>= 6;
   } while (--cnt >= 0);
   return dst;
}

void prName(x)
register pico x;
{
   char buf[8];

   while (!isNum(x)) {
      prString(expShort(unBox(car(x)), buf));
      x = cdr(x);
   }
   prString(expShort(unBox(x), buf));
}

static void strLenErr(x)
pico x;
{
   errStrObj(x,"String too long");
}

uchar *bufString(x,buf,len)
register pico x;
uchar buf[];
long len;
{
   register pico y;
   register int c;
   register uchar *s;

   s = buf;
   y = x;
   while (isCell(y)) {
      NEEDNUM(car(y));
      c = (int)unBox(car(y));
      if (c >= 0) {
         *s++ = (uchar)c;
         if (--len < 1)
            strLenErr(x);
      }
      else do {
         *s++ = ' ';
         if (--len < 1)
            strLenErr(x);
      } while (++c);
      y = cdr(y);
   }
   *s++ = '\0';
   return buf;
}

int strLength(x)
register pico x;
{
   register int c, n;

   NEEDSTRING(x);
   n = 0;
   while (isCell(x)) {
      c = (int)unBox(car(x));
      if (c >= 0)
         ++n;
      else
         n -= c;
      x = cdr(x);
   }
   return n;
}

pico getPname(x)
register pico x;
{
   x = tail(x);
   while (!isNum(x) && !isNum(car(x)))
      x = cdr(x);
   return x;
}

int firstChar(x)
register pico x;
{
   if (!isNum(x = getPname(x)))
      x = car(x);
   return (int)(num(x)>>26 & 0x3F);
}

int ascii(x)
register pico x;
{
   if (!isNum(x = getPname(x)))
      x = car(x);
   return (int)charTable[num(x)>>26 & 0x3F];
}

static pico *hash(name)
pico name;
{
   register unsigned long s;
   register unsigned long h = 0;

   while (isCell(name)) {
      s = num(car(name));
      h += s & 255, s >>= 8;
      h += s & 255, s >>= 8;
      h += s & 255, s >>= 8;
      h += s & 255;
      name = cdr(name);
   }
   s = num(name);
   h += s & 255, s >>= 8;
   h += s & 255, s >>= 8;
   h += s & 255, s >>= 8;
   h += s & 255;
   return env.threads + (h & THREADS-1);
}

pico find(name,flg)
pico name;
bool flg;
{
   register pico x,y,t;

   t  =  flg? *hash(name) : env.threads[THREADS];
   while (isCell(t)) {
      if ((x = getPname(car(t))) == name)
         return car(t);
      y = name;
      while (isCell(x) && isCell(y) && car(x)==car(y))
         if ((x = cdr(x)) == (y = cdr(y)))
            return car(t);
      t = cdr(t);
   }
   return NULL;
}

bool hidden(sym)
register pico sym;
{
	register pico t;

	for (t = *hash(getPname(sym)); isCell(t); t = cdr(t))
		if (sym == car(t))
			return NO;
	return YES;
}

pico intern(sym,flg)
register pico sym;
bool flg;
{
   register pico *p;

   p  =  flg? hash(getPname(sym)) : env.threads+THREADS;
   *p = newCell(sym, *p);
   return sym;
}

pico Oblist(x)
register pico x;
{
   register long i;

   if (isNil(EVAL1(x))) {
      x = nilSym;
      i = 0;
      do
         x = append(env.threads[i], x);
      while (++i <= THREADS);
   }
   else {
      x = nilSym;
      i = THREADS;
      do
         x = newCell(env.threads[i], x);
      while (--i >= 0);
   }
   return x;
}

pico gensym(v)
pico v;
{
   register int i;

   i =  4;
   while (i > 0  &&  env.genSeed[i]++ == '9')
      env.genSeed[i--] = '0';
   if (i == 0 && env.genSeed[0]++ == 'Z')
      env.genSeed[0] = 'A';
   return newSym(compressName(env.genSeed), v);
}

pico Gensym()
{
   return gensym(voidSym);
}

pico Symbol(x)
register pico x;
{
   register pico y, name;
	cell c1;

   if (y = find(name = compressString(EVAL1(x)), YES))
      return y;
   push(y = newSym(name, voidSym), c1);
   x = EVAL1(cdr(x));
   drop(c1);
   if (isNil(x))
      return y;
   return intern(y,YES);
}

pico Remob(x)
register pico x;
{
   register pico s, *t;

   x = EVAL1(x);
   NEEDSYM(x);
   if (x < heaps->cells || x > topSym) {
      t = hash(getPname(x));
      if (car(s = *t) == x) {
         *t = cdr(s);
		   doZap(x);
         return x;
      }
      while (isCell(cdr(s))) {
         if (car(cdr(s)) == x) {
            cdr(s) = cdr(cdr(s));
		      doZap(x);
            return x;
         }
         s = cdr(s);
      }
   }
   return nilSym;
}
