/* xlsubr - xlisp builtin functions */

#include "xlisp.h"

/* external variables */
extern int (*xlgetc)();
extern struct node *xlstack;
extern struct node *self;

/* local variables */
static char *sgetptr;

/* xlsubr - define a builtin function */
xlsubr(sname,subr)
  char *sname; int (*subr)();
{
    struct node *sym;

    /* enter the symbol */
    sym = xlenter(sname);

    /* initialize the value */
    sym->n_symvalue = newnode(SUBR);
    sym->n_symvalue->n_subr = subr;
}

/* xlsvar - define a builtin string variable */
xlsvar(sname,str)
  char *sname,*str;
{
    struct node *sym;

    /* enter the symbol */
    sym = xlenter(sname);

    /* initialize the value */
    sym->n_symvalue = newnode(STR);
    sym->n_symvalue->n_str = strsave(str);
}

/* xlarg - get the next argument */
struct node *xlarg(pargs)
  struct node **pargs;
{
    struct node *arg;

    /* make sure the argument exists */
    if (*pargs == NULL)
	xlfail("too few arguments");

    /* get the argument value */
    arg = (*pargs)->n_listvalue;

    /* move the argument pointer ahead */
    *pargs = (*pargs)->n_listnext;

    /* return the argument */
    return (arg);
}

/* xlmatch - get an argument and match its type */
struct node *xlmatch(type,pargs)
  int type; struct node **pargs;
{
    struct node *arg;

    /* get the argument */
    arg = xlarg(pargs);

    /* check its type */
    if (type == LIST) {
	if (arg != NULL && arg->n_type != LIST)
	    xlfail("bad argument type");
    }
    else {
	if (arg == NULL || arg->n_type != type)
	    xlfail("bad argument type");
    }

    /* return the argument */
    return (arg);
}

/* xlevarg - get the next argument and evaluate it */
struct node *xlevarg(pargs)
  struct node **pargs;
{
    struct node *oldstk,val;

    /* create a new stack frame */
    oldstk = xlsave(&val,NULL);

    /* get the argument */
    val.n_ptr = xlarg(pargs);

    /* evaluate the argument */
    val.n_ptr = xleval(val.n_ptr);

    /* restore the previous stack frame */
    xlstack = oldstk;

    /* return the argument */
    return (val.n_ptr);
}

/* xlevmatch - get an evaluated argument and match its type */
struct node *xlevmatch(type,pargs)
  int type; struct node **pargs;
{
    struct node *arg;

    /* get the argument */
    arg = xlevarg(pargs);

    /* check its type */
    if (type == LIST) {
	if (arg != NULL && arg->n_type != LIST)
	    xlfail("bad argument type");
    }
    else {
	if (arg == NULL || arg->n_type != type)
	    xlfail("bad argument type");
    }

    /* return the argument */
    return (arg);
}

/* assign - assign a value to a symbol */
static assign(sym,val)
  struct node *sym,*val;
{
    struct node *lptr,*bptr,*optr;

    /* check for a current object */
    if ((optr = self->n_symvalue) != NULL && optr->n_type == OBJ)
	for (lptr = optr->n_obdata; lptr != NULL; lptr = lptr->n_listnext)
	    if ((bptr = lptr->n_listvalue) != NULL && bptr->n_type == BND)
		if (bptr->n_bndsym == sym) {
		    bptr->n_bndvalue = val;
		    return;
		}

    /* not an instance variable of the current object */
    sym->n_symvalue = val;
}

/* eval - evaluate an expression */
static struct node *eval(args)
  struct node *args;
{
    struct node *list;

    /* get the list to evaluate */
    list = xlevmatch(LIST,&args);

    /* make sure there aren't any more arguments */
    if (args != NULL)
	xlfail("too many arguments");

    /* return it evaluated */
    return (xleval(list));
}

/* set - builtin function set */
static struct node *set(args)
  struct node *args;
{
    struct node *oldstk,arg,sym,val;

    /* create a new stack frame */
    oldstk = xlsave(&arg,&sym,&val,NULL);

    /* initialize */
    arg.n_ptr = args;

    /* get the symbol */
    sym.n_ptr = xlevmatch(SYM,&arg.n_ptr);

    /* get the new value */
    val.n_ptr = xlevarg(&arg.n_ptr);

    /* make sure there aren't any more arguments */
    if (arg.n_ptr != NULL)
	xlfail("too many arguments");

    /* assign the symbol the value of argument 2 and the return value */
    assign(sym.n_ptr,val.n_ptr);

    /* restore the previous stack frame */
    xlstack = oldstk;

    /* return the result value */
    return (val.n_ptr);
}

/* setq - builtin function setq */
static struct node *setq(args)
  struct node *args;
{
    struct node *oldstk,arg,sym,val;

    /* create a new stack frame */
    oldstk = xlsave(&arg,&sym,&val,NULL);

    /* initialize */
    arg.n_ptr = args;

    /* get the symbol */
    sym.n_ptr = xlmatch(SYM,&arg.n_ptr);

    /* get the new value */
    val.n_ptr = xlevarg(&arg.n_ptr);

    /* make sure there aren't any more arguments */
    if (arg.n_ptr != NULL)
	xlfail("too many arguments");

    /* assign the symbol the value of argument 2 and the return value */
    assign(sym.n_ptr,val.n_ptr);

    /* restore the previous stack frame */
    xlstack = oldstk;

    /* return the result value */
    return (val.n_ptr);
}

/* load - direct input from a file */
static struct node *load(args)
  struct node *args;
{
    struct node *fname;

    /* get the file name */
    fname = xlevmatch(STR,&args);

    /* make sure there aren't any more arguments */
    if (args != NULL)
	xlfail("too many arguments");

    /* direct input from the file */
    xlfin(fname->n_str);

    /* return the filename */
    return (fname);
}

/* defun - builtin function defun */
static struct node *defun(args)
  struct node *args;
{
    struct node *oldstk,arg,sym,fargs,*fun;

    /* create a new stack frame */
    oldstk = xlsave(&arg,&sym,&fargs,&fun,NULL);

    /* initialize */
    arg.n_ptr = args;

    /* get the function symbol */
    sym.n_ptr = xlmatch(SYM,&arg.n_ptr);

    /* get the formal argument list */
    fargs.n_ptr = xlmatch(LIST,&arg.n_ptr);

    /* create a new function definition */
    fun = newnode(FUN);
    fun->n_funargs = fargs.n_ptr;
    fun->n_funcode = arg.n_ptr;

    /* make the symbol point to a new function definition */
    assign(sym.n_ptr,fun);

    /* restore the previous stack frame */
    xlstack = oldstk;

    /* return the function symbol */
    return (sym.n_ptr);
}

/* sgetc - get a character from a string */
static int sgetc()
{
    if (*sgetptr == 0)
	return (-1);
    else
	return (*sgetptr++);
}

/* read - read an expression */
static struct node *read(args)
  struct node *args;
{
    struct node *val;
    int (*oldgetc)();

    /* save the old input stream */
    oldgetc = xlgetc;

    /* get the string or file pointer */
    if (args != NULL) {
	sgetptr = xlevmatch(STR,&args)->n_str;
	xlgetc = sgetc;
    }

    /* make sure there aren't any more arguments */
    if (args != NULL)
	xlfail("too many arguments");

    /* read an expression */
    val = xlread();

    /* restore the old input stream */
    xlgetc = oldgetc;

    /* return the expression read */
    return (val);
}

/* print - builtin function print */
static struct node *print(args)
  struct node *args;
{
    struct node *oldstk,arg,val;

    /* create a new stack frame */
    oldstk = xlsave(&arg,&val,NULL);

    /* initialize */
    arg.n_ptr = args;

    /* evaluate and print each argument */
    while (arg.n_ptr != NULL)
	xlprint(xlevarg(&arg.n_ptr));

    /* restore previous stack frame */
    xlstack = oldstk;

    /* return null */
    return (NULL);
}

/* fwhile - builtin function while */
static struct node *fwhile(args)
  struct node *args;
{
    struct node *oldstk,farg,arg,*val;
    int done;

    /* create a new stack frame */
    oldstk = xlsave(&farg,&arg,NULL);

    /* initialize */
    farg.n_ptr = arg.n_ptr = args;

    /* loop until test fails */
    for (done = FALSE; TRUE; arg.n_ptr = farg.n_ptr) {

	/* evaluate the test expression */
	if ((val = xlevarg(&arg.n_ptr)) == NULL)
	    break;

	/* check the value type */
	switch (val->n_type) {
	case INT:
		if (val->n_int == 0)
		    done = TRUE;
		break;
	case STR:
		if (strlen(val->n_str) == 0)
		    done = TRUE;
		break;
	}

	/* check for loop done */
	if (done)
	    break;

	/* evaluate each remaining argument */
	while (arg.n_ptr != NULL)
	    xlevarg(&arg.n_ptr);
    }

    /* restore the previous stack frame */
    xlstack = oldstk;

    /* return the last test expression value */
    return (val);
}

/* fif - builtin function if */
static struct node *fif(args)
  struct node *args;
{
    struct node *oldstk,arg,testexpr,thenexpr,elseexpr,*val;
    int dothen;

    /* create a new stack frame */
    oldstk = xlsave(&arg,&testexpr,&thenexpr,&elseexpr,NULL);

    /* initialize */
    arg.n_ptr = args;

    /* evaluate the test expression */
    testexpr.n_ptr = xlevarg(&arg.n_ptr);

    /* get the then clause */
    thenexpr.n_ptr = xlmatch(LIST,&arg.n_ptr);

    /* get the else clause */
    if (arg.n_ptr != NULL)
	elseexpr.n_ptr = xlmatch(LIST,&arg.n_ptr);
    else
	elseexpr.n_ptr = NULL;

    /* make sure there aren't any more arguments */
    if (arg.n_ptr != NULL)
	xlfail("too many arguments");

    /* do else if value is null */
    if (testexpr.n_ptr == NULL)
	dothen = FALSE;

    /* check the value */
    else {

	/* check the value type */
	switch (testexpr.n_ptr->n_type) {
	case INT:
		dothen = (testexpr.n_ptr->n_int != 0);
		break;
	case STR:
		dothen = (strlen(testexpr.n_ptr->n_str) != 0);
		break;
	default:
		dothen = TRUE;
		break;
	}
    }

    /* default the result value to the value of the test expression */
    val = testexpr.n_ptr;

    /* evaluate the appropriate clause */
    if (dothen)
	while (thenexpr.n_ptr != NULL)
	    val = xlevarg(&thenexpr.n_ptr);
    else
	while (elseexpr.n_ptr != NULL)
	    val = xlevarg(&elseexpr.n_ptr);

    /* restore the previous stack frame */
    xlstack = oldstk;

    /* return the last value */
    return (val);
}

/* quote - builtin function to quote an expression */
static struct node *quote(args)
  struct node *args;
{
    /* make sure there is exactly one argument */
    if (args == NULL || args->n_listnext != NULL)
	xlfail("incorrect number of arguments");

    /* return the quoted expression */
    return (args->n_listvalue);
}

/* fexit - get out of xlisp */
fexit()
{
    exit();
}

/* xlinit - xlisp initialization routine */
xlinit()
{
    xlsubr("set",set);
    xlsubr("setq",setq);
    xlsubr("load",load);
    xlsubr("read",read);
    xlsubr("print",print);
    xlsubr("quote",quote);
    xlsubr("while",fwhile);
    xlsubr("defun",defun);
    xlsubr("if",fif);
    xlsubr("eval",eval);
    xlsubr("exit",fexit);

    xlsvar("newline","\n");
    xlsvar("tab","\t");
    xlsvar("bell","\007");
}
