/**************************************************************************************
/* Filename:	gr_conncomp.c
/*		Copyright  1998-99 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	implement the connected components marking algorithm for
/*				dense byteplanes images
/*
/***************************************************************************************/

#include <string.h>
#include "gr_bitplns.h"

	/* 
	 *	Global Variables 
	 */

static int *gConstituency;

	/*
	 *	Not exported functions
	 */

static mimage_ptr	float_connected_components_mimage(mimage_ptr mimg);
static mimage_ptr	discrete_connected_components_mimage(mimage_ptr mimg);
static int			lim_val(int x, int *path, int *rep_of);
static int			constituency_compare(unsigned char *x, unsigned char *y);
static void			equiv(int x, int y, int *constituency, int *rep_of, int *path);
static int			fcalc_pixel_data(fmpixel_ptr *fprior_row, fmpixel_ptr *row, int *prior_row_data, int *row_data, int j, int k, int *constituency, float fWhitePixelValue, int *path, int *rep_of, int *component_counter, int row_len, int planes_number);
static int			calc_pixel_data(mpixel_ptr *prior_row, mpixel_ptr *row, int *prior_row_data, int *row_data, int j, int k, int *constituency, mpixel WhitePixelValue, int *path, int *rep_of, int *component_counter, int row_len, int planes_number);

	/* 
	 *	Implementation 
	 */

/**************************************************************************************
/*	Function:		connected_components_mimage
/*	Description:	create a new image from another using as colors its connected
/*					components
/*
/*	Parameters:
/*		<- mimg		pointer to the image
/*
/*	Result:
/*		not NULL	a pointer to the new connected components image
/*		NULL		error condition
/*
/***************************************************************************************/

mimage_ptr connected_components_mimage(mimage_ptr mimg)
{
	mimage_ptr ret;
	
	switch (mimg->kind) {
		case DISCRETE_IMAGE:
			ret = discrete_connected_components_mimage ( mimg );
			break;
		case FLOAT_IMAGE:
			ret = float_connected_components_mimage ( mimg );
			break;
		default:
			ret = NULL;	
	}

	return ret;
}

/**************************************************************************************
/*	Function:		discrete_connected_components_mimage
/*	Description:	create a new image from another discrete one using as colors
/*					its connected components
/*
/*	Parameters:
/*		<- mimg		pointer to the image
/*
/*	Result:
/*		not NULL	a pointer to the new connected components image
/*		NULL		error condition
/*
/***************************************************************************************/

static mimage_ptr discrete_connected_components_mimage(mimage_ptr mimg)
{
	int c, k, j;
	int nri;

	short last_pixel_equals_black;
	short final_row_pixel_equals_black;
	
	mimage_ptr connected_mimg;
	
	mpixel_ptr conn_buffer;
	mpixel_ptr buffer, *row;
			
	mpixel *sorted_values;
	int sorted_values_size;
	
	int width, height, planes_number, kind;
	size_t type_size;
	size_t rep_of_max_size;
		
	int *rep_of = NULL; 
	int *constituency = NULL;			/* count of number of pixels in the component */
	int *path = NULL; 
	int *roots = NULL;
	int *rep_val = NULL;

	int component_counter = 0;			/* we start with component 0: the outer 'universe' */

	mpixel_ptr *prior_row     = NULL;	/* the row prior to the row being processed */
	mpixel_ptr row_minus_one = NULL;		/* a black row, for the outer 'universe' */

	int nroots;
	
	int *prior_row_data = NULL;			/* all these pixels belong to the 'universe' */
	int *row_data = NULL;				/* is pointer to data for the current row */
	int *first_row_data = NULL;			/* is pointer to data fro the current row */
	int **all_rows_data = NULL;			/* is the master results array */
	
 	mpixel WhitePixelValue;

/* we process the sucessive rows, relating each to the prior row, and                             */
/* equivalencing components whenever they are found to touch. Black pixels touch            	  */
/* another along diagonals, i.e X touches any x to which it stands in the relationship   		  */
/* 									...xxx...                                                     */
/* 									...xX....   					                              */
/* and vice-versa. White pixels touch only along horizontals and verticals, i.e. in the  		  */
/* relationship     ....x....                                                      				  */
/*                  ...xX.... 					and vice-versa.                                   */
/*  																							  */
/* The image as a whole must be rectangular, and is considered to be inside an enclosing white    */
/* 'universe'. For each component we keep the topmost-leftmost mpixel. A black component is inside */
/* a white component W, and vice versa if its topmost-leftmost mpixel has a mpixel of W above it.   */
/* This lets us produce the insideness tree of the components as an output. 					  */
/* We get the number of pixels in the componet (its constituency number) as an 					  */
/* additional output. 																																						*/
/* 																																								 								*/
/* note that the code is written to work for images with any number of distinct grey levels, 	  */
/* of which only one (the 'black' mpixel value) should connect diagonal pixels.					  */

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

	type_size = GET_TYPE_SIZE(kind);

	if (!(sorted_values = sort_mimage(mimg, &sorted_values_size)))
		return NULL;
			
	row = (mpixel_ptr *)malloc( sizeof(mpixel_ptr) * planes_number );
	prior_row = (mpixel_ptr *)malloc( sizeof(mpixel_ptr) * planes_number );
		
	if (!row || !prior_row)
		goto abort_discrete_connected_components;

	rep_of_max_size = (height * width);

	rep_of       = (int *) malloc(rep_of_max_size * sizeof(int));
	constituency = (int *) calloc(rep_of_max_size, sizeof(int)); /* 'outside' starts with 0 pixels */
 	path         = (int *) calloc(rep_of_max_size, sizeof(int));
			
	if (!rep_of || !constituency || !path)
		goto abort_discrete_connected_components;

	for (j=0; j < rep_of_max_size; j++)
		rep_of[j] = -1;
		
	if (!(row_data = calloc(width, sizeof(int))))
		goto abort_discrete_connected_components;
	first_row_data = row_data;
	
	if (!(prior_row_data = calloc(width, sizeof(int))))
		goto abort_discrete_connected_components;

	all_rows_data = (int **)malloc(height * sizeof(int) + height * width * sizeof(int));
	if (!all_rows_data)
		goto abort_discrete_connected_components;
	else
		for (k=0;k<height;k++)
			all_rows_data[k] = (int *)(((char *)all_rows_data) + height * sizeof(int *) +  k * width * sizeof(int));

	if (!(connected_mimg = create_mimage(width, height, 1, DISCRETE_IMAGE, DENSE_IMAGE)))
		goto abort_discrete_connected_components;	

	fast_zero_mimage(connected_mimg);

/*
 */

	WhitePixelValue = sorted_values[sorted_values_size - 1];	/* get the highest intensity of any mpixel color in the input image */

	if (!(row_minus_one = (mpixel_ptr)calloc(width, sizeof(mpixel))))
		goto abort_discrete_connected_components;

	for (c=0; c<planes_number; c++)
		prior_row[c] = row_minus_one;

	conn_buffer = get_byteplane_from_mimage(connected_mimg, 0)->buffer;	/* get the buffer of the connected components image to be constructed */

				/* first pass: analyse each row, using the prior row */
	for (j=0; j<height; j++) 
	{
		
		for (c=0; c<planes_number; c++)		/* the row is a vector of pointers to the image rows in each buffer */
		{
			buffer = get_byteplane_from_mimage(mimg, c)->buffer;	
			row[c] = buffer + width * j;
		}
		
		for (k=0; k<width; k++)				/* pass the row and the prior row to calc_pixel_data */
		{
			row_data[k] = calc_pixel_data(prior_row, row, 
				prior_row_data, row_data, j, k, constituency, 
				WhitePixelValue, path, rep_of, &component_counter, width, planes_number);
		}
		
				/* black pixels on the extreme right must be attached to the 'outside' */

		last_pixel_equals_black = TRUE;

		for (c=0; c<planes_number; c++) {
			if (row[c][width-1] != DISCRETE_BLACK)
				last_pixel_equals_black = FALSE;				
		}
		if (last_pixel_equals_black)
			equiv(row_data[width-1], 0, constituency, rep_of, path);
			
		memcpy(all_rows_data[j], row_data, width * sizeof(int));		/* save the data for the now completed row */
		
		for (c=0; c<planes_number; c++) {
			prior_row[c] = row[c];			/* advance to the next row */
		}
		
		memcpy (prior_row_data, row_data, width * sizeof(int));
	}
	
	nri = height;
	row_data = all_rows_data[nri-1];

	for (k=0; k<width; k++)
	{

		final_row_pixel_equals_black = TRUE;

		for (c=0; c<planes_number; c++)
		{
			mpixel_ptr rowc;
			
			buffer = get_byteplane_from_mimage(mimg, c)->buffer;
			rowc = buffer + width * (nri-1);	/* each black mpixel in the final row must be put into the 'outside' */
	
			if (rowc[k] != DISCRETE_BLACK)
				final_row_pixel_equals_black = FALSE;				
		}

		if (final_row_pixel_equals_black)
			equiv(row_data[k], 0, constituency, rep_of, path);
	}
		/* now make a second pass over the image, setting the mpixel value for each mpixel to the */
		/* representative value of its initially assigned component number */

	roots = (int *) calloc((component_counter+1), sizeof(int));
	if (!roots)
		goto abort_discrete_connected_components;
		
		/* we must now sort the connected components according to their size */
	for (nroots=0, k=0; k<component_counter+1; k++)	/* first collect the list of all 'roots' of connected component */
	{
		if (lim_val(k, path, rep_of) == k) 
		{
			roots[nroots++] = k;
		}
	}
	
	rep_val = (int *) calloc(component_counter+1, sizeof(int));

	if(!rep_val)
		goto abort_discrete_connected_components;

	gConstituency = constituency;
			/* sort with biggest first */
	qsort(roots, nroots, sizeof(int), (int (*)(void const *, void const *))constituency_compare);
									
			/* maps each component into its position in size order */
	for (k=0; k<nroots; k++)
		rep_val[roots[k]]= k;
	
			/* set pixels to their final values */
	for (j=0; j<height; j++) 
		for (k=0; k<width; k++) {
			int v = rep_val[lim_val(all_rows_data[j][k], path, rep_of)];
			*(conn_buffer + j*width + k) = (mpixel)v;
		}
/*
-- rewrite this map into a form which uses the final component numbers

finals := [[repval(x),-neg_count]: [neg_count,x] in finals];	-- populations of components

white := repval(lim_val(0));   -- the representative number asssigned to the outer universe

	-- finally, we must determine the containment relationships
outside_of := 
{[x,if i = 1 then 0 elseif (ard := all_rows_data(i - 1)(j)) = white then 0 else ard end if]:
		 [x,[i,j]] in topmost_leftmost | x /= 0};
		 		  -- maps each component into its immediatelyt containing component
*/	

/*
 */

abort_discrete_connected_components:

	SMART_FREE(sorted_values);
	SMART_FREE(row);
	SMART_FREE(prior_row);
	SMART_FREE(all_rows_data);
	SMART_FREE(first_row_data);
	SMART_FREE(prior_row_data);
	SMART_FREE(row_minus_one);
	SMART_FREE(rep_of);
	SMART_FREE(constituency);
	SMART_FREE(path);
	SMART_FREE(rep_val);
	SMART_FREE(roots);

	return connected_mimg;	
}

/**************************************************************************************
/*	Function:		float_connected_components_mimage
/*	Description:	create a new image from another float one using as colors
/*					its connected components
/*
/*	Parameters:
/*		<- mimg		pointer to the image
/*
/*	Result:
/*		not NULL	a pointer to the new connected components image
/*		NULL		error condition
/*
/***************************************************************************************/

static mimage_ptr float_connected_components_mimage(mimage_ptr mimg)
{
	int c, k, j;
	int nri;

	short last_pixel_equals_black;
	short final_row_pixel_equals_black;
	
	mimage_ptr connected_mimg;
	
	fmpixel_ptr conn_buffer;
	fmpixel_ptr fbuffer, *frow;
			
	fmpixel *sorted_values;
	int sorted_values_size;
	
	int width, height, planes_number, kind;
	size_t type_size;
	size_t rep_of_max_size;
		
	int *rep_of = NULL; 
	int *constituency = NULL; 		/* count of number of pixels in the component */
	int *path = NULL; 
	int *roots = NULL;
	int *rep_val = NULL;

	int component_counter = 0;			/* we start with component 0: the outer 'universe' */

	fmpixel_ptr *fprior_row     = NULL;   /* the row prior to the row being processed */
	fmpixel_ptr frow_minus_one = NULL;   /* a black row, for the outer 'universe' */

	int nroots;
	
	int *prior_row_data = NULL;				/* all these pixels belong to the 'universe' */
	int *row_data = NULL;  	/* is pointer to data fro the current row */
	int *first_row_data = NULL;  	/* is pointer to data fro the current row */
	int **all_rows_data = NULL;					/* is the master results array */
	
 	float fWhitePixelValue;
 	unsigned char WhitePixelValue;

#ifdef __MWERKS__
#pragma unused(WhitePixelValue)
#endif
 	
/* we process the sucessive rows, relating each to the prior row, and                             */
/* equivalencing components whenever they are found to touch. Black pixels touch            	  */
/* another along diagonals, i.e X touches any x to which it stands in the relationship   		  */
/* 									...xxx...                                                     */
/* 									...xX....   					                              */
/* and vice-versa. White pixels touch only along horizontals and verticals, i.e. in the  		  */
/* relationship     ....x....                                                      				  */
/*                  ...xX.... 					and vice-versa.                                   */
/*  																							  */
/* The image as a whole must be rectangular, and is considered to be inside an enclosing white    */
/* 'universe'. For each component we keep the topmost-leftmost mpixel. A black component is inside */
/* a white component W, and vice versa if its topmost-leftmost mpixel has a mpixel of W above it.   */
/* This lets us produce the insideness tree of the components as an output. 										  */
/* We get the number of pixels in the componet (its constituency number) as an 										*/
/* additional output. 																																						*/
/* 																																								 								*/
/* note that the code is written to work for images with any number of distinct grey levels, 		  */
/* of which only one (the 'black' mpixel value) should connect diagonal pixels.										*/

	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

	if (kind != FLOAT_IMAGE)
		return NULL;

	type_size = GET_TYPE_SIZE(kind);

	if (!(sorted_values = sort_mimage(mimg, &sorted_values_size)))
		return NULL;
			
	frow = (fmpixel_ptr *)malloc(sizeof(fmpixel_ptr)*planes_number);
	fprior_row = (fmpixel_ptr *)malloc(sizeof(fmpixel_ptr)*planes_number);
		
	if (!frow || !fprior_row)
		goto abort_connected_components;

	rep_of_max_size = (height * width);

	rep_of       = (int *) malloc(rep_of_max_size * sizeof(int));
	constituency = (int *) calloc(rep_of_max_size, sizeof(int)); /* 'outside' starts with 0 pixels */
 	path         = (int *) calloc(rep_of_max_size, sizeof(int));
			
	if (!rep_of || !constituency || !path)
		goto abort_connected_components;

	for (j=0; j < rep_of_max_size; j++)
		rep_of[j] = -1;
		
	if (!(row_data = calloc(width, sizeof(int))))
		goto abort_connected_components;
	first_row_data = row_data;
	
	if (!(prior_row_data = calloc(width, sizeof(int))))
		goto abort_connected_components;

	all_rows_data = (int **)malloc(height * sizeof(int) + height * width * sizeof(int));
	if (!all_rows_data)
		goto abort_connected_components;
	else
		for (k=0;k<height;k++)
			all_rows_data[k] = (int *)(((char *)all_rows_data) + height * sizeof(int *) +  k * width * sizeof(int));

	if (!(connected_mimg = create_mimage(width, height, 1, FLOAT_IMAGE, DENSE_IMAGE)))
		goto abort_connected_components;	

	fast_zero_mimage(connected_mimg);

	fWhitePixelValue = sorted_values[sorted_values_size - 1];	/* get the highest intensity of any mpixel color in the input image */

	if (!(frow_minus_one = (fmpixel_ptr)malloc(width * sizeof(float))))
		goto abort_connected_components;

	for (k=0; k<width; k++)										/* initialize zero row as the row prior to the first */
		frow_minus_one[k] = (float)0.; /* set to zero float values */

	for (c=0; c<planes_number; c++)
		fprior_row[c] = frow_minus_one;

	conn_buffer = get_fbyteplane_from_mimage(connected_mimg, 0)->buffer;	/* get the buffer of the connected components image to be constructed */

				/* first pass: analyse each row, using the prior row */
	for (j=0; j<height; j++) {
		
		for (c=0; c<planes_number; c++) {	/* the row is a vector of pointers to the image rows in each buffer */
			fbuffer     = get_fbyteplane_from_mimage(mimg, c)->buffer;	
			frow[c] = fbuffer + width * j;
		}
		
		for (k=0; k<width; k++) {   /* pass the row and the prior row to fcalc_pixel_data */
			
			row_data[k] = fcalc_pixel_data(fprior_row, frow, 
											prior_row_data, row_data, 
												j, k, constituency, 
													fWhitePixelValue, path, rep_of, 
														&component_counter, width, planes_number);
		}
		
				/* black pixels on the extreme right must be attached to the 'outside' */

		last_pixel_equals_black = TRUE;

		for (c=0; c<planes_number; c++) {
			if (frow[c][width-1] != FLOAT_BLACK)
				last_pixel_equals_black = FALSE;				
		}
		if (last_pixel_equals_black)
			equiv(row_data[width-1], 0, constituency, rep_of, path);
			
		memcpy(all_rows_data[j], row_data, width * sizeof(int));		/* save the data for the now completed row */
		
		for (c=0; c<planes_number; c++) {
			fprior_row[c] = frow[c];			/* advance to the next row */
		}
		
		memcpy (prior_row_data, row_data, width * sizeof(int));
	}
	
	nri = height;
	row_data = all_rows_data[nri-1];

	for (k=0; k<width; k++) {

		final_row_pixel_equals_black = TRUE;

		for (c=0; c<planes_number; c++) {
			fmpixel_ptr frowc;
			
			fbuffer = get_fbyteplane_from_mimage(mimg, c)->buffer;
			frowc = fbuffer + width * (nri-1);	/* each black mpixel in the final row must be put into the 'outside' */
	
			if (frowc[k] != FLOAT_BLACK)
				final_row_pixel_equals_black = FALSE;				
		}

		if (final_row_pixel_equals_black)
			equiv(row_data[k], 0, constituency, rep_of, path);
	}
		/* now make a second pass over the image, setting the mpixel value for each mpixel to the */
		/* representative value of its initially assigned component number */

	roots = (int *) calloc((component_counter+1), sizeof(int));
	if (!roots)
		goto abort_connected_components;
		
		/* we must now sort the connected components according to their size */
	for (nroots=0, k=0; k<component_counter+1; k++)	/* first collect the list of all 'roots' of connected component */
		if (lim_val(k, path, rep_of) == k) {
			roots[nroots++] = k;
		}

	rep_val = (int *) calloc(component_counter+1, sizeof(int));

	if(!rep_val)
		goto abort_connected_components;

	gConstituency = constituency;
			/* sort with biggest first */
	qsort(roots, nroots, sizeof(int), (int (*)(void const *, void const *))constituency_compare);
									
			/* maps each component into its position in size order */
	for (k=0; k<nroots; k++)
		rep_val[roots[k]]= k;
	
			/* set pixels to their final values */
	for (j=0; j<height; j++) 
		for (k=0; k<width; k++) {
			int v = rep_val[lim_val(all_rows_data[j][k], path, rep_of)];
			*(conn_buffer + j*width + k) = (float)v;
		}
/*
-- rewrite this map into a form which uses the final component numbers

finals := [[repval(x),-neg_count]: [neg_count,x] in finals];	-- populations of components

white := repval(lim_val(0));   -- the representative number asssigned to the outer universe

	-- finally, we must determine the containment relationships
outside_of := 
{[x,if i = 1 then 0 elseif (ard := all_rows_data(i - 1)(j)) = white then 0 else ard end if]:
		 [x,[i,j]] in topmost_leftmost | x /= 0};
		 		  -- maps each component into its immediatelyt containing component
*/	
	
abort_connected_components:

	SMART_FREE(sorted_values);
	SMART_FREE(frow);
	SMART_FREE(fprior_row);
	SMART_FREE(all_rows_data);
	SMART_FREE(first_row_data);
	SMART_FREE(prior_row_data);
	SMART_FREE(frow_minus_one);
	SMART_FREE(rep_of);
	SMART_FREE(constituency);
	SMART_FREE(path);
	SMART_FREE(rep_val);
	SMART_FREE(roots);

	return connected_mimg;	
}

/**************************************************************************************
/*	Function:		equiv
/*	Description:	compressed-balanced tree equivalence processing
/*	Note:			more description in the connected components routine
/*
/*	Parameters:
/*		<- x				x coordinate
/*		<- y				y coordinate
/*		<- constituency		constituency array
/*		<- rep_of			representants array
/*		<- path				path array
/*
/*	Result:
/*		none
/*
/***************************************************************************************/

void equiv(int x, int y, int *constituency, int *rep_of, int *path)
{
	int cx, cy;
	int limval_x, limval_y;
	
		/* get the representatives of x and y; compress */
	limval_x = lim_val(x, path, rep_of);
	limval_y = lim_val(y, path, rep_of);
	
	if (limval_x == limval_y)  /* representatives are already the same */
		return;

	 	/* make the root of the larger tree the parent of the root of the smaller */
	if ((cx = constituency[limval_x]) > (cy = constituency[limval_y])) {
		rep_of[limval_y] = limval_x;
		constituency[limval_x] = cx + cy;
	} else {
		rep_of[limval_x] = limval_y;
		constituency[limval_y] = cx + cy;
	}
}	

/**************************************************************************************
/*	Function:		equiv
/*	Description:	lim_val-balanced tree equivalence processing
/*	Note:			more description in the connected components routine
/*
/*	Parameters:
/*		<- x				x coordinate
/*		<- rep_of			representants array
/*		<- path				path array
/*
/*	Result:
/*		the result
/*
/***************************************************************************************/

int lim_val(int x, int *path, int *rep_of)	/* compressed-balanced tree equivalence processing */
{
	int i;
	int z, newz, result;
	int length_path = 0;	/* empty the path array */
	
	z = x;
	while ((newz = rep_of[z]) != -1) {
		path [length_path++] = z;
		z = newz;
	}
	
	result= z;

	for (i=0; i<length_path; i++) {
		z = path[i];	
		rep_of[z] = result;
	}
	
	return result;
}


/**************************************************************************************
/*	Function:		constituency_compare
/*	Description:	compare for ANSI qsort function
/*	Note:			more description in the connected components routine
/*
/*	Parameters:
/*		<- x				x coordinate
/*		<- y				y coordinate
/*
/*	Result:
/*		standard compare results
/*
/***************************************************************************************/

int constituency_compare(unsigned char *x, unsigned char *y)
{
	return -(gConstituency[*x]-gConstituency[*y]);
}

int fcalc_pixel_data(fmpixel_ptr *prior_row, fmpixel_ptr *row, 
			int *prior_row_data, int *row_data, int j, 
				int k, int *constituency, float fWhitePixelValue, 
					int *path, int *rep_of, int *component_counter, int row_len, int planes_number)
{
#ifdef __MWERKS__
#pragma unused(j)
#endif

	short value_left_equals_this_value;
	short value_left_above_equals_this_value;
	short value_right_above_equals_this_value;
	short value_above_equals_this_value;
	short this_value_equals_fwhite_pixel_value;

	float value_left_above, value_right_above;
	int data_left_above, data_right_above;
		
	float value_above;
	int data_above;

	float value_left;
	int data_left;

	float this_value;
	int this_data;

	char done; /* boolean */
	int the_rep;

	int n;
	
	/*  assigns an existing component number to a mpixel, or generates a new component number. */
	/*  may trigger equivalencing of prior component numbers */

	value_left_equals_this_value = value_above_equals_this_value = this_value_equals_fwhite_pixel_value = TRUE;
	
	for (n=0; n<planes_number; n++) {
					
		value_left = (k>0) ? row[n][k-1] : FLOAT_BLACK;		/*  get info for mpixel to the left */
		
		this_value = row[n][k];			/*  color of this mpixel */
		
		value_above = prior_row[n][k];		/*  get info for mpixel above */
		
		if (value_left != this_value)						/* look for inequality in plane n */
			value_left_equals_this_value = FALSE;
			
		if (value_above != this_value)						/* look for inequality in plane n */
			value_above_equals_this_value = FALSE;

		if (this_value != fWhitePixelValue)					/* look for white in plane n */
			this_value_equals_fwhite_pixel_value = FALSE;
	}		
	
	data_left = (k>0) ? row_data[k-1] : 0;
	data_above = prior_row_data[k];

	done = 1; /* true */		/*  will be set false in 4th branch below */
	
	if (value_left_equals_this_value && value_above_equals_this_value) {

		this_data = data_left;	/*  use the component number of the mpixel to the left */
		the_rep = lim_val(this_data, path, rep_of);
		constituency[the_rep] ++;
		equiv(data_left, data_above, constituency, rep_of, path);	/*  note that the two component numbers are equivalent */

	} else if (value_left_equals_this_value) {

		this_data = data_left;	/*  use the component number of the mpixel to the left */
		the_rep = lim_val(this_data, path, rep_of);
		constituency[the_rep] ++;

	} else if (value_above_equals_this_value) {

		this_data = data_above;	/*  use the component number of the mpixel above */
		the_rep = lim_val(this_data, path, rep_of);		
		constituency[the_rep] ++;

	} else
		done = 0; /* false */		/*  more  processing is always required in this case */

	if (this_value_equals_fwhite_pixel_value)	{		/*  this mpixel does not connect across diagonals */
		if (!done) { 		/*  use a new component number, and note that this is the */
										/*  topmost-leftmost mpixel of the component */
			this_data = ++*component_counter;
			constituency[this_data] = 1;
		}
	} else {		/*  this 'black' mpixel connects with others across diagonals */

		value_left_above_equals_this_value = value_right_above_equals_this_value = TRUE;

		for (n=0; n<planes_number; n++) {
								/*  get info for first diagonal mpixel */
			value_left_above = (k>0) ? prior_row[n][k-1] : FLOAT_BLACK;
			value_right_above = (k<(row_len-1)) ? prior_row[n][k+1] : FLOAT_BLACK;
			
			if (value_left_above != this_value)
				value_left_above_equals_this_value = FALSE;
				
			if (value_right_above != this_value)
				value_right_above_equals_this_value = FALSE;
		}

		data_right_above = (k<(row_len-1)) ? prior_row_data[k+1] : 0;
		data_left_above = (k>1) ? prior_row_data[k-1] : 0;

		if (done) { 		/*  only some equivalencing remains */
			if (value_left_above_equals_this_value)
				equiv(data_left_above, this_data, constituency, rep_of, path);	/*  the two component numbers are equivalent */
			
			if (value_right_above_equals_this_value)
				equiv(data_right_above, this_data, constituency, rep_of, path); 	/*  the two component numbers are equivalent */

		} else { 				/*  try to get a component value from the diagonal pixels  */
		
			if (value_left_above_equals_this_value && value_right_above_equals_this_value) {

				this_data = data_left_above;	/*  use the component number of the left diagonal */
				the_rep = lim_val(this_data, path, rep_of);
				constituency[the_rep]++;
				equiv(data_left_above, data_right_above, constituency, rep_of, path);
						/*  note that the two diagonal component numbers are equivalent */
			
			} else if (value_left_above_equals_this_value) {

				this_data = data_left_above;	/*  use the component number of the mpixel to the left */
				the_rep = lim_val(this_data, path, rep_of);
				constituency[the_rep]++;

			} else if (value_right_above_equals_this_value) {

				this_data = data_right_above;	/*  use the component number of the mpixel above */
				the_rep = lim_val(this_data, path, rep_of);
				constituency[the_rep]++;

			} else {		/*  we must start a new component */

				this_data = ++*component_counter;
				constituency[this_data] = 1;
			}
		}
	}

	return this_data;
}

/**************************************************************************************
/*	Function:		calc_pixel_data
/*	Description:	calculate the pixel data
/*	Note:			more description in the connected components routine
/*
/*	Parameters:
/*		?-	prior_row			description in the connected components routine
/*		?-	row					description in the connected components routine
/*		?-	prior_row_data		description in the connected components routine
/*		?-	row_data			description in the connected components routine
/*		?-	j					description in the connected components routine
/*		?-	k					description in the connected components routine
/*		?-	constituency		description in the connected components routine
/*		?-	WhitePixelValue		description in the connected components routine
/*		?-	path				description in the connected components routine
/*		?-	rep_of				description in the connected components routine
/*		?-	component_counter	description in the connected components routine
/*		?-	row_len				description in the connected components routine
/*		?-	planes_number		description in the connected components routine
/*
/*	Result:
/*		description in the connected components routine
/*
/***************************************************************************************/

int calc_pixel_data(mpixel_ptr *prior_row, mpixel_ptr *row, 
	int *prior_row_data, int *row_data, int j, 
	int k, int *constituency, mpixel WhitePixelValue, 
	int *path, int *rep_of, int *component_counter, 
	int row_len, int planes_number)
{
#ifdef __MWERKS__
#pragma unused(j)
#endif

	short value_left_equals_this_value;
	short value_left_above_equals_this_value;
	short value_right_above_equals_this_value;
	short value_above_equals_this_value;
	short this_value_equals_fwhite_pixel_value;

	mpixel value_left_above, value_right_above;
	int data_left_above, data_right_above;
		
	mpixel value_above;
	int data_above;

	mpixel value_left;
	int data_left;

	mpixel this_value;
	int this_data;

	char done; /* boolean */
	int the_rep;

	int n;
	
	/*  assigns an existing component number to a mpixel, or generates a new component number. */
	/*  may trigger equivalencing of prior component numbers */

	value_left_equals_this_value = value_above_equals_this_value = this_value_equals_fwhite_pixel_value = TRUE;
	
	for (n=0; n<planes_number; n++) {
					
		value_left = (k>0) ? row[n][k-1] : DISCRETE_BLACK;		/*  get info for mpixel to the left */
		
		this_value = row[n][k];			/*  color of this mpixel */
		
		value_above = prior_row[n][k];		/*  get info for mpixel above */
		
		if (value_left != this_value)						/* look for inequality in plane n */
			value_left_equals_this_value = FALSE;
			
		if (value_above != this_value)						/* look for inequality in plane n */
			value_above_equals_this_value = FALSE;

		if (this_value != WhitePixelValue)					/* look for white in plane n */
			this_value_equals_fwhite_pixel_value = FALSE;
	}		
	
	data_left = (k>0) ? row_data[k-1] : 0;
	data_above = prior_row_data[k];

	done = 1; /* true */		/*  will be set false in 4th branch below */
	
	if (value_left_equals_this_value && value_above_equals_this_value) {

		this_data = data_left;	/*  use the component number of the mpixel to the left */
		the_rep = lim_val(this_data, path, rep_of);
		constituency[the_rep] ++;
		equiv(data_left, data_above, constituency, rep_of, path);	/*  note that the two component numbers are equivalent */

	} else if (value_left_equals_this_value) {

		this_data = data_left;	/*  use the component number of the mpixel to the left */
		the_rep = lim_val(this_data, path, rep_of);
		constituency[the_rep] ++;

	} else if (value_above_equals_this_value) {

		this_data = data_above;	/*  use the component number of the mpixel above */
		the_rep = lim_val(this_data, path, rep_of);		
		constituency[the_rep] ++;

	} else
		done = 0; /* false */		/*  more  processing is always required in this case */

	if (this_value_equals_fwhite_pixel_value)	{		/*  this mpixel does not connect across diagonals */
		if (!done) { 		/*  use a new component number, and note that this is the */
										/*  topmost-leftmost mpixel of the component */
			this_data = ++*component_counter;
			constituency[this_data] = 1;
		}
	} else {		/*  this 'black' mpixel connects with others across diagonals */

		value_left_above_equals_this_value = value_right_above_equals_this_value = TRUE;

		for (n=0; n<planes_number; n++) {
								/*  get info for first diagonal mpixel */
			value_left_above = (k>0) ? prior_row[n][k-1] : DISCRETE_BLACK;
			value_right_above = (k<(row_len-1)) ? prior_row[n][k+1] : DISCRETE_BLACK;
			
			if (value_left_above != this_value)
				value_left_above_equals_this_value = FALSE;
				
			if (value_right_above != this_value)
				value_right_above_equals_this_value = FALSE;
		}

		data_right_above = (k<(row_len-1)) ? prior_row_data[k+1] : 0;
		data_left_above = (k>1) ? prior_row_data[k-1] : 0;

		if (done) { 		/*  only some equivalencing remains */
			if (value_left_above_equals_this_value)
				equiv(data_left_above, this_data, constituency, rep_of, path);	/*  the two component numbers are equivalent */
			
			if (value_right_above_equals_this_value)
				equiv(data_right_above, this_data, constituency, rep_of, path); 	/*  the two component numbers are equivalent */

		} else { 				/*  try to get a component value from the diagonal pixels  */
		
			if (value_left_above_equals_this_value && value_right_above_equals_this_value) {

				this_data = data_left_above;	/*  use the component number of the left diagonal */
				the_rep = lim_val(this_data, path, rep_of);
				constituency[the_rep]++;
				equiv(data_left_above, data_right_above, constituency, rep_of, path);
						/*  note that the two diagonal component numbers are equivalent */
			
			} else if (value_left_above_equals_this_value) {

				this_data = data_left_above;	/*  use the component number of the mpixel to the left */
				the_rep = lim_val(this_data, path, rep_of);
				constituency[the_rep]++;

			} else if (value_right_above_equals_this_value) {

				this_data = data_right_above;	/*  use the component number of the mpixel above */
				the_rep = lim_val(this_data, path, rep_of);
				constituency[the_rep]++;

			} else {		/*  we must start a new component */

				this_data = ++*component_counter;
				constituency[this_data] = 1;
			}
		}
	}

	return this_data;
}
