/**************************************************************************************
/* Filename:	gr_wavelet.c
/*		Copyright  1999 Giuseppe Di Mauro. All rights reserved.
/*
/* Description:	wavelet transform routines for the graphic library
/*
/* Disclaimer:	part of the code from "Numerical Recipes in C" book
/*
/***************************************************************************************
/* Revision History:
/*		05-24-1999: start coding
/***************************************************************************************/
 
#include <math.h>
#include "gr_bitplns.h"
#include "gr_errors.h"
#include "gr_wavelet.h"
#include "gr_ops.h"

	/*
	 *	Not exported functions
	 */

static void wtn(float a[], unsigned long nn[], int ndim, int isign, void (*wtstep)(float [], unsigned long, int));
static void daub4(float a[], unsigned long n, int isign);
static void free_vector(float *v, long nl, long nh);
static float *vector(long nl, long nh);
static void nrerror(char error_text[]);

	/*
	 *	Implementation
	 */

/**************************************************************************************
/*	Function:		wavelet_mimage
/*	Description:	compute the wavelet transform of an image or its inverse
/*
/*	Parameters:
/*		<- mimg			image we want to make the wavelet transform
/*		-> res_mimg		result transformed image
/*		<- isign		1 for transform, -1 for inverse transform
/*
/*	Result:
/*		kGrOk (0)		ok
/*		else			an error condition
/*
/***************************************************************************************/

int wavelet_mimage(mimage_ptr mimg, mimage_ptr *res_mimg, int isign)
{
	int c, i;
	mimage_ptr target_mimg;
	int ndim;
	unsigned long nn[3];
	float *data;
	int height, width, kind, planes_number;
	
	*res_mimg = NULL;	/* reset the destination image */
	
	if (IS_SPARSE(mimg))	/* check if a sparse image is passed */
		return kGrDensityMismatch;	/* and return the correct error code */

		/* get info about the image */
	get_mimage_info(mimg, &height, &width, &planes_number, &kind, NULL);

		/* check if the size is a power of 2 */	
	if (!is_pow_of_2(height) || !is_pow_of_2(width))
		return kGrNotCorrectSizes;

	ndim = 2;	/* an image is a two dimensions array */

		/* allocate a buffer for wavelet data */
	data = malloc(sizeof(float) * (height * width + 1));	/* data vector */
	if (!data) 
		return kGrOutOfMemory;

		/* prepare to call the wavelet transform */
	nn[1] = height;	/* the first dimension size is the height */
	nn[2] = width;	/* the second is the width */

		/* create the destination image */
	*res_mimg = target_mimg = create_mimage(width, height, planes_number, FLOAT_IMAGE, DENSE_IMAGE);
	if (!target_mimg)
		return kGrOutOfMemory;
		
		/* loop over the planes */
	for (c=0; c<planes_number; c++) {

		fmpixel_ptr target_fbuffer;

			/* move the data from the image to the buffer used for the transform */
		if (IS_DISCRETE(mimg)) {	/* discrete case */
			mpixel_ptr buffer = get_byteplane_from_mimage(mimg, c)->buffer;			
			
			for (i=1; i<=(height*width); i++)
				data[i] = (float)*buffer++;
				
		} else {	/* float case */
			fmpixel_ptr buffer = get_fbyteplane_from_mimage(mimg, c)->buffer;			
			
			for (i=1; i<=(height*width); i++)
				data[i] = *buffer++;
		}

			/* execute the wavelet transform */
		wtn(data, nn, ndim, isign, daub4);

			/* store the transformed data into the destination image */
		target_fbuffer = get_fbyteplane_from_mimage(target_mimg, c)->buffer;
		for (i=1; i<=(height*width); i++)
			*target_fbuffer++ = data[i];
	}
	
		/* empty the buffers used for the transform */
	SMART_FREE(data);
	
	return kGrOk;
}

	/*
	 *	Routines from "Numerical Recipes in C" book
	 */

void wtn(float a[], unsigned long nn[], int ndim, int isign,
	void (*wtstep)(float [], unsigned long, int))
{
	unsigned long i1,i2,i3,k,n,nnew,nprev=1,nt,ntot=1;
	int idim;
	float *wksp;

	for (idim=1;idim<=ndim;idim++) ntot *= nn[idim];
	wksp=vector(1,ntot);
	for (idim=1;idim<=ndim;idim++) {
		n=nn[idim];
		nnew=n*nprev;
		if (n > 4) {
			for (i2=0;i2<ntot;i2+=nnew) {
				for (i1=1;i1<=nprev;i1++) {
					for (i3=i1+i2,k=1;k<=n;k++,i3+=nprev) wksp[k]=a[i3];
					if (isign >= 0) {
						for(nt=n;nt>=4;nt >>= 1)
							(*wtstep)(wksp,nt,isign);
					} else {
						for(nt=4;nt<=n;nt <<= 1)
							(*wtstep)(wksp,nt,isign);
					}

					for (i3=i1+i2,k=1;k<=n;k++,i3+=nprev) a[i3]=wksp[k];
				}
			}
		}
		nprev=nnew;
	}
	free_vector(wksp,1,ntot);
}

#define C0 0.4829629131445341
#define C1 0.8365163037378079
#define C2 0.2241438680420134
#define C3 -0.1294095225512604

void daub4(float a[], unsigned long n, int isign)
{
	float *wksp;
	unsigned long nh,nh1,i,j;

	wksp=vector(1,n);
	if (n < 4) return;
	nh1=(nh=n >> 1)+1;
	if (isign >= 0) {
		for (i=1,j=1;j<=n-3;j+=2,i++) {
			wksp[i]=C0*a[j]+C1*a[j+1]+C2*a[j+2]+C3*a[j+3];
			wksp[i+nh] = C3*a[j]-C2*a[j+1]+C1*a[j+2]-C0*a[j+3];
		}
		wksp[i]=C0*a[n-1]+C1*a[n]+C2*a[1]+C3*a[2];
		wksp[i+nh] = C3*a[n-1]-C2*a[n]+C1*a[1]-C0*a[2];
	} else {
		wksp[1]=C2*a[nh]+C1*a[n]+C0*a[1]+C3*a[nh1];
		wksp[2] = C3*a[nh]-C0*a[n]+C1*a[1]-C2*a[nh1];
		for (i=1,j=3;i<nh;i++) {
			wksp[j++]=C2*a[i]+C1*a[i+nh]+C0*a[i+1]+C3*a[i+nh1];
			wksp[j++] = C3*a[i]-C0*a[i+nh]+C1*a[i+1]-C2*a[i+nh1];
		}
	}
	for (i=1;i<=n;i++) a[i]=wksp[i];
	free_vector(wksp,1,n);
}
#undef C0
#undef C1
#undef C2
#undef C3

#define FREE_ARG char*
#define NR_END 1

void free_vector(float *v, long nl, long nh)
/* free a float vector allocated with vector() */
{
	free((FREE_ARG) (v+nl-NR_END));
}

float *vector(long nl, long nh)
/* allocate a float vector with subscript range v[nl..nh] */
{
	float *v;

	v=(float *)malloc((size_t) ((nh-nl+1+NR_END)*sizeof(float)));
	if (!v) nrerror("allocation failure in vector()");
	return v-nl+NR_END;
}

#undef FREE_ARG
#undef NR_END

void nrerror(char error_text[])
/* Numerical Recipes standard error handler */
{
#ifdef __MWERKS__
#pragma unused(error_text)
#endif
}
