#include "tcl.h"
#include "tkint.h"

#ifdef macintosh
#include "tkMacInt.h"
#endif

#include "tkCanvasSave.h"
#include "tkCanvas.h"

#include "tkgrlib.h"

static int SavePixMap(Drawable pixmap, int width, int height, int xOffset, int yOffset, Tcl_Interp *interp);
static int tkCanvasSave(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int DoSaveCanvas(Tk_Window canvasPtr, Tcl_Interp *interp);
static int tkCanvasISave(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static void	StoreMacDataInMimage(mimage_ptr mimg, mpixel *pixelPtr, int width, int height, int xOffset, int yOffset, int bytesPerLine, short bits_per_pixel);

int registerCanvasISave(Tcl_Interp *interp)
{
	Tcl_CreateCommand(interp, "isave", tkCanvasISave, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
}

static int tkCanvasISave(ClientData clientData, Tcl_Interp *interp, 
	int argc, char *argv[])
{
	Tk_Window tkWin, canvas;
	char *canvasName = argv[1];
	char *newImageName;
	char *evalString;
	int result;
	
	if (argc != 2 && argc != 3) {
		interp->result = "wrong # args";
		return TCL_ERROR;
	}
	
	if (argc == 3)
		newImageName = argv[2];
	else
		newImageName = "";

	result = Tcl_VarEval(interp, "image create mimage ", newImageName, " -canvas ", canvasName, (char *) NULL);

	return result;
}

int registerCanvasSave(Tcl_Interp *interp)
{
	Tcl_CreateCommand(interp, "save", tkCanvasSave, (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
}

static int tkCanvasSave(ClientData clientData, Tcl_Interp *interp, 
	int argc, char *argv[])
{
	Tk_Window tkWin, canvas;
	
	if (argc != 2) {
		interp->result = "wrong # args";
		return TCL_ERROR;
	}
	
	tkWin  = Tk_MainWindow(interp);
	canvas = Tk_NameToWindow(interp, argv[1], tkWin);
	if (!canvas) {
		Tcl_AppendResult(interp, "Canvas ", argv[1], " not found", (char *) NULL);
		return TCL_ERROR;
	}

	return DoSaveCanvas(canvas, interp);
}

static int DoSaveCanvas(Tk_Window canvas, Tcl_Interp *interp) 
{
	int tcl_err = TCL_OK;
	int width, height;
	
	Tk_Window tkwin;
	TkCanvas *canvasPtr;
	Tk_Item *itemPtr;
	Pixmap pixmap;
	
	int saveDrawableXOrigin, saveDrawableYOrigin;	/*  GDM 051499 */

	canvasPtr = ((TkWindow *)canvas)->instanceData;	
	tkwin = canvasPtr->tkwin;	/* get the window of the canvas */
	
	saveDrawableXOrigin = canvasPtr->drawableXOrigin;	/*  GDM 051499 */
	saveDrawableYOrigin	= canvasPtr->drawableYOrigin;	/*  GDM 051499 */
	canvasPtr->drawableXOrigin = 0;	/*  GDM 051499 */
	canvasPtr->drawableYOrigin = 0;	/*  GDM 051499 */

		/*
		 *	get the right and bottom borders of the canvas 
		 */
	width = canvasPtr->xOrigin + Tk_Width(tkwin) - canvasPtr->inset;
	height = canvasPtr->yOrigin + Tk_Height(tkwin) - canvasPtr->inset;


		/*
		 *	allocate a pixmap of the proper size 
		 */
	pixmap = Tk_GetPixmap(Tk_Display(tkwin), Tk_WindowId(tkwin), 
		(width + 30),
		(height + 30),
		Tk_Depth(tkwin));
 
		/*
		 *	blank the pixmap 
		 */
	XFillRectangle(Tk_Display(tkwin), pixmap, canvasPtr->pixmapGC,
		0, 0, (unsigned int) width, (unsigned int) height);

		/*
		 *	draw the contents of the canvas into the 
		 *	pixmap
		 */
	for (itemPtr = canvasPtr->firstItemPtr; itemPtr != NULL;
		itemPtr = itemPtr->nextPtr) {

		(*itemPtr->typePtr->displayProc)((Tk_Canvas) canvasPtr, itemPtr, 
			canvasPtr->display, pixmap, 
			0, 0, width, height);
	}

		/*
		 *	save the pixmap in the mimage format
		 */
	tcl_err = SavePixMap(pixmap, 
		width, 
		height,
		0, 
		0, 
		interp);

/*
{
int i=0;
mimage_ptr mimg;
static	char result[256];
	sprintf(result, "%ld", 	mimg = create_mimage(width, height, 3, DISCRETE_IMAGE, DENSE_IMAGE));
	for (i=0;i<3;i++)
		memset(mimg->planes_array[i]->buffer, 0, mimg->planes_array[i]->height * mimg->planes_array[i]->width);
	interp->result = result;
}
*/

	canvasPtr->drawableXOrigin = saveDrawableXOrigin;	/*  GDM 051499 */
	canvasPtr->drawableYOrigin = saveDrawableYOrigin;	/*  GDM 051499 */
		/*
		 *	free the pixmap
		 */
	Tk_FreePixmap(Tk_Display(tkwin), pixmap);

	return tcl_err;
}

#ifdef macintosh

static int SavePixMap(Drawable pixmap, int width, int height, int xOffset, int yOffset, Tcl_Interp *interp)
{
	static char	result[12];
	mimage_ptr	mimg;

	char	*pixelPtr;
	short	lines;
	short	bytes_per_line;
	short	bytes_per_pixel;
	short	bits_per_pixel;
	short	bits_per_channel;

	PixMap 			*aPixMap;
	MacDrawable		*macWin = (MacDrawable *) pixmap;
	GWorldPtr		srcPort = TkMacGetDrawablePort(pixmap);
	BitMapPtr		srcBit = &((GrafPtr) srcPort)->portBits;
	PixMapHandle	aPixMapHandle = *(PixMapHandle *)srcBit;


	xOffset -= macWin->xOff;
	yOffset -= macWin->yOff;

	HLock(aPixMapHandle);

	aPixMap = *aPixMapHandle;

	pixelPtr = aPixMap->baseAddr;
	bytes_per_line = aPixMap->rowBytes & 0x3fff;
	lines = aPixMap->bounds.bottom;

	bytes_per_pixel = bytes_per_line / aPixMap->bounds.right;

	bits_per_pixel = aPixMap->pixelSize;
	bits_per_channel = aPixMap->cmpSize;

	mimg = create_mimage(width, height, 3, DISCRETE_IMAGE, DENSE_IMAGE);
	if (mimg)
		StoreMacDataInMimage(mimg, pixelPtr, width, height, xOffset, yOffset, bytes_per_line, bits_per_pixel);
	
	HUnlock(aPixMapHandle);

	sprintf(result, "%ld", mimg);
	interp->result = result;

	return TCL_OK;
}


#else /* ! macintosh */

static int SavePixMap(Pixmap pixmap, int width, int height, int xOffset, int yOffset, Tcl_Interp *interp)
{
	interp->result = "Save canvas enabled only under MacOS";
	return TCL_ERROR;
}

#endif /* macintosh */

static void	StoreMacDataInMimage(mimage_ptr mimg, mpixel *pixelPtr, int width, int height, int xOffset, int yOffset, int bytesPerLine, short bits_per_pixel)
{
	int i, j, c, pixels_number;
	mpixel_ptr buffer_ptr[3];
	byteplane_ptr *planes = mimg->planes_array;
	high_color color;
	int xMore;
	
	buffer_ptr[0] = get_plane_buffer(planes[0], DISCRETE_IMAGE);
	buffer_ptr[1] = get_plane_buffer(planes[1], DISCRETE_IMAGE);
	buffer_ptr[2] = get_plane_buffer(planes[2], DISCRETE_IMAGE);
	
	switch (bits_per_pixel) {
		
		case 32:	/* tested ok */
			
			xMore = bytesPerLine - (xOffset + width) * 4;
			pixelPtr = pixelPtr + bytesPerLine * yOffset;			

			for (j=0; j<height; j++) {
				pixelPtr += xOffset * 4;
				for (i=0; i<width; i++) {
					pixelPtr ++;	/* advance over the alpha channel */
					for (c=0; c<3; c++) 
						*buffer_ptr[c]++ = *pixelPtr++;
				}
				pixelPtr += xMore;	/* advance to the next line */
			}
			break;
		
		case 24:	/* not testable */

			xMore = bytesPerLine - (xOffset + width) * 3;
			pixelPtr = pixelPtr + bytesPerLine * yOffset;			

			for (j=0; j<height; j++) {
				pixelPtr += xOffset * 3;
				for (i=0; i<width; i++) {
					for (c=0; c<3; c++) 
						*buffer_ptr[c]++ = *pixelPtr++;
				}
				pixelPtr += xMore;	/* advance to the next line */
			}
			break;
		
		case 16:	/* tested ok */

			xMore = bytesPerLine - (xOffset + width) * 2;
			pixelPtr = pixelPtr + bytesPerLine * yOffset;			

			for (j=0; j<height; j++) {
				pixelPtr += xOffset * 2;
				for (i=0; i<width; i++) {

					color.color = * (short *) pixelPtr;			/* load the color in the union */

					*buffer_ptr[0]++ = color.components.red << 3;
					*buffer_ptr[1]++ = color.components.green << 3;
					*buffer_ptr[2]++ = color.components.blue << 3;
					pixelPtr += 2;

				}
				pixelPtr += xMore;	/* advance to the next line */
			}
			break;

		case 8:	/* not implemented yet */
		
			break;
			
		default:
			break;
	}
}
 
