/*
 *	Voice Native Package
 */

	/* SETL2 system header files */
#include "macros.h"

#include "voice.h"
#include "stubs_utilities.h"
#include "str_queue.h"

	/*
	 *	Structures
	 */

typedef struct setl_voice {             /* Native Object Structure           */
	int32 use_count;                    /* Reference Count                   */
	int32 type;                         /* Encodes Type and Subtype          */

	specifier *callback_spec;

#ifdef macintosh
	SRRecognitionSystem	srSystem;
	SRRecognizer		srRecognizer;
#endif

} setl_voice, *setl_voice_ptr;

	/*
	 *	Variables
	 */

static int32 voice_type;               /* Store the type assigned to us by  */
static char last_command[256];
static StrQueuePtr gQueue;				/* Voice commands queue */

#define QUEUE_SIZE 15
	
	/*
	 *	Prototypes
	 */

void callback(SETL_SYSTEM_PROTO void *user_ptr2, char *str);
int32 VOICEPAK__INIT(SETL_SYSTEM_PROTO_VOID);
int32 VOICEPAK_CREATE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 VOICEPAK_GET_EVENT_SOURCE_FUNCTION(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 voice_event_source_function(SETL_SYSTEM_PROTO specifier *callback);
int32 VOICEPAK_START_RECOGNIZE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 VOICEPAK_STOP_RECOGNIZE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 VOICEPAK_IDLE(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);
int32 VOICEPAK_NEXT_COMMAND(SETL_SYSTEM_PROTO int argc, specifier *argv, specifier *target);

	/*
	 *	Exported functions
	 */

void callback(SETL_SYSTEM_PROTO void *user_ptr2, char *str)
{
	char type_ret;

	strncpy(last_command, str, 256);
	str_enqueue(gQueue, str, strlen(str) + 1);

	if (user_ptr2) 
		callback_caller(SETL_SYSTEM &type_ret, user_ptr2, "S", str);
	
}

	/*
	 *	Package functions
	 */

static void internal_destructor(setl_voice_ptr voice_instance)
{
	if (voice_instance != NULL) {
#ifdef macintosh
		if (gQueue)	destroy_str_queue(&gQueue);
		CleanupSpeechRecognitionStuff (voice_instance->srRecognizer, voice_instance->srSystem);
#endif
		if (voice_instance->callback_spec) free(voice_instance->callback_spec);
		free(voice_instance);
	}
}

int32 VOICEPAK__INIT(
   SETL_SYSTEM_PROTO_VOID)
{
#ifdef macintosh
	if (!HasSpeechRecognitionMgr())
		return 1;	/* voice manager not found */
#endif
	voice_type=register_type(SETL_SYSTEM "voice recognition",internal_destructor);
	if (voice_type==0) return 1;
	return 0;

}

int32 VOICEPAK_CREATE(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	int len;
	string_h_ptr_type string_header;
	char *vocabulary = NULL;
	setl_voice_ptr voice_instance = NULL;
	specifier *callback_spec = NULL;
		
	check_arg(SETL_SYSTEM argv, 0, ft_string, "vocabulary tuple", "start recognize");

	if (argv[1].sp_form == ft_omega) 
		callback_spec = NULL;
	else {
		check_arg(SETL_SYSTEM argv, 1, ft_proc, "callback", "start recognize");
		callback_spec = malloc(sizeof(specifier));
		if (!callback_spec)
			goto end_voice_create;
		memcpy(callback_spec, &argv[1], sizeof(specifier));
	}
	
	voice_instance = calloc(sizeof(setl_voice), 1);
	if (!voice_instance)
		goto end_voice_create;
	
	voice_instance->type = voice_type;
	voice_instance->use_count = 1;
	voice_instance->callback_spec = callback_spec;
#ifdef macintosh
	voice_instance->srSystem = NULL;
	voice_instance->srRecognizer= NULL;
#endif

	vocabulary = malloc(len = STRING_LEN(argv[0]));
	if (!vocabulary) {
		free (voice_instance);
		voice_instance = NULL;
		goto end_voice_create;
	}
	
	gQueue = create_str_queue(QUEUE_SIZE);
	if (!gQueue)
		goto end_voice_create;
		
	string_header = argv[0].sp_val.sp_string_ptr;
	setl2_string_to_cstring(string_header, vocabulary, len);

#ifdef macintosh
	InitVoiceSR(&voice_instance->srRecognizer, &voice_instance->srSystem, vocabulary, SETL_SYSTEM callback_spec /* callback */, (void (*)(void *, void *, char *))callback);
#endif

end_voice_create:

	if (vocabulary)	free(vocabulary);

	if (!voice_instance) {
		unmark_specifier(target);
		target->sp_form = ft_omega;

		if (gQueue)	destroy_str_queue(&gQueue);
		if (callback_spec) free(callback_spec);
		return;
	}

	unmark_specifier(target);
	target->sp_form = ft_opaque;
	target->sp_val.sp_opaque_ptr = (opaque_item_ptr_type)voice_instance;
}

int32 VOICEPAK_GET_EVENT_SOURCE_FUNCTION(  
  SETL_SYSTEM_PROTO
  int argc,                           /* number of arguments passed        */
  specifier *argv,                    /* argument vector (two here)        */
  specifier *target)                  /* return value                      */
{
	unsigned long event_source_function_value;

	event_source_function_value = (unsigned long)voice_event_source_function;

	unmark_specifier(target);
	
	target->sp_form = ft_short;
	target->sp_val.sp_short_value = event_source_function_value;
}

int32 voice_event_source_function(SETL_SYSTEM_PROTO specifier *callback)
{
	int err = 0;
	
	if (callback) {
		char *str = str_dequeue(gQueue, NULL);

		if (str) {
			char type_ret;
			result_type result;

			result = callback_caller(SETL_SYSTEM &type_ret, callback, "S", str);

			free(str);
		}
	}

	return err;
}

int32 VOICEPAK_START_RECOGNIZE(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	setl_voice_ptr voice_instance;

	check_type_arg(SETL_SYSTEM argv, 0, "start_recognize", voice_type);

	voice_instance = (setl_voice_ptr)(argv[0].sp_val.sp_opaque_ptr);
#ifdef macintosh
	if (voice_instance->srRecognizer == NULL)
		abend(SETL_SYSTEM "Cannot start voice before initialization");
		
	StartSpeechRecognition (voice_instance->srRecognizer);
#endif
}

int32 VOICEPAK_STOP_RECOGNIZE(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	setl_voice_ptr voice_instance;

	check_type_arg(SETL_SYSTEM argv, 0, "stop_recognize", voice_type);

	voice_instance = (setl_voice_ptr)(argv[0].sp_val.sp_opaque_ptr);
#ifdef macintosh		
	if (voice_instance->srRecognizer == NULL)
		abend(SETL_SYSTEM "Cannot stop voice before initialization");

	StopSpeechRecognition (voice_instance->srRecognizer);
#endif

	empty_str_queue(gQueue);
}

int32 VOICEPAK_IDLE(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
#ifdef macintosh
	Boolean 		gotEvent;
	EventRecord		event;

	gotEvent = WaitNextEvent(everyEvent, &event, 0xFFFFFFFF, NULL);
	if (gotEvent) {
		switch (event.what) {
			case kHighLevelEvent :
				AEProcessAppleEvent(&event);
				break;
			default :
				break;
		}
	}
		
#endif
}

int32 VOICEPAK_NEXT_COMMAND(
	SETL_SYSTEM_PROTO
	int argc,                           /* number of arguments passed        */
	specifier *argv,                    /* argument vector (two here)        */
	specifier *target)                  /* return value                      */
{
	char *str;
	unmark_specifier(target);
	
	str = str_dequeue(gQueue, NULL);
	if (str) {
		target->sp_form = ft_string;
		target->sp_val.sp_string_ptr = setl2_string(SETL_SYSTEM str, strlen(str));
		free(str);
	} else {
		target->sp_form = ft_omega;
	}
}