/*\
 *  %
 *  %  Copyright (c) W. Kirk Snyder
 *  %  All Rights Reserved
 *  %
 *  %  This is unpublished source code for SETL2.  There should be no
 *  %  copies which are not in the possession of the author.  If you
 *  %  somehow come across a copy, please return or destroy it.
 *  %
 *
 *  \package{Parser}
 *
 *  This is an LALR(1) parser.  It uses tables generated by LALR -- my
 *  own parse table generator.  The error recovery uses Poonen's method.
 *  For now, I'm not doing much work on this area.  I'm waiting to read
 *  Phillippe's thesis, and will improve it then.
 *
 *  \texify{parse.h}
 *
 *  \packagebody{Parser}
\*/


/* SETL2 system header files */

#include "system.h"                    /* SETL2 system constants            */
#include "compiler.h"                  /* SETL2 compiler constants          */
#include "giveup.h"                    /* severe error handler              */
#include "messages.h"                  /* error messages                    */
#include "namtab.h"                    /* name table                        */
#include "lex.h"                       /* lexical analyzer                  */
#include "parse.h"                     /* parser                            */
#include "parsetab.h"                  /* parse tables                      */
#ifdef DEBUG
#include "ntdesc.h"                    /* non-terminal descriptions         */
#include "ruledesc.h"                  /* rule descriptions                 */
#endif
#include "semact.h"                    /* semantic actions                  */
#include "listing.h"                   /* source and error listings         */

#ifdef PLUGIN
#define fprintf plugin_fprintf
#endif

/* performance tuning constants */

#define TKS_BLOCK_SIZE     25          /* state-token stack block size      */

/* state-token structure */

typedef struct {
   token_type ts_token;                /* token                             */
   int ts_state;                       /* state                             */
} state_token;

static state_token *tks_stack = NULL;  /* state-token stack                 */
static int tks_top = -1;               /*    stack top                      */
static int tks_max = 0;                /*    size of stack                  */

/* macro to return a new state_token item */

#define get_tks ((tks_top < tks_max - 1) \
     ? ++tks_top : alloc_tks(SETL_SYSTEM_VOID))

/* package - global data */

static int no_errors;                  /* YES if no syntax errors found     */

/* forward declarations */

static int alloc_tks(SETL_SYSTEM_PROTO_VOID);         
                                       /* allocate a state-token block      */
static void recover(SETL_SYSTEM_PROTO token_type *);   
                                       /* error recovery                    */

/*\
 *  \function{parsefile()}
 *
 *  This function is the parser.  There's nothing special about it -- an
 *  explanation of the algorithm can be found in any compiler book.
\*/

void parsefile(SETL_SYSTEM_PROTO_VOID)

{
token_type token;                      /* current token                     */
int action;                            /* action from parse tables          */
int state;                             /* current state                     */
int *act_ptr;                          /* used to look up action in tables  */
int t;                                 /* temporary pointer to state_token  */
static char is_important[] = {
/* ## begin is_important */
   0,                                  /* end of file                       */
   0,                                  /* error token                       */
   1,                                  /* identifier                        */
   1,                                  /* literal                           */
   1,                                  /* keyword => AND                    */
   1,                                  /* keyword => ASSERT                 */
   0,                                  /* keyword => BODY                   */
   1,                                  /* keyword => CASE                   */
   0,                                  /* keyword => CLASS                  */
   0,                                  /* keyword => CONST                  */
   1,                                  /* keyword => CONTINUE               */
   0,                                  /* keyword => ELSE                   */
   1,                                  /* keyword => ELSEIF                 */
   0,                                  /* keyword => END                    */
   1,                                  /* keyword => EXIT                   */
   1,                                  /* keyword => FOR                    */
   1,                                  /* keyword => IF                     */
   0,                                  /* keyword => INHERIT                */
   1,                                  /* keyword => LAMBDA                 */
   1,                                  /* keyword => LOOP                   */
   1,                                  /* keyword => NOT                    */
   1,                                  /* keyword => NULL                   */
   1,                                  /* keyword => OR                     */
   0,                                  /* keyword => OTHERWISE              */
   0,                                  /* keyword => PACKAGE                */
   1,                                  /* keyword => PROCEDURE              */
   0,                                  /* keyword => PROCESS                */
   0,                                  /* keyword => PROGRAM                */
   0,                                  /* keyword => RD                     */
   1,                                  /* keyword => RETURN                 */
   0,                                  /* keyword => RW                     */
   0,                                  /* keyword => SEL                    */
   1,                                  /* keyword => SELF                   */
   1,                                  /* keyword => STOP                   */
   0,                                  /* keyword => THEN                   */
   1,                                  /* keyword => UNTIL                  */
   0,                                  /* keyword => USE                    */
   0,                                  /* keyword => VAR                    */
   1,                                  /* keyword => WHEN                   */
   1,                                  /* keyword => WHILE                  */
   0,                                  /* keyword => WR                     */
   0,                                  /* ;                                 */
   0,                                  /* ,                                 */
   0,                                  /* :                                 */
   0,                                  /* (                                 */
   0,                                  /* )                                 */
   0,                                  /* [                                 */
   0,                                  /* ]                                 */
   0,                                  /* {                                 */
   0,                                  /* }                                 */
   0,                                  /* .                                 */
   0,                                  /* ..                                */
   1,                                  /* :=                                */
   0,                                  /* |                                 */
   0,                                  /* =>                                */
   1,                                  /* assignment operator               */
   1,                                  /* assignment operator               */
   1,                                  /* unary operator                    */
   1,                                  /* pointer reference                 */
   1,                                  /* addop                             */
   1,                                  /* -                                 */
   1,                                  /* mulop                             */
   1,                                  /* **                                */
   1,                                  /* relop                             */
   1,                                  /* fromop                            */
   1,                                  /* quantifier                        */
/* ## end is_important */
   0};


#ifdef DEBUG

   if (LEX_DEBUG || PRS_DEBUG) {

      fprintf(DEBUG_FILE,"\nPARSE PHASE\n===========\n\n");

   }

#endif

   /* initialize the lexical analyzer */

   no_errors = YES;
   tks_top = -1;
   init_lex(SETL_SYSTEM_VOID);
   init_semacts();

   /* start out in state 0 with an EOF token */

   t = get_tks;
   tks_stack[t].ts_state = state = 0;
   tks_stack[t].ts_token.tk_token_class = tok_eof;

   /* do forever */

   get_token(SETL_SYSTEM &token);
   for (;;) {

      /* look up the action */

      for (act_ptr = state_table[state];
           *act_ptr != -1 && *act_ptr != token.tk_token_class;
           act_ptr += 2);
      action = (*act_ptr == -1) ? -1 : *(act_ptr + 1);

      /* error recovery */

      if (action < 0) {

         recover(SETL_SYSTEM &token);
         no_errors = NO;

         /* if the error handler deleted everything, accept */

         if (tks_top == 0 && token.tk_token_class == tok_eof) {

#ifdef DEBUG

            if (PRS_DEBUG) {

               fprintf(DEBUG_FILE,"PRS : error handler deleted program\n");

            }

#endif

            break;

         }

         state = tks_stack[tks_top].ts_state;

      }

      /* shift */

      else if (action >= num_rules) {

#ifdef DEBUG

         if (PRS_DEBUG) {

            fprintf(DEBUG_FILE,"PRS : shifting ");

            switch (token.tk_token_class) {

               case tok_id :

                  fprintf(DEBUG_FILE,"Identifier => %s\n",
                                     (token.tk_namtab_ptr)->nt_name);
                  break;

               case tok_eof :

                  fprintf(DEBUG_FILE,"End of file\n");
                  break;

               default :

                  fprintf(DEBUG_FILE,"%s\n",(token.tk_namtab_ptr)->nt_name);
                  break;

            }
         }

#endif

         /* we can shift an error, but it should kill the semantic actions */

         if (token.tk_token_class == tok_error)
            no_errors = NO;

         t = get_tks;
         memcpy((void *)&tks_stack[t].ts_token,
                (void *)&token,
                sizeof(token_type));
         tks_stack[t].ts_state = state = action - num_rules;
         if (is_important[token.tk_token_class] && no_errors)
            push_token(SETL_SYSTEM &token);
         get_token(SETL_SYSTEM &token);

      }

      /* accept */

      else if (action == 0) {

#ifdef DEBUG

         if (PRS_DEBUG) {

            fprintf(DEBUG_FILE,"PRS : program accepted\n");

         }

#endif

         break;

      }

      /* reduce */

      else {

#ifdef DEBUG

         if (PRS_DEBUG) {

            fprintf(DEBUG_FILE,"PRS : reducing => %s\n",rule_desc[action]);

         }

#endif

         /* call semantic action */

         if (no_errors)
            semantic_action(SETL_SYSTEM action);

         /* pop the rhs of the rule off the state-token stack */

         tks_top -= rule_rhs[action];
         if (tks_top < -1)
            giveup(SETL_SYSTEM msg_token_stack_overflow);

         /* lookup the next state */

         state = tks_stack[tks_top].ts_state;
         for (act_ptr = state_table[state];
              *act_ptr != -1 && *act_ptr != rule_lhs[action];
              act_ptr += 2);
         if (*act_ptr == -1)
            giveup(SETL_SYSTEM msg_expected_goto);
         state = *(act_ptr + 1) - num_rules;

         /* shift the new token & state */

         t = get_tks;
         memcpy((void *)&tks_stack[t].ts_token,
                (void *)&token,
                sizeof(token_type));
         tks_stack[t].ts_state = state;
         tks_stack[t].ts_token.tk_token_class = rule_lhs[action];

      }
   }

   close_lex(SETL_SYSTEM_VOID);
   close_semacts(SETL_SYSTEM_VOID);

   /* if we didn't process semantic actions update the file error count */

   if (!no_errors) {

      FILE_ERROR_COUNT += UNIT_ERROR_COUNT;
      FILE_WARNING_COUNT += FILE_WARNING_COUNT;

   }

}

/*\
 *  \function{recover()}
 *
 *  This function performs syntactic error recovery.  It uses Poonen's
 *  method.
\*/

static void recover(
   SETL_SYSTEM_PROTO
   token_type *token)                  /* token found to be in error        */

{
int class;                             /* token class                       */
int action;                            /* action from parse tables          */
int state;                             /* current state                     */
int *act_ptr;                          /* used to look up action in tables  */
int t;                                 /* temporary pointer to state_token  */
state_token *save_tks_stack;           /* temporary state-token stack       */
int save_tks_top;                      /*   stack pointer for above         */
int *beacon;                           /* beacon set: terminals which can   */
                                       /*   follow an error mark            */
int i;                                 /* temporary looping variables       */

#ifdef DEBUG

   if (PRS_DEBUG) {

      fprintf(DEBUG_FILE,"PRS : recovering from syntax error\n");

   }

#endif

   /* print an error message, if the lexical analyzer didn't */

   if (token->tk_token_class != tok_error) {

      error_message(SETL_SYSTEM
                    &(token->tk_file_pos),
                    "syntax error at %s",
                    (token->tk_namtab_ptr)->nt_name);

   }

   /* otherwise delete the error token */

   while (token->tk_token_class == tok_error) {

      get_token(SETL_SYSTEM token);

   }

   /* allocate space for a beacon set */

   beacon = (int *)malloc((size_t)(num_terms * sizeof(int)));
   for (i = 0; i < num_terms; i++)
      beacon[i] = -1;

   /* make a copy of the state-token stack */

   save_tks_stack = (state_token *)malloc(
                         (size_t)((tks_top + 1) * sizeof(state_token)));
   if (save_tks_stack == NULL)
      giveup(SETL_SYSTEM msg_malloc_error);

   memcpy((void *)save_tks_stack,
          (void *)tks_stack,
          (size_t)(sizeof(state_token) * (tks_top + 1)));

   save_tks_top = tks_top;

   /* check each class for membership in the beacon set */

   for (class = 0; class < num_terms; class++) {

      if (class == tok_error)
         continue;

      /* check each state on the stack */

      for (i = tks_top; i >= 0; i--) {

         tks_top = i;

         state = tks_stack[i].ts_state;
         for (act_ptr = state_table[state];
              *act_ptr != -1 && *act_ptr != tok_error;
              act_ptr += 2);
         action = (*act_ptr == -1) ? -1 : *(act_ptr + 1);

         /* perform any reduce actions */

         while (action > 0 && action < num_rules) {

            /* pop the rhs of the rule off the state-token stack */

            tks_top -= rule_rhs[action];

            /* lookup the next state */

            state = tks_stack[tks_top].ts_state;
            for (act_ptr = state_table[state];
                 *act_ptr != -1 && *act_ptr != rule_lhs[action];
                 act_ptr += 2);
            state = (*act_ptr == -1) ? -1 : *(act_ptr + 1) - num_rules;
            if (state < 0)
               giveup(SETL_SYSTEM msg_expected_goto);

            /* shift the new token & state */

            t = get_tks;
            tks_stack[t].ts_state = state;
            tks_stack[t].ts_token.tk_token_class = rule_lhs[action];

            for (act_ptr = state_table[state];
                 *act_ptr != -1 && *act_ptr != tok_error;
                 act_ptr += 2);
            action = (*act_ptr == -1) ? -1 : *(act_ptr + 1);

         }

         /* cycle if we can't shift the error token */

         if (action <= 0) {

            tks_top = save_tks_top;

            memcpy((void *)tks_stack,
                   (void *)save_tks_stack,
                   (size_t)(sizeof(state_token) * (tks_top + 1)));

            continue;

         }

         /* at this point, we should have a shift */

         if (action < num_rules)
            giveup(SETL_SYSTEM msg_expected_shift);

         t = get_tks;
         tks_stack[t].ts_state = state = action - num_rules;
         tks_stack[t].ts_token.tk_token_class = tok_error;

         /* look for an action on the class we are checking */

         for (act_ptr = state_table[state];
              *act_ptr != -1 && *act_ptr != class;
              act_ptr += 2);
         action = (*act_ptr == -1) ? -1 : *(act_ptr + 1);

         /* perform any reduce actions */

         while (action > 0 && action < num_rules) {

            /* pop the rhs of the rule off the state-token stack */

            tks_top -= rule_rhs[action];

            /* lookup the next state */

            state = tks_stack[tks_top].ts_state;
            for (act_ptr = state_table[state];
                 *act_ptr != -1 && *act_ptr != rule_lhs[action];
                 act_ptr += 2);
            state = (*act_ptr == -1) ? -1 : *(act_ptr + 1) - num_rules;
            if (state < 0)
               giveup(SETL_SYSTEM msg_expected_goto);

            /* shift the new token & state */

            t = get_tks;
            tks_stack[t].ts_state = state;
            tks_stack[t].ts_token.tk_token_class = rule_lhs[action];

            for (act_ptr = state_table[state];
                 *act_ptr != -1 && *act_ptr != class;
                 act_ptr += 2);
            action = (*act_ptr == -1) ? -1 : *(act_ptr + 1);

         }

         /* cycle if we can't shift the class we are checking */

         if (action < 0) {

            tks_top = save_tks_top;

            memcpy((void *)tks_stack,
                   (void *)save_tks_stack,
                   (size_t)(sizeof(state_token) * (tks_top + 1)));

            continue;

         }

         /* finally, we found something which should be in the beacon set */

         beacon[class] = i;

         tks_top = save_tks_top;

         memcpy((void *)tks_stack,
                (void *)save_tks_stack,
                (size_t)(sizeof(state_token) * (tks_top + 1)));

         break;

      }
   }

   /* we no longer need our saved stack */

   free(save_tks_stack);

   /* end of file should always be in the beacon set */

   if (beacon[tok_eof] == -1) {

      beacon[tok_eof] = -2;

   }

   /* delete input tokens until we find something in the beacon set */

   while (beacon[token->tk_token_class] == -1) {

#ifdef DEBUG

      if (PRS_DEBUG) {

         fprintf(DEBUG_FILE,"PRS : deleting ");

         switch (token->tk_token_class) {

            case tok_id :

               fprintf(DEBUG_FILE,"Identifier => %s\n",
                                  (token->tk_namtab_ptr)->nt_name);
               break;

            case tok_eof :

               fprintf(DEBUG_FILE,"End of file\n");
               break;

            default :

               fprintf(DEBUG_FILE,"%s\n",(token->tk_namtab_ptr)->nt_name);
               break;

         }
      }

#endif

      get_token(SETL_SYSTEM token);

   }

   /*
    *  pop items from state-token stack to highest state with an action
    *  on the token
    */

   if (beacon[token->tk_token_class] == -2) {

      tks_top = 0;
      return;

   }

   tks_top = beacon[token->tk_token_class];

   /* perform any reduce actions on the error mark */

   state = tks_stack[tks_top].ts_state;

   for (;;) {

      for (act_ptr = state_table[state];
           *act_ptr != -1 && *act_ptr != tok_error;
           act_ptr += 2);
      action = (*act_ptr == -1) ? -1 : *(act_ptr + 1);

      if (action < 0)
         giveup(SETL_SYSTEM msg_bad_action);

      if (action >= num_rules)
         break;

      /* pop the rhs of the rule off the state-token stack */

      tks_top -= rule_rhs[action];

      /* lookup the next state */

      state = tks_stack[tks_top].ts_state;
      for (act_ptr = state_table[state];
           *act_ptr != -1 && *act_ptr != rule_lhs[action];
           act_ptr += 2);
      state = (*act_ptr == -1) ? -1 : *(act_ptr + 1) - num_rules;
      if (state < 0)
         giveup(SETL_SYSTEM msg_expected_goto);

      /* shift the new token & state */

      t = get_tks;
      tks_stack[t].ts_state = state;
      tks_stack[t].ts_token.tk_token_class = rule_lhs[action];

   }

   /* shift the error mark onto the stack */

   state = action - num_rules;
   t = get_tks;
   tks_stack[t].ts_state = state;
   tks_stack[t].ts_token.tk_token_class = tok_error;

   /* free the space for the beacon set */

   free(beacon);

   /* that's all, parsing can continue */

#ifdef DEBUG

   if (PRS_DEBUG) {

      fprintf(DEBUG_FILE,"PRS : error recovery successful\n");

   }

#endif

   return;

}

/*\
 *  \function{alloc\_tks()}
 *
 *  This function allocates a block in the state-token stack.  This stack
 *  is organized as an 'expanding array'.  That is, we allocate an array
 *  of a given size, then when that is exceeded, we allocate a larger
 *  array and copy the smaller to the larger.  This makes allocations
 *  slower than for the other tables, but makes it easy to reach down the
 *  stack, which we need to do in this table.  We don't expect the speed
 *  penalty to be a problem, since the stack should never get very big.
\*/

static int alloc_tks(SETL_SYSTEM_PROTO_VOID)

{
state_token *save_tks_stack;           /* temporary state-token stack       */

   /* the first time this is called, allocate an initial block */

   if (tks_max == 0) {

      tks_stack = (state_token *)malloc(
                           (size_t)(TKS_BLOCK_SIZE * sizeof(state_token)));
      if (tks_stack == NULL)
         giveup(SETL_SYSTEM msg_malloc_error);

      tks_max = TKS_BLOCK_SIZE;
      tks_top = 0;

      return 0;

   }

   /* expand the table */

   save_tks_stack = tks_stack;
   tks_stack = (state_token *)malloc(
                (size_t)((tks_max + TKS_BLOCK_SIZE) * sizeof(state_token)));
   if (tks_stack == NULL)
      giveup(SETL_SYSTEM msg_malloc_error);

   /* copy the old table to the new table, and free the old table */

   memcpy((void *)tks_stack,
          (void *)save_tks_stack,
          (size_t)(tks_max * sizeof(state_token)));
   free(save_tks_stack);

   tks_max += TKS_BLOCK_SIZE;

   return ++tks_top;

}
