; AUTOSAVE.EM, EMACSSRC>EMACS*>EXTENSIONS>SOURCES, EMACS DEVELOPMENT, 10/02/89
; Save a buffer to a file periodically and automatically.
; Copyright (c) 1988, Prime Computer, Inc., Natick, MA 01760
;                     All Rights Reserved
;
; Description:
;
; Abnormal conditions:
;
; Implementation:
;
; Modifications:
;   Date    Engineer     Description of modification
; 10/02/89  DMM/Bugos    Added autosave_in_progress$ to prevent an "autosave"
;                        event from occuring during a normal file save (in
;                        order to avoid the possibility of recursion).
; 11/13/88  Bugos, BMK   Initial coding.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;;  COMMANDS / FUNCTIONS:                                                  ;;;
;;;                                                                         ;;;
;;;  autosave                                                               ;;;
;;;      Toggle on and off autosave mode.                                   ;;;
;;;  autosave_on                                                            ;;;
;;;      Turn on autosave mode.                                             ;;;
;;;  autosave_off                                                           ;;;
;;;      Turn off autosave mode.                                            ;;;
;;;  autosave?                                                              ;;;
;;;      Report whether autosave mode is on or off.                         ;;;
;;;  autosave_file$                                                         ;;;
;;;      Store the new save_file until it is used by autosave.              ;;;
;;;  autosave_hook$                                                         ;;;
;;;      Function placed into redisplay hook that checks for autosaving.    ;;;
;;;  autosave_initialization$                                               ;;;
;;;      Function that initializes autosave mode variables.                 ;;;
;;;  autosave_temporary_trim$                                               ;;;
;;;      Function that trims the number of temporary files.                 ;;;
;;;  do_autosave$                                                           ;;;
;;;      Intermediate function that determines whether autosaving should    ;;;
;;;      be done.                                                           ;;;
;;;  orig_redisplay_hook$                                                   ;;;
;;;      Copy of the original redisplay_hook$ function.                     ;;;
;;;  orig_save_file                                                         ;;;
;;;      Copy of the original save_file command.                            ;;;
;;;  save_all_autosave_buffers                                              ;;;
;;;      Save all autosave buffers to files and delete all temporary files. ;;;
;;;  set_autosave_temporaries                                               ;;;
;;;      Set the number of autosave temporary files.                        ;;;
;;;  set_autosave_interval                                                  ;;;
;;;      Set (in number of screen updates) the autosave interval.           ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                         ;;;
;;;  VARIABLES:                                                             ;;;
;;;                                                                         ;;;
;;;  autosave_buffer_list$$                                                 ;;;
;;;      A list of all buffers in which autosave mode is on.                ;;;
;;;  autosave_calling$$                                                     ;;;
;;;      A boolean variable that indicates if autosave mode is calling      ;;;
;;;      save_file.                                                         ;;;
;;;  autosave_count$$                                                       ;;;
;;;      The number of screen updates since the last autosave.              ;;;
;;;  autosave_initialized$$                                                 ;;;
;;;      Indicates if one-time autosave initialization has been performed.  ;;;
;;;  autosave_interval$                                                     ;;;
;;;      Variable that contains the number of characters for the            ;;;
;;;      autosave interval.                                                 ;;;
;;;  autosave_mode$                                                         ;;;
;;;      (find_mode 'autosave)                                              ;;;
;;;  autosave_temporary_count$                                              ;;;
;;;      User variable indicating the number of temporary files to be       ;;;
;;;      retained.                                                          ;;;
;;;  (buffer_info (user autosave_file_name$$))                              ;;;
;;;      The name of the file that is used by autosave.                     ;;;
;;;  (buffer_info (user autosave_temporary_count$$))                        ;;;
;;;      Variable containing the number of the autosave temporary file.     ;;;
;;;  (buffer_info (user autosave_temporary_list$$))                         ;;;
;;;      Variable containing the names of temporary files still on disk.    ;;;
;;;  (buffer_info (user default_file_name$$))                               ;;;
;;;      The default file name of the buffer.                               ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; autosave_on                                                             ;;;
;;;    The command that turns autosave mode on.                             ;;;
;;;                                                                         ;;;
;;;           (autosave_on)                                                 ;;;
;;;                                                                         ;;;
;;;    This command does not take any arguments nor does it return any      ;;;
;;;    values.                                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom autosave_on
    &doc "AUTOSAVE: Turn on autosave mode"
    ; Prevent an attempt to turn on autosave mode if RPG mode is
    ; already on -- since both reset redisplay_hook$.
    (if (member (find_mode 'rpg) (buffer_info modes))
       ;then
            (ring_the_bell)
            (info_message "Autosave mode is not available from within RPG mode.")
            (return)
    )
    (if (null autosave_initialized$$)
       ;then
            (autosave_initialization$)
    )
    (if (null autosave_interval$)
       ;then
            ; Default interval is 300.
            (setq autosave_interval$ 300)
    )
    ; Counting begins at 0.
    (setq autosave_count$$ 0)
    ; Save the default file name.
    (buffer_info (user default_file_name$$) (file_name current_cursor))
    (do_forever
        (if (= (buffer_info default_file) "")
           ;then
                (buffer_info default_file
                    (prompt
                        "Assign a default file name to this buffer"
                    )
                )
                (buffer_info (user default_file_name$$)
                             (buffer_info default_file)
                )
            else
                (stop_doing)
        )
    )
    ; Append this buffer to a list of all autosave mode buffers.
    (if (not (member (buffer_info name) autosave_buffer_list$$))
       ;then
        (setq autosave_buffer_list$$
            (append autosave_buffer_list$$ (list (buffer_info name)))
        )
    )
    ; Verify that the length of the file name used by autosave will be
    ; no more than 24 characters -- i.e., reserve 8 characters for the
    ; autosave temporary file suffix (".T$EMttt").
    (let (
          (lngth (string_length (file_info (file_name current_cursor)
                                       entry_name)))
         )
        (if (> lngth 24)
           ;then
                (setq lngth (- lngth 24))
                (buffer_info (user autosave_file_name$$)
                             (substr (buffer_info (user default_file_name$$)) 1
                                     (- (string_length
                                            (buffer_info (user default_file_name$$))) lngth)
                             )
                )
            else
                (buffer_info (user autosave_file_name$$)
                             (buffer_info (user default_file_name$$))
                )
        )
    )
    (turn_mode_on autosave_mode$ last)
    (autosave?)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; autosave_off                                                            ;;;
;;;    The command that turns autosave mode off.                            ;;;
;;;                                                                         ;;;
;;;           (autosave_off)                                                ;;;
;;;                                                                         ;;;
;;;    This command does not take any arguments nor does it return any      ;;;
;;;    values.                                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom autosave_off
    &doc "AUTOSAVE: Turn off autosave mode"
    (if (null autosave_initialized$$)
       ;then
            (autosave_initialization$)
    )
    (if (member autosave_mode$ (buffer_info modes))
       ;then
            (buffer_info default_file
                         (buffer_info (user default_file_name$$))
            )
            (turn_mode_off autosave_mode$)
    )
    (autosave?)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; autosave                                                                ;;;
;;;    A simple "toggle" command that turns autosave mode on and off.       ;;;
;;;                                                                         ;;;
;;;           (autosave)                                                    ;;;
;;;                                                                         ;;;
;;;    This command does not take any arguments nor does it return any      ;;;
;;;    values.                                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom autosave
    &doc "AUTOSAVE: Toggle switch to turn autosave mode on/off"
    (if (null autosave_initialized$$)
       ;then
            (autosave_initialization$)
    )
    (if (member autosave_mode$ (buffer_info modes))
       ;then
            (autosave_off)
        else
            (autosave_on)
    )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  autosave?                                                              ;;;
;;;     This command reports whether autosave mode is on or off.            ;;;
;;;                                                                         ;;;
;;;              (autosave?)                                                ;;;
;;;                                                                         ;;;
;;;     This command does not take any arguments and does not return any    ;;;
;;;     values.                                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom autosave?
    &doc "AUTOSAVE: Report whether autosave mode is on or off"
    (if (member autosave_mode$ (buffer_info modes))
       ;then
            (info_message "Autosave mode is ""on"":"
                          " save interval = "
                          (integer_to_string autosave_interval$)
                          ";"
                          " temporary files retained = "
                           (integer_to_string autosave_temporary_count$)
                          "."
            )
        else
            (info_message "Autosave mode is ""off"".")
    )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  save_all_autosave_buffers                                              ;;;
;;;     This command explicitly saves to files all buffers in autosave      ;;;
;;;     mode and deletes all associated temporary files.                    ;;;
;;;                                                                         ;;;
;;;              (save_all_autosave_buffers)                                ;;;
;;;                                                                         ;;;
;;;     This command does not take any arguments and does not return any    ;;;
;;;     values.                                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom save_all_autosave_buffers
    &doc "AUTOSAVE: Save all autosave buffers to files"
    (let (
          (index 0)
         )
        (save_position
            (do_n_times (length autosave_buffer_list$$)
                (setq index (1+ index))
                (go_to_buffer (nthcar autosave_buffer_list$$ index))
                (save_file)
            )
        )
    )
    (info_message "All autosave files saved and temporaries deleted")
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  autosave_file$                                                         ;;;
;;;     When autosave is first called, this command overwrites the existing ;;;
;;;     save_file function. If called from autosave, this function will     ;;;
;;;     create a temporary save file with the following autosave suffix:    ;;;
;;;                       .T$EMttt                                          ;;;
;;;     where ttt is a 3-digit file number assigned by autosave. When the   ;;;
;;;     user explicitly executes save_file, autosave temporary files are    ;;;
;;;     deleted and the current contents of the buffer are saved as the     ;;;
;;;     default disk file. Whenever autosave_file$ is called for a buffer   ;;;
;;;     not in autosave mode, a simple save_file is performed.              ;;;
;;;                                                                         ;;;
;;;                       (save_file)                                       ;;;
;;;                                                                         ;;;
;;;     This command does not take any arguments and it does not return     ;;;
;;;     values.                                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom autosave_file$
    &doc "AUTOSAVE: Save file"
; Since saving a file can cause screen updates to occur with the progress
; of the save, avoid potential improprieties that might result from a recursive
; invocation of "autosave". Test here so that an autosave will not occur
; in the middle of a user saving a file normally.
(if autosave_in_progress$
   ;then
        (return)
)
  (let ((autosave_in_progress$ true))
    (setq autosave_count$$ 0)
    (if (null (buffer_info (user default_file_name$$)))
       ;then
            (if (not (buffer_info modified))
               ;then
                    (return)
                else
                    (orig_save_file)
                    (return)
            )
    )
    ; Determine whether or not to perform an autosave.
    (if autosave_calling$$
        (if (not (buffer_info modified))
            (return)
        )
        (if (= 0 autosave_temporary_count$)
            (orig_save_file)
            (return)
        )
        ; Make sure temporary file counter is set.
        (if (null (buffer_info (user autosave_temporary_count$$)))
           ;then
                (buffer_info (user autosave_temporary_count$$) -1)
        )
        ; Increment temporary file counter by 1.
        (buffer_info (user autosave_temporary_count$$)
            (1+ (buffer_info (user autosave_temporary_count$$)))
        )
        (if (> (buffer_info (user autosave_temporary_count$$)) 999)
           ;then
                (buffer_info (user autosave_temporary_count$$) -1)
                (setq autosave_count$$ 0)
                (return)
        )
        ; Determine the last 3 characters of the autosave temporary
        ; file suffix.
        (if (< (buffer_info (user autosave_temporary_count$$)) 10)
            ;then
                 (setq temporary_filler$$ "00")
             else
                 (if (< (buffer_info (user autosave_temporary_count$$)) 100)
                     ;then
                          (setq temporary_filler$$ "0")
                      else
                          (setq temporary_filler$$ "")
                 )
        )
        ; Write the temporary file.
        (write_file
            (catenate
                (buffer_info (user autosave_file_name$$))
                ".t$em"
                temporary_filler$$
                (integer_to_string (buffer_info (user autosave_temporary_count$$)))
            )
        )
        ; From the user's point of view, the buffer has been "modified"
        ; but not yet explicitly saved to a file.
        (buffer_info modified true)
        ; Add the temporary file number to a list.
        (buffer_info (user autosave_temporary_list$$)
            (append
                (buffer_info (user autosave_temporary_list$$))
                (list (buffer_info (user autosave_temporary_count$$)))
            )
        )
        ; Reset the list using autosave_temporary_trim$.
        (buffer_info (user autosave_temporary_list$$)
            (autosave_temporary_trim$ autosave_temporary_count$)
        )
        (return)
    )
    ; An explicit save_file is being executed.
    (info_message
        "Saving the buffer to a file and deleting autosave temporary files . . ."
    )
    (buffer_info (user autosave_temporary_list$$) (autosave_temporary_trim$))
    ; Reset the value of autosave_count$$ to avoid the screen updates which
    ; write_file will generate from causing another autosave.
    (setq autosave_count$$ 0)
    ; Write the current buffer's contents using the default file name.
    (write_file (buffer_info (user default_file_name$$)))
    ; Reset the value of autosave_count$$ to avoid the screen updates which
    ; write_file generated from leading prematurely to another autosave.
    (setq autosave_count$$ 0)
  )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; do_autosave$                                                            ;;;
;;;     Simple function called to check if autosaving should be done.       ;;;
;;;     It sets the value of autosave_calling$$ so that save_file knows     ;;;
;;;     whether or not to perform an autosave.                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun do_autosave$ ()
    ; Increment the screen update counter.
    (setq autosave_count$$ (1+ autosave_count$$))
    ; Determine whether an autosave is to be performed.
    (if (^= (modulo autosave_count$$ autosave_interval$) 0)
       ;then
            (return)
    )
    (setq autosave_calling$$ true)
    (save_file)
    (setq autosave_calling$$ false)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; autosave_temporary_trim$                                                ;;;
;;;   This function determines the number of temporary files for the        ;;;
;;;   current and then trims it to the amount specified by                  ;;;
;;;   autosave_temporary_count$. Note that 0 is a magic number.  If the     ;;;
;;;   number is 0, no (i.e., 0) temporary files are written.                ;;;
;;;                                                                         ;;;
;;;             (autosave_temporary_trim$ optional_integer)                 ;;;
;;;                                                                         ;;;
;;;   The optional_integer argument indicates the number of generations to  ;;;
;;;   keep.  The function returns a list of the temporary file numbers it   ;;;
;;;   has not deleted.                                                      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun autosave_temporary_trim$
    (&optional (temporary_file_count integer)
     &local    (delete_list string)
               (save_list list)
     &returns  list
    )
    (if (null temporary_file_count)
       ;then
            (setq temporary_file_count 0)
    )
    ; Make a copy of the list for future reference.
    (setq save_list (buffer_info (user autosave_temporary_list$$)))
    (if (= save_list () )
       ;then
            (return () )
    )
    ; Test for a list shorter than the number of temporary files wanted.
    (if (<= (length save_list) temporary_file_count)
       ;then
            (return save_list)
    )
    (setq delete_list "")
    (do_forever
        (if (= (length save_list) temporary_file_count)
           ;then
            (stop_doing)
        )
        (if (null save_list)
           ;then
            (stop_doing)
        )
        ; Extract out elements to shorten the list.
        (if (< (car save_list) 10)
            ;then
                 (setq temporary_filler$$ "00")
             else
                 (if (< (car save_list) 100)
                     ;then
                          (setq temporary_filler$$ "0")
                      else
                          (setq temporary_filler$$ "")
                 )
        )
        (setq delete_list
            (catenate delete_list temporary_filler$$ (integer_to_string (car save_list)) " ")
        )
        (setq save_list (cdr save_list))
    )
    ; Delete all autosave temporary files.
    (with_no_redisplay
        (primos_internal_quiet
            (catenate "delete "
                      (buffer_info (user autosave_file_name$$))
                      ".t$em"
                      "("
                      (trim delete_list)
                      ")"
            )
        )
    )
    (return save_list)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; autosave_initialization$                                                ;;;
;;;    This function performs general initialization. One effect is to set  ;;;
;;;    the value of save_file onto orig_save_file and then set save_file    ;;;
;;;    to be a new function.                                                ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun autosave_initialization$ ()
    ; Save original save_file.
    (if (null (fsymeval 'orig_save_file))
       ;then
            (fset 'orig_save_file (fsymeval 'save_file))
    )
    (setq autosave_in_progress$ false)
    ; Reset save_file to autosave_file$.
    (fset 'save_file (fsymeval 'autosave_file$))
    ; Save original redisplay_hook$.
    (fset 'orig_redisplay_hook$ (fsymeval 'redisplay_hook$))
    ; Reset redisplay_hook$ to autosave_hook$.
    (fset 'redisplay_hook$ (fsymeval 'autosave_hook$))
    (setq autosave_mode$ (find_mode 'autosave))
    (setq autosave_calling$$ false)
    (if (null autosave_temporary_count$)
       ;then
            (setq autosave_temporary_count$ 1)
    )
    (if (< autosave_temporary_count$ 0)
       ;then
            (setq autosave_temporary_count$ 0)
    )
    (setq autosave_initialized$$ true)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; autosave_hook$                                                          ;;;
;;;   This function is placed into redisplay_hook$ (actually set to be the  ;;;
;;;   same with an fset).  It is then called after every screen update. The ;;;
;;;   first thing done is increment a counter.  Then, this a modulo of this ;;;
;;;   counter is done using the value of autosave_interval$.  When it is 0, ;;;
;;;   the file is saved.                                                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun autosave_hook$ ()
    ; Check on state of save_file.  Important that this is only done once
    ; per invocation so that problems with recursion do not occur.
    (if (member autosave_mode$ (buffer_info modes))
       ;then
            (do_autosave$)
    )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; set_autosave_temporaries                                                ;;;
;;;    Set the number of temporary files to be retained by autosave.        ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom set_autosave_temporaries
    &doc "AUTOSAVE: Set the number of autosave temporary files"
    (if (null autosave_temporary_count$)
       ;then
            (setq autosave_temporary_count$ 1)
    )
    (info_message (integer_to_string autosave_temporary_count$)
                  " autosave temporary file(s) currently retained. "
                  "[<RETURN> keeps this value.]"
    )
    (do_forever
        (setq autosave_temporary_count$
            (prompt_for_integer
                "New number of autosave temporary files (minimum of 0, maximum of 100)"
                autosave_temporary_count$
            )
        )
        (if (& (>= autosave_temporary_count$ 0)
               (<= autosave_temporary_count$ 100))
           ;then
                (stop_doing)
        )
    )
    (info_message "New number of autosave temporary files retained is "
                  (integer_to_string autosave_temporary_count$)
                  "."
    )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; set_autosave_interval                                                   ;;;
;;;    Set the number of screen updates between autosaves.                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom set_autosave_interval
    &doc "AUTOSAVE: Set the number of screen updates between autosaves"
    (if (null autosave_interval$)
       ;then
            (setq autosave_interval$ 300)
    )
    (info_message "Current autosave interval is "
                  (integer_to_string autosave_interval$)
                  ". [<RETURN> keeps this value.]"
    )
    (do_forever
        (setq autosave_interval$
            (prompt_for_integer
                "New autosave interval (minimum of 3)"
                autosave_interval$
            )
        )
        (if (>= autosave_interval$ 3)
           ;then
                (stop_doing)
        )
    )
    (info_message
        "New autosave interval is "
        (integer_to_string autosave_interval$)
        "."
    )
)

