; BUFFER.EM, EMACSSRC>EMACS*>EXTENSIONS>SOURCES, EMACS DEVELOPMENT, 08/07/89
; Buffer manipulation extensions.
; Copyright (c) 1982, Prime Computer, Inc., Natick, MA 01760
;
; Modifications
;  Date     Engineer     Description of modification
; 08/07/89  Bugos        Modified to ensure that save_all_files works correctly
;                        for files having identical entrynames but different
;                        pathnames. (SPAR 4042351)
; 08/01/89  Bugos        Modified to ensure that write_file works correctly
;                        when a filename or full pathname is entered with one
;                        or more leading blank space characters. (SPAR 4042305)
; 07/25/88  Bugos        Rewrote mod_write_file to eliminate prompting for
;                        a non-existent file. (SPAR 4031744)
; 02/22/88  Bugos        Modified prev_buf$ and next_buf$ to work corectly
;                        for file names beginning with an identical
;                        character string. (SPAR 4026007)
; 02/22/88  Bugos        Rewrote next_buf$ and prev_buf$ and reformatted much
;                        of the other code in this module.
; 02/03/87  Bugos        Modified mod_write_file to prompt for a file name
;                        only when the buffer being written has no default
;                        file name associated with it.
; 10/22/86  Kingsbury    Changed mod_write_file so that it gives an error
;                        message when user doesn't specify a file name.
; 06/30/86  Bugos        Modified mod_split_window_stay to work correctly.
;                        (SPAR 4002411)
; 02/03/83  Zane         Insure DEFINITIONS is loaded:
;                        (if (null loaded$) (load))
; 02/03/83  Zane         Global variables: (setq other_buffer$ ())
; 02/03/83  Zane         (setq buffer_loaded$ true)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                          ;;;
;;; horiz_left            shifts the current window to the left    ^x}       ;;;
;;; horiz_right           shifts the current window to the right   ^x{       ;;;
;;; hscroll               takes the present column as the hcol     {esc}x... ;;;
;;;                       value                                              ;;;
;;; kill_rest_of_buffer   kills from point to end of buffer        {esc}^d   ;;;
;;; mark_bottom           puts a mark at bottom of buffer          ^x^z>     ;;;
;;; mark_top              puts a mark at top of buffer             ^x^z<     ;;;
;;; mod_one_window        same as one_window, except remembers     ^x1       ;;;
;;;                       what used to be in window                          ;;;
;;; mod_select_buf        same as select_buf but knows about file  ^xb       ;;;
;;;                       file hooks                                         ;;;
;;; mod_split_window      same as split_window, except retains     ^x2       ;;;
;;;                       past buffer                                        ;;;
;;; mod_split_window_stay same as split_window_stay, except        ^x3       ;;;
;;;                       remembers what used to be in window                ;;;
;;; mod_write_file        same as write_file except asks if it               ;;;
;;;                       is alright to overwrite                  ^x^w      ;;;
;;; next_buf              cycles through all user buffers forward  {esc}n    ;;;
;;; prev_buf              cycles through all user buffers backward {esc}p    ;;;
;;; repaint               moves point to a line on the screen      {ctrl-x}r ;;;
;;; reset                 refreshes screen, puts in one window               ;;;
;;;                       mode, and resets hcol to 1                         ;;;
;;; save_all_files        saves all files, ignores  modified       {esc}x ...;;;
;;;                       buffers                                            ;;;
;;; scroll_other_backward scrolls the other window backward        ^xv       ;;;
;;; scroll_other_forward  scrolls the other window forward         {esc}^v   ;;;
;;; set_hscroll           prompts for hcol value                             ;;;
;;;                                                                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; kill_rest_of_buffer                                                      ;;;
;;;     Command that kills from point to the end of the buffer.              ;;;
;;;     It is normally bound as {esc}^d.                                     ;;;
;;;                                                                          ;;;
;;;         (kill_rest_of_buffer)                                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom kill_rest_of_buffer
    &doc "Kills from point to end of buffer"
    (mark)
    (move_bottom)
    (kill_region)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mark_top                                                                 ;;;
;;;     Command that places a mark at the top of the buffer and leaves point ;;;
;;;     alone.  Bound as ^x^z<.                                              ;;;
;;;                                                                          ;;;
;;;         (mark_top)                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom mark_top
    &doc "Places a mark at top of buffer"
    (save_position
        (move_top)
        (mark)
    )
    (info_message "Top marked")
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mark_bottom                                                              ;;;
;;;    Command that places a mark at the bottom of the buffer and leaves     ;;;
;;;    point alone.  Bound as ^x^z>.                                         ;;;
;;;                                                                          ;;;
;;;         (mark_bottom)                                                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom mark_bottom
    &doc "Places a mark at bottom of buffer"
    (save_position
        (move_bottom)
        (mark)
    )
    (info_message "Bottom marked")
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; save_all_files                                                           ;;;
;;;      Automatically saves all files that have been modified.  This        ;;;
;;;      command ignores buffers that have been modified.  This is           ;;;
;;;      invoked as an extended command; i.e., {esc}x save_all_files         ;;;
;;;                                                                          ;;;
;;;           (save_all_files)     or     (save_all_files$)                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom save_all_files
    &doc "Saves all files, ignores modified buffers"
    (save_all_files$)
)

(defun save_all_files$ (&local (filename string)
                               (counter integer)
                        &returns integer)
    (setq counter 0)
    (save_position
      (with_no_redisplay               ; necessary so that files aren't shown
         (go_to_cursor (find_buffer ".buffers")) ; where list of files are
         (do_forever
              (if (forward_search " * ")
                                       ; search for modified flag
                  (setq filename (rest_of_line))
                  (setq buffername (stem_of_line))
                  (setq buffername
                    (substr buffername 1 (- (string_length buffername) 3))
                  )
                  (if (^= filename "") ; if = null, then am looking at
                                       ; a buffer, not file
                      (select_buf buffername)
                      (save_file)
                      (setq counter (1+ counter))
                                       ; keep track of number saved
                      (select_buf ".buffers"))
               else
                   (stop_doing)))))
    (info_message (catenate (integer_to_string counter)
                            " file(s) saved "))
    (return counter)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; next_buf                                                                 ;;;
;;;      Command that cycles forward through all buffers.  This command      ;;;
;;;      goes to the next buffer.  The order of searching is as defined in   ;;;
;;;      the ".buffers" buffer.  This command is bound as {esc}n,            ;;;
;;;      does not take arguments, and returns the name of the next buffer.   ;;;
;;;                                                                          ;;;
;;;         (next_buf)       or       (next_buf$)                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom next_buf
    &doc "Cycles forward through all user buffers"
    (next_buf$)
)

(defun next_buf$ (&local (start_buf string)
                         (start_buf_cursor cursor)
                  &returns string)
    ; Retain the current buffer name.
    (setq start_buf (buffer_name current_cursor))
    ; Suppress screen update.
    (with_no_redisplay
        ; Go to the ".buffers" buffer.
        (go_to_cursor (find_buffer ".buffers"))
        (do_forever
            ; Locate the name of the "starting" buffer.
            (forward_search (catenate  start_buf " "))
            (if (= (catenate start_buf " ")
                (substr (current_line) 1 (string_length (catenate start_buf " "))))
               ;then
                    (stop_doing)
            )
        )
        (begin_line)
        ; Remember the beginning cursor position.
        (setq start_buf_cursor (copy_cursor current_cursor))
        (do_forever
            (if (next_line)
               ;then
                    ; Make sure we're looking at a user buffer, not a temporary
                    ; "." buffer.
                    (if (^ (| (line_is_blank) (looking_at ".")))
                       ;then
                            (stop_doing)
                    )
                else
                    (move_top)
            )
        )
        ; Check to see that there is more than one user buffer.
        (if (^ (cursor_on_current_line_p start_buf_cursor))
           ;then
                (begin_line)
                (with_cursor here
                     (search_fd_in_line " ")
                     (select_buf (point_cursor_to_string here))
                )
                (return (buffer_info name))
            else
                (select_buf start_buf)
        )
    )
    (ring_the_bell)
    (info_message "There is only one user buffer.")
    (sleep_for_n_milliseconds 1000)
    (info_message "")
    (return "")
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; prev_buf                                                                 ;;;
;;;      Command that cycles backward through all buffers.  This command     ;;;
;;;      goes to the previous buffer.  The order of searching is as defined  ;;;
;;;      in the ".buffers" buffer.  This command is bound as {esc}p,         ;;;
;;;      does not take arguments, and returns the name of previous buffer.   ;;;
;;;                                                                          ;;;
;;;             (prev_buf)    or   (prev_buf$)                               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom prev_buf
    &doc "Cycles backward through all user buffers"
    (prev_buf$)
)

(defun prev_buf$ (&local (start_buf string)
                         (start_buf_cursor cursor)
                  &returns string)
    ; Retain the current buffer name.
    (setq start_buf (buffer_name current_cursor))
    ; Suppress screen update.
    (with_no_redisplay
        ; Go to the ".buffers" buffer.
        (go_to_cursor (find_buffer ".buffers"))
        (do_forever
            ; Locate the name of the "starting" buffer.
            (forward_search (catenate  start_buf " "))
            (if (= (catenate start_buf " ")
                (substr (current_line) 1 (string_length (catenate start_buf " "))))
               ;then
                    (stop_doing)
            )
        )
        (begin_line)
        ; Remember the beginning cursor position.
        (setq start_buf_cursor (copy_cursor current_cursor))
        (do_forever
            (if (prev_line)
               ;then
                    ; Make sure we're looking at a user buffer, not a temporary
                    ; "." buffer.
                    (if (^ (| (line_is_blank) (looking_at ".")))
                       ;then
                            (stop_doing)
                    )
                else
                    (move_bottom)
            )
        )
        ; Check to see that there is more than one user buffer.
        (if (^ (cursor_on_current_line_p start_buf_cursor))
           ;then
                (begin_line)
                (with_cursor here
                     (search_fd_in_line " ")
                     (select_buf (point_cursor_to_string here))
                )
                (return (buffer_info name))
            else
                (select_buf start_buf)
        )
    )
    (ring_the_bell)
    (info_message "There is only one user buffer.")
    (sleep_for_n_milliseconds 1000)
    (info_message "")
    (return "")
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; repaint                                                                  ;;;
;;;     This command moves point to the line specified.  With no argument    ;;;
;;;     or with an argument of one, point is moved to the first line of      ;;;
;;;     the screen.  It is bound as {control-x}r.                            ;;;
;;;                                                                          ;;;
;;;         (repaint optional-integer)                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom repaint
     &doc "Moves cursor to first line of screen"
     &na (&pass count &default 1)
     (goto_line (line_number (window_info top_line_cursor))) ; goto top line
     (let ((window_length (1+ (- (window_info bottom_line)   ; compute length
                                 (window_info top_line)))))  ; of window
          (if (> count window_length)                        ; make sure arg is
              (setq count window_length)                     ; in range
              (ring_the_bell)
              (info_message "Argument too big, it has been reset")))
     (begin_line)
     (if (> count 1)                  ; process argument
         (next_line (1- count))
      else
         (if (< count 1)
             (info_message "Negative arguments are not accepted")
             (ring_the_bell)))
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; scroll_other_forward                                                     ;;;
;;;     This command scrolls the window reached by other_window forward.     ;;;
;;;     It will accept both positive and negative arguments.  This command   ;;;
;;;     is bound as {esc}^v.                                                 ;;;
;;;                                                                          ;;;
;;;        (scroll_other_forward arg)                                        ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom scroll_other_forward
    &doc "Page the other window"
    &na (&pass count &default 1)
    (if (= (major_window_count) 1)
       ;then
            (error_message "Only one window")
        else
            (other_window)
            (next_page count)
            (other_window)
    )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; scroll_other_backward                                                    ;;;
;;;     This command scrolls the window reached by other_window backward.    ;;;
;;;     It will accept both positive and negative arguments.  It is          ;;;
;;;     normally bound as ^xv.                                               ;;;
;;;                                                                          ;;;
;;;        (scroll_other_backward arg)                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom scroll_other_backward
    &doc "Page the other window backward"
    &na (&pass count &default 1)
    (scroll_other_forward (- count))
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; hscroll                                                                  ;;;
;;;    Extended command that uses the current column position to set         ;;;
;;;    hcol (horizontal scrolling).                                          ;;;
;;;                                                                          ;;;
;;;         (hscroll)                                                        ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom hscroll
    &doc "Sets horizontal column to current pos"
    (hcol (cur_hpos))
)

(defun take_horiz ()
    (hscroll)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; set_hscroll                                                              ;;;
;;;     Extended comand that prompts for hcol value (horizontal              ;;;
;;;     scrolling).                                                          ;;;
;;;                                                                          ;;;
;;;         (set_hscroll)                                                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom set_hscroll
    &doc "Lets you say where hcol position will be"
    (hcol (prompt_for_integer "What is the horizontal column" 1))
)

(defun set_horiz ()
    (set_hscroll)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; horiz_left                                                               ;;;
;;;    Shifts the current window to the left by either an argument           ;;;
;;;    or by default.  This will not work in macros.                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom horiz_left
    &doc "Shift current window to the left"
    &numeric_arg (&pass offset &default 40)
    (hcol (- (window_info column_offset) offset))
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; horiz_right                                                              ;;;
;;;    Shifts the current window to the right by either an argument          ;;;
;;;    or by default.  This will not work in macros.                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom horiz_right
    &doc "Shift current window to the right"
    &numeric_arg (&pass offset &default 40)
    (hcol (+ (window_info column_offset) offset))
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; reset                                                                    ;;;
;;;     Extended command that puts you back in one window mode, sets         ;;;
;;;     hcol to 1, and refreshes the screen.                                 ;;;
;;;                                                                          ;;;
;;;         (reset)                                                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom reset
    &doc "Resets windows and columns"
    (hcol 1)
    (2doff)
    (#off)
    (set_mode "")
    (one_window)
    (refresh)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mod_split_window                                                         ;;;
;;;   This is the same as the fundamental command split_window.  The         ;;;
;;;   difference is that it goes to the buffer that previously was           ;;;
;;;   there.                                                                 ;;;
;;;                                                                          ;;;
;;;        (mod_split_window optional-integer)                               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom mod_split_window
    &doc "Split windows"
    &na (&pass count &default -99999)
    (if (= count -99999)
        (setq count (/ (1+ (- (window_info bottom_line)
                              (window_info top_line)))
                       2)))
    (split_window count)               ; same as split_window
    (if (not (null other_buffer$))      ; go to what was there previously
       ;then
            (go_to_cursor other_buffer$)
    )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mod_one_window                                                           ;;;
;;;  Same as one_window in fundamental Emacs.  The difference is that        ;;;
;;;  this saves the name of the other buffer to be used by                   ;;;
;;;  mod_split_window                                                        ;;;
;;;                                                                          ;;;
;;;      (mod_split_window)                                                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom mod_one_window
    &doc "Create single window"
    (other_window)                     ; retain name of other window's buffer
    (setq other_buffer$ (copy_cursor current_cursor))
    (other_window)
    (one_window)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mod_split_window_stay                                                    ;;;
;;;   This is the same as fundamental split_window_stay except that          ;;;
;;;   contents of a previous window are remembered                           ;;;
;;;                                                                          ;;;
;;;        (mod_split_window_stay)                                           ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom mod_split_window_stay
    &doc "Split window, stay in current"
    &na (&pass count &default -99999)
    (setq orig_cursor (copy_cursor current_cursor))
    (mod_split_window count)
    (do_forever
        (select_any_window)
        (if (= current_cursor orig_cursor)
           ;then
            (stop_doing)
        )
    )
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mod_write_file                                                           ;;;
;;;   Similar to write_file; however, mod_write_file prompts the user if the ;;;
;;;   file being written will overwrite a file.                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom mod_write_file
    &doc "Write specified file"
    &args ((place &prompt "Write file" &string))
    ; If the user has failed to specify a file name, verify that
    ; the buffer to be written has been assigned a default file name.
    (setq place (trim place))
    (if (= place "")
       ;then
        (if (^ (= (file_name current_cursor) ""))
           ;then
                (setq place (file_name current_cursor))
            else
                (info_message "No default file name for this buffer.")
                (ring_the_bell)
                (return)
        )
    )
    ; If the file already exists, ask if the user really wants to
    ; overwrite it.
    (if (file_info place exists)
        (if (^ (yesno "This file already exists. Do you want to overwrite it"))
            (mod_write_file)
            (return)
        )
    )
    (write_file place)
)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; mod_select_buf                                                           ;;;
;;;    The file hook macro does not work with select buffer. The following   ;;;
;;;    macro lets it be used.                                                ;;;
;;;                                                                          ;;;
;;;                  (mod_select_buf)                                        ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom mod_select_buf
    &doc "Select buffer"
    &args ((buf_name &prompt "Buffer" &string))
    (select_buf buf_name)
    (if (null (buffer_info modes))
       ;then
            (found_file_hook)
    )
)
