; BUFFER_INSERT.EM, EMACSSRC>EMACS*>EXTENSIONS>SOURCES, EMACS DEVELOPMENT, 08/02/89
; Contains buffer manipulation extensions in EMACS
; Copyright (c) 1982, Prime Computer, Inc., Natick, MA 01760
;
; Modification history:
;
;  Date     Engineer     Modification
; 08/02/89  Bugos        Modified to ensure that a buffer region will not be
;                        deleted when an attempt is made to prepend or append a
;                        buffer region to a file and the file name entered is
;                        not that of a file (e.g., directory or segment
;                        directory). (SPAR 4042264)
; 10/23/86  B. Kingsbury Changed all functions so that they call get_region$
;                        and insert_region$ rather than use kill_ring and
;                        point_cursor_to_string.  All commands now call
;                        subroutines so that all variables are local. Finally,
;                        added three functions: get_region$, insert_region$,
;                        and verified_prompt$.
; 06/30/86 Bugos         Added test for a null buffer name to both
;                        append_to_buf and prepend_to_buf (SPAR 4003190).
; 11/10/82 DNK           Improved error handling in insert_buff (Lou Gross
;                        suggestion).
;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; append_to_buf         takes a region and puts it at end of     ^xa      ;;;
;;;                       named buffer.                                     ;;;
;;; append_to_buf$        Where the work gets done.                         ;;;
;;; append_to_file        takes a region and puts it at end of a   ^x^z^a   ;;;
;;;                       named file                                        ;;;
;;; append_to_file$       Where the work gets done.                         ;;;
;;; get_region$           Function that returns a region between two cursors;;;
;;; insert_region$        Function that inserts the text in an array        ;;;
;;; insert_buf            save as insert file, but on buffers      ^x^zi    ;;;
;;; insert_buff           Where the work gets done.                         ;;;
;;; prepend_to_buf        takes a region and puts it at beginning  ^xp      ;;;
;;;                       of a named buffer.                                ;;;
;;; prepend_to_buf$       Where the work gets done.                         ;;;
;;; prepend_to_file       takes a region and puts it at beginning  ^x^zp    ;;;
;;;                       of a named file                                   ;;;
;;; prepend_to_file$      Where the work gets done.                         ;;;
;;; verified_prompt$      Function similar to prompt but won't let user     ;;;
;;;                       type a null response.                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; append_to_buf                                                           ;;;
;;;     Command that appends the current region to a buffer (which it       ;;;
;;;     prompts for). Append means put it at the end of the buffer.  If     ;;;
;;;     you use a numerical argument (any value), the region is copied.     ;;;
;;;     Otherwise, the region is deleted.  This is bound as ^xa.            ;;;
;;;                                                                         ;;;
;;;         (append_to_buf integer)                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom append_to_buf
     &doc "Appends current region to a buffer"
     &na (&pass count &default 1)
     (append_to_buf$ count))
(defun append_to_buf$ (       (count integer)
                       &local (region array)
                              (buf_name string)
                              (bottom_hor_pos integer)
                              (bottom_vert_pos integer))
                     ; insure that the user types a buffer name
     (setq buf_name (verified_prompt$ "Buffer Name"))
     (save_position
         (if (= count 1)      ; if arg, kill text
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) true))
          else
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) false)))
         (if (= 0 (array_dimension region))
             (info_message "The region is empty.")
             (ring_the_bell)
             (return))
         (go_to_buffer buf_name)
         (move_bottom)
                     ; set the original end_buffer position
         (setq bottom_hor_pos (cur_hpos))
         (setq bottom_vert_pos (line_number current_cursor))
                     ; for an insert of data, the inserted data is placed
                     ; before the cursor.  Thus, you need to record the
                     ; position absolutely.
         (insert_region$ region)
         (goto_line bottom_vert_pos)
         (go_to_hpos bottom_hor_pos)
         (mark)
         (move_bottom))
     (info_message "Region appended"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; prepend_to_buf                                                           ;;;
;;;     Command that prepends the current region to a buffer (which it       ;;;
;;;     prompts for). Prepend means put it at the beginning of the buffer.   ;;;
;;;     If you use a numerical argument (any value), the region is copied.   ;;;
;;;     Otherwise, the region is deleted.  This is bound as ^xp.             ;;;
;;;                                                                          ;;;
;;;         (prepend_to_buf integer)                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom prepend_to_buf
     &doc "Prepends current region to a buffer"
     &na (&pass count &default 1)
     (prepend_to_buf$ count))
(defun prepend_to_buf$ (       (count integer)
                       &local (region array)
                              (buf_name string)
                              (bottom_hor_pos integer)
                              (bottom_vert_pos integer))
                     ; insure that the user types a buffer name
     (setq buf_name (verified_prompt$ "Buffer Name"))
     (save_position
         (if (= count 1)      ; if arg, kill text
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) true))
          else
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) false)))
         (if (= 0 (array_dimension region))
             (info_message "The region is empty.")
             (ring_the_bell)
             (return))
         (go_to_buffer buf_name)
         (move_top)
         (insert_region$ region)
         (buffer_info mark (buffer_info top_cursor)))
       (info_message "Region prepended"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; append_to_file                                                           ;;;
;;;     Command that appends the current region to a file (which it          ;;;
;;;     prompts for).  Append means put it at the end of the file.  If       ;;;
;;;     you use a numerical argument (any value), the region is copied.      ;;;
;;;     Otherwise, the region is deleted.  This is bound as ^x^za.           ;;;
;;;                                                                          ;;;
;;;         (append_to_file integer)                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom append_to_file
     &doc "Appends current region to a file"
     &na (&pass count &default 1)
     (append_to_file$ count))
(defun append_to_file$ (       (count integer)
                        &local (region array)
                               (file_name string)
                               (bottom_hor_pos integer)
                               (bottom_vert_pos integer))
                     ; insure that the user types a file name
     (setq file_name (verified_prompt$ "File Name"))
     (if (^= "file" (file_info file_name type))
        ;then
             (ring_the_bell)
             (info_message (catenate "Error -- not a file: "
                                     (file_info file_name path_name) " ("
                                     (file_info file_name type) ")"
                           )
             )
             (return)
     )
     (save_position
         (if (= count 1)      ; if arg, kill text
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) true))
          else
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) false)))
         (if (= 0 (array_dimension region))
             (info_message "The region is empty.")
             (ring_the_bell)
             (return))
         (with_no_redisplay
             (if (file_info file_name exists)  ;check if file exists
                 (find_file file_name)
              else                         ; if it doesn't, create it
                 (go_to_buffer file_name)
                 (write_file file_name))
             (move_bottom)
                     ; set the original end_buffer position
             (setq bottom_hor_pos (cur_hpos))
             (setq bottom_vert_pos (line_number current_cursor))
                     ; for an insert of data, the inserted data is placed
                     ; before the cursor.  Thus, you need to record the
                     ; position absolutely.
            (insert_region$ region)
            (goto_line bottom_vert_pos)
            (go_to_hpos bottom_hor_pos)
            (mark) (move_bottom) (save_file)))
      (info_message "Region appended"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; prepend_to_file                                                          ;;;
;;;      Command that prepends the current region to a file (which it        ;;;
;;;      prompts for).  Prepend means put it at the beginning of th file.    ;;;
;;;      if you use a numerical argument (any value), the region is copied.  ;;;
;;;      Otherwise, the region is deleted.  This is bound as ^x^zp.          ;;;
;;;                                                                          ;;;
;;;         (prepend_to_file integer)                                        ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom prepend_to_file
   &doc "Prepends current region to a file"
   &na (&pass count &default 1)
   (prepend_to_file$ count))
(defun prepend_to_file$ (      (count integer)
                        &local (region array)
                               (file_name string)
                               (bottom_hor_pos integer)
                               (bottom_vert_pos integer))
                     ; insure that the user types a file name
     (setq file_name (verified_prompt$ "File Name"))
     (if (^= "file" (file_info file_name type))
        ;then
             (ring_the_bell)
             (info_message (catenate "Error -- not a file: "
                                     (file_info file_name path_name) " ("
                                     (file_info file_name type) ")"
                           )
             )
             (return)
     )
     (save_position
         (if (= count 1)      ; if arg, kill text
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) true))
          else
             (setq region (get_region$ (copy_cursor current_cursor)
                                       (buffer_info mark) false)))
         (if (= 0 (array_dimension region))
             (info_message "The region is empty.")
             (ring_the_bell)
             (return))
         (with_no_redisplay
             (if (file_info file_name exists)  ;check if file exists
                 (find_file file_name)
              else                         ; if it doesn't, create it
                 (go_to_buffer file_name)
                 (write_file file_name))
             (move_top)
             (insert_region$ region)
             (mark)
             (move_top)
             (save_file)))
         (info_message "Region prepended"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; insert_buf                                                               ;;;
;;;     This command takes the contents of a buffer and inserts it at point. ;;;
;;;                                                                          ;;;
;;;         (insert_buf)   or     (insert_buff)                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom insert_buf
     &doc "Inserts a buffer at point"
     (insert_buff))
(defun insert_buff (&local (region string)
                           (hor_pos integer)
                           (vert_pos integer))
       (save_excursion
           (go_to_buffer (verified_prompt$ "Buffer name"))
           (setq region (get_region$ (buffer_info top_cursor)
                                     (buffer_info bottom_cursor))))
        ; if no data passed back, then region is an array of
        ; with a diminsion of 0
       (if (= 0 (array_dimension region))
           (info_message "Named buffer is empty.")
           (ring_the_bell)
           (return))
        ; for an insert of data, the inserted data is placed before the
        ; cursor.  Thus, you need to record the position absolutely.
       (setq hor_pos (cur_hpos))
       (setq vert_pos (line_number current_cursor))
       (insert_region$ region)
       (mark)
       (goto_line vert_pos)
       (go_to_hpos hor_pos)
       (info_message "Buffer inserted"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; get_region$                                                             ;;;
;;;     This function returns all text between two cursors.  It is similar  ;;;
;;;     to range_to_string except that there is no limitation on the length ;;;
;;;     of the text betwen the two cursors.  This function places the       ;;;
;;;     region into an array, which is returned.  Each element of the array ;;;
;;;     consists of 100 lines of the file.  Thus, each line can have about  ;;;
;;;     3276 characters.                                                    ;;;
;;;                                                                         ;;;
;;;          (get_region$ cursor_1 cursor_2 [delete_boolean])               ;;;
;;;                                                                         ;;;
;;;     cursor_1 and cursor_2 are the cursors that describe the limits of   ;;;
;;;                           the region.                                   ;;;
;;;     delete_boolean        is a true/false value that indicates if the   ;;;
;;;                           region should also be deleted.  If the value  ;;;
;;;                           is null or false, the region is NOT deleted.  ;;;
;;;                                                                         ;;;
;;;     Returned entity:                                                    ;;;
;;;          An array of unknown dimension                                  ;;;
;;;                                                                         ;;;
;;;     Usage:                                                              ;;;
;;;            (setq region (get_region$ cursor_1 cursor_2))                ;;;
;;;            (setq region (get_region$ cursor_1 cursor_2 true))           ;;;
;;;                                                                         ;;;
;;;     Note:  Do not pass the value current_cursor to get_region. Instead, ;;;
;;;            pass (copy_cursor current_cursor)                            ;;;
;;;                                                                         ;;;
;;;     Global Variables: None                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun get_region$ (           (top_buff_cursor cursor)
                               (bottom_buff_cursor cursor)
                    &optional  (delete_region boolean)
                    &local     (temp_cursor cursor)
                               (region array)
                               (counter integer)
                               (region_number integer)
                    &returns   array)
   (save_position
                     ; if the two cursors are identical, pass back an array
                     ; with 0 length
       (if (= bottom_buff_cursor top_buff_cursor)
           (return (make_array 'array 0)))
       (if (null delete_region) (setq delete_region false))
                     ; make sure that the top_buff_cursor is the one with the
                     ; lowest line and horizontal positon.  If they are not,
                     ; exchange them
       (if (< bottom_buff_cursor top_buff_cursor)
           (setq temp_cursor bottom_buff_cursor)
           (setq bottom_buff_cursor top_buff_cursor)
           (setq top_buff_cursor temp_cursor))
                      ; figure out how many lines are in the region.
       (setq region_number (1+ (- (line_number bottom_buff_cursor)
                                  (line_number top_buff_cursor))))
                      ; figure out how many 100 line regions exist in the file
       (setq region_number (1+ (/ region_number 100)))
                     ; set an array where each element in the array will hold
                     ; 100 lines.
       (setq region (make_array 'string region_number))
       (setq counter 0)
                     ; to to beginning of region
       (go_to_cursor top_buff_cursor)
       (setq temp_cursor top_buff_cursor)
                     ; add 100 lines to each array element
       (do_n_times region_number
          (if (> 101 (- (line_number bottom_buff_cursor)
                        (line_number current_cursor)))
                     ; less than 100 lines left
              (go_to_cursor bottom_buff_cursor)
           else
                     ; more than 100 lines left
              (goto_line (+ 100 (line_number current_cursor))))
          (aset (range_to_string current_cursor top_buff_cursor)
                region counter)
          (setq counter (1+ counter))
          (setq top_buff_cursor (copy_cursor current_cursor)))
       (if delete_region
           (delete_point_cursor temp_cursor))
       (return region)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; insert_region$                                                          ;;;
;;;    This function accepts inserts text into the buffer.  The argument    ;;;
;;;    should be an array where each element is a string.                   ;;;
;;;                                                                         ;;;
;;;         (insert_region$ region_array)                                   ;;;
;;;                                                                         ;;;
;;;    region_array is an array where each element contains a string.       ;;;
;;;                                                                         ;;;
;;;    Returned Entity:                                                     ;;;
;;;         A boolean indicating success.                                   ;;;
;;;                                                                         ;;;
;;;    Usage:                                                               ;;;
;;;        (if (insert_region$ region_array) ...                            ;;;
;;;                                                                         ;;;
;;;    Global Varables:  None                                               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun insert_region$ (          (region array)
                       &local    (num_dimensions integer)
                       &returns  boolean)
       (setq num_dimensions (array_dimension region))
       (if (= 0 num_dimensions)
           (info_message "The region list is empty.")
           (ring_the_bell)
           (return false))
       (setq counter 0)
       (do_n_times num_dimensions
           (if (^= (typef (aref region counter)) 5)
               (info_message "The array does not contain a string elements")
               (ring_the_bell)
               (return false))
           (insert (aref region counter))
           (setq counter (1+ counter)))
       (return true))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; verified_prompt$                                                        ;;;
;;;    Function similar to prompt.  The major difference is that the user   ;;;
;;;    must type a response, rather than a carriage return.  Also, when an  ;;;
;;;    error occurs, the user must type a carriage return to continue to    ;;;
;;;    acknowledge the error                                                ;;;
;;;                                                                         ;;;
;;;         (verified_prompt$ prompt_text [ help_text ])                    ;;;
;;;                                                                         ;;;
;;;    prompt_text is any text string                                       ;;;
;;;    help_text is an optional string to be transmitted to the function    ;;;
;;;              such that if the user types a ? in response to the prompt, ;;;
;;;              the help text is displayed                                 ;;;
;;;                                                                         ;;;
;;;    Returned entity:                                                     ;;;
;;;         A string that is greater than 0 length                          ;;;
;;;                                                                         ;;;
;;;    Usage:                                                               ;;;
;;;         (setq string (verified_prompt$ string))                         ;;;
;;;                                                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun verified_prompt$ (          (text string)
                         &optional (help string)
                         &returns  string
                         &local    (prompt_string string))
     (if (null help) (setq help ""))
     (if (^= help "")
         (setq text (catenate text " [? for help]")))
     (do_forever
         (setq prompt_string (trim (prompt text)))
         (select prompt_string
            ""   (info_message "You must type a response.")
                 (ring_the_bell)
                 (prompt "Type <RET> to continue")
                 (info_message "")
            "?"
                 (if (= help "")
                     (info_message "Help is not available.")
                  else
                     (info_message help))
             otherwise
                 (return prompt_string)  )))
