;  FILL.EM, EMACS*>EXTENSIONS>SOURCES, EMACS DEVELOPMENT, 03/25/87
;  Miscellaneous routines that deal with margins in EMACS
;  Copyright (c) 1982, Prime Computer, Inc., Natick, MA 01760
;
; Modifications:
;
;   Date    Engineer     Description of modification
; 03/25/87  Bugos        Modified build_para$ (in fill_para) to disable
;                        justification if the prefix count / integer argument
;                        of fill_para is either 1 or -1. (SPAR 4011434)
; 12/11/86  Kingsbury    Fixed wrap$ so it wraps at fill column, not after
; 07/09/86  Bugos        Fixed build_para$ (in fill_para) to ensure text
;                        justification if negative prefix count / integer
;                        argument is used with fill_para. (SPAR 4003137)
; 04/05/85  Sarkisian    Fixed build_para$; restored justification
;                        funcionality.
; 02/06/85  SARKISIAN    Fixed build_para$ to catch infinite loop problem
;                        in splitting lines with no blanks.
; 11/14/84  MAM          Fixed a bug to prevent gutters in filling
;                        (SPAR 3009063).
; 07/15/83  MAM          Fixed typo, placment to placement, line 248.
; 11/02/82  BMZ          Added fasload of text package for back_para.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; all_modes_off     extended command that shuts off all                    ;;;
;;;                   modes                                                  ;;;
;;; fill              inserts cr so you don't have to         {space}        ;;;
;;; fill_on           turns on fill_mode                                     ;;;
;;; fill_off          turns off fill_mode                                    ;;;
;;; fill_para         fills and justifies a paragraph         {esc}q         ;;;
;;; indent_to_fill_prefix                                                    ;;;
;;;                   indents left margin to fill prefix      {esc}^i        ;;;
;;; set_left_margin   extended command that sets the left                    ;;;
;;;                   margin                                                 ;;;
;;; set_right_margin  extended command that sets the right                   ;;;
;;;                   margin for filling                                     ;;;
;;; take_left_margin  same as set_left_margin except user is  ^x^zf          ;;;
;;;                   not prompted for position                              ;;;
;;; take_right_margin same as set_right_margin except user    ^x.            ;;;
;;;                   is not prompted for position                           ;;;
;;; tell_modes        extended command that prints lists of                  ;;;
;;;                   all modes                                              ;;;
;;; tell_left_margin  displays what the left margin is                       ;;;
;;; tell_right_margin displays what the right margin is                      ;;;
;;;                                                                          ;;;

; 02/03/83 Zane   (if (null loaded$) (load))
; 02/03/83 Zane   (if (null text_loaded$)
; 02/03/83 Zane       (fasload "emacs*>extensions>text"))

;;; Global Variables:                                                        ;;;
;
;   Now set up as part of core initializtion
;
; 02/03/83 Zane         (if (null fill_prefix)
; 02/03/83 Zane             (setq fill_prefix 0))
; 02/03/83 Zane
; 02/03/83 Zane         (if (null default_right_margin)
; 02/03/83 Zane             (setq default_right_margin 70))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; fill_on                                                                  ;;;
;;;   Extended command that turns on fill mode.                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom fill_on
     &doc "Allows character wrapping for this buffer"
     (set_mode_key "fill" " "  "wrap")
     (set_mode_key "fill" "^j" "wrap")
     (if (= (buffer_info fill_column) 0)
         (buffer_info fill_column default_right_margin))
     (turn_mode_on (find_mode 'fill) last)
     (info_message (catenate "Word wrapping is on at a right margin of "
                             (integer_to_string default_right_margin))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; fill_off                                                                 ;;;
;;;    Extended command that turns off fill mode.                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom fill_off
     &doc "Shuts off wrapping for this buffer"
     (buffer_info fill_column 0)
     (turn_mode_off (find_mode 'fill))
     (buffer_info fill_column 0)
     (info_message "Word wrapping is now off"))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; wrap                                                                     ;;;
;;;   Command that takes a space and carriage return and checks its          ;;;
;;;   horizontal position. If it is greater than fill column, wrap           ;;;
;;;   puts in a carriage return.                                             ;;;
;;;                                                                          ;;;
;;;        (wrap optional-integer)                                           ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom wrap
     &doc "Inserts cr so you don't have to"
     &na (&pass count &default 1)
     (do_n_times count (wrap$)))

(defun wrap$ ()
       (if (null character_argument) (setq character_argument " "))
       (if (and (>= (cur_hpos) (buffer_info fill_column))
                (> (buffer_info fill_column) 10))
                                       ; insure enough space
           (insert "")              ; insert three ^z garbage chars
           (go_to_hpos (buffer_info fill_column))
                                       ; go to fill_column
           (if (search_bk_in_line " ") ; find whitespace previous and delete
               (delete_white_sides)
               (cr)                    ; finally add cr
               (save_excursion         ; if next line is blank, get rid of
                   (next_line)         ; it.  try to do something similar
                   (if (line_is_blank) ; to fundamental in this respect
                       (if_at " "
                              (kill_line 2)
                           else
                              (delete_char)))
                   )
               )
           (forward_search "")      ; get rid of garbage
           (rubout_char 3))
       (insert (character_argument)))  ; insert character causing invokation

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; indent_to_fill_prefix                                                    ;;;
;;;    This function indents a region to the fill_prefix.  This is bound as  ;;;
;;;    {esc}{tab}.                                                           ;;;
;;;                                                                          ;;;
;;;         (indent_for_fill_prefix)                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom indent_to_fill_prefix
     &doc "Indents line to a fill prefix"
     &na (&pass count &default 1)
     (if (< count 0)                   ; set different point if negative
         (setq count (- count))        ; argument, makes things easier for
         (prev_line count))            ; next step
     (do_n_times count
         (begin_line)
         (delete_white_sides)
         (whitespace_to_hpos fill_prefix)   ; this must be pre-set
         (next_line))                  ; default to zero
     (prev_line)
     (begin_line)
     (skip_over_white))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; fill_para                                                                ;;;
;;;   Command that fills a paragraph so that each line does not have         ;;;
;;;   more than the number of characters indicated in fill_column.  It       ;;;
;;;   will take characters from other lines as it wishes to have each        ;;;
;;;   line have about the same number of characters.  If a fill_prefix       ;;;
;;;   is in force, the text is indented the number of characters indicated   ;;;
;;;   by the prefix.  If this command is used with an argument other than 1, ;;;
;;;   the paragraph is justified as well as filled.  Additionally, if the    ;;;
;;;   argument is negative, the ability to create a "list"-type paragraph is ;;;
;;;   disabled. If a fill_prefix is in force with a fill and justify, the    ;;;
;;;   text is indented by that amount as well. This is bound as {esc}q.      ;;;
;;;                                                                          ;;;
;;;        (fill_para optional-integer)                                      ;;;
;;;                                                                          ;;;
;;;   where optional-integer (other than 1) tells Emacs to fill and justify  ;;;
;;;   and, possibly, to disable the ability to create a "list"-type          ;;;
;;;   paragraph.                                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom fill_para
     &doc "Re-fills a paragraph a fill column"
     &na (&pass count &default 1)
     (fill_para$ count))

(defun fill_para$ (       (count integer)
                   &local (paragraph string)
                          (pos integer)
                          (list_para boolean))
   (with_no_redisplay
     (if (< (buffer_info fill_column) 10)
                                       ; make sure enough space to be
                                       ; meaningful
         (info_message "Not enough space to fill, Reset fill column")
         (ring_the_bell)
         (return))
     (if (= fill_prefix 1)             ; fill_prefix of 1 should be allowed
         (setq fill_prefix 0))         ; however, EMACS offsets different
     (backward_para)                   ; find beginning of para
     (if (not (beginning_of_buffer_p)) ; at this point, assume that nothing
         (next_line))                  ; starts para
     (if_at "." (next_line))           ; however, dont screw up for runoff
     (with_cursor here
         (forward_para)                ; isolate paragraph
         (setq paragraph (point_cursor_to_string here))
                                       ; write it to special buffer
         (create_text_save_buffer$ paragraph)
                                       ; now get rid of it
         (delete_point_cursor here))
     (open_line)                       ; this sometimes causes problems
                                       ; get rid of carriage returns
     (insert (translate paragraph " " "~n"))
     (begin_line)
     (if_at "o "                       ; determine if list para
          (setq list_para true)
      else
          (forward_char)
          (if_at ". "
               (setq list_para true)
           else
               (forward_char)
               (if_at ". "
                   (setq list_para true)
                else
                   (setq list_para false))))
     (begin_line)
     (if (< count 0) (setq list_para false))
                                       ; if there is a neg argument,user
                                       ; wont get list_para
     (if list_para
         (build_list_para$ count)
      else
         (build_para$ count))))

(defun build_para$ (         (count integer)
                    &local   (start_point cursor)
                             (end_point cursor)
                             (end_num integer)
                             (placement integer)
                             (badlines integer)
                             (badline_linenum integer)
                    &returns cursor)
    (setq badlines 0)                  ; number of lines too long for
                                       ; fill_column
    (setq start_point (copy_cursor current_cursor))
    (self_insert " " fill_prefix)      ; insert spaces for fill prefix
    (do_forever
       (if (not (go_to_hpos (buffer_info fill_column)))
                                       ; special check if they match
           (if (= (buffer_info fill_column)
                  (string_length (current_line)))
               (end_line)
               (delete_char 2))
           (stop_doing))
                                              ;
       (if (not (search_bk_in_line " ")) then ; cannot find blank...can't wrap
                                              ; write out as is and inform user
                                              ; later
           (setq badlines (1+ badlines))
                                              ; keep track of last bad line #
           (setq badline_linenum (line_number current_cursor))
           (search_fd_in_line " "))            ; find next blank and write out
                                               ; as is
       (delete_white_sides)
       (cr)
       (self_insert " " fill_prefix)) ; add additional fill prefix
                                       ; if arg = 1, do not fill

    (if (& (^= count 1) (^= count -1))
      then
        ; reset paragraph limits, i.e. do right justification
        (setq end_num (line_number current_cursor))
        (setq end_point (copy_cursor current_cursor))
        (setq placement 1)                 ; variable for left/right space insertion
        (go_to_cursor start_point)
                                           ; do the number of times = line number
        (do_n_times (- end_num (line_number current_cursor))
            (if (= (pad$ placement) 1)
                (setq placement 1)         ; this flip flops so don't get gutters
             else                          ; when inserting spaces
                (setq placement 0))
            (next_line))
        (go_to_cursor end_point))

    (if (> badlines 0) then            ; inform user some lines in paragraph
        (ring_the_bell)                ; couldn't be wrapped
        (if (= badlines 1) then
            (info_message "Line " (integer_to_string badline_linenum)
                          " couldn't be split...no blank found before fill column "
                          "of " (integer_to_string (buffer_info fill_column)))
        else
            (info_message (integer_to_string badlines) " lines couldn't be "
                          "split...no blanks found in them before fill column "
                          "of "(integer_to_string (buffer_info fill_column)))))
    (return start_point))

(defun build_list_para$ (       (count integer)
                        &local (temp_fill_prefix integer)
                               (tag string)
                               (length integer)
                               (start_point cursor))
                                       ; this routine sets up environment
                                       ; so that build_para can do its thing
      (setq temp_fill_prefix fill_prefix)
                                       ; set up right indentation
      (buffer_info fill_column (- (buffer_info fill_column) 4))
      (if_at "o"                       ; different list types have dif. indents
           (setq fill_prefix (+ fill_prefix 7))
       else
           (setq fill_prefix (+ fill_prefix 8)))
      (with_cursor here                ; save the indicator such as bullet
         (skip_to_white)               ; or number
         (setq tag (point_cursor_to_string here))
         (setq length (string_length tag))
         (delete_point_cursor here))
      (white_delete)                   ; make sure no garbage spaces
                                       ; now do a build_para
      (setq start_point (build_para$ count))
                                       ; restore fill_column
      (buffer_info fill_column (+ (buffer_info fill_column) 4))
      (save_excursion                  ; restore indicator
         (go_to_cursor start_point)
         (go_to_hpos (- fill_prefix (1+ length)))
         (delete_char length)
         (insert tag)
         (setq fill_prefix temp_fill_prefix)))

(defun pad$ (         (placement integer)
             &local   (to_pad integer)
             &returns integer)
                                       ; figure out how many spaces to add
     (setq to_pad (- (buffer_info fill_column) (string_length (current_line))))
     (if (= to_pad 0) (return placement))
                                       ; if = 1 pad from right
                                       ; otherwise pad from left
     (if (= placement 1)
         (end_line)
         (do_forever
            (if (search_bk_in_line " ")  ; find a space and make sure after
                                         ; fill prefix
                (if (<= (cur_hpos) fill_prefix)
                    (end_line)
                 else
                    (insert " ")
                    (setq to_pad (1- to_pad))   ; decrement amount of spaces to
                    (if (= to_pad 0)            ; add.  If zero, exit
                        (setq placement 0)
                        (return placement))
                    (skip_back_over_white))
             else
                 (end_line))))
                                       ; padding from the left begins
     (begin_line)
     (skip_over_white)                 ; avoid fill prefix
     (do_forever
          (if (search_fd_in_line " ")  ; same as above except reverse
              (insert " ")             ; direction
              (setq to_pad (1- to_pad))
              (if (= to_pad 0)
                  (setq placement 1)
                  (return placement))
              (skip_over_white)
           else
              (begin_line)
              (skip_over_white))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;untidy                                                                    ;;;
;;;  Unjustifies a paragraph made neat by a fill_para                        ;;;
;;;                                                                          ;;;
;;;      (untidy)     or      (untidy$)                                      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defcom untidy
   &doc "Unjustifies a paragraph"
   (untidy$))
(defun untidy$ (&local (paragraph string)
                       (pos integer)
                       (wrap_column integer)
                       (line_num integer)
                       (temp_fill_prefix integer))
     (if (< (buffer_info fill_column) 10)
                                       ; check for sufficient space
         (info_message "Not enough space to fill, Reset fill column")
         (ring_the_bell)
         (return))
                                       ; essentially, all that occurs is
                                       ; that initial blanks are shaved
                                       ; then region is filled
     (setq temp_fill_prefix fill_prefix)
     (setq fill_prefix 0)
     (begin_line)
     (if (line_is_blank)
         (prev_line)
      else
         (if_at "."
              (prev_line)))
     (do_forever
         (if_at "." (next_line) (stop_doing))
         (if (line_is_blank) (next_line) (stop_doing))
         (if (beginning_of_buffer_p) (stop_doing))
         (prev_line))
     (with_cursor here
         (do_forever
            (if_at "." (stop_doing))
            (if (line_is_blank) (stop_doing))
            (if (lastlinep) (move_bottom) (stop_doing))
            (next_line))
         (setq paragraph (point_cursor_to_string here))
         (delete_point_cursor here))
      (create_text_save_buffer$ paragraph)
      (setq paragraph (trim paragraph))
      (open_line)
      (save_excursion
        (insert (trim (translate paragraph " " "~n"))))
      (setq line_num (line_number current_cursor))
      (do_forever                      ; get rid of double spaces
          (if (forward_search "  ")
              (if (^= (line_number current_cursor) line_num)
                  (stop_doing))
              (rubout_char)
              (back_char)
           else
              (stop_doing)))
       (goto_line line_num)
       (checker$ ".")
       (checker$ "!")
       (checker$ "?")
       (fill_para -1)
       (setq fill_prefix temp_fill_prefix))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; checker                                                                  ;;;
;;;      This is a pretty stupid routine that adds a double space after some ;;;
;;;      punctuation points.  One of these days, should spend some time and  ;;;
;;;      add some intelligence.                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun checker$ (       (to_check string)
                &local (line_num integer))
   (setq line_num (line_number current_cursor))
   (save_excursion
     (do_forever
         (if (forward_search to_check)
             (if (^= (line_number current_cursor) line_num)
                 (stop_doing))
             (if_at " "
                 (insert " "))
           else
              (stop_doing)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; set_right_margin                                                         ;;;
;;;   Command that sets the right margin for wrapping.  This is bound        ;;;
;;;   as ^x^zf.                                                              ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom set_right_margin
     &doc "Sets the right margin to whatever you specify"
     &args ((column &prompt "What is the right margin"
                    &default default_right_margin
                    &integer))
     (buffer_info fill_column column)
     (if (> (buffer_info fill_column) 10)
         (tell_right_margin)))
(defun set_right ()
     (set_right_margin))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;take_right_margin                                                         ;;;
;;;   Command that takes the right margin from the current column position   ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom take_right_margin
   &doc "Takes the right magin from current position"
   (buffer_info fill_column (numeric_argument (cur_hpos)))
   (tell_right_margin))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; tell_right_margin                                                        ;;;
;;;   Extended command that tells what the right margin is.                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom tell_right_margin
     &doc "Prints what the right margin is"
     (if (> (buffer_info fill_column) 10)
         (info_message
             (catenate "Right margin is "
                       (integer_to_string (buffer_info fill_column))))
      else
         (info_message
              (catenate "Right margin is "
                        (integer_to_string (buffer_info fill_column))
                   ".  Word wrapping disabled because margin is too small."))))
(defun tell_right ()
     (tell_right_margin))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; take_left_margin                                                         ;;;
;;;   Extended command that sets the left margin to the current position     ;;;
;;;   of point.  This is bound as ^x.                                        ;;;
;;;                                                                          ;;;
;;;        (take_left_margin)                                                ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom take_left_margin
     &doc "Takes the left margin from the current column"
     (setq fill_prefix (cur_hpos))
     (tell_left_margin))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; tell_left_margin                                                         ;;;
;;;   Extended command that prints what the left column is.                  ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom tell_left_margin
    &doc "Tells what the left margin is"
    (info_message
      (catenate "The left column is " (integer_to_string fill_prefix))))
(defun tell_left ()
    (tell_left_margin))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; set_left_margin                                                          ;;;
;;;   Extended command that prompts for a left column number.                ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defcom set_left_margin
    &doc "Sets the left margin"
    &args ((column &prompt "What is the left margin"
                   &default 0
                   &integer))
    (setq fill_prefix column))
(defun set_left ()
    (set_left_margin))


; 02/03/83 Zane  (setq fill_loaded$ true)
