;  INDENT_AND_WHITE.EM,  EMACS*>EXTENSIONS>SOURCES, EMACS DEVELOPMENT, 10/07/87
;  The library of whitespace functions for EMACS.
;  Copyright (c) 1982, Prime Computer, Inc., Natick, MA 01760


; Modifications:
; Date      Engineer     Description of modification
; 10/07/87  Bugos        Added tests for read-only buffer prior to attempting to
;                        delete blank lines around point (delete_blank_lines).
;                        (SPAR 4019899)
; 07/17/85  MAM          Changed white_delete and delete_blank_lines to use
;                        kill_line instead of delete_char to fix overlay mode.
; 02/03/83  Zane         (setq indent_and_white_loaded$ true)
; 02/03/83  Zane         (setq init_whitespace whitespace)
; 02/03/83  Zane         (setq whitespace " ")


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; back_to_nonwhite      goes to first nonwhite char on line      {esc}m    ;;;
;;; cret_indent_relative  same as indent_relative but adds {cr}    ^x^j      ;;;
;;; delete_blank_lines    deletes blank lines around point         ^x^o      ;;;
;;; indent_relative       lines up text or tabs text over more     {esc}i    ;;;
;;; leave_one_white       deletes whitespace around point, but  {esc}{space} ;;;
;;; merge_lines           joins two lines together                 {esc}^    ;;;
;;; split_line            breaks a line, preserving the horizontal {esc}^o   ;;;
;;;                       position of the second half by indenting           ;;;
;;;                       it to its old starting position                    ;;;
;;; white_delete          deletes whitespace around point          {esc}\    ;;;
;;;                       leaves one space                                   ;;;
;;;                                                                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                          ;;;
;;; Global Variables: whitespace, init_whitespace                            ;;;
;;;                   makes sure atom whitespace is only blank characters,   ;;;
;;;                   not newline                                            ;;;
;;;                                                                          ;;;
;;; Part of core initialization                                              ;;;
;;;                                                                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; white_delete                                                             ;;;
;;;    Command that deletes the whitespace around point. The function        ;;;
;;;    white_delete$ returns a boolean that indicates whether or not         ;;;
;;;    the operation was successful. The command is bound as {esc}\.         ;;;
;;;                                                                          ;;;
;;;        (white_delete)                                                    ;;;
;;;                                                                          ;;;
;;;        (if (white_delete$) ... )                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom white_delete
     &doc "Deletes space around the cursor"
     (white_delete$))

(defun white_delete$ (&returns boolean)
    (if (or (looked_at " ") (looking_at " "))
        (delete_white_sides)
        (return true)
    else
        (return false)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; leave_one_white                                                          ;;;
;;;   Deletes the whitespace around point.  The command is bound as          ;;;
;;;   {esc}{space}.                                                          ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom leave_one_white
     &doc "Deletes all but one space around cursor"
     (if (white_delete$)
         (insert " ")))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; split_line                                                               ;;;
;;;     Breaks a line, preserving the horizontal position of the second      ;;;
;;;     half by indenting it to its old starting position. This command      ;;;
;;;     is bound as {esc}^o.  This command does not take an argument.        ;;;
;;;                                                                          ;;;
;;;        (split_line)                                                      ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom split_line
     &doc "Breaks line at cursor and indents next"
     &na (&pass count &default 1)
     (let ((indentation (cur_hpos)))
          (do_n_times count
             (cr)
             (indent_line_to_hpos indentation)
             (prev_line_command))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; merge_lines                                                              ;;;
;;;     This command merges two lines together.  With a positive aregument   ;;;
;;;     it merges the current line with the next line.  With a negative      ;;;
;;;     argument, it merges the current line with the previous line.         ;;;
;;;     It is bound as {esc}^.                                               ;;;
;;;                                                                          ;;;
;;;         (merge_lines arg)                                                ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom merge_lines
     &doc "Merges lines together"
     &na (&pass count &default 1)
     (save_excursion
          (if (> count 0)
              (do_n_times count
                  (end_line)
                  (kill_line)
                  (white_delete)
                  (insert " "))
          else
              (do_n_times (- count)
                  (prev_line)
                  (end_line)
                  (kill_line)
                  (white_delete)
                  (insert " ")))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; delete_blank_lines                                                       ;;;
;;;     This comand deletes blank lines.  Because it is often used in        ;;;
;;;     conjunction with an arg open_line, it works downward, rather         ;;;
;;;     than upward if executed while on a non-blank line.  It is bound      ;;;
;;;     as ^x^o.                                                             ;;;
;;;                                                                          ;;;
;;;         (delete_blank_lines)                                             ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom delete_blank_lines
     &doc "Deletes blank lines"
     (save_position
          (if (^ (line_is_blank))    ; you want to execute while point
              (next_line)            ; is on a blank line
              (if (^ (line_is_blank))
                  (prev_line 2)
                  (if (^ (line_is_blank)) return)))
          (do_forever
               (if (buffer_info read_only)
                   (info_message "Attempt to modify read-only buffer.")
                   (ring_the_bell)
                   (stop_doing)
               )
               (if (end_of_buffer_p) (stop_doing))
               (if (line_is_blank)
                   (kill_line)   ; must kill_line for overlay mode
               else
                   (stop_doing)))
          (do_forever
               (if (buffer_info read_only)
                   (stop_doing)
               )
               (prev_line)                         ;delete backward
               (if (beginning_of_buffer_p) (stop_doing))
               (if (line_is_blank)
                   (kill_line)
                else
                   (stop_doing)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; back_to_nonwhite                                                        ;;;
;;;     Command that puts the cursor at the first non-blank character        ;;;
;;;     on a line.  With a positive argument, it puts point on the           ;;;
;;;     first non-blank character on that many lines from the current        ;;;
;;;     line.  With a negative argument, it moves back that many lines       ;;;
;;;     before moving to the first non_white.  This is bound as {esc}m.      ;;;
;;;                                                                          ;;;
;;;         (back_to_nonwhite)                                               ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom back_to_nonwhite
     &doc "Puts cursor at first non blank character on line"
     &na (&pass count &default 1)
     (if (< count 0)                    ; want to treat 0 and 1 the same
         (setq count (1+ count)))
     (setq count (1- count))
     (next_line count)
     (skip_over_white))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; indent_relative                                                          ;;;
;;;    This command lines up the current line with the line                  ;;;
;;;    right above it.  With an argument, it tabs the line                   ;;;
;;;    over that many more tab stops. This is bound as {esc}i.               ;;;
;;;                                                                          ;;;
;;;         (indent_relative arg)                                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom indent_relative
     &doc "Lines up text or tabs text over more"
     &na (&pass count &default 0)
     (indent_relative$ count))

(defun indent_relative$ (&optional (count integer)
                         &local (indentation integer))
     (back_to_nonwhite -1)        ;finds column position of previous
     (setq indentation (cur_hpos))
     (next_line)
     (begin_line)
     (white_delete)                     ; deletes space if there
     (whitespace_to_hpos indentation)   ; indents line
     (if (null count) (setq count 0))
     (if (^= count 0)
         (insert_tab count)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; cret_indent_relative                                                     ;;;
;;;    This command does a carriage return, then indents the line to the     ;;;
;;;    the current position.  If you type an argument, you get that many     ;;;
;;;    more tabs.  This is bound as ^x^j.                                    ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom cret_indent_relative
     &doc "Lines up text or tabs text over more"
     &na (&pass count &default 0)
     (cret_indent_relative$ count))

(defun cret_indent_relative$ (&optional (count integer)
                              &local (indentation integer))
      (save_excursion
        (begin_line)
        (if (line_is_blank)            ; if current = blank no indent wanted
            (setq indentation 0)
         else
            (skip_over_white)
            (setq indentation (cur_hpos))))  ; find current indent
      (cr)
      (indent_line_to_hpos indentation)   ; no add the indent
      (if (null count) (setq count 0))    ; see if there is a tab and them
      (if (^= count 0)                    ; do some more
          (insert_tab count)))
