; LISP.EM, EMACSSRC>EMACS*>EXTENSIONS>SOURCES, EMACS DEVELOPMENT, 04/09/87
; Lisp mode for PEEL and Common Lisp
; Copyright (c) 1986, Prime Computer, Inc., Natick, MA 01760
;                     All Rights Reserved

; Description:
;
; Code is from myself and from SAI
;
; To set the pprinting style for a form execute:
;
; (pp-style form-name first-indent second-indent ... rest-indent)
;
; for example (pp-style do 0 2 0) tells the pretty printer to indent
; the first form of the do (the var init step) to the indentation level.
; the second form of the do (the test) in 2 spaces and all the other forms
; of the do to the indentation level.  Note that the default is the equivalent
; of (pp-style name 0) and this results in the following kind of pprint:
;
; (random-thing
;    form1
;    form2
;    )
;
; whereas the do example:
;
; (do ((i 1 (1+ i)))
;     ((> i 100) (return))
;   form1
;   form2
;   )
;
; the initialization will not reset the style.  pp-style installs the list
; of indentations on the indentation property of the symbols plist.  This
; only happens if there isn't one there already.  Use remove-style to
; take off the property for redefinition.
;
; Modifications:
;   Date   Programmer     Description of modification
; 04/09/87 Rand           Wrote new pretty printer
; 10/10/86 Rand           Fixed balfor
; 07/15/86 Rand           Added Typin mode to LISP mode
; 06/02/86 Rand           Fixing balbak
; 01/30/86 Rand           Initial coding.

(defun setq_default (&quote atom &eval value &returns any)
       (if (null (eval atom))
           (set atom value))
       (return value))

(defcom lisp_on
        &doc "Lisp: Set lisp mode"
        (turn_mode_on (find_mode 'lisp) first)
        (buffer_info (user language_name$) "lisp")
        (if (null lisp_mode_initialized$)
            (initialize_lisp_mode))
        )

(defcom lisp_off
        &doc "Lisp: Turn off lisp mode"
        (turn_mode_off (find_mode 'lisp)))

;;;; Balfor reads one sexp

(defcom balfor
       &doc "Lisp: Balance the current paren or move forward an sexp"
       (if (^ (looking_at "(")) (search_fd "("))
       (setq parse_list (lisp_reader))
       (go_to_cursor (car (car (reverse parse_list)))))

(defcom close_paren
        &doc "Lisp: Flashes matching open parenthesis (iff on screen)"
        (lisp_close_parenf))

(defun lisp_close_parenf (&local s l t b)
       (insert ")")                    ; insert the close parenthesis
       (if (have_input_p) (return))    ; ignore if type ahead anyway
       (setq l (copy_cursor current_cursor))
       (setq t (window_info top_line_cursor))
       (with_no_redisplay
          (save_excursion
             (if (not (balbak_f))
                 (ring_the_bell)
                 (info_message "Unbalanced parenthesis")
                 (return))
             (setq b (copy_cursor current_cursor))
             (forward_char)
             (setq s (copy_cursor current_cursor))
             (forward_word)
             (setq s (range_to_string s current_cursor))
             ))
       ;; in range?
       (if (>= b t)
           (go_to_cursor b)
           (redisplay)
           (sleep_for_n_milliseconds lisp.paren_time)
           (go_to_cursor l)
           else
           (info_message (catenate "Closed: " s))
           )
       )

;;;; Balbak balances one set of parens

(defcom balbak
        &doc "Lisp: balance lisp parenthesis"
        (balbak_f))

(defun balbak_f () (lisp_balance_back_f 1))

;;;;
;;;; Start of Pretty printer code
;;;;

(defcom lisp_pp_command
        &doc "Lisp: Pretty prints a region of lisp text"
        (let ((left_margin$ (list (cur_hpos)))
              (parse_list (lisp_reader))
              (lisp_pp$_0 lisp_indent_amount$)
              (lisp_pp$_1 lisp.comment_column)
              (item-indentation '(0))
              (ln (line_number current_cursor))
              (first false)
              (i 0)
              (indent-list nil))
             (do_forever
                (setq item (car parse_list))
                (setq parse_list (cdr parse_list))
                (if (null item) (stop_doing))
                (go_to_cursor (car item))
                (select (cdr item)
                        atom (do_atom)
                        paren (do_paren)
                        thesis (do_thesis)
                        comment1 (if (^= 1 (cur_hpos)) (pp (car item) "m#1"))
                        comment2 (pp (car item) "s0")
                        comment3 (pp (car item) "b")
                        comment4 (pp (car item) "b")
                        else (if (^= ln (line_number current_cursor))
                            (setq ln (line_number current_cursor))
                            (pp (car item) "s-#0")
                            )
                        otherwise (do_standard_indent)
                        )
                )
             )
        )

(defun do_atom ()
       (if first
           (setq name (rest_of_line))
           (if (< 0 (setq i (index name " ")))
               (setq name (substr name 1 (1- i))))
           (setq name (intern name))
           (setq item-indentation
                 (get name 'indentation))
           (if (null item-indentation)
               (setq item-indentation
                     '(0))
               )
           (setq first false)
           else
           (do_standard_indent)
           )
       )

(defun do_paren ()
       (redisplay)
       (do_standard_indent)
       ;; push current indent list on saver
       (setq indent-list (cons item-indentation indent-list))
       (if_at "(("
              (pp (car item) "*+1")
              (setq item-indentation '(0))
              else
              (pp (car item) "*+#0")
              (setq item-indentation (list lisp_indent_amount$)))
       (setq first true)
       )

(defun do_thesis ()
       ;; pop indent-list
       (setq item-indentation (car indent-list))
       (setq indent-list (cdr indent-list))
       ;;
       (if (^= ln (line_number current_cursor))
           (setq ln (line_number current_cursor))
           (pp (car item) "s0^")
           else
           (pp (car item) "^")
           )
       )

(defun do_standard_indent ()
       (setq lisp_pp$_2 (car item-indentation))
       (if (numberp (car (cdr item-indentation)))
           (setq item-indentation (cdr item-indentation)))
       (if (^= ln (line_number current_cursor))
           (setq ln (line_number current_cursor))
           (pp (car item) "s#2")))

(defun pp-style (&quote symbol &rest list)
       (if (null (get symbol 'indentation))
           (putprop symbol list 'indentation)
           ))

(defun remove-style (&quote symbol)
       (remprop symbol 'indentation))

(defun init-pp-styles ()
       (pp-style dolist 2 0)
       (pp-style dotimes 2 0)
       (pp-style do 0 2 0)
       (pp-style do* 0 2 0)
       (pp-style defun 5 5 0)
       (pp-style defcom 6 2 0)
       (pp-style defmethod 9 9 0)
       (pp-style defmacro 8 9 0)
       (pp-style defclass 8 0 2)
       (pp-style defflavor 9 4 4)
       (pp-style when 2 0)
       (pp-style unless 2 0)
       (pp-style prog 4)
       (pp-style progn 5)
       (pp-style prog1 5)
       (pp-style prog2 5)
       (pp-style values 6)
       (pp-style list 4)
       (pp-style if 2 0)
       (pp-style return-from 3 0)
       (pp-style unwind-protect 2 0)
       (pp-style and 3)
       (pp-style or 2)
       (pp-style let 2 0)
       (pp-style let* 2 0)
       (pp-style flet 2 0)
       (pp-style labels 2 0)
       (pp-style macrolet 2 0)
       (pp-style setq 4)
       (pp-style psetq 5)
       (pp-style setf 4)
       (pp-style set 3)
       )

;; (defun pp ((cursor cursor) (comstring string) &returns any)
;;    (with_command_abort_handler
;;       (epf_defun pp "CL$RTPPRINT" "*" "ks" "")
;;     command_abort_handler
;;       (init_local_displays "Can't find external routine")
;;       (info_message "See system administrator: probable problem is bad ENTRY$ search rules")
;;       )
;;    (return (pp cursor comstring)))

(defun lisp_reader (&local
          (start_cur cursor)
          (end_cur cursor)
          (p_list list)
          &returns list)
       (with_no_redisplay
          (setq start_cur (copy_cursor current_cursor))
          (move_bottom)
          (setq end_cur (copy_cursor current_cursor))
          (setq p_list
                ((fsymeval (intern (buffer_info (user language_name$))))
                 start_cur end_cur))
          (go_to_cursor start_cur))
       (return (sort_list p_list))
       )

;;;;
;;;; End of pretty printer code
;;;;

(defcom lisp_comment
        &doc "Lisp: Move to lisp comment column"
;;        (lisp_start_of_comment)
;;        (if_at "; " (forward_char 2) (return))
        (end_line)
        (whitespace_to_hpos (1- lisp.comment_column))
        (insert " ; ")
        (return))

(defun begin_defun_f ()                ; go to beginning of this
                                       ; actually  -- go to paren in colum
       (if (reverse_search "~n(")      ; is it easy? )
           (forward_char)              ; back to the paren
           (return))
       (move_top)                      ; must be near the beginning
       (return))                       ; must be here or else we lose

(defun end_defun_f ()                  ; go to beginning of this defun
                                       ; actually  -- go to paren in column 1
       (if_at "("
              (if (= (cur_hpos) 1)
                  (forward_char)))     ; stay in this function
       (begin_defun_f)
       (balfor)                        ; must be near the beginning
       (return))                       ; must be here or else we lose

(defcom begin_defun
        &doc "Lisp: Go to the beginning of the current defun or whatever"
        (begin_defun_f))

(defcom end_defun
        &doc "Lisp: Go to the end of the current defun or whatever"
        (end_defun_f))

(defcom move_defun_to_screen_top
        &doc "Lisp: Move defun to screen top"
        (with_cursor here
                     (begin_defun_f)
                     (window_info top_line_cursor current_cursor)
                     (go_to_cursor here)))

(defcom pl_current_expression
   &doc "Lisp: PL's current defun or defcom"
   (mark_defun)
   (exchange_mark)
   (print (catenate "PL: " (get_pname (eval (read)))))
)

(defcom mark_defun
        &doc "Lisp: Put point and mark around defun"
        (begin_defun_f)
        (mark)
        (balfor)
        (forward_char)
        )

(defcom lisp_cr
        &doc "Lisp: Cret to correct indentation for program"
        (cr)
        (lisp_indent))

(defcom lisp_indent
        &doc "Lisp: indent to correct indentation for program"
        (let ((indent_column 1))
             (insert ")")
             (with_no_redisplay
                (save_position
                   (if (balbak_f)
                       (if_at "(("
                         (setq indent_column (1+ (cur_hpos)))
                         else
                         (setq indent_column (+ lisp_indent_amount$ (cur_hpos))))
                       )))
             (rubout_char)
             (indent_line_to_hpos indent_column)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;  Lisp key bindings                                     ;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defun setup_lisp_dispatch ()
       (dispatch_info lisp_mode     \~cj 'lisp_cr)
       (dispatch_info lisp_esc_mode \~ca 'begin_defun)
       (dispatch_info lisp_esc_mode \~ce 'end_defun)
       (dispatch_info lisp_esc_mode \e 'pl_current_expression)
       (dispatch_info lisp_esc_mode \~cf 'balfor)
       (dispatch_info lisp_esc_mode \h 'mark_defun)
       (dispatch_info lisp_esc_mode \i  'lisp_indent)
       (dispatch_info lisp_esc_mode \~cr 'move_defun_to_screen_top)
       (dispatch_info lisp_esc_mode \~cb 'balbak)
       (dispatch_info lisp_esc_mode \~cj 'lisp_cr)
       (dispatch_info lisp_esc_mode \~hbb 'lisp_comment) ; Semicolon
       (dispatch_info lisp_mode lisp.close_parenthesis 'close_paren)
       (dispatch_info lisp_esc_mode \q 'lisp_pp_command))

(defcom initialize_lisp_mode
        (setq lisp_mode_initialized$ true)
        (with_command_abort_handler
           (epf_defun lisp "CL$RTLISP" "l" "kk" "")
         command_abort_handler
           (init_local_displays "Can't find external routine")
           (info_message "See system administrator: probable problem is bad ENTRY$ search rules")
           )
        (setq_default lisp_indent_amount$ 2)
        (setq_default lisp_mode (find_mode 'lisp))
        (setq lisp_esc_mode (dispatch_info lisp_mode \~h9b))
        (if (null lisp_esc_mode)
            (dispatch_info lisp_mode \~h9b
                           (setq lisp_esc_mode
                                 (find_mode 'lisp_esc_mode))))

        (setq_default lisp.open_parenthesis "(")
        (setq_default lisp.close_parenthesis ")")
        (if (and (not (null lisp_comment_column))
                 (null lisp.comment_column))
            (setq lisp.comment_column lisp_comment_column))
        (setq_default lisp.comment_column 40)
        (setq_default lisp.paren_time 750); milliseconds
        (init-pp-styles)
        (setup_lisp_dispatch)
        )
