; TAB.EM, EMACS*>EXTENSIONS>SOURCES, ENVIRONMENTS GROUP, 03/26/85
; PART ONE OF THE EMACS SUPPLEMENTAL TAB LIBRARY IMPLEMENTATION
; COPYRIGHT (C) 1985, Prime computer, inc., Natick, Ma 01760
;
; Description:
;   This file contains a library of tabbing and tab related functions
;   for EMACS.
;
; Abnormal conditions:
;
; Implementation:
;        This file contains many of the base functions used in implementing
;   tabs in EMACS. This particular implementation allows for the user to
;   have many different sets of tabs in existence at once. The tab stops that
;   can be accessed from any buffer are hereby referred to as Global tabs.
;   The tab stops that are set specifically for the buffer the user is presently
;   using are called Local tabs.
;        The Global tab stops are stored in the global variable tab_array$.
;   The last tab stop for the Global tabs is stored in the global variable
;   last_tab$.
;        The Local tabs stops for each buffer are stored in the global variable
;   (buffer_info (user tab_array)). The last tab stop for the Local tabs is
;   likewise stored in (buffer_info (user last_tab)).
;        The tab stop arrays are declared as (make_array 'integer 141). {This
;   rather uneven number was chosen to avoid the problems associated with
;   PEEL's rather nasty habit of starting array subscripts at 0 instead of 1.}
;   Of course, the last tab variables are also integers.
;        The global variable which tells the tab routines which set of tab
;   stops to use is global_tab_on$; if it is true, use the Global tabs, else
;   use the Local tabs. Should the variable global_tabs_on$ be "nulled", the
;   tab routines should restore it to true via the functions check_set_tab_vars
;   and check_tab_vars, which are stored in this file.
;        If the user happens to be using a language mode that sets up tabs,
;   they will and up using Local tabs regardless of the value of
;   global_tabs_on$. This is done so that the tab stops that the particular
;   language mode uses don't harass the user as he/she moves from one buffer
;   to another.
;
; Modifications:
;   Date   Programmer   Description of modification
; 03/26/85 Sarkisian    Added special handling of buffers in a language mode
;                       that expects tabs to be used.
; 02/15/85 Sarkisian    Added correct documentation.
; 02/12/85 Sarkisian    Initial coding.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; C/F       Name                     Description                        ;;;
;;; ---       ---                      -----------                        ;;;
;;; F         tab_langauage_mode       returns true if the current buffer ;;;
;;;                                    is in a language mode that uses    ;;;
;;;                                    tabs; returns false otherwise.     ;;;
;;;                                                                       ;;;
;;; F         check_set_tab_vars       makes certain global_tabs_on$ is set;;
;;;                                    as well as approriate tab variables;;;
;;;                                                                       ;;;
;;; F         check_tab_vars           makes certain global_tabs_on$ is set;;
;;;                                    but only makes certain that the    ;;;
;;;                                    appropriate tab variables exist!   ;;;
;;;                             SHOULD ONLY TO BE USED WITH ROUTINES THAT ;;;
;;;                                    DEFINITELY ALTER TABS!             ;;;
;;;                                                                       ;;;
;;; C         global_tabs              sets global_tabs_on$ true, which   ;;;
;;;                                    activates global tabs.             ;;;
;;;                                                                       ;;;
;;; C         local_tabs               sets global_tabs_on$ false, which  ;;;
;;;                                    activates local tabs.              ;;;
;;;                                                                       ;;;
;;; C         which_tabs               informs users which set of tabs is ;;;
;;;                                    presently in use.                  ;;;
;;;                                                                       ;;;
;;; C         default_tabs             resets tabs that are presently in  ;;;
;;;                                    use to default settings. defaults  ;;;
;;;                                    are tab stops every 5th column,    ;;;
;;;                                    up to column 140.                  ;;;
;;;                                                                       ;;;
;;; C         type_tab                 moves point to next tab stop.      ;;;
;;;                                                                       ;;;
;;; F         type_tabf                does the work for type_tab.        ;;;
;;;                                                                       ;;;
;;; C         insert_tab               inserts whitespace from point to   ;;;
;;;                                    next tab stop.                     ;;;
;;;                                                                       ;;;
;;; F         insert_tabf              does the work for insert_tab.      ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; tab_language_mode                                                     ;;;
;;;               function that returns true if the current buffer is in  ;;;
;;;           a language mode that uses tabs.                             ;;;
;;;                                                                       ;;;
;;;           (tab_language_mode)                                         ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun tab_language_mode (&returns boolean)
    (if (| (member (find_mode 'fortran) (buffer_info modes))
           (member (find_mode 'cobol) (buffer_info modes)))
        (return true)
    else
        (return false)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; check_set_tab_vars                                                    ;;;
;;;                Makes certain that the global variable global_tabs_on$ ;;;
;;;                exists and checks for the existence of either          ;;;
;;;                     o tab_array$ and last_tab$ - global tabs data     ;;;
;;;                     o (buffer_info (user tab_array))                  ;;;
;;;                       (buffer_info (user last_tab)) - local buffer    ;;;
;;;                                                       tab data.       ;;;
;;;                If either of the above is null, it is then created and ;;;
;;;                initialized.                                           ;;;
;;;                Will make certain that the last tab variables exist and;;;
;;;                are set.                                               ;;;
;;;                                                                       ;;;
;;;                                                                       ;;;
;;;           (check_set_tab_vars)                                        ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun check_set_tab_vars ()

    (if (null global_tabs_on$)
        (setq global_tabs_on$ true))   ; assume global tabs are desired

    (if (& global_tabs_on$ (^ (tab_language_mode)))
                                       ; make certain global tab variables are
                                       ; existent and set properly
        (if (null last_tab$)
            (setq last_tab$ 1))
        (if (null tab_array$)
            (default_tabs)        ; calls check_tab_vars which will set
                                  ; up tab_array$. will initialize tabs to
                                  ; defaults.
            (sleep_for_n_milliseconds 1000)) ; let user read message from
                                             ; previous function.

    else                          ; make certain local tab variables are
                                  ; existent and set properly
         (if (null (buffer_info (user last_tab)))
             (buffer_info (user last_tab) 1))
         (if (null (buffer_info (user tab_array)))
             (default_tabs)            ; see note to above call to default_tabs.
             (sleep_for_n_milliseconds 1000))))    ; let user read message from
                                                   ; previous function.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; check_tab_vars      Function that makes certain global_tabs_on$ is set;;;
;;;                     and makes certain that the appropriate tab        ;;;
;;;                     variables exist. DOES NOT SET THE TAB VARIALES.   ;;;
;;;                     If this routine is called by modules that expect  ;;;
;;;                     the tab variables to be initialized with default  ;;;
;;;                     values or that may leave the variables unset,     ;;;
;;;                     expect disasterous results.                       ;;;
;;;                 NOTE: If global_tabs_on$ is nill, it will be set to   ;;;
;;;                       true by this routine!                           ;;;
;;;                                                                       ;;;
;;;           (check_tab_vars)                                            ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun check_tab_vars ()

    (if (null global_tabs_on$)
        (setq global_tabs_on$ true))

    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (if (null last_tab$)
            (setq last_tab$ 1))
        (if (null tab_array$) (setq tab_array$ (make_array 'integer 141)))
    else
        (if (null (buffer_info (user last_tab)))
            (buffer_info (user last_tab) 1))
        (if (null (buffer_info (user tab_array)))
            (buffer_info (user tab_array) (make_array 'integer 141)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; global_tabs    Command that sets the variable global_tabs_on$ to true ;;;
;;;                and informs the user that global tabs are in effect.   ;;;
;;;                                                                       ;;;
;;;                (global_tabs)                                          ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom global_tabs
    &doc "Activates global tabs"
    (if (tab_language_mode)   ; don't do anything...inform user what's going on
        (info_message "Language mode set that uses local tabs.")
        (ring_the_bell)
        (sleep_for_n_milliseconds 1000)
        (info_message "Local tabs are still in effect.")
    else
        (setq global_tabs_on$ true)
        (check_set_tab_vars)       ; make certain all tab variables are there.
        (info_message "Global tabs are now in effect")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; local_tabs     Command that sets the variable global_tabs_on$ to false;;;
;;;                and informs the user that local tabs are in effect.    ;;;
;;;                                                                       ;;;
;;;                (local_tabs)                                           ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom local_tabs
    &doc "Activates local (to buffer) tabs"
    (if (tab_language_mode)   ; don't do anything...local tabs are in use for
                              ; buffer in particular language mode...if user
                              ; leaves buffer or cancels mode, will revert to
                              ; previous tabs usage, local or global.
        (info_message "Local tabs in use for present language mode.")
    else
        (setq global_tabs_on$ false)  ;done to make certain that check_set_tab_vars
                                      ; will set the local tab variables.
        (check_set_tab_vars)          ; make certain all tab variables are there.
        (info_message "Local tabs are now in effect")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; which_tabs          Command that informs the user which set of tabs   ;;;
;;;                     are currently in effect (global or local).        ;;;
;;;                                                                       ;;;
;;;                     (which_tabs)                                      ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom which_tabs
    &doc "Tells which tabs are in use"
    (check_set_tab_vars)         ; make certain that all tab variables are there.
    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (info_message "Global tabs are in effect")
    else
        (info_message "Local tabs are in effect")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; default_tabs   Command that resets tab stops to every 5th column.     ;;;
;;;                See the function default_tabsf.                        ;;;
;;;                                                                       ;;;
;;;                (default_tabs)                                         ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom default_tabs
    &doc "Sets up default tabs every 5 spaces"
    (default_tabsf))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; default_tabsf  Function that resets tab stops to every 5th column.    ;;;
;;;                Will reset global tabs (tab_array$, last_tab$) if      ;;;
;;;                global tabs are in effect (global_tabs_on$ = true and  ;;;
;;;                (not (tab_language_mode)), else...                     ;;;
;;;                it will reset local tabs (buffer_tabs (user tab_array));;;
;;;                                         (buffer_tabs (user last_tab)) ;;;
;;;                if local tabs are in effect (global_tabs_on$ = false   ;;;
;;;                or (tab_language_mode)                                 ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun tablist_to_array ((list list))
       (if (null tab_array$) (setq tab_array$ (make_array 'integer 139)))
       (tablist_to_array$ list tab_array$ last_tab$))

(defun default_tabsf (&local (dummy_var integer))

    (check_tab_vars)         ; make certain tab variables are there.
    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (info_message "Setting default tabs (global)")
        (tablist_to_array$ '(5 10 15 20 25 30 35 40 45 50 55 60 65 70 75 80 85
                            90 95 100 105 110 115 120 125 130 135 140)
                            tab_array$ last_tab$)
    else
        (info_message "Setting default tabs (local to buffer)")
        (tablist_to_array$ '(5 10 15 20 25 30 35 40 45 50 55 60 65 70 75 80 85
                            90 95 100 105 110 115 120 125 130 135 140)
                           (buffer_info (user tab_array))
                           dummy_var)
        (buffer_info (user last_tab) dummy_var)) ; done in this manner due to
                                                 ; the fact that the routine
                                                 ; tablist_to_array$ expects an
                                                 ; atom as the last arg.
    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (info_message "Default tabs set (global)")
    else
        (info_message "Default tabs set (local_to_buffer)")))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; type_tab       Command that moves point to next tab stop.             ;;;
;;;                Will not attempt to insert whitespace in a read-only   ;;;
;;;                buffer (see type_tabf).                                ;;;
;;;                                                                       ;;;
;;;           (type_tab optional-integer)                                 ;;;
;;;                                                                       ;;;
;;;       where the optional integer can be either positive or negative.  ;;;
;;;      If it is negative, the back_tab routine is called.               ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom type_tab
    &doc "Tab"
    &na (&pass count &default 1)
    (check_set_tab_vars)   ; make certain that all tab variables exist and
                           ; are initialized.
    (if (>= count 0)
        (do_n_times count (type_tabf))
    else
        (do_n_times (- count) (back_tabf)))) ; negative tabbing is treated as
                                             ; back tabbing!

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; type_tabf      Function that does the actual work for type_tab.       ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun type_tabf (&local cur_pos
                        array_name
                        last_tab_name)
;
;   The local variables array_name and last_tab_name are used for
; making this routine more generic...it must do the same things using
; either local or global tabs, so simply set some "pointers" to the
; proper set of data and take off!
;
    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (setq array_name 'tab_array$)
        (setq last_tab_name 'last_tab$)
    else
         (setq array_name '(buffer_info (user tab_array)))
         (setq last_tab_name '(buffer_info (user last_tab))))

    (if (>= (cur_hpos) (eval last_tab_name))     ; user is past last tab stop
        (info_message "Attempted to tab past last tab stop of "
                      (integer_to_string (eval last_tab_name)))
        (ring_the_bell)
        (return))
                             ; get column should be going to...
    (setq cur_pos (aref (eval array_name) (cur_hpos)))
                             ; if got to column, return...
    (if (go_to_hpos cur_pos) (return))
                             ; didn't make it, so...
    (end_line)
    (if (buffer_info read_only)
        (info_message "You have reached the end of line (read_only buffer)")
        (return))                           ; NEVER attempt to write to a
                                            ; read_only buffer.

    (whitespace_to_hpos cur_pos))           ; all is well, insert whitespace at
                                            ; end of the line.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; insert_tab     Command that inserts whitespace from point to the next ;;;
;;;                tab stop. Does not attempt to modify a read-only buffer;;;
;;;                                                                       ;;;
;;;                (insert_tab optional-integer)                          ;;;
;;;                                                                       ;;;
;;;           where the optional-integer is a positive value for how many ;;;
;;;          tabs to insert. If it is <= to 0, this routine does nothing. ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom insert_tab
    &doc "Inserts spaces to next tab stop"
    &na (&pass count &default 1)
    (if (buffer_info read_only) (return))   ; NEVER attempt to modify a
                                            ; read-only buffer.
    (if (<= count 0) (return))              ; can't insert null or backwards!
    (check_set_tab_vars)        ; make certain that all tab variables exist
                                ; and are initialized.
    (insert_tabf count))                    ; fufill the user's request

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; insert_tabf    Function that really does the work for insert_tab.     ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun insert_tabf (       (count integer)
                    &local array_name
                           last_tab_name )


; Determine which set of tabs to use and store their names in local variables.
;
    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (setq array_name 'tab_array$)
        (setq last_tab_name 'last_tab$)
    else
        (setq array_name '(buffer_info (user tab_array)))
        (setq last_tab_name '(buffer_info (user last_tab))))

    (do_n_times count
         (if (>= (cur_hpos) (eval last_tab_name))
             (info_message "Attempted to insert tab past last tab stop of "
                           (integer_to_string (eval last_tab_name)))
             (ring_the_bell)
             (return))
                                  ; fill in the whitespace for them...
         (whitespace_to_hpos (aref (eval array_name) (cur_hpos)))))
