; TAB1.EM, EMACS*>EXTENSIONS>SOURCES, EMACS DEVELEOPMENT, 12/04/86
; Second part of the EMACS Tab supplemental library implementation
; COPYRIGHT (C) 1985, Prime computer, inc., Natick, Ma 01760
;
; Description:
;        This file contains a library of tabbing and tab related functions
;   for EMACS.
;
; Abnormal conditions:
;
; Implementation:
;        This file contains many of the base functions used in implementing
;   tabs in EMACS. This particular implementation allows for the user to
;   have many different sets of tabs in existence at once. The tab stops that
;   can be accessed from any buffer are hereby referred to as Global tabs.
;   The tab stops that are set specifically for the buffer the user is presently
;   using are called Local tabs.
;        The Global tab stops are stored in the global variable tab_array$.
;   The last tab stop for the Global tabs is stored in the global variable
;   last_tab$.
;        The Local tabs stops for each buffer are stored in the global variable
;   (buffer_info (user tab_array)). The last tab stop for the Local tabs is
;   likewise stored in (buffer_info (user last_tab)).
;        The tab stop arrays are declared as (make_array 'integer 141). {This
;   rather uneven number was chosen to avoid the problems associated with
;   PEEL's rather nasty habit of starting array subscripts at 0 instead of 1.}
;   Of course, the last tab variables are also integers.
;        The global variable which tells the tab routines which set of tab
;   stops to use is global_tab_on$; if it is true, use the Global tabs, else
;   use the Local tabs. Should the variable global_tabs_on$ be "nulled", the
;   tab routines should restore it to true via the function check_set_tab_vars.
;        If the user happens to be using a language mode that sets up tabs,
;   they will and up using Local tabs regardless of the value of
;   global_tabs_on$. This is done so that the tab stops that the particular
;   language mode uses don't harass the user as he/she moves from one buffer
;   to another.
;
; Related files: TAB.EM
;
;
; Modifications:
;   Date   Programmer   Description of modification
; 12/04/86 Kingsbury    Changed backtab to check for <= 1 rather than 1
; 03/27/85 Sarkisian    Fixed settabf to check for bad tab interval.
; 03/26/85 Sarkisian    Added handling of buffers in language mode(s) that
;                       expect to use tabs.
; 02/20/85 Sarkisian    Initial coding.
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; C/F       Name                     Description                        ;;;
;;; ---       ----                     -----------                        ;;;
;;; F         tablist_to_array$         takes a list of integers(tab stops);;;
;;;                                    and places the appropriate values  ;;;
;;;                                    into the appropriate tab_array.    ;;;
;;;                                    NOW IMPLEMENTED IN SPL.            ;;;
;;;                                                                       ;;;
;;; C         back_tab                 moves point to previous tab stop.  ;;;
;;;                                                                       ;;;
;;; F         back_tabf                does the work for back_tab.        ;;;
;;;                                                                       ;;;
;;; C         settab,set_tab,                                             ;;;
;;;           set_tabs, settabs        synonymous commands that allow the ;;;
;;;                                    user to choose tab stops.          ;;;
;;;                                                                       ;;;
;;; F         settabf                  does the work for the above commands;;
;;;                                                                       ;;;
;;; F         paint_tab_menu           displays the tab ruler intructions.;;;
;;;                                                                       ;;;
;;; F         move                     handles the user interaction while ;;;
;;;                                    in the tab ruler.                  ;;;
;;;                                                                       ;;;
;;; C         settabs_from_table,                                         ;;;
;;;           setft                    synonymous commands that look at a ;;;
;;;                                    line of text and sets tab stops    ;;;
;;;                                    based on the column positions of   ;;;
;;;                                    words.                             ;;;
;;;                                                                       ;;;
;;; F         settabs_from_tablef      does the work for the above commands;;
;;;                                                                       ;;;
;;; C         tablist                  accepts a series of numbers from   ;;;
;;;                                    the terminal and converts them into;;;
;;;                                    tab stops.                         ;;;
;;;                                                                       ;;;
;;; F         convert_tabs             does the work for tablist.         ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;
;
;
;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; back_tab       Command that moves point to previous tab stop.         ;;;
;;;                                                                       ;;;
;;;           (back_tab optional-integer)                                 ;;;
;;;                                                                       ;;;
;;;      where optional-integer is how many tab stops to move back to.    ;;;
;;;     If it is negative, the command is interpreted is a normal tab.    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom back_tab
    &doc "Moves cursor to previous tab stop"
    &na (&pass count &default 1)
    (check_set_tab_vars)            ; make certain that tab variables are set.
    (if (> count 0)
        (do_n_times count (back_tabf))
    else
        (do_n_times (- count) (type_tabf) ))) ; treat as a normal tab.


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; back_tabf      Function that does the work for back_tab.              ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun back_tabf (&local current_tab
                         down_counter
                         array_name
                         last_tab_name)

    (if (= (cur_hpos) 1) (return))     ; cannot backtab before beginning of
                                       ; line
; Determine which tabs to use...
    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (setq array_name 'tab_array$)
        (setq last_tab_name 'last_tab$)
    else
        (setq array_name '(buffer_info (user tab_array)))
        (setq last_tab_name '(buffer_info (user last_tab))))

    (if (> (cur_hpos) (eval last_tab_name)) ; that's easy..go to last valid
        (go_to_hpos (eval last_tab_name))   ; tab stop!
        (return))

    (setq current_tab (aref (eval array_name) (1- (cur_hpos))))
    (setq down_counter (1- (cur_hpos)))
    ;
    ; determine which tab stop is before current position and go there.
    ;
    (do_forever
         (if (= (aref (eval array_name) down_counter) current_tab)
             (setq down_counter (1- down_counter))
             (if (<= down_counter 1)    ; at beginning of array...no more stops.
                 (begin_line)
                 (return))
         else                           ; found it! go to that tab position.
             (go_to_hpos (aref (eval array_name) down_counter))
             (return))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; settab, set_tab, set_tabs, settabs                                    ;;;
;;;                                                                       ;;;
;;;           All equivalent commands that allow the user to set tabs to  ;;;
;;;      whatever he/she wants.                                           ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom settab
    &doc "Set tabs to what you want"
    (settabf))

(defcom set_tab
    &doc "Set tabs to what you want"
    (settabf))

(defcom settabs
    &doc "Set tabs to what you want"
    (settabf))

(defcom set_tabs
    &doc "Set tabs to what you want"
    (settabf))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; paint_tab_menu     Function that displays the tab ruler instructions. ;;;
;;;                   Fits in a small window.                             ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun paint_tab_menu ()
    (insert "~nspace    enter a space and move forward~n")
    (insert "t        enter a T--signals a tab stop~n")
    (insert "b,f      move Backward or Forward a column~n")
    (insert "q,RETURN Quit and set these tab stops~n")
    (insert "CTRL-G   Leave the tab stops as they were before this ruler~n")
    (insert "Horizontal scrolling is done automatically for you.~n"))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; settabf        Function that does the actual work for the commands    ;;;
;;;           settab, settabs, set_tab, set_tabs.                         ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun settabf (&local counter
                       start_hcol
                       interval
                       tablist
                       dummy_last_tab
                       tab_language_mode_val)

    (setq start_hcol (window_info column_offset))     ; keep track of where we
                                                      ; are in case user aborts
                                                      ; settab function.
    ; find out if the current buffer is using a language mode that sets up
    ; tabs. Done here to avoid the problem of switching buffers and not being
    ; able to see the modes in the present buffer.

    (setq tab_language_mode_val (tab_language_mode))

    (with_command_abort_handler   ; Lets user abort setting new tabs,
                                  ; keeping old ones intact.
        (save_position
             (go_to_cursor (find_buffer ".tab"))
             (buffer_info dont_show true)
             (mark_whole)
             (delete_region)     ; if this isn't 1st time setting tabs, get
                                 ; rid of previous garbage in .tab buffer.
             (setq counter 1)

             (do_n_times 14                ; set up tabs ruler.
                  (insert "         ")     ; nine spaces.
                  (insert (integer_to_string (modulo counter 10)))
                  (setq counter (1+ counter)))
             (cr)
             (do_n_times 14
                  (insert "....5....0"))
             (cr)

             (if (yesno "Is there a default interval")
                 (do_forever
                     (setq interval (1- (string_to_integer
                                         (prompt "How far apart"))))
                     (if (| (> interval 140) (< interval 0))
                         (info_message "Invalid interval supplied.")
                         (ring_the_bell)
                         (sleep_for_n_milliseconds 1000)
                     else (stop_doing)))

                 (do_forever
                    (if (> (+ interval (cur_hpos)) 140) (stop_doing))
                    (self_insert " " interval)       ; put in interval spaces.
                    (insert "T"))
                 (whitespace_to_hpos 140)   ; fill up any spaces left.
             else
                (self_insert " " 79)
                (insert "T")           ; give user single tab stop in col. 80
                (self_insert " " 60))
              (begin_line)        ; output tabs menu.
              (save_position
                (with_no_redisplay
                  (move_bottom)
                  (insert "~n")
                  (paint_tab_menu)))          ;give user the real instructions.

              (do_forever
                 (if (^ (move)) (stop_doing))) ; get user input until done.

              (unmodify)
              (check_tab_vars)              ;make certain that tab variables
                                            ; exist.
              (if (& global_tabs_on$ (^ tab_language_mode_val))
                  (info_message "Now setting these tab stops...(global)")
              else
                  (info_message "Now setting these tab stops..."
                                "(local to buffer)"))

              ; convert the tab ruler into a tab list.
              ;
              (begin_line)
              (setq tablist ())
              (do_forever
                   (if (search_fd_in_line "T")
                       (setq tablist (append tablist (list (cur_hpos))))
                       (if (^ (eolp)) (forward_char) ; don't want same "T".
                       else  (stop_doing))
                   else
                       (stop_doing))))  ; no more tab stops.

          (if (& global_tabs_on$ (^ tab_language_mode_val))
              (tablist_to_array$ tablist tab_array$ last_tab$)
          else
              (tablist_to_array$ tablist (buffer_info (user tab_array))
                                         dummy_last_tab)
              (buffer_info (user last_tab) dummy_last_tab)) ; done in this
                                                     ; manner due to the fact
                                                     ; that the routine
                                                     ; tablist_to_array$ expects
                                                     ; an atom as its last arg.

          (hcol start_hcol)             ; reset window to starting point
                                        ; of this window.
          (if (& global_tabs_on$ (^ tab_language_mode_val))
              (info_message "Tabs set (global)")
          else
              (info_message "Tabs set (local_to_buffer)"))

  command_abort_handler                 ; user wants to stop this session.

         (hcol start_hcol)             ; reset window to starting point of
                                       ; this window.
         (info_message "Abandoned that tab ruler. Your previous tab stops "
                       "are still in effect.")))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; move      Function that handles user interaction while in the tab     ;;;
;;;           ruler. Returns true unless user types a q, Q or RETURN, in  ;;;
;;;           which case it reutrns false.                                ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun move (&local movement
             &returns boolean)

    (info_message "Type a space, t, b, f, q, RETURN, or CTRL-G. ")
    (setq movement (char_to_string (read_character)))
    (select movement

         " "       (if (^ (eolp))
                       (delete_char)
                       (insert " ")
                       (if (= 81 (cur_hpos))
                           (if (^= (window_info column_offset) 61)
                               (hcol 61)))
                   else
                       (ring_the_bell)
                       (info_message "Sorry, you can't space forward "
                                     "past here.")
                       (flush_typeahead)
                       (sleep_for_n_milliseconds 1000))
                   (return true)

         "t"
         "T"       (if (^ (eolp))
                       (delete_char)
                       (insert "T")
                       (if (= 81 (cur_hpos))
                           (if (^= (window_info column_offset) 61)
                               (hcol 61)))
                   else
                       (ring_the_bell)
                       (info_message "Sorry, you can't set a tab past here.")
                       (flush_typeahead)
                       (sleep_for_n_milliseconds 1000))
                   (return true)

         "f"
         "F"       (if (^ (eolp))
                       (forward_char)
                       (if (= 81 (cur_hpos))
                           (if (^= (window_info column_offset) 61)
                               (hcol 61)))
                   else
                       (info_message "Sorry, you can't move Forward past here.")
                       (flush_typeahead)
                       (sleep_for_n_milliseconds 1000))
                   (return true)

         "b"
         "B"       (if (^ (bolp))
                       (if (= 61 (cur_hpos))
                           (if (^= (window_info column_offset) 1)
                               (hcol 1)))
                       (back_char))
                   (return true)

         "q"                           ; if a user types a q or Q or RETURN
         "Q"
         "~cJ"     (return false)

         "~cG"     (flush_typeahead)
                   (abort_command)

         otherwise
                   (flush_typeahead)
                   (ring_the_bell)
                   (info_message "Unknown response.")
                   (sleep_for_n_milliseconds 1000)
                   (return true)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; settabs_from_table, setf                                              ;;;
;;;                                                                       ;;;
;;;      Synonymous commands that look at a line of text and sets tabs    ;;;
;;;      based on the column position of words.                           ;;;
;;;                                                                       ;;;
;;;      (settabs_from_table)     or   (setft)                            ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom settabs_from_table ()
    &doc "Sets the tabs from a line"
    (settabs_from_tablef))

(defcom setft ()
    &doc "Sets the tabs from a line"
    (settabs_from_tablef))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; settabs_from_tablef                                                   ;;;
;;;                                                                       ;;;
;;;           Routine that does the actual work for the commands          ;;;
;;;           settabs_from_table and setft.                               ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun settabs_from_tablef (&local cur_cursor
                                   tablist
                                   dummy_last_tab)
    (save_excursion
        (if (line_is_blank)       ; try to find text before cursor.
            (back_word))
        (if (line_is_blank)       ; have been placed in a blank region. Not too
                                  ; bright of the user...abort.
            (info_message "Must have text on or before cursor's current "
                          "position. Try again")
            (ring_the_bell)
            (return))

         (check_tab_vars)         ; make certain that tab variables exist.

                                  ; save cursor so that can tell that we're
                                  ; still on the same line.
         (setq cur_cursor (copy_cursor current_cursor))
         (setq tablist ())
         (begin_line)
         (if (^ (looking_at " ")) ; check for initial whitespace.
             (setq tablist (list 1))
             (skip_to_white))

         (do_forever
             (if (skip_over_white)     ; get the position of the 1st words.
                 (if (^ (cursor_on_current_line_p cur_cursor)) (stop_doing))
                 ;
                 ; found a word...enter it's position into the tablist
                 ;
                 (setq tablist (append tablist (list (cur_hpos))))
             else
                 (stop_doing))

             (skip_to_white)
             (if (^ (cursor_on_current_line_p cur_cursor)) (stop_doing))
             (if (eolp)           ; get end-of-line...include it in tablist.
                 (setq tablist (append tablist (list (cur_hpos))))
                 (stop_doing)))

             (if (& global_tabs_on$ (^ (tab_language_mode)))
                 (info_message "Setting tabs...(global)")
                 (tablist_to_array$ tablist tab_array$ last_tab$)
                 (info_message "Tabs set (global).")
             else
                 (info_message "Setting tabs...(local to buffer)")
                 (tablist_to_array$ tablist (buffer_info (user tab_array))
                                            dummy_last_tab)
                 (buffer_info (user last_tab) dummy_last_tab) ; done this way
                                                    ; due to the fact that the
                                                    ; routine tablist_to_array$
                                                    ; expects an atom as its
                                                    ; last arg.
                 (info_message "Tabs set (local to buffer)"))))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; tablist        Command that accepts a series of numbers from the      ;;;
;;;                terminal and converts them into tab stops.             ;;;
;;;                                                                       ;;;
;;;           (tablist)                                                   ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom tablist
    &doc "Sets tabs from user-supplied list"
    &args ((tabs &prompt "Please supply tab column seperated by blanks"))
    (convert_tabs tabs))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; convert_tabs        Function that takes a list of numbers and converts;;;
;;;                     them into a tablist. It then calls will turn the  ;;;
;;;                     tablist into tab stops by calling tablist_to_array$;;;
;;;                                                                       ;;;
;;;      (convert_tabs string)                                            ;;;
;;;                                                                       ;;;
;;;      where string is a series of numbers seperated by spaces.         ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun convert_tabs (       (stops string)
                     &local source
                            destination
                            next_tab_string
                            tablist
                            next_tab
                            dummy_last_tab)
    (setq tablist ())
    (setq source 1)                    ; position on line to begin next search.
    (if (null stops)                   ; supply user with tab @ column 1.
        (setq stops "1"))
    (setq stops (catenate stops " "))  ; makes processing easier.
    (setq next_tab_string "")
    (do_forever
         (setq destination (index (substr stops source) " "))
         (setq next_tab_string (substr stops source destination))
         (setq source (+ source destination))
         (if (= next_tab_string "") (stop_doing))
                                       ; make certain that next value supplied
                                       ; is not > max tab of 140.
         (if (> (string_to_integer next_tab_string) 140) (stop_doing))
         (setq next_tab (string_to_integer next_tab_string))
         (setq tablist (append tablist (list next_tab))))

                                  ; if null tablist, either no tabs were given
                                  ; or the 1st one given was > max tab of 140.
                                  ; In either case, there is no point in going
                                  ; on with operation, so...
         (if (null tablist)
             (info_message "No valid tabs supplied. Tabs have not been set.")
             (ring_the_bell)
             (return))

    (check_tab_vars)              ; make certain tab variables exist.
    (if (& global_tabs_on$ (^ (tab_language_mode)))
        (info_message "Setting tabs (global)")
        (tablist_to_array$ tablist tab_array$ last_tab$)
        (info_message "Tabs set (global)")
     else
        (info_message "Setting tabs (local to buffer)")
        (tablist_to_array$ tablist (buffer_info (user tab_array)) dummy_last_tab)
        (buffer_info (user last_tab) dummy_last_tab)  ; done this way due to
                                                      ; fact that the routine
                                                      ; tablist_to_array$ wants
                                                      ; an atom for last arg.
        (info_message "Tabs set (local to buffer)")))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;; cobol_back_tab      Command that does back-tabbing in cobol mode.     ;;;
;;;                                                                       ;;;
;;;      (cobol_back_tab optional-integer)                                ;;;
;;;                                                                       ;;;
;;;      where optional-integer is how many tab stops to move back to.    ;;;
;;;     If it is negative, the command is interpreted as a normal tab.    ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defcom cobol_back_tab
    &doc "Special back_tab for COBOL mode"
    &na (&pass count &default 1)
    (back_tab count)
    (if (< (cur_hpos) 7)
        (cobol_begin_line)))      ; too far out...goto correct column for a
                                  ; COBOL program.
