/* This module does various manipulations with type-strings and classdefs. */

#include <memory.h>
#include "barbados.h"
#include "array.h"



/*---------------- Type-str manipulations: ---------------*/

interface size_t TypeSize(Type type)
/* Returns the size of this type. */
{       Classdef* classdef;
	int dimension;

	switch (*type) {
	    case tp_function:
	    case tp_error:
	    case tp_void:   return 0;
	    case tp_uchar:
	    case tp_char:   return 1;
	    case tp_ushort:
	    case tp_short:  return 2;
	    case tp_enumerated:
	    case tp_ulong:
	    case tp_long:
	    case tp_uint:
	    case tp_bool:
	    case tp_int:    return 4;
	    case tp_int64:  return 8;
	    case tp_float:  return sizeof(float);
	    case tp_double: return sizeof(double);
            case tp_longdouble:
                            return 10;
	    case tp_container:
			    return sizeof(container_id);
	    case tp_const:
	    case tp_volatile:
			    return TypeSize(++type);
	    case tp_pointer:
	    case tp_reference:
	    case tp_ptrmemberfn:
			    return sizeof(void*);
	    case tp_dynarray:
			    return sizeof(void*) + sizeof(int);
	    case tp_array:  type++;
			    GetDimension(dimension, type);  // macro
			    return dimension * TypeSize(type);
	    case tp_class:  type++;
			    GetPtr(classdef,type);         // macro
			    if (classdef == NULL)
				return 0;
			    return classdef->size;
	    default:        assert(false);
			    return 0;
	}
}


interface size_t TypeSizeWord(Type type)
/* Returns the size of this type, or the size of a */
/* word if the type is a char or enum or short.    */
{       Classdef* classdef;
	int dimension;

	switch (*type) {
	    case tp_error:
	    case tp_void:   return 0;
	    case tp_char:
	    case tp_uint:
	    case tp_bool:
	    case tp_long:
	    case tp_short:
	    case tp_uchar:
	    case tp_ulong:
	    case tp_ushort:
	    case tp_enumerated:
	    case tp_int:    return sizeof(int);
	    case tp_int64:  return 8;
	    case tp_float:  return sizeof(float);
	    case tp_double: return sizeof(double);
	    case tp_longdouble:
			    return 10;
	    case tp_container:
			    return sizeof(container_id);
	    case tp_pointer:
	    case tp_function:
	    case tp_reference:
	    case tp_ptrmemberfn:
			    return sizeof(char*);
	    case tp_const:
	    case tp_volatile:
			    return TypeSize(++type);
	    case tp_dynarray:
			    return sizeof(void*) + sizeof(int);
	    case tp_array:  type++;
			    GetDimension(dimension, type);  // macro
			    dimension *= TypeSize(type);
			    if (dimension & 3)
				dimension = (dimension | 3) + 1;
			    return dimension;
	    case tp_class: type++;
			    GetPtr(classdef,type);         // macro
			    if (classdef == NULL)
				return 0;
			    dimension = classdef->size;
			    if (dimension & 3)
				dimension = (dimension | 3) + 1;
			    return dimension;
	    default:        assert(false);
			    return 0;
	}
}


interface int TpSize(tp_enum tp)
{
        switch (tp) {
	    case tp_uchar:
            case tp_char:       return 1;
	    case tp_ushort:
            case tp_short:      return 2;
	    case tp_bool:
	    case tp_uint:
            case tp_int:
            case tp_float:
            case tp_pointer:
	    case tp_container:
            case tp_enumerated: return 4;
            case tp_long:
            case tp_double:     return 8;
            default:            return 9999;
        }
}


interface Type EndOfTypeString(Type r)
/* Returns the byte after the end this type-str. */
{       int i;

	forever switch (*r++) {
	    case tp_reference:
	    case tp_volatile:
	    case tp_dynarray:
	    case tp_pointer:
	    case tp_const:	continue;
	    case tp_array:      r += sizeof(int);
				continue;
	    case tp_ptrmemberfn:
	    case tp_function:   for (i=(unsigned char)*r++ & 127; i; i--)
				    r = EndOfTypeString(r);
				assert(*r == tp_terminated);
				r++;
				r += LengthOfTypeString(r);
				return r;
	    case tp_enumerated:
	    case tp_class:	return r + sizeof(int*);
	    default:            return r;
	}
}


interface int LengthOfTypeString(Type type)
/* Returns the length in bytes of this type descriptor. */
{
	return EndOfTypeString(type) - type;
}


interface bool TypeEqual(Type a, Type b)
{
	return memcmp(a,b,LengthOfTypeString(a)) == 0;
}


interface Classdef* TypeToClassdef(Type type)
/* If this type refers to a classdef, extract that classdef. */
{       Classdef* classdef;

	/* Get the 'classdef': */
        if (type == NULL)
            return NULL;	// Only happens if we've had an error.
	if (*type != tp_class)
	    return NULL;
	type++;
	GetPtr(classdef, type);
	return classdef;
}


interface tp_enum TpToSigned(tp_enum tp)
/* Convert this tp_enum into the signed version of the same thing. */
/* Also converts tp_enumerated's into tp_int's. */
{
	if (tp == tp_uint)
	    return tp_int;
	if (tp == tp_ushort)
	    return tp_short;
	if (tp == tp_uchar)
	    return tp_char;
	if (tp == tp_ulong)
	    return tp_long;
	if (tp == tp_enumerated or tp == tp_bool)
	    return tp_int;
	return tp;
}


interface bool TpIsUnsigned(tp_enum tp)
/* Is this an unsigned datum? */
{
	return (tp == tp_uint or tp == tp_ushort or tp == tp_uchar
                or tp == tp_ulong);
}


interface bool TypesNearlyEqual_i(Type *ap, Type *bp)
/* Are these types equal in every respect except signed/unsigned  */
/* and array size and reference level?  If YES, then the ptrs are */
/* both updated to the end of that type. */
{	Type a=*ap, b=*bp;
	int arity, i;

	do {
	    while (*a == tp_volatile or *a == tp_const or *a == tp_reference)
		a++;
	    while (*b == tp_volatile or *b == tp_const or *b == tp_reference)
		b++;
	    if (*a++ != *b++) {
		*ap = a;
		*bp = b;
		return no;
		//(TypeToSigned(tp_enum(a[-1])) == TypeToSigned(tp_enum(b[-1])));
		// If we return the above, then it means that eg. f(int) == f(unsigned int).
		// As it stands, f(int) != f(unsigned int).
	    }
	    *ap = a;
	    *bp = b;
	    switch (a[-1]) {
		case tp_dynarray:
		case tp_pointer:    continue;
		case tp_array:      a += sizeof(int);
				    b += sizeof(int);
				    continue;
	        case tp_ptrmemberfn:
		case tp_function:   arity = *a++;
				    if (*b++ != arity)
					return no;
				    arity &= 127;
				    for (i=arity-1; i >= 0; i--) {
					if (not TypesNearlyEqual_i(&a, &b))
					    return no;
				    }
				    assert(*a == tp_terminated and *b == tp_terminated);
				    a++;
				    b++;
				    *ap = a;
				    *bp = b;
				    if (not TypesNearlyEqual_i(ap, bp))
					return no;
				    return yes;
		case tp_enumerated:
		case tp_class:     return *(*(int**)ap)++ == *(*(int**)bp)++;

                case tp_longdouble:
		case tp_container:
		case tp_ushort:
		case tp_short:
		case tp_uchar:
                case tp_ulong:
		case tp_bool:
		case tp_uint:
		case tp_int:
		case tp_char:
		case tp_long:
		case tp_float:
		case tp_int64:
		case tp_double:
                case tp_void:       return yes;
		default:            assert(false);
	    }
	} forever;
}


interface bool TypesNearlyEqual(Type a, Type b)
/* Are these types equal in every respect except signed/unsigned  */
/* and array size and reference level?   We want to answer yes if */
/* the types are equivalent with "no or only trivial conversions".*/
{
	return TypesNearlyEqual_i(&a, &b);
}


interface void TypeCopy(Type dest, Type src)
/* Copy 'src' into 'dest': */
{
	memcpy(dest, src, LengthOfTypeString(src));
}


interface bool TypeIsBigStruct(Type type)
/* Is this type a non-datum, i.e. a struct which is > 4 bytes? */
{	Classdef* classdef;

	if (*type++ != tp_class)
            return no;
        classdef = *(Classdef**)type;
        return classdef->size > 4;
}


interface bool EqualFootprint(Type func, Type footprint)
/* Does this function-type have this footprint?  A footprint */
/* is the sequence of types which form the parameters.  We   */
/* must ignore the return type and ignore references.        */
{
	if (*func++ != tp_function or *footprint++ != tp_function)
	    return no;

	// Compare from here onward:
	if (*func++ != *footprint++)
	    return no;                          // Arity
	until (*func == tp_terminated) {
	    if (not TypesNearlyEqual_i(&func, &footprint))
		return no;
	}
	return yes;
}



interface bool TypeValidated(Type type)
/* Returns true iff this type is a meaningful type description. */
/* Used for debugging and for checking types when given in      */
/* library functions. */
{	unsigned int dim, arity, i;
	tp_enum tp;

	tp = tp_enum(*type++);
	switch (tp) {
            case tp_longdouble:
	    case tp_enumerated:
	    case tp_ushort:
	    case tp_short:
	    case tp_int64:
	    case tp_long:
	    case tp_void:
	    case tp_char:
	    case tp_bool:
	    case tp_int:
	    case tp_uint:
	    case tp_uchar:
	    case tp_float:
            case tp_ulong:
	    case tp_double:
	    case tp_container:	return yes;

	    case tp_class:	return yes;

	    case tp_pointer:
	    case tp_dynarray:
	    case tp_reference:  return TypeValidated(type);

	    case tp_const:
	    case tp_volatile:	return TypeValidated(++type);

	    case tp_array:      GetDimension(dim, type);
				return TypeValidated(type);
	    case tp_ptrmemberfn:
	    case tp_function:   arity = (*type++) & 127;
				for (i=0; i < arity; i++) {
				    if (not TypeValidated(type))
					return no;
				    type += LengthOfTypeString(type);
				}
				if (*type++ != tp_terminated)
				    return no;
				if (not TypeValidated(type))
				    return no;
				return yes;
	    case tp_error:
	    default:            return no;
	}
}


interface Type TypeAppend(Type d, Type suffix)
/* Copy 'suffix' into 'd' and return 'd'. */
{	int len;

	len = LengthOfTypeString(suffix);
	memcpy(d, suffix, len);
	return d + len;
}


interface Type TypeFunctionReturnType(Type type)
{	int i,arity;

	assert(*type == tp_function);
	type++;
	arity = *type++;
	for (i=0; i < arity; i++)
	    type += LengthOfTypeString(type);
	assert(*type == tp_terminated);
	type++;
	return type;
}


interface int TypeFunctionArity(Type type)
{
	assert(*type == tp_function or *type == tp_ptrmemberfn);
	type++;
	return *type++;
}


interface bool TpIsIntegral(tp_enum tp)
/* Is this tp an integral tp? */
{
	return TpGroup(tp) == 'i' or tp == tp_enumerated;
}


interface char TpGroup(tp_enum tp)
/* What kind of tp is this?  'i'=integral, 'f'=fpu, '\0'=other. */
/* Enums are counted as '\0'. */
{
	switch (tp) {
	    case tp_short:
	    case tp_uchar:
	    case tp_char:
	    case tp_uint:
	    case tp_bool:
	    case tp_int:
	    case tp_long:
            case tp_ulong:
	    case tp_ushort:	return 'i';

	    case tp_float:
	    case tp_int64:
	    case tp_double:
            case tp_longdouble: return 'f';

	    default:		return '\0';
	}
}


interface bool IsDirectory(Namedobj* obj, bool include_dirref, bool include_containers)
/* Is this a subdirectory? */
{       Type type=obj->type;

	if (*type == tp_container)
	    return include_containers;
	if (*type == tp_reference and include_dirref)
	    type++;
	return TypeEqual(type, direc_typstr);
}


interface bool IsConstructor(Namedobj *obj)
/* Does this object identify a constructor member function? */
{	Classdef* classdef;

	if (obj->storage != member_fn)
	    return no;
	classdef = (Classdef*)obj->owner;
	return classdef->typedef_obj and streq(classdef->typedef_obj->name,
			obj->name);
}


interface bool IsDestructor(Namedobj *obj)
/* Does this object identify a destructor member function? */
{	
	if (obj->storage != member_fn and obj->storage != virtual_fn)
	    return no;
        return strieq(obj->name, "~");
}


interface bool TypeHasConstructor(Type type)
/* Do we need a constructor for an object of this type? */
/* The answer is yes iff the type has a constructor     */
/* (or is an array of constructors).  Note that during  */
/* the class declaration we do additional checks for 	*/
/* whether the user was amiss in not providing a	*/
/* constructor or whether we must provide a default	*/
/* constructor. */
{	Classdef* classdef;
	int dimension;

	do switch (*type++) {
	    case tp_class:
		classdef = *(Classdef**)type;
                if (classdef->VirtualFns)
                    return yes;	// At the very least we'll need a function
                    // to insert virtual function pointers.
                for (Namedobj* obj=classdef->member; obj; obj=obj->next) {
                    if (IsConstructor(obj))
			return yes;
                }
                return no;

	    case tp_const:
	    case tp_volatile:
	    case tp_dynarray:
		continue;

	    case tp_array:
		GetDimension(dimension, type);  // macro
		continue;

	    default:
		return no;
	} forever;
}


interface bool TypeHasDestructor(Type type)
/* Do we need to call a destructor for an object of this type? */
{	Classdef* classdef;
	int dimension;

	do switch (*type++) {
	    case tp_class:
		classdef = *(Classdef**)type;
                if (classdef->VirtualFns)
                    return yes;	// At the very least we'll need a function
                    // to insert virtual function pointers.
                for (Namedobj* obj=classdef->member; obj; obj=obj->next) {
                    if (streq(obj->name, "~"))
			return yes;
                }
                return no;

	    case tp_const:
	    case tp_volatile:
	    case tp_dynarray:
		continue;

	    case tp_array:
		GetDimension(dimension, type);  // macro
		continue;

	    default:
		return no;
	} forever;
}


interface FunctionNamedobj* ClassDefaultConstructor(Classdef* classdef,
		Namedobj **any_constructor)
/* Return the default constructor for this class.  If there is no default */
/* constructor, return any old constructor in 'any_constructor'.  If 	  */
/* there's no constructor at all, then both will be NULL. */
{	Namedobj *obj;

	*any_constructor = NULL;
        for (obj=classdef->member; obj; obj=obj->next) {
            if (IsConstructor(obj)) {
            	*any_constructor = obj;
             	if (obj->type[1] == 0)
                    return (FunctionNamedobj*)obj;
            }
        }
        return NULL;
}


interface bool BisBaseOfD(Classdef* bclass, Classdef* dclass, int *offset)
/* Returns true if the 1st parameter is equal to or a base class  */
/* of the second parameter.  Return also via '*offset' the offset */
/* of the base class (quite often zero). */
{       Namedobj *obj;

	*offset = 0;
	if (bclass == dclass)
            return yes;
	for (obj=dclass->member; obj; obj=obj->next) {
            if (obj->storage == inherit_storage) {
            	Classdef* b2;
                int suboffset;
                b2 = TypeToClassdef(obj->type);
            	if (BisBaseOfD(bclass, b2, &suboffset)) {
                    *offset = ((FieldNamedobj*)obj)->offset + suboffset;
                    return yes;
                }
            }
        }
        return no;
}



/*-------------------- Bit-fields: -------------------*/

interface int EncodeBitField(int offset, int bit_offset, int bit_width)
{	int x;

	assert(bit_offset >= 0 and bit_offset < 8 and bit_width + bit_offset <= 32 and bit_width < 32);
	x = (1<<31) | (offset << 8) | (bit_offset << 5) | (bit_width&31);
	return x;
}


interface void DecodeBitField(int x, int *offsetp, int *bit_offsetp, int *bit_widthp)
{
	assert(x & (1<<31));
	*bit_widthp = x & 31;
	*bit_offsetp = (x >> 5) & 7;
	*offsetp = (x - (1<<31)) >> 8;
}




