#include <stdio.h>
#include <memory.h>
#include <math.h>
#include <windows.h>
#include <winuser.h>
#include <wingdi.h>
#include "barbados.h"
#include "bitmap.h"
#include "tfc.h"
#include "c_editor.h"

#pragma warning ( disable : 4245 )
#pragma warning ( disable : 4244 4761 )


struct bitmap_node {
	int cx, cy;
	void* mDC;
	void* Bmp;

	void init(bool alsoDC=yes);
};



static int fontHeight;


interface bitmap_type BM_New(int width, int height)
{	bitmap_type bm;

	bm = new bitmap_node;
	bm->cx = width;
	bm->cy = height;
	bm->mDC = NULL;
	bm->Bmp = NULL;
	return bm;
}


void bitmap_node::init(bool alsoDC)
/* Check that this bitmap's Bmp is initialised, and perhaps */
/* also it's DC. */
{
	if (Bmp == NULL) {
	    HDC tmpDC;
	    tmpDC = GetDC(NULL);
	    Bmp = CreateCompatibleBitmap(tmpDC, cx, cy);
	    SetBitmapDimensionEx((HBITMAP)Bmp, cx, cy, NULL);
	}
	if (alsoDC and mDC == NULL) {
	    mDC = CreateCompatibleDC(NULL);
	    SetMapMode((HDC)mDC, MM_TEXT);
	    SelectObject((HDC)mDC, Bmp);
	    BM_Clear(this, 0);
	}
}


interface void BM_Free(bitmap_type bm)
{
	if (bm->mDC)
	    DeleteDC((HDC)bm->mDC);
	if (bm->Bmp)
	    DeleteObject((HBITMAP)bm->Bmp);
	delete bm;
}


interface void BM_Clear(bitmap_type bm, int rgb)
{	static int old_rgb=-1;
	static HBRUSH hbr;
	RECT Rect;

	if (bm == NULL)
	    return;
	bm->init();
	if (rgb != old_rgb) {
	    if (hbr)
		DeleteObject(hbr);
	    hbr = CreateSolidBrush(rgb | 0x02000000);
	    old_rgb = rgb;
	}
	Rect.top = 0;
	Rect.bottom = bm->cy;
	Rect.left = 0;
	Rect.right = bm->cx;
	FillRect((HDC)bm->mDC, &Rect, hbr);
}


static void BM_SetPen(bitmap_type bm, int rgb)
{	static int old_rgb=-1;
	static HPEN hpen;

	if (rgb != old_rgb) {
	    bm->init();
	    if (hpen)
		DeleteObject(hpen);
	    hpen = CreatePen(PS_SOLID, 0, rgb);
	    old_rgb = rgb;
	    SelectObject((HDC)bm->mDC, hpen);
	}
}


interface void BM_Line(bitmap_type bm, int x0,int y0,int x1,int y1, int rgb)
/* Draw a line from (x0,y0) to (x1,y1) in colour 'col' */
/* on bitmap 'bm'. */
{
	bm->init();
	BM_SetPen(bm, rgb);
	MoveToEx((HDC)bm->mDC, x0, y0, NULL);
	LineTo((HDC)bm->mDC, x1, y1);
}


interface void BM_Rect(bitmap_type bm, int x0,int y0,int x1,int y1, int rgb)
/* Draw a line from (x0,y0) to (x1,y1) in colour 'col' */
/* on bitmap 'bm'. */
{	HBRUSH brush;
	RECT Rect;

	bm->init();
	Rect.top = y0;
	Rect.left = x0;
	Rect.right = x1;
	Rect.bottom = y1;
	brush = CreateSolidBrush(rgb);
	FillRect((HDC)bm->mDC, &Rect, brush);
	DeleteObject(brush);
}


interface void BM_Arrow(bitmap_type bm, int x0,int y0,int x1,int y1, int rgb)
/* Draw an arrow from (x0,y0) to (x1,y1) in colour 'col'. */
{	double x,y,r,cosa,sina;

	bm->init();
	BM_Line(bm, x0,y0,x1,y1, rgb);
	x = x0 - x1;
	y = y0 - y1;
	r = sqrt(x*x + y*y) / 4;        // Arrowheads of 4 pixels
	x /= r;
	y /= r;
	cosa = 0.866;
	sina = 0.5;

	x0 = x1 + cosa*x - sina*y;
	y0 = y1 + sina*x + cosa*y;
	BM_Line(bm, x0,y0,x1,y1, rgb);
    
	x0 = x1 + cosa*x + sina*y;
	y0 = y1 - sina*x + cosa*y;
	BM_Line(bm, x0,y0,x1,y1, rgb);
}


interface void BM_ArrowArc(bitmap_type bm, int x0,int y0,int x1,int y1, int d, int rgb)
/* Draw an arrow from (x0,y0) to (x1,y1) diverting it by 'd' pixels, in colour 'rgb'. */
{	double x,y,px,py,r,cosa,sina;
	POINT Bez[4];

	bm->init();

	/* The curve: */
	BM_SetPen(bm, rgb);
	x = x1 - x0;
	y = y1 - y0;
	r = sqrt(x*x + y*y);
	px = -y * d / r;
	py = x * d / r;
	x = (x0 + x1) / 2 + px;
	y = (y0 + y1) / 2 + py;
	Bez[0].x = x0;
	Bez[0].y = y0;
	Bez[1].x = (x0*2 + x1) / 3 + px;
	Bez[1].y = (y0*2 + y1) / 3 + py;
	Bez[2].x = (x0 + x1*2) / 3 + px;
	Bez[2].y = (y0 + y1*2) / 3 + py;
	Bez[3].x = x1;
	Bez[3].y = y1;
	if (d < 0) {
	    /* Self-referential loops: */
	    d = -d;
	    Bez[1].x = x0 - d;
	    Bez[1].y = y0 - d;
	    Bez[2].x = x1 - d;
	    Bez[2].y = y1 + d;
	}
	PolyBezier((HDC)bm->mDC, Bez, 4);
    
	/* The arrowhead: */
	x0 = Bez[2].x;
	y0 = Bez[2].y;
	x = x0 - x1;
	y = y0 - y1;
	r = sqrt(x*x + y*y) / 4;        // Arrowheads of 4 pixels
	x /= r;
	y /= r;
	cosa = 0.866;
	sina = 0.5;

	x0 = x1 + cosa*x - sina*y;
	y0 = y1 + sina*x + cosa*y;
	BM_Line(bm, x0,y0,x1,y1, rgb);
    
	x0 = x1 + cosa*x + sina*y;
	y0 = y1 - sina*x + cosa*y;
	BM_Line(bm, x0,y0,x1,y1, rgb);
}


interface void BM_SetPixel(bitmap_type bm, int x,int y, int rgb)
/* Set (x,y) in colour 'rgb' on bitmap 'bm'. */
{
	bm->init();
	SetPixel((HDC)bm->mDC, x,y, rgb);
}


static void SelectFont(bitmap_type bm)
/* Create and select a small font into this bitmap's mDC. */
{	static HFONT font;

	bm->init();
	if (font == NULL) {
	    font = CreateFont(10,0, 0,0, 400, no, no, no, 
		    DEFAULT_CHARSET, OUT_DEFAULT_PRECIS,
		    CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
		    FF_ROMAN, NULL);
	}
	SelectObject((HDC)bm->mDC, font);
}


static void TextDimensions(bitmap_type bm, str text, SIZE *Sizep)
/* What are the dimensions of this (possibly multiline) str? */
{	bool usefont;
	int cx, cy;
	SIZE Size;
	str s,d;
    
	if (bm == NULL)
	    usefont = no;
	else {
	    bm->init();
	    SelectFont(bm);
	    usefont = yes;
	}
	cx = cy = 0;
	for (s=d=text; ; s++) {
	    if (*s == '\0' or *s == '\n') {
		if (usefont) {
		    GetTextExtentPoint32((HDC)bm->mDC, d, s-d, &Size);
		    fontHeight = Size.cy;
		}
		else {
		    fontHeight = 13;
		    Size.cx = (s-d) * 9;
		    Size.cy = fontHeight;
		}
		cx = max(cx, Size.cx);
		cy += Size.cy;
		if (*s == '\0')
		    break;
		else 
		    d = ++s;
	    }
	}
	Sizep->cx = cx;
	Sizep->cy = cy;
}


interface void BM_BubbleSize(bitmap_type bm, str text, int *xp, int *yp)
/* What are the dimensions of an ellipse enclosing 'text'? */
{	SIZE Size;

	Size.cx = 8;
	Size.cy = 13;
	TextDimensions(bm, text, &Size);
	*xp = Size.cx * 1.414;
	*yp = Size.cy * 1.414;
}


interface void BM_Bubble(bitmap_type bm, str text, int x, int y, 
                            int rgb_text, int rgb_outline, int rgb_background)
/* Display text at the specified location inside an ellipse. */
{	static HBRUSH hbr, old_br;
	int width, height;
	SIZE Size;
	RECT Rect;
	str s,d;

	bm->init();

	/* Calculate width and height: */
	Size.cx = 8;
	Size.cy = 13;
	TextDimensions(bm, text, &Size);
	width = Size.cx;
	height = Size.cy;

	/* Draw the ellipse: */
	hbr = CreateSolidBrush(rgb_background);
	BM_SetPen(bm, rgb_outline);
	old_br = (HBRUSH) SelectObject((HDC)bm->mDC,(HBRUSH) hbr);
	Ellipse((HDC) bm->mDC, x - 0.707*width, y - 0.707*height,
			 x + 0.707*width + 1, y + 0.707*height + 1);
	SelectObject((HDC) bm->mDC, (HBRUSH) old_br);
	DeleteObject(hbr);
    
	/* Draw the text: */
	Rect.top = y - 0.5*height;
	Rect.left = x - 0.5*width;
	Rect.bottom = y + 0.5*height;
	Rect.right = x + 0.5*height;
	SetBkColor((HDC) bm->mDC, rgb_background);
	SetBkMode((HDC) bm->mDC, TRANSPARENT);  // OPAQUE
	SetTextColor((HDC) bm->mDC, rgb_text);
	for (s=d=text; ; s++) {
	    if (*s == '\0' or *s == '\n') {
		DrawText((HDC) bm->mDC, d, s-d, &Rect, 
		    DT_LEFT | DT_NOPREFIX | DT_TOP | DT_SINGLELINE | DT_NOCLIP);
		if (*s == '\0')
		    break;
		d = ++s;
		Rect.top += fontHeight;
	    }
	}
}


static double sqr(double a)
{
	return a*a;
}


interface void BM_BubbleIntersect(bitmap_type bm, str text, 
                            int x0, int y0, int x1, int y1, int *p)
/* Works out the point of intersection of this line and the */
/* bubble containing 'text' centred on (x0,y0).             */
/* Returns the point in p[0] and p[1].                      */
{	int width, height;
	double t, dx, dy;
	SIZE Size;
    
	/* Calculate width and height: */
	Size.cx = 8;
	Size.cy = 13;
	//TextDimensions(bm, text, &Size);
	width = Size.cx;
	height = Size.cy;

	/* Calculate the quadratic coefficients: */
	dx = x1 - x0;
	dy = y1 - y0;
	t = 1 / sqrt(2 * (sqr(dx/width) + sqr(dy/height)));
    
	/* Give the answers: */
	p[0] = x0 + t*dx;
	p[1] = y0 + t*dy;
}





/*----------------------- Displaying Bitmaps: --------------------*/

static HWND BmpWindow;
static WNDCLASS BmpClass;
static HDC BmpDC;
static bool ClassRegistered;
static struct {
	HDC mDC;
	HBITMAP Bmp;
	int cx, cy;
} B;


static void Reprint(HDC hDC)
/* Reprint 'current_bm' with this device context. */
{   
        BitBlt(hDC, 0,0, B.cx, B.cy, B.mDC, 0,0, SRCCOPY);
}       



static void OnPaint(HWND hwnd)
{	PAINTSTRUCT paintStruct;
	HDC hDC;
    
	hDC = BeginPaint(hwnd, &paintStruct);
	if (hDC != NULL) {
	    Reprint(hDC);
	    EndPaint(hwnd, &paintStruct);
	}
}


static LRESULT CALLBACK WndProc(HWND wnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg) {
	    case WM_PAINT:  OnPaint(wnd);
		break;
	    case WM_DESTROY:BmpWindow = 0;
		break;
	    case WM_KEYUP:
	    case WM_KEYDOWN:
	    case WM_LBUTTONDOWN:
	    case WM_LBUTTONUP:
	    case WM_MOUSEMOVE:
		//IO_WindowPost(uMsg, wParam, lParam);
		break;

	    case WM_SETFOCUS:
		return 0;

	    default:return DefWindowProc(wnd, uMsg, wParam, lParam);
	}
	return 0;
}



static void ShowBitmapWindow(int width, int height)
/* Set up the bitmap window if not already done, and */
/* set the device context to it. */
{	static int old_height, old_width;

        if (not BmpWindow) {
	    /* Set up the window again. */

	    if (not ClassRegistered) {
		/* Set up the Bitmap window class: */

		memset(&BmpClass, 0, sizeof(BmpClass));
		BmpClass.style = CS_HREDRAW | CS_VREDRAW | CS_BYTEALIGNCLIENT;
		BmpClass.lpfnWndProc = WndProc;
		BmpClass.hInstance = NULL;
		BmpClass.hCursor = LoadCursor(NULL, IDC_ARROW);
		BmpClass.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
		BmpClass.lpszClassName = "BarbBitmap";
		RegisterClass(&BmpClass);
		ClassRegistered = yes;
	    }

	    /* Create the window: */
	    BmpWindow = CreateWindow("BarbBitmap", "Bitmap", WS_OVERLAPPEDWINDOW,
			0, 400, width + 8, height + 32,
			NULL, NULL, NULL, NULL);
	    UpdateWindow(BmpWindow);
	    old_height = height;
	    old_width = width;
    
	    /* Create a device context: */
	    BmpDC = GetDC(BmpWindow);
	}
	else {
	    WINDOWPLACEMENT wplace;
	    if (height != old_height or width != old_width) {
		wplace.length = sizeof(WINDOWPLACEMENT);
		GetWindowPlacement(BmpWindow, &wplace);
		wplace.rcNormalPosition.right = wplace.rcNormalPosition.left + width + 8;
		wplace.rcNormalPosition.bottom = wplace.rcNormalPosition.top + height + 27;
		SetWindowPlacement(BmpWindow, &wplace);
		old_width = width;
		old_height = height;
	    }
	}
	ShowWindow(BmpWindow, SW_RESTORE);
	ShowWindow(BmpWindow, SW_SHOW);
}



interface void BM_Display(bitmap_type bm)
/* Display this bitmap. */
{	HBITMAP oldBmp;

	if (bm == NULL)
	    return;
	bm->init(yes);

	/* Take a copy of this bitmap: */
	if (B.mDC == NULL)
	    B.mDC = CreateCompatibleDC(NULL);
	oldBmp = B.Bmp;
	if (oldBmp and B.cx == bm->cx and B.cy == bm->cy)
	    ;       /* The existing bitmap is Ok. */
	else {
	    B.Bmp = CreateCompatibleBitmap(GetDC(NULL), bm->cx, bm->cy);
	    SelectObject(B.mDC, B.Bmp);
	    if (oldBmp != NULL) 
		DeleteObject(oldBmp);
	    B.cx = bm->cx;
	    B.cy = bm->cy;
	}
	BitBlt((HDC)B.mDC, 0,0, B.cx, B.cy, (HDC) bm->mDC, 0,0, SRCCOPY);

	/* Print it on the screen: */
	ShowBitmapWindow(bm->cx, bm->cy);
	Reprint(BmpDC);
	CompilerEditor->Focus();
}


interface str BM_Output(bitmap_type bm)
/* Output this bitmap to the log. */
{
	return "Not implemented";
}

