#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include "resource.h"
#include "barbados.h"
#include "join.h"
#include "c_editor.h"
#include "array.h"


class CLine : public Line {
	virtual void Paragrapher(Paragrapher_type P);
	/* Do everything related to the display of this paragraph. */

	virtual bool Mousestroke(int op, int _x, int _y);

	friend class C_Editor;
};


typedef enum { le_normal, le_string, le_char, le_comment, 
	le_preprocessor, le_keyword, le_highlight } le_enum;

typedef enum { ty_current, ty_pastinput, ty_output, ty_error, ty_history } ty_enum;

#define COMPILEABLE_FLAG	(1<<15)
#define NONEMPTY_FLAG		(1<<14)



static int IsKeyword(char *s)
/* Is 's' a keyword (perhaps followed by punctuation?) */
{	static const char *reserved[] = {
                "typedef",
                "class", "protected", "private", "public", "this", "virtual",
                "cout", "cin", "endl",
                "for", "printf", "scanf", "if", "do", "while",
                "const", "static", "extern", "inline",
                "bool", "false", "true",
                "new", "delete", "sizeof",
                "int", "unsigned", "double", "float", "string", "std",
                "char", "void", "long",
                "static_cast", "const_cast", "reinterpret_cast", "dynamic_cast",
                NULL };
	char buf[40], *d=buf;
	int i;

	while ((isalpha(*s) or *s == '_') and d < buf+sizeof(buf)-1)
	    *d++ = *s++;
	*d = '\0';
	for (i=0; reserved[i]; i++)
	    if (strcmp(reserved[i], buf) == 0)
		return d - buf;
	return 0;
}


static int LexicalAnalyse(str buf, str toks, int les)
#define is(le)	    *t++ = le, s++
{	bool Compileable, NonEmpty;
	int brackets, keyword_len;
	str s, t;

	s = buf, t = toks;
	brackets = (les >> 3) & 255;
	Compileable = no;
	NonEmpty = no;
	les &= 7;
	if (les == le_comment)
	    goto SLASHSTAR_COMMENT;
	else if (les == le_string)
	    goto STRING;
	else if (les == le_preprocessor)
	    goto PREPROCESSOR;
	
	NORMAL:
	switch (*s) {
	    case '/':	if (s[1] == '*') {
			    is(le_comment);
			    is(le_comment);
			    SLASHSTAR_COMMENT:
			    until (*s == '*' and s[1] == '/') {
				if (*s == '\0') {
				    les = le_comment;
				    goto RETURN;
				}
				is(le_comment);
			    }
			    is(le_comment);
			    is(le_comment);
			    goto NORMAL;
			}
			else if (s[1] == '/') {
			    is(le_comment);
			    while (*s)
				is(le_comment);
			    les = le_normal;
			    goto RETURN;
			}
			else {
			    is(le_normal);
			    goto NORMAL;
			}

	    case '"':	is(le_string);
			NonEmpty = yes;
			STRING:
			do {
			    if (*s == '\\') {
				is(le_string);
				if (s[1] == '\0') {
				    les = le_string;
				    goto RETURN;
				}
				is(le_string);
 			    }
			    else if (*s == '"') {
				is(le_string);
				goto NORMAL;
			    }
			    else if (*s == '\0') {
				les = le_string;
				goto RETURN;
			    }
			    else is(le_string);
			} forever;

	    case '\'':	is(le_char);
			NonEmpty = yes;
			do {
			    if (*s == '\\') {
				is(le_char);
				if (s[1] == '\0') {
				    les = le_char;
				    goto RETURN;
				}
				is(le_char);
			    }
			    else if (*s == '\'') {
				is(le_char);
				goto NORMAL;
			    }
			    else if (*s == '\0') {
				les = le_char;
				goto RETURN;
			    }
			    else is(le_char);
			} forever;

	    case '#':	is(le_preprocessor);
			NonEmpty = yes;
			PREPROCESSOR:
			while (*s) {
			    if (*s == '\\')
				les = le_preprocessor, Compileable = no;
			    else if (*s != ' ' and *s != '\t')
				les = le_normal, Compileable = yes;
			    is(le_preprocessor);
			}
			goto RETURN;

	    case '\0':	les = le_normal;
			goto RETURN;

	    case '{':
	    case '[':
	    case '(':	is(le_normal);
			brackets++;
			NonEmpty = yes;
			goto NORMAL;

	    case '}':	is(le_normal);
			if (--brackets == 0)
			    Compileable = yes;
			NonEmpty = yes;
			goto NORMAL;

	    case ']':
	    case ')':	is(le_normal);
			NonEmpty = yes;
			brackets--;
			goto NORMAL;

	    case ';':	is(le_normal);
			if (brackets == 0)
			    Compileable = yes;
			NonEmpty = yes;
			goto NORMAL;

	    case ' ':
	    case '\t':	is(le_normal);
			goto NORMAL;

	    default:	if (s == buf or (not isalpha(s[-1]) and s[-1] != '_')) {
			    keyword_len = IsKeyword(s);
			    if (keyword_len) {
				while (keyword_len--)
				    is(le_keyword);
				goto NORMAL;
			    }
			}
			is(le_normal);
			NonEmpty = yes;
			goto NORMAL;
	}

	RETURN:
	*t = le_normal;
	if (brackets < 0)
	    brackets = 0;
	les |= brackets << 3;
	if (Compileable)
	    les |= COMPILEABLE_FLAG;
	if (NonEmpty)
	    les |= NONEMPTY_FLAG;
	return les;
}


static void TypeToColours(ty_enum ty, int *fg_colourp, int *bg_colourp, bool for_printer)
{
	switch (ty) {
	    case ty_pastinput:
		    *fg_colourp = WHITE;
                    *bg_colourp = BLACK;
		    return;

	    case ty_output:
		    *fg_colourp = ORANGE;
                    *bg_colourp = BLACK;
		    return;

	    case ty_error:
		    *fg_colourp = MAGENTA;
                    *bg_colourp = BLACK;
		    return;

	    case ty_history:
		    *fg_colourp = YELLOW;
                    *bg_colourp = DARK(RED);
		    return;

	    default:
		    *fg_colourp = YELLOW;
		    return;
	}
}


static void LeToColours(le_enum le, int *fg_colourp, int *bg_colourp, bool for_printer)
{
	if (not for_printer) {
	    switch (le & 7) {
		case le_highlight:
			*fg_colourp = BLACK, *bg_colourp = CYAN;
			return;

		case le_comment:
			*fg_colourp = GREEN;
			return;

		case le_preprocessor:
			*fg_colourp = RED, *bg_colourp = DARK(BLUE);
			return;

		case le_string:
		case le_char:
			*fg_colourp = WHITE;
			return;

		case le_keyword:
			*fg_colourp = YELLOW;
			return;

		default:
			*fg_colourp = YELLOW;
			return;
	    }
	}
	else {
	    switch (le & 7) {
		case le_comment:
			*fg_colourp = GREEN, *bg_colourp = WHITE;
			return;

		default:
			*fg_colourp = BLACK, *bg_colourp = WHITE;
			return;
	    }
	}
}


static str StateToPrompt(int incoming, int outgoing)
/* Map this line's state field to a prompt character. */
{	int ib, ic, it, ob, oc, ot;

	ib = (incoming >> 3) & 255;
	ic = (incoming & COMPILEABLE_FLAG) != 0;
	it = (incoming & NONEMPTY_FLAG) != 0;
	incoming &= 7;
	ob = (outgoing >> 3) & 255;
	oc = (outgoing & COMPILEABLE_FLAG) != 0;
	ot = (outgoing & NONEMPTY_FLAG) != 0;
	outgoing &= 7;

	if (incoming != le_normal) {
	    switch (incoming) {
		case le_comment:
			    return "/";
		case le_string:
			    return "\"";
		case le_char:
			    return "\'";
		case le_preprocessor:
			    return "#";
		default:    return "?";
	    }
	}
	else if (ib == 0) {
	    if (ob)
		return "{";
	    else if (oc)
		return ";";
	    else if (not ot)
		return " ";
	    else if (ic or not it)
		return ">";
	    else return " ";
	}
	else {
	    if (ob == 0)
		return "}";
	    else {
            	static char buf[2];
                buf[0] = ib+'0';
                buf[1] = '\0';
                return buf;
            }
	}
}


static int BracketColour(int brackets, int _out, int _in)
{	union {
	    struct { char r,g,b,dummy; } rgb;
            int colour;
	} out, in, bra;
        int scaler1,scaler2;

	out.colour = _out;
	in.colour = _in;
        if (brackets > 4)
            brackets = 4;
        scaler1 = brackets * 64;
        scaler2 = 256 - brackets*64;
        bra.rgb.r = (scaler2*out.rgb.r + scaler1*in.rgb.r)>>8;
        bra.rgb.g = (scaler2*out.rgb.g + scaler1*in.rgb.g)>>8;
        bra.rgb.b = (scaler2*out.rgb.b + scaler1*in.rgb.b)>>8;
        bra.rgb.dummy = 0;
	return bra.colour;
}


Line* C_Editor::NewLine()
{
	return new CLine;
}


bool C_Editor::MoveUp(Line *ln)
/* The user has just exited this line by moving up. */
{
	if (parse_line == ln) {
	    HistoryMode();
	    return no;
	}

	/* Repaint the prompt: */
	ln->PaintWhole();
	if (ln->prev)
	    ln->prev->PaintWhole();

	return yes;
}


static str GatherSourceCode(Line *a, Line *b)
/* Gather all the characters from a to b, inclusive, and put them */
/* in a big string.  The user must free this at some stage. */
{	Line *ln, *end;
	str s, s0;
	int size;

	/* This shouldn't happen... */
	if (a->y > b->y)
	    return strdup(anon_heap, "");

	/* How big is it? */
	size = 1;
	end = (Line*)b->next;
	for (ln=a; ln != end; ln = (Line*)ln->next)
	    size += strlen(ln->buf) + 1;

	/* Allocate it: */
	s = s0 = (char*)malloc(anon_heap, size);

	/* Fill it: */
	for (ln=a; ln != end; ln = (Line*)ln->next) {
	    if (ln->type > ty_pastinput)
		break;
            if (s == s0 and *ln->buf == '\0')
                continue;
	    strcpy(s, ln->buf);
	    s += strlen(s);
	    *s++ = '\n';
	}
	*s = '\0';
	return s0;
}


static LineN_struct GatherGetErrorPos(Line *a, Line *b, int n)
/* After gathering source-code and identifying an error in it, */
/* move to the exact spot with the error. */
{	LineN_struct e;
	Line *end;

	end = (Line*)b->next;
	for (e.ln=a, e.n=n; e.ln and e.ln != end; e.ln = (Line*)e.ln->next) {
	    if (e.ln->type > ty_pastinput)
		continue;
	    if (e.n <= strlen(e.ln->buf))
		break;
	    else e.n -= strlen(e.ln->buf) + 1;
	}
	return e;
}


interface bool C_Editor::MoveDown(Line *ln)
/* The user has just exited this line by moving down. */
{	bool Compileable, success;
	error_node Error;
	char toks[1024];
	short outgoing;
        CLine *oldln;
	str s;

	if (ln->type > ty_pastinput)
	    return yes;

	/* Update compiler info: */
	outgoing = LexicalAnalyse(ln->buf, toks, ln->state);
	Compileable = (outgoing & COMPILEABLE_FLAG) != 0 and
                        (outgoing & 0x7) == le_normal;
	if (Compileable and NotifyCompileable) {
	    ClearErrorLines();
	    if (parse_line == NULL)
		parse_line = ln;
	    s = GatherSourceCode(parse_line, ln);
	    if (s)
		JoinCompileAndRun(s, &Error);
	    else Error.err = err_none;
	    if (Error.err and Error.err != err_runtime) {
                LineN_struct e;
                e = GatherGetErrorPos(parse_line, ln, Error.pos - s);
                if (e.ln)
                    MoveTo(e);
                InsertLineB(c.ln, no, Error.message, ty_error, no);
		success = no;
	    }
	    else {
		if (Error.err == err_runtime)
		    InsertLineB(ln, no, Error.message, ty_error, no);
                /* Move the parse-line down: */
                oldln = (CLine*)parse_line;
		parse_line = GetPosition();
	        for (; oldln->y < parse_line->y; oldln=(CLine*)oldln->next) {
                    if (oldln->type == ty_current) {
                        oldln->type = ty_pastinput;
                        oldln->PaintWhole();
                    }
                }
		success = yes;
	    }
	    free(anon_heap, s);
	}
	else success = yes;

	/* Repaint the prompt: */
	ln->PaintWhole();
	if (ln->next)
	    ln->next->PaintWhole();
	return success;
}


interface int C_Editor::DefineState(Line *prev)
/* Define this line's state by processing the previous line. */
{	char tbuf[1024];

	if (prev == NULL)
	    return le_normal;
	if (prev->type > ty_pastinput)
	    return prev->state;
	return LexicalAnalyse(prev->buf, tbuf, prev->state);
}


interface void CLine::Paragrapher(Paragrapher_type P)
/* Do everything related to the display of this paragraph. */
{	int i, j, W_idx, minY, maxY, maxX, limitX, marginX;
	C_Editor *Editor=(C_Editor*)parent;
	static unsigned int PromptWidth;
	struct {
	    int x;
	    int height;
	    str s;
	    int sn;
	    le_enum le;
	} W[200];
	int fg_colour, bg_colour, les;
	unsigned int height, width;
	str s, sl, sw, prompt;
        int bracket_colour;
	int x, y, alignX;
	char LE[1024];
	le_enum le;

        /*** Setting up: ***/
        les = LexicalAnalyse(buf, LE, state);
	sl = buf;
	y = P->y1;
	if (PromptWidth == 0) {
	    TextDimensions("H", 1, Editor->thisfont(), &PromptWidth, &height);
	    PromptWidth++;
	}
	marginX = PromptWidth;
	limitX = Editor->PaginatedWidth;
	if (limitX < 10) {
	    assert(false);
	    return;
	}
	maxX = 0;


        /*** Paint the background: ***/
        if (P->pa == pa_paint) {
            if (type > ty_pastinput) {
            	bg_colour = Editor->Background;
                TypeToColours((ty_enum)type, &fg_colour, &bg_colour, no);
                DrawRectangle(P->x1,P->y1,P->x2,P->y2,bg_colour);
            }
            else {
            	int i,brackets1,brackets2,brackets,x1,x2,x3;
                brackets1 = (state>>3) & 255;
                brackets2 = (les>>3) & 255;
                brackets = min(brackets1,brackets2);
                bracket_colour = BracketColour(brackets,
                            Editor->Background, BLACK);
                x1 = PromptWidth;
                x3 = Editor->PaginatedWidth;
                /*if (Editor->realWidth < Editor->PaginatedWidth)
                    x3 = Editor->realWidth;*/
                for (i=0; i < brackets; i++) {
                    x2 = x1 + Editor->TabSize;
                    DrawRectangle(x1,P->y1,x2,P->y2,
                    	BracketColour(i, Editor->Background, BLACK));
                    x1 = x2;
                }
                x2 = x3 - brackets*Editor->TabSize;
                for (i=brackets; i >= 0; i--) {
                    DrawRectangle(x1,P->y1,x2,P->y2,
                    	BracketColour(i, Editor->Background, BLACK));
                    x1 = x2;
                    x2 += Editor->TabSize;
                }
                //DrawRectangle(P->x1,P->y1,P->x2,P->y2,bracket_colour);
            }
        }


	do {
	    /* Take the first (next) line of the paragraph: */
	    W_idx = 0;
	    sw = sl;
	    minY = 99999;
	    maxY = 0;
	    x = P->x1 + marginX;

	    /* Moving to Home: */
	    if (P->pa == pa_xy2cn and P->mouse_x <= marginX and P->mouse_y <= y) {
		P->cn = sl - buf;
		return;
	    }

	    do {
		/* Take the first (next) word of the line: */
		s = sw;
		le = (le_enum)LE[s - buf];
		if (*s == '\t') {
		    TextDimensions("H", 1, Editor->thisfont(), &width, &height);
		    width = Editor->TabSize - ((x - marginX) % Editor->TabSize);
		    if (P->ha >= 0) {
			i = s - buf;
			if (i >= P->ha and i < P->hb)
			    le = le_highlight;
		    }
		    s++;
		    goto HAVE_WIDTH;
		}
		if (*s <= ' ') {
		    while (*s <= ' ' and *s and *s != '\t') {
			s++;
			if (LE[s - buf] != le)
			    break;
		    }
		}
		else {
		    while (*s > ' ') {
			s++;
			if (LE[s - buf] != le)
			    break;
		    }
		    if (*s == ' ')
			s++;
		}

		/* What about highlighting? */
		if (P->ha >= 0) {
		    i = sw - buf;
		    j = s - buf;
		    if (i < P->hb and j >= P->ha) {
			if (i >= P->ha and j < P->hb)
			    le = le_highlight;	// The whole word is highlighted
			else if (i >= P->ha) {
			    s = buf + P->hb;	// Highlighting stops halfway through the word
			    le = le_highlight;
			}
			else {
			    s = buf + P->ha;	// Highlighting starts halfway through the word
			}
		    }
		}

		/* Measure this word: */
		TextDimensions(sw, s - sw, Editor->thisfont(), &width, &height);
		HAVE_WIDTH:
		if (x + width > limitX) {
		    char *old_s=s, *old_sw=sw;
		    int old_W_idx=W_idx, old_x=x;

		    if (width >= limitX / 2) {
			/* We have one very long token. Extend the limit to the right. */
			limitX = 999999;
			goto HAVE_WIDTH;
		    }

		    /* Rewind all the tokens that are joined to this token: */
		    while (W_idx > 0 and sw[-1] != ' ' and sw[-1] != '\t') {
			sw = s = W[--W_idx].s;
			x = W[W_idx].x;
		    }

		    if (W_idx == 0 or W[W_idx-1].x < limitX / 3) {
			/* We have a very long connected token. */
			limitX = 999999;
			s = old_s;
			sw = old_sw;
			W_idx = old_W_idx;
			x = old_x;
			goto HAVE_WIDTH;
		    }
		    break;
		}
		if (minY > height)
		    minY = height;
		if (maxY < height)
		    maxY = height;
		W[W_idx].x = x;
		W[W_idx].le = le;
		x += width;
		W[W_idx].height = height;
		W[W_idx].s = sw;
		W[W_idx].sn = s - sw;
		if (++W_idx >= sizeof(W)/sizeof(W[0]))
		    assert(false);
		sw = s;
	    } while (*sw);

	    W[W_idx].x = x;
	    y += maxY;

	    /* Take care of left (or centre) alignment: */
	    if (sl > buf) {
		if (limitX > 99999)
		    alignX = 50;
		else alignX = limitX - 5 - x;
		for (i=0; i <= W_idx; i++)
		    W[i].x += alignX;
		x += alignX;
	    }

	    if (maxX < x)
		maxX = x;

	    if (P->pa == pa_paint) {
		/* We are painting. */
		if (sl > buf)
		    prompt = " ";
		else if (P->IsCurrent)
		    prompt = ">";
		else prompt = StateToPrompt(state, les);
		DrawString(prompt, -1, Editor->thisfont(), P->x1, y - maxY, P->x1 + marginX, y, BLACK, GREY);
		for (i=0; i < W_idx; i++) {
		    le = W[i].le;
                    bg_colour = NOCOLOUR;
		    if (type)
			TypeToColours((ty_enum)type, &fg_colour, &bg_colour, no);
		    else LeToColours(le, &fg_colour, &bg_colour, no);
		    if (W[i].s[0] == '\t')
			DrawRectangle(W[i].x, y - W[i].height,
				    W[i+1].x, y, bg_colour, NOCOLOUR);
		    else DrawString(W[i].s, W[i].sn, Editor->thisfont(),
			    W[i].x, y - W[i].height,
			    W[i+1].x, y,
			    fg_colour, bg_colour);
		}
		if (W[i].x < P->x2) {
		    if (P->hb > 999)
			bg_colour = CYAN;
		    else if (type)
			;   // we already have bg_colour
		    else bg_colour = NOCOLOUR;
		    DrawRectangle(W[i].x,
				y - minY,
				P->x2,
				y,
				bg_colour, NOCOLOUR);
		}
	    }
	    else if (P->pa == pa_cn2rect and sw - buf > P->cn) {
		/* Have we found the cursor position already? */
		if (W_idx == 0) {
		    P->x1 = 0;
		    P->x2 = 3;
		    P->y1 = P->y1;
		    P->y2 = P->y2;
		    return;
		}
		for (i=0; i < W_idx; i++) {
		    if (W[i].s + W[i].sn > buf + P->cn)
			break;
		}
		if (i == W_idx)
		    i--;
		int n = buf + P->cn - W[i].s;
		if (n <= 0)
		    width = 0;
		else TextDimensions(W[i].s, buf + P->cn - W[i].s, Editor->thisfont(), &width, &height);
		P->x1 = W[i].x + width;
		P->y1 = y - W[i].height;
		P->y2 = y;
		TextDimensions(W[i].s, buf + P->cn - W[i].s + 1, Editor->thisfont(), &width, &height);
		P->x2 = W[i].x + width;
		return;
	    }
	    else if (P->pa == pa_xy2cn and y > P->mouse_y) {
		if (W_idx == 0) {
		    P->cn = 0;
		    return;
		}
		else if (P->mouse_x < W[0].x) {
		    P->cn = W[0].s - buf;
		    return;
		}
		for (i=0; i < W_idx; i++) {
		    if (W[i+1].x > P->mouse_x)
			break;
		}
		if (i >= W_idx) {
		    P->cn = sw - buf;
		    if (buf[P->cn])
			P->cn--;
		    return;
		}
		s = W[i].s;
		assert(*s);
		for (j=0; j <= W[i].sn; j++) {
		    TextDimensions(W[i].s, j, Editor->thisfont(), &width, &height);
		    if (W[i].x + width > P->mouse_x) {
			P->cn = W[i].s + j - 1 - buf;
			return;
		    }
		}
		P->cn = W[i].s + j - 1 - buf;
		return;
	    }

	    sl = sw;
	} while (*sl);

	if (P->pa == pa_measure) {
	    P->x2 = maxX;
	    P->y2 = y;
	}
	else if (P->pa == pa_cn2rect) {
	    P->x1 = x;
	    P->x2 = x;
	    P->y1 = y - maxY;
	    P->y2 = y;
	}
	else if (P->pa == pa_xy2cn) {
	    P->cn = sl - buf;
	}
}


C_Editor::C_Editor(str caption, int width, int height)
    : EditorWin(caption, width, height)
{
	Background = DARK(BLUE);
	NotifyCompileable = yes;
	FixedWidth = yes;
	font = TfcFindFont(13,no,no,yes);
        TabSize = 4 * AverageCharWidth();

	/* Avoid having root == NULL. */
	c.ln = InsertLineNoUndo(NULL, "");

	/* Have a valid cursor: */
	SetCursor();
}


void C_Editor::InsertLineB(Line* ln, bool abovecurrent, str s, char type, bool moveto)
/* Insert this line either above or below the current line. */
/* Then, if requested, we move onto this line.  No undo info*/
/* is created. */
{
	if (abovecurrent and ln->prev)
	    ln = InsertLineNoUndo((Line*)ln->prev, s, type);
	else ln = InsertLineNoUndo(ln, s, type);
	if (moveto)
	    MoveTo(ln, 0);
}


void C_Editor::DeleteLineB(Line* ln)
/* Delete this line from the editor and call its destructor. */
/* Don't create any undo info. */
{
	ClearAllUndos();
	Delete(ln);
	delete ln;
}


void C_Editor::ClearOutputLines(void)
/* Clear all <<< output lines near the current location. */
{	Line *ln;

	do {
	    ln = GetPosition();
	    if (ln == NULL)
		break;
	    if (ln->type > ty_pastinput) {
		DeleteLineB(ln);
		continue;
	    }
	    else break;
	} forever;
	do {
	    ln = GetPosition();
	    if (ln == NULL)
		break;
	    ln = (Line*)ln->prev;
	    if (ln == NULL)
		break;
	    if (ln->type > ty_pastinput) {
		DeleteLineB(ln);
		continue;
	    }
	    else break;
	} forever;
}


void C_Editor::ClearErrorLines(void)
{	Line *ln, *ln_next;

	for (ln=parse_line; ln; ln=ln_next) {
	    ln_next = (Line*)ln->next;
	    if (ln->type == ty_error)
		DeleteLineB(ln);
	}
}


bool C_Editor::Include(str filename)
{
	TfcMessage("#include", 'i', "Unimplemented.");
	return no;
}


bool CLine::Mousestroke(int op, int _x, int _y)
/* Process a mousestroke.  Mostly it will be passed to Line. */
{	C_Editor *W = (C_Editor*)parent;

	if (W->parse_line and W->parse_line->y > y)
	    return yes;
        if (op == MOUSE_PRESS and W->parse_line and y > W->parse_line->y)
            W->MoveTo(this, 0);         // This is necessary in order to
            // execute the 'MoveDown()' calls which process the commands
            // we're moving past.
	return Line::Mousestroke(op, _x, _y);
}


bool C_Editor::Keystroke(int key)
{
        if (key == DEL) {
            if (c.ln->buf[c.n] == '\0' and c.ln->next) {
                CLine *next = (CLine*)c.ln->next;
                if (next->type > ty_pastinput)
                    DeleteLineB(next);
            }
        }
        else if (key == BACKSPACE) {
            if (c.n == 0 and c.ln->prev) {
                CLine *prev = (CLine*)c.ln->prev;
                if (prev->type > ty_pastinput)
                    return yes;
            }
        }
        else if (key == WINDOW_QUIT)
            Save();
        return EditorWin::Keystroke(key);
}


void C_Editor::Save()
/* Save the contents of the editor to disk. */
/* Return 'yes' for success. */
{       FILE *output;
        Line *ln;

        output = fopen("barbados.bar", "wt");
        if (output == NULL)
            return;
        for (ln=(Line*)root; ln; ln=(Line*)ln->next) {
            if (ln->type == ty_pastinput) {
                fputs(ln->buf, output);
                fputc('\n', output);
            }
        }
        fclose(output);
        NeedsSave = no;
}


void C_Editor::GetPhraseAtCursor(char dest[], int sizeofdest)
/* If there's a 1-line selection, take that, otherwise take the */
/* word at the cursor. */
{
        if (Marker1.ln and Marker2.ln == Marker1.ln) {
            /* There's a 1-line selection */
            int n = Marker2.n - Marker1.n;
            if (n < 0) {
                dest[0] = '\0';
                return;
            }
            if (n > sizeofdest-1)
                n = sizeofdest - 1;
            strncpy(dest, Marker1.ln->buf + Marker1.n, n);
            dest[n] = '\0';
        }
        else {
            str s,d;
            s = c.ln->buf + c.n;
            while (s > c.ln->buf and isalnum(s[-1]))
                s--;
            d = dest;
            while (isalnum(*s) and d < dest + sizeofdest - 1)
                *d++ = *s++;
            *d = '\0';
        }
}









/*----------------- History: -----------------*/

#define history		temp.ln


static bool LineIsEmpty(Line* ln)
{
	if (ln->next == NULL)
	    return no;
	ln = (Line*)ln->next;
	return (ln->state & NONEMPTY_FLAG) == 0;
}


static bool LineIsEndOfCompileable(Line* ln)
{
	if (ln->next == NULL)
	    return no;
	ln = (Line*)ln->next;
	return (ln->state & COMPILEABLE_FLAG) != 0;
}


bool C_Editor::HistoryAtStartOf(Line* ln)
/* Is this line the start of a compileable entity? */
{
	if (ln->type > ty_pastinput or LineIsEmpty(ln))
	    return no;

	do {
	    if (ln->state & COMPILEABLE_FLAG)
		return yes;
	    ln = (Line*)ln->prev;
	    if (ln == NULL)
		return yes;
	    if (not LineIsEmpty(ln))
		return no;
	    if (ln->type == ty_output)
		return yes;
	} forever;
}


bool C_Editor::HistoryUp(void)
/* Move the 'history_line' to the previous compileable entity. */
{       
	do {
	    if (history->prev == NULL)
		return no;
	    history = (Line*)history->prev;
	    if (HistoryAtStartOf(history))
		return yes;
	} forever;
}


bool C_Editor::HistoryDown(void)
/* Move the 'history_line' to the next compileable entity. */
{
	do {
	    if (history->next == NULL)
		return no;
	    if (history->type == ty_history)
		return no;
	    history = (Line*)history->next;
	    if (history == GetPosition())
		return no;
	    if (history->type == ty_history)
		return no;
	    if (HistoryAtStartOf(history))
		break;
	} forever;
	return yes;
}


void C_Editor::HistoryClear(void)
/* Clear the history lines */
{	Line* ln;

	do {
	    ln = GetPosition();
	    if (ln == NULL or ln->type != ty_history)
		break;
	    DeleteLineB(ln);
	} forever;
}


Line* C_Editor::HistoryDisplay(Line* lower)
/* Display the current historical command just above 'lower'. */
/* Return the first line we inserted. */
{       Line *ln, *ln_end, *newbie;

	/* Turn off the display (it's turned back on when we next wait for a keystroke): */
	SuppressPaints();

	/* Clear the current display: */
	HistoryClear();

	/* Work out the extent of this command: */
	ln = history;
	ln_end = ln;
	do {
	    if (ln == NULL)
		break;
	    if (ln == lower)
		break;
	    if (not LineIsEmpty(ln) and ln->type == ty_pastinput)
		ln_end = ln;
	    if (LineIsEndOfCompileable(ln))
		break;
	    ln = (Line*)ln->next;
	} forever;

	/* Insert it just above the current line: */
	ln = history;
	newbie = NULL;
	do {
	    InsertLineB(lower, yes, ln->buf, ty_history, no);
	    if (newbie == NULL)
		newbie = (Line*)lower->prev;
	    if (ln == ln_end)
		break;
	    ln = (Line*)ln->next;
	} while (ln);
	return newbie;
}


void C_Editor::HistoryInsert(Line* lower, bool goto_bottom)
/* Insert the current historical command into the line-buffer. */
{	Line *ln, *top;

	SuppressPaints();
	top = NULL;
	for (ln=(Line*)lower->prev; ln and ln->type == ty_history; ln = (Line*)ln->prev)
	    top = ln;
	assert(top != NULL);
	parse_line = top;
	SetPosDirectly(top, 0);
	lower = NULL;
	for (ln=top; ln and ln->type == ty_history; ln = (Line*)ln->next) {
	    ln->type = ty_current;
	    ln->state = DefineState((Line*)ln->prev);
	    lower = ln;
	}
	if (ln)
	    ln->state = DefineState((Line*)ln->prev);
	if (goto_bottom and lower)
	    MoveTo(lower, strlen(lower->buf));
}


void C_Editor::HistoryMode(void)
/* Browse through past commands. */
{	Line *lower, *displaystart;
	int i;
        
	history = GetPosition();
	lower = (Line*)history->next;
	if (lower == NULL)
	    return;
	if (history->type > ty_pastinput)
	    HistoryUp();

	do {
	    displaystart = HistoryDisplay(lower);
	    SetPosDirectly(displaystart, strlen(displaystart->buf));
            SetCursor();
	    switch (GetKey()) {

		case UP:    HistoryUp();
			    break;

		case DOWN:  if (not HistoryDown())
				goto ABORT;
			    break;

                case PG_UP: for (i=0; i < 3; i++)
                		HistoryUp();
                	    break;

                case PG_DOWN:for (i=0; i < 3; i++)
                		if (not HistoryDown())
                                    goto ABORT;
                	    break;

		case MOUSE_PRESS:
		case ESC:   goto ABORT;
		case BACKSPACE:
		case LEFT:
		case HOME:
		case RIGHT: HistoryInsert(lower, no);
			    return;

		case END:
		case ENTER: HistoryInsert(lower, yes);
			    return;
	    }
	} forever;

	ABORT:
	HistoryClear();
	SetPosDirectly(lower, 0);
}






/*----------------- Interface functions: ----------------*/

interface C_Editor* CompilerEditor;


void OutputLine(str buf)
/* Output this line to the screen. */
{
	CompilerEditor->ClearAllUndos();
	CompilerEditor->InsertLineB(CompilerEditor->GetPosition(), yes, buf, ty_output, no);
}


interface int EditorGetch(void)
/* Input a single keystroke from the keyboard. */
{
	CompilerEditor->SetCursor();
	//CompilerEditor->SuppressMouse = yes;
	int ch = CompilerEditor->GetKey();
	//CompilerEditor->SuppressMouse = no;
	return ch;
}


interface bool EditorInclude(str filename)
{
	return CompilerEditor->Include(filename);
}


void ClearOutputLines(void)
{
	CompilerEditor->ClearOutputLines();
}


void EditorAddSource(str source)
/* Insert this str at the current location. */
{
	CompilerEditor->InsertBigString(CompilerEditor->GetPosition(), source);
}


int EditorWidth(void)
/* How many characters can we fit on one line? */
{
	if (CompilerEditor == NULL)
	    return 80;
	return CompilerEditor->clientWidth / 8;
}


void EditorReprint(void)
/* Flush the display so all changes are visible. */
{
	TfcYield(no);
}


str EditorGets(char dest[], int sizeofdest)
/* Input a string from the keyboard. */
{	str s;

	s = CompilerEditor->EditOneLine();
	if (strlen(s) + 1 > sizeofdest)
	    memcpy(dest, s, sizeofdest-1), dest[sizeofdest-1] = '\0';
	else strcpy(dest, s);
	return dest;
}


void EditorClearLog(void)
/* Clear the whole editor. */
{
	CompilerEditor->Clear();
}


interface str SubEdit(void)
/* Re-enter the editor to get a single str.  Return NULL if they   */
/* leave the window without entering a compileable str. The caller */
/* must free the returned str. */
{
	return CompilerEditor->EditOneLine();
}
