/*------ The Intermediate Program Representation (IPR): -------*/

/*
Our compiler consists of a front end and a back end.  The front end does lexical
analysis and parsing and type-decoration.  The back end does optimisation and
code generation.  They communicate via the IPR - a data-structure.  The IPR is
language-independent - it consists of a reduced set of constructs: expressions,
branches, returns and switches.  Everything the language generates must be
reduced to this common denominator.

Each statement the function or code-fragment uses must be registered.  This is
so that we can detect dead code, as well as to make it easier to iterate over
statements.  Call:  IprRegisterStatement().

Each statement must 'go somewhere'.  This means that expressions and null statements
must have a non-NULL S->next field and branches must have 2 non-NULL branches.
To exit the code-fragment these should point to a return statement.

Note that 'long' and 'int' are considered different types by C++, but for
code generation they are considered the same (4-byte int's). Therefore we
must convert all tp_long's to tp_int's and tp_ulong's to tp_uint's in the
e->tp field.
*/


typedef enum { oUnknown,

	    /* Atomic expressions: */
	    oConstant, oLocalVar,

	    /* Unary operators: */
	    oNOP1, oCONVERT, oNEG, oNOT, oNOTT, oDER,
	    oSIN, oCOS, oTAN, oATAN, oSQRT,

	    /* Binary operators: */
	    oADD, oSUB, oMUL, oDIV, oMOD,
	    oLT, oGT, oEQ, oNE, oLE, oGE,
	    oAND, oOR, oXOR, oANND, oORR, oSHL, oSHR, oCOMMA,

	    /* Ternary operators: */
	    oTERNARY, oBITFIELD,

	    /* Assignment operators: */
	    oASSIGN, oAADD, oASUB, oAMUL, oADIV, oAMOD, oAXOR, oAOR, oAAND,
	    oASHL, oASHR, oAADDAFTER,

	    /* Functions: */
	    oFUNC,

	    /* Internal use only: */
	    oPUSH, oPUSHSPPLUS, oMEMIA, oSentinel
} o_enum;


enum reg_enum {
        reg_unknown, reg_BX, reg_CX, reg_DX, reg_SI, reg_DI, reg_AX, reg_error,
        reg_stack, reg_error2, reg_fpstack, reg_SP, reg_BP,
        reg_R0, reg_R1, reg_R2, reg_R3, reg_R4, reg_R5, reg_R6, reg_R7,
        reg_maxrealregs, reg_mem, reg_imm, reg_discard, reg_operand,
        reg_somegeneral };
        /* Really, this belongs in compcode.cpp.  But I can't find a nice */
        /* way to segregate this enum from the Expr definition. */


struct Expr {
	/* Ipr fields: */
	o_enum o;                       // What operator.
	tp_enum tp;                     // What type.

	/* Parsing/typing fields: */
	char opr;                       // What parsed operator.
	Type type;                      // The type-ref
	int ref;                        // The type-ref
	str src;                        // Where is it in source-code?

	/* Other fields: */
	union {
	    struct {                    // For binary expressions
		Expr *_left;
		Expr *_right;
	    } b;
	    double f;			// For constant expressions
	    int i;                      // For enum constants
	    void *p;
	    long l;
            struct {                    // For undecorated 'identifier's
            	void* x;
                struct Resolved_node *resolve;
            } idfier;
	    struct variable_node *var;	// For variables
	    struct exprf_node *func;	// For functions
	    struct bitfield_node *bitfield;	// For oBitfield
            class AutoNamedobj *obj;    // For oLocalVar
	} u;

	/* Code generation fields: */
	reg_enum storereg;      // What register holds the result while its brother is calculated?
	reg_enum workreg;       // What register holds the result as the parent is calculated?
	uchar right_first;      // Do we compute the right-hand-side first?
	uchar numiregs;         // How many general-purpose registers do we need?
	uchar numfregs;         // How many floating-point registers do we need?
	uchar needregs;         // Which registers do we need to compute subexpr's?
	short idx;              // How old is this expression?
};

		    // A special expression for function calls:  if opr == op_func, then
		    // the e->right is a pointer to an exprf_node.
typedef struct exprf_node {     
	Expr* func;             	// What is the location of the function?  Usually a constant.
	Expr* owner;            	// Is there an owner object?
	int stack_size;             	// How many bytes do the parameters together consume?
	int arity;                  	// How many parameters are there?
        int flags;                 	// 1=pop 1st param into AX
	Expr* param[1];
		    /* When creating the parse-tree, we store the 'this' expression in 'owner'	*/
		    /* ('owner'==NULL for non-member-function calls).  However, in the process	*/
		    /* of decorating the expression-tree, we move this expression to be the	*/
		    /* first parameter (and we shift the other ones up). */
} *exprf_type;


typedef struct bitfield_node {
	int bit_width;			// How many bits wide?
	int bit_offset;			// What is the bit offset?
	uchar o;			// What operation are we doing?  oDER, oASSIGN, oAADD etc.
	Expr* addr;			// The address of the bit-field
	Expr* operand;		// The optional operand
} *bitfield_type;


typedef enum { st_unknown, st_null, st_expr, st_if, st_switch,
		st_return, st_sentinel } st_enum;


struct Statement {
	st_enum st;                     // What statement? st_expr, st_if, etc.
	char flag1;                     // internal use
	char flag2;                     // internal use
	Statement *next;                // Where do we flow to after doing this statement?
	str src;                        // Where is it in source code?
	uint location;                  // Where is it in compiled code?
        AutoNamedobj *scope;            // What local var's are in scope?
	Statement *next_registered;     // For the list of registered statements
};


/* The 4 allowable statement-types: */

struct ExprStatement : public Statement {
        Expr* e;
};


struct IfStatement : public Statement {
	Expr* Test;
	Statement *True, *Fals;
};


typedef struct switchcase_node {
	int tag;
	Statement* Body;
	uint backpatch;
	struct switchcase_node *next;
} *switchcase_type;


struct SwitchStatement : public Statement {
	Expr* Test;
	Statement *Body;
	switchcase_type Cases;
	Statement *Default;
};


struct ReturnStatement : public Statement {
	Expr* e;
        Expr* destructors;      // 'e' is computed before 'destructors', yet
                                // the value of 'e' is returned in register EAX.
        /* If it's a 'void' return statement, then scope == NULL,
        and this will automatically take care of any destruction
        required.
                If it returns something, then 'scope' can contain variables,
        and if these variables have destructors then the destructors
        must be called after 'e' is computed, and we need a special case
        to make the destructor calls. */
};


class AutoNamedobj : public Namedobj {     // for auto_storage and local_static.
public:
        int offset;             // Offset w.r.t. to the frame pointer.
                                // offset is +ve for parameters, -ve for locals.
	Expr *initialisation;       // The initialisation expression
                                // (if any). For checking constructors etc.
        AutoNamedobj *nextlocal;
	struct vargraph_node *vargraph, *root;
	Expr *last_use;
};


typedef enum {  ph_unoptimised=0,
		    ph_optimised=1,
		    ph_jmp_checked=2
		 } phase_type;


typedef void (*ParamChecker_fn)(exprf_type ef);



#define each_statement      S=first_statement; S; S=S->next_registered
#define left                u.b._left
#define right               u.b._right
#define localstatic_base    ((void*)0x1)
#define Unaries         oDER: case oNOP1: case oNEG: case oNOT: case oNOTT: case oCONVERT:\
			    case oSIN: case oCOS: case oTAN: case oATAN: case oSQRT
#define Atomics          oConstant: case oLocalVar: case oPUSHSPPLUS
#define Binaries        oADD: case oSUB: case oMUL: case oDIV: case oMOD: \
	    case oLT: case oGT: case oEQ: case oNE: case oLE: case oGE: case oAND: \
	    case oOR: case oXOR: case oANND: case oORR: case oSHL: case oSHR: \
	    case oCOMMA: case oASSIGN: case oAADD: case oASUB: case oAMUL: \
	    case oADIV: case oAMOD: case oAXOR: case oAOR: case oAAND: case oASHL: \
	    case oASHR: case oAADDAFTER



/*------------- Constructing IPR's: ----------------*/

extern AutoNamedobj* localvar_root;
extern LocalStaticNamedobj* localstaticvar_root;

void *qmalloc(int n);
void* qcalloc(int n);
bool qptrcheck(void* p);
Expr* NewExpr(int opr, Expr* _right, Expr* _left);
Expr* NewExprFloat(double f);
Expr* NewExprInt(int i);
Expr* NewExprFn(Expr* func, Expr* *parameters, int arity);
Expr* NewEnlargedExprFn(Expr *e, int arity);
Expr* NewExprTernary(Expr* a, Expr* b, Expr* c);
Expr* NewExprVirtualFnTable(void** VirtualFns);
void* NewStatement(st_enum statement);

bool IsComparison(o_enum o);
	/* Is this a comparison operation? */

int TpSize(tp_enum tp);
	/* How big is this datum? */

str OtoString(int o);
	/* For debugging purposes. */

void IprRegisterStatement(Statement* S);
	/* Register this statement as something in use in this code fragment. */
	/* This way we can detect dead code. */

int IprEvaluateIntConstant(Expr* e, bool *successp);
	/* Hopefully this expression evaluates to an integer constant. */
	/* If so, return it. */

bool IprEvaluateConstant(Expr **ep);
	/* Optimise this expression.  Hopefully it evaluates to a constant. */
	/* If so, leave it in 'e->u'. */

funcblock_type IprCompile(Statement* S, void* preferred_location);
	/* Returns a funcblock.  If it can use 'preferred_location',	*/
	/* then it returns this same pointer.  Otherwise it returns a 	*/
	/* block that the caller must use or free. */

void OptimiseExpr(Expr **ep, bool need_result);
        /* Optimise this expression. */

void ErrorParse(str src, str mess, ...);

void DumpExpr(Expr* e, int depth=0);
	/* Dump this expression as a str to stdout. */

void DumpIpr(Statement* S);
	/* Dump this whole function IPR to stdout. */



/*---------------- Generating code: --------------*/

extern Statement *first_statement;

funcblock_type GenerateCode(Statement *S, void* preferred_location);
	/* Emit Pentium code for this IPR.			     */
	/* Returns a malloc'd block: the first 4 bytes is the length */
	/* of the malloc block, the next 4 bytes is the location of  */
	/* the 'extra' info, the next byte is the entry point. */

str RegisterToString(uchar reg);
	/* Display this register. */

void InitLocalStatic(void);
	/* Initialise the local static data-structure. */

int AllocateLocalStatic(uint size, Type type);
	/* Allocate some data storage space inside the current compileable */
	/* entity.  The data lasts as long as the compileable entity does, */
	/* so if you're inside a function then it should last a while.     */

int AllocateAndCopyLocalStatic(uint size, void* s, Type type);
	/* Allocate space with the above function and then copy this data into it. */

int CopyLocalStatic(int offset, uint size, void* s);
	/* Copy this data into a local-static variable that has already been created. */

void* RetrieveLocalStatic(int offset);
	/* Get the ptr corresponding to this offset in the localstatic_buf. */

void InitLocalVars(void);
	/* Initialise the local static adt and local-variables adt. */

DebugInfo* IprGetDebugInfo(DebugInfo *dinfo);
	/* Return a malloc'd block with all debug info for the current compileable. */

void IprClearDebugInfo(void);
	/* Free the malloc'd block which we create with any compilation. */





/*----------------- The parser provides: ------------------*/

extern FunctionNamedobj* CompilerCurrentFuncObj;
	/* The Namedobj* of the entity we're currently compiling, */
	/* for putting in the debugheader. */

extern str CurrentSource;
	/* The start str of the current compileable. */

