/********************************************************\
** MAKE_CONTAINER.CPP:					**
**          The container-level make module.		**
**							**
\********************************************************/

#include "barbados.h"
#include "make_container.h"
#include "source.h"
#include "memory.h"
#include "array.h"
#include "trace.h"
#include "make.h"




/*------------------ Container Make: -------------------*\
This module is concerned with Make operating across 
containers.  

Defn:-  A container is said to be source-linked if its SRC
	    directory is linked in, i.e. conim->src points to it and
	    each string inside it has been compiled, so as to generate
	    make-nodes and dependency information for each object.
	    (Make-nodes and dependency lists are not stored persistently).
	    Of course, container-level Make is only concerned with 
	    directory containers.

Invariant:-   The following invariant is maintained:
	    All open containers have their source linked in if there's any
	    chance that they're out-of-date.   'Out-of-date' means they
	    contain an object which is out-of-date in the ordinary 'make'
	    sense.  'Any chance' is taken to mean the following:   if 
	    this containers 'remake' time-stamp is lower than either of
	    x.interface_change or x.interaddr_change, where 'x' is any
	    container it depends on, then we say it might be out of date.

Algorithm:
	    We follow the following rules in order to maintain this
	    invariant.   (a) Whenever an container is opened, if it needs
	    to be source-linked, we source-link it.  This applies to
	    containers being read from disk and containers retrieved from
	    the cache.  Note that containers coming from the cache may or
	    may not be source-linked already.   (b) Whenever an container
	    has an interface object modified, we scan through all open containers
	    that are not source-link, and check whether they need to be
	    source-linked now.
		When we 'chdir()' out of an container, we bring everything
	    inside up-to-date.  This is called a 'container remake', invoked
	    by ContainerMakeAllObjs(), as opposed to a 'container compile' which
	    is invoked by 'SourceLinkSource()'.  The reason we do this is
	    as an optimisation - to minimise the need for recompilation 
	    when the container is read back in, and to detect errors as 
	    early as possible.

Functions:
	    ContainerMakeAllObjs(conim):       Call Make on every object 
					    inside 'conim'.

	    SourceLinkSource(conim):     Link in the source for 'conim'.

	    ContainerCheckSource(conim):       If 'conim' needs source, mark it
					    so.  (for LinkOldies()).

	    ContainerInterfaceChange(conim):   Process the fact that this conim
					    has had an interface obj change.
					    (Mark relevant conim's as needing
					    to be source-linked).

	    ContainerLinkOldies():	Source-link the above conims, the
					'out-of-date' containers.


\*------------------------------------------------------------------*/


interface void ContainerMakeAllObjs(Conim *conim)
/* Bring everything in this container up-to-date, given that	*/
/* all the objects have been compiled (and hence their  */
/* dependencies are set).  Used for bringing an container	*/
/* up-to-date prior to writing it out. */
{	Make* make, *make_root;
	Directory* dir;
	Namedobj* obj;
	uint h;

	assert(conim != NULL);

	/* Only remake the container if the source is linked in.  */
	/* Otherwise the source can not have changed and    */
	/* the objects are presumably up-to-date already.   */
	if (conim->src) {

	    /* Step 1: Collect all the objects that need recompilation: */
	    dir = conim->directory();
	    make_root = NULL;
	    for (each_dir_obj(dir)) {
		if (obj->make) {
		    if (obj->storage == straight_fn and ((FunctionNamedobj*)obj)->u.fn == NULL)
			continue;
			/* It hasn't even been defined yet, so it won't */
			/* do any good to try to recompile it. */
		    obj->make->next = make_root;
		    make_root = obj->make;
		}
	    }

	    /* Step 2: Remake all these objects. */
	    for (make=make_root; make; make=make->next)
		MakeThis(make);

	    /* This 2-step process is necessary because during compilation, */
	    /* the directory can change. */
	    conim->remake = time(NULL);
	}
}



/*--------------------------*/

static Conim **needsrc_list;
static int ns_idx;


static void NeedSrc_Add(Conim *conim)
/* Add this conim to the 'needsrc_list'. */
{       int i, size;

	/* Is it already there? */
	for (i=0; i < ns_idx; i++)
	    if (needsrc_list[i] == conim)
		return;

	/* Add it in: */
	size = (ns_idx + 1) * sizeof(Conim*);
	size = ((size - 1) | 127) + 1;
	needsrc_list = (Conim**)realloc(anon_heap, needsrc_list, size);
	needsrc_list[ns_idx++] = conim;
}


static Conim* NeedSrc_Next(void)
/* Take an openseg out of the 'needsrc_list' and return it. */
/* Returns NULL when they have all been removed.  */
{       Conim *conim;

	if (ns_idx == 0)
	    return NULL;
	conim = needsrc_list[--ns_idx];
	if (ns_idx == 0) {
	    free(anon_heap, needsrc_list);
	    needsrc_list = NULL;
	}
	return conim;
}


interface void ContainerInterfaceChange(Conim *this_conim, bool just_address)
/* Process the fact that this container has had an interface	*/
/* object change.  All open containers which depend directly  */
/* on it must have their source linked in. */
{       ConimIterator iter;
	Conim *conim;


	/* Update the time-stamps: */
	this_conim->interaddr_change = time(NULL);
	if (not just_address) 
	    this_conim->interface_change = time(NULL);


	/* Add all relevant conim's into 'NeedSrc'. */
	for (each_conim) 
	    if (conim->src == NULL and conim->LinksTo(this_conim))
		NeedSrc_Add(conim);
		// this could be optimised by storing the subset of
		// opensegs which do not have source linked.
}


interface void ContainerCheckSource(Conim *conim)
/* If this container needs SRC linked in, link it in.	    */
/* It needs SRC if it doesn't already have it, and  */
/* conim->remake < x.interface_change or	    */
/* conim->remake < x.interaddr_change for some container  */
/* 'x' that it depends on. */
{       
	/* $$$
	Conim *B;
	int i;

	if (conim->src)
	    return;
	if (not TypeEqual(conim->root.type, direc_typstr))
	    return;
	for (each_aeli(B, conim->prerequisites)) {
	    if (conim->remake < B->interface_change or
		conim->remake < B->interaddr_change)
		goto NEEDS_SRC;
	}
	return;

	NEEDS_SRC:
	NeedSrc_Add(conim);
	*/
}



interface void ContainerLinkOldies(void)
{
/*       Conim *conim;

	while ((conim = NeedSrc_Next()) != NULL) {
	    if (ContainerIsValid(conim))
		ContainerLinkSource(conim);
	}
	$$$
*/
}

